/***************************************************************************
 *   Copyright (C) 2006 by Bram Biesbrouck                                 *
 *   b@beligum.org                                                         *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA.             *
 *
 *   In addition, as a special exception, the copyright holders give	   *
 *   permission to link the code of portions of this program with the	   *
 *   OpenSSL library under certain conditions as described in each	   *
 *   individual source file, and distribute linked combinations		   *
 *   including the two.							   *
 *   You must obey the GNU General Public License in all respects	   *
 *   for all of the code used other than OpenSSL.  If you modify	   *
 *   file(s) with this exception, you may extend this exception to your	   *
 *   version of the file(s), but you are not obligated to do so.  If you   *
 *   do not wish to do so, delete this exception statement from your	   *
 *   version.  If you delete this exception statement from all source	   *
 *   files in the program, then also delete it here.			   *
 ***************************************************************************/

#include <cstdio>
#include <cstdlib>
#include <iostream>

#include <sys/stat.h> 

#include <libinstrudeo/isdwscommunicator.h>
#include <libinstrudeo/isdlogger.h>
#include <libinstrudeo/isdutils.h>
#include <libinstrudeo/isdwsftptransmitter.h>

#undef LOG_HEADER
#define LOG_HEADER "Error while transmitting FTP file.\n"
#include <libinstrudeo/isdloggermacros.h>

void* ISDWSFtpTransmitter::callbackClass = NULL;
FtpCallbackFunc ISDWSFtpTransmitter::callbackPtr = NULL;

//-----CONSTRUCTORS-----
ISDWSFtpTransmitter::ISDWSFtpTransmitter()
    : ISDObject(),
      ftpHandle(NULL)
{
}

//-----DESTRUCTOR-----
ISDWSFtpTransmitter::~ISDWSFtpTransmitter()
{
}

//-----PUBLIC METHODS-----
ISDObject::ISDErrorCode ISDWSFtpTransmitter::sendRecording(string& recFile, string& server, int port,
							   string& directory, string& filename,
							   string& username, string& password)
{
    //Check the arguments
    if (!VALID_FILE(recFile)) {
	LOG_WARNING("The specified file for FTP upload doesn't exist.");
	RETURN_ERROR(ISD_FTP_FILE_ERROR);
    }
    if (port<=0) {
	LOG_WARNING("The specified port for FTP upload is invalid.");
	RETURN_ERROR(ISD_ARGS_ERROR);
    }

    //get the file size of the local recording-file
    struct stat file_info;
    int fd = open(recFile.c_str(), O_RDONLY) ;
    fstat(fd, &file_info);
    FILE* file = fdopen(fd, "rb");
    if (file==NULL) {
	LOG_WARNING("Error while initialising the FTP subsystem.");
	RETURN_ERROR(ISD_FTP_FILE_ERROR);
    }

    //init libcurl
    if (curl_global_init(CURL_GLOBAL_ALL) != CURLE_OK) {
	LOG_WARNING("Error while initialising the FTP subsystem.");
	RETURN_ERROR(ISD_FTP_FILE_ERROR);
    }
    ftpHandle = curl_easy_init();
    if (ftpHandle == NULL) {
	LOG_WARNING("Error while initialising the FTP subsystem.");
	RETURN_ERROR(ISD_FTP_FILE_ERROR);
    }

    //set credentials
    string loginStr = username+":"+password;
    curl_easy_setopt(ftpHandle, CURLOPT_USERPWD, loginStr.c_str());

    //disable extended passive mode, didn't work
    curl_easy_setopt(ftpHandle, CURLOPT_FTP_USE_EPSV, FALSE) ;
    
    // specify that we're uploading data
    curl_easy_setopt(ftpHandle, CURLOPT_UPLOAD, 1);

    //specify target
    string urlStr;
    if (urlStr.substr(0, 6)!="ftp://") {
	urlStr += "ftp://";
    }
    urlStr += server;

    //captorials' service doesn't use directories
    //curl_easy_setopt(ftpHandle, CURLOPT_FTP_CREATE_MISSING_DIRS, 1);

    if (directory=="") {
	urlStr += "/";
    }
    else {
	if (directory[0]!='/' && directory!="") {
	    urlStr += "/";
	}
	urlStr += directory;
	if (directory[directory.size()-1]!='/' && directory!="") {
	    urlStr += "/";
	}
    }
    urlStr += filename;
    curl_easy_setopt(ftpHandle, CURLOPT_URL, urlStr.c_str());

    //set callback
    if (ISDWSFtpTransmitter::callbackPtr != NULL) {
	curl_easy_setopt(ftpHandle, CURLOPT_NOPROGRESS, FALSE);
	curl_easy_setopt(ftpHandle, CURLOPT_PROGRESSFUNCTION, ISDWSFtpTransmitter::progressCallback);
    }

    //specify port
    curl_easy_setopt(ftpHandle, CURLOPT_PORT, port);

    //specify which file to upload
    curl_easy_setopt(ftpHandle, CURLOPT_READDATA, file);

    //and give the size of the upload (optional)
    curl_easy_setopt(ftpHandle, CURLOPT_INFILESIZE, file_info.st_size); 

    //upload the file
    CURLcode res = curl_easy_perform(ftpHandle); 

    //always cleanup
    cleanup();
    fclose(file);
    close(fd);

    curl_global_cleanup();

    //if the user aborted, don't return an error.
    
    //TODO: when sending a very large file (tested with 50MB+),
    //      a timeout occurs, although the file was sent successfully.
    //      This is a quick and dirty solution that needs revising.
    if (res != CURLE_OK &&
	res != CURLE_ABORTED_BY_CALLBACK &&
	res != CURLE_OPERATION_TIMEOUTED)
	{
	    string err = "Error while transmitting recording to the FTP server ("+ISDUtils::getInstance()->intToString(res)+").";
	    LOG_WARNING(err);
	    RETURN_ERROR(ISD_FTP_FILE_ERROR);
	}
    
    RETURN_SUCCESS;
}
int ISDWSFtpTransmitter::progressCallback(void *clientp, double dltotal, double dlnow, double ultotal, double ulnow)
{
    if (ISDWSFtpTransmitter::callbackPtr != NULL) {
	int ret = (*ISDWSFtpTransmitter::callbackPtr)(ISDWSFtpTransmitter::callbackClass, ulnow, ultotal);
	//if this function (callbackPtr) returns 0, the upload needs to be aborted,
	//libcurl aborts if this function (progressCallback) returns a non-zero value.
	return ret;
    }
}
ISDObject::ISDErrorCode ISDWSFtpTransmitter::setSentBytesCallback(ISDWSCommunicator* callbackClass, FtpCallbackFunc funcPtr)
{
    ISDWSFtpTransmitter::callbackClass = callbackClass;
    ISDWSFtpTransmitter::callbackPtr = funcPtr;

    RETURN_SUCCESS;
}

//-----PROTECTED METHODS-----
void ISDWSFtpTransmitter::cleanup()
{
    if (ftpHandle!=NULL) {
	curl_easy_cleanup(ftpHandle);
    }
}
