/***************************************************************************
 *   Copyright (C) 2006 by Bram Biesbrouck                                 *
 *   b@beligum.org                                                         *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA.             *
 *
 *   In addition, as a special exception, the copyright holders give	   *
 *   permission to link the code of portions of this program with the	   *
 *   OpenSSL library under certain conditions as described in each	   *
 *   individual source file, and distribute linked combinations		   *
 *   including the two.							   *
 *   You must obey the GNU General Public License in all respects	   *
 *   for all of the code used other than OpenSSL.  If you modify	   *
 *   file(s) with this exception, you may extend this exception to your	   *
 *   version of the file(s), but you are not obligated to do so.  If you   *
 *   do not wish to do so, delete this exception statement from your	   *
 *   version.  If you delete this exception statement from all source	   *
 *   files in the program, then also delete it here.			   *
 ***************************************************************************/

#ifndef ISDGLWIDGET_H
#define ISDGLWIDGET_H

/**
 * Abstract base class of the Composite pattern that represents a displayable OpenGL widget.
 * The display() method of this class can be used as the display function
 * of an OpenGL implementation. The method only delivers the calculations
 * though, no windows, etc. are created.
 *
 * @author Bram Biesbrouck <b@beligum.org>
 */

#include <string>
#include <set>

#include <libinstrudeo/isdobject.h>

#include <GL/gl.h>
#include <GL/glu.h>
#include <GL/glut.h>

using namespace std;

class ISDGLWidget : public ISDObject
{
public:
    //-----CONSTRUCTORS-----
    /**
     * Standard constructor that initialises the OpenGL structures.
     * Make sure it's safe to call this contructor for every created
     * child, so don't add GL's one-time-initialisation routines here,
     * because they will be called for every new child.
     */
    ISDGLWidget();

    /**
     * Warning: all children of this widget are deleted too,
     * so don't delete them (eg. the commentboxes) if your keep
     * a reference to one, since it will result in a crash.
     */
    virtual ~ISDGLWidget();

    //-----METHODS-----
    /**
     * Adds a child widget to the composite class.
     * When this widget is displayed, the call is
     * passed on to all its child widgets too.
     * If the widget is already a child of this widget,
     * nothing happens.
     *
     * @param widget The child widget.
     * @return Returns a code that indicates success or failure.
     */
    ISDErrorCode addChild(ISDGLWidget* widget);

    /**
     * Removes the specified child widget from the child tree of this
     * widget. It just removes the child from the child-list, no destroying
     * or something like that is performed.
     * If the widget is not a child of this widget, nothing happens.
     *
     * @param widget The child-widget to remove.
     * @return Returns a code that indicates success or failure.
     */
    ISDErrorCode removeChild(ISDGLWidget* widget);

    /**
     * Removes all child widgets from the child tree of this widget.
     * It just removes the child from the child-list, no destroying
     * or something like that is performed.
     *
     * @return Returns a code that indicates success or failure.
     */
    ISDErrorCode removeAllChildren();

    /**
     * Removes all child widgets from the child tree of this widget
     * and deletes them too.
     *
     * @return Returns a code that indicates success or failure.
     */
    ISDErrorCode removeAndDeleteAllChildren();

    /**
     * Returns a collection of the children of this widget.
     *
     * @return The set of children, empty if this widget had no children, never NULL.
     */
    set<ISDGLWidget*>* getChildren();

    /**
     * Checks if the specified widget is a child widget of this widget.
     *
     * @return true if widget is a child of this widget, false otherwise.
     */
    bool hasChild(ISDGLWidget* widget);

    /**
     * Sets the parent widget if this widget.
     *
     * @param parent The parent.
     */
    void setParent(ISDGLWidget* parent);

    /**
     * Returns the parent of this widget or NULL if none is set.
     *
     * @return The parent widget or NULL if none is set.
     */
    ISDGLWidget* getParent();

    /**
     * Abstract display function for OpenGL; implement this method
     * with the specific render-logic for each derived widget.
     */
    virtual void display() = 0;

 protected:
    //-----METHODS-----
    /**
     * Abstract function that gets called when the widget is created.
     *
     * @return Returns a code that indicates success or failure.
     */
    virtual ISDErrorCode initGL() = 0;

    /**
     * This function is useful when the OpenGL context changed,
     * re-initialised the context of the canvas and all its children.
     *
     * @return Returns a code that indicates success or failure.
     */
    virtual ISDErrorCode reInitGL() = 0;

    /**
     * Convenience-method for subclasses that initialises the children of this widget.
     *
     * @return Returns a code that indicates success or failure.
     */
    ISDErrorCode initChildren();

    /**
     * Convenience-method for subclasses that re-initialises the children of this widget.
     *
     * @return Returns a code that indicates success or failure.
     */
    ISDErrorCode reInitChildren();
    
    /**
     * Convenience-methods for subclasses that draws the children of this widget.
     *
     * @return Returns a code that indicates success or failure.
     */
    ISDErrorCode drawChildren();

    //-----VARIABLES-----
    set<ISDGLWidget*> children;
    ISDGLWidget* parent;
};

#endif
