/***************************************************************************
 *   Copyright (C) 2006 by Bram Biesbrouck                                 *
 *   b@beligum.org                                                         *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA.             *
 *
 *   In addition, as a special exception, the copyright holders give	   *
 *   permission to link the code of portions of this program with the	   *
 *   OpenSSL library under certain conditions as described in each	   *
 *   individual source file, and distribute linked combinations		   *
 *   including the two.							   *
 *   You must obey the GNU General Public License in all respects	   *
 *   for all of the code used other than OpenSSL.  If you modify	   *
 *   file(s) with this exception, you may extend this exception to your	   *
 *   version of the file(s), but you are not obligated to do so.  If you   *
 *   do not wish to do so, delete this exception statement from your	   *
 *   version.  If you delete this exception statement from all source	   *
 *   files in the program, then also delete it here.			   *
 ***************************************************************************/

#ifndef ISDCOMMENTBOX_H
#define ISDCOMMENTBOX_H

/**
 * General abstract superclass for all commentboxes.
 * It holds information about the commentbox and implements
 * the ISDGLWidget interface, so it knows how to draw itself.
 * The added functionality for the derived classes is very limited,
 * since nearly all information and functionality is embedded in this
 * superclass. Check the existing-implementations for samples.
 * Make sure every derived class constructs only one ISDGLObjFile that is statically
 * shared between the classes of the same type to avoid loading the .obj
 * twice. The object is fetched through the abstract getObjFile() method.
 * 
 * @author Bram Biesbrouck <b@beligum.org>
 */

#include <list>
#include <glibmm/ustring.h>

#include <libinstrudeo/isdglwidget.h>
#include <libinstrudeo/isdutils.h> //for DIR_DELIM

using namespace std;

class ISDGLObjFile;

/*
 * For now, this is the only file that needs configuration files, so it is
 * hardcoded in this header file. It can be useful to abstract this and create
 * a ISDApplicationSettings class (see isdprocessd for an example) that parses
 * a configuration file instead.
 *
 * SOLVED: used a compile -DDATADIR flag instead, defined in Makefile.am
 */
//#define BASE_CONFIG_DIR "/etc/instrudeo"
#define COMMENTBOXES_DIR_NAME (string(DATADIR)+string(DIR_DELIM)+string("commentboxes"))
#define FONTS_DIR_NAME "fonts"
#define FONT_SCALE_SIZE 0.75
#define NEWLINE_DELIMITER "\n"
#define LINE_SPACER 2

//-----DEFAULT CONSTANTS-----
#define DEFAULT_FONT_FILE (string(DATADIR)+string(DIR_DELIM)+string(FONTS_DIR_NAME)+string(DIR_DELIM)+string("FreeSans.ttf"))
#define DEFAULT_FONT_FACE_SIZE 20
#define DEFAULT_COMMENTBOX_TEXT ""
#define DEFAULT_COMMENTBOX_LANGUAGE ""
#define DEFAULT_COMMENTBOX_STARTTIME 0
#define DEFAULT_COMMENTBOX_DURATION 3000

//use this for the language of a commentbox if you want the box to be shown
//under all languages
#define ISD_COMMENTBOX_LANGUAGE_CATCHALL "*"

class ISDVideoCanvas;
class FTFont;

class ISDCommentbox : public ISDGLWidget
{

 public:
    //-----CONSTANTS-----
    enum commentboxMirrorMode {
	ISD_COMMENTBOX_MIRROR_NONE,
	ISD_COMMENTBOX_MIRROR_HORIZONTAL,
	ISD_COMMENTBOX_MIRROR_VERTICAL,
	ISD_COMMENTBOX_MIRROR_BOTH
    };
    enum commentboxType {
	ISD_COMMENT_TEXTBALLOON_TYPE,
	ISD_COMMENT_ROUNDED_BOX_TYPE,
	ISD_COMMENT_TEXTBALLOON_2D_TYPE,
	ISD_COMMENT_CAPTOVERLAY_TYPE,
    };

    /**
     * Constructor that creates an empty comment, filling all variables with default values.
     * Only an ID must be specified, since it uniquely represents the commentbox.
     * Don't forget to initialise position, size, color, ... in the constructor of the derived
     * subclass.
     *
     * @param parent The parent-canvas widget or NULL if none (use with caution).
     * @param id The id of this commentbox.
     */
    ISDCommentbox(ISDVideoCanvas* parent, int id);

    virtual ~ISDCommentbox();

    //-----FACTORY METHODS-----
    /**
     * Creates a new commentbox of the specified type.
     * This way, we have a centralized way to create different commentbox-types.
     *
     * @param type The commentbox-type to create.
     * @param parent The parent-canvas widget or NULL if none (use with caution).
     * @param id The id of this commentbox.
     * @return The newly allocated commentbox of the specified type or 
     *         NULL if the type wasn't recognised or an error occurred.
     */
    static ISDCommentbox* createCommentbox(commentboxType type, ISDVideoCanvas* parent, int id);

    //-----GETTERS-----
    /**
     * Returns the id of this commentbox.
     *
     * @return The id.
     */
    int getId();

    /**
     * Returns the size of the model unit cube to where every loaded .obj is scaled.
     * It is used in scaling the mesh to the current width and height.
     */
    float getModelUnit();

    /**
     * Returns the hotspot of this commentbox.
     * The hotspot is used on some commentbox-types,
     * like the textballoon to indicate the tail-end.
     * It defaults to (0, 0) if not used.
     *
     * @param pos An array of size 2 that will contain the x and y hotspot-position.
     */
    virtual void getHotspot(int pos[2]);

    /**
     * Returns the type of this commentbox.
     * Make sure you add an entry to the enum above
     * for every subclass to enforce a unique id.
     *
     * @return The type.
     */
    virtual int getType() = 0;

    /**
     * Returns the language of the text of the commentbox.
     * Note: It's possible to set the language of a comment to "*",
     *       in this case, the commentbox is alway show, no matter what 
     *       language is currently selected. Use the constant
     *       ISD_COMMENTBOX_LANGUAGE_CATCHALL
     *       for this instead of "*".
     *
     * @return The language.
     */
    Glib::ustring& getLanguage();
    
    /**
     * Returns the position of this commentbox.
     *
     * @param pos An array of size 2 that will contain the x and y position.
     */
    void getPosition(int pos[2]);

    /**
     * Returns the position of this commentbox, as scaled to the current settings.
     *
     * @param pos An array of size 2 that will contain the scaled x and y position.
     */
    void getScaledPosition(int pos[2]);

    /**
     * Returns the original (unscaled) size of this commentbox.
     *
     * @param pos An array of size 2 that will contain the width and height.
     */
    void getSize(int size[2]);

    /**
     * Returns the size of this commentbox, as scaled to the current settings.
     *
     * @param pos An array of size 2 that will contain the scaled width and height.
     */
    void getScaledSize(int size[2]);
    
    /**
     * Returns the text of this commentbox.
     *
     * @return The text in unicode-encoding.
     */
    Glib::ustring& getText();

    /**
     * Returns the mirror mode of this commentbox.
     *
     * @return The mode.
     */
    commentboxMirrorMode getMirrorMode();

    /**
     * Returns the start time of this commentbox in milliseconds.
     *
     * @return The starttime.
     */
    int getStartTime();

    /**
     * Returns the duration of this commentbox in milliseconds.
     *
     * @return The duration.
     */
    int getDuration();

    /**
     * Returns the color of this commentbox.µ
     * This can be overridden in the subclass to provide a correct
     * "fill color hint". This means the user can specify one color,
     * that can be used in the coloring of the object. This isn't
     * necessary though.
     *
     * @param pos An array of size 4 that will contain RGBA color.
     */
    virtual void getColor(float color[4]);

    /**
     * Returns the textcolor of this commentbox.
     *
     * @param pos An array of size 4 that will contain RGBA color.
     */
    virtual void getTextColor(float color[4]);

    //-----SETTERS-----
    /**
     * See the corresponding getter for documentation.
     * @see getLanguage()
     */
    void setLanguage(Glib::ustring& language);

    /**
     * See the corresponding getter for documentation.
     * @see getPosition()
     */
    void setPosition(int x, int y);

    /**
     * See the corresponding getter for documentation.
     * @see getScaledPosition()
     */
    void setScaledPosition(int x, int y);

    /**
     * See the corresponding getter for documentation.
     * @see getSize()
     */
    void setSize(int width, int height);

    /**
     * See the corresponding getter for documentation.
     * @see getScaledSize()
     */
    void setScaledSize(int width, int height);
    
    /**
     * See the corresponding getter for documentation.
     * @see getText()
     */
    void setText(Glib::ustring& text);

    /**
     * See the corresponding getter for documentation.
     * @see getMirrorMode()
     */
    void setMirrorMode(commentboxMirrorMode mode);

    /**
     * See the corresponding getter for documentation.
     * @see getStartTime()
     */
    void setStartTime(int time);

    /**
     * See the corresponding getter for documentation.
     * @see getDuration()
     */
    void setDuration(int duration);

    /**
     * See the corresponding getter for documentation.
     * @see getColor()
     */
    virtual void setColor(float r, float g, float b, float a);

    /**
     * See the corresponding getter for documentation.
     * @see getTextColor()
     */
    virtual void setTextColor(float r, float g, float b, float a);

    //-----OVERLOADED METHODS-----
    /**
     * Draws the commentbox onto the current (initialized) OpenGL canvas.
     */
    virtual void display();

    /**
     * Initializes the OpenGL system for commentboxes in
     * general. This method usually suffices and doesn't need to
     * be overloaded for every subclass, but if some additional initialisation
     * is required, this option is available.
     *
     * @return Returns a code that indicates success or failure.
     */
    virtual ISDErrorCode initGL();

    /**
     * This function is useful when the OpenGL context changed,
     * re-initialised the context of the canvas and all its children.
     *
     * @return Returns a code that indicates success or failure.
     */
    virtual ISDErrorCode reInitGL();

    //-----METHODS-----
    /**
     * Recalculates the scale, based on the (new) size.
     * Because the memory-mesh-model never changes size,
     * we must recalc these values every time something happens
     * that can affect the size or position of the displayed commentbox.
     * This method is quite complex, see the source code for a detailed
     * explanation of the different modifiers.
     */
    void recalcScales();

    /**
     * Returns a reference to the .obj file wrapper. This method must be
     * implemented for every commentbox-type. Make sure every subclass stores
     * this object statically to avoid loading the .obj file multiple times when
     * multiple commentbox-instances are created.
     *
     * @return A reference to the .obj file wrapper class or NULL if something went wrong.
     */
    virtual ISDGLObjFile* getObjFile() = 0;

 protected:
    //-----CONSTANTS-----
    enum { X, Y, Z, W };
    
    static float MIRROR_MATRIX_NONE[16];
    static float MIRROR_MATRIX_HORIZONTAL[16];
    static float MIRROR_MATRIX_VERTICAL[16];
    static float MIRROR_MATRIX_BOTH[16];

    //-----METHODS-----
    /**
     * Initialises the 3D-font.
     *
     * @return Returns a code that indicates success or failure.
     */
    ISDErrorCode initFont();

    /**
     * This method is called from within the display() method to draw the mesh model.
     * The default implementation draws the model using GL lists, but it can be overloaded
     * in a child class to use the ISDGLObjFile::drawModel() method instead to provide
     * more flexibility.
     */
    virtual void displayModel();

    /**
     * Help-routine for the display method that draws the text.
     */
    void displayText();

    /**
     * This method is used to optimize the scales-calculations for every subclass.
     * The scale values in the recalcScales() method will be multiplied with the
     * values returned by this method, so it is an opportunity for the subclass
     * to tweak the scaling-process.
     * Default return values are 1.0, 1.0, 1.0
     *
     * @param values Three values that will be used in the XYZ-scale calculations.
     */
    virtual void getScalesTweakValues(float values[3]);

    /**
     * Cuts a textstring so it fits inside the bound of the text-plane.
     * Note: The height-bounds of the plane are not checked, this is done
     * inside the displayText routine.
     * The textplaneWidth, textplaneHeight and font variables are used in this
     * method and must be properly initialized.
     *
     * @param text The text to display.
     * @param maxWidth Pass a variable to know the maximal width of all lines.
     * @return A list of lines that fits inside the text-plane-width.
     */
    list<Glib::ustring> splitTextHorizontally(Glib::ustring text, float* maxWidth=NULL);

    //-----VARIABLES-----
    int id;
    commentboxMirrorMode mirrorMode;
    float* mirrorMatrix;
    /*
     * This holds a backup of the position of light 0, as initialised.
     */
    float lightPosition[4];
    /**
     * This position represents the position, as requested by the user.
     */
    int position[2];
    /**
     * This position represents the position, scaled to the current
     * situation of the OpenGL world.
     * The same count for the following two sizes.
     */
    float scaledPosition[2];
    int size[2];
    float scales[3];
    Glib::ustring text;
    Glib::ustring language;
    float color[4];
    float textColor[4];
    int startTime;
    int duration;
    FTFont* font;
    /**
     * This is the upper left position of the textplane,
     * exactly as it is in the centered unit cube in memory.
     * The same for the following two dimensions.
     */
    float textplaneTopLeft[3];
    float textplaneWidth;
    float textplaneHeight;
    /**
     * This position is the upper left position of the textplane,
     * scaled to the current situation of the OpenGL world.
     */
    float scaledTextplaneOrigin[3];
};

#endif
