/*    
 *  GLM library.  Wavefront .obj file format reader/writer/manipulator.
 *
 *  Written by Nate Robins, 1997.
 *  email: ndr@pobox.com
 *  www: http://www.pobox.com/~ndr
 */

#include <math.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <assert.h>
#include <locale.h>

#include <libinstrudeo/glm.h>

/* Some <math.h> files do not define M_PI... */
#ifndef M_PI
#define M_PI 3.14159265358979323846
#endif

/* defines */
#define T(x) object->triangles[(x)]


/* enums */
enum { X, Y, Z, W };			/* elements of a vertex */


/* typedefs */

/* _GLMnode: general purpose node
 */
typedef struct _GLMnode {
    GLuint           index;
    GLboolean        averaged;
    struct _GLMnode* next;
} GLMnode;

/* strdup is actually not a standard ANSI C or POSIX routine
   so implement a private one.  OpenVMS does not have a strdup; Linux's
   standard libc doesn't declare strdup by default (unless BSD or SVID
   interfaces are requested). */
static char *
stralloc(const char *string)
{
    char *copy;

    copy = (char*)malloc(strlen(string) + 1);
    if (copy == NULL)
	return NULL;
    strcpy(copy, string);
    return copy;
}

/* private functions */

/* _glmMax: returns the maximum of two floats */
static GLfloat
_glmMax(GLfloat a, GLfloat b) 
{
    if (a > b)
	return a;
    return b;
}

/* _glmAbs: returns the absolute value of a float */
static GLfloat
_glmAbs(GLfloat f)
{
    if (f < 0)
	return -f;
    return f;
}

/* _glmDot: compute the dot product of two vectors
 *
 * u - array of 3 GLfloats (GLfloat u[3])
 * v - array of 3 GLfloats (GLfloat v[3])
 */
static GLfloat
_glmDot(GLfloat* u, GLfloat* v)
{
    assert(u);
    assert(v);

    /* compute the dot product */
    return u[X] * v[X] + u[Y] * v[Y] + u[Z] * v[Z];
}

/* _glmCross: compute the cross product of two vectors
 *
 * u - array of 3 GLfloats (GLfloat u[3])
 * v - array of 3 GLfloats (GLfloat v[3])
 * n - array of 3 GLfloats (GLfloat n[3]) to return the cross product in
 */
static GLvoid
_glmCross(GLfloat* u, GLfloat* v, GLfloat* n)
{
    assert(u);
    assert(v);
    assert(n);

    /* compute the cross product (u x v for right-handed [ccw]) */
    n[X] = u[Y] * v[Z] - u[Z] * v[Y];
    n[Y] = u[Z] * v[X] - u[X] * v[Z];
    n[Z] = u[X] * v[Y] - u[Y] * v[X];
}

/* _glmNormalize: normalize a vector
 *
 * n - array of 3 GLfloats (GLfloat n[3]) to be normalized
 */
static GLvoid
_glmNormalize(GLfloat* n)
{
    GLfloat l;

    assert(n);

    /* normalize */
    l = (GLfloat)sqrt(n[X] * n[X] + n[Y] * n[Y] + n[Z] * n[Z]);
    n[0] /= l;
    n[1] /= l;
    n[2] /= l;
}

/* _glmEqual: compares two vectors and returns GL_TRUE if they are
 * equal (within a certain threshold) or GL_FALSE if not. An epsilon
 * that works fairly well is 0.000001.
 *
 * u - array of 3 GLfloats (GLfloat u[3])
 * v - array of 3 GLfloats (GLfloat v[3]) 
 */
static GLboolean
_glmEqual(GLfloat* u, GLfloat* v, GLfloat epsilon)
{
    if (_glmAbs(u[0] - v[0]) < epsilon &&
	_glmAbs(u[1] - v[1]) < epsilon &&
	_glmAbs(u[2] - v[2]) < epsilon) 
	{
	    return GL_TRUE;
	}
    return GL_FALSE;
}

/* _glmWeldVectors: eliminate (weld) vectors that are within an
 * epsilon of each other.
 *
 * vectors    - array of GLfloat[3]'s to be welded
 * numvectors - number of GLfloat[3]'s in vectors
 * epsilon    - maximum difference between vectors 
 *
 */
GLfloat*
_glmWeldVectors(GLfloat* vectors, GLuint* numvectors, GLfloat epsilon)
{
    GLfloat* copies;
    GLuint   copied;
    GLuint   i, j;

    copies = (GLfloat*)malloc(sizeof(GLfloat) * 3 * (*numvectors + 1));
    memcpy(copies, vectors, (sizeof(GLfloat) * 3 * (*numvectors + 1)));

    copied = 1;
    for (i = 1; i <= *numvectors; i++) {
	for (j = 1; j <= copied; j++) {
	    if (_glmEqual(&vectors[3 * i], &copies[3 * j], epsilon)) {
		goto duplicate;
	    }
	}

	/* must not be any duplicates -- add to the copies array */
	copies[3 * copied + 0] = vectors[3 * i + 0];
	copies[3 * copied + 1] = vectors[3 * i + 1];
	copies[3 * copied + 2] = vectors[3 * i + 2];
	j = copied;				/* pass this along for below */
	copied++;

    duplicate:
	/* set the first component of this vector to point at the correct
	   index into the new copies array */
	vectors[3 * i + 0] = (GLfloat)j;
    }

    *numvectors = copied-1;
    return copies;
}

/* glmFindObject: Find an object in the model
 * made public by Bram Biesbrouck for object-access support
 */
GLMobject*
glmFindObject(GLMmodel* model, char* name)
{
    GLMobject* object;

    assert(model);

    object = model->objects;
    while(object) {
	if (!strcmp(name, object->name))
	    break;
	object = object->next;
    }

    return object;
}

/* _glmFindGroup: Find a group in the object
 */
GLMgroup*
_glmFindGroup(GLMobject* object, char* name)
{
    GLMgroup* group;

    assert(object);

    group = object->groups;
    while(group) {
	if (!strcmp(name, group->name))
	    break;
	group = group->next;
    }

    return group;
}

/* _glmAddObject: Add an object to the model
 */
GLMobject*
_glmAddObject(GLMmodel* model, char* name)
{
    GLMobject* object;

    object = glmFindObject(model, name);
    if (!object) {
	object = (GLMobject*)malloc(sizeof(GLMobject));
    
	//init members
	object->name = stralloc(name);
	object->numvertices   = 0;
	object->vertices      = NULL;
	object->numnormals    = 0;
	object->normals       = NULL;
	object->numtexcoords  = 0;
	object->texcoords     = NULL;
	object->numfacetnorms = 0;
	object->facetnorms    = NULL;
	object->numtriangles  = 0;
	object->triangles     = NULL;
	object->numgroups     = 0;
	object->groups        = NULL;
	object->position[0]   = 0.0;
	object->position[1]   = 0.0;
	object->position[2]   = 0.0;

	//new object is PREpended to the list of objects
	object->next = model->objects;
	model->objects = object;
	model->numobjects++;
    }

    return object;
}

/* _glmAddGroup: Add a group to the object
 */
GLMgroup*
_glmAddGroup(GLMobject* object, char* name)
{
    GLMgroup* group;

    group = _glmFindGroup(object, name);
    if (!group) {
	group = (GLMgroup*)malloc(sizeof(GLMgroup));
	group->name = stralloc(name);
	group->material = 0;
	group->numtriangles = 0;
	group->triangles = NULL;
	group->next = object->groups;
	object->groups = group;
	object->numgroups++;
    }

    return group;
}

/* _glmFindMaterial: Find a material in the model
 */
GLuint
_glmFindMaterial(GLMmodel* model, char* name)
{
    GLuint i;

    for (i = 0; i < model->nummaterials; i++) {
	if (!strcmp(model->materials[i].name, name))
	    goto found;
    }

    /* didn't find the name, so set it as the default material */
    if (PRINT_DEBUG)
	printf("_glmFindMaterial():  can't find material \"%s\".\n", name);
    i = 0;

 found:
    return i;
}


/* _glmDirName: return the directory given a path
 *
 * path - filesystem path
 *
 * The return value should be free'd.
 */
static char*
_glmDirName(char* path)
{
    char* dir;
    char* s;

    dir = stralloc(path);

    s = strrchr(dir, '/');
    if (s)
	s[1] = '\0';
    else
	dir[0] = '\0';

    return dir;
}


/* _glmReadMTL: read a wavefront material library file
 *
 * model - properly initialized GLMmodel structure
 * name  - name of the material library
 */
static GLvoid
_glmReadMTL(GLMmodel* model, char* name)
{
    FILE* file;
    char* dir;
    char* filename;
    char  buf[128];
    GLuint nummaterials, i;

    dir = _glmDirName(model->pathname);
    filename = (char*)malloc(sizeof(char) * (strlen(dir) + strlen(name) + 1));
    strcpy(filename, dir);
    strcat(filename, name);
    free(dir);

    /* open the file */
    file = fopen(filename, "r");
    if (!file) {
	if (PRINT_DEBUG)
	    fprintf(stderr, "_glmReadMTL() failed: can't open material file \"%s\".\n", filename);
	exit(1);
    }
    free(filename);

    /* count the number of materials in the file */
    nummaterials = 1;
    while(fscanf(file, "%s", buf) != EOF) {
	switch(buf[0]) {
	case '#':				/* comment */
	    /* eat up rest of line */
	    fgets(buf, sizeof(buf), file);
	    break;
	case 'n':				/* newmtl */
	    fgets(buf, sizeof(buf), file);
	    nummaterials++;
	    sscanf(buf, "%s %s", buf, buf);
	    break;
	default:
	    /* eat up rest of line */
	    fgets(buf, sizeof(buf), file);
	    break;
	}
    }

    rewind(file);

    /* allocate memory for the materials */
    model->materials = (GLMmaterial*)malloc(sizeof(GLMmaterial) * nummaterials);
    model->nummaterials = nummaterials;

    /* set the default material */
    for (i = 0; i < nummaterials; i++) {
	model->materials[i].name = NULL;
	model->materials[i].shininess = 0;
	model->materials[i].diffuse[0] = 0.8;
	model->materials[i].diffuse[1] = 0.8;
	model->materials[i].diffuse[2] = 0.8;
	model->materials[i].diffuse[3] = 1.0;
	model->materials[i].ambient[0] = 0.2;
	model->materials[i].ambient[1] = 0.2;
	model->materials[i].ambient[2] = 0.2;
	model->materials[i].ambient[3] = 1.0;
	model->materials[i].specular[0] = 0.0;
	model->materials[i].specular[1] = 0.0;
	model->materials[i].specular[2] = 0.0;
	model->materials[i].specular[3] = 1.0;
    }
    model->materials[0].name = stralloc("default");

    /* now, read in the data */
    nummaterials = 0;
    while(fscanf(file, "%s", buf) != EOF) {
	switch(buf[0]) {
	case '#':				/* comment */
	    /* eat up rest of line */
	    fgets(buf, sizeof(buf), file);
	    break;
	case 'n':				/* newmtl */
	    fgets(buf, sizeof(buf), file);
	    sscanf(buf, "%s %s", buf, buf);
	    nummaterials++;
	    model->materials[nummaterials].name = stralloc(buf);
	    break;
	case 'N':
	    fscanf(file, "%f", &model->materials[nummaterials].shininess);
	    /* wavefront shininess is from [0, 1000], so scale for OpenGL */
	    model->materials[nummaterials].shininess /= 1000.0;
	    model->materials[nummaterials].shininess *= 128.0;
	    break;
	case 'K':
	    switch(buf[1]) {
	    case 'd':
		fscanf(file, "%f %f %f",
		       &model->materials[nummaterials].diffuse[0],
		       &model->materials[nummaterials].diffuse[1],
		       &model->materials[nummaterials].diffuse[2]);
		break;
	    case 's':
		fscanf(file, "%f %f %f",
		       &model->materials[nummaterials].specular[0],
		       &model->materials[nummaterials].specular[1],
		       &model->materials[nummaterials].specular[2]);
		break;
	    case 'a':
		fscanf(file, "%f %f %f",
		       &model->materials[nummaterials].ambient[0],
		       &model->materials[nummaterials].ambient[1],
		       &model->materials[nummaterials].ambient[2]);
		break;
	    default:
		/* eat up rest of line */
		fgets(buf, sizeof(buf), file);
		break;
	    }
	    break;
	    /* added support for transparency by Bram Biesbrouck */
	case 'T':
	case 'd':
	    fscanf(file, "%f", &model->materials[nummaterials].diffuse[3]);
	    //this is not really necessary, since only diffuse is used in alpha-lighting computation
	    //model->materials[nummaterials].specular[3] = model->materials[nummaterials].diffuse[3];
	    //model->materials[nummaterials].ambient[3] = model->materials[nummaterials].diffuse[3];
	    //model->materials[nummaterials].emmissive[3] = model->materials[nummaterials].diffuse[3];
	    break;
	default:
	    /* eat up rest of line */
	    fgets(buf, sizeof(buf), file);
	    break;
	}
    }
}

/* _glmFirstPass: first pass at a Wavefront OBJ file that gets all the
 * statistics of the model (such as #vertices, #normals, etc)
 *
 * model - properly initialized GLMmodel structure
 * file  - (fopen'd) file descriptor 
 */
static GLvoid
_glmFirstPass(GLMmodel* model, FILE* file) 
{
    GLuint    numvertices;		/* number of vertices in model */
    GLuint    numnormals;			/* number of normals in model */
    GLuint    numtexcoords;		/* number of texcoords in model */
    GLuint    numtriangles;		/* number of triangles in model */
    GLMobject* object = NULL; 			/* current object */
    GLMgroup* group;				/* current group */
    unsigned  v, n, t;
    char      buf[128];

    numvertices = numnormals = numtexcoords = numtriangles = 0;
    while(fscanf(file, "%s", buf) != EOF) {
	switch(buf[0]) {
	case '#':				/* comment */
	    /* eat up rest of line */
	    fgets(buf, sizeof(buf), file);
	    break;
	case 'o':				/* object */
	    /* eat up rest of line */
	    fgets(buf, sizeof(buf), file);
	    sscanf(buf, "%s", buf);
      
	    //write the stats of the previous object if any
	    if (object){
		object->numvertices  = numvertices;
		object->numnormals   = numnormals;
		object->numtexcoords = numtexcoords;
		object->numtriangles = numtriangles;
		numvertices = numnormals = numtexcoords = numtriangles = 0;
	    }
      
	    object = _glmAddObject(model, buf);
	    // make default group
	    group = _glmAddGroup(object, "default");
	    break;
	case 'v':				/* v, vn, vt */
	    switch(buf[1]) {
	    case '\0':			/* vertex */
		/* eat up rest of line */
		fgets(buf, sizeof(buf), file);
		numvertices++;
		break;
	    case 'n':				/* normal */
		/* eat up rest of line */
		fgets(buf, sizeof(buf), file);
		numnormals++;
		break;
	    case 't':				/* texcoord */
		/* eat up rest of line */
		fgets(buf, sizeof(buf), file);
		numtexcoords++;
		break;
	    default:
		if (PRINT_DEBUG)
		    printf("_glmFirstPass(): Unknown token \"%s\".\n", buf);
		exit(1);
		break;
	    }
	    break;
	case 'm':				/* mtllib */
	    fgets(buf, sizeof(buf), file);
	    sscanf(buf, "%s %s", buf, buf);
	    model->mtllibname = stralloc(buf);
	    _glmReadMTL(model, buf);
	    break;
	case 'u':				/* usemtl */
	    /* eat up rest of line */
	    fgets(buf, sizeof(buf), file);
	    break;
	case 'g':				/* group */
	    /* eat up rest of line */
	    fgets(buf, sizeof(buf), file);
	    sscanf(buf, "%s", buf);
	    group = _glmAddGroup(object, buf);
	    break;
	case 'f':				/* face */
	    v = n = t = 0;
	    fscanf(file, "%s", buf);
	    /* can be one of %d, %d//%d, %d/%d, %d/%d/%d %d//%d */
	    if (strstr(buf, "//")) {
		/* v//n */
		sscanf(buf, "%d//%d", &v, &n);
		fscanf(file, "%d//%d", &v, &n);
		fscanf(file, "%d//%d", &v, &n);
		numtriangles++;
		group->numtriangles++;
		while(fscanf(file, "%d//%d", &v, &n) > 0) {
		    numtriangles++;
		    group->numtriangles++;
		}
	    } else if (sscanf(buf, "%d/%d/%d", &v, &t, &n) == 3) {
		/* v/t/n */
		fscanf(file, "%d/%d/%d", &v, &t, &n);
		fscanf(file, "%d/%d/%d", &v, &t, &n);
		numtriangles++;
		group->numtriangles++;
		while(fscanf(file, "%d/%d/%d", &v, &t, &n) > 0) {
		    numtriangles++;
		    group->numtriangles++;
		}
	    } else if (sscanf(buf, "%d/%d", &v, &t) == 2) {
		/* v/t */
		fscanf(file, "%d/%d", &v, &t);
		fscanf(file, "%d/%d", &v, &t);
		numtriangles++;
		group->numtriangles++;
		while(fscanf(file, "%d/%d", &v, &t) > 0) {
		    numtriangles++;
		    group->numtriangles++;
		}
	    } else {
		/* v */
		fscanf(file, "%d", &v);
		fscanf(file, "%d", &v);
		numtriangles++;
		group->numtriangles++;
		while(fscanf(file, "%d", &v) > 0) {
		    numtriangles++;
		    group->numtriangles++;
		}
	    }
	    break;

	default:
	    /* eat up rest of line */
	    fgets(buf, sizeof(buf), file);
	    break;
	}
    }
  
    //write the stats of the last object
    if (object){
	object->numvertices  = numvertices;
	object->numnormals   = numnormals;
	object->numtexcoords = numtexcoords;
	object->numtriangles = numtriangles;
    }
  
    /* allocate memory for the triangles in each group */
    object = model->objects;
    while (object){
	group = object->groups;
	while(group) {
	    group->triangles = (GLuint*)malloc(sizeof(GLuint) * group->numtriangles);
	    group->numtriangles = 0;
	    group = group->next;
	}
	object = object->next;
    }
}

/* _glmSecondPass: second pass at a Wavefront OBJ file that gets all
 * the data.
 *
 * model - properly initialized GLMmodel structure
 * file  - (fopen'd) file descriptor 
 */
static GLvoid
_glmSecondPass(GLMmodel* model, FILE* file) 
{
    GLuint    numvertices;		/* number of vertices in model */
    GLuint    numnormals;			/* number of normals in model */
    GLuint    numtexcoords;		/* number of texcoords in model */
    GLuint    numtriangles;		/* number of triangles in model */
    GLfloat*  vertices;			/* array of vertices  */
    GLfloat*  normals;			/* array of normals */
    GLfloat*  texcoords;			/* array of texture coordinates */
    GLMobject* object = NULL; 	/* current object */
    GLMgroup* group;				/* current group pointer */
    GLuint    material;			/* current material */
    GLuint    v, n, t;
    char      buf[128];
    GLfloat x, y, z;
    int ret;

    /* set the pointer shortcuts */
    vertices     = model->objects->vertices;
    normals      = model->objects->normals;
    texcoords    = model->objects->texcoords;
    group        = model->objects->groups;

    /* on the second pass through the file, read all the data into the
       allocated arrays */
    numvertices = numnormals = numtexcoords = 1;
    numtriangles = 0;
    material = 0;

    while(fscanf(file, "%s", buf) != EOF) {
	switch(buf[0]) {
	case '#':				/* comment */
	    /* eat up rest of line */
	    fgets(buf, sizeof(buf), file);
	    break;
	case 'o':				/* object */
	    /* eat up rest of line */
	    fgets(buf, sizeof(buf), file);
	    sscanf(buf, "%s", buf);
      
	    object = glmFindObject(model, buf);
	    assert(object);
	    object->groups->material = material;
      
	    //set the new pointer shortcuts
	    vertices     = object->vertices;
	    normals      = object->normals;
	    texcoords    = object->texcoords;
	    group        = object->groups;
	    numvertices = numnormals = numtexcoords = 1;
	    numtriangles = 0;
	    material = 0;
      
	    break;
	case 'v':				/* v, vn, vt */
	    switch(buf[1]) {
	    case '\0':			/* vertex */
	        fscanf(file, "%f %f %f", 
		       &vertices[3 * numvertices + X], 
		       &vertices[3 * numvertices + Y], 
		       &vertices[3 * numvertices + Z]);
		numvertices++;
	      
		break;
	    case 'n':				/* normal */
		fscanf(file, "%f %f %f", 
		       &normals[3 * numnormals + X],
		       &normals[3 * numnormals + Y], 
		       &normals[3 * numnormals + Z]);
		numnormals++;
		break;
	    case 't':				/* texcoord */
		fscanf(file, "%f %f", 
		       &texcoords[2 * numtexcoords + X],
		       &texcoords[2 * numtexcoords + Y]);
		numtexcoords++;
		break;
	    }
	    break;
	case 'u':
	    fgets(buf, sizeof(buf), file);
	    sscanf(buf, "%s %s", buf, buf);
	    group->material = material = _glmFindMaterial(model, buf);
	    break;
	case 'g':				/* group */
	    /* eat up rest of line */
	    fgets(buf, sizeof(buf), file);
	    sscanf(buf, "%s", buf);
	    group = _glmFindGroup(object, buf);
	    group->material = material;
	    break;
	case 'f':				/* face */
	    v = n = t = 0;
	    fscanf(file, "%s", buf);
	    /* can be one of %d, %d//%d, %d/%d, %d/%d/%d %d//%d */
	    if (strstr(buf, "//")) {
		/* v//n */
		sscanf(buf, "%d//%d", &v, &n);
		T(numtriangles).vindices[0] = v;
		T(numtriangles).nindices[0] = n;
		fscanf(file, "%d//%d", &v, &n);
		T(numtriangles).vindices[1] = v;
		T(numtriangles).nindices[1] = n;
		fscanf(file, "%d//%d", &v, &n);
		T(numtriangles).vindices[2] = v;
		T(numtriangles).nindices[2] = n;
		group->triangles[group->numtriangles++] = numtriangles;
		numtriangles++;
		while(fscanf(file, "%d//%d", &v, &n) > 0) {
		    T(numtriangles).vindices[0] = T(numtriangles-1).vindices[0];
		    T(numtriangles).nindices[0] = T(numtriangles-1).nindices[0];
		    T(numtriangles).vindices[1] = T(numtriangles-1).vindices[2];
		    T(numtriangles).nindices[1] = T(numtriangles-1).nindices[2];
		    T(numtriangles).vindices[2] = v;
		    T(numtriangles).nindices[2] = n;
		    group->triangles[group->numtriangles++] = numtriangles;
		    numtriangles++;
		}
	    } else if (sscanf(buf, "%d/%d/%d", &v, &t, &n) == 3) {
		/* v/t/n */
		T(numtriangles).vindices[0] = v;
		T(numtriangles).tindices[0] = t;
		T(numtriangles).nindices[0] = n;
		fscanf(file, "%d/%d/%d", &v, &t, &n);
		T(numtriangles).vindices[1] = v;
		T(numtriangles).tindices[1] = t;
		T(numtriangles).nindices[1] = n;
		fscanf(file, "%d/%d/%d", &v, &t, &n);
		T(numtriangles).vindices[2] = v;
		T(numtriangles).tindices[2] = t;
		T(numtriangles).nindices[2] = n;
		group->triangles[group->numtriangles++] = numtriangles;
		numtriangles++;
		while(fscanf(file, "%d/%d/%d", &v, &t, &n) > 0) {
		    T(numtriangles).vindices[0] = T(numtriangles-1).vindices[0];
		    T(numtriangles).tindices[0] = T(numtriangles-1).tindices[0];
		    T(numtriangles).nindices[0] = T(numtriangles-1).nindices[0];
		    T(numtriangles).vindices[1] = T(numtriangles-1).vindices[2];
		    T(numtriangles).tindices[1] = T(numtriangles-1).tindices[2];
		    T(numtriangles).nindices[1] = T(numtriangles-1).nindices[2];
		    T(numtriangles).vindices[2] = v;
		    T(numtriangles).tindices[2] = t;
		    T(numtriangles).nindices[2] = n;
		    group->triangles[group->numtriangles++] = numtriangles;
		    numtriangles++;
		}
	    } else if (sscanf(buf, "%d/%d", &v, &t) == 2) {
		/* v/t */
		T(numtriangles).vindices[0] = v;
		T(numtriangles).tindices[0] = t;
		fscanf(file, "%d/%d", &v, &t);
		T(numtriangles).vindices[1] = v;
		T(numtriangles).tindices[1] = t;
		fscanf(file, "%d/%d", &v, &t);
		T(numtriangles).vindices[2] = v;
		T(numtriangles).tindices[2] = t;
		group->triangles[group->numtriangles++] = numtriangles;
		numtriangles++;
		while(fscanf(file, "%d/%d", &v, &t) > 0) {
		    T(numtriangles).vindices[0] = T(numtriangles-1).vindices[0];
		    T(numtriangles).tindices[0] = T(numtriangles-1).tindices[0];
		    T(numtriangles).vindices[1] = T(numtriangles-1).vindices[2];
		    T(numtriangles).tindices[1] = T(numtriangles-1).tindices[2];
		    T(numtriangles).vindices[2] = v;
		    T(numtriangles).tindices[2] = t;
		    group->triangles[group->numtriangles++] = numtriangles;
		    numtriangles++;
		}
	    } else {
		/* v */
		sscanf(buf, "%d", &v);
		T(numtriangles).vindices[0] = v;
		fscanf(file, "%d", &v);
		T(numtriangles).vindices[1] = v;
		fscanf(file, "%d", &v);
		T(numtriangles).vindices[2] = v;
		group->triangles[group->numtriangles++] = numtriangles;
		numtriangles++;
		while(fscanf(file, "%d", &v) > 0) {
		    T(numtriangles).vindices[0] = T(numtriangles-1).vindices[0];
		    T(numtriangles).vindices[1] = T(numtriangles-1).vindices[2];
		    T(numtriangles).vindices[2] = v;
		    group->triangles[group->numtriangles++] = numtriangles;
		    numtriangles++;
		}
	    }
	    break;

	default:
	    /* eat up rest of line */
	    fgets(buf, sizeof(buf), file);
	    break;
	}
    }
}




/* public functions */

/* glmUnitize: "unitize" a model by translating it to the origin and
 * scaling it to fit in a unit cube around the origin.  Returns the
 * scalefactor used.
 *
 * model - properly initialized GLMmodel structure 
 */
GLfloat
glmUnitize(GLMmodel* model)
{
    GLuint  i;
    GLfloat maxx, minx, maxy, miny, maxz, minz;
    GLfloat cx, cy, cz, w, h, d;
    GLfloat scale;

    assert(model);
  
    //calc the values for the first object and apply them to the rest too
    //doing this, we save the relative proportions of the objects
    GLMobject* object = model->objects;
    assert(object->vertices);
  
    /* get the max/mins */
    maxx = minx = object->vertices[3 + X];
    maxy = miny = object->vertices[3 + Y];
    maxz = minz = object->vertices[3 + Z];
    for (i = 1; i <= object->numvertices; i++) {
	if (maxx < object->vertices[3 * i + X])
	    maxx = object->vertices[3 * i + X];
	if (minx > object->vertices[3 * i + X])
	    minx = object->vertices[3 * i + X];

	if (maxy < object->vertices[3 * i + Y])
	    maxy = object->vertices[3 * i + Y];
	if (miny > object->vertices[3 * i + Y])
	    miny = object->vertices[3 * i + Y];

	if (maxz < object->vertices[3 * i + Z])
	    maxz = object->vertices[3 * i + Z];
	if (minz > object->vertices[3 * i + Z])
	    minz = object->vertices[3 * i + Z];
    }

    /* calculate object width, height, and depth */
    w = _glmAbs(maxx) + _glmAbs(minx);
    h = _glmAbs(maxy) + _glmAbs(miny);
    d = _glmAbs(maxz) + _glmAbs(minz);

    /* calculate center of the object */
    cx = (maxx + minx) / 2.0;
    cy = (maxy + miny) / 2.0;
    cz = (maxz + minz) / 2.0;

    /* calculate unitizing scale factor */
    scale = 1.0 / _glmMax(_glmMax(w, h), d);

    //now, appply the values to all objects
    while(object){
	/* translate around center then scale */
	for (i = 1; i <= object->numvertices; i++) {
	    object->vertices[3 * i + X] -= cx;
	    object->vertices[3 * i + Y] -= cy;
	    object->vertices[3 * i + Z] -= cz;
	    object->vertices[3 * i + X] *= scale;
	    object->vertices[3 * i + Y] *= scale;
	    object->vertices[3 * i + Z] *= scale;
	}
	  
	object = object->next;
    }

    return scale;
}

/* glmDimensions: Calculates the dimensions (width, height, depth) of
 * an object.
 *
 * object      - initialized GLMobject structure
 * dimensions - array of 3 GLfloats (GLfloat dimensions[3])
 */
GLvoid
glmDimensions(GLMobject* object, GLfloat* dimensions)
{
    GLuint i;
    GLfloat maxx, minx, maxy, miny, maxz, minz;

    assert(object);
    assert(object->vertices);
    assert(dimensions);

    /* get the max/mins */
    maxx = minx = object->vertices[3 + X];
    maxy = miny = object->vertices[3 + Y];
    maxz = minz = object->vertices[3 + Z];
    for (i = 1; i <= object->numvertices; i++) {
	if (maxx < object->vertices[3 * i + X])
	    maxx = object->vertices[3 * i + X];
	if (minx > object->vertices[3 * i + X])
	    minx = object->vertices[3 * i + X];

	if (maxy < object->vertices[3 * i + Y])
	    maxy = object->vertices[3 * i + Y];
	if (miny > object->vertices[3 * i + Y])
	    miny = object->vertices[3 * i + Y];

	if (maxz < object->vertices[3 * i + Z])
	    maxz = object->vertices[3 * i + Z];
	if (minz > object->vertices[3 * i + Z])
	    minz = object->vertices[3 * i + Z];
    }

    /* calculate object width, height, and depth */
    dimensions[X] = _glmAbs(maxx) + _glmAbs(minx);
    dimensions[Y] = _glmAbs(maxy) + _glmAbs(miny);
    dimensions[Z] = _glmAbs(maxz) + _glmAbs(minz);
}

/* glmScale: Scales a model by a given amount.
 * 
 * model - properly initialized GLMmodel structure
 * scale - scalefactor (0.5 = half as large, 2.0 = twice as large)
 */
GLvoid
glmScale(GLMmodel* model, GLfloat scale)
{
    GLuint i;
  
    GLMobject* object;
    object = model->objects;
    while (object){
	for (i = 1; i <= object->numvertices; i++) {
	    object->vertices[3 * i + X] *= scale;
	    object->vertices[3 * i + Y] *= scale;
	    object->vertices[3 * i + Z] *= scale;
	}
	object = object->next;
    }
}

/* glmReverseWinding: Reverse the polygon winding for all polygons in
 * this model.  Default winding is counter-clockwise.  Also changes
 * the direction of the normals.
 * 
 * model - properly initialized GLMmodel structure 
 */
GLvoid
glmReverseWinding(GLMmodel* model)
{
    GLuint i, swap;

    assert(model);
  
    GLMobject* object;
    object = model->objects;
    while (object){
	for (i = 0; i < object->numtriangles; i++) {
	    swap = T(i).vindices[0];
	    T(i).vindices[0] = T(i).vindices[2];
	    T(i).vindices[2] = swap;
	
	    if (object->numnormals) {
		swap = T(i).nindices[0];
		T(i).nindices[0] = T(i).nindices[2];
		T(i).nindices[2] = swap;
	    }
	
	    if (object->numtexcoords) {
		swap = T(i).tindices[0];
		T(i).tindices[0] = T(i).tindices[2];
		T(i).tindices[2] = swap;
	    }
	}
	
	/* reverse facet normals */
	for (i = 1; i <= object->numfacetnorms; i++) {
	    object->facetnorms[3 * i + X] = -object->facetnorms[3 * i + X];
	    object->facetnorms[3 * i + Y] = -object->facetnorms[3 * i + Y];
	    object->facetnorms[3 * i + Z] = -object->facetnorms[3 * i + Z];
	}
	
	/* reverse vertex normals */
	for (i = 1; i <= object->numnormals; i++) {
	    object->normals[3 * i + X] = -object->normals[3 * i + X];
	    object->normals[3 * i + Y] = -object->normals[3 * i + Y];
	    object->normals[3 * i + Z] = -object->normals[3 * i + Z];
	}
	  
	object = object->next;
    }
}

/* glmFacetNormals: Generates facet normals for a model (by taking the
 * cross product of the two vectors derived from the sides of each
 * triangle).  Assumes a counter-clockwise winding.
 *
 * model - initialized GLMmodel structure
 */
GLvoid
glmFacetNormals(GLMmodel* model)
{
    GLuint  i;
    GLfloat u[3];
    GLfloat v[3];
  
    assert(model);
  
    GLMobject* object;
    object = model->objects;
    while (object){
	assert(object->vertices);
	
	/* clobber any old facetnormals */
	if (object->facetnorms)
	    free(object->facetnorms);
	
	/* allocate memory for the new facet normals */
	object->numfacetnorms = object->numtriangles;
	object->facetnorms = (GLfloat*)malloc(sizeof(GLfloat) *
					      3 * (object->numfacetnorms + 1));
	
	for (i = 0; i < object->numtriangles; i++) {
	    object->triangles[i].findex = i+1;
	
	    u[X] = object->vertices[3 * T(i).vindices[1] + X] -
		object->vertices[3 * T(i).vindices[0] + X];
	    u[Y] = object->vertices[3 * T(i).vindices[1] + Y] -
		object->vertices[3 * T(i).vindices[0] + Y];
	    u[Z] = object->vertices[3 * T(i).vindices[1] + Z] -
		object->vertices[3 * T(i).vindices[0] + Z];
	
	    v[X] = object->vertices[3 * T(i).vindices[2] + X] -
		object->vertices[3 * T(i).vindices[0] + X];
	    v[Y] = object->vertices[3 * T(i).vindices[2] + Y] -
		object->vertices[3 * T(i).vindices[0] + Y];
	    v[Z] = object->vertices[3 * T(i).vindices[2] + Z] -
		object->vertices[3 * T(i).vindices[0] + Z];
	
	    _glmCross(u, v, &object->facetnorms[3 * (i+1)]);
	    _glmNormalize(&object->facetnorms[3 * (i+1)]);
	}
	  
	object = object->next;
    }
}

/* glmVertexNormals: Generates smooth vertex normals for a model.
 * First builds a list of all the triangles each vertex is in.  Then
 * loops through each vertex in the the list averaging all the facet
 * normals of the triangles each vertex is in.  Finally, sets the
 * normal index in the triangle for the vertex to the generated smooth
 * normal.  If the dot product of a facet normal and the facet normal
 * associated with the first triangle in the list of triangles the
 * current vertex is in is greater than the cosine of the angle
 * parameter to the function, that facet normal is not added into the
 * average normal calculation and the corresponding vertex is given
 * the facet normal.  This tends to preserve hard edges.  The angle to
 * use depends on the model, but 90 degrees is usually a good start.
 *
 * model - initialized GLMmodel structure
 * angle - maximum angle (in degrees) to smooth across
 */
GLvoid
glmVertexNormals(GLMmodel* model, GLfloat angle)
{
    GLMnode*  node;
    GLMnode*  tail;
    GLMnode** members;
    GLfloat*  normals;
    GLuint    numnormals;
    GLfloat   average[3];
    GLfloat   dot, cos_angle;
    GLuint    i, avg;

    assert(model);
  
    GLMobject* object;
    object = model->objects;
    while (object){
	assert(object->facetnorms);
	
	/* calculate the cosine of the angle (in degrees) */
	cos_angle = cos(angle * M_PI / 180.0);
	
	/* nuke any previous normals */
	if (object->normals)
	    free(object->normals);
	
	/* allocate space for new normals */
	object->numnormals = object->numtriangles * 3; /* 3 normals per triangle */
	object->normals = (GLfloat*)malloc(sizeof(GLfloat)* 3* (object->numnormals+1));
	
	/* allocate a structure that will hold a linked list of triangle
	   indices for each vertex */
	members = (GLMnode**)malloc(sizeof(GLMnode*) * (object->numvertices + 1));
	for (i = 1; i <= object->numvertices; i++)
	    members[i] = NULL;
	  
	/* for every triangle, create a node for each vertex in it */
	for (i = 0; i < object->numtriangles; i++) {
	    node = (GLMnode*)malloc(sizeof(GLMnode));
	    node->index = i;
	    node->next  = members[T(i).vindices[0]];
	    members[T(i).vindices[0]] = node;
	
	    node = (GLMnode*)malloc(sizeof(GLMnode));
	    node->index = i;
	    node->next  = members[T(i).vindices[1]];
	    members[T(i).vindices[1]] = node;
	
	    node = (GLMnode*)malloc(sizeof(GLMnode));
	    node->index = i;
	    node->next  = members[T(i).vindices[2]];
	    members[T(i).vindices[2]] = node;
	}
	
	/* calculate the average normal for each vertex */
	numnormals = 1;
	for (i = 1; i <= object->numvertices; i++) {
	    /* calculate an average normal for this vertex by averaging the
	       facet normal of every triangle this vertex is in */
	    node = members[i];
	    if (!node && PRINT_DEBUG)
		fprintf(stderr, "glmVertexNormals(): vertex w/o a triangle\n");
	    average[0] = 0.0; average[1] = 0.0; average[2] = 0.0;
	    avg = 0;
	    while (node) {
		/* only average if the dot product of the angle between the two
		   facet normals is greater than the cosine of the threshold
		   angle -- or, said another way, the angle between the two
		   facet normals is less than (or equal to) the threshold angle */
		dot = _glmDot(&object->facetnorms[3 * T(node->index).findex],
			      &object->facetnorms[3 * T(members[i]->index).findex]);
		if (dot > cos_angle) {
		    node->averaged = GL_TRUE;
		    average[0] += object->facetnorms[3 * T(node->index).findex + 0];
		    average[1] += object->facetnorms[3 * T(node->index).findex + 1];
		    average[2] += object->facetnorms[3 * T(node->index).findex + 2];
		    avg = 1;			/* we averaged at least one normal! */
		} else {
		    node->averaged = GL_FALSE;
		}
		node = node->next;
	    }
	
	    if (avg) {
		/* normalize the averaged normal */
		_glmNormalize(average);
	
		/* add the normal to the vertex normals list */
		object->normals[3 * numnormals + 0] = average[0];
		object->normals[3 * numnormals + 1] = average[1];
		object->normals[3 * numnormals + 2] = average[2];
		avg = numnormals;
		numnormals++;
	    }
	
	    /* set the normal of this vertex in each triangle it is in */
	    node = members[i];
	    while (node) {
		if (node->averaged) {
		    /* if this node was averaged, use the average normal */
		    if (T(node->index).vindices[0] == i)
			T(node->index).nindices[0] = avg;
		    else if (T(node->index).vindices[1] == i)
			T(node->index).nindices[1] = avg;
		    else if (T(node->index).vindices[2] == i)
			T(node->index).nindices[2] = avg;
		} else {
		    /* if this node wasn't averaged, use the facet normal */
		    object->normals[3 * numnormals + 0] = 
			object->facetnorms[3 * T(node->index).findex + 0];
		    object->normals[3 * numnormals + 1] = 
			object->facetnorms[3 * T(node->index).findex + 1];
		    object->normals[3 * numnormals + 2] = 
			object->facetnorms[3 * T(node->index).findex + 2];
		    if (T(node->index).vindices[0] == i)
			T(node->index).nindices[0] = numnormals;
		    else if (T(node->index).vindices[1] == i)
			T(node->index).nindices[1] = numnormals;
		    else if (T(node->index).vindices[2] == i)
			T(node->index).nindices[2] = numnormals;
		    numnormals++;
		}
		node = node->next;
	    }
	}
	  
	object->numnormals = numnormals - 1;
	
	/* free the member information */
	for (i = 1; i <= object->numvertices; i++) {
	    node = members[i];
	    while (node) {
		tail = node;
		node = node->next;
		free(tail);
	    }
	}
	free(members);
	
	/* pack the normals array (we previously allocated the maximum
	   number of normals that could possibly be created (numtriangles *
	   3), so get rid of some of them (usually alot unless none of the
	   facet normals were averaged)) */
	normals = object->normals;
	object->normals = (GLfloat*)malloc(sizeof(GLfloat)* 3* (object->numnormals+1));
	for (i = 1; i <= object->numnormals; i++) {
	    object->normals[3 * i + 0] = normals[3 * i + 0];
	    object->normals[3 * i + 1] = normals[3 * i + 1];
	    object->normals[3 * i + 2] = normals[3 * i + 2];
	}
	free(normals);
	
	if (PRINT_DEBUG)
	    printf("glmVertexNormals(): %d normals generated\n", object->numnormals);
	  
	object = object->next;
    }
}


/* glmLinearTexture: Generates texture coordinates according to a
 * linear projection of the texture map.  It generates these by
 * linearly mapping the vertices onto a square.
 *
 * model - pointer to initialized GLMmodel structure
 */
GLvoid
glmLinearTexture(GLMmodel* model)
{
    GLMgroup *group;
    GLfloat dimensions[3];
    GLfloat x, y, scalefactor;
    GLuint i;
  
    assert(model);
	
    GLMobject* object;
    object = model->objects;
    while (object){	
	if (object->texcoords)
	    free(object->texcoords);
	object->numtexcoords = object->numvertices;
	object->texcoords=(GLfloat*)malloc(sizeof(GLfloat)*2*(object->numtexcoords+1));
	  
	glmDimensions(object, dimensions);
	scalefactor = 2.0 / 
	    _glmAbs(_glmMax(_glmMax(dimensions[0], dimensions[1]), dimensions[2]));
	
	/* do the calculations */
	for(i = 1; i <= object->numvertices; i++) {
	    x = object->vertices[3 * i + 0] * scalefactor;
	    y = object->vertices[3 * i + 2] * scalefactor;
	    object->texcoords[2 * i + 0] = (x + 1.0) / 2.0;
	    object->texcoords[2 * i + 1] = (y + 1.0) / 2.0;
	}
	  
	/* go through and put texture coordinate indices in all the triangles */
	group = object->groups;
	while(group) {
	    for(i = 0; i < group->numtriangles; i++) {
		T(group->triangles[i]).tindices[0] = T(group->triangles[i]).vindices[0];
		T(group->triangles[i]).tindices[1] = T(group->triangles[i]).vindices[1];
		T(group->triangles[i]).tindices[2] = T(group->triangles[i]).vindices[2];
	    }    
	    group = group->next;
	}
	
	object = object->next;
    }

}

/* glmSpheremapTexture: Generates texture coordinates according to a
 * spherical projection of the texture map.  Sometimes referred to as
 * spheremap, or reflection map texture coordinates.  It generates
 * these by using the normal to calculate where that vertex would map
 * onto a sphere.  Since it is impossible to map something flat
 * perfectly onto something spherical, there is distortion at the
 * poles.  This particular implementation causes the poles along the X
 * axis to be distorted.
 *
 * model - pointer to initialized GLMmodel structure
 */
GLvoid
glmSpheremapTexture(GLMmodel* model)
{
    GLMgroup* group;
    GLfloat theta, phi, rho, x, y, z, r;
    GLuint i;
  
    assert(model);
  
    GLMobject* object;
    object = model->objects;
    while (object){
	assert(object->normals);
	
	if (object->texcoords)
	    free(object->texcoords);
	object->numtexcoords = object->numnormals;
	object->texcoords=(GLfloat*)malloc(sizeof(GLfloat)*2*(object->numtexcoords+1));
	     
	/* do the calculations */
	for (i = 1; i <= object->numnormals; i++) {
	    z = object->normals[3 * i + 0];	/* re-arrange for pole distortion */
	    y = object->normals[3 * i + 1];
	    x = object->normals[3 * i + 2];
	    r = sqrt((x * x) + (y * y));
	    rho = sqrt((r * r) + (z * z));
	      
	    if(r == 0.0) {
		theta = 0.0;
		phi = 0.0;
	    } else {
		if(z == 0.0)
		    phi = M_PI / 2.0;
		else
		    phi = acos(z / rho);
	      
#if WE_DONT_NEED_THIS_CODE
		if(x == 0.0)
		    theta = M_PI / 2.0;	/* asin(y / r); */
		else
		    theta = acos(x / r);
#endif
	      
		if(y == 0.0)
		    theta = M_PI / 2.0;	/* acos(x / r); */
		else
		    theta = asin(y / r) + (M_PI / 2.0);
	    }
	    
	    object->texcoords[2 * i + 0] = theta / M_PI;
	    object->texcoords[2 * i + 1] = phi / M_PI;
	}
	  
	/* go through and put texcoord indices in all the triangles */
	group = object->groups;
	while(group) {
	    for (i = 0; i < group->numtriangles; i++) {
		T(group->triangles[i]).tindices[0] = T(group->triangles[i]).nindices[0];
		T(group->triangles[i]).tindices[1] = T(group->triangles[i]).nindices[1];
		T(group->triangles[i]).tindices[2] = T(group->triangles[i]).nindices[2];
	    }
	    group = group->next;
	}
	  
	object = object->next;
    }
}

/* glmDelete: Deletes a GLMmodel structure.
 *
 * object - initialized GLMmodel structure
 */
GLvoid
glmDelete(GLMmodel* model)
{
    GLMobject* object;
    GLMgroup* group;
    GLuint i;

    assert(model);
  
    object = model->objects;
    while (model->objects){
	object = model->objects;
	model->objects = model->objects->next;
	if (object->name)		  free(object->name);
	if (object->vertices)   free(object->vertices);
	if (object->normals)    free(object->normals);
	if (object->texcoords)  free(object->texcoords);
	if (object->facetnorms) free(object->facetnorms);
	if (object->triangles)  free(object->triangles);
	  
	while(object->groups) {
	    group = object->groups;
	    object->groups = object->groups->next;
	    free(group->name);
	    free(group->triangles);
	    free(group);
	}
	  
	free(object);
    }
  
    if (model->materials) {
	for (i = 0; i < model->nummaterials; i++)
	    free(model->materials[i].name);
	free(model->materials);
    }
  
    if (model->pathname)   free(model->pathname);
    if (model->mtllibname) free(model->mtllibname);
  
    free(model);
}

/* glmReadOBJ: Reads a object description from a Wavefront .OBJ file.
 * Returns a pointer to the created model which should be free'd with
 * glmDelete().
 *
 * filename - name of the file containing the Wavefront .OBJ format data.  
 */
GLMmodel* 
glmReadOBJ(char* filename)
{
    GLMmodel* model;
    GLMobject* object;
    FILE*     file;
    char* locale;

    //backup the current number locale and set it to C
    //This makes sure the .obj file will be read with '.' as floating point separator
    locale = setlocale(LC_NUMERIC, NULL);
    setlocale(LC_NUMERIC, "C");

    /* open the file */
    file = fopen(filename, "r");
    if (!file) {
	if (PRINT_DEBUG)
	    fprintf(stderr, "glmReadOBJ() failed: can't open data file \"%s\".\n", filename);
	exit(1);
    }

    /* allocate a new object */
    model = (GLMmodel*)malloc(sizeof(GLMmodel));
    model->pathname      = stralloc(filename);
    model->mtllibname    = NULL;
    model->nummaterials  = 0;
    model->materials     = NULL;
    model->numobjects     = 0;
    model->objects        = NULL;

    /* make a first pass through the file to get a count of the number
       of vertices, normals, texcoords & triangles */
    _glmFirstPass(model, file);

    /* allocate memory */
    object = model->objects;
    while (object){
	object->vertices = (GLfloat*)malloc(sizeof(GLfloat) *
					    3 * (object->numvertices + 1));
	object->triangles = (GLMtriangle*)malloc(sizeof(GLMtriangle) *
						 object->numtriangles);
	if (object->numnormals) {
	    object->normals = (GLfloat*)malloc(sizeof(GLfloat) *
					       3 * (object->numnormals + 1));
	}
	if (object->numtexcoords) {
	    object->texcoords = (GLfloat*)malloc(sizeof(GLfloat) *
						 2 * (object->numtexcoords + 1));
	}
	  
	object = object->next;
    }

    /* rewind to beginning of file and read in the data this pass */
    rewind(file);
    
    _glmSecondPass(model, file);

    /* close the file */
    fclose(file);

    //reset the locale
    setlocale(LC_NUMERIC, locale);

    return model;
}

/* glmDraw: Renders the object to the current OpenGL context using the
 * mode specified.
 *
 * model    - initialized GLMmodel structure
 * mode     - a bitwise OR of values describing what is to be rendered.
 *            GLM_NONE     -  render with only vertices
 *            GLM_FLAT     -  render with facet normals
 *            GLM_SMOOTH   -  render with vertex normals
 *            GLM_TEXTURE  -  render with texture coords
 *            GLM_COLOR    -  render with colors (color material)
 *            GLM_MATERIAL -  render with materials
 *            GLM_COLOR and GLM_MATERIAL should not both be specified.  
 *            GLM_FLAT and GLM_SMOOTH should not both be specified.  
 */
GLvoid
glmDraw(GLMmodel* model, GLuint mode)
{
    GLuint i;
    GLMobject* object;
    GLMgroup* group;
    GLfloat colBuf[4];

    assert(model);
  
    object = model->objects;
    while (object){
	assert(object->vertices);
	
	/* do a bit of warning */
	if (mode & GLM_FLAT && !object->facetnorms) {
	    if (PRINT_DEBUG)
		printf("glmDraw() warning: flat render mode requested "
		       "with no facet normals defined.\n");
	    mode &= ~GLM_FLAT;
	}
	if (mode & GLM_SMOOTH && !object->normals) {
	    if (PRINT_DEBUG)
		printf("glmDraw() warning: smooth render mode requested "
		       "with no normals defined.\n");
	    mode &= ~GLM_SMOOTH;
	}
	if (mode & GLM_TEXTURE && !object->texcoords) {
	    if (PRINT_DEBUG)
		printf("glmDraw() warning: texture render mode requested "
		       "with no texture coordinates defined.\n");
	    mode &= ~GLM_TEXTURE;
	}
	if (mode & GLM_FLAT && mode & GLM_SMOOTH) {
	    if (PRINT_DEBUG)
		printf("glmDraw() warning: flat render mode requested "
		       "and smooth render mode requested (using smooth).\n");
	    mode &= ~GLM_FLAT;
	}
	if (mode & GLM_COLOR && !model->materials) {
	    if (PRINT_DEBUG)
		printf("glmDraw() warning: color render mode requested "
		       "with no materials defined.\n");
	    mode &= ~GLM_COLOR;
	}
	if (mode & GLM_MATERIAL && !model->materials) {
	    if (PRINT_DEBUG)
		printf("glmDraw() warning: material render mode requested "
		       "with no materials defined.\n");
	    mode &= ~GLM_MATERIAL;
	}
	if (mode & GLM_COLOR && mode & GLM_MATERIAL) {
	    if (PRINT_DEBUG)
		printf("glmDraw() warning: color and material render mode requested "
		       "using only material mode\n");
	    mode &= ~GLM_COLOR;
	}
	if (mode & GLM_COLOR)
	    glEnable(GL_COLOR_MATERIAL);
	if (mode & GLM_MATERIAL)
	    glDisable(GL_COLOR_MATERIAL);
	
	glPushMatrix();
	glTranslatef(object->position[0], object->position[1], object->position[2]);
	
	glBegin(GL_TRIANGLES);
	group = object->groups;
	while (group) {
	    if (mode & GLM_MATERIAL) {
		glMaterialfv(GL_FRONT_AND_BACK, GL_AMBIENT, 
			     model->materials[group->material].ambient);
		glMaterialfv(GL_FRONT_AND_BACK, GL_DIFFUSE, 
			     model->materials[group->material].diffuse);
		glMaterialfv(GL_FRONT_AND_BACK, GL_SPECULAR, 
			     model->materials[group->material].specular);
		glMaterialf(GL_FRONT_AND_BACK, GL_SHININESS, 
			    model->materials[group->material].shininess);
	    }
	
	    if (mode & GLM_COLOR) {
		//glMaterial will control the polygon's specular and emission colours,
		// and the ambient and diffuse will both be set using glColor.
		glColorMaterial(GL_FRONT_AND_BACK, GL_AMBIENT_AND_DIFFUSE);
		
		//set specular and emission to default values
		colBuf[0] = 0;
		colBuf[1] = 0;
		colBuf[2] = 0;
		colBuf[3] = 1;
		glMaterialfv(GL_FRONT_AND_BACK, GL_SPECULAR, colBuf);
		colBuf[0] = 0;
		colBuf[1] = 0;
		colBuf[2] = 0;
		colBuf[3] = 1;
		glMaterialfv(GL_FRONT_AND_BACK, GL_EMISSION, colBuf);
		
		//set the 'real' color
		glColor4fv(model->materials[group->material].diffuse);
	    }
	
	    for (i = 0; i < group->numtriangles; i++) {
		if (mode & GLM_FLAT)
		    glNormal3fv(&object->facetnorms[3 * T(group->triangles[i]).findex]);
	      
		if (mode & GLM_SMOOTH)
		    glNormal3fv(&object->normals[3 * T(group->triangles[i]).nindices[0]]);
		if (mode & GLM_TEXTURE)
		    glTexCoord2fv(&object->texcoords[2*T(group->triangles[i]).tindices[0]]);
		glVertex3fv(&object->vertices[3 * T(group->triangles[i]).vindices[0]]);
		      
		if (mode & GLM_SMOOTH)
		    glNormal3fv(&object->normals[3 * T(group->triangles[i]).nindices[1]]);
		if (mode & GLM_TEXTURE)
		    glTexCoord2fv(&object->texcoords[2*T(group->triangles[i]).tindices[1]]);
		glVertex3fv(&object->vertices[3 * T(group->triangles[i]).vindices[1]]);
		      
		if (mode & GLM_SMOOTH)
		    glNormal3fv(&object->normals[3 * T(group->triangles[i]).nindices[2]]);
		if (mode & GLM_TEXTURE)
		    glTexCoord2fv(&object->texcoords[2*T(group->triangles[i]).tindices[2]]);
		glVertex3fv(&object->vertices[3 * T(group->triangles[i]).vindices[2]]);
		      
	    }
	    
	    group = group->next;
	}
	glEnd();
	
	glPopMatrix();
	  
	object = object->next;
    }
}

/* glmList: Generates and returns a display list for the object using
 * the mode specified.
 *
 * model    - initialized GLMmodel structure
 * mode     - a bitwise OR of values describing what is to be rendered.
 *            GLM_NONE     -  render with only vertices
 *            GLM_FLAT     -  render with facet normals
 *            GLM_SMOOTH   -  render with vertex normals
 *            GLM_TEXTURE  -  render with texture coords
 *            GLM_COLOR    -  render with colors (color material)
 *            GLM_MATERIAL -  render with materials
 *            GLM_COLOR and GLM_MATERIAL should not both be specified.  
 *            GLM_FLAT and GLM_SMOOTH should not both be specified.  
 */
GLuint
glmList(GLMmodel* model, GLuint mode)
{
    GLuint list;
  
    list = glGenLists(1);
    glNewList(list, GL_COMPILE);
    glmDraw(model, mode);
    glEndList();

    return list;
}

/* glmWeld: eliminate (weld) vectors that are within an epsilon of
 * each other.
 *
 * model      - initialized GLMmodel structure
 * epsilon    - maximum difference between vertices
 *              ( 0.00001 is a good start for a unitized object)
 *
 */
GLvoid
glmWeld(GLMmodel* model, GLfloat epsilon)
{
    GLMobject* object;
    GLfloat* vectors;
    GLfloat* copies;
    GLuint   numvectors;
    GLuint   i;
  
    object = model->objects;
    while (object){
	/* vertices */
	numvectors = object->numvertices;
	vectors    = object->vertices;
	copies = _glmWeldVectors(vectors, &numvectors, epsilon);
	
	if (PRINT_DEBUG)
	    printf("glmWeld(): %d redundant vertices.\n", 
		   object->numvertices - numvectors - 1);
	
	for (i = 0; i < object->numtriangles; i++) {
	    T(i).vindices[0] = (GLuint)vectors[3 * T(i).vindices[0] + 0];
	    T(i).vindices[1] = (GLuint)vectors[3 * T(i).vindices[1] + 0];
	    T(i).vindices[2] = (GLuint)vectors[3 * T(i).vindices[2] + 0];
	}
	
	/* free space for old vertices */
	free(vectors);
	
	/* allocate space for the new vertices */
	object->numvertices = numvectors;
	object->vertices = (GLfloat*)malloc(sizeof(GLfloat) * 
					    3 * (object->numvertices + 1));
	
	/* copy the optimized vertices into the actual vertex list */
	for (i = 1; i <= object->numvertices; i++) {
	    object->vertices[3 * i + 0] = copies[3 * i + 0];
	    object->vertices[3 * i + 1] = copies[3 * i + 1];
	    object->vertices[3 * i + 2] = copies[3 * i + 2];
	}
	
	free(copies);
	  
	object = object->next;
    }
}
