/* iksemel (XML parser for Jabber)
** Copyright (C) 2000-2003 Gurer Ozen <madcat@e-kolay.net>
** This code is free software; you can redistribute it and/or
** modify it under the terms of GNU Lesser General Public License.
*/

#include "common.h"
#include "iksemel.h"

#ifdef HAVE_GETOPT_LONG
#include <getopt.h>
#endif

#ifdef HAVE_GETOPT_LONG
static struct option longopts[] = {
	{ "backup", required_argument, 0, 'b' },
	{ "restore", required_argument, 0, 'r' },
	{ "file", required_argument, 0, 'f' },
	{ "timeout", required_argument, 0, 't' },
	{ "help", 0, 0, 'h' },
	{ "version", 0, 0, 'V' },
	{ 0, 0, 0, 0 }
};
#endif

static char *shortopts = "b:r:f:t:hV";

static void
print_usage (void)
{
	puts ("Usage: iksroster [OPTIONS]\n"
		"This is a backup tool for your jabber roster.\n"
		" -b, --backup=JID    Download roster from the server.\n"
		" -r, --restore=JID   Upload roster to the server.\n"
		" -f, --file=FILE     Load/Save roster to this file.\n"
		" -t, --timeout=SECS  Set network timeout.\n"
		" -h, --help          Print this text and exit.\n"
		" -V, --version       Print version and exit.\n"
#ifndef HAVE_GETOPT_LONG
		"(long options are not supported on your system)\n"
#endif
		"Report bugs to <iksemel-dev@jabberstudio.org>.");
}

struct {
	iksparser *prs;
	iksfilter *f;
	iksid *from;
	iksid *to;
	char *file;
	int op;
	int timeout;
	int counter;
	int state;
	char from_pw[128];
	char to_pw[128];
	iks *x;
} roster;

void
j_error (char *msg)
{
	fprintf (stderr, "iksroster: %s\n", msg);
	exit (2);
}

int
stream_hook (void *user_data, int type, iks *node)
{
	roster.counter = roster.timeout;
	switch (type) {
		case IKS_NODE_START:
		{
			iks *x;

			if (roster.op)
				x = iks_make_auth (roster.to, roster.to_pw, iks_find_attrib (node, "id"));
			else
				x = iks_make_auth (roster.from, roster.from_pw, iks_find_attrib (node, "id"));
			iks_insert_attrib (x, "id", "auth");
			iks_send (roster.prs, x);
			iks_delete (x);
			break;
		}
		case IKS_NODE_NORMAL:
		{
			ikspak *pak;

			pak = iks_packet (node);
			iks_filter_packet (roster.f, pak);
			if (roster.state == 1) return IKS_HOOK;
			break;
		}
		case IKS_NODE_STOP:
			j_error ("server disconnected");
		case IKS_NODE_ERROR:
			j_error ("stream error");
	}
	if (node) iks_delete (node);
	return IKS_OK;
}

int
on_result (void *user_data, ikspak *pak)
{
	iks *x;

	if (roster.op == 0) {
		x = iks_make_iq (IKS_TYPE_GET, IKS_NS_ROSTER);
		iks_insert_attrib (x, "id", "roster");
		iks_send (roster.prs, x);
		iks_delete (x);
	} else {
		iks_insert_attrib (roster.x, "type", "set");
		iks_send (roster.prs, roster.x);
	}
	return IKS_FILTER_EAT;
}

int
on_error (void *user_data, ikspak *pak)
{
	j_error ("authorization failed");
	return IKS_FILTER_EAT;
}

int
on_roster (void *user_data, ikspak *pak)
{
	roster.x = pak->x;
	roster.state = 1;

	return IKS_FILTER_EAT;
}

void
on_log (void *user_data, const char *data, size_t size, int is_incoming)
{
	if (is_incoming) printf ("[%s]\n",data);
}

void
j_setup (void)
{
	roster.prs = iks_stream_new (IKS_NS_CLIENT, NULL, stream_hook);
//	iks_set_log_hook (roster.prs, on_log);
	roster.f = iks_filter_new ();
	iks_filter_add_rule (roster.f, on_result, NULL,
		IKS_RULE_TYPE, IKS_PAK_IQ,
		IKS_RULE_SUBTYPE, IKS_TYPE_RESULT,
		IKS_RULE_ID, "auth",
		IKS_RULE_DONE);
	iks_filter_add_rule (roster.f, on_error, NULL,
		IKS_RULE_TYPE, IKS_PAK_IQ,
		IKS_RULE_SUBTYPE, IKS_TYPE_ERROR,
		IKS_RULE_ID, "auth",
		IKS_RULE_DONE);
	iks_filter_add_rule (roster.f, on_roster, NULL,
		IKS_RULE_TYPE, IKS_PAK_IQ,
		IKS_RULE_SUBTYPE, IKS_TYPE_RESULT,
		IKS_RULE_ID, "roster",
		IKS_RULE_DONE);
}

void
j_connect (void)
{
	int ret;

	roster.state = 0;

	if (roster.op)
		ret = iks_connect_tcp (roster.prs, roster.to->server, IKS_JABBER_PORT);
	else
		ret = iks_connect_tcp (roster.prs, roster.from->server, IKS_JABBER_PORT);

	switch (ret) {
		case IKS_OK:
			break;
		case IKS_NET_NODNS:
			j_error ("hostname lookup failed");
		case IKS_NET_NOCONN:
			j_error ("connection failed");
		default:
			j_error ("io error");
	}

	roster.counter = roster.timeout;
	while (1) {
		ret = iks_recv (roster.prs, 1);
		if (IKS_HOOK == ret) break;
		if (IKS_OK != ret) j_error ("io error");
		roster.counter--;
		if (roster.counter == 0) j_error ("network timeout");
	}
	iks_disconnect (roster.prs);
}

int
main (int argc, char *argv[])
{
	ikstack *s;
	int c;
#ifdef HAVE_GETOPT_LONG
	int i;
#endif

	s = iks_stack_new (1024);
	memset (&roster, 0, sizeof (roster));
	roster.timeout = 30;
#ifdef HAVE_GETOPT_LONG
	while ((c = getopt_long (argc, argv, shortopts, longopts, &i)) != -1) {
#else
	while ((c = getopt (argc, argv, shortopts)) != -1) {
#endif
		switch (c) {
			case 'b':
				roster.from = iks_id_new (s, optarg);
				printf ("Password for %s: ", optarg);
				fflush (stdout);
				fgets (roster.from_pw, 127, stdin);
				strtok (roster.from_pw, "\r\n");
				break;
			case 'r':
				roster.to = iks_id_new (s, optarg);
				printf ("Password for %s: ", optarg);
				fflush (stdout);
				fgets (roster.to_pw, 127, stdin);
				strtok (roster.to_pw, "\r\n");
				break;
			case 'f':
				roster.file = strdup (optarg);
				break;
			case 't':
				roster.timeout = atoi (optarg);
				if (roster.timeout < 10) roster.timeout = 10;
				break;
			case 'h':
				print_usage ();
				exit (0);
			case 'V':
				puts ("iksroster (iksemel) "VERSION);
				exit (0);
		}
	}
	if (roster.from == NULL && roster.to == NULL) {
		puts ("What I'm supposed to do?");
		print_usage ();
		exit (1);
	}
	if (roster.to && (roster.from == NULL && roster.file == NULL)) {
		puts ("Store which roster?");
		print_usage ();
		exit (1);
	}

	j_setup ();

	if (roster.from) {
		j_connect ();
		if (roster.file) {
			switch (iks_save (roster.file, roster.x)) {
				case IKS_OK:
					break;
				case IKS_FILE_NOACCESS:
					j_error ("cannot write to file");
				default:
					j_error ("file io error");
			}
		}
	} else {
		switch (iks_load (roster.file, &roster.x)) {
			case IKS_OK:
				break;
			case IKS_FILE_NOFILE:
				j_error ("file not found");
			case IKS_FILE_NOACCESS:
				j_error ("cannot read file");
			default:
				j_error ("file io error");
		}
	}
	if (roster.to) {
		roster.op = 1;
		j_connect ();
	}

	return 0;
}
