/*****************************************************************************************************************************************
 ctoolbar.c
*****************************************************************************************************************************************/

#include "ctoolbar.h"
#include "cimage.h"

//---------------------------------------------------------------------------------------------------------------------------------------
// metaclass
//---------------------------------------------------------------------------------------------------------------------------------------
RESOLVE_DYNAMIC_METACLASS (CToolButton);

//---------------------------------------------------------------------------------------------------------------------------------------
// OnClick
//---------------------------------------------------------------------------------------------------------------------------------------
void CToolButton::OnClick (GtkToolButton *, gpointer inData)
{
	// retreive the gtkol instance
	CToolButton *theToolButton = reinterpret_cast <CToolButton *> (inData);

	// send the notification if any listener is available
	if (theToolButton != NULL)
	{
		if (theToolButton -> GetMenuItem() != NULL && theToolButton -> GetMenuItem() != NULL)
			theToolButton -> GetMenuItem() -> Click();
		else if (theToolButton -> GetListener() != NULL)
			static_cast <CButtonListener *> (theToolButton -> GetListener()) -> OnClick (theToolButton);
	}
}

//---------------------------------------------------------------------------------------------------------------------------------------
// constructor
//---------------------------------------------------------------------------------------------------------------------------------------
CToolButton::CToolButton   	(CToolBar *inOwner, const CString &inStockId, const CMenuItem *inMenuItem, 
				 const CButtonListener *inListener)
	    :CButton	   	(NULL, inListener),
	     m_StockId	   	(inStockId),
	     m_GtkToolTips 	(NULL),
	     m_CaptionLock	(false),
	     m_ToolTipsText	(),
	     m_ToolTipsPrivate	(),
	     m_MenuItem		(const_cast <CMenuItem *> (inMenuItem))
{
	// instanciation requested
	if (inOwner != NULL) SetOwner (inOwner);
}

//---------------------------------------------------------------------------------------------------------------------------------------
// destructor
//---------------------------------------------------------------------------------------------------------------------------------------
CToolButton::~CToolButton ()
{
	// delete the tooltips instance
	if (m_GtkToolTips != NULL) ::gtk_object_destroy (GTK_OBJECT(m_GtkToolTips));

	// deletion coherence requested
	CWidget::DestroyWidget (this);
}

//---------------------------------------------------------------------------------------------------------------------------------------
// specific gtk widget instanciation
//---------------------------------------------------------------------------------------------------------------------------------------
GtkWidget * CToolButton::PerformWidgetInstanciate ()
{
	// filled in stock id ?
	if (m_StockId != CString())

		// stock id instance
		return GTK_WIDGET(::gtk_tool_button_new_from_stock (m_StockId.Get()));

	// no stock id specification, default instance to be personalized
	return GTK_WIDGET(::gtk_tool_button_new (NULL, NULL));
}

//---------------------------------------------------------------------------------------------------------------------------------------
// gtk widget initialization
//---------------------------------------------------------------------------------------------------------------------------------------
void CToolButton::PerformWidgetInitialize ()
{
	// our gtk widget
	GtkWidget *inGtkWidget = GetGtkWidget ();

	// check pointer
	if (inGtkWidget == NULL) return;

	// specific signal connection (overwrite the generic definition because of gtk listener pointer type modification and potential
	// menu item event redirection)
	::g_signal_handlers_disconnect_by_func (G_OBJECT(inGtkWidget), (void*)G_CALLBACK(CButton::OnClick), this);
	::g_signal_connect (G_OBJECT(inGtkWidget), "clicked", G_CALLBACK(CToolButton::OnClick), this);

	// default caption
	if (m_StockId == CString()) SetCaption ("id"+CString(GetId()));

	// show the tool item
	Show ();
}

//---------------------------------------------------------------------------------------------------------------------------------------
// container add request
//---------------------------------------------------------------------------------------------------------------------------------------
void CToolButton::PerformContainerAdd (CWidget *inChild)
{
	// child instance check
	if (inChild && inChild -> ClassIs (__metaclass(CImage)))
	{
		// retreive our gtk instance
		GtkWidget *inGtkWidget (GetGtkWidget());

		// pointer check
		if (inGtkWidget == NULL) return;

		// set it
		::gtk_tool_button_set_icon_widget (GTK_TOOL_BUTTON(inGtkWidget), inChild -> GetGtkWidget());
	}
	// generic call...
	else CButton::PerformContainerAdd (inChild);
}

//---------------------------------------------------------------------------------------------------------------------------------------
// container remove request
//---------------------------------------------------------------------------------------------------------------------------------------
void CToolButton::PerformContainerRemove (CWidget *inChild)
{
	// child instance check
	if (inChild && inChild -> ClassIs (__metaclass(CImage)))
	{
		// retreive our gtk instance
		GtkWidget *inGtkWidget (GetGtkWidget());

		// pointer check
		if (inGtkWidget == NULL) return;

		// set it
		::gtk_tool_button_set_icon_widget (GTK_TOOL_BUTTON(inGtkWidget), NULL);
	}
	// generic call...
	else CButton::PerformContainerRemove (inChild);
}

//---------------------------------------------------------------------------------------------------------------------------------------
// expected owner type
//---------------------------------------------------------------------------------------------------------------------------------------
CMetaClasses CToolButton::OwnerMustBe () const
{
	return __metaclasses(CToolBar);
}

//---------------------------------------------------------------------------------------------------------------------------------------
// draggable state of the tool button
//---------------------------------------------------------------------------------------------------------------------------------------
void CToolButton::SetDraggable (const bool inDraggable)
{
	// generic call first
	CButton::SetDraggable (inDraggable);

	// get our widget
	GtkWidget *inGtkWidget (GetGtkWidget());

	// specific property affectation
	::gtk_tool_item_set_use_drag_window (GTK_TOOL_ITEM(inGtkWidget), inDraggable);
}

//---------------------------------------------------------------------------------------------------------------------------------------
// caption writer
//---------------------------------------------------------------------------------------------------------------------------------------
void CToolButton::SetCaption (const CString &inCaption)
{
	// from serialization, do we handle caption affectation
	if (m_CaptionLock) return;

	// our widget
	GtkWidget *inGtkWidget = GetGtkWidget();

	// gtk affectation
	if (inGtkWidget != NULL) ::gtk_tool_button_set_label (GTK_TOOL_BUTTON(inGtkWidget), inCaption.Get());
}

//---------------------------------------------------------------------------------------------------------------------------------------
// caption reader
//---------------------------------------------------------------------------------------------------------------------------------------
CString CToolButton::GetCaption () const
{
	// our widget
	GtkWidget *inGtkWidget = GetGtkWidget();

	// ok
	return inGtkWidget == NULL ? CString() : CString(::gtk_tool_button_get_label (GTK_TOOL_BUTTON(inGtkWidget)));
}

//---------------------------------------------------------------------------------------------------------------------------------------
// click
//---------------------------------------------------------------------------------------------------------------------------------------
void CToolButton::Click ()
{
	// get the instance's listener if any
	if (m_Listener != NULL)
		static_cast <CButtonListener *> (m_Listener) -> OnClick (this);
}

//---------------------------------------------------------------------------------------------------------------------------------------
// set the toolbutton tips
//---------------------------------------------------------------------------------------------------------------------------------------
void CToolButton::SetToolTips (const CString &inText, const CString &inPrivate)
{
	// our widget
	GtkWidget *inGtkWidget = GetGtkWidget();

	// tooltips affectation
	if (inGtkWidget != NULL && (inText != CString() || inPrivate != CString())) 
	{
		if (m_GtkToolTips == NULL) m_GtkToolTips = ::gtk_tooltips_new ();
		::gtk_tool_item_set_tooltip (GTK_TOOL_ITEM(inGtkWidget), m_GtkToolTips, inText.Get(), inPrivate.Get());
	}
	else
	{
		if (m_GtkToolTips != NULL) ::gtk_object_destroy (GTK_OBJECT(m_GtkToolTips));
		m_GtkToolTips = NULL;
	}

	// local copy of the attributes
	m_ToolTipsText 	  = inText;
	m_ToolTipsPrivate = inPrivate;
}

//---------------------------------------------------------------------------------------------------------------------------------------
// get tooltips text
//---------------------------------------------------------------------------------------------------------------------------------------
CString CToolButton::GetToolTipsText () const
{
	return m_ToolTipsText;
}

//---------------------------------------------------------------------------------------------------------------------------------------
// get tooltips private
//---------------------------------------------------------------------------------------------------------------------------------------
CString CToolButton::GetToolTipsPrivate () const
{
	return m_ToolTipsPrivate;
}

//---------------------------------------------------------------------------------------------------------------------------------------
// menu item affectation
//---------------------------------------------------------------------------------------------------------------------------------------
void CToolButton::SetMenuItem (const CMenuItem *inMenuItem)
{
	m_MenuItem = const_cast <CMenuItem *> (inMenuItem);
}

//---------------------------------------------------------------------------------------------------------------------------------------
// menu item reader
//---------------------------------------------------------------------------------------------------------------------------------------
CMenuItem * CToolButton::GetMenuItem () const
{
	return const_cast <CMenuItem *> (m_MenuItem);
}

//---------------------------------------------------------------------------------------------------------------------------------------
// xml serialization
//---------------------------------------------------------------------------------------------------------------------------------------
void CToolButton::Serialize (CXMLElementNode *&ioXMLElementNode, const int inMode) THROWABLE
{
	// serialization request analyse
	switch (inMode)
	{
		// xml read
		case XML_READ :
		{
			// we first have to find the ctoolbutton xml node because it might instanciate a specific stock tool item; so put
			// on our make up for the PerformWidgetInstanciate implicit handling...
			CXMLElementNode *inXMLNode = ::xml_node_search (ioXMLElementNode, XML_TOOLBUTTON_ELEMENT);

			// check we got an expected ctoolbutton node
                        if (inXMLNode == NULL)
                                throw new CException (CString("CToolButton::Serialize, specified xml node is not a \"") +
                                                              XML_TOOLBUTTON_ELEMENT + CString("\" element one."), __exception(XMLPARSE));

			// get the stock id attribute if any
			m_StockId = ::xml_node_get_attribute (inXMLNode, XML_TOOLBUTTON_ATTR_STOCKID).GetValue();

			// now get the xml button element and see if potential default caption has been overwritten
			CXMLElementNode *inXMLNodeButton = ::xml_node_search (ioXMLElementNode, XML_BUTTON_ELEMENT);

			// check we got an expected ctoolbutton node
                        if (inXMLNodeButton == NULL)
                                throw new CException (CString("CToolButton::Serialize, specified xml node is not a \"") +
                                                              XML_BUTTON_ELEMENT + CString("\" element one."), __exception(XMLPARSE));

			// get the caption if any
			CString inCaption (::xml_node_get_attribute (inXMLNodeButton, XML_BUTTON_ATTR_CAPTION).GetValue());

			// so, now see if the SetCaption call will lock or not
			if (m_StockId != CString() && inCaption == CString()) m_CaptionLock = true;

			// generic call, at this point the m_StockId as been loaded, so the gtk widget instanciator will generate a stock
			// tool item if any and lock or not the caption affectation...
			CButton::Serialize (ioXMLElementNode, inMode);

			// do not lock caption affectation anymore...
			m_CaptionLock = false;

			// set the tooltips properties
			SetToolTips (::xml_node_get_attribute (inXMLNode, XML_TOOLBUTTON_ATTR_TOOLTIPS_TEXT   ).GetValue(),
				     ::xml_node_get_attribute (inXMLNode, XML_TOOLBUTTON_ATTR_TOOLTIPS_PRIVATE).GetValue());

			// get the potential associated menu item
			CString inMenuItemCaption (::xml_node_get_attribute (inXMLNode, XML_TOOLBUTTON_ATTR_MENUITEM).GetValue());

			// check the specified menu item caption
			if (inMenuItemCaption != CString())
			{
				// get the actual instanciated menu items...
				CComponents inMenuItems (CComponent::GetComponents (__metaclass(CMenuItem)));

				// go through the instance an try to retreive the specified one
				for (size_t i=inMenuItems.GetLength(), j=0; i>0; i--, j++)
				{
					// retreive the menu item instance
					CMenuItem *inMenuItem = static_cast <CMenuItem *> (*inMenuItems[j]);

					// check the caption
					if (inMenuItem -> GetCaption() == inMenuItemCaption)
					{
						// set the local attribute
						m_MenuItem = inMenuItem;

						// stop the loop
						break;
					}
				}
			}

			// modify the io xml node
			ioXMLElementNode = inXMLNode;
		}
		break;

		// xml dump
		case XML_WRITE :
		{
			// generic call first
			CButton::Serialize (ioXMLElementNode, inMode);

			// instanciate a new xml element under the current one
			CXMLElement *newXMLElement = new CXMLElement (ioXMLElementNode, XML_TOOLBUTTON_ELEMENT);

			// add the attributes
			newXMLElement -> AddAttribute (XML_TOOLBUTTON_ATTR_STOCKID, 	     m_StockId);
			newXMLElement -> AddAttribute (XML_TOOLBUTTON_ATTR_TOOLTIPS_TEXT,    m_ToolTipsText);
			newXMLElement -> AddAttribute (XML_TOOLBUTTON_ATTR_TOOLTIPS_PRIVATE, m_ToolTipsPrivate);
			newXMLElement -> AddAttribute (XML_TOOLBUTTON_ATTR_MENUITEM,	     m_MenuItem != NULL ? 
											     m_MenuItem-> GetCaption() : CString());

			// modify the input output pointer so that the overwritten serialization process continues under this element
			ioXMLElementNode = newXMLElement -> GetXMLElementNode ();
		}
		break;
	}
}

//---------------------------------------------------------------------------------------------------------------------------------------
// metaclass
//---------------------------------------------------------------------------------------------------------------------------------------
RESOLVE_DYNAMIC_METACLASS (CToolToggleButton);

//---------------------------------------------------------------------------------------------------------------------------------------
// OnClick
//---------------------------------------------------------------------------------------------------------------------------------------
void CToolToggleButton::OnClick	(GtkToggleToolButton *, gpointer inData)
{
	// retreive the gtkol instance
	CToolToggleButton *inToolToggleButton = reinterpret_cast <CToolToggleButton *> (inData);

	// send the notification if any listener is available
	if (inToolToggleButton != NULL)
	{
		if (inToolToggleButton -> GetMenuItem() != NULL && inToolToggleButton -> GetMenuItem() != NULL)
			inToolToggleButton -> GetMenuItem() -> Click();
		else if (inToolToggleButton -> GetListener() != NULL)
			static_cast <CButtonListener *> (inToolToggleButton -> GetListener()) -> OnClick (inToolToggleButton);
	}
}

//---------------------------------------------------------------------------------------------------------------------------------------
// constructor
//---------------------------------------------------------------------------------------------------------------------------------------
CToolToggleButton::CToolToggleButton (CToolBar *inOwner, const CString &inStockId, const CMenuItem *inMenuItem, 
				      const CButtonListener *inListener)
		  :CToolButton	     (NULL, inStockId, inMenuItem, inListener)
{
	// instanciation requested
	if (inOwner != NULL) SetOwner (inOwner);
}

//--------------------------------------------------------------------------------------------------------------------------------------
// destructor
//---------------------------------------------------------------------------------------------------------------------------------------
CToolToggleButton::~CToolToggleButton ()
{
	// deletion coherence requested
	CWidget::DestroyWidget (this);
}

//---------------------------------------------------------------------------------------------------------------------------------------
// gtk widget instanciation
//---------------------------------------------------------------------------------------------------------------------------------------
GtkWidget * CToolToggleButton::PerformWidgetInstanciate ()
{
	// filled in stock id ?
	if (m_StockId != CString())

		// stock id instance
		return GTK_WIDGET(::gtk_toggle_tool_button_new_from_stock (m_StockId.Get()));

	// no stock id specification, default instance to be personalized
	return GTK_WIDGET(::gtk_toggle_tool_button_new());
}

//---------------------------------------------------------------------------------------------------------------------------------------
// gtk widget initialization
//---------------------------------------------------------------------------------------------------------------------------------------
void CToolToggleButton::PerformWidgetInitialize ()
{
	// generic call first
	CToolButton::PerformWidgetInitialize ();

	// our gtk widget
	GtkWidget *inGtkWidget = GetGtkWidget ();

	// pointer check
	if (inGtkWidget == NULL) return;

	// disconnect the default click function
	::g_signal_handlers_disconnect_by_func (G_OBJECT(inGtkWidget), (void*)G_CALLBACK(CToolButton::OnClick), this);

	// connect the toggle signal
	::g_signal_connect (G_OBJECT(inGtkWidget), "toggled", G_CALLBACK(CToolToggleButton::OnClick), this);
}

//---------------------------------------------------------------------------------------------------------------------------------------
// click
//---------------------------------------------------------------------------------------------------------------------------------------
void CToolToggleButton::Click ()
{
	// get our widget
	GtkWidget *inGtkWidget (GetGtkWidget());

	// pointer check
	if (inGtkWidget == NULL) return;

	// get the check status
	if (IsChecked()) Check (false); else Check (true);

	// generic call, signal notification
	CToolButton::Click ();
}

//---------------------------------------------------------------------------------------------------------------------------------------
// check
//---------------------------------------------------------------------------------------------------------------------------------------
void CToolToggleButton::Check (const bool inCheck)
{
	// get our widget
	GtkWidget *inGtkWidget (GetGtkWidget());

	// pointer check
	if (inGtkWidget == NULL) return;

	// disconnect the default click function
	::g_signal_handlers_disconnect_by_func (G_OBJECT(inGtkWidget), (void*)G_CALLBACK(CToolToggleButton::OnClick), this);

	// property affectation
	::gtk_toggle_tool_button_set_active (GTK_TOGGLE_TOOL_BUTTON(inGtkWidget), inCheck);

	// connect the toggle signal
	::g_signal_connect (G_OBJECT(inGtkWidget), "toggled", G_CALLBACK(CToolToggleButton::OnClick), this);

	// check menu item associated ?
	if (m_MenuItem != NULL && m_MenuItem -> ClassIs (__metaclass(CMenuItemCheck))) 
		static_cast <CMenuItemCheck *> (m_MenuItem) -> Check (inCheck);

}

//---------------------------------------------------------------------------------------------------------------------------------------
// is checked ?
//---------------------------------------------------------------------------------------------------------------------------------------
bool CToolToggleButton::IsChecked () const
{
	// get our widget
	GtkWidget *inGtkWidget (GetGtkWidget());

	// ok ?
	return inGtkWidget == NULL ? false : ::gtk_toggle_tool_button_get_active (GTK_TOGGLE_TOOL_BUTTON(inGtkWidget));
}

//---------------------------------------------------------------------------------------------------------------------------------------
// tool toggle button serialization
//---------------------------------------------------------------------------------------------------------------------------------------
void CToolToggleButton::Serialize (CXMLElementNode *&ioXMLElementNode, const int inMode) THROWABLE
{
	// generic call
	CToolButton::Serialize (ioXMLElementNode, inMode);

	// request analyse
	switch (inMode)
	{
		// xml dump
		case XML_WRITE :
		{
			// instanciate a new xml element under the current one
			CXMLElement *newXMLElement = new CXMLElement (ioXMLElementNode, XML_TOOLTOGGLEBUTTON_ELEMENT);

			// add the attribute
			newXMLElement -> AddAttribute (XML_TOOLTOGGLEBUTTON_ATTR_CHECKED, CString(IsChecked()?"true":"false"));

			// modify the input output pointer so that the overwritten serialization process continues under this element
			ioXMLElementNode = newXMLElement -> GetXMLElementNode ();
		}
		break;

		// xml load
		case XML_READ :
		{
			// get the xml element node
			CXMLElementNode *inXMLNode = ::xml_node_get_child (ioXMLElementNode, XML_TOOLTOGGLEBUTTON_ELEMENT);

			// check we got the expected node
                        if (inXMLNode == NULL)
                                throw new CException (CString("CToolToggleButton::Serialize, specified xml node is not a \"") +
                                                              XML_TOOLTOGGLEBUTTON_ELEMENT + CString("\" element one."),
							      __exception(XMLPARSE));

			// io xml element node modification
			ioXMLElementNode = inXMLNode;

			// check property affectation
			Check (::xml_node_get_attribute (inXMLNode, XML_TOOLTOGGLEBUTTON_ATTR_CHECKED).GetValue().ToBool());
		}
		break;
	}
}

//---------------------------------------------------------------------------------------------------------------------------------------
// metaclass
//---------------------------------------------------------------------------------------------------------------------------------------
RESOLVE_DYNAMIC_METACLASS (CToolRadioButton);

//---------------------------------------------------------------------------------------------------------------------------------------
// constructor
//---------------------------------------------------------------------------------------------------------------------------------------
CToolRadioButton::CToolRadioButton  (CToolBar *inOwner, const CString &inStockId, const CMenuItem *inMenuItem, 
				     const CButtonListener *inListener)
		 :CToolToggleButton (NULL, inStockId, inMenuItem, inListener)
{
	// instanciation requested
	if (inOwner != NULL) SetOwner (inOwner);
}

//---------------------------------------------------------------------------------------------------------------------------------------
// destructor
//---------------------------------------------------------------------------------------------------------------------------------------
CToolRadioButton::~CToolRadioButton ()
{
	// deletion coherence requested
	CWidget::DestroyWidget (this);
}

//---------------------------------------------------------------------------------------------------------------------------------------
// gtk widget instanciation
//---------------------------------------------------------------------------------------------------------------------------------------
GtkWidget * CToolRadioButton::PerformWidgetInstanciate ()
{
	// filled in stock id ?
	if (m_StockId != CString())

		// stock id instance
		return GTK_WIDGET(::gtk_radio_tool_button_new_from_stock (NULL, m_StockId.Get()));

	// no stock id specification, default instance to be personalized
	return GTK_WIDGET(::gtk_radio_tool_button_new (NULL));
}

//---------------------------------------------------------------------------------------------------------------------------------------
// check
//---------------------------------------------------------------------------------------------------------------------------------------
void CToolRadioButton::Check (const bool inToggle) 
{
	// check gtk pointer
	if (GetGtkWidget() == NULL) return;

	// get our siblings
	CComponents inToolRadioButtons (GetSiblings (__metaclass(CToolRadioButton)) - this);

	// all the click signals are disconnected
	for (size_t i=0; i<inToolRadioButtons.GetLength(); i++)
		::g_signal_handlers_disconnect_by_func 
			(G_OBJECT(static_cast <CToolRadioButton *> (*inToolRadioButtons[i]) -> GetGtkWidget()), 
				(void*)G_CALLBACK(CToolToggleButton::OnClick), this);

	// generic call (singal auto ignition)
	CToolToggleButton::Check (inToggle);

	// all signals are connected again
	for (size_t i=0; i<inToolRadioButtons.GetLength(); i++)
		::g_signal_connect 
			(G_OBJECT(static_cast <CToolRadioButton *> (*inToolRadioButtons[i]) -> GetGtkWidget()), 
			 	"toggled", G_CALLBACK(CToolToggleButton::OnClick), this);

	// radio menu item associated ?
	if (m_MenuItem != NULL && m_MenuItem -> ClassIs (__metaclass(CMenuItemRadio))) 
		static_cast <CMenuItemRadio *> (m_MenuItem) -> Check (inToggle);
}

//---------------------------------------------------------------------------------------------------------------------------------------
// set owner
//---------------------------------------------------------------------------------------------------------------------------------------
bool CToolRadioButton::SetOwner (CComponent *inOwner, const SInt16 inIndex)
{
	// generic call
	if (!CToolToggleButton::SetOwner (inOwner, inIndex)) return false;

	// get our widget
	GtkWidget *inGtkWidget (GetGtkWidget());

	// pointer check
	if (inGtkWidget == NULL) return false;

	// the potential gtk radio group this radio item should be added to
	GSList *inToolItemRadioGroup = NULL;

	// get the siblings tool items
	CComponents inToolItems (GetSiblings (__metaclass(CToolButton)));

	// go through the siblings
	for (size_t i=inToolItems.GetLength(), j=0; i>0; i--, j++)
	{
		// is it our gtkol instance ?
		if ((*inToolItems[j]) == this) break;

		// get the gtk group of the item if it is radio one
		if ((*inToolItems[j]) -> ClassIs (__metaclass(CToolRadioButton)))
			inToolItemRadioGroup = ::gtk_radio_tool_button_get_group 
				(GTK_RADIO_TOOL_BUTTON (static_cast <CToolRadioButton *> (*inToolItems[j]) -> GetGtkWidget()));

		// if the item is a logical items separator, consider a new item radio group and do not use the previous one
		if ((*inToolItems[j]) -> ClassIs (__metaclass(CToolSeparatorButton))) inToolItemRadioGroup = NULL;
	}

	// group affectation
	::gtk_radio_tool_button_set_group (GTK_RADIO_TOOL_BUTTON(inGtkWidget), inToolItemRadioGroup);

	// ok
	return true;
}

//---------------------------------------------------------------------------------------------------------------------------------------
// metaclass
//---------------------------------------------------------------------------------------------------------------------------------------
RESOLVE_DYNAMIC_METACLASS (CToolSeparatorButton);

//---------------------------------------------------------------------------------------------------------------------------------------
// constructor
//---------------------------------------------------------------------------------------------------------------------------------------
CToolSeparatorButton::CToolSeparatorButton (CToolBar *inOwner, const CButtonListener *inListener)
		     :CToolButton	   (NULL, CString(), NULL, inListener)
{
	// instanciation requested
	if (inOwner != NULL) SetOwner (inOwner);
}

//---------------------------------------------------------------------------------------------------------------------------------------
// destructor
//---------------------------------------------------------------------------------------------------------------------------------------
CToolSeparatorButton::~CToolSeparatorButton ()
{
	// deletion coherence requested
	CWidget::DestroyWidget (this);
}

//---------------------------------------------------------------------------------------------------------------------------------------
// gtk widget instanciation
//---------------------------------------------------------------------------------------------------------------------------------------
GtkWidget * CToolSeparatorButton::PerformWidgetInstanciate ()
{
	// ok, cast the tool item
	return GTK_WIDGET(::gtk_separator_tool_item_new());
}

//---------------------------------------------------------------------------------------------------------------------------------------
// gtk widget initialization
//---------------------------------------------------------------------------------------------------------------------------------------
void CToolSeparatorButton::PerformWidgetInitialize ()
{
	// show the tool item separator
	Show ();
}

//---------------------------------------------------------------------------------------------------------------------------------------
// xml serialization
//---------------------------------------------------------------------------------------------------------------------------------------
void CToolSeparatorButton::Serialize (CXMLElementNode *&ioXMLElementNode, const int inMode) THROWABLE
{
	// no cbutton nor ctoolbutton serialization needed...
	CContainer::Serialize (ioXMLElementNode, inMode);
}

//---------------------------------------------------------------------------------------------------------------------------------------
// metaclass
//---------------------------------------------------------------------------------------------------------------------------------------
RESOLVE_DYNAMIC_METACLASS (CToolBar);

//---------------------------------------------------------------------------------------------------------------------------------------
// constructor
//---------------------------------------------------------------------------------------------------------------------------------------
CToolBar::CToolBar (CContainer *inOwner, const CWidgetListener *inListener)
	 :CLayout  (inOwner, inListener)
{
	// instanciation requested
	if (inOwner != NULL) CWidget::CreateWidget (this);
}

//---------------------------------------------------------------------------------------------------------------------------------------
// destructor
//---------------------------------------------------------------------------------------------------------------------------------------
CToolBar::~CToolBar ()
{
	// deletion coherence requested
	CWidget::DestroyWidget (this);
}

//---------------------------------------------------------------------------------------------------------------------------------------
// gtk widget instanciation
//---------------------------------------------------------------------------------------------------------------------------------------
GtkWidget * CToolBar::PerformWidgetInstanciate ()
{
	// ok
	return ::gtk_toolbar_new ();
}

//---------------------------------------------------------------------------------------------------------------------------------------
// gtk widget initialization
//---------------------------------------------------------------------------------------------------------------------------------------
void CToolBar::PerformWidgetInitialize ()
{
	// get our gtk widget
	GtkWidget *inGtkWidget = GetGtkWidget ();

	// check pointers
	if (inGtkWidget == NULL) ::gtk_toolbar_set_style (GTK_TOOLBAR(inGtkWidget), GTK_TOOLBAR_ICONS);

	// show our job
	Show ();
}

//---------------------------------------------------------------------------------------------------------------------------------------
// container add
//---------------------------------------------------------------------------------------------------------------------------------------
void CToolBar::PerformContainerAdd (CWidget *inChild)
{
	// get the child gtk widget
	GtkWidget *inGtkChildWidget = inChild != NULL ? inChild -> GetGtkWidget() : NULL;

	// get our gtk widget
	GtkWidget *inGtkWidget = GetGtkWidget ();

	// check pointers
	if (inGtkWidget == NULL || inGtkChildWidget == NULL) return;

	// get the child in owner index from a gtkol hierarchy view, count only on tool button components !
	SInt16 inIndex = CComponent::GetInOwnerIndex (this, inChild, __metaclass(CToolButton));

	// perform the gtk gui insertion
	::gtk_toolbar_insert (GTK_TOOLBAR(inGtkWidget), GTK_TOOL_ITEM(inGtkChildWidget), inIndex);
}

//---------------------------------------------------------------------------------------------------------------------------------------
// container remove
//---------------------------------------------------------------------------------------------------------------------------------------
void CToolBar::PerformContainerRemove (CWidget *inChild)
{
	// generic calls
	CLayout::PerformContainerRemove (inChild);
}

//---------------------------------------------------------------------------------------------------------------------------------------
// expected children type
//---------------------------------------------------------------------------------------------------------------------------------------
CMetaClasses CToolBar::ChildMustBe () const
{
	return __metaclasses(CToolButton);
}

//---------------------------------------------------------------------------------------------------------------------------------------
// drop index
//---------------------------------------------------------------------------------------------------------------------------------------
SInt16 CToolBar::GetDropIndexAtPoint (const TPoint &inRelativePoint, const CControl *) const
{
	// get our gtk widget
	GtkWidget *inGtkWidget = GetGtkWidget ();

	// ok, I'm tired so let gtk compute for us...
	return inGtkWidget != NULL ? ::gtk_toolbar_get_drop_index (GTK_TOOLBAR(inGtkWidget), inRelativePoint.x, inRelativePoint.y) : -1;
}

//---------------------------------------------------------------------------------------------------------------------------------------
// set orientation
//---------------------------------------------------------------------------------------------------------------------------------------
void CToolBar::SetOrientation (const GtkOrientation inGtkOrientation)
{
	// get our gtk widget
	GtkWidget *inGtkWidget = GetGtkWidget ();

	// check pointer
	if (inGtkWidget == NULL) return;

	// property affectation
	::gtk_toolbar_set_orientation (GTK_TOOLBAR(inGtkWidget), inGtkOrientation);
}

//---------------------------------------------------------------------------------------------------------------------------------------
// get orientation
//---------------------------------------------------------------------------------------------------------------------------------------
GtkOrientation CToolBar::GetOrientation () const
{
	// get our gtk widget
	GtkWidget *inGtkWidget = GetGtkWidget ();

	// get the property
	return inGtkWidget == NULL ? GTK_ORIENTATION_HORIZONTAL : ::gtk_toolbar_get_orientation (GTK_TOOLBAR(inGtkWidget));
}

//---------------------------------------------------------------------------------------------------------------------------------------
// toolbar style
//---------------------------------------------------------------------------------------------------------------------------------------
void CToolBar::SetStyle (const GtkToolbarStyle inGtkToolbarStyle)
{
	// get our gtk widget
	GtkWidget *inGtkWidget = GetGtkWidget ();

	// check pointer
	if (inGtkWidget == NULL) return;

	// property affectation
	::gtk_toolbar_set_style (GTK_TOOLBAR(inGtkWidget), inGtkToolbarStyle);
}

//---------------------------------------------------------------------------------------------------------------------------------------
// get style
//---------------------------------------------------------------------------------------------------------------------------------------
GtkToolbarStyle CToolBar::GetStyle () const
{
	// get our gtk widget
	GtkWidget *inGtkWidget = GetGtkWidget ();

	// get the property
	return inGtkWidget == NULL ? GTK_TOOLBAR_ICONS : ::gtk_toolbar_get_style (GTK_TOOLBAR(inGtkWidget));
}

//---------------------------------------------------------------------------------------------------------------------------------------
// toolbar icon size
//---------------------------------------------------------------------------------------------------------------------------------------
void CToolBar::SetIconSize (const GtkIconSize inGtkIconSize)
{
	// get our gtk widget
	GtkWidget *inGtkWidget = GetGtkWidget ();

	// check pointer
	if (inGtkWidget == NULL) return;

	// property affectation	
	::gtk_toolbar_set_icon_size (GTK_TOOLBAR(inGtkWidget), inGtkIconSize);
}

//---------------------------------------------------------------------------------------------------------------------------------------
// get icon size
//---------------------------------------------------------------------------------------------------------------------------------------
GtkIconSize CToolBar::GetIconSize () const
{
	// get our gtk widget
	GtkWidget *inGtkWidget = GetGtkWidget ();

	// get the property
	return inGtkWidget == NULL ? GTK_ICON_SIZE_INVALID : ::gtk_toolbar_get_icon_size (GTK_TOOLBAR(inGtkWidget));
}

//---------------------------------------------------------------------------------------------------------------------------------------
// toolbar serialization
//---------------------------------------------------------------------------------------------------------------------------------------
void CToolBar::Serialize (CXMLElementNode *&ioXMLElementNode, const int inMode) THROWABLE
{
	// generic call first 
	CLayout::Serialize (ioXMLElementNode, inMode);
	
	// request analyse
	switch (inMode)
	{
		// xml dump
		case XML_WRITE :
		{
			// instanciate a new xml element
			CXMLElement *newXMLElement = new CXMLElement (ioXMLElementNode, XML_TOOLBAR_ELEMENT);

			// convert the attributes to strings
			CString inOrientation (GetOrientation() == GTK_ORIENTATION_HORIZONTAL ? "horizontal" : "vertical");
			CString inStyle; switch (GetStyle())
			{
				case GTK_TOOLBAR_ICONS 	    : inStyle = CString("icons"); 	break;
				case GTK_TOOLBAR_TEXT 	    : inStyle = CString("text");  	break;
				case GTK_TOOLBAR_BOTH 	    : inStyle = CString("both");  	break;
				case GTK_TOOLBAR_BOTH_HORIZ : inStyle = CString("both-horiz"); 	break;
			}
			CString inIconSize; switch (GetIconSize())
			{
				case GTK_ICON_SIZE_INVALID 	 : inIconSize = CString("invalid");	  break;
				case GTK_ICON_SIZE_MENU		 : inIconSize = CString("menu");	  break;
				case GTK_ICON_SIZE_SMALL_TOOLBAR : inIconSize = CString("small-toolbar"); break;
				case GTK_ICON_SIZE_LARGE_TOOLBAR : inIconSize = CString("large-toolbar"); break;
				case GTK_ICON_SIZE_BUTTON	 : inIconSize = CString("button");	  break;
				case GTK_ICON_SIZE_DND		 : inIconSize = CString("dnd");		  break;
				case GTK_ICON_SIZE_DIALOG	 : inIconSize = CString("dialog"); 	  break;
			}

			// add the attributes
			newXMLElement -> AddAttribute (XML_TOOLBAR_ATTR_ORIENTATION, inOrientation);
			newXMLElement -> AddAttribute (XML_TOOLBAR_ATTR_STYLE, 	     inStyle);
			newXMLElement -> AddAttribute (XML_TOOLBAR_ATTR_ICON_SIZE,   inIconSize);

			// io xml element node modification
			ioXMLElementNode = newXMLElement -> GetXMLElementNode ();
		}
		break;
		
		// xml load
		case XML_READ :
		{
			// get the xml element node
			CXMLElementNode *inXMLNode = ::xml_node_get_child (ioXMLElementNode, XML_TOOLBAR_ELEMENT);

			// check we got the expected node
                        if (inXMLNode == NULL)
                                throw new CException (CString("CToolBar::Serialize, specified xml node is not a \"") +
                                                              XML_TOOLBAR_ELEMENT + CString("\" element one."), __exception(XMLPARSE));

			// retreive our attributes
			CString inOrientation (::xml_node_get_attribute (inXMLNode, XML_TOOLBAR_ATTR_ORIENTATION).GetValue());
			CString inStyle	      (::xml_node_get_attribute (inXMLNode, XML_TOOLBAR_ATTR_STYLE).	  GetValue());
			CString inIconSize    (::xml_node_get_attribute (inXMLNode, XML_TOOLBAR_ATTR_ICON_SIZE).  GetValue());

			// orientation affectation
			if (inOrientation == CString("vertical")) 
				SetOrientation (GTK_ORIENTATION_VERTICAL); 
			else 
				SetOrientation (GTK_ORIENTATION_HORIZONTAL);

			// style affectation
			if (inStyle == CString("text"))
				SetStyle (GTK_TOOLBAR_TEXT);
			else if (inStyle == CString("both"))
				SetStyle (GTK_TOOLBAR_BOTH);
			else if (inStyle == CString("both-horiz"))
				SetStyle (GTK_TOOLBAR_BOTH_HORIZ);
			else
				SetStyle (GTK_TOOLBAR_ICONS);

			// icon size affectation
			if (inIconSize == CString("menu"))
				SetIconSize (GTK_ICON_SIZE_MENU);
			else if (inIconSize == CString("small-toolbar"))
				SetIconSize (GTK_ICON_SIZE_SMALL_TOOLBAR);
			else if (inIconSize == CString("large-toolbar"))
				SetIconSize (GTK_ICON_SIZE_LARGE_TOOLBAR);
			else if (inIconSize == CString("button"))
				SetIconSize (GTK_ICON_SIZE_BUTTON);
			else if (inIconSize == CString("dnd"))
				SetIconSize (GTK_ICON_SIZE_DND);
			else if (inIconSize == CString("dialog"))
				SetIconSize (GTK_ICON_SIZE_DIALOG);
			else
				SetIconSize (GTK_ICON_SIZE_INVALID);

			// io xml element node modification
			ioXMLElementNode = inXMLNode;
		}
		break;
	}
}

