/*
 * channel.vala
 *
 * Authored by Michael 'Mickey' Lauer <mlauer@vanille-media.de>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301  USA
 *
 */

//===========================================================================
using GLib;
using CONST;
using Gsm0710;

//===========================================================================
// The Channel class
//
internal class Channel
{
    public enum Status
    {
        Requested,  /* requested on 07.10 layer, but not acknowledged by modem */
        Acked,      /* acknowledged by modem, but not opened by any client */
        Open,       /* acknowledged and opened by a client */
        Denied,     /* denied by the modem. this status is persistent */
        Shutdown,   /* shutting down, will no longer be openable */
    }

    // FIXME: Do we really want to expose the whole multiplexer object to the channel? Consider only using the relevant delegates.
    Multiplexer _multiplexer;
    Status _status;
    FsoFramework.PtyTransport _pty;
    FsoFramework.Logger logger;

    string _name;
    int _number;
    int _serial_status;

    public Channel( Multiplexer? multiplexer, string name, int number )
    {
        var smk = new FsoFramework.SmartKeyFile();
        smk.loadFromFile( ABYSS_CONFIG_FILE );
        logger = FsoFramework.Logger.createFromKeyFile( smk, ABYSS_CONFIG_SECTION, ABYSS_LOGGING_DOMAIN );
        logger.setReprDelegate( repr );

        _multiplexer = multiplexer;
        _status = Status.Requested;
        _name = name;
        _number = number;
        _pty = new FsoFramework.PtyTransport();
        _pty.setPriorities( PTY_READ_PRIORITY, PTY_WRITE_PRIORITY );
        _pty.setDelegates( onRead, onHup );
        logger.debug( "constructed" );
    }

    ~Channel()
    {
        logger.debug( "destructed" );
    }

    public string repr()
    {
        return "<Channel %d (%s) connected via %s>".printf( _number, _name, _pty != null? _pty.getName() : "(none)" );
    }

    public string acked()
    {
        logger.debug( "acked()" );

        if ( !_pty.open() )
        {
            logger.error( "could not open pty: %s".printf( Posix.strerror( Posix.errno ) ) );
            return "";
        }

        _status = Status.Acked;
        return _pty.getName();
    }

    public void close()
    {
        logger.debug( "close()" );

        var oldstatus = _status;
        _status = Status.Shutdown;

        if ( oldstatus != Status.Requested )
        {
            if (_multiplexer != null )
                _multiplexer.channel_closed( _number );
        }

        if ( _pty != null )
        {
            _pty.close();
            _pty = null;
        }
    }

    public string name()
    {
        return _name;
    }

    public string path()
    {
        return _pty.getName();
    }

    public bool isAcked()
    {
        return _status != Status.Requested;
    }

    public void setSerialStatus( int newstatus )
    {
        logger.debug( "setSerialStatus()" );

        var oldstatus = _serial_status;
        _serial_status = newstatus;

        // check whether the FC bit has been set
        if ( ( ( oldstatus & SerialStatus.FC ) == 0 ) &&
                 ( ( newstatus & SerialStatus.FC ) == SerialStatus.FC ) )
        {
            logger.warning( "FC has been set. Disabling read from PTY" );
            _pty.freeze();
        }

        // check whether the FC bit has been cleared
        if ( ( ( oldstatus & SerialStatus.FC ) == SerialStatus.FC ) &&
             ( ( newstatus & SerialStatus.FC ) == 0 ) )
        {
            logger.warning( "FC has been cleared. Reenabling read from PTY" );
            _pty.thaw();
        }
    }

    public void deliverData( void* data, int len )
    {
        _pty.write( data, len );
        MainContext.default().iteration( false ); // give other channels a chance (round-robin)
    }

    //
    // delegates from Pty object
    //
    public void onRead( FsoFramework.Transport transport )
    {
        logger.debug( "onRead() from Pty; reading." );
        assert( _multiplexer != null );

        if ( ( _serial_status & SerialStatus.FC ) == SerialStatus.FC )
        {
            logger.warning( "FC active... reading anyways..." );
        }

        var buffer = new char[8192];
        int bytesread = transport.read( buffer, 8192 );
        logger.debug( "read %d bytes".printf( bytesread ) );

        _multiplexer.submit_data( _number, buffer, (int)bytesread );
    }

    public void onHup( FsoFramework.Transport transport )
    {
        logger.debug( "onHup() from Pty; closing." );
        close();
    }

}
