require 'test/unit'

require 'gettext.rb'

require 'testlib1.rb'
require 'testlib2.rb'
require 'testlib3.rb'
require 'testlib4.rb'
require 'test_rubyparser.rb'
require 'test_sgettext.rb'

class TestGetText < Test::Unit::TestCase
  include GetText

  def test_bindtextdomain
    GetText.locale = nil
    return if /linux/ !~ RUBY_PLATFORM
    GetText.bindtextdomain("libc")
    GetText.locale = "ja_JP.eucJP"
    GetText.charset = "euc-jp"
    assert_equal("λޤ", GetText._("Terminated"))

    GetText.locale = nil
    GetText.bindtextdomain("test1", "locale")
    assert_equal("japanese", GetText._("language"))

    GetText.bindtextdomain("libc", "/usr/share/locale")
    assert_equal("λޤ", GetText._("Terminated"))

    GetText.bindtextdomain("test2", "locale")
    assert_equal("JAPANESE", GetText._("LANGUAGE"))

    GetText.bindtextdomain("libc", "/usr/share/locale/")
    assert_equal("λޤ", GetText._("Terminated"))

    GetText.bindtextdomain("libc", "/usr/share/locale", "de")
    assert_equal("Beendet", GetText._("Terminated"))

    GetText.bindtextdomain("test1", "locale")
    GetText.locale = "fr"
    assert_equal("french", GetText._("language"))

    GetText.locale = nil
    GetText.bindtextdomain("test1", "locale", "ja")
    assert_equal("japanese", GetText._("language"))
  end

  def test_bindtextdomain_include_module
    GetText.locale = nil
    bindtextdomain("test1", "locale")
    assert_equal("japanese", _("language"))
  end

  def test_gettext
    GetText.locale = nil
    bindtextdomain("test1", "locale")
    assert_equal("japanese", gettext("language"))
  end

  def test_class
    GetText.locale = nil
    bindtextdomain("test2", "locale")

    testlib1 = TestLib1.new
    assert_equal("japanese", testlib1.test)
    assert_equal("japanese", testlib1.test) #no influence of previous line
    assert_equal("ONE IS 1.", testlib1.test_formatted_string)
    testlib1.setlocale("fr")
    assert_equal("JAPANESE", _("LANGUAGE")) #no influence of previous line
    assert_equal("french", testlib1.test)
  end

  def test_subclass
    GetText.locale = nil
    testlib2 = TestLib2.new
    assert_equal("JAPANESE", testlib2.test2)
    assert_equal("japanese", testlib2.test)
  end

  def test_nested_module
    GetText.locale = nil
    testlib3 = TestLib3::Test3.new
    assert_equal("JAPANESE", testlib3.test3)
  end

  def test_no_msgstr
    GetText.locale = nil
    bindtextdomain("test1", "locale", "ja")
    assert_equal("nomsgstr", _("nomsgstr"))
  end

  def test_complex
    GetText.locale = nil
    testlib4 = TestRubyParser.new
    assert_equal("AAA", testlib4.test_1)
    assert_equal("AAA\n", testlib4.test_2)
    assert_equal("BBB\nCCC", testlib4.test_3)
    assert_equal("BBB
CCC
DDD
", testlib4.test_4)
    assert_equal("EEE", testlib4.test_5)
    assert_equal("EEEfooFFF", testlib4.test_6)
    assert_equal("GGGHHHIII", testlib4.test_7)
  end

  def test_noop
    GetText.locale = nil
    assert_equal("test", N_("test"))
  end

  def test_sgettext
    GetText.locale = nil
    testlib5 = TestSGetText.new
    assert_equal("MATCHED", testlib5.test_1)
    assert_equal("MATCHED", testlib5.test_2)
    assert_equal("AAA", testlib5.test_3)
    assert_equal("CCC", testlib5.test_4)
    assert_equal("CCC", testlib5.test_5)
    assert_equal("BBB", testlib5.test_6)
    assert_equal("B|BB", testlib5.test_7)
    assert_equal("MATCHED", testlib5.test_8)
    assert_equal("BBB", testlib5.test_9)
  end

  def test_plural
    GetText.locale = nil
    bindtextdomain("plural", "locale", "ja")
    assert_equal("all", n_("one", "two", 0))
    assert_equal("all", n_("one", "two", 1))
    assert_equal("all", n_("one", "two", 2))

    setlocale("da")
    assert_equal("da_plural", n_("one", "two", 0))
    assert_equal("da_one", n_("one", "two", 1))
    assert_equal("da_plural", n_("one", "two", 2))

    setlocale("fr")
    assert_equal("fr_one", ngettext("one", "two", 0))
    assert_equal("fr_one", ngettext("one", "two", 1))
    assert_equal("fr_plural", ngettext("one", "two", 2))

    setlocale("la")
    assert_equal("la_one", ngettext("one", "two", 21))
    assert_equal("la_one", ngettext("one", "two", 1))
    assert_equal("la_plural", ngettext("one", "two", 2))
    assert_equal("la_zero", ngettext("one", "two", 0))

    setlocale("ir")
    assert_equal("ir_one", ngettext("one", "two", 1))
    assert_equal("ir_two", ngettext("one", "two", 2))
    assert_equal("ir_plural", ngettext("one", "two", 3))
    assert_equal("ir_plural", ngettext("one", "two", 0))

    setlocale("li")
    assert_equal("li_one", ngettext("one", "two", 1))
    assert_equal("li_two", ngettext("one", "two", 22))
    assert_equal("li_three", ngettext("one", "two", 11))

    setlocale("cr")
    assert_equal("cr_one", ngettext("one", "two", 1))
    assert_equal("cr_two", ngettext("one", "two", 2))
    assert_equal("cr_three", ngettext("one", "two", 5))

    setlocale("po")
    assert_equal("po_one", ngettext("one", "two", 1))
    assert_equal("po_two", ngettext("one", "two", 2))
    assert_equal("po_three", ngettext("one", "two", 5))

    setlocale("sl")
    assert_equal("sl_one", ngettext("one", "two", 1))
    assert_equal("sl_two", ngettext("one", "two", 2))
    assert_equal("sl_three", ngettext("one", "two", 3))
    assert_equal("sl_three", ngettext("one", "two", 4))
    assert_equal("sl_four", ngettext("one", "two", 5))
  end

  def test_plural_format_invalid
    setlocale(nil)
    bindtextdomain("plural_error", "locale", "ja")
    assert_equal("a", n_("one", "two", 0)) 
    # Use default(plural = 0)
    setlocale("fr")
    assert_equal("fr_one", n_("one", "two", 0))   
    assert_equal("fr_one", n_("one", "two", 1))
    assert_equal("fr_one", n_("one", "two", 2))
    setlocale("da")
    assert_equal("da_one", n_("one", "two", 0))   
    assert_equal("da_one", n_("one", "two", 1))
    assert_equal("da_one", n_("one", "two", 2))
  end

  def test_setlocale
    GetText.locale = nil
    bindtextdomain("test1", "locale")
    assert_equal("japanese", _("language"))
    setlocale("en")
    assert_equal("language", _("language"))
    setlocale("fr")
    assert_equal("french", _("language"))
  end

  def test_locale
    GetText.locale = nil
    loc = Locale.set(Locale::CTYPE, "ja_JP.eucJP")
    assert_equal("ja_JP.eucJP", loc)
    assert_equal("ja_JP.eucJP", Locale.get)
    assert_equal("EUC-JP", Locale.codeset)

    loc = Locale.set(Locale::CTYPE, "ja_JP.UTF-8")
    assert_equal("ja_JP.UTF-8", loc)
    assert_equal("ja_JP.UTF-8", Locale.get)
    assert_equal("UTF-8", Locale.codeset)

    Locale.setlocale(Locale::CTYPE, "ja_JP.eucJP")
  end

  def test_textdomain
    GetText.locale = nil
    Locale.setlocale(Locale::CTYPE, "ja_JP.eucJP")
    testlib = TestLib4.new
    assert_equal("japanese", testlib.test)
    assert_raises(GetText::NoboundTextDomainError) {
      GetText.textdomain("nodomainisdefined")
    }
  end
end
