/* hashmap.vala
 *
 * Copyright (C) 1995-1997  Peter Mattis, Spencer Kimball and Josh MacDonald
 * Copyright (C) 1997-2000  GLib Team and others
 * Copyright (C) 2007-2009  Jürg Billeter
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.

 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.

 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301  USA
 *
 * Author:
 * 	Jürg Billeter <j@bitron.ch>
 */

#include <glib.h>
#include <glib-object.h>


#define GEE_TYPE_MAP (gee_map_get_type ())
#define GEE_MAP(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), GEE_TYPE_MAP, GeeMap))
#define GEE_IS_MAP(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), GEE_TYPE_MAP))
#define GEE_MAP_GET_INTERFACE(obj) (G_TYPE_INSTANCE_GET_INTERFACE ((obj), GEE_TYPE_MAP, GeeMapIface))

typedef struct _GeeMap GeeMap;
typedef struct _GeeMapIface GeeMapIface;

#define GEE_TYPE_ITERABLE (gee_iterable_get_type ())
#define GEE_ITERABLE(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), GEE_TYPE_ITERABLE, GeeIterable))
#define GEE_IS_ITERABLE(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), GEE_TYPE_ITERABLE))
#define GEE_ITERABLE_GET_INTERFACE(obj) (G_TYPE_INSTANCE_GET_INTERFACE ((obj), GEE_TYPE_ITERABLE, GeeIterableIface))

typedef struct _GeeIterable GeeIterable;
typedef struct _GeeIterableIface GeeIterableIface;

#define GEE_TYPE_ITERATOR (gee_iterator_get_type ())
#define GEE_ITERATOR(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), GEE_TYPE_ITERATOR, GeeIterator))
#define GEE_IS_ITERATOR(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), GEE_TYPE_ITERATOR))
#define GEE_ITERATOR_GET_INTERFACE(obj) (G_TYPE_INSTANCE_GET_INTERFACE ((obj), GEE_TYPE_ITERATOR, GeeIteratorIface))

typedef struct _GeeIterator GeeIterator;
typedef struct _GeeIteratorIface GeeIteratorIface;

#define GEE_TYPE_COLLECTION (gee_collection_get_type ())
#define GEE_COLLECTION(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), GEE_TYPE_COLLECTION, GeeCollection))
#define GEE_IS_COLLECTION(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), GEE_TYPE_COLLECTION))
#define GEE_COLLECTION_GET_INTERFACE(obj) (G_TYPE_INSTANCE_GET_INTERFACE ((obj), GEE_TYPE_COLLECTION, GeeCollectionIface))

typedef struct _GeeCollection GeeCollection;
typedef struct _GeeCollectionIface GeeCollectionIface;

#define GEE_TYPE_SET (gee_set_get_type ())
#define GEE_SET(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), GEE_TYPE_SET, GeeSet))
#define GEE_IS_SET(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), GEE_TYPE_SET))
#define GEE_SET_GET_INTERFACE(obj) (G_TYPE_INSTANCE_GET_INTERFACE ((obj), GEE_TYPE_SET, GeeSetIface))

typedef struct _GeeSet GeeSet;
typedef struct _GeeSetIface GeeSetIface;

#define GEE_TYPE_ABSTRACT_MAP (gee_abstract_map_get_type ())
#define GEE_ABSTRACT_MAP(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), GEE_TYPE_ABSTRACT_MAP, GeeAbstractMap))
#define GEE_ABSTRACT_MAP_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), GEE_TYPE_ABSTRACT_MAP, GeeAbstractMapClass))
#define GEE_IS_ABSTRACT_MAP(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), GEE_TYPE_ABSTRACT_MAP))
#define GEE_IS_ABSTRACT_MAP_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), GEE_TYPE_ABSTRACT_MAP))
#define GEE_ABSTRACT_MAP_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), GEE_TYPE_ABSTRACT_MAP, GeeAbstractMapClass))

typedef struct _GeeAbstractMap GeeAbstractMap;
typedef struct _GeeAbstractMapClass GeeAbstractMapClass;
typedef struct _GeeAbstractMapPrivate GeeAbstractMapPrivate;

#define GEE_TYPE_HASH_MAP (gee_hash_map_get_type ())
#define GEE_HASH_MAP(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), GEE_TYPE_HASH_MAP, GeeHashMap))
#define GEE_HASH_MAP_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), GEE_TYPE_HASH_MAP, GeeHashMapClass))
#define GEE_IS_HASH_MAP(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), GEE_TYPE_HASH_MAP))
#define GEE_IS_HASH_MAP_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), GEE_TYPE_HASH_MAP))
#define GEE_HASH_MAP_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), GEE_TYPE_HASH_MAP, GeeHashMapClass))

typedef struct _GeeHashMap GeeHashMap;
typedef struct _GeeHashMapClass GeeHashMapClass;
typedef struct _GeeHashMapPrivate GeeHashMapPrivate;
typedef struct _GeeHashMapNode GeeHashMapNode;

#define GEE_TYPE_ABSTRACT_COLLECTION (gee_abstract_collection_get_type ())
#define GEE_ABSTRACT_COLLECTION(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), GEE_TYPE_ABSTRACT_COLLECTION, GeeAbstractCollection))
#define GEE_ABSTRACT_COLLECTION_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), GEE_TYPE_ABSTRACT_COLLECTION, GeeAbstractCollectionClass))
#define GEE_IS_ABSTRACT_COLLECTION(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), GEE_TYPE_ABSTRACT_COLLECTION))
#define GEE_IS_ABSTRACT_COLLECTION_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), GEE_TYPE_ABSTRACT_COLLECTION))
#define GEE_ABSTRACT_COLLECTION_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), GEE_TYPE_ABSTRACT_COLLECTION, GeeAbstractCollectionClass))

typedef struct _GeeAbstractCollection GeeAbstractCollection;
typedef struct _GeeAbstractCollectionClass GeeAbstractCollectionClass;

#define GEE_HASH_MAP_TYPE_KEY_SET (gee_hash_map_key_set_get_type ())
#define GEE_HASH_MAP_KEY_SET(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), GEE_HASH_MAP_TYPE_KEY_SET, GeeHashMapKeySet))
#define GEE_HASH_MAP_KEY_SET_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), GEE_HASH_MAP_TYPE_KEY_SET, GeeHashMapKeySetClass))
#define GEE_HASH_MAP_IS_KEY_SET(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), GEE_HASH_MAP_TYPE_KEY_SET))
#define GEE_HASH_MAP_IS_KEY_SET_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), GEE_HASH_MAP_TYPE_KEY_SET))
#define GEE_HASH_MAP_KEY_SET_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), GEE_HASH_MAP_TYPE_KEY_SET, GeeHashMapKeySetClass))

typedef struct _GeeHashMapKeySet GeeHashMapKeySet;
typedef struct _GeeHashMapKeySetClass GeeHashMapKeySetClass;

#define GEE_HASH_MAP_TYPE_VALUE_COLLECTION (gee_hash_map_value_collection_get_type ())
#define GEE_HASH_MAP_VALUE_COLLECTION(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), GEE_HASH_MAP_TYPE_VALUE_COLLECTION, GeeHashMapValueCollection))
#define GEE_HASH_MAP_VALUE_COLLECTION_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), GEE_HASH_MAP_TYPE_VALUE_COLLECTION, GeeHashMapValueCollectionClass))
#define GEE_HASH_MAP_IS_VALUE_COLLECTION(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), GEE_HASH_MAP_TYPE_VALUE_COLLECTION))
#define GEE_HASH_MAP_IS_VALUE_COLLECTION_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), GEE_HASH_MAP_TYPE_VALUE_COLLECTION))
#define GEE_HASH_MAP_VALUE_COLLECTION_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), GEE_HASH_MAP_TYPE_VALUE_COLLECTION, GeeHashMapValueCollectionClass))

typedef struct _GeeHashMapValueCollection GeeHashMapValueCollection;
typedef struct _GeeHashMapValueCollectionClass GeeHashMapValueCollectionClass;
typedef struct _GeeAbstractCollectionPrivate GeeAbstractCollectionPrivate;
typedef struct _GeeHashMapKeySetPrivate GeeHashMapKeySetPrivate;

#define GEE_HASH_MAP_TYPE_KEY_ITERATOR (gee_hash_map_key_iterator_get_type ())
#define GEE_HASH_MAP_KEY_ITERATOR(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), GEE_HASH_MAP_TYPE_KEY_ITERATOR, GeeHashMapKeyIterator))
#define GEE_HASH_MAP_KEY_ITERATOR_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), GEE_HASH_MAP_TYPE_KEY_ITERATOR, GeeHashMapKeyIteratorClass))
#define GEE_HASH_MAP_IS_KEY_ITERATOR(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), GEE_HASH_MAP_TYPE_KEY_ITERATOR))
#define GEE_HASH_MAP_IS_KEY_ITERATOR_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), GEE_HASH_MAP_TYPE_KEY_ITERATOR))
#define GEE_HASH_MAP_KEY_ITERATOR_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), GEE_HASH_MAP_TYPE_KEY_ITERATOR, GeeHashMapKeyIteratorClass))

typedef struct _GeeHashMapKeyIterator GeeHashMapKeyIterator;
typedef struct _GeeHashMapKeyIteratorClass GeeHashMapKeyIteratorClass;
typedef struct _GeeHashMapKeyIteratorPrivate GeeHashMapKeyIteratorPrivate;
typedef struct _GeeHashMapValueCollectionPrivate GeeHashMapValueCollectionPrivate;

#define GEE_HASH_MAP_TYPE_VALUE_ITERATOR (gee_hash_map_value_iterator_get_type ())
#define GEE_HASH_MAP_VALUE_ITERATOR(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), GEE_HASH_MAP_TYPE_VALUE_ITERATOR, GeeHashMapValueIterator))
#define GEE_HASH_MAP_VALUE_ITERATOR_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), GEE_HASH_MAP_TYPE_VALUE_ITERATOR, GeeHashMapValueIteratorClass))
#define GEE_HASH_MAP_IS_VALUE_ITERATOR(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), GEE_HASH_MAP_TYPE_VALUE_ITERATOR))
#define GEE_HASH_MAP_IS_VALUE_ITERATOR_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), GEE_HASH_MAP_TYPE_VALUE_ITERATOR))
#define GEE_HASH_MAP_VALUE_ITERATOR_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), GEE_HASH_MAP_TYPE_VALUE_ITERATOR, GeeHashMapValueIteratorClass))

typedef struct _GeeHashMapValueIterator GeeHashMapValueIterator;
typedef struct _GeeHashMapValueIteratorClass GeeHashMapValueIteratorClass;
typedef struct _GeeHashMapValueIteratorPrivate GeeHashMapValueIteratorPrivate;

/**
 * Implemented by classes that support a simple iteration over instances of the
 * collection.
 */
struct _GeeIteratorIface {
	GTypeInterface parent_iface;
	gboolean (*next) (GeeIterator* self);
	gpointer (*get) (GeeIterator* self);
};

/**
 * Implemented by classes that support a simple iteration over instances of the
 * collection.
 */
struct _GeeIterableIface {
	GTypeInterface parent_iface;
	GeeIterator* (*iterator) (GeeIterable* self);
	GType (*get_element_type) (GeeIterable* self);
};

/**
 * Serves as the base interface for implementing collection classes. Defines
 * size, iteration, and modification methods.
 */
struct _GeeCollectionIface {
	GTypeInterface parent_iface;
	gboolean (*contains) (GeeCollection* self, gconstpointer item);
	gboolean (*add) (GeeCollection* self, gconstpointer item);
	gboolean (*remove) (GeeCollection* self, gconstpointer item);
	void (*clear) (GeeCollection* self);
	gboolean (*add_all) (GeeCollection* self, GeeCollection* collection);
	gboolean (*contains_all) (GeeCollection* self, GeeCollection* collection);
	gboolean (*remove_all) (GeeCollection* self, GeeCollection* collection);
	gboolean (*retain_all) (GeeCollection* self, GeeCollection* collection);
	gpointer* (*to_array) (GeeCollection* self, int* result_length1);
	gint (*get_size) (GeeCollection* self);
	gboolean (*get_is_empty) (GeeCollection* self);
};

/**
 * A set is a collection without duplicates.
 */
struct _GeeSetIface {
	GTypeInterface parent_iface;
};

/**
 * A map is a generic collection of key/value pairs.
 */
struct _GeeMapIface {
	GTypeInterface parent_iface;
	GeeSet* (*get_keys) (GeeMap* self);
	GeeCollection* (*get_values) (GeeMap* self);
	gboolean (*contains) (GeeMap* self, gconstpointer key);
	gpointer (*get) (GeeMap* self, gconstpointer key);
	void (*set) (GeeMap* self, gconstpointer key, gconstpointer value);
	gboolean (*remove) (GeeMap* self, gconstpointer key, gpointer* value);
	void (*clear) (GeeMap* self);
	void (*set_all) (GeeMap* self, GeeMap* map);
	gboolean (*remove_all) (GeeMap* self, GeeMap* map);
	gboolean (*contains_all) (GeeMap* self, GeeMap* map);
	gint (*get_size) (GeeMap* self);
	gboolean (*get_is_empty) (GeeMap* self);
};

/**
 * Serves as the base class for implementing map classes.
 */
struct _GeeAbstractMap {
	GObject parent_instance;
	GeeAbstractMapPrivate * priv;
};

struct _GeeAbstractMapClass {
	GObjectClass parent_class;
	GeeSet* (*get_keys) (GeeAbstractMap* self);
	GeeCollection* (*get_values) (GeeAbstractMap* self);
	gboolean (*contains) (GeeAbstractMap* self, gconstpointer key);
	gpointer (*get) (GeeAbstractMap* self, gconstpointer key);
	void (*set) (GeeAbstractMap* self, gconstpointer key, gconstpointer value);
	gboolean (*remove) (GeeAbstractMap* self, gconstpointer key, gpointer* value);
	void (*clear) (GeeAbstractMap* self);
	void (*set_all) (GeeAbstractMap* self, GeeMap* map);
	gboolean (*remove_all) (GeeAbstractMap* self, GeeMap* map);
	gboolean (*contains_all) (GeeAbstractMap* self, GeeMap* map);
	gint (*get_size) (GeeAbstractMap* self);
	gboolean (*get_is_empty) (GeeAbstractMap* self);
};

/**
 * Hashtable implementation of the Map interface.
 */
struct _GeeHashMap {
	GeeAbstractMap parent_instance;
	GeeHashMapPrivate * priv;
};

struct _GeeHashMapClass {
	GeeAbstractMapClass parent_class;
};

struct _GeeHashMapPrivate {
	GType k_type;
	GBoxedCopyFunc k_dup_func;
	GDestroyNotify k_destroy_func;
	GType v_type;
	GBoxedCopyFunc v_dup_func;
	GDestroyNotify v_destroy_func;
	GHashFunc _key_hash_func;
	GEqualFunc _key_equal_func;
	GEqualFunc _value_equal_func;
	gint _array_size;
	gint _nnodes;
	GeeHashMapNode** _nodes;
	gint _nodes_length1;
	gint _nodes_size;
	gint _stamp;
};

struct _GeeHashMapNode {
	gpointer key;
	gpointer value;
	GeeHashMapNode* next;
	guint key_hash;
};

/**
 * Serves as the base class for implementing collection classes.
 */
struct _GeeAbstractCollection {
	GObject parent_instance;
	GeeAbstractCollectionPrivate * priv;
};

struct _GeeAbstractCollectionClass {
	GObjectClass parent_class;
	gboolean (*contains) (GeeAbstractCollection* self, gconstpointer item);
	gboolean (*add) (GeeAbstractCollection* self, gconstpointer item);
	gboolean (*remove) (GeeAbstractCollection* self, gconstpointer item);
	void (*clear) (GeeAbstractCollection* self);
	gpointer* (*to_array) (GeeAbstractCollection* self, int* result_length1);
	gboolean (*add_all) (GeeAbstractCollection* self, GeeCollection* collection);
	gboolean (*contains_all) (GeeAbstractCollection* self, GeeCollection* collection);
	gboolean (*remove_all) (GeeAbstractCollection* self, GeeCollection* collection);
	gboolean (*retain_all) (GeeAbstractCollection* self, GeeCollection* collection);
	GeeIterator* (*iterator) (GeeAbstractCollection* self);
	gint (*get_size) (GeeAbstractCollection* self);
	gboolean (*get_is_empty) (GeeAbstractCollection* self);
};

struct _GeeHashMapKeySet {
	GeeAbstractCollection parent_instance;
	GeeHashMapKeySetPrivate * priv;
};

struct _GeeHashMapKeySetClass {
	GeeAbstractCollectionClass parent_class;
};

struct _GeeHashMapKeySetPrivate {
	GType k_type;
	GBoxedCopyFunc k_dup_func;
	GDestroyNotify k_destroy_func;
	GType v_type;
	GBoxedCopyFunc v_dup_func;
	GDestroyNotify v_destroy_func;
	GeeHashMap* _map;
};

struct _GeeHashMapKeyIterator {
	GObject parent_instance;
	GeeHashMapKeyIteratorPrivate * priv;
};

struct _GeeHashMapKeyIteratorClass {
	GObjectClass parent_class;
};

struct _GeeHashMapKeyIteratorPrivate {
	GType k_type;
	GBoxedCopyFunc k_dup_func;
	GDestroyNotify k_destroy_func;
	GType v_type;
	GBoxedCopyFunc v_dup_func;
	GDestroyNotify v_destroy_func;
	GeeHashMap* _map;
	gint _index;
	GeeHashMapNode* _node;
	gint _stamp;
};

struct _GeeHashMapValueCollection {
	GeeAbstractCollection parent_instance;
	GeeHashMapValueCollectionPrivate * priv;
};

struct _GeeHashMapValueCollectionClass {
	GeeAbstractCollectionClass parent_class;
};

struct _GeeHashMapValueCollectionPrivate {
	GType k_type;
	GBoxedCopyFunc k_dup_func;
	GDestroyNotify k_destroy_func;
	GType v_type;
	GBoxedCopyFunc v_dup_func;
	GDestroyNotify v_destroy_func;
	GeeHashMap* _map;
};

struct _GeeHashMapValueIterator {
	GObject parent_instance;
	GeeHashMapValueIteratorPrivate * priv;
};

struct _GeeHashMapValueIteratorClass {
	GObjectClass parent_class;
};

struct _GeeHashMapValueIteratorPrivate {
	GType k_type;
	GBoxedCopyFunc k_dup_func;
	GDestroyNotify k_destroy_func;
	GType v_type;
	GBoxedCopyFunc v_dup_func;
	GDestroyNotify v_destroy_func;
	GeeHashMap* _map;
	gint _index;
	GeeHashMapNode* _node;
	gint _stamp;
};


static gpointer gee_hash_map_key_set_parent_class = NULL;
static GeeSetIface* gee_hash_map_key_set_gee_set_parent_iface = NULL;
static gpointer gee_hash_map_key_iterator_parent_class = NULL;
static GeeIteratorIface* gee_hash_map_key_iterator_gee_iterator_parent_iface = NULL;
static gpointer gee_hash_map_value_collection_parent_class = NULL;
static gpointer gee_hash_map_value_iterator_parent_class = NULL;
static GeeIteratorIface* gee_hash_map_value_iterator_gee_iterator_parent_iface = NULL;
static gpointer gee_hash_map_parent_class = NULL;

GType gee_iterator_get_type (void);
GType gee_iterable_get_type (void);
GType gee_collection_get_type (void);
GType gee_set_get_type (void);
GType gee_map_get_type (void);
GType gee_abstract_map_get_type (void);
GType gee_hash_map_get_type (void);
#define GEE_HASH_MAP_GET_PRIVATE(o) (G_TYPE_INSTANCE_GET_PRIVATE ((o), GEE_TYPE_HASH_MAP, GeeHashMapPrivate))
enum  {
	GEE_HASH_MAP_DUMMY_PROPERTY,
	GEE_HASH_MAP_SIZE,
	GEE_HASH_MAP_KEY_HASH_FUNC,
	GEE_HASH_MAP_KEY_EQUAL_FUNC,
	GEE_HASH_MAP_VALUE_EQUAL_FUNC,
	GEE_HASH_MAP_K_TYPE,
	GEE_HASH_MAP_K_DUP_FUNC,
	GEE_HASH_MAP_K_DESTROY_FUNC,
	GEE_HASH_MAP_V_TYPE,
	GEE_HASH_MAP_V_DUP_FUNC,
	GEE_HASH_MAP_V_DESTROY_FUNC
};
#define GEE_HASH_MAP_MIN_SIZE 11
#define GEE_HASH_MAP_MAX_SIZE 13845163
static void gee_hash_map_set_key_hash_func (GeeHashMap* self, GHashFunc value);
static void gee_hash_map_set_key_equal_func (GeeHashMap* self, GEqualFunc value);
static void gee_hash_map_set_value_equal_func (GeeHashMap* self, GEqualFunc value);
GeeHashMap* gee_hash_map_new (GType k_type, GBoxedCopyFunc k_dup_func, GDestroyNotify k_destroy_func, GType v_type, GBoxedCopyFunc v_dup_func, GDestroyNotify v_destroy_func, GHashFunc key_hash_func, GEqualFunc key_equal_func, GEqualFunc value_equal_func);
GeeHashMap* gee_hash_map_construct (GType object_type, GType k_type, GBoxedCopyFunc k_dup_func, GDestroyNotify k_destroy_func, GType v_type, GBoxedCopyFunc v_dup_func, GDestroyNotify v_destroy_func, GHashFunc key_hash_func, GEqualFunc key_equal_func, GEqualFunc value_equal_func);
static GeeHashMapKeySet* gee_hash_map_key_set_new (GType k_type, GBoxedCopyFunc k_dup_func, GDestroyNotify k_destroy_func, GType v_type, GBoxedCopyFunc v_dup_func, GDestroyNotify v_destroy_func, GeeHashMap* map);
static GeeHashMapKeySet* gee_hash_map_key_set_construct (GType object_type, GType k_type, GBoxedCopyFunc k_dup_func, GDestroyNotify k_destroy_func, GType v_type, GBoxedCopyFunc v_dup_func, GDestroyNotify v_destroy_func, GeeHashMap* map);
GType gee_abstract_collection_get_type (void);
static GType gee_hash_map_key_set_get_type (void);
static GeeSet* gee_hash_map_real_get_keys (GeeAbstractMap* base);
static GeeHashMapValueCollection* gee_hash_map_value_collection_new (GType k_type, GBoxedCopyFunc k_dup_func, GDestroyNotify k_destroy_func, GType v_type, GBoxedCopyFunc v_dup_func, GDestroyNotify v_destroy_func, GeeHashMap* map);
static GeeHashMapValueCollection* gee_hash_map_value_collection_construct (GType object_type, GType k_type, GBoxedCopyFunc k_dup_func, GDestroyNotify k_destroy_func, GType v_type, GBoxedCopyFunc v_dup_func, GDestroyNotify v_destroy_func, GeeHashMap* map);
static GType gee_hash_map_value_collection_get_type (void);
static GeeCollection* gee_hash_map_real_get_values (GeeAbstractMap* base);
GHashFunc gee_hash_map_get_key_hash_func (GeeHashMap* self);
GEqualFunc gee_hash_map_get_key_equal_func (GeeHashMap* self);
static GeeHashMapNode** gee_hash_map_lookup_node (GeeHashMap* self, gconstpointer key);
static gboolean gee_hash_map_real_contains (GeeAbstractMap* base, gconstpointer key);
static gpointer gee_hash_map_real_get (GeeAbstractMap* base, gconstpointer key);
static GeeHashMapNode* gee_hash_map_node_new (gpointer k, gpointer v, guint hash);
static GeeHashMapNode* gee_hash_map_node_new (gpointer k, gpointer v, guint hash);
static void gee_hash_map_resize (GeeHashMap* self);
static void gee_hash_map_real_set (GeeAbstractMap* base, gconstpointer key, gconstpointer value);
static gboolean gee_hash_map_real_remove (GeeAbstractMap* base, gconstpointer key, gpointer* value);
static void gee_hash_map_real_clear (GeeAbstractMap* base);
GEqualFunc gee_hash_map_get_value_equal_func (GeeHashMap* self);
static GObject * gee_hash_map_constructor (GType type, guint n_construct_properties, GObjectConstructParam * construct_properties);
void gee_abstract_map_clear (GeeAbstractMap* self);
static void gee_hash_map_node_free (GeeHashMapNode* self);
static void gee_hash_map_node_instance_init (GeeHashMapNode * self);
#define GEE_HASH_MAP_KEY_SET_GET_PRIVATE(o) (G_TYPE_INSTANCE_GET_PRIVATE ((o), GEE_HASH_MAP_TYPE_KEY_SET, GeeHashMapKeySetPrivate))
enum  {
	GEE_HASH_MAP_KEY_SET_DUMMY_PROPERTY,
	GEE_HASH_MAP_KEY_SET_MAP,
	GEE_HASH_MAP_KEY_SET_SIZE,
	GEE_HASH_MAP_KEY_SET_K_TYPE,
	GEE_HASH_MAP_KEY_SET_K_DUP_FUNC,
	GEE_HASH_MAP_KEY_SET_K_DESTROY_FUNC,
	GEE_HASH_MAP_KEY_SET_V_TYPE,
	GEE_HASH_MAP_KEY_SET_V_DUP_FUNC,
	GEE_HASH_MAP_KEY_SET_V_DESTROY_FUNC
};
static void gee_hash_map_key_set_set_map (GeeHashMapKeySet* self, GeeHashMap* value);
static GeeHashMap* gee_hash_map_key_set_get_map (GeeHashMapKeySet* self);
static GeeHashMapKeyIterator* gee_hash_map_key_iterator_new (GType k_type, GBoxedCopyFunc k_dup_func, GDestroyNotify k_destroy_func, GType v_type, GBoxedCopyFunc v_dup_func, GDestroyNotify v_destroy_func, GeeHashMap* map);
static GeeHashMapKeyIterator* gee_hash_map_key_iterator_construct (GType object_type, GType k_type, GBoxedCopyFunc k_dup_func, GDestroyNotify k_destroy_func, GType v_type, GBoxedCopyFunc v_dup_func, GDestroyNotify v_destroy_func, GeeHashMap* map);
static GType gee_hash_map_key_iterator_get_type (void);
static GeeIterator* gee_hash_map_key_set_real_iterator (GeeAbstractCollection* base);
static gboolean gee_hash_map_key_set_real_add (GeeAbstractCollection* base, gconstpointer key);
static void gee_hash_map_key_set_real_clear (GeeAbstractCollection* base);
static gboolean gee_hash_map_key_set_real_remove (GeeAbstractCollection* base, gconstpointer key);
gboolean gee_abstract_map_contains (GeeAbstractMap* self, gconstpointer key);
static gboolean gee_hash_map_key_set_real_contains (GeeAbstractCollection* base, gconstpointer key);
static gboolean gee_hash_map_key_set_real_add_all (GeeAbstractCollection* base, GeeCollection* collection);
static gboolean gee_hash_map_key_set_real_remove_all (GeeAbstractCollection* base, GeeCollection* collection);
static gboolean gee_hash_map_key_set_real_retain_all (GeeAbstractCollection* base, GeeCollection* collection);
gint gee_abstract_map_get_size (GeeAbstractMap* self);
static void gee_hash_map_key_set_finalize (GObject* obj);
static void gee_hash_map_key_set_get_property (GObject * object, guint property_id, GValue * value, GParamSpec * pspec);
static void gee_hash_map_key_set_set_property (GObject * object, guint property_id, const GValue * value, GParamSpec * pspec);
#define GEE_HASH_MAP_KEY_ITERATOR_GET_PRIVATE(o) (G_TYPE_INSTANCE_GET_PRIVATE ((o), GEE_HASH_MAP_TYPE_KEY_ITERATOR, GeeHashMapKeyIteratorPrivate))
enum  {
	GEE_HASH_MAP_KEY_ITERATOR_DUMMY_PROPERTY,
	GEE_HASH_MAP_KEY_ITERATOR_MAP,
	GEE_HASH_MAP_KEY_ITERATOR_K_TYPE,
	GEE_HASH_MAP_KEY_ITERATOR_K_DUP_FUNC,
	GEE_HASH_MAP_KEY_ITERATOR_K_DESTROY_FUNC,
	GEE_HASH_MAP_KEY_ITERATOR_V_TYPE,
	GEE_HASH_MAP_KEY_ITERATOR_V_DUP_FUNC,
	GEE_HASH_MAP_KEY_ITERATOR_V_DESTROY_FUNC
};
static void gee_hash_map_key_iterator_set_map (GeeHashMapKeyIterator* self, GeeHashMap* value);
static gboolean gee_hash_map_key_iterator_real_next (GeeIterator* base);
static gpointer gee_hash_map_key_iterator_real_get (GeeIterator* base);
static void gee_hash_map_key_iterator_finalize (GObject* obj);
static void gee_hash_map_key_iterator_get_property (GObject * object, guint property_id, GValue * value, GParamSpec * pspec);
static void gee_hash_map_key_iterator_set_property (GObject * object, guint property_id, const GValue * value, GParamSpec * pspec);
#define GEE_HASH_MAP_VALUE_COLLECTION_GET_PRIVATE(o) (G_TYPE_INSTANCE_GET_PRIVATE ((o), GEE_HASH_MAP_TYPE_VALUE_COLLECTION, GeeHashMapValueCollectionPrivate))
enum  {
	GEE_HASH_MAP_VALUE_COLLECTION_DUMMY_PROPERTY,
	GEE_HASH_MAP_VALUE_COLLECTION_MAP,
	GEE_HASH_MAP_VALUE_COLLECTION_SIZE,
	GEE_HASH_MAP_VALUE_COLLECTION_K_TYPE,
	GEE_HASH_MAP_VALUE_COLLECTION_K_DUP_FUNC,
	GEE_HASH_MAP_VALUE_COLLECTION_K_DESTROY_FUNC,
	GEE_HASH_MAP_VALUE_COLLECTION_V_TYPE,
	GEE_HASH_MAP_VALUE_COLLECTION_V_DUP_FUNC,
	GEE_HASH_MAP_VALUE_COLLECTION_V_DESTROY_FUNC
};
static void gee_hash_map_value_collection_set_map (GeeHashMapValueCollection* self, GeeHashMap* value);
static GeeHashMap* gee_hash_map_value_collection_get_map (GeeHashMapValueCollection* self);
static GeeHashMapValueIterator* gee_hash_map_value_iterator_new (GType k_type, GBoxedCopyFunc k_dup_func, GDestroyNotify k_destroy_func, GType v_type, GBoxedCopyFunc v_dup_func, GDestroyNotify v_destroy_func, GeeHashMap* map);
static GeeHashMapValueIterator* gee_hash_map_value_iterator_construct (GType object_type, GType k_type, GBoxedCopyFunc k_dup_func, GDestroyNotify k_destroy_func, GType v_type, GBoxedCopyFunc v_dup_func, GDestroyNotify v_destroy_func, GeeHashMap* map);
static GType gee_hash_map_value_iterator_get_type (void);
static GeeIterator* gee_hash_map_value_collection_real_iterator (GeeAbstractCollection* base);
static gboolean gee_hash_map_value_collection_real_add (GeeAbstractCollection* base, gconstpointer value);
static void gee_hash_map_value_collection_real_clear (GeeAbstractCollection* base);
static gboolean gee_hash_map_value_collection_real_remove (GeeAbstractCollection* base, gconstpointer value);
GeeIterator* gee_abstract_collection_iterator (GeeAbstractCollection* self);
gboolean gee_iterator_next (GeeIterator* self);
gpointer gee_iterator_get (GeeIterator* self);
static gboolean gee_hash_map_value_collection_real_contains (GeeAbstractCollection* base, gconstpointer value);
static gboolean gee_hash_map_value_collection_real_add_all (GeeAbstractCollection* base, GeeCollection* collection);
static gboolean gee_hash_map_value_collection_real_remove_all (GeeAbstractCollection* base, GeeCollection* collection);
static gboolean gee_hash_map_value_collection_real_retain_all (GeeAbstractCollection* base, GeeCollection* collection);
static void gee_hash_map_value_collection_finalize (GObject* obj);
static void gee_hash_map_value_collection_get_property (GObject * object, guint property_id, GValue * value, GParamSpec * pspec);
static void gee_hash_map_value_collection_set_property (GObject * object, guint property_id, const GValue * value, GParamSpec * pspec);
#define GEE_HASH_MAP_VALUE_ITERATOR_GET_PRIVATE(o) (G_TYPE_INSTANCE_GET_PRIVATE ((o), GEE_HASH_MAP_TYPE_VALUE_ITERATOR, GeeHashMapValueIteratorPrivate))
enum  {
	GEE_HASH_MAP_VALUE_ITERATOR_DUMMY_PROPERTY,
	GEE_HASH_MAP_VALUE_ITERATOR_MAP,
	GEE_HASH_MAP_VALUE_ITERATOR_K_TYPE,
	GEE_HASH_MAP_VALUE_ITERATOR_K_DUP_FUNC,
	GEE_HASH_MAP_VALUE_ITERATOR_K_DESTROY_FUNC,
	GEE_HASH_MAP_VALUE_ITERATOR_V_TYPE,
	GEE_HASH_MAP_VALUE_ITERATOR_V_DUP_FUNC,
	GEE_HASH_MAP_VALUE_ITERATOR_V_DESTROY_FUNC
};
static void gee_hash_map_value_iterator_set_map (GeeHashMapValueIterator* self, GeeHashMap* value);
static gboolean gee_hash_map_value_iterator_real_next (GeeIterator* base);
static gpointer gee_hash_map_value_iterator_real_get (GeeIterator* base);
static void gee_hash_map_value_iterator_finalize (GObject* obj);
static void gee_hash_map_value_iterator_get_property (GObject * object, guint property_id, GValue * value, GParamSpec * pspec);
static void gee_hash_map_value_iterator_set_property (GObject * object, guint property_id, const GValue * value, GParamSpec * pspec);
static void gee_hash_map_finalize (GObject* obj);
static void gee_hash_map_get_property (GObject * object, guint property_id, GValue * value, GParamSpec * pspec);
static void gee_hash_map_set_property (GObject * object, guint property_id, const GValue * value, GParamSpec * pspec);
static void _vala_array_destroy (gpointer array, gint array_length, GDestroyNotify destroy_func);
static void _vala_array_free (gpointer array, gint array_length, GDestroyNotify destroy_func);



GeeHashMap* gee_hash_map_construct (GType object_type, GType k_type, GBoxedCopyFunc k_dup_func, GDestroyNotify k_destroy_func, GType v_type, GBoxedCopyFunc v_dup_func, GDestroyNotify v_destroy_func, GHashFunc key_hash_func, GEqualFunc key_equal_func, GEqualFunc value_equal_func) {
	GParameter * __params;
	GParameter * __params_it;
	GeeHashMap * self;
	__params = g_new0 (GParameter, 9);
	__params_it = __params;
	__params_it->name = "k-type";
	g_value_init (&__params_it->value, G_TYPE_GTYPE);
	g_value_set_gtype (&__params_it->value, k_type);
	__params_it++;
	__params_it->name = "k-dup-func";
	g_value_init (&__params_it->value, G_TYPE_POINTER);
	g_value_set_pointer (&__params_it->value, k_dup_func);
	__params_it++;
	__params_it->name = "k-destroy-func";
	g_value_init (&__params_it->value, G_TYPE_POINTER);
	g_value_set_pointer (&__params_it->value, k_destroy_func);
	__params_it++;
	__params_it->name = "v-type";
	g_value_init (&__params_it->value, G_TYPE_GTYPE);
	g_value_set_gtype (&__params_it->value, v_type);
	__params_it++;
	__params_it->name = "v-dup-func";
	g_value_init (&__params_it->value, G_TYPE_POINTER);
	g_value_set_pointer (&__params_it->value, v_dup_func);
	__params_it++;
	__params_it->name = "v-destroy-func";
	g_value_init (&__params_it->value, G_TYPE_POINTER);
	g_value_set_pointer (&__params_it->value, v_destroy_func);
	__params_it++;
	__params_it->name = "key-hash-func";
	g_value_init (&__params_it->value, G_TYPE_POINTER);
	g_value_set_pointer (&__params_it->value, key_hash_func);
	__params_it++;
	__params_it->name = "key-equal-func";
	g_value_init (&__params_it->value, G_TYPE_POINTER);
	g_value_set_pointer (&__params_it->value, key_equal_func);
	__params_it++;
	__params_it->name = "value-equal-func";
	g_value_init (&__params_it->value, G_TYPE_POINTER);
	g_value_set_pointer (&__params_it->value, value_equal_func);
	__params_it++;
	self = g_object_newv (object_type, __params_it - __params, __params);
	while (__params_it > __params) {
		--__params_it;
		g_value_unset (&__params_it->value);
	}
	g_free (__params);
	return self;
}


GeeHashMap* gee_hash_map_new (GType k_type, GBoxedCopyFunc k_dup_func, GDestroyNotify k_destroy_func, GType v_type, GBoxedCopyFunc v_dup_func, GDestroyNotify v_destroy_func, GHashFunc key_hash_func, GEqualFunc key_equal_func, GEqualFunc value_equal_func) {
	return gee_hash_map_construct (GEE_TYPE_HASH_MAP, k_type, k_dup_func, k_destroy_func, v_type, v_dup_func, v_destroy_func, key_hash_func, key_equal_func, value_equal_func);
}


static GeeSet* gee_hash_map_real_get_keys (GeeAbstractMap* base) {
	GeeHashMap * self;
	GeeSet* result;
	self = (GeeHashMap*) base;
	result = (GeeSet*) gee_hash_map_key_set_new (self->priv->k_type, (GBoxedCopyFunc) self->priv->k_dup_func, self->priv->k_destroy_func, self->priv->v_type, (GBoxedCopyFunc) self->priv->v_dup_func, self->priv->v_destroy_func, self);
	return result;
}


static GeeCollection* gee_hash_map_real_get_values (GeeAbstractMap* base) {
	GeeHashMap * self;
	GeeCollection* result;
	self = (GeeHashMap*) base;
	result = (GeeCollection*) gee_hash_map_value_collection_new (self->priv->k_type, (GBoxedCopyFunc) self->priv->k_dup_func, self->priv->k_destroy_func, self->priv->v_type, (GBoxedCopyFunc) self->priv->v_dup_func, self->priv->v_destroy_func, self);
	return result;
}


static GeeHashMapNode** gee_hash_map_lookup_node (GeeHashMap* self, gconstpointer key) {
	GeeHashMapNode** result;
	guint hash_value;
	GeeHashMapNode** node;
	g_return_val_if_fail (self != NULL, NULL);
	hash_value = gee_hash_map_get_key_hash_func (self) (key);
	node = &self->priv->_nodes[hash_value % self->priv->_array_size];
	while (TRUE) {
		gboolean _tmp0_;
		_tmp0_ = FALSE;
		if ((*node) != NULL) {
			gboolean _tmp1_;
			_tmp1_ = FALSE;
			if (hash_value != (*node)->key_hash) {
				_tmp1_ = TRUE;
			} else {
				_tmp1_ = !gee_hash_map_get_key_equal_func (self) ((*node)->key, key);
			}
			_tmp0_ = _tmp1_;
		} else {
			_tmp0_ = FALSE;
		}
		if (!_tmp0_) {
			break;
		}
		node = &(*node)->next;
	}
	result = node;
	return result;
}


static gboolean gee_hash_map_real_contains (GeeAbstractMap* base, gconstpointer key) {
	GeeHashMap * self;
	gboolean result;
	GeeHashMapNode** node;
	self = (GeeHashMap*) base;
	node = gee_hash_map_lookup_node (self, key);
	result = (*node) != NULL;
	return result;
}


static gpointer gee_hash_map_real_get (GeeAbstractMap* base, gconstpointer key) {
	GeeHashMap * self;
	gpointer result;
	GeeHashMapNode* node;
	self = (GeeHashMap*) base;
	node = *gee_hash_map_lookup_node (self, key);
	if (node != NULL) {
		gconstpointer _tmp0_;
		_tmp0_ = NULL;
		result = (_tmp0_ = node->value, ((_tmp0_ == NULL) || (self->priv->v_dup_func == NULL)) ? ((gpointer) _tmp0_) : self->priv->v_dup_func ((gpointer) _tmp0_));
		return result;
	} else {
		result = NULL;
		return result;
	}
}


static void gee_hash_map_real_set (GeeAbstractMap* base, gconstpointer key, gconstpointer value) {
	GeeHashMap * self;
	GeeHashMapNode** node;
	self = (GeeHashMap*) base;
	node = gee_hash_map_lookup_node (self, key);
	if ((*node) != NULL) {
		gpointer _tmp1_;
		gconstpointer _tmp0_;
		_tmp1_ = NULL;
		_tmp0_ = NULL;
		(*node)->value = (_tmp1_ = (_tmp0_ = value, ((_tmp0_ == NULL) || (self->priv->v_dup_func == NULL)) ? ((gpointer) _tmp0_) : self->priv->v_dup_func ((gpointer) _tmp0_)), (((*node)->value == NULL) || (self->priv->v_destroy_func == NULL)) ? NULL : ((*node)->value = (self->priv->v_destroy_func ((*node)->value), NULL)), _tmp1_);
	} else {
		guint hash_value;
		gconstpointer _tmp3_;
		gconstpointer _tmp2_;
		hash_value = gee_hash_map_get_key_hash_func (self) (key);
		_tmp3_ = NULL;
		_tmp2_ = NULL;
		*node = gee_hash_map_node_new ((_tmp2_ = key, ((_tmp2_ == NULL) || (self->priv->k_dup_func == NULL)) ? ((gpointer) _tmp2_) : self->priv->k_dup_func ((gpointer) _tmp2_)), (_tmp3_ = value, ((_tmp3_ == NULL) || (self->priv->v_dup_func == NULL)) ? ((gpointer) _tmp3_) : self->priv->v_dup_func ((gpointer) _tmp3_)), hash_value);
		self->priv->_nnodes++;
		gee_hash_map_resize (self);
	}
	self->priv->_stamp++;
}


static gboolean gee_hash_map_real_remove (GeeAbstractMap* base, gconstpointer key, gpointer* value) {
	GeeHashMap * self;
	gboolean result;
	GeeHashMapNode** node;
	self = (GeeHashMap*) base;
	node = gee_hash_map_lookup_node (self, key);
	if ((*node) != NULL) {
		GeeHashMapNode* _tmp0_;
		GeeHashMapNode* next;
		gpointer _tmp3_;
		gpointer _tmp4_;
		GeeHashMapNode* _tmp5_;
		_tmp0_ = NULL;
		next = (_tmp0_ = (*node)->next, (*node)->next = NULL, _tmp0_);
		if ((&(*value)) != NULL) {
			gpointer _tmp2_;
			gpointer _tmp1_;
			_tmp2_ = NULL;
			_tmp1_ = NULL;
			(*value) = (_tmp2_ = (_tmp1_ = (*node)->value, (*node)->value = NULL, _tmp1_), (((*value) == NULL) || (self->priv->v_destroy_func == NULL)) ? NULL : ((*value) = (self->priv->v_destroy_func ((*value)), NULL)), _tmp2_);
		}
		_tmp3_ = NULL;
		(*node)->key = (_tmp3_ = NULL, (((*node)->key == NULL) || (self->priv->k_destroy_func == NULL)) ? NULL : ((*node)->key = (self->priv->k_destroy_func ((*node)->key), NULL)), _tmp3_);
		_tmp4_ = NULL;
		(*node)->value = (_tmp4_ = NULL, (((*node)->value == NULL) || (self->priv->v_destroy_func == NULL)) ? NULL : ((*node)->value = (self->priv->v_destroy_func ((*node)->value), NULL)), _tmp4_);
		gee_hash_map_node_free (*node);
		_tmp5_ = NULL;
		*node = (_tmp5_ = next, next = NULL, _tmp5_);
		self->priv->_nnodes--;
		gee_hash_map_resize (self);
		self->priv->_stamp++;
		result = TRUE;
		(next == NULL) ? NULL : (next = (gee_hash_map_node_free (next), NULL));
		return result;
	}
	result = FALSE;
	return result;
}


static void gee_hash_map_real_clear (GeeAbstractMap* base) {
	GeeHashMap * self;
	self = (GeeHashMap*) base;
	{
		gint i;
		i = 0;
		{
			gboolean _tmp0_;
			_tmp0_ = TRUE;
			while (TRUE) {
				GeeHashMapNode* _tmp1_;
				GeeHashMapNode* node;
				if (!_tmp0_) {
					i++;
				}
				_tmp0_ = FALSE;
				if (!(i < self->priv->_array_size)) {
					break;
				}
				_tmp1_ = NULL;
				node = (_tmp1_ = self->priv->_nodes[i], self->priv->_nodes[i] = NULL, _tmp1_);
				while (TRUE) {
					GeeHashMapNode* _tmp2_;
					GeeHashMapNode* next;
					gpointer _tmp3_;
					gpointer _tmp4_;
					GeeHashMapNode* _tmp6_;
					GeeHashMapNode* _tmp5_;
					if (!(node != NULL)) {
						break;
					}
					_tmp2_ = NULL;
					next = (_tmp2_ = node->next, node->next = NULL, _tmp2_);
					_tmp3_ = NULL;
					node->key = (_tmp3_ = NULL, ((node->key == NULL) || (self->priv->k_destroy_func == NULL)) ? NULL : (node->key = (self->priv->k_destroy_func (node->key), NULL)), _tmp3_);
					_tmp4_ = NULL;
					node->value = (_tmp4_ = NULL, ((node->value == NULL) || (self->priv->v_destroy_func == NULL)) ? NULL : (node->value = (self->priv->v_destroy_func (node->value), NULL)), _tmp4_);
					_tmp6_ = NULL;
					_tmp5_ = NULL;
					node = (_tmp6_ = (_tmp5_ = next, next = NULL, _tmp5_), (node == NULL) ? NULL : (node = (gee_hash_map_node_free (node), NULL)), _tmp6_);
					(next == NULL) ? NULL : (next = (gee_hash_map_node_free (next), NULL));
				}
				(node == NULL) ? NULL : (node = (gee_hash_map_node_free (node), NULL));
			}
		}
	}
	self->priv->_nnodes = 0;
	gee_hash_map_resize (self);
}


static void gee_hash_map_resize (GeeHashMap* self) {
	gboolean _tmp0_;
	gboolean _tmp1_;
	g_return_if_fail (self != NULL);
	_tmp0_ = FALSE;
	_tmp1_ = FALSE;
	if (self->priv->_array_size >= (3 * self->priv->_nnodes)) {
		_tmp1_ = self->priv->_array_size >= GEE_HASH_MAP_MIN_SIZE;
	} else {
		_tmp1_ = FALSE;
	}
	if (_tmp1_) {
		_tmp0_ = TRUE;
	} else {
		gboolean _tmp2_;
		_tmp2_ = FALSE;
		if ((3 * self->priv->_array_size) <= self->priv->_nnodes) {
			_tmp2_ = self->priv->_array_size < GEE_HASH_MAP_MAX_SIZE;
		} else {
			_tmp2_ = FALSE;
		}
		_tmp0_ = _tmp2_;
	}
	if (_tmp0_) {
		gint new_array_size;
		GeeHashMapNode** _tmp3_;
		gint new_nodes_size;
		gint new_nodes_length1;
		GeeHashMapNode** new_nodes;
		GeeHashMapNode** _tmp17_;
		GeeHashMapNode** _tmp16_;
		new_array_size = (gint) g_spaced_primes_closest ((guint) self->priv->_nnodes);
		new_array_size = CLAMP (new_array_size, GEE_HASH_MAP_MIN_SIZE, GEE_HASH_MAP_MAX_SIZE);
		_tmp3_ = NULL;
		new_nodes = (_tmp3_ = g_new0 (GeeHashMapNode*, new_array_size + 1), new_nodes_length1 = new_array_size, new_nodes_size = new_nodes_length1, _tmp3_);
		{
			gint i;
			i = 0;
			{
				gboolean _tmp4_;
				_tmp4_ = TRUE;
				while (TRUE) {
					GeeHashMapNode* node;
					GeeHashMapNode* next;
					if (!_tmp4_) {
						i++;
					}
					_tmp4_ = FALSE;
					if (!(i < self->priv->_array_size)) {
						break;
					}
					node = NULL;
					next = NULL;
					{
						GeeHashMapNode* _tmp6_;
						GeeHashMapNode* _tmp5_;
						gboolean _tmp7_;
						_tmp6_ = NULL;
						_tmp5_ = NULL;
						node = (_tmp6_ = (_tmp5_ = self->priv->_nodes[i], self->priv->_nodes[i] = NULL, _tmp5_), (node == NULL) ? NULL : (node = (gee_hash_map_node_free (node), NULL)), _tmp6_);
						_tmp7_ = TRUE;
						while (TRUE) {
							GeeHashMapNode* _tmp11_;
							GeeHashMapNode* _tmp10_;
							guint hash_val;
							GeeHashMapNode* _tmp13_;
							GeeHashMapNode* _tmp12_;
							GeeHashMapNode* _tmp15_;
							GeeHashMapNode* _tmp14_;
							if (!_tmp7_) {
								GeeHashMapNode* _tmp9_;
								GeeHashMapNode* _tmp8_;
								_tmp9_ = NULL;
								_tmp8_ = NULL;
								node = (_tmp9_ = (_tmp8_ = next, next = NULL, _tmp8_), (node == NULL) ? NULL : (node = (gee_hash_map_node_free (node), NULL)), _tmp9_);
							}
							_tmp7_ = FALSE;
							if (!(node != NULL)) {
								break;
							}
							_tmp11_ = NULL;
							_tmp10_ = NULL;
							next = (_tmp11_ = (_tmp10_ = node->next, node->next = NULL, _tmp10_), (next == NULL) ? NULL : (next = (gee_hash_map_node_free (next), NULL)), _tmp11_);
							hash_val = node->key_hash % new_array_size;
							_tmp13_ = NULL;
							_tmp12_ = NULL;
							node->next = (_tmp13_ = (_tmp12_ = new_nodes[hash_val], new_nodes[hash_val] = NULL, _tmp12_), (node->next == NULL) ? NULL : (node->next = (gee_hash_map_node_free (node->next), NULL)), _tmp13_);
							_tmp15_ = NULL;
							_tmp14_ = NULL;
							new_nodes[hash_val] = (_tmp15_ = (_tmp14_ = node, node = NULL, _tmp14_), (new_nodes[hash_val] == NULL) ? NULL : (new_nodes[hash_val] = (gee_hash_map_node_free (new_nodes[hash_val]), NULL)), _tmp15_);
						}
					}
					(node == NULL) ? NULL : (node = (gee_hash_map_node_free (node), NULL));
					(next == NULL) ? NULL : (next = (gee_hash_map_node_free (next), NULL));
				}
			}
		}
		_tmp17_ = NULL;
		_tmp16_ = NULL;
		self->priv->_nodes = (_tmp17_ = (_tmp16_ = new_nodes, new_nodes = NULL, _tmp16_), self->priv->_nodes = (_vala_array_free (self->priv->_nodes, self->priv->_nodes_length1, (GDestroyNotify) gee_hash_map_node_free), NULL), self->priv->_nodes_length1 = new_nodes_length1, self->priv->_nodes_size = self->priv->_nodes_length1, _tmp17_);
		self->priv->_array_size = new_array_size;
		new_nodes = (_vala_array_free (new_nodes, new_nodes_length1, (GDestroyNotify) gee_hash_map_node_free), NULL);
	}
}


static gint gee_hash_map_real_get_size (GeeAbstractMap* base) {
	gint result;
	GeeHashMap* self;
	self = (GeeHashMap*) base;
	result = self->priv->_nnodes;
	return result;
}


GHashFunc gee_hash_map_get_key_hash_func (GeeHashMap* self) {
	GHashFunc result;
	g_return_val_if_fail (self != NULL, NULL);
	result = self->priv->_key_hash_func;
	return result;
}


static void gee_hash_map_set_key_hash_func (GeeHashMap* self, GHashFunc value) {
	g_return_if_fail (self != NULL);
	self->priv->_key_hash_func = value;
	g_object_notify ((GObject *) self, "key-hash-func");
}


GEqualFunc gee_hash_map_get_key_equal_func (GeeHashMap* self) {
	GEqualFunc result;
	g_return_val_if_fail (self != NULL, NULL);
	result = self->priv->_key_equal_func;
	return result;
}


static void gee_hash_map_set_key_equal_func (GeeHashMap* self, GEqualFunc value) {
	g_return_if_fail (self != NULL);
	self->priv->_key_equal_func = value;
	g_object_notify ((GObject *) self, "key-equal-func");
}


GEqualFunc gee_hash_map_get_value_equal_func (GeeHashMap* self) {
	GEqualFunc result;
	g_return_val_if_fail (self != NULL, NULL);
	result = self->priv->_value_equal_func;
	return result;
}


static void gee_hash_map_set_value_equal_func (GeeHashMap* self, GEqualFunc value) {
	g_return_if_fail (self != NULL);
	self->priv->_value_equal_func = value;
	g_object_notify ((GObject *) self, "value-equal-func");
}


static GObject * gee_hash_map_constructor (GType type, guint n_construct_properties, GObjectConstructParam * construct_properties) {
	GObject * obj;
	GeeHashMapClass * klass;
	GObjectClass * parent_class;
	GeeHashMap * self;
	klass = GEE_HASH_MAP_CLASS (g_type_class_peek (GEE_TYPE_HASH_MAP));
	parent_class = G_OBJECT_CLASS (g_type_class_peek_parent (klass));
	obj = parent_class->constructor (type, n_construct_properties, construct_properties);
	self = GEE_HASH_MAP (obj);
	{
		GeeHashMapNode** _tmp0_;
		self->priv->_array_size = GEE_HASH_MAP_MIN_SIZE;
		_tmp0_ = NULL;
		self->priv->_nodes = (_tmp0_ = g_new0 (GeeHashMapNode*, self->priv->_array_size + 1), self->priv->_nodes = (_vala_array_free (self->priv->_nodes, self->priv->_nodes_length1, (GDestroyNotify) gee_hash_map_node_free), NULL), self->priv->_nodes_length1 = self->priv->_array_size, self->priv->_nodes_size = self->priv->_nodes_length1, _tmp0_);
	}
	return obj;
}


static GeeHashMapNode* gee_hash_map_node_new (gpointer k, gpointer v, guint hash) {
	GeeHashMapNode* self;
	gpointer _tmp0_;
	gpointer _tmp1_;
	self = g_slice_new0 (GeeHashMapNode);
	gee_hash_map_node_instance_init (self);
	_tmp0_ = NULL;
	self->key = (_tmp0_ = k, k = NULL, _tmp0_);
	_tmp1_ = NULL;
	self->value = (_tmp1_ = v, v = NULL, _tmp1_);
	self->key_hash = hash;
	return self;
}


static void gee_hash_map_node_instance_init (GeeHashMapNode * self) {
}


static void gee_hash_map_node_free (GeeHashMapNode* self) {
	(self->next == NULL) ? NULL : (self->next = (gee_hash_map_node_free (self->next), NULL));
	g_slice_free (GeeHashMapNode, self);
}


static GeeHashMapKeySet* gee_hash_map_key_set_construct (GType object_type, GType k_type, GBoxedCopyFunc k_dup_func, GDestroyNotify k_destroy_func, GType v_type, GBoxedCopyFunc v_dup_func, GDestroyNotify v_destroy_func, GeeHashMap* map) {
	GParameter * __params;
	GParameter * __params_it;
	GeeHashMapKeySet * self;
	g_return_val_if_fail (map != NULL, NULL);
	__params = g_new0 (GParameter, 7);
	__params_it = __params;
	__params_it->name = "k-type";
	g_value_init (&__params_it->value, G_TYPE_GTYPE);
	g_value_set_gtype (&__params_it->value, k_type);
	__params_it++;
	__params_it->name = "k-dup-func";
	g_value_init (&__params_it->value, G_TYPE_POINTER);
	g_value_set_pointer (&__params_it->value, k_dup_func);
	__params_it++;
	__params_it->name = "k-destroy-func";
	g_value_init (&__params_it->value, G_TYPE_POINTER);
	g_value_set_pointer (&__params_it->value, k_destroy_func);
	__params_it++;
	__params_it->name = "v-type";
	g_value_init (&__params_it->value, G_TYPE_GTYPE);
	g_value_set_gtype (&__params_it->value, v_type);
	__params_it++;
	__params_it->name = "v-dup-func";
	g_value_init (&__params_it->value, G_TYPE_POINTER);
	g_value_set_pointer (&__params_it->value, v_dup_func);
	__params_it++;
	__params_it->name = "v-destroy-func";
	g_value_init (&__params_it->value, G_TYPE_POINTER);
	g_value_set_pointer (&__params_it->value, v_destroy_func);
	__params_it++;
	__params_it->name = "map";
	g_value_init (&__params_it->value, GEE_TYPE_HASH_MAP);
	g_value_set_object (&__params_it->value, map);
	__params_it++;
	self = g_object_newv (object_type, __params_it - __params, __params);
	while (__params_it > __params) {
		--__params_it;
		g_value_unset (&__params_it->value);
	}
	g_free (__params);
	return self;
}


static GeeHashMapKeySet* gee_hash_map_key_set_new (GType k_type, GBoxedCopyFunc k_dup_func, GDestroyNotify k_destroy_func, GType v_type, GBoxedCopyFunc v_dup_func, GDestroyNotify v_destroy_func, GeeHashMap* map) {
	return gee_hash_map_key_set_construct (GEE_HASH_MAP_TYPE_KEY_SET, k_type, k_dup_func, k_destroy_func, v_type, v_dup_func, v_destroy_func, map);
}


static GeeIterator* gee_hash_map_key_set_real_iterator (GeeAbstractCollection* base) {
	GeeHashMapKeySet * self;
	GeeIterator* result;
	self = (GeeHashMapKeySet*) base;
	result = (GeeIterator*) gee_hash_map_key_iterator_new (self->priv->k_type, (GBoxedCopyFunc) self->priv->k_dup_func, self->priv->k_destroy_func, self->priv->v_type, (GBoxedCopyFunc) self->priv->v_dup_func, self->priv->v_destroy_func, self->priv->_map);
	return result;
}


static gboolean gee_hash_map_key_set_real_add (GeeAbstractCollection* base, gconstpointer key) {
	GeeHashMapKeySet * self;
	gboolean result;
	self = (GeeHashMapKeySet*) base;
	g_assert_not_reached ();
}


static void gee_hash_map_key_set_real_clear (GeeAbstractCollection* base) {
	GeeHashMapKeySet * self;
	self = (GeeHashMapKeySet*) base;
	g_assert_not_reached ();
}


static gboolean gee_hash_map_key_set_real_remove (GeeAbstractCollection* base, gconstpointer key) {
	GeeHashMapKeySet * self;
	gboolean result;
	self = (GeeHashMapKeySet*) base;
	g_assert_not_reached ();
}


static gboolean gee_hash_map_key_set_real_contains (GeeAbstractCollection* base, gconstpointer key) {
	GeeHashMapKeySet * self;
	gboolean result;
	self = (GeeHashMapKeySet*) base;
	result = gee_abstract_map_contains ((GeeAbstractMap*) self->priv->_map, key);
	return result;
}


static gboolean gee_hash_map_key_set_real_add_all (GeeAbstractCollection* base, GeeCollection* collection) {
	GeeHashMapKeySet * self;
	gboolean result;
	self = (GeeHashMapKeySet*) base;
	g_return_val_if_fail (collection != NULL, FALSE);
	g_assert_not_reached ();
}


static gboolean gee_hash_map_key_set_real_remove_all (GeeAbstractCollection* base, GeeCollection* collection) {
	GeeHashMapKeySet * self;
	gboolean result;
	self = (GeeHashMapKeySet*) base;
	g_return_val_if_fail (collection != NULL, FALSE);
	g_assert_not_reached ();
}


static gboolean gee_hash_map_key_set_real_retain_all (GeeAbstractCollection* base, GeeCollection* collection) {
	GeeHashMapKeySet * self;
	gboolean result;
	self = (GeeHashMapKeySet*) base;
	g_return_val_if_fail (collection != NULL, FALSE);
	g_assert_not_reached ();
}


static GeeHashMap* gee_hash_map_key_set_get_map (GeeHashMapKeySet* self) {
	GeeHashMap* result;
	g_return_val_if_fail (self != NULL, NULL);
	result = self->priv->_map;
	return result;
}


static void gee_hash_map_key_set_set_map (GeeHashMapKeySet* self, GeeHashMap* value) {
	GeeHashMap* _tmp1_;
	GeeHashMap* _tmp0_;
	g_return_if_fail (self != NULL);
	_tmp1_ = NULL;
	_tmp0_ = NULL;
	self->priv->_map = (_tmp1_ = (_tmp0_ = value, (_tmp0_ == NULL) ? NULL : g_object_ref (_tmp0_)), (self->priv->_map == NULL) ? NULL : (self->priv->_map = (g_object_unref (self->priv->_map), NULL)), _tmp1_);
	g_object_notify ((GObject *) self, "map");
}


static gint gee_hash_map_key_set_real_get_size (GeeAbstractCollection* base) {
	gint result;
	GeeHashMapKeySet* self;
	self = (GeeHashMapKeySet*) base;
	result = gee_map_get_size ((GeeMap*) self->priv->_map);
	return result;
}


static void gee_hash_map_key_set_class_init (GeeHashMapKeySetClass * klass) {
	gee_hash_map_key_set_parent_class = g_type_class_peek_parent (klass);
	g_type_class_add_private (klass, sizeof (GeeHashMapKeySetPrivate));
	GEE_ABSTRACT_COLLECTION_CLASS (klass)->iterator = gee_hash_map_key_set_real_iterator;
	GEE_ABSTRACT_COLLECTION_CLASS (klass)->add = gee_hash_map_key_set_real_add;
	GEE_ABSTRACT_COLLECTION_CLASS (klass)->clear = gee_hash_map_key_set_real_clear;
	GEE_ABSTRACT_COLLECTION_CLASS (klass)->remove = gee_hash_map_key_set_real_remove;
	GEE_ABSTRACT_COLLECTION_CLASS (klass)->contains = gee_hash_map_key_set_real_contains;
	GEE_ABSTRACT_COLLECTION_CLASS (klass)->add_all = gee_hash_map_key_set_real_add_all;
	GEE_ABSTRACT_COLLECTION_CLASS (klass)->remove_all = gee_hash_map_key_set_real_remove_all;
	GEE_ABSTRACT_COLLECTION_CLASS (klass)->retain_all = gee_hash_map_key_set_real_retain_all;
	GEE_ABSTRACT_COLLECTION_CLASS (klass)->get_size = gee_hash_map_key_set_real_get_size;
	G_OBJECT_CLASS (klass)->get_property = gee_hash_map_key_set_get_property;
	G_OBJECT_CLASS (klass)->set_property = gee_hash_map_key_set_set_property;
	G_OBJECT_CLASS (klass)->finalize = gee_hash_map_key_set_finalize;
	g_object_class_install_property (G_OBJECT_CLASS (klass), GEE_HASH_MAP_KEY_SET_K_TYPE, g_param_spec_gtype ("k-type", "type", "type", G_TYPE_NONE, G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB | G_PARAM_WRITABLE | G_PARAM_CONSTRUCT_ONLY));
	g_object_class_install_property (G_OBJECT_CLASS (klass), GEE_HASH_MAP_KEY_SET_K_DUP_FUNC, g_param_spec_pointer ("k-dup-func", "dup func", "dup func", G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB | G_PARAM_WRITABLE | G_PARAM_CONSTRUCT_ONLY));
	g_object_class_install_property (G_OBJECT_CLASS (klass), GEE_HASH_MAP_KEY_SET_K_DESTROY_FUNC, g_param_spec_pointer ("k-destroy-func", "destroy func", "destroy func", G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB | G_PARAM_WRITABLE | G_PARAM_CONSTRUCT_ONLY));
	g_object_class_install_property (G_OBJECT_CLASS (klass), GEE_HASH_MAP_KEY_SET_V_TYPE, g_param_spec_gtype ("v-type", "type", "type", G_TYPE_NONE, G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB | G_PARAM_WRITABLE | G_PARAM_CONSTRUCT_ONLY));
	g_object_class_install_property (G_OBJECT_CLASS (klass), GEE_HASH_MAP_KEY_SET_V_DUP_FUNC, g_param_spec_pointer ("v-dup-func", "dup func", "dup func", G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB | G_PARAM_WRITABLE | G_PARAM_CONSTRUCT_ONLY));
	g_object_class_install_property (G_OBJECT_CLASS (klass), GEE_HASH_MAP_KEY_SET_V_DESTROY_FUNC, g_param_spec_pointer ("v-destroy-func", "destroy func", "destroy func", G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB | G_PARAM_WRITABLE | G_PARAM_CONSTRUCT_ONLY));
	g_object_class_install_property (G_OBJECT_CLASS (klass), GEE_HASH_MAP_KEY_SET_MAP, g_param_spec_object ("map", "map", "map", GEE_TYPE_HASH_MAP, G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB | G_PARAM_READABLE | G_PARAM_WRITABLE | G_PARAM_CONSTRUCT_ONLY));
	g_object_class_override_property (G_OBJECT_CLASS (klass), GEE_HASH_MAP_KEY_SET_SIZE, "size");
}


static void gee_hash_map_key_set_gee_set_interface_init (GeeSetIface * iface) {
	gee_hash_map_key_set_gee_set_parent_iface = g_type_interface_peek_parent (iface);
}


static void gee_hash_map_key_set_instance_init (GeeHashMapKeySet * self) {
	self->priv = GEE_HASH_MAP_KEY_SET_GET_PRIVATE (self);
}


static void gee_hash_map_key_set_finalize (GObject* obj) {
	GeeHashMapKeySet * self;
	self = GEE_HASH_MAP_KEY_SET (obj);
	(self->priv->_map == NULL) ? NULL : (self->priv->_map = (g_object_unref (self->priv->_map), NULL));
	G_OBJECT_CLASS (gee_hash_map_key_set_parent_class)->finalize (obj);
}


static GType gee_hash_map_key_set_get_type (void) {
	static GType gee_hash_map_key_set_type_id = 0;
	if (gee_hash_map_key_set_type_id == 0) {
		static const GTypeInfo g_define_type_info = { sizeof (GeeHashMapKeySetClass), (GBaseInitFunc) NULL, (GBaseFinalizeFunc) NULL, (GClassInitFunc) gee_hash_map_key_set_class_init, (GClassFinalizeFunc) NULL, NULL, sizeof (GeeHashMapKeySet), 0, (GInstanceInitFunc) gee_hash_map_key_set_instance_init, NULL };
		static const GInterfaceInfo gee_set_info = { (GInterfaceInitFunc) gee_hash_map_key_set_gee_set_interface_init, (GInterfaceFinalizeFunc) NULL, NULL};
		gee_hash_map_key_set_type_id = g_type_register_static (GEE_TYPE_ABSTRACT_COLLECTION, "GeeHashMapKeySet", &g_define_type_info, 0);
		g_type_add_interface_static (gee_hash_map_key_set_type_id, GEE_TYPE_SET, &gee_set_info);
	}
	return gee_hash_map_key_set_type_id;
}


static void gee_hash_map_key_set_get_property (GObject * object, guint property_id, GValue * value, GParamSpec * pspec) {
	GeeHashMapKeySet * self;
	gpointer boxed;
	self = GEE_HASH_MAP_KEY_SET (object);
	switch (property_id) {
		case GEE_HASH_MAP_KEY_SET_MAP:
		g_value_set_object (value, gee_hash_map_key_set_get_map (self));
		break;
		case GEE_HASH_MAP_KEY_SET_SIZE:
		g_value_set_int (value, gee_abstract_collection_get_size ((GeeAbstractCollection*) self));
		break;
		default:
		G_OBJECT_WARN_INVALID_PROPERTY_ID (object, property_id, pspec);
		break;
	}
}


static void gee_hash_map_key_set_set_property (GObject * object, guint property_id, const GValue * value, GParamSpec * pspec) {
	GeeHashMapKeySet * self;
	self = GEE_HASH_MAP_KEY_SET (object);
	switch (property_id) {
		case GEE_HASH_MAP_KEY_SET_MAP:
		gee_hash_map_key_set_set_map (self, g_value_get_object (value));
		break;
		default:
		G_OBJECT_WARN_INVALID_PROPERTY_ID (object, property_id, pspec);
		break;
		case GEE_HASH_MAP_KEY_SET_K_TYPE:
		self->priv->k_type = g_value_get_gtype (value);
		break;
		case GEE_HASH_MAP_KEY_SET_K_DUP_FUNC:
		self->priv->k_dup_func = g_value_get_pointer (value);
		break;
		case GEE_HASH_MAP_KEY_SET_K_DESTROY_FUNC:
		self->priv->k_destroy_func = g_value_get_pointer (value);
		break;
		case GEE_HASH_MAP_KEY_SET_V_TYPE:
		self->priv->v_type = g_value_get_gtype (value);
		break;
		case GEE_HASH_MAP_KEY_SET_V_DUP_FUNC:
		self->priv->v_dup_func = g_value_get_pointer (value);
		break;
		case GEE_HASH_MAP_KEY_SET_V_DESTROY_FUNC:
		self->priv->v_destroy_func = g_value_get_pointer (value);
		break;
	}
}


static GeeHashMapKeyIterator* gee_hash_map_key_iterator_construct (GType object_type, GType k_type, GBoxedCopyFunc k_dup_func, GDestroyNotify k_destroy_func, GType v_type, GBoxedCopyFunc v_dup_func, GDestroyNotify v_destroy_func, GeeHashMap* map) {
	GParameter * __params;
	GParameter * __params_it;
	GeeHashMapKeyIterator * self;
	g_return_val_if_fail (map != NULL, NULL);
	__params = g_new0 (GParameter, 7);
	__params_it = __params;
	__params_it->name = "k-type";
	g_value_init (&__params_it->value, G_TYPE_GTYPE);
	g_value_set_gtype (&__params_it->value, k_type);
	__params_it++;
	__params_it->name = "k-dup-func";
	g_value_init (&__params_it->value, G_TYPE_POINTER);
	g_value_set_pointer (&__params_it->value, k_dup_func);
	__params_it++;
	__params_it->name = "k-destroy-func";
	g_value_init (&__params_it->value, G_TYPE_POINTER);
	g_value_set_pointer (&__params_it->value, k_destroy_func);
	__params_it++;
	__params_it->name = "v-type";
	g_value_init (&__params_it->value, G_TYPE_GTYPE);
	g_value_set_gtype (&__params_it->value, v_type);
	__params_it++;
	__params_it->name = "v-dup-func";
	g_value_init (&__params_it->value, G_TYPE_POINTER);
	g_value_set_pointer (&__params_it->value, v_dup_func);
	__params_it++;
	__params_it->name = "v-destroy-func";
	g_value_init (&__params_it->value, G_TYPE_POINTER);
	g_value_set_pointer (&__params_it->value, v_destroy_func);
	__params_it++;
	__params_it->name = "map";
	g_value_init (&__params_it->value, GEE_TYPE_HASH_MAP);
	g_value_set_object (&__params_it->value, map);
	__params_it++;
	self = g_object_newv (object_type, __params_it - __params, __params);
	while (__params_it > __params) {
		--__params_it;
		g_value_unset (&__params_it->value);
	}
	g_free (__params);
	return self;
}


static GeeHashMapKeyIterator* gee_hash_map_key_iterator_new (GType k_type, GBoxedCopyFunc k_dup_func, GDestroyNotify k_destroy_func, GType v_type, GBoxedCopyFunc v_dup_func, GDestroyNotify v_destroy_func, GeeHashMap* map) {
	return gee_hash_map_key_iterator_construct (GEE_HASH_MAP_TYPE_KEY_ITERATOR, k_type, k_dup_func, k_destroy_func, v_type, v_dup_func, v_destroy_func, map);
}


static gboolean gee_hash_map_key_iterator_real_next (GeeIterator* base) {
	GeeHashMapKeyIterator * self;
	gboolean result;
	self = (GeeHashMapKeyIterator*) base;
	if (self->priv->_node != NULL) {
		self->priv->_node = self->priv->_node->next;
	}
	while (TRUE) {
		gboolean _tmp0_;
		_tmp0_ = FALSE;
		if (self->priv->_node == NULL) {
			_tmp0_ = (self->priv->_index + 1) < self->priv->_map->priv->_array_size;
		} else {
			_tmp0_ = FALSE;
		}
		if (!_tmp0_) {
			break;
		}
		self->priv->_index++;
		self->priv->_node = self->priv->_map->priv->_nodes[self->priv->_index];
	}
	result = self->priv->_node != NULL;
	return result;
}


static gpointer gee_hash_map_key_iterator_real_get (GeeIterator* base) {
	GeeHashMapKeyIterator * self;
	gpointer result;
	gconstpointer _tmp0_;
	self = (GeeHashMapKeyIterator*) base;
	g_assert (self->priv->_stamp == self->priv->_map->priv->_stamp);
	g_assert (self->priv->_node != NULL);
	_tmp0_ = NULL;
	result = (_tmp0_ = self->priv->_node->key, ((_tmp0_ == NULL) || (self->priv->k_dup_func == NULL)) ? ((gpointer) _tmp0_) : self->priv->k_dup_func ((gpointer) _tmp0_));
	return result;
}


static void gee_hash_map_key_iterator_set_map (GeeHashMapKeyIterator* self, GeeHashMap* value) {
	GeeHashMap* _tmp1_;
	GeeHashMap* _tmp0_;
	g_return_if_fail (self != NULL);
	_tmp1_ = NULL;
	_tmp0_ = NULL;
	self->priv->_map = (_tmp1_ = (_tmp0_ = value, (_tmp0_ == NULL) ? NULL : g_object_ref (_tmp0_)), (self->priv->_map == NULL) ? NULL : (self->priv->_map = (g_object_unref (self->priv->_map), NULL)), _tmp1_);
	self->priv->_stamp = self->priv->_map->priv->_stamp;
	g_object_notify ((GObject *) self, "map");
}


static void gee_hash_map_key_iterator_class_init (GeeHashMapKeyIteratorClass * klass) {
	gee_hash_map_key_iterator_parent_class = g_type_class_peek_parent (klass);
	g_type_class_add_private (klass, sizeof (GeeHashMapKeyIteratorPrivate));
	G_OBJECT_CLASS (klass)->get_property = gee_hash_map_key_iterator_get_property;
	G_OBJECT_CLASS (klass)->set_property = gee_hash_map_key_iterator_set_property;
	G_OBJECT_CLASS (klass)->finalize = gee_hash_map_key_iterator_finalize;
	g_object_class_install_property (G_OBJECT_CLASS (klass), GEE_HASH_MAP_KEY_ITERATOR_K_TYPE, g_param_spec_gtype ("k-type", "type", "type", G_TYPE_NONE, G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB | G_PARAM_WRITABLE | G_PARAM_CONSTRUCT_ONLY));
	g_object_class_install_property (G_OBJECT_CLASS (klass), GEE_HASH_MAP_KEY_ITERATOR_K_DUP_FUNC, g_param_spec_pointer ("k-dup-func", "dup func", "dup func", G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB | G_PARAM_WRITABLE | G_PARAM_CONSTRUCT_ONLY));
	g_object_class_install_property (G_OBJECT_CLASS (klass), GEE_HASH_MAP_KEY_ITERATOR_K_DESTROY_FUNC, g_param_spec_pointer ("k-destroy-func", "destroy func", "destroy func", G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB | G_PARAM_WRITABLE | G_PARAM_CONSTRUCT_ONLY));
	g_object_class_install_property (G_OBJECT_CLASS (klass), GEE_HASH_MAP_KEY_ITERATOR_V_TYPE, g_param_spec_gtype ("v-type", "type", "type", G_TYPE_NONE, G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB | G_PARAM_WRITABLE | G_PARAM_CONSTRUCT_ONLY));
	g_object_class_install_property (G_OBJECT_CLASS (klass), GEE_HASH_MAP_KEY_ITERATOR_V_DUP_FUNC, g_param_spec_pointer ("v-dup-func", "dup func", "dup func", G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB | G_PARAM_WRITABLE | G_PARAM_CONSTRUCT_ONLY));
	g_object_class_install_property (G_OBJECT_CLASS (klass), GEE_HASH_MAP_KEY_ITERATOR_V_DESTROY_FUNC, g_param_spec_pointer ("v-destroy-func", "destroy func", "destroy func", G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB | G_PARAM_WRITABLE | G_PARAM_CONSTRUCT_ONLY));
	g_object_class_install_property (G_OBJECT_CLASS (klass), GEE_HASH_MAP_KEY_ITERATOR_MAP, g_param_spec_object ("map", "map", "map", GEE_TYPE_HASH_MAP, G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB | G_PARAM_WRITABLE | G_PARAM_CONSTRUCT_ONLY));
}


static void gee_hash_map_key_iterator_gee_iterator_interface_init (GeeIteratorIface * iface) {
	gee_hash_map_key_iterator_gee_iterator_parent_iface = g_type_interface_peek_parent (iface);
	iface->next = gee_hash_map_key_iterator_real_next;
	iface->get = gee_hash_map_key_iterator_real_get;
}


static void gee_hash_map_key_iterator_instance_init (GeeHashMapKeyIterator * self) {
	self->priv = GEE_HASH_MAP_KEY_ITERATOR_GET_PRIVATE (self);
	self->priv->_index = -1;
}


static void gee_hash_map_key_iterator_finalize (GObject* obj) {
	GeeHashMapKeyIterator * self;
	self = GEE_HASH_MAP_KEY_ITERATOR (obj);
	(self->priv->_map == NULL) ? NULL : (self->priv->_map = (g_object_unref (self->priv->_map), NULL));
	G_OBJECT_CLASS (gee_hash_map_key_iterator_parent_class)->finalize (obj);
}


static GType gee_hash_map_key_iterator_get_type (void) {
	static GType gee_hash_map_key_iterator_type_id = 0;
	if (gee_hash_map_key_iterator_type_id == 0) {
		static const GTypeInfo g_define_type_info = { sizeof (GeeHashMapKeyIteratorClass), (GBaseInitFunc) NULL, (GBaseFinalizeFunc) NULL, (GClassInitFunc) gee_hash_map_key_iterator_class_init, (GClassFinalizeFunc) NULL, NULL, sizeof (GeeHashMapKeyIterator), 0, (GInstanceInitFunc) gee_hash_map_key_iterator_instance_init, NULL };
		static const GInterfaceInfo gee_iterator_info = { (GInterfaceInitFunc) gee_hash_map_key_iterator_gee_iterator_interface_init, (GInterfaceFinalizeFunc) NULL, NULL};
		gee_hash_map_key_iterator_type_id = g_type_register_static (G_TYPE_OBJECT, "GeeHashMapKeyIterator", &g_define_type_info, 0);
		g_type_add_interface_static (gee_hash_map_key_iterator_type_id, GEE_TYPE_ITERATOR, &gee_iterator_info);
	}
	return gee_hash_map_key_iterator_type_id;
}


static void gee_hash_map_key_iterator_get_property (GObject * object, guint property_id, GValue * value, GParamSpec * pspec) {
	GeeHashMapKeyIterator * self;
	gpointer boxed;
	self = GEE_HASH_MAP_KEY_ITERATOR (object);
	switch (property_id) {
		default:
		G_OBJECT_WARN_INVALID_PROPERTY_ID (object, property_id, pspec);
		break;
	}
}


static void gee_hash_map_key_iterator_set_property (GObject * object, guint property_id, const GValue * value, GParamSpec * pspec) {
	GeeHashMapKeyIterator * self;
	self = GEE_HASH_MAP_KEY_ITERATOR (object);
	switch (property_id) {
		case GEE_HASH_MAP_KEY_ITERATOR_MAP:
		gee_hash_map_key_iterator_set_map (self, g_value_get_object (value));
		break;
		default:
		G_OBJECT_WARN_INVALID_PROPERTY_ID (object, property_id, pspec);
		break;
		case GEE_HASH_MAP_KEY_ITERATOR_K_TYPE:
		self->priv->k_type = g_value_get_gtype (value);
		break;
		case GEE_HASH_MAP_KEY_ITERATOR_K_DUP_FUNC:
		self->priv->k_dup_func = g_value_get_pointer (value);
		break;
		case GEE_HASH_MAP_KEY_ITERATOR_K_DESTROY_FUNC:
		self->priv->k_destroy_func = g_value_get_pointer (value);
		break;
		case GEE_HASH_MAP_KEY_ITERATOR_V_TYPE:
		self->priv->v_type = g_value_get_gtype (value);
		break;
		case GEE_HASH_MAP_KEY_ITERATOR_V_DUP_FUNC:
		self->priv->v_dup_func = g_value_get_pointer (value);
		break;
		case GEE_HASH_MAP_KEY_ITERATOR_V_DESTROY_FUNC:
		self->priv->v_destroy_func = g_value_get_pointer (value);
		break;
	}
}


static GeeHashMapValueCollection* gee_hash_map_value_collection_construct (GType object_type, GType k_type, GBoxedCopyFunc k_dup_func, GDestroyNotify k_destroy_func, GType v_type, GBoxedCopyFunc v_dup_func, GDestroyNotify v_destroy_func, GeeHashMap* map) {
	GParameter * __params;
	GParameter * __params_it;
	GeeHashMapValueCollection * self;
	g_return_val_if_fail (map != NULL, NULL);
	__params = g_new0 (GParameter, 7);
	__params_it = __params;
	__params_it->name = "k-type";
	g_value_init (&__params_it->value, G_TYPE_GTYPE);
	g_value_set_gtype (&__params_it->value, k_type);
	__params_it++;
	__params_it->name = "k-dup-func";
	g_value_init (&__params_it->value, G_TYPE_POINTER);
	g_value_set_pointer (&__params_it->value, k_dup_func);
	__params_it++;
	__params_it->name = "k-destroy-func";
	g_value_init (&__params_it->value, G_TYPE_POINTER);
	g_value_set_pointer (&__params_it->value, k_destroy_func);
	__params_it++;
	__params_it->name = "v-type";
	g_value_init (&__params_it->value, G_TYPE_GTYPE);
	g_value_set_gtype (&__params_it->value, v_type);
	__params_it++;
	__params_it->name = "v-dup-func";
	g_value_init (&__params_it->value, G_TYPE_POINTER);
	g_value_set_pointer (&__params_it->value, v_dup_func);
	__params_it++;
	__params_it->name = "v-destroy-func";
	g_value_init (&__params_it->value, G_TYPE_POINTER);
	g_value_set_pointer (&__params_it->value, v_destroy_func);
	__params_it++;
	__params_it->name = "map";
	g_value_init (&__params_it->value, GEE_TYPE_HASH_MAP);
	g_value_set_object (&__params_it->value, map);
	__params_it++;
	self = g_object_newv (object_type, __params_it - __params, __params);
	while (__params_it > __params) {
		--__params_it;
		g_value_unset (&__params_it->value);
	}
	g_free (__params);
	return self;
}


static GeeHashMapValueCollection* gee_hash_map_value_collection_new (GType k_type, GBoxedCopyFunc k_dup_func, GDestroyNotify k_destroy_func, GType v_type, GBoxedCopyFunc v_dup_func, GDestroyNotify v_destroy_func, GeeHashMap* map) {
	return gee_hash_map_value_collection_construct (GEE_HASH_MAP_TYPE_VALUE_COLLECTION, k_type, k_dup_func, k_destroy_func, v_type, v_dup_func, v_destroy_func, map);
}


static GeeIterator* gee_hash_map_value_collection_real_iterator (GeeAbstractCollection* base) {
	GeeHashMapValueCollection * self;
	GeeIterator* result;
	self = (GeeHashMapValueCollection*) base;
	result = (GeeIterator*) gee_hash_map_value_iterator_new (self->priv->k_type, (GBoxedCopyFunc) self->priv->k_dup_func, self->priv->k_destroy_func, self->priv->v_type, (GBoxedCopyFunc) self->priv->v_dup_func, self->priv->v_destroy_func, self->priv->_map);
	return result;
}


static gboolean gee_hash_map_value_collection_real_add (GeeAbstractCollection* base, gconstpointer value) {
	GeeHashMapValueCollection * self;
	gboolean result;
	self = (GeeHashMapValueCollection*) base;
	g_assert_not_reached ();
}


static void gee_hash_map_value_collection_real_clear (GeeAbstractCollection* base) {
	GeeHashMapValueCollection * self;
	self = (GeeHashMapValueCollection*) base;
	g_assert_not_reached ();
}


static gboolean gee_hash_map_value_collection_real_remove (GeeAbstractCollection* base, gconstpointer value) {
	GeeHashMapValueCollection * self;
	gboolean result;
	self = (GeeHashMapValueCollection*) base;
	g_assert_not_reached ();
}


static gboolean gee_hash_map_value_collection_real_contains (GeeAbstractCollection* base, gconstpointer value) {
	GeeHashMapValueCollection * self;
	gboolean result;
	GeeIterator* it;
	self = (GeeHashMapValueCollection*) base;
	it = gee_abstract_collection_iterator ((GeeAbstractCollection*) self);
	while (TRUE) {
		gpointer _tmp0_;
		gboolean _tmp1_;
		if (!gee_iterator_next (it)) {
			break;
		}
		_tmp0_ = NULL;
		if ((_tmp1_ = gee_hash_map_get_value_equal_func (self->priv->_map) (_tmp0_ = gee_iterator_get (it), value), ((_tmp0_ == NULL) || (self->priv->v_destroy_func == NULL)) ? NULL : (_tmp0_ = (self->priv->v_destroy_func (_tmp0_), NULL)), _tmp1_)) {
			result = TRUE;
			(it == NULL) ? NULL : (it = (g_object_unref (it), NULL));
			return result;
		}
	}
	result = FALSE;
	(it == NULL) ? NULL : (it = (g_object_unref (it), NULL));
	return result;
}


static gboolean gee_hash_map_value_collection_real_add_all (GeeAbstractCollection* base, GeeCollection* collection) {
	GeeHashMapValueCollection * self;
	gboolean result;
	self = (GeeHashMapValueCollection*) base;
	g_return_val_if_fail (collection != NULL, FALSE);
	g_assert_not_reached ();
}


static gboolean gee_hash_map_value_collection_real_remove_all (GeeAbstractCollection* base, GeeCollection* collection) {
	GeeHashMapValueCollection * self;
	gboolean result;
	self = (GeeHashMapValueCollection*) base;
	g_return_val_if_fail (collection != NULL, FALSE);
	g_assert_not_reached ();
}


static gboolean gee_hash_map_value_collection_real_retain_all (GeeAbstractCollection* base, GeeCollection* collection) {
	GeeHashMapValueCollection * self;
	gboolean result;
	self = (GeeHashMapValueCollection*) base;
	g_return_val_if_fail (collection != NULL, FALSE);
	g_assert_not_reached ();
}


static GeeHashMap* gee_hash_map_value_collection_get_map (GeeHashMapValueCollection* self) {
	GeeHashMap* result;
	g_return_val_if_fail (self != NULL, NULL);
	result = self->priv->_map;
	return result;
}


static void gee_hash_map_value_collection_set_map (GeeHashMapValueCollection* self, GeeHashMap* value) {
	GeeHashMap* _tmp1_;
	GeeHashMap* _tmp0_;
	g_return_if_fail (self != NULL);
	_tmp1_ = NULL;
	_tmp0_ = NULL;
	self->priv->_map = (_tmp1_ = (_tmp0_ = value, (_tmp0_ == NULL) ? NULL : g_object_ref (_tmp0_)), (self->priv->_map == NULL) ? NULL : (self->priv->_map = (g_object_unref (self->priv->_map), NULL)), _tmp1_);
	g_object_notify ((GObject *) self, "map");
}


static gint gee_hash_map_value_collection_real_get_size (GeeAbstractCollection* base) {
	gint result;
	GeeHashMapValueCollection* self;
	self = (GeeHashMapValueCollection*) base;
	result = gee_map_get_size ((GeeMap*) self->priv->_map);
	return result;
}


static void gee_hash_map_value_collection_class_init (GeeHashMapValueCollectionClass * klass) {
	gee_hash_map_value_collection_parent_class = g_type_class_peek_parent (klass);
	g_type_class_add_private (klass, sizeof (GeeHashMapValueCollectionPrivate));
	GEE_ABSTRACT_COLLECTION_CLASS (klass)->iterator = gee_hash_map_value_collection_real_iterator;
	GEE_ABSTRACT_COLLECTION_CLASS (klass)->add = gee_hash_map_value_collection_real_add;
	GEE_ABSTRACT_COLLECTION_CLASS (klass)->clear = gee_hash_map_value_collection_real_clear;
	GEE_ABSTRACT_COLLECTION_CLASS (klass)->remove = gee_hash_map_value_collection_real_remove;
	GEE_ABSTRACT_COLLECTION_CLASS (klass)->contains = gee_hash_map_value_collection_real_contains;
	GEE_ABSTRACT_COLLECTION_CLASS (klass)->add_all = gee_hash_map_value_collection_real_add_all;
	GEE_ABSTRACT_COLLECTION_CLASS (klass)->remove_all = gee_hash_map_value_collection_real_remove_all;
	GEE_ABSTRACT_COLLECTION_CLASS (klass)->retain_all = gee_hash_map_value_collection_real_retain_all;
	GEE_ABSTRACT_COLLECTION_CLASS (klass)->get_size = gee_hash_map_value_collection_real_get_size;
	G_OBJECT_CLASS (klass)->get_property = gee_hash_map_value_collection_get_property;
	G_OBJECT_CLASS (klass)->set_property = gee_hash_map_value_collection_set_property;
	G_OBJECT_CLASS (klass)->finalize = gee_hash_map_value_collection_finalize;
	g_object_class_install_property (G_OBJECT_CLASS (klass), GEE_HASH_MAP_VALUE_COLLECTION_K_TYPE, g_param_spec_gtype ("k-type", "type", "type", G_TYPE_NONE, G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB | G_PARAM_WRITABLE | G_PARAM_CONSTRUCT_ONLY));
	g_object_class_install_property (G_OBJECT_CLASS (klass), GEE_HASH_MAP_VALUE_COLLECTION_K_DUP_FUNC, g_param_spec_pointer ("k-dup-func", "dup func", "dup func", G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB | G_PARAM_WRITABLE | G_PARAM_CONSTRUCT_ONLY));
	g_object_class_install_property (G_OBJECT_CLASS (klass), GEE_HASH_MAP_VALUE_COLLECTION_K_DESTROY_FUNC, g_param_spec_pointer ("k-destroy-func", "destroy func", "destroy func", G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB | G_PARAM_WRITABLE | G_PARAM_CONSTRUCT_ONLY));
	g_object_class_install_property (G_OBJECT_CLASS (klass), GEE_HASH_MAP_VALUE_COLLECTION_V_TYPE, g_param_spec_gtype ("v-type", "type", "type", G_TYPE_NONE, G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB | G_PARAM_WRITABLE | G_PARAM_CONSTRUCT_ONLY));
	g_object_class_install_property (G_OBJECT_CLASS (klass), GEE_HASH_MAP_VALUE_COLLECTION_V_DUP_FUNC, g_param_spec_pointer ("v-dup-func", "dup func", "dup func", G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB | G_PARAM_WRITABLE | G_PARAM_CONSTRUCT_ONLY));
	g_object_class_install_property (G_OBJECT_CLASS (klass), GEE_HASH_MAP_VALUE_COLLECTION_V_DESTROY_FUNC, g_param_spec_pointer ("v-destroy-func", "destroy func", "destroy func", G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB | G_PARAM_WRITABLE | G_PARAM_CONSTRUCT_ONLY));
	g_object_class_install_property (G_OBJECT_CLASS (klass), GEE_HASH_MAP_VALUE_COLLECTION_MAP, g_param_spec_object ("map", "map", "map", GEE_TYPE_HASH_MAP, G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB | G_PARAM_READABLE | G_PARAM_WRITABLE | G_PARAM_CONSTRUCT_ONLY));
	g_object_class_override_property (G_OBJECT_CLASS (klass), GEE_HASH_MAP_VALUE_COLLECTION_SIZE, "size");
}


static void gee_hash_map_value_collection_instance_init (GeeHashMapValueCollection * self) {
	self->priv = GEE_HASH_MAP_VALUE_COLLECTION_GET_PRIVATE (self);
}


static void gee_hash_map_value_collection_finalize (GObject* obj) {
	GeeHashMapValueCollection * self;
	self = GEE_HASH_MAP_VALUE_COLLECTION (obj);
	(self->priv->_map == NULL) ? NULL : (self->priv->_map = (g_object_unref (self->priv->_map), NULL));
	G_OBJECT_CLASS (gee_hash_map_value_collection_parent_class)->finalize (obj);
}


static GType gee_hash_map_value_collection_get_type (void) {
	static GType gee_hash_map_value_collection_type_id = 0;
	if (gee_hash_map_value_collection_type_id == 0) {
		static const GTypeInfo g_define_type_info = { sizeof (GeeHashMapValueCollectionClass), (GBaseInitFunc) NULL, (GBaseFinalizeFunc) NULL, (GClassInitFunc) gee_hash_map_value_collection_class_init, (GClassFinalizeFunc) NULL, NULL, sizeof (GeeHashMapValueCollection), 0, (GInstanceInitFunc) gee_hash_map_value_collection_instance_init, NULL };
		gee_hash_map_value_collection_type_id = g_type_register_static (GEE_TYPE_ABSTRACT_COLLECTION, "GeeHashMapValueCollection", &g_define_type_info, 0);
	}
	return gee_hash_map_value_collection_type_id;
}


static void gee_hash_map_value_collection_get_property (GObject * object, guint property_id, GValue * value, GParamSpec * pspec) {
	GeeHashMapValueCollection * self;
	gpointer boxed;
	self = GEE_HASH_MAP_VALUE_COLLECTION (object);
	switch (property_id) {
		case GEE_HASH_MAP_VALUE_COLLECTION_MAP:
		g_value_set_object (value, gee_hash_map_value_collection_get_map (self));
		break;
		case GEE_HASH_MAP_VALUE_COLLECTION_SIZE:
		g_value_set_int (value, gee_abstract_collection_get_size ((GeeAbstractCollection*) self));
		break;
		default:
		G_OBJECT_WARN_INVALID_PROPERTY_ID (object, property_id, pspec);
		break;
	}
}


static void gee_hash_map_value_collection_set_property (GObject * object, guint property_id, const GValue * value, GParamSpec * pspec) {
	GeeHashMapValueCollection * self;
	self = GEE_HASH_MAP_VALUE_COLLECTION (object);
	switch (property_id) {
		case GEE_HASH_MAP_VALUE_COLLECTION_MAP:
		gee_hash_map_value_collection_set_map (self, g_value_get_object (value));
		break;
		default:
		G_OBJECT_WARN_INVALID_PROPERTY_ID (object, property_id, pspec);
		break;
		case GEE_HASH_MAP_VALUE_COLLECTION_K_TYPE:
		self->priv->k_type = g_value_get_gtype (value);
		break;
		case GEE_HASH_MAP_VALUE_COLLECTION_K_DUP_FUNC:
		self->priv->k_dup_func = g_value_get_pointer (value);
		break;
		case GEE_HASH_MAP_VALUE_COLLECTION_K_DESTROY_FUNC:
		self->priv->k_destroy_func = g_value_get_pointer (value);
		break;
		case GEE_HASH_MAP_VALUE_COLLECTION_V_TYPE:
		self->priv->v_type = g_value_get_gtype (value);
		break;
		case GEE_HASH_MAP_VALUE_COLLECTION_V_DUP_FUNC:
		self->priv->v_dup_func = g_value_get_pointer (value);
		break;
		case GEE_HASH_MAP_VALUE_COLLECTION_V_DESTROY_FUNC:
		self->priv->v_destroy_func = g_value_get_pointer (value);
		break;
	}
}


static GeeHashMapValueIterator* gee_hash_map_value_iterator_construct (GType object_type, GType k_type, GBoxedCopyFunc k_dup_func, GDestroyNotify k_destroy_func, GType v_type, GBoxedCopyFunc v_dup_func, GDestroyNotify v_destroy_func, GeeHashMap* map) {
	GParameter * __params;
	GParameter * __params_it;
	GeeHashMapValueIterator * self;
	g_return_val_if_fail (map != NULL, NULL);
	__params = g_new0 (GParameter, 7);
	__params_it = __params;
	__params_it->name = "k-type";
	g_value_init (&__params_it->value, G_TYPE_GTYPE);
	g_value_set_gtype (&__params_it->value, k_type);
	__params_it++;
	__params_it->name = "k-dup-func";
	g_value_init (&__params_it->value, G_TYPE_POINTER);
	g_value_set_pointer (&__params_it->value, k_dup_func);
	__params_it++;
	__params_it->name = "k-destroy-func";
	g_value_init (&__params_it->value, G_TYPE_POINTER);
	g_value_set_pointer (&__params_it->value, k_destroy_func);
	__params_it++;
	__params_it->name = "v-type";
	g_value_init (&__params_it->value, G_TYPE_GTYPE);
	g_value_set_gtype (&__params_it->value, v_type);
	__params_it++;
	__params_it->name = "v-dup-func";
	g_value_init (&__params_it->value, G_TYPE_POINTER);
	g_value_set_pointer (&__params_it->value, v_dup_func);
	__params_it++;
	__params_it->name = "v-destroy-func";
	g_value_init (&__params_it->value, G_TYPE_POINTER);
	g_value_set_pointer (&__params_it->value, v_destroy_func);
	__params_it++;
	__params_it->name = "map";
	g_value_init (&__params_it->value, GEE_TYPE_HASH_MAP);
	g_value_set_object (&__params_it->value, map);
	__params_it++;
	self = g_object_newv (object_type, __params_it - __params, __params);
	while (__params_it > __params) {
		--__params_it;
		g_value_unset (&__params_it->value);
	}
	g_free (__params);
	return self;
}


static GeeHashMapValueIterator* gee_hash_map_value_iterator_new (GType k_type, GBoxedCopyFunc k_dup_func, GDestroyNotify k_destroy_func, GType v_type, GBoxedCopyFunc v_dup_func, GDestroyNotify v_destroy_func, GeeHashMap* map) {
	return gee_hash_map_value_iterator_construct (GEE_HASH_MAP_TYPE_VALUE_ITERATOR, k_type, k_dup_func, k_destroy_func, v_type, v_dup_func, v_destroy_func, map);
}


static gboolean gee_hash_map_value_iterator_real_next (GeeIterator* base) {
	GeeHashMapValueIterator * self;
	gboolean result;
	self = (GeeHashMapValueIterator*) base;
	if (self->priv->_node != NULL) {
		self->priv->_node = self->priv->_node->next;
	}
	while (TRUE) {
		gboolean _tmp0_;
		_tmp0_ = FALSE;
		if (self->priv->_node == NULL) {
			_tmp0_ = (self->priv->_index + 1) < self->priv->_map->priv->_array_size;
		} else {
			_tmp0_ = FALSE;
		}
		if (!_tmp0_) {
			break;
		}
		self->priv->_index++;
		self->priv->_node = self->priv->_map->priv->_nodes[self->priv->_index];
	}
	result = self->priv->_node != NULL;
	return result;
}


static gpointer gee_hash_map_value_iterator_real_get (GeeIterator* base) {
	GeeHashMapValueIterator * self;
	gpointer result;
	gconstpointer _tmp0_;
	self = (GeeHashMapValueIterator*) base;
	g_assert (self->priv->_stamp == self->priv->_map->priv->_stamp);
	g_assert (self->priv->_node != NULL);
	_tmp0_ = NULL;
	result = (_tmp0_ = self->priv->_node->value, ((_tmp0_ == NULL) || (self->priv->v_dup_func == NULL)) ? ((gpointer) _tmp0_) : self->priv->v_dup_func ((gpointer) _tmp0_));
	return result;
}


static void gee_hash_map_value_iterator_set_map (GeeHashMapValueIterator* self, GeeHashMap* value) {
	GeeHashMap* _tmp1_;
	GeeHashMap* _tmp0_;
	g_return_if_fail (self != NULL);
	_tmp1_ = NULL;
	_tmp0_ = NULL;
	self->priv->_map = (_tmp1_ = (_tmp0_ = value, (_tmp0_ == NULL) ? NULL : g_object_ref (_tmp0_)), (self->priv->_map == NULL) ? NULL : (self->priv->_map = (g_object_unref (self->priv->_map), NULL)), _tmp1_);
	self->priv->_stamp = self->priv->_map->priv->_stamp;
	g_object_notify ((GObject *) self, "map");
}


static void gee_hash_map_value_iterator_class_init (GeeHashMapValueIteratorClass * klass) {
	gee_hash_map_value_iterator_parent_class = g_type_class_peek_parent (klass);
	g_type_class_add_private (klass, sizeof (GeeHashMapValueIteratorPrivate));
	G_OBJECT_CLASS (klass)->get_property = gee_hash_map_value_iterator_get_property;
	G_OBJECT_CLASS (klass)->set_property = gee_hash_map_value_iterator_set_property;
	G_OBJECT_CLASS (klass)->finalize = gee_hash_map_value_iterator_finalize;
	g_object_class_install_property (G_OBJECT_CLASS (klass), GEE_HASH_MAP_VALUE_ITERATOR_K_TYPE, g_param_spec_gtype ("k-type", "type", "type", G_TYPE_NONE, G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB | G_PARAM_WRITABLE | G_PARAM_CONSTRUCT_ONLY));
	g_object_class_install_property (G_OBJECT_CLASS (klass), GEE_HASH_MAP_VALUE_ITERATOR_K_DUP_FUNC, g_param_spec_pointer ("k-dup-func", "dup func", "dup func", G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB | G_PARAM_WRITABLE | G_PARAM_CONSTRUCT_ONLY));
	g_object_class_install_property (G_OBJECT_CLASS (klass), GEE_HASH_MAP_VALUE_ITERATOR_K_DESTROY_FUNC, g_param_spec_pointer ("k-destroy-func", "destroy func", "destroy func", G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB | G_PARAM_WRITABLE | G_PARAM_CONSTRUCT_ONLY));
	g_object_class_install_property (G_OBJECT_CLASS (klass), GEE_HASH_MAP_VALUE_ITERATOR_V_TYPE, g_param_spec_gtype ("v-type", "type", "type", G_TYPE_NONE, G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB | G_PARAM_WRITABLE | G_PARAM_CONSTRUCT_ONLY));
	g_object_class_install_property (G_OBJECT_CLASS (klass), GEE_HASH_MAP_VALUE_ITERATOR_V_DUP_FUNC, g_param_spec_pointer ("v-dup-func", "dup func", "dup func", G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB | G_PARAM_WRITABLE | G_PARAM_CONSTRUCT_ONLY));
	g_object_class_install_property (G_OBJECT_CLASS (klass), GEE_HASH_MAP_VALUE_ITERATOR_V_DESTROY_FUNC, g_param_spec_pointer ("v-destroy-func", "destroy func", "destroy func", G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB | G_PARAM_WRITABLE | G_PARAM_CONSTRUCT_ONLY));
	g_object_class_install_property (G_OBJECT_CLASS (klass), GEE_HASH_MAP_VALUE_ITERATOR_MAP, g_param_spec_object ("map", "map", "map", GEE_TYPE_HASH_MAP, G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB | G_PARAM_WRITABLE | G_PARAM_CONSTRUCT_ONLY));
}


static void gee_hash_map_value_iterator_gee_iterator_interface_init (GeeIteratorIface * iface) {
	gee_hash_map_value_iterator_gee_iterator_parent_iface = g_type_interface_peek_parent (iface);
	iface->next = gee_hash_map_value_iterator_real_next;
	iface->get = gee_hash_map_value_iterator_real_get;
}


static void gee_hash_map_value_iterator_instance_init (GeeHashMapValueIterator * self) {
	self->priv = GEE_HASH_MAP_VALUE_ITERATOR_GET_PRIVATE (self);
	self->priv->_index = -1;
}


static void gee_hash_map_value_iterator_finalize (GObject* obj) {
	GeeHashMapValueIterator * self;
	self = GEE_HASH_MAP_VALUE_ITERATOR (obj);
	(self->priv->_map == NULL) ? NULL : (self->priv->_map = (g_object_unref (self->priv->_map), NULL));
	G_OBJECT_CLASS (gee_hash_map_value_iterator_parent_class)->finalize (obj);
}


static GType gee_hash_map_value_iterator_get_type (void) {
	static GType gee_hash_map_value_iterator_type_id = 0;
	if (gee_hash_map_value_iterator_type_id == 0) {
		static const GTypeInfo g_define_type_info = { sizeof (GeeHashMapValueIteratorClass), (GBaseInitFunc) NULL, (GBaseFinalizeFunc) NULL, (GClassInitFunc) gee_hash_map_value_iterator_class_init, (GClassFinalizeFunc) NULL, NULL, sizeof (GeeHashMapValueIterator), 0, (GInstanceInitFunc) gee_hash_map_value_iterator_instance_init, NULL };
		static const GInterfaceInfo gee_iterator_info = { (GInterfaceInitFunc) gee_hash_map_value_iterator_gee_iterator_interface_init, (GInterfaceFinalizeFunc) NULL, NULL};
		gee_hash_map_value_iterator_type_id = g_type_register_static (G_TYPE_OBJECT, "GeeHashMapValueIterator", &g_define_type_info, 0);
		g_type_add_interface_static (gee_hash_map_value_iterator_type_id, GEE_TYPE_ITERATOR, &gee_iterator_info);
	}
	return gee_hash_map_value_iterator_type_id;
}


static void gee_hash_map_value_iterator_get_property (GObject * object, guint property_id, GValue * value, GParamSpec * pspec) {
	GeeHashMapValueIterator * self;
	gpointer boxed;
	self = GEE_HASH_MAP_VALUE_ITERATOR (object);
	switch (property_id) {
		default:
		G_OBJECT_WARN_INVALID_PROPERTY_ID (object, property_id, pspec);
		break;
	}
}


static void gee_hash_map_value_iterator_set_property (GObject * object, guint property_id, const GValue * value, GParamSpec * pspec) {
	GeeHashMapValueIterator * self;
	self = GEE_HASH_MAP_VALUE_ITERATOR (object);
	switch (property_id) {
		case GEE_HASH_MAP_VALUE_ITERATOR_MAP:
		gee_hash_map_value_iterator_set_map (self, g_value_get_object (value));
		break;
		default:
		G_OBJECT_WARN_INVALID_PROPERTY_ID (object, property_id, pspec);
		break;
		case GEE_HASH_MAP_VALUE_ITERATOR_K_TYPE:
		self->priv->k_type = g_value_get_gtype (value);
		break;
		case GEE_HASH_MAP_VALUE_ITERATOR_K_DUP_FUNC:
		self->priv->k_dup_func = g_value_get_pointer (value);
		break;
		case GEE_HASH_MAP_VALUE_ITERATOR_K_DESTROY_FUNC:
		self->priv->k_destroy_func = g_value_get_pointer (value);
		break;
		case GEE_HASH_MAP_VALUE_ITERATOR_V_TYPE:
		self->priv->v_type = g_value_get_gtype (value);
		break;
		case GEE_HASH_MAP_VALUE_ITERATOR_V_DUP_FUNC:
		self->priv->v_dup_func = g_value_get_pointer (value);
		break;
		case GEE_HASH_MAP_VALUE_ITERATOR_V_DESTROY_FUNC:
		self->priv->v_destroy_func = g_value_get_pointer (value);
		break;
	}
}


static void gee_hash_map_class_init (GeeHashMapClass * klass) {
	gee_hash_map_parent_class = g_type_class_peek_parent (klass);
	g_type_class_add_private (klass, sizeof (GeeHashMapPrivate));
	GEE_ABSTRACT_MAP_CLASS (klass)->get_keys = gee_hash_map_real_get_keys;
	GEE_ABSTRACT_MAP_CLASS (klass)->get_values = gee_hash_map_real_get_values;
	GEE_ABSTRACT_MAP_CLASS (klass)->contains = gee_hash_map_real_contains;
	GEE_ABSTRACT_MAP_CLASS (klass)->get = gee_hash_map_real_get;
	GEE_ABSTRACT_MAP_CLASS (klass)->set = gee_hash_map_real_set;
	GEE_ABSTRACT_MAP_CLASS (klass)->remove = gee_hash_map_real_remove;
	GEE_ABSTRACT_MAP_CLASS (klass)->clear = gee_hash_map_real_clear;
	GEE_ABSTRACT_MAP_CLASS (klass)->get_size = gee_hash_map_real_get_size;
	G_OBJECT_CLASS (klass)->get_property = gee_hash_map_get_property;
	G_OBJECT_CLASS (klass)->set_property = gee_hash_map_set_property;
	G_OBJECT_CLASS (klass)->constructor = gee_hash_map_constructor;
	G_OBJECT_CLASS (klass)->finalize = gee_hash_map_finalize;
	g_object_class_install_property (G_OBJECT_CLASS (klass), GEE_HASH_MAP_K_TYPE, g_param_spec_gtype ("k-type", "type", "type", G_TYPE_NONE, G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB | G_PARAM_WRITABLE | G_PARAM_CONSTRUCT_ONLY));
	g_object_class_install_property (G_OBJECT_CLASS (klass), GEE_HASH_MAP_K_DUP_FUNC, g_param_spec_pointer ("k-dup-func", "dup func", "dup func", G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB | G_PARAM_WRITABLE | G_PARAM_CONSTRUCT_ONLY));
	g_object_class_install_property (G_OBJECT_CLASS (klass), GEE_HASH_MAP_K_DESTROY_FUNC, g_param_spec_pointer ("k-destroy-func", "destroy func", "destroy func", G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB | G_PARAM_WRITABLE | G_PARAM_CONSTRUCT_ONLY));
	g_object_class_install_property (G_OBJECT_CLASS (klass), GEE_HASH_MAP_V_TYPE, g_param_spec_gtype ("v-type", "type", "type", G_TYPE_NONE, G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB | G_PARAM_WRITABLE | G_PARAM_CONSTRUCT_ONLY));
	g_object_class_install_property (G_OBJECT_CLASS (klass), GEE_HASH_MAP_V_DUP_FUNC, g_param_spec_pointer ("v-dup-func", "dup func", "dup func", G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB | G_PARAM_WRITABLE | G_PARAM_CONSTRUCT_ONLY));
	g_object_class_install_property (G_OBJECT_CLASS (klass), GEE_HASH_MAP_V_DESTROY_FUNC, g_param_spec_pointer ("v-destroy-func", "destroy func", "destroy func", G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB | G_PARAM_WRITABLE | G_PARAM_CONSTRUCT_ONLY));
	g_object_class_override_property (G_OBJECT_CLASS (klass), GEE_HASH_MAP_SIZE, "size");
	g_object_class_install_property (G_OBJECT_CLASS (klass), GEE_HASH_MAP_KEY_HASH_FUNC, g_param_spec_pointer ("key-hash-func", "key-hash-func", "key-hash-func", G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB | G_PARAM_READABLE | G_PARAM_WRITABLE | G_PARAM_CONSTRUCT_ONLY));
	g_object_class_install_property (G_OBJECT_CLASS (klass), GEE_HASH_MAP_KEY_EQUAL_FUNC, g_param_spec_pointer ("key-equal-func", "key-equal-func", "key-equal-func", G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB | G_PARAM_READABLE | G_PARAM_WRITABLE | G_PARAM_CONSTRUCT_ONLY));
	g_object_class_install_property (G_OBJECT_CLASS (klass), GEE_HASH_MAP_VALUE_EQUAL_FUNC, g_param_spec_pointer ("value-equal-func", "value-equal-func", "value-equal-func", G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB | G_PARAM_READABLE | G_PARAM_WRITABLE | G_PARAM_CONSTRUCT_ONLY));
}


static void gee_hash_map_instance_init (GeeHashMap * self) {
	self->priv = GEE_HASH_MAP_GET_PRIVATE (self);
	self->priv->_stamp = 0;
}


static void gee_hash_map_finalize (GObject* obj) {
	GeeHashMap * self;
	self = GEE_HASH_MAP (obj);
	{
		gee_abstract_map_clear ((GeeAbstractMap*) self);
	}
	self->priv->_nodes = (_vala_array_free (self->priv->_nodes, self->priv->_nodes_length1, (GDestroyNotify) gee_hash_map_node_free), NULL);
	G_OBJECT_CLASS (gee_hash_map_parent_class)->finalize (obj);
}


GType gee_hash_map_get_type (void) {
	static GType gee_hash_map_type_id = 0;
	if (gee_hash_map_type_id == 0) {
		static const GTypeInfo g_define_type_info = { sizeof (GeeHashMapClass), (GBaseInitFunc) NULL, (GBaseFinalizeFunc) NULL, (GClassInitFunc) gee_hash_map_class_init, (GClassFinalizeFunc) NULL, NULL, sizeof (GeeHashMap), 0, (GInstanceInitFunc) gee_hash_map_instance_init, NULL };
		gee_hash_map_type_id = g_type_register_static (GEE_TYPE_ABSTRACT_MAP, "GeeHashMap", &g_define_type_info, 0);
	}
	return gee_hash_map_type_id;
}


static void gee_hash_map_get_property (GObject * object, guint property_id, GValue * value, GParamSpec * pspec) {
	GeeHashMap * self;
	gpointer boxed;
	self = GEE_HASH_MAP (object);
	switch (property_id) {
		case GEE_HASH_MAP_SIZE:
		g_value_set_int (value, gee_abstract_map_get_size ((GeeAbstractMap*) self));
		break;
		case GEE_HASH_MAP_KEY_HASH_FUNC:
		g_value_set_pointer (value, gee_hash_map_get_key_hash_func (self));
		break;
		case GEE_HASH_MAP_KEY_EQUAL_FUNC:
		g_value_set_pointer (value, gee_hash_map_get_key_equal_func (self));
		break;
		case GEE_HASH_MAP_VALUE_EQUAL_FUNC:
		g_value_set_pointer (value, gee_hash_map_get_value_equal_func (self));
		break;
		default:
		G_OBJECT_WARN_INVALID_PROPERTY_ID (object, property_id, pspec);
		break;
	}
}


static void gee_hash_map_set_property (GObject * object, guint property_id, const GValue * value, GParamSpec * pspec) {
	GeeHashMap * self;
	self = GEE_HASH_MAP (object);
	switch (property_id) {
		case GEE_HASH_MAP_KEY_HASH_FUNC:
		gee_hash_map_set_key_hash_func (self, g_value_get_pointer (value));
		break;
		case GEE_HASH_MAP_KEY_EQUAL_FUNC:
		gee_hash_map_set_key_equal_func (self, g_value_get_pointer (value));
		break;
		case GEE_HASH_MAP_VALUE_EQUAL_FUNC:
		gee_hash_map_set_value_equal_func (self, g_value_get_pointer (value));
		break;
		default:
		G_OBJECT_WARN_INVALID_PROPERTY_ID (object, property_id, pspec);
		break;
		case GEE_HASH_MAP_K_TYPE:
		self->priv->k_type = g_value_get_gtype (value);
		break;
		case GEE_HASH_MAP_K_DUP_FUNC:
		self->priv->k_dup_func = g_value_get_pointer (value);
		break;
		case GEE_HASH_MAP_K_DESTROY_FUNC:
		self->priv->k_destroy_func = g_value_get_pointer (value);
		break;
		case GEE_HASH_MAP_V_TYPE:
		self->priv->v_type = g_value_get_gtype (value);
		break;
		case GEE_HASH_MAP_V_DUP_FUNC:
		self->priv->v_dup_func = g_value_get_pointer (value);
		break;
		case GEE_HASH_MAP_V_DESTROY_FUNC:
		self->priv->v_destroy_func = g_value_get_pointer (value);
		break;
	}
}


static void _vala_array_destroy (gpointer array, gint array_length, GDestroyNotify destroy_func) {
	if ((array != NULL) && (destroy_func != NULL)) {
		int i;
		for (i = 0; i < array_length; i = i + 1) {
			if (((gpointer*) array)[i] != NULL) {
				destroy_func (((gpointer*) array)[i]);
			}
		}
	}
}


static void _vala_array_free (gpointer array, gint array_length, GDestroyNotify destroy_func) {
	_vala_array_destroy (array, array_length, destroy_func);
	g_free (array);
}




