/* gEDA - GPL Electronic Design Automation
 * libgeda - gEDA's library
 * Copyright (C) 1998-2000 Ales V. Hvezda
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111 USA
 */
#include <config.h>

#include <stdio.h>
#include <unistd.h>
#include <time.h>
#include <math.h>

#include <gtk/gtk.h>
#include <libguile.h>

#include "defines.h"
#include "struct.h"
#include "globals.h"
#include "o_types.h"
#include "funcs.h"
#include "colors.h"

#include "../include/prototype.h"

/*! \brief Hashtable storing mapping between character and
 *         postscript glyph name
 */
GHashTable *unicode_char_to_glyph = NULL;

static int f_print_get_unicode_chars(TOPLEVEL * w_current, OBJECT * head,
				     int count, gunichar * table);
static void f_print_unicode_map(FILE * fp, int count, gunichar * table);

/*! \brief Prints the line width in mils to a postscript document.
 *  \par Function Description
 *  This function writes the line width in mils to the specified
 *  postscript document.
 *
 *  \param [in]     fp     The postscript document to write to.
 *  \param [in]     width  The width of the line in mils to set.
 */
void f_print_set_line_width(FILE *fp, int width)
{
  if (width > 0) {
    fprintf(fp, "%d mils setlinewidth\n", width);
  }
}

/*! \brief Prints the color to a postscript document
 *  \par Function Description
 *  This function converts the color number passed to a string
 *  and prints it to the postscript document provided.
 *
 *  \param [in]     fp     The postscript document to print the color to.
 *  \param [in]     color  Integer color to convert and print.
 */
void f_print_set_color(FILE *fp, int color) 
{
  char *string;

  /* DO NOT free string... it's a reference to a malloced */
  /* string, there is *no* memory leak here */
  string = s_color_ps_string(color);

  if (string) {
    fprintf(fp, "%s setrgbcolor\n", string);
  } else {
    fprintf(fp, "0 0 0 setrgbcolor\n");
  }
}

/*! \brief Prints the header to a postscript document.
 *  \par Function Description
 *  This function will print a document preamble and header
 *  for a postscript document.
 *
 *  \param [in] w_current     The TOPLEVEL object to create document for.
 *  \param [in] fp            The postscript document to write to.
 *  \param [in] paper_size_x  The width of the document on paper in inches.
 *  \param [in] paper_size_y  The height of the document on paper in inches.
 *  \return 0 on success, -1 on failure.
 */
int f_print_header(TOPLEVEL *w_current, FILE *fp, 
		   int paper_size_x, int paper_size_y) 
{
  char *buf;
  FILE *prolog;
  size_t bytes;
  int llx,lly,urx,ury;
  time_t current_time,time_rc;

  /* Compute bounding box */
  llx=0;                          /* So, right now, the box starts at (0,0) */
  lly=0;
  urx=((float)paper_size_y * 72.0)/1000; 
  ury=((float)paper_size_x * 72.0)/1000; 
  
  /* Get Time of day for creation date */
  time_rc = time(&current_time);
  if(time_rc == (time_t)-1) {
    s_log_message("Unable to get time of day in f_print_header()\n");
    current_time=0; /* Just set it to 1970... */
  }

  /* Output the DSC comments at the beginning of the document */
  fprintf(fp, "%%!PS-Adobe-3.0\n"
	  "%%%%Creator: gEDA gschem %s\n"
	  "%%%%CreationDate: %s"
	  "%%%%Title: %s\n"
	  "%%%%Author: %s\n"
	  "%%%%BoundingBox: %d %d %d %d\n"
	  "%%%%Orientation: %s\n"
	  "%%%%Pages: 1\n"
	  "%%%%Endcomments\n"
	  "%%%%BeginProlog\n",
	  VERSION,
	  ctime(&current_time),
	  w_current->page_current->page_filename,
	  getlogin(),
	  llx, lly, urx, ury,
	  ((w_current->print_orientation == LANDSCAPE) 
	   ? "Landscape" : "Portrait")
	  );

  /* Fetch and insert the Postscript prolog from disk here */
  
  /* Allocate a buffer to use during copy */
#define PROLOG_BUFFER_SIZE 8192
  buf = g_malloc(PROLOG_BUFFER_SIZE);
  if(buf == NULL) {
    s_log_message("Unable to allocate %d bytes in f_print_header()\n"
		  "Giving up on printing.\n",PROLOG_BUFFER_SIZE);
    return -1;
  }
  /* Open the prolog file */
  prolog = fopen(w_current->postscript_prolog,"r");
  if(prolog == NULL) {
    s_log_message("Unable to open the prolog file `%s' for reading "
		  "in f_print_header()\n"
		  "Giving up on printing\n", w_current->postscript_prolog);
    g_free(buf);  /* If we got to here, the buffer was allocated. */
    return -1;
  }
  /* Loop while reading file into buffer and dump it
   * back out to the supplied file handle
   */
  do {
    bytes = fread(buf, 1, PROLOG_BUFFER_SIZE, prolog);
    if(ferror(prolog)) break;
    fwrite(buf, 1, bytes, fp);
  } while(!feof(prolog) && !ferror(prolog) && !ferror(fp));

  if(ferror(prolog)) {
    s_log_message("Error during reading of the prolog file `%s' "
		  "in f_print_header()\n"
		  "Giving up on printing\n", w_current->postscript_prolog);
    g_free(buf);  /* If we got to here, the buffer was allocated. */
    return -1;
  }

  if(ferror(fp)) {
    s_log_message("Error during writing of the output postscript file "
		  "in f_print_header()\n"
		  "Giving up on printing\n");
    g_free(buf);  /* If we got to here, the buffer was allocated. */
    return -1;
  }
  g_free(buf);  /* If we got to here, the buffer was allocated. */

  fprintf(fp,"%%%%EndProlog\n"
	  "%%%%Page: 1 1\n");     /* Just name it `page 1' for now */
  
  /* These go after the prolog ends because they affect the graphics
   * state
   */

  return 0;
}

/*! \brief Prints the footer to a postscript document
 *  \par Function Description
 *  This function will print a document footer and end
 *  a postscript document.
 *  
 *  \param [in] fp The postscript document to write footer to.
 */
void f_print_footer(FILE *fp)
{
  fprintf(fp,"showpage\n"
	  "%%%%End\n");
}

/*! \brief Print all objects from the w_current TOPLEVEL object.
 *  \par Function Description
 *  This function will parse the head parameter for all objects
 *  and write the to the postscript document.
 *
 *  \param [in] w_current      The current TOPLEVEL object.
 *  \param [in] fp             The postscript document to print to.
 *  \param [in] head           Container for objects to be printed.
 *  \param [in] start_x        X origin on page to start printing objects.
 *  \param [in] start_y        Y origin on page to start printing objects.
 *  \param [in] scale          Scale factor for object output.
 *  \param [in] unicode_count  Number of items in unicode table.
 *  \param [in] unicode_table  Table of unicode items.
 *  \return void
 *
 *  \todo  what happens if snap is off? hack deal with this !!!!!!!!
 */
void f_print_objects(TOPLEVEL *w_current, FILE *fp, OBJECT *head, 
		     int start_x, int start_y, float scale, 
		     int unicode_count, gunichar *unicode_table)
{
  OBJECT *o_current=NULL;
  int origin_x, origin_y;
	
  origin_x = start_x;
  origin_y = start_y;

  if (head == NULL) {
    return;
  }

  /* Apply a translation to move the origin to where we want it */
  if (origin_x != 0 || origin_y != 0) {
    fprintf(fp, "%d %d translate\n", -origin_x, -origin_y);
  } 


  /* no longer change the coords, the postscript translate takes care 
   * of this */
  origin_x = 0;
  origin_y = 0;

  o_current = head;

  while ( o_current != NULL ) {

    if (o_current->type != OBJ_HEAD) {

      switch (o_current->type) {
        case(OBJ_LINE):
          o_line_print(w_current, fp, o_current,
                       origin_x, origin_y);
          break;
	
        case(OBJ_NET):
          o_net_print(w_current, fp, o_current,
                      origin_x, origin_y);
          break;

        case(OBJ_BUS):
          o_bus_print(w_current, fp, o_current,
                      origin_x, origin_y);
          break;
	
        case(OBJ_BOX):
          o_box_print(w_current, fp, o_current,
                      origin_x, origin_y);
          break;
			
        case(OBJ_CIRCLE):
          o_circle_print(w_current, fp, o_current,
                         origin_x, origin_y);
          break;

        case(OBJ_COMPLEX):
        case(OBJ_PLACEHOLDER): /* new object -- 1.20.2005 SDB */
          fprintf(fp, "gsave\n");

          f_print_objects(w_current, fp, 
                          o_current->complex->prim_objs,
                          origin_x, origin_y, scale,
			  unicode_count, unicode_table);
          fprintf(fp, "grestore\n");
          break;

        case(OBJ_TEXT):
          if (o_current->visibility == VISIBLE) {
	    int vectors = 0;

	    /* Look at flags and determine if vector text should
	     * be used for output.
	     */
	    if(w_current->text_output == VECTOR_FONTS) {
	      vectors = 1;
	    } else if( (w_current->print_vector_threshold > 0)
			&& ( o_text_num_lines(o_current->text->string) > 
			     w_current->print_vector_threshold)) {
	      vectors = 1;
	    }
            
            fprintf(fp, "gsave\n");
			
            if (vectors)
            {	
	      /* Output vectors */
              f_print_objects(w_current, 
                              fp, 
                              o_current->text->
                              prim_objs,
                              origin_x, origin_y, 
                              scale, unicode_count, unicode_table);
            } else {
	      /* Output text */
              o_text_print(w_current, fp, 
                           o_current,
                           origin_x, origin_y, unicode_count, unicode_table);
            }

            fprintf(fp, "grestore\n");
          }
          break;


        case(OBJ_PIN):
          o_pin_print(w_current, fp, o_current,
                      origin_x, origin_y);
          break;
	
        case(OBJ_ARC):
          o_arc_print(w_current, fp, o_current,
                      origin_x, origin_y);
          break;

  	case(OBJ_PICTURE):
          o_picture_print(w_current, fp, o_current,
			  origin_x, origin_y);
	  break;

        default:
          fprintf(stderr, "Error type!\n");
          exit(-1);
          break;
      }

    } 
    o_current = o_current->next;
  }

  s_cue_output_all(w_current, head, fp, POSTSCRIPT);
  return;
}

/*! \brief Print the current TOPLEVEL object to a file as a postscript
 *  document.  
 *
 *  \par Function Description
 *
 *  \param [in] w_current  The TOPLEVEL object to print.
 *  \param [in] filename   The file name of the output postscript document.
 *  \return 0 on success, -1 on failure.
 */
int f_print_file (TOPLEVEL *w_current, const char *filename)
{
  FILE *fp;
  int result;

  /* dots are breaking my filename selection hack hack !!!! */
  fp = fopen(filename, "wb"); /* Use "wb" for safety on e.g. Win32 */

  /* check to see if it worked */ 
  if (fp == NULL) {
    s_log_message("Could not open [%s] for printing\n", filename);
    return -1;
  }

  result = f_print_stream(w_current, fp);
  
  if (result != 0) {
    /* If there was an error in f_print_stream, then unlink the output file. */
    unlink(filename);
  }
  fclose (fp);
  return result;
}

/*! \brief Opens a pipe to the specified command and prints the
 *  current TOPLEVEL object to the pipe as a postscript document.
 *
 *  \par Function Description
 *
 *  \param [in] w_current  The TOPLEVEL object to print.
 *  \param [in] command    The command to print to.
 *  \return 0 on success, -1 on failure.
 */
int f_print_command (TOPLEVEL *w_current, const char *command)
{
  FILE *fp;
  int result;

  fp = popen (command, "w");

  /* check to see if it worked */ 
  if (fp == NULL) 
    {
      s_log_message("Could not execute command [%s] for printing\n", command);
      return -1;
    }

  result = f_print_stream (w_current, fp);
  pclose (fp);
  return result;
}

/*! \brief Print the current TOPLEVEL object to a stream as a
 *  postscript document.  
 *  \par Function Description
 *
 *  \param [in] w_current  The TOPLEVEL object to print.
 *  \param [in] fp         A pointer to an open IO stream
 *  \return 0 on success, -1 on failure.
 */

int f_print_stream(TOPLEVEL *w_current, FILE *fp)
{
  int origin_x, origin_y, bottom, right;
  int margin_x, margin_y;
  int dx,dy;
  float scale;
  int unicode_count;
  gunichar unicode_table [128];  /* to contain the list of unicode
				    characters that need mapping */


  /* Unicode support */
  f_print_initialize_glyph_table();  /* Fill up unicode map */

  /* Find all the unicode characters */
  unicode_count = f_print_get_unicode_chars(w_current,
			 w_current->page_current->object_head, 
			 0, unicode_table);

  /*	printf("%d %d\n", w_current->paper_width, w_current->paper_height);*/

  world_get_object_list_bounds(w_current,
                               w_current->page_current->object_head,
                               &origin_x, &origin_y,
                               &right, &bottom);

  /* Calculate scale factor that will make the image fit on the page */
  dx = 0; dy = 0;
  margin_x = 0; margin_y = 0;
  switch (w_current->print_output_type) {
  case EXTENTS:
    dx = right  - origin_x;
    dy = bottom - origin_y;
    
    /* Add a 10% margin */
    margin_x = dx/10;
    margin_y = dy/10;
    dx = dx + margin_x;
    dy = dy + margin_y;
    break;

  case WINDOW:
    dx = w_current->page_current->right - w_current->page_current->left;
    dy = w_current->page_current->bottom - w_current->page_current->top;
    origin_x = w_current->page_current->left;
    origin_y = w_current->page_current->top;
    right = w_current->page_current->right;
    bottom = w_current->page_current->bottom;
    break;

  case EXTENTS_NOMARGINS:
    dx = right  - origin_x;
    dy = bottom - origin_y;
    break;

  default:
    break;

  }

  scale = 0.0;
  if(w_current->print_orientation == LANDSCAPE) {
    /* First attempt to fit in x direction. */
    scale = w_current->paper_width / (float)dx;
    if((w_current->paper_height / (float)dy) < scale ) {
      /* Else fit with y direction */
      scale = (w_current->paper_height / (float)dy);
    }
  } else { /* portrait */
    /* First attempt to fit in y direction. */
    scale = w_current->paper_width / (float) dy;
    if((w_current->paper_height / (float)dx) < scale ) {
      /* Else fit with x direction */
      scale = (w_current->paper_height / (float)dx);
    }
  }

#if 0
  /* Debug */
  printf("dx: %d dy:%d, origin_x:%d origin_y:%d, right:%d bottom:%d\n",
	 dx,dy,origin_x,origin_y,right,bottom);
  printf("scale:%f\n",scale);
#endif  

  /* Output the header */
  if (f_print_header(w_current, fp, 
		     w_current->paper_width, 
		     w_current->paper_height) != 0) {

    /* There was an error in f_print_header */
    return -1;
  }

  /* Output font re-encoding */
  if (unicode_count) {
    f_print_unicode_map(fp, unicode_count, unicode_table);
    /* output font re-encodeing command, use our re-encoding */
    fprintf(fp,"/gEDAFont UTFencoding /Helvetica RE\n");
  } else {
    /* Otherwise, use LATIN1 extended encoding from prolog */
    fprintf(fp,"/gEDAFont ISOLatin1Extended /Helvetica RE\n");
  }


  /* XXX - Do page orientation selection */

/*   if (w_current->setpagedevice_orientation) { */
/*     if (w_current->print_orientation == LANDSCAPE) { */
/*       fprintf(fp, "%c%c /Orientation 1 %c%c setpagedevice\n\n", '<', '<', */
/*               '>', '>'); */
/*     } else { */
/*       fprintf(fp, "%c%c /Orientation 0 %c%c setpagedevice\n\n", '<', '<', */
/*               '>', '>'); */
/*     } */
/*   } */

  /* the height and width are in the right order here, since they were
   * specified in landscape order in the system-gschemrc file.
   * \074 is '<', \076 is '>' */
  if (w_current->setpagedevice_pagesize) {
    fprintf(fp, "\074\074 /PageSize [%d %d] \076\076 setpagedevice\n",
            (w_current->paper_height * 72) / MILS_PER_INCH,
            (w_current->paper_width * 72) / MILS_PER_INCH);
  }


  /* Set line end style */
  if (w_current->print_output_capstyle == BUTT_CAP) {
    fprintf(fp, "0 setlinecap\n");
  } else if (w_current->print_output_capstyle == SQUARE_CAP) {
    fprintf(fp, "2 setlinecap\n");
  } else if (w_current->print_output_capstyle == ROUND_CAP) {
    fprintf(fp, "1 setlinecap\n");
  }

  /* Apply mils to postscript native units scaling to CTM */
  fprintf(fp,"%f %f scale\n",
	  72.0 / 1000.0 , 72.0 / 1000.0);

  /* Now the output is defined in terms of mils */
  /* Draw a box with the background colour covering the whole page */
  if (w_current->print_color) {
    f_print_set_color(fp, w_current->print_color_background);
    fprintf(fp,"%d %d 0 0 fbox\n",
	    w_current->paper_height,
	    w_current->paper_width);
  }

  /* Now rotate and translate the graphics to fit onto the desired
   * page with the orientation we want. Center it too */
  if(w_current->print_orientation == LANDSCAPE) {
    fprintf(fp,
	    "%d %d translate 90 rotate\n",
	    (int)((w_current->paper_height + ( dy-margin_y) * scale)/2.0),
	    (int)((w_current->paper_width  + (-dx+margin_x) * scale)/2.0));
  } else { /* portrait */
    fprintf(fp,"%d %d translate\n",
	    (int)((w_current->paper_height + (-dx + margin_x) * scale)/2.0),
	    (int)((w_current->paper_width  + (-dy + margin_y) * scale)/2.0));

  }

  /* Now apply final mils to output scaling factor */
  fprintf(fp,"%f %f scale\n",
	  scale, scale);

  /* Print the objects */
  f_print_objects(w_current, fp, 
		  w_current->page_current->object_head,
		  origin_x, origin_y, scale, unicode_count, unicode_table);

  f_print_footer(fp);

  return(0);
}

/*! \brief Sets the current TOPLEVEL object output type
 *  \par Function Description
 *  Sets the current TOPLEVEL object output type.
 *
 *  \param [in,out] w_current  The TOPLEVEL object to set output type in.
 *  \param [in]     type       The print type to set.
 */
void f_print_set_type(TOPLEVEL *w_current, int type)
{
  w_current->print_output_type = type;
}

/*! \brief Converts all text strings to unicode format.
 *  \par Function Description
 *  Converts all text strings to unicode format.
 *
 *  \param [in,out] w_current  The output TOPLEVEL element to store converted
 *                             strings in.
 *  \param [in]     head       The object containing strings for conversion.
 *  \param [in]     count      The number of elements in the unicode table.
 *  \param [in]     table      The unicode table.
 *  \return count on success, 0 otherwise.
 */
static int f_print_get_unicode_chars(TOPLEVEL *w_current, OBJECT *head,
				     int count, gunichar *table)
{
  OBJECT *o_current = NULL;
  gchar *aux;
  gunichar current_char;
  int i, found;

  if (head == NULL) {
    return(0);
  }

  o_current = head;

  while (o_current != NULL) {

    if (o_current->type != OBJ_HEAD) {

      switch (o_current->type) {

      case (OBJ_COMPLEX):
      case (OBJ_PLACEHOLDER):	/* new object -- 1.20.2005 SDB */

	count = f_print_get_unicode_chars(w_current, 
                                          o_current->complex->prim_objs, 
                                          count, table);
	break;

      case (OBJ_TEXT):
	if (o_current->visibility == VISIBLE) {

          aux = o_current->text->string;
          while (aux && ((gunichar) (*aux) != 0)) {
            current_char = g_utf8_get_char_validated(aux, -1);
            if (current_char >= 127) {
	      found = 0;  
	      i = 0;
	      while (i < count) {
	        if (table[i] == current_char)
	          found = 1;
		i++;  
	      }
              if (!found) {
	        if (count < 128)
		  table[count++] = current_char;
	        else 
	          s_log_message("Too many UTF-8 characters, cannot print\n");
	      }
	    }  
            aux = g_utf8_find_next_char(aux, NULL);
	  }
	}
	break;

      default:
	break;
      }
    }
    o_current = o_current->next;
  }
  return (count);
}

/*! \brief Prints unicode map to postscript document.
 *  \par Function Description
 *  Prints unicode map to postscript document.
 *
 *  \param [in] fp     The postscript document to write unicode map to.
 *  \param [in] count  Size of unicode map table.
 *  \param [in] table  The unicode map to write.
 */
static void f_print_unicode_map(FILE * fp, int count, gunichar * table)
{
  unsigned int i;
  int line_count;
  char *glyph_map[256];     /* To contain the postscript remapping */

  /* update the glyph map, undefine everything */
  for(i=0; i<256; i++) glyph_map[i]="/.notdef";

  /* Now fill in the active characters */
  for (i=0; i<128; i++) {  /* Copy in the regular latin chars */
      glyph_map[i] = g_hash_table_lookup (unicode_char_to_glyph,
					  GUINT_TO_POINTER (i));
  }
  /* Finish by using up the rest of the spares */
  for (i=128; i<(count+128); i++) { 
    if(i < (count+128)) {
      glyph_map[i] = g_hash_table_lookup (unicode_char_to_glyph,
					  GUINT_TO_POINTER (table[i-128]));
    }
  }

  fprintf(fp, "%%%%BeginResource: encoding UTFencoding\n");
  fprintf(fp, "/UTFencoding [\n");
  
  /* Output the re-encoding vector, prettily */
  line_count = 0;
  for (i=0; i<256; i++) {
      line_count += fprintf(fp, "%s ", glyph_map[i]);
      if(line_count > 60) {
	line_count = 0;
	fprintf(fp, "\n");
      }
  }

  fprintf(fp, "] def\n");
  fprintf(fp, "%%%%EndResource\n");
}


/*! \brief */
static struct glyph_list {
  gpointer key;
  gpointer name;
} glyphs[] = {
  { GUINT_TO_POINTER (0x0000), "/.notdef" },
  { GUINT_TO_POINTER (0x0041), "/A" },
  { GUINT_TO_POINTER (0x00C6), "/AE" },
  { GUINT_TO_POINTER (0x01FC), "/AEacute" },
  { GUINT_TO_POINTER (0x01E2), "/AEmacron" },
  { GUINT_TO_POINTER (0xF7E6), "/AEsmall" },
  { GUINT_TO_POINTER (0x00C1), "/Aacute" },
  { GUINT_TO_POINTER (0xF7E1), "/Aacutesmall" },
  { GUINT_TO_POINTER (0x0102), "/Abreve" },
  { GUINT_TO_POINTER (0x1EAE), "/Abreveacute" },
  { GUINT_TO_POINTER (0x04D0), "/Abrevecyrillic" },
  { GUINT_TO_POINTER (0x1EB6), "/Abrevedotbelow" },
  { GUINT_TO_POINTER (0x1EB0), "/Abrevegrave" },
  { GUINT_TO_POINTER (0x1EB2), "/Abrevehookabove" },
  { GUINT_TO_POINTER (0x1EB4), "/Abrevetilde" },
  { GUINT_TO_POINTER (0x01CD), "/Acaron" },
  { GUINT_TO_POINTER (0x24B6), "/Acircle" },
  { GUINT_TO_POINTER (0x00C2), "/Acircumflex" },
  { GUINT_TO_POINTER (0x1EA4), "/Acircumflexacute" },
  { GUINT_TO_POINTER (0x1EAC), "/Acircumflexdotbelow" },
  { GUINT_TO_POINTER (0x1EA6), "/Acircumflexgrave" },
  { GUINT_TO_POINTER (0x1EA8), "/Acircumflexhookabove" },
  { GUINT_TO_POINTER (0xF7E2), "/Acircumflexsmall" },
  { GUINT_TO_POINTER (0x1EAA), "/Acircumflextilde" },
  { GUINT_TO_POINTER (0xF6C9), "/Acute" },
  { GUINT_TO_POINTER (0xF7B4), "/Acutesmall" },
  { GUINT_TO_POINTER (0x0410), "/Acyrillic" },
  { GUINT_TO_POINTER (0x0200), "/Adblgrave" },
  { GUINT_TO_POINTER (0x00C4), "/Adieresis" },
  { GUINT_TO_POINTER (0x04D2), "/Adieresiscyrillic" },
  { GUINT_TO_POINTER (0x01DE), "/Adieresismacron" },
  { GUINT_TO_POINTER (0xF7E4), "/Adieresissmall" },
  { GUINT_TO_POINTER (0x1EA0), "/Adotbelow" },
  { GUINT_TO_POINTER (0x01E0), "/Adotmacron" },
  { GUINT_TO_POINTER (0x00C0), "/Agrave" },
  { GUINT_TO_POINTER (0xF7E0), "/Agravesmall" },
  { GUINT_TO_POINTER (0x1EA2), "/Ahookabove" },
  { GUINT_TO_POINTER (0x04D4), "/Aiecyrillic" },
  { GUINT_TO_POINTER (0x0202), "/Ainvertedbreve" },
  { GUINT_TO_POINTER (0x0391), "/Alpha" },
  { GUINT_TO_POINTER (0x0386), "/Alphatonos" },
  { GUINT_TO_POINTER (0x0100), "/Amacron" },
  { GUINT_TO_POINTER (0xFF21), "/Amonospace" },
  { GUINT_TO_POINTER (0x0104), "/Aogonek" },
  { GUINT_TO_POINTER (0x00C5), "/Aring" },
  { GUINT_TO_POINTER (0x01FA), "/Aringacute" },
  { GUINT_TO_POINTER (0x1E00), "/Aringbelow" },
  { GUINT_TO_POINTER (0xF7E5), "/Aringsmall" },
  { GUINT_TO_POINTER (0xF761), "/Asmall" },
  { GUINT_TO_POINTER (0x00C3), "/Atilde" },
  { GUINT_TO_POINTER (0xF7E3), "/Atildesmall" },
  { GUINT_TO_POINTER (0x0531), "/Aybarmenian" },
  { GUINT_TO_POINTER (0x0042), "/B" },
  { GUINT_TO_POINTER (0x24B7), "/Bcircle" },
  { GUINT_TO_POINTER (0x1E02), "/Bdotaccent" },
  { GUINT_TO_POINTER (0x1E04), "/Bdotbelow" },
  { GUINT_TO_POINTER (0x0411), "/Becyrillic" },
  { GUINT_TO_POINTER (0x0532), "/Benarmenian" },
  { GUINT_TO_POINTER (0x0392), "/Beta" },
  { GUINT_TO_POINTER (0x0181), "/Bhook" },
  { GUINT_TO_POINTER (0x1E06), "/Blinebelow" },
  { GUINT_TO_POINTER (0xFF22), "/Bmonospace" },
  { GUINT_TO_POINTER (0xF6F4), "/Brevesmall" },
  { GUINT_TO_POINTER (0xF762), "/Bsmall" },
  { GUINT_TO_POINTER (0x0182), "/Btopbar" },
  { GUINT_TO_POINTER (0x0043), "/C" },
  { GUINT_TO_POINTER (0x053E), "/Caarmenian" },
  { GUINT_TO_POINTER (0x0106), "/Cacute" },
  { GUINT_TO_POINTER (0xF6CA), "/Caron" },
  { GUINT_TO_POINTER (0xF6F5), "/Caronsmall" },
  { GUINT_TO_POINTER (0x010C), "/Ccaron" },
  { GUINT_TO_POINTER (0x00C7), "/Ccedilla" },
  { GUINT_TO_POINTER (0x1E08), "/Ccedillaacute" },
  { GUINT_TO_POINTER (0xF7E7), "/Ccedillasmall" },
  { GUINT_TO_POINTER (0x24B8), "/Ccircle" },
  { GUINT_TO_POINTER (0x0108), "/Ccircumflex" },
  { GUINT_TO_POINTER (0x010A), "/Cdot" },
  { GUINT_TO_POINTER (0x010A), "/Cdotaccent" },
  { GUINT_TO_POINTER (0xF7B8), "/Cedillasmall" },
  { GUINT_TO_POINTER (0x0549), "/Chaarmenian" },
  { GUINT_TO_POINTER (0x04BC), "/Cheabkhasiancyrillic" },
  { GUINT_TO_POINTER (0x0427), "/Checyrillic" },
  { GUINT_TO_POINTER (0x04BE), "/Chedescenderabkhasiancyrillic" },
  { GUINT_TO_POINTER (0x04B6), "/Chedescendercyrillic" },
  { GUINT_TO_POINTER (0x04F4), "/Chedieresiscyrillic" },
  { GUINT_TO_POINTER (0x0543), "/Cheharmenian" },
  { GUINT_TO_POINTER (0x04CB), "/Chekhakassiancyrillic" },
  { GUINT_TO_POINTER (0x04B8), "/Cheverticalstrokecyrillic" },
  { GUINT_TO_POINTER (0x03A7), "/Chi" },
  { GUINT_TO_POINTER (0x0187), "/Chook" },
  { GUINT_TO_POINTER (0xF6F6), "/Circumflexsmall" },
  { GUINT_TO_POINTER (0xFF23), "/Cmonospace" },
  { GUINT_TO_POINTER (0x0551), "/Coarmenian" },
  { GUINT_TO_POINTER (0xF763), "/Csmall" },
  { GUINT_TO_POINTER (0x0044), "/D" },
  { GUINT_TO_POINTER (0x01F1), "/DZ" },
  { GUINT_TO_POINTER (0x01C4), "/DZcaron" },
  { GUINT_TO_POINTER (0x0534), "/Daarmenian" },
  { GUINT_TO_POINTER (0x0189), "/Dafrican" },
  { GUINT_TO_POINTER (0x010E), "/Dcaron" },
  { GUINT_TO_POINTER (0x1E10), "/Dcedilla" },
  { GUINT_TO_POINTER (0x24B9), "/Dcircle" },
  { GUINT_TO_POINTER (0x1E12), "/Dcircumflexbelow" },
  { GUINT_TO_POINTER (0x0110), "/Dcroat" },
  { GUINT_TO_POINTER (0x1E0A), "/Ddotaccent" },
  { GUINT_TO_POINTER (0x1E0C), "/Ddotbelow" },
  { GUINT_TO_POINTER (0x0414), "/Decyrillic" },
  { GUINT_TO_POINTER (0x03EE), "/Deicoptic" },
  { GUINT_TO_POINTER (0x2206), "/Delta" },
  { GUINT_TO_POINTER (0x0394), "/Deltagreek" },
  { GUINT_TO_POINTER (0x018A), "/Dhook" },
  { GUINT_TO_POINTER (0xF6CB), "/Dieresis" },
  { GUINT_TO_POINTER (0xF6CC), "/DieresisAcute" },
  { GUINT_TO_POINTER (0xF6CD), "/DieresisGrave" },
  { GUINT_TO_POINTER (0xF7A8), "/Dieresissmall" },
  { GUINT_TO_POINTER (0x03DC), "/Digammagreek" },
  { GUINT_TO_POINTER (0x0402), "/Djecyrillic" },
  { GUINT_TO_POINTER (0x1E0E), "/Dlinebelow" },
  { GUINT_TO_POINTER (0xFF24), "/Dmonospace" },
  { GUINT_TO_POINTER (0xF6F7), "/Dotaccentsmall" },
  { GUINT_TO_POINTER (0x0110), "/Dslash" },
  { GUINT_TO_POINTER (0xF764), "/Dsmall" },
  { GUINT_TO_POINTER (0x018B), "/Dtopbar" },
  { GUINT_TO_POINTER (0x01F2), "/Dz" },
  { GUINT_TO_POINTER (0x01C5), "/Dzcaron" },
  { GUINT_TO_POINTER (0x04E0), "/Dzeabkhasiancyrillic" },
  { GUINT_TO_POINTER (0x0405), "/Dzecyrillic" },
  { GUINT_TO_POINTER (0x040F), "/Dzhecyrillic" },
  { GUINT_TO_POINTER (0x0045), "/E" },
  { GUINT_TO_POINTER (0x00C9), "/Eacute" },
  { GUINT_TO_POINTER (0xF7E9), "/Eacutesmall" },
  { GUINT_TO_POINTER (0x0114), "/Ebreve" },
  { GUINT_TO_POINTER (0x011A), "/Ecaron" },
  { GUINT_TO_POINTER (0x1E1C), "/Ecedillabreve" },
  { GUINT_TO_POINTER (0x0535), "/Echarmenian" },
  { GUINT_TO_POINTER (0x24BA), "/Ecircle" },
  { GUINT_TO_POINTER (0x00CA), "/Ecircumflex" },
  { GUINT_TO_POINTER (0x1EBE), "/Ecircumflexacute" },
  { GUINT_TO_POINTER (0x1E18), "/Ecircumflexbelow" },
  { GUINT_TO_POINTER (0x1EC6), "/Ecircumflexdotbelow" },
  { GUINT_TO_POINTER (0x1EC0), "/Ecircumflexgrave" },
  { GUINT_TO_POINTER (0x1EC2), "/Ecircumflexhookabove" },
  { GUINT_TO_POINTER (0xF7EA), "/Ecircumflexsmall" },
  { GUINT_TO_POINTER (0x1EC4), "/Ecircumflextilde" },
  { GUINT_TO_POINTER (0x0404), "/Ecyrillic" },
  { GUINT_TO_POINTER (0x0204), "/Edblgrave" },
  { GUINT_TO_POINTER (0x00CB), "/Edieresis" },
  { GUINT_TO_POINTER (0xF7EB), "/Edieresissmall" },
  { GUINT_TO_POINTER (0x0116), "/Edot" },
  { GUINT_TO_POINTER (0x0116), "/Edotaccent" },
  { GUINT_TO_POINTER (0x1EB8), "/Edotbelow" },
  { GUINT_TO_POINTER (0x0424), "/Efcyrillic" },
  { GUINT_TO_POINTER (0x00C8), "/Egrave" },
  { GUINT_TO_POINTER (0xF7E8), "/Egravesmall" },
  { GUINT_TO_POINTER (0x0537), "/Eharmenian" },
  { GUINT_TO_POINTER (0x1EBA), "/Ehookabove" },
  { GUINT_TO_POINTER (0x2167), "/Eightroman" },
  { GUINT_TO_POINTER (0x0206), "/Einvertedbreve" },
  { GUINT_TO_POINTER (0x0464), "/Eiotifiedcyrillic" },
  { GUINT_TO_POINTER (0x041B), "/Elcyrillic" },
  { GUINT_TO_POINTER (0x216A), "/Elevenroman" },
  { GUINT_TO_POINTER (0x0112), "/Emacron" },
  { GUINT_TO_POINTER (0x1E16), "/Emacronacute" },
  { GUINT_TO_POINTER (0x1E14), "/Emacrongrave" },
  { GUINT_TO_POINTER (0x041C), "/Emcyrillic" },
  { GUINT_TO_POINTER (0xFF25), "/Emonospace" },
  { GUINT_TO_POINTER (0x041D), "/Encyrillic" },
  { GUINT_TO_POINTER (0x04A2), "/Endescendercyrillic" },
  { GUINT_TO_POINTER (0x014A), "/Eng" },
  { GUINT_TO_POINTER (0x04A4), "/Enghecyrillic" },
  { GUINT_TO_POINTER (0x04C7), "/Enhookcyrillic" },
  { GUINT_TO_POINTER (0x0118), "/Eogonek" },
  { GUINT_TO_POINTER (0x0190), "/Eopen" },
  { GUINT_TO_POINTER (0x0395), "/Epsilon" },
  { GUINT_TO_POINTER (0x0388), "/Epsilontonos" },
  { GUINT_TO_POINTER (0x0420), "/Ercyrillic" },
  { GUINT_TO_POINTER (0x018E), "/Ereversed" },
  { GUINT_TO_POINTER (0x042D), "/Ereversedcyrillic" },
  { GUINT_TO_POINTER (0x0421), "/Escyrillic" },
  { GUINT_TO_POINTER (0x04AA), "/Esdescendercyrillic" },
  { GUINT_TO_POINTER (0x01A9), "/Esh" },
  { GUINT_TO_POINTER (0xF765), "/Esmall" },
  { GUINT_TO_POINTER (0x0397), "/Eta" },
  { GUINT_TO_POINTER (0x0538), "/Etarmenian" },
  { GUINT_TO_POINTER (0x0389), "/Etatonos" },
  { GUINT_TO_POINTER (0x00D0), "/Eth" },
  { GUINT_TO_POINTER (0xF7F0), "/Ethsmall" },
  { GUINT_TO_POINTER (0x1EBC), "/Etilde" },
  { GUINT_TO_POINTER (0x1E1A), "/Etildebelow" },
  { GUINT_TO_POINTER (0x20AC), "/Euro" },
  { GUINT_TO_POINTER (0x01B7), "/Ezh" },
  { GUINT_TO_POINTER (0x01EE), "/Ezhcaron" },
  { GUINT_TO_POINTER (0x01B8), "/Ezhreversed" },
  { GUINT_TO_POINTER (0x0046), "/F" },
  { GUINT_TO_POINTER (0x24BB), "/Fcircle" },
  { GUINT_TO_POINTER (0x1E1E), "/Fdotaccent" },
  { GUINT_TO_POINTER (0x0556), "/Feharmenian" },
  { GUINT_TO_POINTER (0x03E4), "/Feicoptic" },
  { GUINT_TO_POINTER (0x0191), "/Fhook" },
  { GUINT_TO_POINTER (0x0472), "/Fitacyrillic" },
  { GUINT_TO_POINTER (0x2164), "/Fiveroman" },
  { GUINT_TO_POINTER (0xFF26), "/Fmonospace" },
  { GUINT_TO_POINTER (0x2163), "/Fourroman" },
  { GUINT_TO_POINTER (0xF766), "/Fsmall" },
  { GUINT_TO_POINTER (0x0047), "/G" },
  { GUINT_TO_POINTER (0x3387), "/GBsquare" },
  { GUINT_TO_POINTER (0x01F4), "/Gacute" },
  { GUINT_TO_POINTER (0x0393), "/Gamma" },
  { GUINT_TO_POINTER (0x0194), "/Gammaafrican" },
  { GUINT_TO_POINTER (0x03EA), "/Gangiacoptic" },
  { GUINT_TO_POINTER (0x011E), "/Gbreve" },
  { GUINT_TO_POINTER (0x01E6), "/Gcaron" },
  { GUINT_TO_POINTER (0x0122), "/Gcedilla" },
  { GUINT_TO_POINTER (0x24BC), "/Gcircle" },
  { GUINT_TO_POINTER (0x011C), "/Gcircumflex" },
  { GUINT_TO_POINTER (0x0122), "/Gcommaaccent" },
  { GUINT_TO_POINTER (0x0120), "/Gdot" },
  { GUINT_TO_POINTER (0x0120), "/Gdotaccent" },
  { GUINT_TO_POINTER (0x0413), "/Gecyrillic" },
  { GUINT_TO_POINTER (0x0542), "/Ghadarmenian" },
  { GUINT_TO_POINTER (0x0494), "/Ghemiddlehookcyrillic" },
  { GUINT_TO_POINTER (0x0492), "/Ghestrokecyrillic" },
  { GUINT_TO_POINTER (0x0490), "/Gheupturncyrillic" },
  { GUINT_TO_POINTER (0x0193), "/Ghook" },
  { GUINT_TO_POINTER (0x0533), "/Gimarmenian" },
  { GUINT_TO_POINTER (0x0403), "/Gjecyrillic" },
  { GUINT_TO_POINTER (0x1E20), "/Gmacron" },
  { GUINT_TO_POINTER (0xFF27), "/Gmonospace" },
  { GUINT_TO_POINTER (0xF6CE), "/Grave" },
  { GUINT_TO_POINTER (0xF760), "/Gravesmall" },
  { GUINT_TO_POINTER (0xF767), "/Gsmall" },
  { GUINT_TO_POINTER (0x029B), "/Gsmallhook" },
  { GUINT_TO_POINTER (0x01E4), "/Gstroke" },
  { GUINT_TO_POINTER (0x0048), "/H" },
  { GUINT_TO_POINTER (0x25CF), "/H18533" },
  { GUINT_TO_POINTER (0x25AA), "/H18543" },
  { GUINT_TO_POINTER (0x25AB), "/H18551" },
  { GUINT_TO_POINTER (0x25A1), "/H22073" },
  { GUINT_TO_POINTER (0x33CB), "/HPsquare" },
  { GUINT_TO_POINTER (0x04A8), "/Haabkhasiancyrillic" },
  { GUINT_TO_POINTER (0x04B2), "/Hadescendercyrillic" },
  { GUINT_TO_POINTER (0x042A), "/Hardsigncyrillic" },
  { GUINT_TO_POINTER (0x0126), "/Hbar" },
  { GUINT_TO_POINTER (0x1E2A), "/Hbrevebelow" },
  { GUINT_TO_POINTER (0x1E28), "/Hcedilla" },
  { GUINT_TO_POINTER (0x24BD), "/Hcircle" },
  { GUINT_TO_POINTER (0x0124), "/Hcircumflex" },
  { GUINT_TO_POINTER (0x1E26), "/Hdieresis" },
  { GUINT_TO_POINTER (0x1E22), "/Hdotaccent" },
  { GUINT_TO_POINTER (0x1E24), "/Hdotbelow" },
  { GUINT_TO_POINTER (0xFF28), "/Hmonospace" },
  { GUINT_TO_POINTER (0x0540), "/Hoarmenian" },
  { GUINT_TO_POINTER (0x03E8), "/Horicoptic" },
  { GUINT_TO_POINTER (0xF768), "/Hsmall" },
  { GUINT_TO_POINTER (0xF6CF), "/Hungarumlaut" },
  { GUINT_TO_POINTER (0xF6F8), "/Hungarumlautsmall" },
  { GUINT_TO_POINTER (0x3390), "/Hzsquare" },
  { GUINT_TO_POINTER (0x0049), "/I" },
  { GUINT_TO_POINTER (0x042F), "/IAcyrillic" },
  { GUINT_TO_POINTER (0x0132), "/IJ" },
  { GUINT_TO_POINTER (0x042E), "/IUcyrillic" },
  { GUINT_TO_POINTER (0x00CD), "/Iacute" },
  { GUINT_TO_POINTER (0xF7ED), "/Iacutesmall" },
  { GUINT_TO_POINTER (0x012C), "/Ibreve" },
  { GUINT_TO_POINTER (0x01CF), "/Icaron" },
  { GUINT_TO_POINTER (0x24BE), "/Icircle" },
  { GUINT_TO_POINTER (0x00CE), "/Icircumflex" },
  { GUINT_TO_POINTER (0xF7EE), "/Icircumflexsmall" },
  { GUINT_TO_POINTER (0x0406), "/Icyrillic" },
  { GUINT_TO_POINTER (0x0208), "/Idblgrave" },
  { GUINT_TO_POINTER (0x00CF), "/Idieresis" },
  { GUINT_TO_POINTER (0x1E2E), "/Idieresisacute" },
  { GUINT_TO_POINTER (0x04E4), "/Idieresiscyrillic" },
  { GUINT_TO_POINTER (0xF7EF), "/Idieresissmall" },
  { GUINT_TO_POINTER (0x0130), "/Idot" },
  { GUINT_TO_POINTER (0x0130), "/Idotaccent" },
  { GUINT_TO_POINTER (0x1ECA), "/Idotbelow" },
  { GUINT_TO_POINTER (0x04D6), "/Iebrevecyrillic" },
  { GUINT_TO_POINTER (0x0415), "/Iecyrillic" },
  { GUINT_TO_POINTER (0x2111), "/Ifraktur" },
  { GUINT_TO_POINTER (0x00CC), "/Igrave" },
  { GUINT_TO_POINTER (0xF7EC), "/Igravesmall" },
  { GUINT_TO_POINTER (0x1EC8), "/Ihookabove" },
  { GUINT_TO_POINTER (0x0418), "/Iicyrillic" },
  { GUINT_TO_POINTER (0x020A), "/Iinvertedbreve" },
  { GUINT_TO_POINTER (0x0419), "/Iishortcyrillic" },
  { GUINT_TO_POINTER (0x012A), "/Imacron" },
  { GUINT_TO_POINTER (0x04E2), "/Imacroncyrillic" },
  { GUINT_TO_POINTER (0xFF29), "/Imonospace" },
  { GUINT_TO_POINTER (0x053B), "/Iniarmenian" },
  { GUINT_TO_POINTER (0x0401), "/Iocyrillic" },
  { GUINT_TO_POINTER (0x012E), "/Iogonek" },
  { GUINT_TO_POINTER (0x0399), "/Iota" },
  { GUINT_TO_POINTER (0x0196), "/Iotaafrican" },
  { GUINT_TO_POINTER (0x03AA), "/Iotadieresis" },
  { GUINT_TO_POINTER (0x038A), "/Iotatonos" },
  { GUINT_TO_POINTER (0xF769), "/Ismall" },
  { GUINT_TO_POINTER (0x0197), "/Istroke" },
  { GUINT_TO_POINTER (0x0128), "/Itilde" },
  { GUINT_TO_POINTER (0x1E2C), "/Itildebelow" },
  { GUINT_TO_POINTER (0x0474), "/Izhitsacyrillic" },
  { GUINT_TO_POINTER (0x0476), "/Izhitsadblgravecyrillic" },
  { GUINT_TO_POINTER (0x004A), "/J" },
  { GUINT_TO_POINTER (0x0541), "/Jaarmenian" },
  { GUINT_TO_POINTER (0x24BF), "/Jcircle" },
  { GUINT_TO_POINTER (0x0134), "/Jcircumflex" },
  { GUINT_TO_POINTER (0x0408), "/Jecyrillic" },
  { GUINT_TO_POINTER (0x054B), "/Jheharmenian" },
  { GUINT_TO_POINTER (0xFF2A), "/Jmonospace" },
  { GUINT_TO_POINTER (0xF76A), "/Jsmall" },
  { GUINT_TO_POINTER (0x004B), "/K" },
  { GUINT_TO_POINTER (0x3385), "/KBsquare" },
  { GUINT_TO_POINTER (0x33CD), "/KKsquare" },
  { GUINT_TO_POINTER (0x04A0), "/Kabashkircyrillic" },
  { GUINT_TO_POINTER (0x1E30), "/Kacute" },
  { GUINT_TO_POINTER (0x041A), "/Kacyrillic" },
  { GUINT_TO_POINTER (0x049A), "/Kadescendercyrillic" },
  { GUINT_TO_POINTER (0x04C3), "/Kahookcyrillic" },
  { GUINT_TO_POINTER (0x039A), "/Kappa" },
  { GUINT_TO_POINTER (0x049E), "/Kastrokecyrillic" },
  { GUINT_TO_POINTER (0x049C), "/Kaverticalstrokecyrillic" },
  { GUINT_TO_POINTER (0x01E8), "/Kcaron" },
  { GUINT_TO_POINTER (0x0136), "/Kcedilla" },
  { GUINT_TO_POINTER (0x24C0), "/Kcircle" },
  { GUINT_TO_POINTER (0x0136), "/Kcommaaccent" },
  { GUINT_TO_POINTER (0x1E32), "/Kdotbelow" },
  { GUINT_TO_POINTER (0x0554), "/Keharmenian" },
  { GUINT_TO_POINTER (0x053F), "/Kenarmenian" },
  { GUINT_TO_POINTER (0x0425), "/Khacyrillic" },
  { GUINT_TO_POINTER (0x03E6), "/Kheicoptic" },
  { GUINT_TO_POINTER (0x0198), "/Khook" },
  { GUINT_TO_POINTER (0x040C), "/Kjecyrillic" },
  { GUINT_TO_POINTER (0x1E34), "/Klinebelow" },
  { GUINT_TO_POINTER (0xFF2B), "/Kmonospace" },
  { GUINT_TO_POINTER (0x0480), "/Koppacyrillic" },
  { GUINT_TO_POINTER (0x03DE), "/Koppagreek" },
  { GUINT_TO_POINTER (0x046E), "/Ksicyrillic" },
  { GUINT_TO_POINTER (0xF76B), "/Ksmall" },
  { GUINT_TO_POINTER (0x004C), "/L" },
  { GUINT_TO_POINTER (0x01C7), "/LJ" },
  { GUINT_TO_POINTER (0xF6BF), "/LL" },
  { GUINT_TO_POINTER (0x0139), "/Lacute" },
  { GUINT_TO_POINTER (0x039B), "/Lambda" },
  { GUINT_TO_POINTER (0x013D), "/Lcaron" },
  { GUINT_TO_POINTER (0x013B), "/Lcedilla" },
  { GUINT_TO_POINTER (0x24C1), "/Lcircle" },
  { GUINT_TO_POINTER (0x1E3C), "/Lcircumflexbelow" },
  { GUINT_TO_POINTER (0x013B), "/Lcommaaccent" },
  { GUINT_TO_POINTER (0x013F), "/Ldot" },
  { GUINT_TO_POINTER (0x013F), "/Ldotaccent" },
  { GUINT_TO_POINTER (0x1E36), "/Ldotbelow" },
  { GUINT_TO_POINTER (0x1E38), "/Ldotbelowmacron" },
  { GUINT_TO_POINTER (0x053C), "/Liwnarmenian" },
  { GUINT_TO_POINTER (0x01C8), "/Lj" },
  { GUINT_TO_POINTER (0x0409), "/Ljecyrillic" },
  { GUINT_TO_POINTER (0x1E3A), "/Llinebelow" },
  { GUINT_TO_POINTER (0xFF2C), "/Lmonospace" },
  { GUINT_TO_POINTER (0x0141), "/Lslash" },
  { GUINT_TO_POINTER (0xF6F9), "/Lslashsmall" },
  { GUINT_TO_POINTER (0xF76C), "/Lsmall" },
  { GUINT_TO_POINTER (0x004D), "/M" },
  { GUINT_TO_POINTER (0x3386), "/MBsquare" },
  { GUINT_TO_POINTER (0xF6D0), "/Macron" },
  { GUINT_TO_POINTER (0xF7AF), "/Macronsmall" },
  { GUINT_TO_POINTER (0x1E3E), "/Macute" },
  { GUINT_TO_POINTER (0x24C2), "/Mcircle" },
  { GUINT_TO_POINTER (0x1E40), "/Mdotaccent" },
  { GUINT_TO_POINTER (0x1E42), "/Mdotbelow" },
  { GUINT_TO_POINTER (0x0544), "/Menarmenian" },
  { GUINT_TO_POINTER (0xFF2D), "/Mmonospace" },
  { GUINT_TO_POINTER (0xF76D), "/Msmall" },
  { GUINT_TO_POINTER (0x019C), "/Mturned" },
  { GUINT_TO_POINTER (0x039C), "/Mu" },
  { GUINT_TO_POINTER (0x004E), "/N" },
  { GUINT_TO_POINTER (0x01CA), "/NJ" },
  { GUINT_TO_POINTER (0x0143), "/Nacute" },
  { GUINT_TO_POINTER (0x0147), "/Ncaron" },
  { GUINT_TO_POINTER (0x0145), "/Ncedilla" },
  { GUINT_TO_POINTER (0x24C3), "/Ncircle" },
  { GUINT_TO_POINTER (0x1E4A), "/Ncircumflexbelow" },
  { GUINT_TO_POINTER (0x0145), "/Ncommaaccent" },
  { GUINT_TO_POINTER (0x1E44), "/Ndotaccent" },
  { GUINT_TO_POINTER (0x1E46), "/Ndotbelow" },
  { GUINT_TO_POINTER (0x019D), "/Nhookleft" },
  { GUINT_TO_POINTER (0x2168), "/Nineroman" },
  { GUINT_TO_POINTER (0x01CB), "/Nj" },
  { GUINT_TO_POINTER (0x040A), "/Njecyrillic" },
  { GUINT_TO_POINTER (0x1E48), "/Nlinebelow" },
  { GUINT_TO_POINTER (0xFF2E), "/Nmonospace" },
  { GUINT_TO_POINTER (0x0546), "/Nowarmenian" },
  { GUINT_TO_POINTER (0xF76E), "/Nsmall" },
  { GUINT_TO_POINTER (0x00D1), "/Ntilde" },
  { GUINT_TO_POINTER (0xF7F1), "/Ntildesmall" },
  { GUINT_TO_POINTER (0x039D), "/Nu" },
  { GUINT_TO_POINTER (0x004F), "/O" },
  { GUINT_TO_POINTER (0x0152), "/OE" },
  { GUINT_TO_POINTER (0xF6FA), "/OEsmall" },
  { GUINT_TO_POINTER (0x00D3), "/Oacute" },
  { GUINT_TO_POINTER (0xF7F3), "/Oacutesmall" },
  { GUINT_TO_POINTER (0x04E8), "/Obarredcyrillic" },
  { GUINT_TO_POINTER (0x04EA), "/Obarreddieresiscyrillic" },
  { GUINT_TO_POINTER (0x014E), "/Obreve" },
  { GUINT_TO_POINTER (0x01D1), "/Ocaron" },
  { GUINT_TO_POINTER (0x019F), "/Ocenteredtilde" },
  { GUINT_TO_POINTER (0x24C4), "/Ocircle" },
  { GUINT_TO_POINTER (0x00D4), "/Ocircumflex" },
  { GUINT_TO_POINTER (0x1ED0), "/Ocircumflexacute" },
  { GUINT_TO_POINTER (0x1ED8), "/Ocircumflexdotbelow" },
  { GUINT_TO_POINTER (0x1ED2), "/Ocircumflexgrave" },
  { GUINT_TO_POINTER (0x1ED4), "/Ocircumflexhookabove" },
  { GUINT_TO_POINTER (0xF7F4), "/Ocircumflexsmall" },
  { GUINT_TO_POINTER (0x1ED6), "/Ocircumflextilde" },
  { GUINT_TO_POINTER (0x041E), "/Ocyrillic" },
  { GUINT_TO_POINTER (0x0150), "/Odblacute" },
  { GUINT_TO_POINTER (0x020C), "/Odblgrave" },
  { GUINT_TO_POINTER (0x00D6), "/Odieresis" },
  { GUINT_TO_POINTER (0x04E6), "/Odieresiscyrillic" },
  { GUINT_TO_POINTER (0xF7F6), "/Odieresissmall" },
  { GUINT_TO_POINTER (0x1ECC), "/Odotbelow" },
  { GUINT_TO_POINTER (0xF6FB), "/Ogoneksmall" },
  { GUINT_TO_POINTER (0x00D2), "/Ograve" },
  { GUINT_TO_POINTER (0xF7F2), "/Ogravesmall" },
  { GUINT_TO_POINTER (0x0555), "/Oharmenian" },
  { GUINT_TO_POINTER (0x2126), "/Ohm" },
  { GUINT_TO_POINTER (0x1ECE), "/Ohookabove" },
  { GUINT_TO_POINTER (0x01A0), "/Ohorn" },
  { GUINT_TO_POINTER (0x1EDA), "/Ohornacute" },
  { GUINT_TO_POINTER (0x1EE2), "/Ohorndotbelow" },
  { GUINT_TO_POINTER (0x1EDC), "/Ohorngrave" },
  { GUINT_TO_POINTER (0x1EDE), "/Ohornhookabove" },
  { GUINT_TO_POINTER (0x1EE0), "/Ohorntilde" },
  { GUINT_TO_POINTER (0x0150), "/Ohungarumlaut" },
  { GUINT_TO_POINTER (0x01A2), "/Oi" },
  { GUINT_TO_POINTER (0x020E), "/Oinvertedbreve" },
  { GUINT_TO_POINTER (0x014C), "/Omacron" },
  { GUINT_TO_POINTER (0x1E52), "/Omacronacute" },
  { GUINT_TO_POINTER (0x1E50), "/Omacrongrave" },
  { GUINT_TO_POINTER (0x2126), "/Omega" },
  { GUINT_TO_POINTER (0x0460), "/Omegacyrillic" },
  { GUINT_TO_POINTER (0x03A9), "/Omegagreek" },
  { GUINT_TO_POINTER (0x047A), "/Omegaroundcyrillic" },
  { GUINT_TO_POINTER (0x047C), "/Omegatitlocyrillic" },
  { GUINT_TO_POINTER (0x038F), "/Omegatonos" },
  { GUINT_TO_POINTER (0x039F), "/Omicron" },
  { GUINT_TO_POINTER (0x038C), "/Omicrontonos" },
  { GUINT_TO_POINTER (0xFF2F), "/Omonospace" },
  { GUINT_TO_POINTER (0x2160), "/Oneroman" },
  { GUINT_TO_POINTER (0x01EA), "/Oogonek" },
  { GUINT_TO_POINTER (0x01EC), "/Oogonekmacron" },
  { GUINT_TO_POINTER (0x0186), "/Oopen" },
  { GUINT_TO_POINTER (0x00D8), "/Oslash" },
  { GUINT_TO_POINTER (0x01FE), "/Oslashacute" },
  { GUINT_TO_POINTER (0xF7F8), "/Oslashsmall" },
  { GUINT_TO_POINTER (0xF76F), "/Osmall" },
  { GUINT_TO_POINTER (0x01FE), "/Ostrokeacute" },
  { GUINT_TO_POINTER (0x047E), "/Otcyrillic" },
  { GUINT_TO_POINTER (0x00D5), "/Otilde" },
  { GUINT_TO_POINTER (0x1E4C), "/Otildeacute" },
  { GUINT_TO_POINTER (0x1E4E), "/Otildedieresis" },
  { GUINT_TO_POINTER (0xF7F5), "/Otildesmall" },
  { GUINT_TO_POINTER (0x0050), "/P" },
  { GUINT_TO_POINTER (0x1E54), "/Pacute" },
  { GUINT_TO_POINTER (0x24C5), "/Pcircle" },
  { GUINT_TO_POINTER (0x1E56), "/Pdotaccent" },
  { GUINT_TO_POINTER (0x041F), "/Pecyrillic" },
  { GUINT_TO_POINTER (0x054A), "/Peharmenian" },
  { GUINT_TO_POINTER (0x04A6), "/Pemiddlehookcyrillic" },
  { GUINT_TO_POINTER (0x03A6), "/Phi" },
  { GUINT_TO_POINTER (0x01A4), "/Phook" },
  { GUINT_TO_POINTER (0x03A0), "/Pi" },
  { GUINT_TO_POINTER (0x0553), "/Piwrarmenian" },
  { GUINT_TO_POINTER (0xFF30), "/Pmonospace" },
  { GUINT_TO_POINTER (0x03A8), "/Psi" },
  { GUINT_TO_POINTER (0x0470), "/Psicyrillic" },
  { GUINT_TO_POINTER (0xF770), "/Psmall" },
  { GUINT_TO_POINTER (0x0051), "/Q" },
  { GUINT_TO_POINTER (0x24C6), "/Qcircle" },
  { GUINT_TO_POINTER (0xFF31), "/Qmonospace" },
  { GUINT_TO_POINTER (0xF771), "/Qsmall" },
  { GUINT_TO_POINTER (0x0052), "/R" },
  { GUINT_TO_POINTER (0x054C), "/Raarmenian" },
  { GUINT_TO_POINTER (0x0154), "/Racute" },
  { GUINT_TO_POINTER (0x0158), "/Rcaron" },
  { GUINT_TO_POINTER (0x0156), "/Rcedilla" },
  { GUINT_TO_POINTER (0x24C7), "/Rcircle" },
  { GUINT_TO_POINTER (0x0156), "/Rcommaaccent" },
  { GUINT_TO_POINTER (0x0210), "/Rdblgrave" },
  { GUINT_TO_POINTER (0x1E58), "/Rdotaccent" },
  { GUINT_TO_POINTER (0x1E5A), "/Rdotbelow" },
  { GUINT_TO_POINTER (0x1E5C), "/Rdotbelowmacron" },
  { GUINT_TO_POINTER (0x0550), "/Reharmenian" },
  { GUINT_TO_POINTER (0x211C), "/Rfraktur" },
  { GUINT_TO_POINTER (0x03A1), "/Rho" },
  { GUINT_TO_POINTER (0xF6FC), "/Ringsmall" },
  { GUINT_TO_POINTER (0x0212), "/Rinvertedbreve" },
  { GUINT_TO_POINTER (0x1E5E), "/Rlinebelow" },
  { GUINT_TO_POINTER (0xFF32), "/Rmonospace" },
  { GUINT_TO_POINTER (0xF772), "/Rsmall" },
  { GUINT_TO_POINTER (0x0281), "/Rsmallinverted" },
  { GUINT_TO_POINTER (0x02B6), "/Rsmallinvertedsuperior" },
  { GUINT_TO_POINTER (0x0053), "/S" },
  { GUINT_TO_POINTER (0x250C), "/SF010000" },
  { GUINT_TO_POINTER (0x2514), "/SF020000" },
  { GUINT_TO_POINTER (0x2510), "/SF030000" },
  { GUINT_TO_POINTER (0x2518), "/SF040000" },
  { GUINT_TO_POINTER (0x253C), "/SF050000" },
  { GUINT_TO_POINTER (0x252C), "/SF060000" },
  { GUINT_TO_POINTER (0x2534), "/SF070000" },
  { GUINT_TO_POINTER (0x251C), "/SF080000" },
  { GUINT_TO_POINTER (0x2524), "/SF090000" },
  { GUINT_TO_POINTER (0x2500), "/SF100000" },
  { GUINT_TO_POINTER (0x2502), "/SF110000" },
  { GUINT_TO_POINTER (0x2561), "/SF190000" },
  { GUINT_TO_POINTER (0x2562), "/SF200000" },
  { GUINT_TO_POINTER (0x2556), "/SF210000" },
  { GUINT_TO_POINTER (0x2555), "/SF220000" },
  { GUINT_TO_POINTER (0x2563), "/SF230000" },
  { GUINT_TO_POINTER (0x2551), "/SF240000" },
  { GUINT_TO_POINTER (0x2557), "/SF250000" },
  { GUINT_TO_POINTER (0x255D), "/SF260000" },
  { GUINT_TO_POINTER (0x255C), "/SF270000" },
  { GUINT_TO_POINTER (0x255B), "/SF280000" },
  { GUINT_TO_POINTER (0x255E), "/SF360000" },
  { GUINT_TO_POINTER (0x255F), "/SF370000" },
  { GUINT_TO_POINTER (0x255A), "/SF380000" },
  { GUINT_TO_POINTER (0x2554), "/SF390000" },
  { GUINT_TO_POINTER (0x2569), "/SF400000" },
  { GUINT_TO_POINTER (0x2566), "/SF410000" },
  { GUINT_TO_POINTER (0x2560), "/SF420000" },
  { GUINT_TO_POINTER (0x2550), "/SF430000" },
  { GUINT_TO_POINTER (0x256C), "/SF440000" },
  { GUINT_TO_POINTER (0x2567), "/SF450000" },
  { GUINT_TO_POINTER (0x2568), "/SF460000" },
  { GUINT_TO_POINTER (0x2564), "/SF470000" },
  { GUINT_TO_POINTER (0x2565), "/SF480000" },
  { GUINT_TO_POINTER (0x2559), "/SF490000" },
  { GUINT_TO_POINTER (0x2558), "/SF500000" },
  { GUINT_TO_POINTER (0x2552), "/SF510000" },
  { GUINT_TO_POINTER (0x2553), "/SF520000" },
  { GUINT_TO_POINTER (0x256B), "/SF530000" },
  { GUINT_TO_POINTER (0x256A), "/SF540000" },
  { GUINT_TO_POINTER (0x015A), "/Sacute" },
  { GUINT_TO_POINTER (0x1E64), "/Sacutedotaccent" },
  { GUINT_TO_POINTER (0x03E0), "/Sampigreek" },
  { GUINT_TO_POINTER (0x0160), "/Scaron" },
  { GUINT_TO_POINTER (0x1E66), "/Scarondotaccent" },
  { GUINT_TO_POINTER (0xF6FD), "/Scaronsmall" },
  { GUINT_TO_POINTER (0x015E), "/Scedilla" },
  { GUINT_TO_POINTER (0x018F), "/Schwa" },
  { GUINT_TO_POINTER (0x04D8), "/Schwacyrillic" },
  { GUINT_TO_POINTER (0x04DA), "/Schwadieresiscyrillic" },
  { GUINT_TO_POINTER (0x24C8), "/Scircle" },
  { GUINT_TO_POINTER (0x015C), "/Scircumflex" },
  { GUINT_TO_POINTER (0x0218), "/Scommaaccent" },
  { GUINT_TO_POINTER (0x1E60), "/Sdotaccent" },
  { GUINT_TO_POINTER (0x1E62), "/Sdotbelow" },
  { GUINT_TO_POINTER (0x1E68), "/Sdotbelowdotaccent" },
  { GUINT_TO_POINTER (0x054D), "/Seharmenian" },
  { GUINT_TO_POINTER (0x2166), "/Sevenroman" },
  { GUINT_TO_POINTER (0x0547), "/Shaarmenian" },
  { GUINT_TO_POINTER (0x0428), "/Shacyrillic" },
  { GUINT_TO_POINTER (0x0429), "/Shchacyrillic" },
  { GUINT_TO_POINTER (0x03E2), "/Sheicoptic" },
  { GUINT_TO_POINTER (0x04BA), "/Shhacyrillic" },
  { GUINT_TO_POINTER (0x03EC), "/Shimacoptic" },
  { GUINT_TO_POINTER (0x03A3), "/Sigma" },
  { GUINT_TO_POINTER (0x2165), "/Sixroman" },
  { GUINT_TO_POINTER (0xFF33), "/Smonospace" },
  { GUINT_TO_POINTER (0x042C), "/Softsigncyrillic" },
  { GUINT_TO_POINTER (0xF773), "/Ssmall" },
  { GUINT_TO_POINTER (0x03DA), "/Stigmagreek" },
  { GUINT_TO_POINTER (0x0054), "/T" },
  { GUINT_TO_POINTER (0x03A4), "/Tau" },
  { GUINT_TO_POINTER (0x0166), "/Tbar" },
  { GUINT_TO_POINTER (0x0164), "/Tcaron" },
  { GUINT_TO_POINTER (0x0162), "/Tcedilla" },
  { GUINT_TO_POINTER (0x24C9), "/Tcircle" },
  { GUINT_TO_POINTER (0x1E70), "/Tcircumflexbelow" },
  { GUINT_TO_POINTER (0x0162), "/Tcommaaccent" },
  { GUINT_TO_POINTER (0x1E6A), "/Tdotaccent" },
  { GUINT_TO_POINTER (0x1E6C), "/Tdotbelow" },
  { GUINT_TO_POINTER (0x0422), "/Tecyrillic" },
  { GUINT_TO_POINTER (0x04AC), "/Tedescendercyrillic" },
  { GUINT_TO_POINTER (0x2169), "/Tenroman" },
  { GUINT_TO_POINTER (0x04B4), "/Tetsecyrillic" },
  { GUINT_TO_POINTER (0x0398), "/Theta" },
  { GUINT_TO_POINTER (0x01AC), "/Thook" },
  { GUINT_TO_POINTER (0x00DE), "/Thorn" },
  { GUINT_TO_POINTER (0xF7FE), "/Thornsmall" },
  { GUINT_TO_POINTER (0x2162), "/Threeroman" },
  { GUINT_TO_POINTER (0xF6FE), "/Tildesmall" },
  { GUINT_TO_POINTER (0x054F), "/Tiwnarmenian" },
  { GUINT_TO_POINTER (0x1E6E), "/Tlinebelow" },
  { GUINT_TO_POINTER (0xFF34), "/Tmonospace" },
  { GUINT_TO_POINTER (0x0539), "/Toarmenian" },
  { GUINT_TO_POINTER (0x01BC), "/Tonefive" },
  { GUINT_TO_POINTER (0x0184), "/Tonesix" },
  { GUINT_TO_POINTER (0x01A7), "/Tonetwo" },
  { GUINT_TO_POINTER (0x01AE), "/Tretroflexhook" },
  { GUINT_TO_POINTER (0x0426), "/Tsecyrillic" },
  { GUINT_TO_POINTER (0x040B), "/Tshecyrillic" },
  { GUINT_TO_POINTER (0xF774), "/Tsmall" },
  { GUINT_TO_POINTER (0x216B), "/Twelveroman" },
  { GUINT_TO_POINTER (0x2161), "/Tworoman" },
  { GUINT_TO_POINTER (0x0055), "/U" },
  { GUINT_TO_POINTER (0x00DA), "/Uacute" },
  { GUINT_TO_POINTER (0xF7FA), "/Uacutesmall" },
  { GUINT_TO_POINTER (0x016C), "/Ubreve" },
  { GUINT_TO_POINTER (0x01D3), "/Ucaron" },
  { GUINT_TO_POINTER (0x24CA), "/Ucircle" },
  { GUINT_TO_POINTER (0x00DB), "/Ucircumflex" },
  { GUINT_TO_POINTER (0x1E76), "/Ucircumflexbelow" },
  { GUINT_TO_POINTER (0xF7FB), "/Ucircumflexsmall" },
  { GUINT_TO_POINTER (0x0423), "/Ucyrillic" },
  { GUINT_TO_POINTER (0x0170), "/Udblacute" },
  { GUINT_TO_POINTER (0x0214), "/Udblgrave" },
  { GUINT_TO_POINTER (0x00DC), "/Udieresis" },
  { GUINT_TO_POINTER (0x01D7), "/Udieresisacute" },
  { GUINT_TO_POINTER (0x1E72), "/Udieresisbelow" },
  { GUINT_TO_POINTER (0x01D9), "/Udieresiscaron" },
  { GUINT_TO_POINTER (0x04F0), "/Udieresiscyrillic" },
  { GUINT_TO_POINTER (0x01DB), "/Udieresisgrave" },
  { GUINT_TO_POINTER (0x01D5), "/Udieresismacron" },
  { GUINT_TO_POINTER (0xF7FC), "/Udieresissmall" },
  { GUINT_TO_POINTER (0x1EE4), "/Udotbelow" },
  { GUINT_TO_POINTER (0x00D9), "/Ugrave" },
  { GUINT_TO_POINTER (0xF7F9), "/Ugravesmall" },
  { GUINT_TO_POINTER (0x1EE6), "/Uhookabove" },
  { GUINT_TO_POINTER (0x01AF), "/Uhorn" },
  { GUINT_TO_POINTER (0x1EE8), "/Uhornacute" },
  { GUINT_TO_POINTER (0x1EF0), "/Uhorndotbelow" },
  { GUINT_TO_POINTER (0x1EEA), "/Uhorngrave" },
  { GUINT_TO_POINTER (0x1EEC), "/Uhornhookabove" },
  { GUINT_TO_POINTER (0x1EEE), "/Uhorntilde" },
  { GUINT_TO_POINTER (0x0170), "/Uhungarumlaut" },
  { GUINT_TO_POINTER (0x04F2), "/Uhungarumlautcyrillic" },
  { GUINT_TO_POINTER (0x0216), "/Uinvertedbreve" },
  { GUINT_TO_POINTER (0x0478), "/Ukcyrillic" },
  { GUINT_TO_POINTER (0x016A), "/Umacron" },
  { GUINT_TO_POINTER (0x04EE), "/Umacroncyrillic" },
  { GUINT_TO_POINTER (0x1E7A), "/Umacrondieresis" },
  { GUINT_TO_POINTER (0xFF35), "/Umonospace" },
  { GUINT_TO_POINTER (0x0172), "/Uogonek" },
  { GUINT_TO_POINTER (0x03A5), "/Upsilon" },
  { GUINT_TO_POINTER (0x03D2), "/Upsilon1" },
  { GUINT_TO_POINTER (0x03D3), "/Upsilonacutehooksymbolgreek" },
  { GUINT_TO_POINTER (0x01B1), "/Upsilonafrican" },
  { GUINT_TO_POINTER (0x03AB), "/Upsilondieresis" },
  { GUINT_TO_POINTER (0x03D4), "/Upsilondieresishooksymbolgreek" },
  { GUINT_TO_POINTER (0x03D2), "/Upsilonhooksymbol" },
  { GUINT_TO_POINTER (0x038E), "/Upsilontonos" },
  { GUINT_TO_POINTER (0x016E), "/Uring" },
  { GUINT_TO_POINTER (0x040E), "/Ushortcyrillic" },
  { GUINT_TO_POINTER (0xF775), "/Usmall" },
  { GUINT_TO_POINTER (0x04AE), "/Ustraightcyrillic" },
  { GUINT_TO_POINTER (0x04B0), "/Ustraightstrokecyrillic" },
  { GUINT_TO_POINTER (0x0168), "/Utilde" },
  { GUINT_TO_POINTER (0x1E78), "/Utildeacute" },
  { GUINT_TO_POINTER (0x1E74), "/Utildebelow" },
  { GUINT_TO_POINTER (0x0056), "/V" },
  { GUINT_TO_POINTER (0x24CB), "/Vcircle" },
  { GUINT_TO_POINTER (0x1E7E), "/Vdotbelow" },
  { GUINT_TO_POINTER (0x0412), "/Vecyrillic" },
  { GUINT_TO_POINTER (0x054E), "/Vewarmenian" },
  { GUINT_TO_POINTER (0x01B2), "/Vhook" },
  { GUINT_TO_POINTER (0xFF36), "/Vmonospace" },
  { GUINT_TO_POINTER (0x0548), "/Voarmenian" },
  { GUINT_TO_POINTER (0xF776), "/Vsmall" },
  { GUINT_TO_POINTER (0x1E7C), "/Vtilde" },
  { GUINT_TO_POINTER (0x0057), "/W" },
  { GUINT_TO_POINTER (0x1E82), "/Wacute" },
  { GUINT_TO_POINTER (0x24CC), "/Wcircle" },
  { GUINT_TO_POINTER (0x0174), "/Wcircumflex" },
  { GUINT_TO_POINTER (0x1E84), "/Wdieresis" },
  { GUINT_TO_POINTER (0x1E86), "/Wdotaccent" },
  { GUINT_TO_POINTER (0x1E88), "/Wdotbelow" },
  { GUINT_TO_POINTER (0x1E80), "/Wgrave" },
  { GUINT_TO_POINTER (0xFF37), "/Wmonospace" },
  { GUINT_TO_POINTER (0xF777), "/Wsmall" },
  { GUINT_TO_POINTER (0x0058), "/X" },
  { GUINT_TO_POINTER (0x24CD), "/Xcircle" },
  { GUINT_TO_POINTER (0x1E8C), "/Xdieresis" },
  { GUINT_TO_POINTER (0x1E8A), "/Xdotaccent" },
  { GUINT_TO_POINTER (0x053D), "/Xeharmenian" },
  { GUINT_TO_POINTER (0x039E), "/Xi" },
  { GUINT_TO_POINTER (0xFF38), "/Xmonospace" },
  { GUINT_TO_POINTER (0xF778), "/Xsmall" },
  { GUINT_TO_POINTER (0x0059), "/Y" },
  { GUINT_TO_POINTER (0x00DD), "/Yacute" },
  { GUINT_TO_POINTER (0xF7FD), "/Yacutesmall" },
  { GUINT_TO_POINTER (0x0462), "/Yatcyrillic" },
  { GUINT_TO_POINTER (0x24CE), "/Ycircle" },
  { GUINT_TO_POINTER (0x0176), "/Ycircumflex" },
  { GUINT_TO_POINTER (0x0178), "/Ydieresis" },
  { GUINT_TO_POINTER (0xF7FF), "/Ydieresissmall" },
  { GUINT_TO_POINTER (0x1E8E), "/Ydotaccent" },
  { GUINT_TO_POINTER (0x1EF4), "/Ydotbelow" },
  { GUINT_TO_POINTER (0x042B), "/Yericyrillic" },
  { GUINT_TO_POINTER (0x04F8), "/Yerudieresiscyrillic" },
  { GUINT_TO_POINTER (0x1EF2), "/Ygrave" },
  { GUINT_TO_POINTER (0x01B3), "/Yhook" },
  { GUINT_TO_POINTER (0x1EF6), "/Yhookabove" },
  { GUINT_TO_POINTER (0x0545), "/Yiarmenian" },
  { GUINT_TO_POINTER (0x0407), "/Yicyrillic" },
  { GUINT_TO_POINTER (0x0552), "/Yiwnarmenian" },
  { GUINT_TO_POINTER (0xFF39), "/Ymonospace" },
  { GUINT_TO_POINTER (0xF779), "/Ysmall" },
  { GUINT_TO_POINTER (0x1EF8), "/Ytilde" },
  { GUINT_TO_POINTER (0x046A), "/Yusbigcyrillic" },
  { GUINT_TO_POINTER (0x046C), "/Yusbigiotifiedcyrillic" },
  { GUINT_TO_POINTER (0x0466), "/Yuslittlecyrillic" },
  { GUINT_TO_POINTER (0x0468), "/Yuslittleiotifiedcyrillic" },
  { GUINT_TO_POINTER (0x005A), "/Z" },
  { GUINT_TO_POINTER (0x0536), "/Zaarmenian" },
  { GUINT_TO_POINTER (0x0179), "/Zacute" },
  { GUINT_TO_POINTER (0x017D), "/Zcaron" },
  { GUINT_TO_POINTER (0xF6FF), "/Zcaronsmall" },
  { GUINT_TO_POINTER (0x24CF), "/Zcircle" },
  { GUINT_TO_POINTER (0x1E90), "/Zcircumflex" },
  { GUINT_TO_POINTER (0x017B), "/Zdot" },
  { GUINT_TO_POINTER (0x017B), "/Zdotaccent" },
  { GUINT_TO_POINTER (0x1E92), "/Zdotbelow" },
  { GUINT_TO_POINTER (0x0417), "/Zecyrillic" },
  { GUINT_TO_POINTER (0x0498), "/Zedescendercyrillic" },
  { GUINT_TO_POINTER (0x04DE), "/Zedieresiscyrillic" },
  { GUINT_TO_POINTER (0x0396), "/Zeta" },
  { GUINT_TO_POINTER (0x053A), "/Zhearmenian" },
  { GUINT_TO_POINTER (0x04C1), "/Zhebrevecyrillic" },
  { GUINT_TO_POINTER (0x0416), "/Zhecyrillic" },
  { GUINT_TO_POINTER (0x0496), "/Zhedescendercyrillic" },
  { GUINT_TO_POINTER (0x04DC), "/Zhedieresiscyrillic" },
  { GUINT_TO_POINTER (0x1E94), "/Zlinebelow" },
  { GUINT_TO_POINTER (0xFF3A), "/Zmonospace" },
  { GUINT_TO_POINTER (0xF77A), "/Zsmall" },
  { GUINT_TO_POINTER (0x01B5), "/Zstroke" },
  { GUINT_TO_POINTER (0x0061), "/a" },
  { GUINT_TO_POINTER (0x0986), "/aabengali" },
  { GUINT_TO_POINTER (0x00E1), "/aacute" },
  { GUINT_TO_POINTER (0x0906), "/aadeva" },
  { GUINT_TO_POINTER (0x0A86), "/aagujarati" },
  { GUINT_TO_POINTER (0x0A06), "/aagurmukhi" },
  { GUINT_TO_POINTER (0x0A3E), "/aamatragurmukhi" },
  { GUINT_TO_POINTER (0x3303), "/aarusquare" },
  { GUINT_TO_POINTER (0x09BE), "/aavowelsignbengali" },
  { GUINT_TO_POINTER (0x093E), "/aavowelsigndeva" },
  { GUINT_TO_POINTER (0x0ABE), "/aavowelsigngujarati" },
  { GUINT_TO_POINTER (0x055F), "/abbreviationmarkarmenian" },
  { GUINT_TO_POINTER (0x0970), "/abbreviationsigndeva" },
  { GUINT_TO_POINTER (0x0985), "/abengali" },
  { GUINT_TO_POINTER (0x311A), "/abopomofo" },
  { GUINT_TO_POINTER (0x0103), "/abreve" },
  { GUINT_TO_POINTER (0x1EAF), "/abreveacute" },
  { GUINT_TO_POINTER (0x04D1), "/abrevecyrillic" },
  { GUINT_TO_POINTER (0x1EB7), "/abrevedotbelow" },
  { GUINT_TO_POINTER (0x1EB1), "/abrevegrave" },
  { GUINT_TO_POINTER (0x1EB3), "/abrevehookabove" },
  { GUINT_TO_POINTER (0x1EB5), "/abrevetilde" },
  { GUINT_TO_POINTER (0x01CE), "/acaron" },
  { GUINT_TO_POINTER (0x24D0), "/acircle" },
  { GUINT_TO_POINTER (0x00E2), "/acircumflex" },
  { GUINT_TO_POINTER (0x1EA5), "/acircumflexacute" },
  { GUINT_TO_POINTER (0x1EAD), "/acircumflexdotbelow" },
  { GUINT_TO_POINTER (0x1EA7), "/acircumflexgrave" },
  { GUINT_TO_POINTER (0x1EA9), "/acircumflexhookabove" },
  { GUINT_TO_POINTER (0x1EAB), "/acircumflextilde" },
  { GUINT_TO_POINTER (0x00B4), "/acute" },
  { GUINT_TO_POINTER (0x0317), "/acutebelowcmb" },
  { GUINT_TO_POINTER (0x0301), "/acutecmb" },
  { GUINT_TO_POINTER (0x0301), "/acutecomb" },
  { GUINT_TO_POINTER (0x0954), "/acutedeva" },
  { GUINT_TO_POINTER (0x02CF), "/acutelowmod" },
  { GUINT_TO_POINTER (0x0341), "/acutetonecmb" },
  { GUINT_TO_POINTER (0x0430), "/acyrillic" },
  { GUINT_TO_POINTER (0x0201), "/adblgrave" },
  { GUINT_TO_POINTER (0x0A71), "/addakgurmukhi" },
  { GUINT_TO_POINTER (0x0905), "/adeva" },
  { GUINT_TO_POINTER (0x00E4), "/adieresis" },
  { GUINT_TO_POINTER (0x04D3), "/adieresiscyrillic" },
  { GUINT_TO_POINTER (0x01DF), "/adieresismacron" },
  { GUINT_TO_POINTER (0x1EA1), "/adotbelow" },
  { GUINT_TO_POINTER (0x01E1), "/adotmacron" },
  { GUINT_TO_POINTER (0x00E6), "/ae" },
  { GUINT_TO_POINTER (0x01FD), "/aeacute" },
  { GUINT_TO_POINTER (0x3150), "/aekorean" },
  { GUINT_TO_POINTER (0x01E3), "/aemacron" },
  { GUINT_TO_POINTER (0x2015), "/afii00208" },
  { GUINT_TO_POINTER (0x20A4), "/afii08941" },
  { GUINT_TO_POINTER (0x0410), "/afii10017" },
  { GUINT_TO_POINTER (0x0411), "/afii10018" },
  { GUINT_TO_POINTER (0x0412), "/afii10019" },
  { GUINT_TO_POINTER (0x0413), "/afii10020" },
  { GUINT_TO_POINTER (0x0414), "/afii10021" },
  { GUINT_TO_POINTER (0x0415), "/afii10022" },
  { GUINT_TO_POINTER (0x0401), "/afii10023" },
  { GUINT_TO_POINTER (0x0416), "/afii10024" },
  { GUINT_TO_POINTER (0x0417), "/afii10025" },
  { GUINT_TO_POINTER (0x0418), "/afii10026" },
  { GUINT_TO_POINTER (0x0419), "/afii10027" },
  { GUINT_TO_POINTER (0x041A), "/afii10028" },
  { GUINT_TO_POINTER (0x041B), "/afii10029" },
  { GUINT_TO_POINTER (0x041C), "/afii10030" },
  { GUINT_TO_POINTER (0x041D), "/afii10031" },
  { GUINT_TO_POINTER (0x041E), "/afii10032" },
  { GUINT_TO_POINTER (0x041F), "/afii10033" },
  { GUINT_TO_POINTER (0x0420), "/afii10034" },
  { GUINT_TO_POINTER (0x0421), "/afii10035" },
  { GUINT_TO_POINTER (0x0422), "/afii10036" },
  { GUINT_TO_POINTER (0x0423), "/afii10037" },
  { GUINT_TO_POINTER (0x0424), "/afii10038" },
  { GUINT_TO_POINTER (0x0425), "/afii10039" },
  { GUINT_TO_POINTER (0x0426), "/afii10040" },
  { GUINT_TO_POINTER (0x0427), "/afii10041" },
  { GUINT_TO_POINTER (0x0428), "/afii10042" },
  { GUINT_TO_POINTER (0x0429), "/afii10043" },
  { GUINT_TO_POINTER (0x042A), "/afii10044" },
  { GUINT_TO_POINTER (0x042B), "/afii10045" },
  { GUINT_TO_POINTER (0x042C), "/afii10046" },
  { GUINT_TO_POINTER (0x042D), "/afii10047" },
  { GUINT_TO_POINTER (0x042E), "/afii10048" },
  { GUINT_TO_POINTER (0x042F), "/afii10049" },
  { GUINT_TO_POINTER (0x0490), "/afii10050" },
  { GUINT_TO_POINTER (0x0402), "/afii10051" },
  { GUINT_TO_POINTER (0x0403), "/afii10052" },
  { GUINT_TO_POINTER (0x0404), "/afii10053" },
  { GUINT_TO_POINTER (0x0405), "/afii10054" },
  { GUINT_TO_POINTER (0x0406), "/afii10055" },
  { GUINT_TO_POINTER (0x0407), "/afii10056" },
  { GUINT_TO_POINTER (0x0408), "/afii10057" },
  { GUINT_TO_POINTER (0x0409), "/afii10058" },
  { GUINT_TO_POINTER (0x040A), "/afii10059" },
  { GUINT_TO_POINTER (0x040B), "/afii10060" },
  { GUINT_TO_POINTER (0x040C), "/afii10061" },
  { GUINT_TO_POINTER (0x040E), "/afii10062" },
  { GUINT_TO_POINTER (0xF6C4), "/afii10063" },
  { GUINT_TO_POINTER (0xF6C5), "/afii10064" },
  { GUINT_TO_POINTER (0x0430), "/afii10065" },
  { GUINT_TO_POINTER (0x0431), "/afii10066" },
  { GUINT_TO_POINTER (0x0432), "/afii10067" },
  { GUINT_TO_POINTER (0x0433), "/afii10068" },
  { GUINT_TO_POINTER (0x0434), "/afii10069" },
  { GUINT_TO_POINTER (0x0435), "/afii10070" },
  { GUINT_TO_POINTER (0x0451), "/afii10071" },
  { GUINT_TO_POINTER (0x0436), "/afii10072" },
  { GUINT_TO_POINTER (0x0437), "/afii10073" },
  { GUINT_TO_POINTER (0x0438), "/afii10074" },
  { GUINT_TO_POINTER (0x0439), "/afii10075" },
  { GUINT_TO_POINTER (0x043A), "/afii10076" },
  { GUINT_TO_POINTER (0x043B), "/afii10077" },
  { GUINT_TO_POINTER (0x043C), "/afii10078" },
  { GUINT_TO_POINTER (0x043D), "/afii10079" },
  { GUINT_TO_POINTER (0x043E), "/afii10080" },
  { GUINT_TO_POINTER (0x043F), "/afii10081" },
  { GUINT_TO_POINTER (0x0440), "/afii10082" },
  { GUINT_TO_POINTER (0x0441), "/afii10083" },
  { GUINT_TO_POINTER (0x0442), "/afii10084" },
  { GUINT_TO_POINTER (0x0443), "/afii10085" },
  { GUINT_TO_POINTER (0x0444), "/afii10086" },
  { GUINT_TO_POINTER (0x0445), "/afii10087" },
  { GUINT_TO_POINTER (0x0446), "/afii10088" },
  { GUINT_TO_POINTER (0x0447), "/afii10089" },
  { GUINT_TO_POINTER (0x0448), "/afii10090" },
  { GUINT_TO_POINTER (0x0449), "/afii10091" },
  { GUINT_TO_POINTER (0x044A), "/afii10092" },
  { GUINT_TO_POINTER (0x044B), "/afii10093" },
  { GUINT_TO_POINTER (0x044C), "/afii10094" },
  { GUINT_TO_POINTER (0x044D), "/afii10095" },
  { GUINT_TO_POINTER (0x044E), "/afii10096" },
  { GUINT_TO_POINTER (0x044F), "/afii10097" },
  { GUINT_TO_POINTER (0x0491), "/afii10098" },
  { GUINT_TO_POINTER (0x0452), "/afii10099" },
  { GUINT_TO_POINTER (0x0453), "/afii10100" },
  { GUINT_TO_POINTER (0x0454), "/afii10101" },
  { GUINT_TO_POINTER (0x0455), "/afii10102" },
  { GUINT_TO_POINTER (0x0456), "/afii10103" },
  { GUINT_TO_POINTER (0x0457), "/afii10104" },
  { GUINT_TO_POINTER (0x0458), "/afii10105" },
  { GUINT_TO_POINTER (0x0459), "/afii10106" },
  { GUINT_TO_POINTER (0x045A), "/afii10107" },
  { GUINT_TO_POINTER (0x045B), "/afii10108" },
  { GUINT_TO_POINTER (0x045C), "/afii10109" },
  { GUINT_TO_POINTER (0x045E), "/afii10110" },
  { GUINT_TO_POINTER (0x040F), "/afii10145" },
  { GUINT_TO_POINTER (0x0462), "/afii10146" },
  { GUINT_TO_POINTER (0x0472), "/afii10147" },
  { GUINT_TO_POINTER (0x0474), "/afii10148" },
  { GUINT_TO_POINTER (0xF6C6), "/afii10192" },
  { GUINT_TO_POINTER (0x045F), "/afii10193" },
  { GUINT_TO_POINTER (0x0463), "/afii10194" },
  { GUINT_TO_POINTER (0x0473), "/afii10195" },
  { GUINT_TO_POINTER (0x0475), "/afii10196" },
  { GUINT_TO_POINTER (0xF6C7), "/afii10831" },
  { GUINT_TO_POINTER (0xF6C8), "/afii10832" },
  { GUINT_TO_POINTER (0x04D9), "/afii10846" },
  { GUINT_TO_POINTER (0x200E), "/afii299" },
  { GUINT_TO_POINTER (0x200F), "/afii300" },
  { GUINT_TO_POINTER (0x200D), "/afii301" },
  { GUINT_TO_POINTER (0x066A), "/afii57381" },
  { GUINT_TO_POINTER (0x060C), "/afii57388" },
  { GUINT_TO_POINTER (0x0660), "/afii57392" },
  { GUINT_TO_POINTER (0x0661), "/afii57393" },
  { GUINT_TO_POINTER (0x0662), "/afii57394" },
  { GUINT_TO_POINTER (0x0663), "/afii57395" },
  { GUINT_TO_POINTER (0x0664), "/afii57396" },
  { GUINT_TO_POINTER (0x0665), "/afii57397" },
  { GUINT_TO_POINTER (0x0666), "/afii57398" },
  { GUINT_TO_POINTER (0x0667), "/afii57399" },
  { GUINT_TO_POINTER (0x0668), "/afii57400" },
  { GUINT_TO_POINTER (0x0669), "/afii57401" },
  { GUINT_TO_POINTER (0x061B), "/afii57403" },
  { GUINT_TO_POINTER (0x061F), "/afii57407" },
  { GUINT_TO_POINTER (0x0621), "/afii57409" },
  { GUINT_TO_POINTER (0x0622), "/afii57410" },
  { GUINT_TO_POINTER (0x0623), "/afii57411" },
  { GUINT_TO_POINTER (0x0624), "/afii57412" },
  { GUINT_TO_POINTER (0x0625), "/afii57413" },
  { GUINT_TO_POINTER (0x0626), "/afii57414" },
  { GUINT_TO_POINTER (0x0627), "/afii57415" },
  { GUINT_TO_POINTER (0x0628), "/afii57416" },
  { GUINT_TO_POINTER (0x0629), "/afii57417" },
  { GUINT_TO_POINTER (0x062A), "/afii57418" },
  { GUINT_TO_POINTER (0x062B), "/afii57419" },
  { GUINT_TO_POINTER (0x062C), "/afii57420" },
  { GUINT_TO_POINTER (0x062D), "/afii57421" },
  { GUINT_TO_POINTER (0x062E), "/afii57422" },
  { GUINT_TO_POINTER (0x062F), "/afii57423" },
  { GUINT_TO_POINTER (0x0630), "/afii57424" },
  { GUINT_TO_POINTER (0x0631), "/afii57425" },
  { GUINT_TO_POINTER (0x0632), "/afii57426" },
  { GUINT_TO_POINTER (0x0633), "/afii57427" },
  { GUINT_TO_POINTER (0x0634), "/afii57428" },
  { GUINT_TO_POINTER (0x0635), "/afii57429" },
  { GUINT_TO_POINTER (0x0636), "/afii57430" },
  { GUINT_TO_POINTER (0x0637), "/afii57431" },
  { GUINT_TO_POINTER (0x0638), "/afii57432" },
  { GUINT_TO_POINTER (0x0639), "/afii57433" },
  { GUINT_TO_POINTER (0x063A), "/afii57434" },
  { GUINT_TO_POINTER (0x0640), "/afii57440" },
  { GUINT_TO_POINTER (0x0641), "/afii57441" },
  { GUINT_TO_POINTER (0x0642), "/afii57442" },
  { GUINT_TO_POINTER (0x0643), "/afii57443" },
  { GUINT_TO_POINTER (0x0644), "/afii57444" },
  { GUINT_TO_POINTER (0x0645), "/afii57445" },
  { GUINT_TO_POINTER (0x0646), "/afii57446" },
  { GUINT_TO_POINTER (0x0648), "/afii57448" },
  { GUINT_TO_POINTER (0x0649), "/afii57449" },
  { GUINT_TO_POINTER (0x064A), "/afii57450" },
  { GUINT_TO_POINTER (0x064B), "/afii57451" },
  { GUINT_TO_POINTER (0x064C), "/afii57452" },
  { GUINT_TO_POINTER (0x064D), "/afii57453" },
  { GUINT_TO_POINTER (0x064E), "/afii57454" },
  { GUINT_TO_POINTER (0x064F), "/afii57455" },
  { GUINT_TO_POINTER (0x0650), "/afii57456" },
  { GUINT_TO_POINTER (0x0651), "/afii57457" },
  { GUINT_TO_POINTER (0x0652), "/afii57458" },
  { GUINT_TO_POINTER (0x0647), "/afii57470" },
  { GUINT_TO_POINTER (0x06A4), "/afii57505" },
  { GUINT_TO_POINTER (0x067E), "/afii57506" },
  { GUINT_TO_POINTER (0x0686), "/afii57507" },
  { GUINT_TO_POINTER (0x0698), "/afii57508" },
  { GUINT_TO_POINTER (0x06AF), "/afii57509" },
  { GUINT_TO_POINTER (0x0679), "/afii57511" },
  { GUINT_TO_POINTER (0x0688), "/afii57512" },
  { GUINT_TO_POINTER (0x0691), "/afii57513" },
  { GUINT_TO_POINTER (0x06BA), "/afii57514" },
  { GUINT_TO_POINTER (0x06D2), "/afii57519" },
  { GUINT_TO_POINTER (0x06D5), "/afii57534" },
  { GUINT_TO_POINTER (0x20AA), "/afii57636" },
  { GUINT_TO_POINTER (0x05BE), "/afii57645" },
  { GUINT_TO_POINTER (0x05C3), "/afii57658" },
  { GUINT_TO_POINTER (0x05D0), "/afii57664" },
  { GUINT_TO_POINTER (0x05D1), "/afii57665" },
  { GUINT_TO_POINTER (0x05D2), "/afii57666" },
  { GUINT_TO_POINTER (0x05D3), "/afii57667" },
  { GUINT_TO_POINTER (0x05D4), "/afii57668" },
  { GUINT_TO_POINTER (0x05D5), "/afii57669" },
  { GUINT_TO_POINTER (0x05D6), "/afii57670" },
  { GUINT_TO_POINTER (0x05D7), "/afii57671" },
  { GUINT_TO_POINTER (0x05D8), "/afii57672" },
  { GUINT_TO_POINTER (0x05D9), "/afii57673" },
  { GUINT_TO_POINTER (0x05DA), "/afii57674" },
  { GUINT_TO_POINTER (0x05DB), "/afii57675" },
  { GUINT_TO_POINTER (0x05DC), "/afii57676" },
  { GUINT_TO_POINTER (0x05DD), "/afii57677" },
  { GUINT_TO_POINTER (0x05DE), "/afii57678" },
  { GUINT_TO_POINTER (0x05DF), "/afii57679" },
  { GUINT_TO_POINTER (0x05E0), "/afii57680" },
  { GUINT_TO_POINTER (0x05E1), "/afii57681" },
  { GUINT_TO_POINTER (0x05E2), "/afii57682" },
  { GUINT_TO_POINTER (0x05E3), "/afii57683" },
  { GUINT_TO_POINTER (0x05E4), "/afii57684" },
  { GUINT_TO_POINTER (0x05E5), "/afii57685" },
  { GUINT_TO_POINTER (0x05E6), "/afii57686" },
  { GUINT_TO_POINTER (0x05E7), "/afii57687" },
  { GUINT_TO_POINTER (0x05E8), "/afii57688" },
  { GUINT_TO_POINTER (0x05E9), "/afii57689" },
  { GUINT_TO_POINTER (0x05EA), "/afii57690" },
  { GUINT_TO_POINTER (0xFB2A), "/afii57694" },
  { GUINT_TO_POINTER (0xFB2B), "/afii57695" },
  { GUINT_TO_POINTER (0xFB4B), "/afii57700" },
  { GUINT_TO_POINTER (0xFB1F), "/afii57705" },
  { GUINT_TO_POINTER (0x05F0), "/afii57716" },
  { GUINT_TO_POINTER (0x05F1), "/afii57717" },
  { GUINT_TO_POINTER (0x05F2), "/afii57718" },
  { GUINT_TO_POINTER (0xFB35), "/afii57723" },
  { GUINT_TO_POINTER (0x05B4), "/afii57793" },
  { GUINT_TO_POINTER (0x05B5), "/afii57794" },
  { GUINT_TO_POINTER (0x05B6), "/afii57795" },
  { GUINT_TO_POINTER (0x05BB), "/afii57796" },
  { GUINT_TO_POINTER (0x05B8), "/afii57797" },
  { GUINT_TO_POINTER (0x05B7), "/afii57798" },
  { GUINT_TO_POINTER (0x05B0), "/afii57799" },
  { GUINT_TO_POINTER (0x05B2), "/afii57800" },
  { GUINT_TO_POINTER (0x05B1), "/afii57801" },
  { GUINT_TO_POINTER (0x05B3), "/afii57802" },
  { GUINT_TO_POINTER (0x05C2), "/afii57803" },
  { GUINT_TO_POINTER (0x05C1), "/afii57804" },
  { GUINT_TO_POINTER (0x05B9), "/afii57806" },
  { GUINT_TO_POINTER (0x05BC), "/afii57807" },
  { GUINT_TO_POINTER (0x05BD), "/afii57839" },
  { GUINT_TO_POINTER (0x05BF), "/afii57841" },
  { GUINT_TO_POINTER (0x05C0), "/afii57842" },
  { GUINT_TO_POINTER (0x02BC), "/afii57929" },
  { GUINT_TO_POINTER (0x2105), "/afii61248" },
  { GUINT_TO_POINTER (0x2113), "/afii61289" },
  { GUINT_TO_POINTER (0x2116), "/afii61352" },
  { GUINT_TO_POINTER (0x202C), "/afii61573" },
  { GUINT_TO_POINTER (0x202D), "/afii61574" },
  { GUINT_TO_POINTER (0x202E), "/afii61575" },
  { GUINT_TO_POINTER (0x200C), "/afii61664" },
  { GUINT_TO_POINTER (0x066D), "/afii63167" },
  { GUINT_TO_POINTER (0x02BD), "/afii64937" },
  { GUINT_TO_POINTER (0x00E0), "/agrave" },
  { GUINT_TO_POINTER (0x0A85), "/agujarati" },
  { GUINT_TO_POINTER (0x0A05), "/agurmukhi" },
  { GUINT_TO_POINTER (0x3042), "/ahiragana" },
  { GUINT_TO_POINTER (0x1EA3), "/ahookabove" },
  { GUINT_TO_POINTER (0x0990), "/aibengali" },
  { GUINT_TO_POINTER (0x311E), "/aibopomofo" },
  { GUINT_TO_POINTER (0x0910), "/aideva" },
  { GUINT_TO_POINTER (0x04D5), "/aiecyrillic" },
  { GUINT_TO_POINTER (0x0A90), "/aigujarati" },
  { GUINT_TO_POINTER (0x0A10), "/aigurmukhi" },
  { GUINT_TO_POINTER (0x0A48), "/aimatragurmukhi" },
  { GUINT_TO_POINTER (0x0639), "/ainarabic" },
  { GUINT_TO_POINTER (0xFECA), "/ainfinalarabic" },
  { GUINT_TO_POINTER (0xFECB), "/aininitialarabic" },
  { GUINT_TO_POINTER (0xFECC), "/ainmedialarabic" },
  { GUINT_TO_POINTER (0x0203), "/ainvertedbreve" },
  { GUINT_TO_POINTER (0x09C8), "/aivowelsignbengali" },
  { GUINT_TO_POINTER (0x0948), "/aivowelsigndeva" },
  { GUINT_TO_POINTER (0x0AC8), "/aivowelsigngujarati" },
  { GUINT_TO_POINTER (0x30A2), "/akatakana" },
  { GUINT_TO_POINTER (0xFF71), "/akatakanahalfwidth" },
  { GUINT_TO_POINTER (0x314F), "/akorean" },
  { GUINT_TO_POINTER (0x05D0), "/alef" },
  { GUINT_TO_POINTER (0x0627), "/alefarabic" },
  { GUINT_TO_POINTER (0xFB30), "/alefdageshhebrew" },
  { GUINT_TO_POINTER (0xFE8E), "/aleffinalarabic" },
  { GUINT_TO_POINTER (0x0623), "/alefhamzaabovearabic" },
  { GUINT_TO_POINTER (0xFE84), "/alefhamzaabovefinalarabic" },
  { GUINT_TO_POINTER (0x0625), "/alefhamzabelowarabic" },
  { GUINT_TO_POINTER (0xFE88), "/alefhamzabelowfinalarabic" },
  { GUINT_TO_POINTER (0x05D0), "/alefhebrew" },
  { GUINT_TO_POINTER (0xFB4F), "/aleflamedhebrew" },
  { GUINT_TO_POINTER (0x0622), "/alefmaddaabovearabic" },
  { GUINT_TO_POINTER (0xFE82), "/alefmaddaabovefinalarabic" },
  { GUINT_TO_POINTER (0x0649), "/alefmaksuraarabic" },
  { GUINT_TO_POINTER (0xFEF0), "/alefmaksurafinalarabic" },
  { GUINT_TO_POINTER (0xFEF3), "/alefmaksurainitialarabic" },
  { GUINT_TO_POINTER (0xFEF4), "/alefmaksuramedialarabic" },
  { GUINT_TO_POINTER (0xFB2E), "/alefpatahhebrew" },
  { GUINT_TO_POINTER (0xFB2F), "/alefqamatshebrew" },
  { GUINT_TO_POINTER (0x2135), "/aleph" },
  { GUINT_TO_POINTER (0x224C), "/allequal" },
  { GUINT_TO_POINTER (0x03B1), "/alpha" },
  { GUINT_TO_POINTER (0x03AC), "/alphatonos" },
  { GUINT_TO_POINTER (0x0101), "/amacron" },
  { GUINT_TO_POINTER (0xFF41), "/amonospace" },
  { GUINT_TO_POINTER (0x0026), "/ampersand" },
  { GUINT_TO_POINTER (0xFF06), "/ampersandmonospace" },
  { GUINT_TO_POINTER (0xF726), "/ampersandsmall" },
  { GUINT_TO_POINTER (0x33C2), "/amsquare" },
  { GUINT_TO_POINTER (0x3122), "/anbopomofo" },
  { GUINT_TO_POINTER (0x3124), "/angbopomofo" },
  { GUINT_TO_POINTER (0x0E5A), "/angkhankhuthai" },
  { GUINT_TO_POINTER (0x2220), "/angle" },
  { GUINT_TO_POINTER (0x3008), "/anglebracketleft" },
  { GUINT_TO_POINTER (0xFE3F), "/anglebracketleftvertical" },
  { GUINT_TO_POINTER (0x3009), "/anglebracketright" },
  { GUINT_TO_POINTER (0xFE40), "/anglebracketrightvertical" },
  { GUINT_TO_POINTER (0x2329), "/angleleft" },
  { GUINT_TO_POINTER (0x232A), "/angleright" },
  { GUINT_TO_POINTER (0x212B), "/angstrom" },
  { GUINT_TO_POINTER (0x0387), "/anoteleia" },
  { GUINT_TO_POINTER (0x0952), "/anudattadeva" },
  { GUINT_TO_POINTER (0x0982), "/anusvarabengali" },
  { GUINT_TO_POINTER (0x0902), "/anusvaradeva" },
  { GUINT_TO_POINTER (0x0A82), "/anusvaragujarati" },
  { GUINT_TO_POINTER (0x0105), "/aogonek" },
  { GUINT_TO_POINTER (0x3300), "/apaatosquare" },
  { GUINT_TO_POINTER (0x249C), "/aparen" },
  { GUINT_TO_POINTER (0x055A), "/apostrophearmenian" },
  { GUINT_TO_POINTER (0x02BC), "/apostrophemod" },
  { GUINT_TO_POINTER (0xF8FF), "/apple" },
  { GUINT_TO_POINTER (0x2250), "/approaches" },
  { GUINT_TO_POINTER (0x2248), "/approxequal" },
  { GUINT_TO_POINTER (0x2252), "/approxequalorimage" },
  { GUINT_TO_POINTER (0x2245), "/approximatelyequal" },
  { GUINT_TO_POINTER (0x318E), "/araeaekorean" },
  { GUINT_TO_POINTER (0x318D), "/araeakorean" },
  { GUINT_TO_POINTER (0x2312), "/arc" },
  { GUINT_TO_POINTER (0x1E9A), "/arighthalfring" },
  { GUINT_TO_POINTER (0x00E5), "/aring" },
  { GUINT_TO_POINTER (0x01FB), "/aringacute" },
  { GUINT_TO_POINTER (0x1E01), "/aringbelow" },
  { GUINT_TO_POINTER (0x2194), "/arrowboth" },
  { GUINT_TO_POINTER (0x21E3), "/arrowdashdown" },
  { GUINT_TO_POINTER (0x21E0), "/arrowdashleft" },
  { GUINT_TO_POINTER (0x21E2), "/arrowdashright" },
  { GUINT_TO_POINTER (0x21E1), "/arrowdashup" },
  { GUINT_TO_POINTER (0x21D4), "/arrowdblboth" },
  { GUINT_TO_POINTER (0x21D3), "/arrowdbldown" },
  { GUINT_TO_POINTER (0x21D0), "/arrowdblleft" },
  { GUINT_TO_POINTER (0x21D2), "/arrowdblright" },
  { GUINT_TO_POINTER (0x21D1), "/arrowdblup" },
  { GUINT_TO_POINTER (0x2193), "/arrowdown" },
  { GUINT_TO_POINTER (0x2199), "/arrowdownleft" },
  { GUINT_TO_POINTER (0x2198), "/arrowdownright" },
  { GUINT_TO_POINTER (0x21E9), "/arrowdownwhite" },
  { GUINT_TO_POINTER (0x02C5), "/arrowheaddownmod" },
  { GUINT_TO_POINTER (0x02C2), "/arrowheadleftmod" },
  { GUINT_TO_POINTER (0x02C3), "/arrowheadrightmod" },
  { GUINT_TO_POINTER (0x02C4), "/arrowheadupmod" },
  { GUINT_TO_POINTER (0xF8E7), "/arrowhorizex" },
  { GUINT_TO_POINTER (0x2190), "/arrowleft" },
  { GUINT_TO_POINTER (0x21D0), "/arrowleftdbl" },
  { GUINT_TO_POINTER (0x21CD), "/arrowleftdblstroke" },
  { GUINT_TO_POINTER (0x21C6), "/arrowleftoverright" },
  { GUINT_TO_POINTER (0x21E6), "/arrowleftwhite" },
  { GUINT_TO_POINTER (0x2192), "/arrowright" },
  { GUINT_TO_POINTER (0x21CF), "/arrowrightdblstroke" },
  { GUINT_TO_POINTER (0x279E), "/arrowrightheavy" },
  { GUINT_TO_POINTER (0x21C4), "/arrowrightoverleft" },
  { GUINT_TO_POINTER (0x21E8), "/arrowrightwhite" },
  { GUINT_TO_POINTER (0x21E4), "/arrowtableft" },
  { GUINT_TO_POINTER (0x21E5), "/arrowtabright" },
  { GUINT_TO_POINTER (0x2191), "/arrowup" },
  { GUINT_TO_POINTER (0x2195), "/arrowupdn" },
  { GUINT_TO_POINTER (0x21A8), "/arrowupdnbse" },
  { GUINT_TO_POINTER (0x21A8), "/arrowupdownbase" },
  { GUINT_TO_POINTER (0x2196), "/arrowupleft" },
  { GUINT_TO_POINTER (0x21C5), "/arrowupleftofdown" },
  { GUINT_TO_POINTER (0x2197), "/arrowupright" },
  { GUINT_TO_POINTER (0x21E7), "/arrowupwhite" },
  { GUINT_TO_POINTER (0xF8E6), "/arrowvertex" },
  { GUINT_TO_POINTER (0x005E), "/asciicircum" },
  { GUINT_TO_POINTER (0xFF3E), "/asciicircummonospace" },
  { GUINT_TO_POINTER (0x007E), "/asciitilde" },
  { GUINT_TO_POINTER (0xFF5E), "/asciitildemonospace" },
  { GUINT_TO_POINTER (0x0251), "/ascript" },
  { GUINT_TO_POINTER (0x0252), "/ascriptturned" },
  { GUINT_TO_POINTER (0x3041), "/asmallhiragana" },
  { GUINT_TO_POINTER (0x30A1), "/asmallkatakana" },
  { GUINT_TO_POINTER (0xFF67), "/asmallkatakanahalfwidth" },
  { GUINT_TO_POINTER (0x002A), "/asterisk" },
  { GUINT_TO_POINTER (0x066D), "/asteriskaltonearabic" },
  { GUINT_TO_POINTER (0x066D), "/asteriskarabic" },
  { GUINT_TO_POINTER (0x2217), "/asteriskmath" },
  { GUINT_TO_POINTER (0xFF0A), "/asteriskmonospace" },
  { GUINT_TO_POINTER (0xFE61), "/asterisksmall" },
  { GUINT_TO_POINTER (0x2042), "/asterism" },
  { GUINT_TO_POINTER (0xF6E9), "/asuperior" },
  { GUINT_TO_POINTER (0x2243), "/asymptoticallyequal" },
  { GUINT_TO_POINTER (0x0040), "/at" },
  { GUINT_TO_POINTER (0x00E3), "/atilde" },
  { GUINT_TO_POINTER (0xFF20), "/atmonospace" },
  { GUINT_TO_POINTER (0xFE6B), "/atsmall" },
  { GUINT_TO_POINTER (0x0250), "/aturned" },
  { GUINT_TO_POINTER (0x0994), "/aubengali" },
  { GUINT_TO_POINTER (0x3120), "/aubopomofo" },
  { GUINT_TO_POINTER (0x0914), "/audeva" },
  { GUINT_TO_POINTER (0x0A94), "/augujarati" },
  { GUINT_TO_POINTER (0x0A14), "/augurmukhi" },
  { GUINT_TO_POINTER (0x09D7), "/aulengthmarkbengali" },
  { GUINT_TO_POINTER (0x0A4C), "/aumatragurmukhi" },
  { GUINT_TO_POINTER (0x09CC), "/auvowelsignbengali" },
  { GUINT_TO_POINTER (0x094C), "/auvowelsigndeva" },
  { GUINT_TO_POINTER (0x0ACC), "/auvowelsigngujarati" },
  { GUINT_TO_POINTER (0x093D), "/avagrahadeva" },
  { GUINT_TO_POINTER (0x0561), "/aybarmenian" },
  { GUINT_TO_POINTER (0x05E2), "/ayin" },
  { GUINT_TO_POINTER (0xFB20), "/ayinaltonehebrew" },
  { GUINT_TO_POINTER (0x05E2), "/ayinhebrew" },
  { GUINT_TO_POINTER (0x0062), "/b" },
  { GUINT_TO_POINTER (0x09AC), "/babengali" },
  { GUINT_TO_POINTER (0x005C), "/backslash" },
  { GUINT_TO_POINTER (0xFF3C), "/backslashmonospace" },
  { GUINT_TO_POINTER (0x092C), "/badeva" },
  { GUINT_TO_POINTER (0x0AAC), "/bagujarati" },
  { GUINT_TO_POINTER (0x0A2C), "/bagurmukhi" },
  { GUINT_TO_POINTER (0x3070), "/bahiragana" },
  { GUINT_TO_POINTER (0x0E3F), "/bahtthai" },
  { GUINT_TO_POINTER (0x30D0), "/bakatakana" },
  { GUINT_TO_POINTER (0x007C), "/bar" },
  { GUINT_TO_POINTER (0xFF5C), "/barmonospace" },
  { GUINT_TO_POINTER (0x3105), "/bbopomofo" },
  { GUINT_TO_POINTER (0x24D1), "/bcircle" },
  { GUINT_TO_POINTER (0x1E03), "/bdotaccent" },
  { GUINT_TO_POINTER (0x1E05), "/bdotbelow" },
  { GUINT_TO_POINTER (0x266C), "/beamedsixteenthnotes" },
  { GUINT_TO_POINTER (0x2235), "/because" },
  { GUINT_TO_POINTER (0x0431), "/becyrillic" },
  { GUINT_TO_POINTER (0x0628), "/beharabic" },
  { GUINT_TO_POINTER (0xFE90), "/behfinalarabic" },
  { GUINT_TO_POINTER (0xFE91), "/behinitialarabic" },
  { GUINT_TO_POINTER (0x3079), "/behiragana" },
  { GUINT_TO_POINTER (0xFE92), "/behmedialarabic" },
  { GUINT_TO_POINTER (0xFC9F), "/behmeeminitialarabic" },
  { GUINT_TO_POINTER (0xFC08), "/behmeemisolatedarabic" },
  { GUINT_TO_POINTER (0xFC6D), "/behnoonfinalarabic" },
  { GUINT_TO_POINTER (0x30D9), "/bekatakana" },
  { GUINT_TO_POINTER (0x0562), "/benarmenian" },
  { GUINT_TO_POINTER (0x05D1), "/bet" },
  { GUINT_TO_POINTER (0x03B2), "/beta" },
  { GUINT_TO_POINTER (0x03D0), "/betasymbolgreek" },
  { GUINT_TO_POINTER (0xFB31), "/betdagesh" },
  { GUINT_TO_POINTER (0xFB31), "/betdageshhebrew" },
  { GUINT_TO_POINTER (0x05D1), "/bethebrew" },
  { GUINT_TO_POINTER (0xFB4C), "/betrafehebrew" },
  { GUINT_TO_POINTER (0x09AD), "/bhabengali" },
  { GUINT_TO_POINTER (0x092D), "/bhadeva" },
  { GUINT_TO_POINTER (0x0AAD), "/bhagujarati" },
  { GUINT_TO_POINTER (0x0A2D), "/bhagurmukhi" },
  { GUINT_TO_POINTER (0x0253), "/bhook" },
  { GUINT_TO_POINTER (0x3073), "/bihiragana" },
  { GUINT_TO_POINTER (0x30D3), "/bikatakana" },
  { GUINT_TO_POINTER (0x0298), "/bilabialclick" },
  { GUINT_TO_POINTER (0x0A02), "/bindigurmukhi" },
  { GUINT_TO_POINTER (0x3331), "/birusquare" },
  { GUINT_TO_POINTER (0x25CF), "/blackcircle" },
  { GUINT_TO_POINTER (0x25C6), "/blackdiamond" },
  { GUINT_TO_POINTER (0x25BC), "/blackdownpointingtriangle" },
  { GUINT_TO_POINTER (0x25C4), "/blackleftpointingpointer" },
  { GUINT_TO_POINTER (0x25C0), "/blackleftpointingtriangle" },
  { GUINT_TO_POINTER (0x3010), "/blacklenticularbracketleft" },
  { GUINT_TO_POINTER (0xFE3B), "/blacklenticularbracketleftvertical" },
  { GUINT_TO_POINTER (0x3011), "/blacklenticularbracketright" },
  { GUINT_TO_POINTER (0xFE3C), "/blacklenticularbracketrightvertical" },
  { GUINT_TO_POINTER (0x25E3), "/blacklowerlefttriangle" },
  { GUINT_TO_POINTER (0x25E2), "/blacklowerrighttriangle" },
  { GUINT_TO_POINTER (0x25AC), "/blackrectangle" },
  { GUINT_TO_POINTER (0x25BA), "/blackrightpointingpointer" },
  { GUINT_TO_POINTER (0x25B6), "/blackrightpointingtriangle" },
  { GUINT_TO_POINTER (0x25AA), "/blacksmallsquare" },
  { GUINT_TO_POINTER (0x263B), "/blacksmilingface" },
  { GUINT_TO_POINTER (0x25A0), "/blacksquare" },
  { GUINT_TO_POINTER (0x2605), "/blackstar" },
  { GUINT_TO_POINTER (0x25E4), "/blackupperlefttriangle" },
  { GUINT_TO_POINTER (0x25E5), "/blackupperrighttriangle" },
  { GUINT_TO_POINTER (0x25B4), "/blackuppointingsmalltriangle" },
  { GUINT_TO_POINTER (0x25B2), "/blackuppointingtriangle" },
  { GUINT_TO_POINTER (0x2423), "/blank" },
  { GUINT_TO_POINTER (0x1E07), "/blinebelow" },
  { GUINT_TO_POINTER (0x2588), "/block" },
  { GUINT_TO_POINTER (0xFF42), "/bmonospace" },
  { GUINT_TO_POINTER (0x0E1A), "/bobaimaithai" },
  { GUINT_TO_POINTER (0x307C), "/bohiragana" },
  { GUINT_TO_POINTER (0x30DC), "/bokatakana" },
  { GUINT_TO_POINTER (0x249D), "/bparen" },
  { GUINT_TO_POINTER (0x33C3), "/bqsquare" },
  { GUINT_TO_POINTER (0xF8F4), "/braceex" },
  { GUINT_TO_POINTER (0x007B), "/braceleft" },
  { GUINT_TO_POINTER (0xF8F3), "/braceleftbt" },
  { GUINT_TO_POINTER (0xF8F2), "/braceleftmid" },
  { GUINT_TO_POINTER (0xFF5B), "/braceleftmonospace" },
  { GUINT_TO_POINTER (0xFE5B), "/braceleftsmall" },
  { GUINT_TO_POINTER (0xF8F1), "/bracelefttp" },
  { GUINT_TO_POINTER (0xFE37), "/braceleftvertical" },
  { GUINT_TO_POINTER (0x007D), "/braceright" },
  { GUINT_TO_POINTER (0xF8FE), "/bracerightbt" },
  { GUINT_TO_POINTER (0xF8FD), "/bracerightmid" },
  { GUINT_TO_POINTER (0xFF5D), "/bracerightmonospace" },
  { GUINT_TO_POINTER (0xFE5C), "/bracerightsmall" },
  { GUINT_TO_POINTER (0xF8FC), "/bracerighttp" },
  { GUINT_TO_POINTER (0xFE38), "/bracerightvertical" },
  { GUINT_TO_POINTER (0x005B), "/bracketleft" },
  { GUINT_TO_POINTER (0xF8F0), "/bracketleftbt" },
  { GUINT_TO_POINTER (0xF8EF), "/bracketleftex" },
  { GUINT_TO_POINTER (0xFF3B), "/bracketleftmonospace" },
  { GUINT_TO_POINTER (0xF8EE), "/bracketlefttp" },
  { GUINT_TO_POINTER (0x005D), "/bracketright" },
  { GUINT_TO_POINTER (0xF8FB), "/bracketrightbt" },
  { GUINT_TO_POINTER (0xF8FA), "/bracketrightex" },
  { GUINT_TO_POINTER (0xFF3D), "/bracketrightmonospace" },
  { GUINT_TO_POINTER (0xF8F9), "/bracketrighttp" },
  { GUINT_TO_POINTER (0x02D8), "/breve" },
  { GUINT_TO_POINTER (0x032E), "/brevebelowcmb" },
  { GUINT_TO_POINTER (0x0306), "/brevecmb" },
  { GUINT_TO_POINTER (0x032F), "/breveinvertedbelowcmb" },
  { GUINT_TO_POINTER (0x0311), "/breveinvertedcmb" },
  { GUINT_TO_POINTER (0x0361), "/breveinverteddoublecmb" },
  { GUINT_TO_POINTER (0x032A), "/bridgebelowcmb" },
  { GUINT_TO_POINTER (0x033A), "/bridgeinvertedbelowcmb" },
  { GUINT_TO_POINTER (0x00A6), "/brokenbar" },
  { GUINT_TO_POINTER (0x0180), "/bstroke" },
  { GUINT_TO_POINTER (0xF6EA), "/bsuperior" },
  { GUINT_TO_POINTER (0x0183), "/btopbar" },
  { GUINT_TO_POINTER (0x3076), "/buhiragana" },
  { GUINT_TO_POINTER (0x30D6), "/bukatakana" },
  { GUINT_TO_POINTER (0x2022), "/bullet" },
  { GUINT_TO_POINTER (0x25D8), "/bulletinverse" },
  { GUINT_TO_POINTER (0x2219), "/bulletoperator" },
  { GUINT_TO_POINTER (0x25CE), "/bullseye" },
  { GUINT_TO_POINTER (0x0063), "/c" },
  { GUINT_TO_POINTER (0x056E), "/caarmenian" },
  { GUINT_TO_POINTER (0x099A), "/cabengali" },
  { GUINT_TO_POINTER (0x0107), "/cacute" },
  { GUINT_TO_POINTER (0x091A), "/cadeva" },
  { GUINT_TO_POINTER (0x0A9A), "/cagujarati" },
  { GUINT_TO_POINTER (0x0A1A), "/cagurmukhi" },
  { GUINT_TO_POINTER (0x3388), "/calsquare" },
  { GUINT_TO_POINTER (0x0981), "/candrabindubengali" },
  { GUINT_TO_POINTER (0x0310), "/candrabinducmb" },
  { GUINT_TO_POINTER (0x0901), "/candrabindudeva" },
  { GUINT_TO_POINTER (0x0A81), "/candrabindugujarati" },
  { GUINT_TO_POINTER (0x21EA), "/capslock" },
  { GUINT_TO_POINTER (0x2105), "/careof" },
  { GUINT_TO_POINTER (0x02C7), "/caron" },
  { GUINT_TO_POINTER (0x032C), "/caronbelowcmb" },
  { GUINT_TO_POINTER (0x030C), "/caroncmb" },
  { GUINT_TO_POINTER (0x21B5), "/carriagereturn" },
  { GUINT_TO_POINTER (0x3118), "/cbopomofo" },
  { GUINT_TO_POINTER (0x010D), "/ccaron" },
  { GUINT_TO_POINTER (0x00E7), "/ccedilla" },
  { GUINT_TO_POINTER (0x1E09), "/ccedillaacute" },
  { GUINT_TO_POINTER (0x24D2), "/ccircle" },
  { GUINT_TO_POINTER (0x0109), "/ccircumflex" },
  { GUINT_TO_POINTER (0x0255), "/ccurl" },
  { GUINT_TO_POINTER (0x010B), "/cdot" },
  { GUINT_TO_POINTER (0x010B), "/cdotaccent" },
  { GUINT_TO_POINTER (0x33C5), "/cdsquare" },
  { GUINT_TO_POINTER (0x00B8), "/cedilla" },
  { GUINT_TO_POINTER (0x0327), "/cedillacmb" },
  { GUINT_TO_POINTER (0x00A2), "/cent" },
  { GUINT_TO_POINTER (0x2103), "/centigrade" },
  { GUINT_TO_POINTER (0xF6DF), "/centinferior" },
  { GUINT_TO_POINTER (0xFFE0), "/centmonospace" },
  { GUINT_TO_POINTER (0xF7A2), "/centoldstyle" },
  { GUINT_TO_POINTER (0xF6E0), "/centsuperior" },
  { GUINT_TO_POINTER (0x0579), "/chaarmenian" },
  { GUINT_TO_POINTER (0x099B), "/chabengali" },
  { GUINT_TO_POINTER (0x091B), "/chadeva" },
  { GUINT_TO_POINTER (0x0A9B), "/chagujarati" },
  { GUINT_TO_POINTER (0x0A1B), "/chagurmukhi" },
  { GUINT_TO_POINTER (0x3114), "/chbopomofo" },
  { GUINT_TO_POINTER (0x04BD), "/cheabkhasiancyrillic" },
  { GUINT_TO_POINTER (0x2713), "/checkmark" },
  { GUINT_TO_POINTER (0x0447), "/checyrillic" },
  { GUINT_TO_POINTER (0x04BF), "/chedescenderabkhasiancyrillic" },
  { GUINT_TO_POINTER (0x04B7), "/chedescendercyrillic" },
  { GUINT_TO_POINTER (0x04F5), "/chedieresiscyrillic" },
  { GUINT_TO_POINTER (0x0573), "/cheharmenian" },
  { GUINT_TO_POINTER (0x04CC), "/chekhakassiancyrillic" },
  { GUINT_TO_POINTER (0x04B9), "/cheverticalstrokecyrillic" },
  { GUINT_TO_POINTER (0x03C7), "/chi" },
  { GUINT_TO_POINTER (0x3277), "/chieuchacirclekorean" },
  { GUINT_TO_POINTER (0x3217), "/chieuchaparenkorean" },
  { GUINT_TO_POINTER (0x3269), "/chieuchcirclekorean" },
  { GUINT_TO_POINTER (0x314A), "/chieuchkorean" },
  { GUINT_TO_POINTER (0x3209), "/chieuchparenkorean" },
  { GUINT_TO_POINTER (0x0E0A), "/chochangthai" },
  { GUINT_TO_POINTER (0x0E08), "/chochanthai" },
  { GUINT_TO_POINTER (0x0E09), "/chochingthai" },
  { GUINT_TO_POINTER (0x0E0C), "/chochoethai" },
  { GUINT_TO_POINTER (0x0188), "/chook" },
  { GUINT_TO_POINTER (0x3276), "/cieucacirclekorean" },
  { GUINT_TO_POINTER (0x3216), "/cieucaparenkorean" },
  { GUINT_TO_POINTER (0x3268), "/cieuccirclekorean" },
  { GUINT_TO_POINTER (0x3148), "/cieuckorean" },
  { GUINT_TO_POINTER (0x3208), "/cieucparenkorean" },
  { GUINT_TO_POINTER (0x321C), "/cieucuparenkorean" },
  { GUINT_TO_POINTER (0x25CB), "/circle" },
  { GUINT_TO_POINTER (0x2297), "/circlemultiply" },
  { GUINT_TO_POINTER (0x2299), "/circleot" },
  { GUINT_TO_POINTER (0x2295), "/circleplus" },
  { GUINT_TO_POINTER (0x3036), "/circlepostalmark" },
  { GUINT_TO_POINTER (0x25D0), "/circlewithlefthalfblack" },
  { GUINT_TO_POINTER (0x25D1), "/circlewithrighthalfblack" },
  { GUINT_TO_POINTER (0x02C6), "/circumflex" },
  { GUINT_TO_POINTER (0x032D), "/circumflexbelowcmb" },
  { GUINT_TO_POINTER (0x0302), "/circumflexcmb" },
  { GUINT_TO_POINTER (0x2327), "/clear" },
  { GUINT_TO_POINTER (0x01C2), "/clickalveolar" },
  { GUINT_TO_POINTER (0x01C0), "/clickdental" },
  { GUINT_TO_POINTER (0x01C1), "/clicklateral" },
  { GUINT_TO_POINTER (0x01C3), "/clickretroflex" },
  { GUINT_TO_POINTER (0x2663), "/club" },
  { GUINT_TO_POINTER (0x2663), "/clubsuitblack" },
  { GUINT_TO_POINTER (0x2667), "/clubsuitwhite" },
  { GUINT_TO_POINTER (0x33A4), "/cmcubedsquare" },
  { GUINT_TO_POINTER (0xFF43), "/cmonospace" },
  { GUINT_TO_POINTER (0x33A0), "/cmsquaredsquare" },
  { GUINT_TO_POINTER (0x0581), "/coarmenian" },
  { GUINT_TO_POINTER (0x003A), "/colon" },
  { GUINT_TO_POINTER (0x20A1), "/colonmonetary" },
  { GUINT_TO_POINTER (0xFF1A), "/colonmonospace" },
  { GUINT_TO_POINTER (0x20A1), "/colonsign" },
  { GUINT_TO_POINTER (0xFE55), "/colonsmall" },
  { GUINT_TO_POINTER (0x02D1), "/colontriangularhalfmod" },
  { GUINT_TO_POINTER (0x02D0), "/colontriangularmod" },
  { GUINT_TO_POINTER (0x002C), "/comma" },
  { GUINT_TO_POINTER (0x0313), "/commaabovecmb" },
  { GUINT_TO_POINTER (0x0315), "/commaaboverightcmb" },
  { GUINT_TO_POINTER (0xF6C3), "/commaaccent" },
  { GUINT_TO_POINTER (0x060C), "/commaarabic" },
  { GUINT_TO_POINTER (0x055D), "/commaarmenian" },
  { GUINT_TO_POINTER (0xF6E1), "/commainferior" },
  { GUINT_TO_POINTER (0xFF0C), "/commamonospace" },
  { GUINT_TO_POINTER (0x0314), "/commareversedabovecmb" },
  { GUINT_TO_POINTER (0x02BD), "/commareversedmod" },
  { GUINT_TO_POINTER (0xFE50), "/commasmall" },
  { GUINT_TO_POINTER (0xF6E2), "/commasuperior" },
  { GUINT_TO_POINTER (0x0312), "/commaturnedabovecmb" },
  { GUINT_TO_POINTER (0x02BB), "/commaturnedmod" },
  { GUINT_TO_POINTER (0x263C), "/compass" },
  { GUINT_TO_POINTER (0x2245), "/congruent" },
  { GUINT_TO_POINTER (0x222E), "/contourintegral" },
  { GUINT_TO_POINTER (0x2303), "/control" },
  { GUINT_TO_POINTER (0x0006), "/controlACK" },
  { GUINT_TO_POINTER (0x0007), "/controlBEL" },
  { GUINT_TO_POINTER (0x0008), "/controlBS" },
  { GUINT_TO_POINTER (0x0018), "/controlCAN" },
  { GUINT_TO_POINTER (0x000D), "/controlCR" },
  { GUINT_TO_POINTER (0x0011), "/controlDC1" },
  { GUINT_TO_POINTER (0x0012), "/controlDC2" },
  { GUINT_TO_POINTER (0x0013), "/controlDC3" },
  { GUINT_TO_POINTER (0x0014), "/controlDC4" },
  { GUINT_TO_POINTER (0x007F), "/controlDEL" },
  { GUINT_TO_POINTER (0x0010), "/controlDLE" },
  { GUINT_TO_POINTER (0x0019), "/controlEM" },
  { GUINT_TO_POINTER (0x0005), "/controlENQ" },
  { GUINT_TO_POINTER (0x0004), "/controlEOT" },
  { GUINT_TO_POINTER (0x001B), "/controlESC" },
  { GUINT_TO_POINTER (0x0017), "/controlETB" },
  { GUINT_TO_POINTER (0x0003), "/controlETX" },
  { GUINT_TO_POINTER (0x000C), "/controlFF" },
  { GUINT_TO_POINTER (0x001C), "/controlFS" },
  { GUINT_TO_POINTER (0x001D), "/controlGS" },
  { GUINT_TO_POINTER (0x0009), "/controlHT" },
  { GUINT_TO_POINTER (0x000A), "/controlLF" },
  { GUINT_TO_POINTER (0x0015), "/controlNAK" },
  { GUINT_TO_POINTER (0x001E), "/controlRS" },
  { GUINT_TO_POINTER (0x000F), "/controlSI" },
  { GUINT_TO_POINTER (0x000E), "/controlSO" },
  { GUINT_TO_POINTER (0x0002), "/controlSOT" },
  { GUINT_TO_POINTER (0x0001), "/controlSTX" },
  { GUINT_TO_POINTER (0x001A), "/controlSUB" },
  { GUINT_TO_POINTER (0x0016), "/controlSYN" },
  { GUINT_TO_POINTER (0x001F), "/controlUS" },
  { GUINT_TO_POINTER (0x000B), "/controlVT" },
  { GUINT_TO_POINTER (0x00A9), "/copyright" },
  { GUINT_TO_POINTER (0xF8E9), "/copyrightsans" },
  { GUINT_TO_POINTER (0xF6D9), "/copyrightserif" },
  { GUINT_TO_POINTER (0x300C), "/cornerbracketleft" },
  { GUINT_TO_POINTER (0xFF62), "/cornerbracketlefthalfwidth" },
  { GUINT_TO_POINTER (0xFE41), "/cornerbracketleftvertical" },
  { GUINT_TO_POINTER (0x300D), "/cornerbracketright" },
  { GUINT_TO_POINTER (0xFF63), "/cornerbracketrighthalfwidth" },
  { GUINT_TO_POINTER (0xFE42), "/cornerbracketrightvertical" },
  { GUINT_TO_POINTER (0x337F), "/corporationsquare" },
  { GUINT_TO_POINTER (0x33C7), "/cosquare" },
  { GUINT_TO_POINTER (0x33C6), "/coverkgsquare" },
  { GUINT_TO_POINTER (0x249E), "/cparen" },
  { GUINT_TO_POINTER (0x20A2), "/cruzeiro" },
  { GUINT_TO_POINTER (0x0297), "/cstretched" },
  { GUINT_TO_POINTER (0x22CF), "/curlyand" },
  { GUINT_TO_POINTER (0x22CE), "/curlyor" },
  { GUINT_TO_POINTER (0x00A4), "/currency" },
  { GUINT_TO_POINTER (0xF6D1), "/cyrBreve" },
  { GUINT_TO_POINTER (0xF6D2), "/cyrFlex" },
  { GUINT_TO_POINTER (0xF6D4), "/cyrbreve" },
  { GUINT_TO_POINTER (0xF6D5), "/cyrflex" },
  { GUINT_TO_POINTER (0x0064), "/d" },
  { GUINT_TO_POINTER (0x0564), "/daarmenian" },
  { GUINT_TO_POINTER (0x09A6), "/dabengali" },
  { GUINT_TO_POINTER (0x0636), "/dadarabic" },
  { GUINT_TO_POINTER (0x0926), "/dadeva" },
  { GUINT_TO_POINTER (0xFEBE), "/dadfinalarabic" },
  { GUINT_TO_POINTER (0xFEBF), "/dadinitialarabic" },
  { GUINT_TO_POINTER (0xFEC0), "/dadmedialarabic" },
  { GUINT_TO_POINTER (0x05BC), "/dagesh" },
  { GUINT_TO_POINTER (0x05BC), "/dageshhebrew" },
  { GUINT_TO_POINTER (0x2020), "/dagger" },
  { GUINT_TO_POINTER (0x2021), "/daggerdbl" },
  { GUINT_TO_POINTER (0x0AA6), "/dagujarati" },
  { GUINT_TO_POINTER (0x0A26), "/dagurmukhi" },
  { GUINT_TO_POINTER (0x3060), "/dahiragana" },
  { GUINT_TO_POINTER (0x30C0), "/dakatakana" },
  { GUINT_TO_POINTER (0x062F), "/dalarabic" },
  { GUINT_TO_POINTER (0x05D3), "/dalet" },
  { GUINT_TO_POINTER (0xFB33), "/daletdagesh" },
  { GUINT_TO_POINTER (0xFB33), "/daletdageshhebrew" },
  { GUINT_TO_POINTER (0x05D3), "/dalethatafpatah" },
  { GUINT_TO_POINTER (0x05B2), "/dalethatafpatah" },
  { GUINT_TO_POINTER (0x05D3), "/dalethatafpatahhebrew" },
  { GUINT_TO_POINTER (0x05B2), "/dalethatafpatahhebrew" },
  { GUINT_TO_POINTER (0x05D3), "/dalethatafsegol" },
  { GUINT_TO_POINTER (0x05B1), "/dalethatafsegol" },
  { GUINT_TO_POINTER (0x05D3), "/dalethatafsegolhebrew" },
  { GUINT_TO_POINTER (0x05B1), "/dalethatafsegolhebrew" },
  { GUINT_TO_POINTER (0x05D3), "/dalethebrew" },
  { GUINT_TO_POINTER (0x05D3), "/dalethiriq" },
  { GUINT_TO_POINTER (0x05B4), "/dalethiriq" },
  { GUINT_TO_POINTER (0x05D3), "/dalethiriqhebrew" },
  { GUINT_TO_POINTER (0x05B4), "/dalethiriqhebrew" },
  { GUINT_TO_POINTER (0x05D3), "/daletholam" },
  { GUINT_TO_POINTER (0x05B9), "/daletholam" },
  { GUINT_TO_POINTER (0x05D3), "/daletholamhebrew" },
  { GUINT_TO_POINTER (0x05B9), "/daletholamhebrew" },
  { GUINT_TO_POINTER (0x05D3), "/daletpatah" },
  { GUINT_TO_POINTER (0x05B7), "/daletpatah" },
  { GUINT_TO_POINTER (0x05D3), "/daletpatahhebrew" },
  { GUINT_TO_POINTER (0x05B7), "/daletpatahhebrew" },
  { GUINT_TO_POINTER (0x05D3), "/daletqamats" },
  { GUINT_TO_POINTER (0x05B8), "/daletqamats" },
  { GUINT_TO_POINTER (0x05D3), "/daletqamatshebrew" },
  { GUINT_TO_POINTER (0x05B8), "/daletqamatshebrew" },
  { GUINT_TO_POINTER (0x05D3), "/daletqubuts" },
  { GUINT_TO_POINTER (0x05BB), "/daletqubuts" },
  { GUINT_TO_POINTER (0x05D3), "/daletqubutshebrew" },
  { GUINT_TO_POINTER (0x05BB), "/daletqubutshebrew" },
  { GUINT_TO_POINTER (0x05D3), "/daletsegol" },
  { GUINT_TO_POINTER (0x05B6), "/daletsegol" },
  { GUINT_TO_POINTER (0x05D3), "/daletsegolhebrew" },
  { GUINT_TO_POINTER (0x05B6), "/daletsegolhebrew" },
  { GUINT_TO_POINTER (0x05D3), "/daletsheva" },
  { GUINT_TO_POINTER (0x05B0), "/daletsheva" },
  { GUINT_TO_POINTER (0x05D3), "/daletshevahebrew" },
  { GUINT_TO_POINTER (0x05B0), "/daletshevahebrew" },
  { GUINT_TO_POINTER (0x05D3), "/dalettsere" },
  { GUINT_TO_POINTER (0x05B5), "/dalettsere" },
  { GUINT_TO_POINTER (0x05D3), "/dalettserehebrew" },
  { GUINT_TO_POINTER (0x05B5), "/dalettserehebrew" },
  { GUINT_TO_POINTER (0xFEAA), "/dalfinalarabic" },
  { GUINT_TO_POINTER (0x064F), "/dammaarabic" },
  { GUINT_TO_POINTER (0x064F), "/dammalowarabic" },
  { GUINT_TO_POINTER (0x064C), "/dammatanaltonearabic" },
  { GUINT_TO_POINTER (0x064C), "/dammatanarabic" },
  { GUINT_TO_POINTER (0x0964), "/danda" },
  { GUINT_TO_POINTER (0x05A7), "/dargahebrew" },
  { GUINT_TO_POINTER (0x05A7), "/dargalefthebrew" },
  { GUINT_TO_POINTER (0x0485), "/dasiapneumatacyrilliccmb" },
  { GUINT_TO_POINTER (0xF6D3), "/dblGrave" },
  { GUINT_TO_POINTER (0x300A), "/dblanglebracketleft" },
  { GUINT_TO_POINTER (0xFE3D), "/dblanglebracketleftvertical" },
  { GUINT_TO_POINTER (0x300B), "/dblanglebracketright" },
  { GUINT_TO_POINTER (0xFE3E), "/dblanglebracketrightvertical" },
  { GUINT_TO_POINTER (0x032B), "/dblarchinvertedbelowcmb" },
  { GUINT_TO_POINTER (0x21D4), "/dblarrowleft" },
  { GUINT_TO_POINTER (0x21D2), "/dblarrowright" },
  { GUINT_TO_POINTER (0x0965), "/dbldanda" },
  { GUINT_TO_POINTER (0xF6D6), "/dblgrave" },
  { GUINT_TO_POINTER (0x030F), "/dblgravecmb" },
  { GUINT_TO_POINTER (0x222C), "/dblintegral" },
  { GUINT_TO_POINTER (0x2017), "/dbllowline" },
  { GUINT_TO_POINTER (0x0333), "/dbllowlinecmb" },
  { GUINT_TO_POINTER (0x033F), "/dbloverlinecmb" },
  { GUINT_TO_POINTER (0x02BA), "/dblprimemod" },
  { GUINT_TO_POINTER (0x2016), "/dblverticalbar" },
  { GUINT_TO_POINTER (0x030E), "/dblverticallineabovecmb" },
  { GUINT_TO_POINTER (0x3109), "/dbopomofo" },
  { GUINT_TO_POINTER (0x33C8), "/dbsquare" },
  { GUINT_TO_POINTER (0x010F), "/dcaron" },
  { GUINT_TO_POINTER (0x1E11), "/dcedilla" },
  { GUINT_TO_POINTER (0x24D3), "/dcircle" },
  { GUINT_TO_POINTER (0x1E13), "/dcircumflexbelow" },
  { GUINT_TO_POINTER (0x0111), "/dcroat" },
  { GUINT_TO_POINTER (0x09A1), "/ddabengali" },
  { GUINT_TO_POINTER (0x0921), "/ddadeva" },
  { GUINT_TO_POINTER (0x0AA1), "/ddagujarati" },
  { GUINT_TO_POINTER (0x0A21), "/ddagurmukhi" },
  { GUINT_TO_POINTER (0x0688), "/ddalarabic" },
  { GUINT_TO_POINTER (0xFB89), "/ddalfinalarabic" },
  { GUINT_TO_POINTER (0x095C), "/dddhadeva" },
  { GUINT_TO_POINTER (0x09A2), "/ddhabengali" },
  { GUINT_TO_POINTER (0x0922), "/ddhadeva" },
  { GUINT_TO_POINTER (0x0AA2), "/ddhagujarati" },
  { GUINT_TO_POINTER (0x0A22), "/ddhagurmukhi" },
  { GUINT_TO_POINTER (0x1E0B), "/ddotaccent" },
  { GUINT_TO_POINTER (0x1E0D), "/ddotbelow" },
  { GUINT_TO_POINTER (0x066B), "/decimalseparatorarabic" },
  { GUINT_TO_POINTER (0x066B), "/decimalseparatorpersian" },
  { GUINT_TO_POINTER (0x0434), "/decyrillic" },
  { GUINT_TO_POINTER (0x00B0), "/degree" },
  { GUINT_TO_POINTER (0x05AD), "/dehihebrew" },
  { GUINT_TO_POINTER (0x3067), "/dehiragana" },
  { GUINT_TO_POINTER (0x03EF), "/deicoptic" },
  { GUINT_TO_POINTER (0x30C7), "/dekatakana" },
  { GUINT_TO_POINTER (0x232B), "/deleteleft" },
  { GUINT_TO_POINTER (0x2326), "/deleteright" },
  { GUINT_TO_POINTER (0x03B4), "/delta" },
  { GUINT_TO_POINTER (0x018D), "/deltaturned" },
  { GUINT_TO_POINTER (0x09F8), "/denominatorminusonenumeratorbengali" },
  { GUINT_TO_POINTER (0x02A4), "/dezh" },
  { GUINT_TO_POINTER (0x09A7), "/dhabengali" },
  { GUINT_TO_POINTER (0x0927), "/dhadeva" },
  { GUINT_TO_POINTER (0x0AA7), "/dhagujarati" },
  { GUINT_TO_POINTER (0x0A27), "/dhagurmukhi" },
  { GUINT_TO_POINTER (0x0257), "/dhook" },
  { GUINT_TO_POINTER (0x0385), "/dialytikatonos" },
  { GUINT_TO_POINTER (0x0344), "/dialytikatonoscmb" },
  { GUINT_TO_POINTER (0x2666), "/diamond" },
  { GUINT_TO_POINTER (0x2662), "/diamondsuitwhite" },
  { GUINT_TO_POINTER (0x00A8), "/dieresis" },
  { GUINT_TO_POINTER (0xF6D7), "/dieresisacute" },
  { GUINT_TO_POINTER (0x0324), "/dieresisbelowcmb" },
  { GUINT_TO_POINTER (0x0308), "/dieresiscmb" },
  { GUINT_TO_POINTER (0xF6D8), "/dieresisgrave" },
  { GUINT_TO_POINTER (0x0385), "/dieresistonos" },
  { GUINT_TO_POINTER (0x3062), "/dihiragana" },
  { GUINT_TO_POINTER (0x30C2), "/dikatakana" },
  { GUINT_TO_POINTER (0x3003), "/dittomark" },
  { GUINT_TO_POINTER (0x00F7), "/divide" },
  { GUINT_TO_POINTER (0x2223), "/divides" },
  { GUINT_TO_POINTER (0x2215), "/divisionslash" },
  { GUINT_TO_POINTER (0x0452), "/djecyrillic" },
  { GUINT_TO_POINTER (0x2593), "/dkshade" },
  { GUINT_TO_POINTER (0x1E0F), "/dlinebelow" },
  { GUINT_TO_POINTER (0x3397), "/dlsquare" },
  { GUINT_TO_POINTER (0x0111), "/dmacron" },
  { GUINT_TO_POINTER (0xFF44), "/dmonospace" },
  { GUINT_TO_POINTER (0x2584), "/dnblock" },
  { GUINT_TO_POINTER (0x0E0E), "/dochadathai" },
  { GUINT_TO_POINTER (0x0E14), "/dodekthai" },
  { GUINT_TO_POINTER (0x3069), "/dohiragana" },
  { GUINT_TO_POINTER (0x30C9), "/dokatakana" },
  { GUINT_TO_POINTER (0x0024), "/dollar" },
  { GUINT_TO_POINTER (0xF6E3), "/dollarinferior" },
  { GUINT_TO_POINTER (0xFF04), "/dollarmonospace" },
  { GUINT_TO_POINTER (0xF724), "/dollaroldstyle" },
  { GUINT_TO_POINTER (0xFE69), "/dollarsmall" },
  { GUINT_TO_POINTER (0xF6E4), "/dollarsuperior" },
  { GUINT_TO_POINTER (0x20AB), "/dong" },
  { GUINT_TO_POINTER (0x3326), "/dorusquare" },
  { GUINT_TO_POINTER (0x02D9), "/dotaccent" },
  { GUINT_TO_POINTER (0x0307), "/dotaccentcmb" },
  { GUINT_TO_POINTER (0x0323), "/dotbelowcmb" },
  { GUINT_TO_POINTER (0x0323), "/dotbelowcomb" },
  { GUINT_TO_POINTER (0x30FB), "/dotkatakana" },
  { GUINT_TO_POINTER (0x0131), "/dotlessi" },
  { GUINT_TO_POINTER (0xF6BE), "/dotlessj" },
  { GUINT_TO_POINTER (0x0284), "/dotlessjstrokehook" },
  { GUINT_TO_POINTER (0x22C5), "/dotmath" },
  { GUINT_TO_POINTER (0x25CC), "/dottedcircle" },
  { GUINT_TO_POINTER (0xFB1F), "/doubleyodpatah" },
  { GUINT_TO_POINTER (0xFB1F), "/doubleyodpatahhebrew" },
  { GUINT_TO_POINTER (0x031E), "/downtackbelowcmb" },
  { GUINT_TO_POINTER (0x02D5), "/downtackmod" },
  { GUINT_TO_POINTER (0x249F), "/dparen" },
  { GUINT_TO_POINTER (0xF6EB), "/dsuperior" },
  { GUINT_TO_POINTER (0x0256), "/dtail" },
  { GUINT_TO_POINTER (0x018C), "/dtopbar" },
  { GUINT_TO_POINTER (0x3065), "/duhiragana" },
  { GUINT_TO_POINTER (0x30C5), "/dukatakana" },
  { GUINT_TO_POINTER (0x01F3), "/dz" },
  { GUINT_TO_POINTER (0x02A3), "/dzaltone" },
  { GUINT_TO_POINTER (0x01C6), "/dzcaron" },
  { GUINT_TO_POINTER (0x02A5), "/dzcurl" },
  { GUINT_TO_POINTER (0x04E1), "/dzeabkhasiancyrillic" },
  { GUINT_TO_POINTER (0x0455), "/dzecyrillic" },
  { GUINT_TO_POINTER (0x045F), "/dzhecyrillic" },
  { GUINT_TO_POINTER (0x0065), "/e" },
  { GUINT_TO_POINTER (0x00E9), "/eacute" },
  { GUINT_TO_POINTER (0x2641), "/earth" },
  { GUINT_TO_POINTER (0x098F), "/ebengali" },
  { GUINT_TO_POINTER (0x311C), "/ebopomofo" },
  { GUINT_TO_POINTER (0x0115), "/ebreve" },
  { GUINT_TO_POINTER (0x090D), "/ecandradeva" },
  { GUINT_TO_POINTER (0x0A8D), "/ecandragujarati" },
  { GUINT_TO_POINTER (0x0945), "/ecandravowelsigndeva" },
  { GUINT_TO_POINTER (0x0AC5), "/ecandravowelsigngujarati" },
  { GUINT_TO_POINTER (0x011B), "/ecaron" },
  { GUINT_TO_POINTER (0x1E1D), "/ecedillabreve" },
  { GUINT_TO_POINTER (0x0565), "/echarmenian" },
  { GUINT_TO_POINTER (0x0587), "/echyiwnarmenian" },
  { GUINT_TO_POINTER (0x24D4), "/ecircle" },
  { GUINT_TO_POINTER (0x00EA), "/ecircumflex" },
  { GUINT_TO_POINTER (0x1EBF), "/ecircumflexacute" },
  { GUINT_TO_POINTER (0x1E19), "/ecircumflexbelow" },
  { GUINT_TO_POINTER (0x1EC7), "/ecircumflexdotbelow" },
  { GUINT_TO_POINTER (0x1EC1), "/ecircumflexgrave" },
  { GUINT_TO_POINTER (0x1EC3), "/ecircumflexhookabove" },
  { GUINT_TO_POINTER (0x1EC5), "/ecircumflextilde" },
  { GUINT_TO_POINTER (0x0454), "/ecyrillic" },
  { GUINT_TO_POINTER (0x0205), "/edblgrave" },
  { GUINT_TO_POINTER (0x090F), "/edeva" },
  { GUINT_TO_POINTER (0x00EB), "/edieresis" },
  { GUINT_TO_POINTER (0x0117), "/edot" },
  { GUINT_TO_POINTER (0x0117), "/edotaccent" },
  { GUINT_TO_POINTER (0x1EB9), "/edotbelow" },
  { GUINT_TO_POINTER (0x0A0F), "/eegurmukhi" },
  { GUINT_TO_POINTER (0x0A47), "/eematragurmukhi" },
  { GUINT_TO_POINTER (0x0444), "/efcyrillic" },
  { GUINT_TO_POINTER (0x00E8), "/egrave" },
  { GUINT_TO_POINTER (0x0A8F), "/egujarati" },
  { GUINT_TO_POINTER (0x0567), "/eharmenian" },
  { GUINT_TO_POINTER (0x311D), "/ehbopomofo" },
  { GUINT_TO_POINTER (0x3048), "/ehiragana" },
  { GUINT_TO_POINTER (0x1EBB), "/ehookabove" },
  { GUINT_TO_POINTER (0x311F), "/eibopomofo" },
  { GUINT_TO_POINTER (0x0038), "/eight" },
  { GUINT_TO_POINTER (0x0668), "/eightarabic" },
  { GUINT_TO_POINTER (0x09EE), "/eightbengali" },
  { GUINT_TO_POINTER (0x2467), "/eightcircle" },
  { GUINT_TO_POINTER (0x2791), "/eightcircleinversesansserif" },
  { GUINT_TO_POINTER (0x096E), "/eightdeva" },
  { GUINT_TO_POINTER (0x2471), "/eighteencircle" },
  { GUINT_TO_POINTER (0x2485), "/eighteenparen" },
  { GUINT_TO_POINTER (0x2499), "/eighteenperiod" },
  { GUINT_TO_POINTER (0x0AEE), "/eightgujarati" },
  { GUINT_TO_POINTER (0x0A6E), "/eightgurmukhi" },
  { GUINT_TO_POINTER (0x0668), "/eighthackarabic" },
  { GUINT_TO_POINTER (0x3028), "/eighthangzhou" },
  { GUINT_TO_POINTER (0x266B), "/eighthnotebeamed" },
  { GUINT_TO_POINTER (0x3227), "/eightideographicparen" },
  { GUINT_TO_POINTER (0x2088), "/eightinferior" },
  { GUINT_TO_POINTER (0xFF18), "/eightmonospace" },
  { GUINT_TO_POINTER (0xF738), "/eightoldstyle" },
  { GUINT_TO_POINTER (0x247B), "/eightparen" },
  { GUINT_TO_POINTER (0x248F), "/eightperiod" },
  { GUINT_TO_POINTER (0x06F8), "/eightpersian" },
  { GUINT_TO_POINTER (0x2177), "/eightroman" },
  { GUINT_TO_POINTER (0x2078), "/eightsuperior" },
  { GUINT_TO_POINTER (0x0E58), "/eightthai" },
  { GUINT_TO_POINTER (0x0207), "/einvertedbreve" },
  { GUINT_TO_POINTER (0x0465), "/eiotifiedcyrillic" },
  { GUINT_TO_POINTER (0x30A8), "/ekatakana" },
  { GUINT_TO_POINTER (0xFF74), "/ekatakanahalfwidth" },
  { GUINT_TO_POINTER (0x0A74), "/ekonkargurmukhi" },
  { GUINT_TO_POINTER (0x3154), "/ekorean" },
  { GUINT_TO_POINTER (0x043B), "/elcyrillic" },
  { GUINT_TO_POINTER (0x2208), "/element" },
  { GUINT_TO_POINTER (0x246A), "/elevencircle" },
  { GUINT_TO_POINTER (0x247E), "/elevenparen" },
  { GUINT_TO_POINTER (0x2492), "/elevenperiod" },
  { GUINT_TO_POINTER (0x217A), "/elevenroman" },
  { GUINT_TO_POINTER (0x2026), "/ellipsis" },
  { GUINT_TO_POINTER (0x22EE), "/ellipsisvertical" },
  { GUINT_TO_POINTER (0x0113), "/emacron" },
  { GUINT_TO_POINTER (0x1E17), "/emacronacute" },
  { GUINT_TO_POINTER (0x1E15), "/emacrongrave" },
  { GUINT_TO_POINTER (0x043C), "/emcyrillic" },
  { GUINT_TO_POINTER (0x2014), "/emdash" },
  { GUINT_TO_POINTER (0xFE31), "/emdashvertical" },
  { GUINT_TO_POINTER (0xFF45), "/emonospace" },
  { GUINT_TO_POINTER (0x055B), "/emphasismarkarmenian" },
  { GUINT_TO_POINTER (0x2205), "/emptyset" },
  { GUINT_TO_POINTER (0x3123), "/enbopomofo" },
  { GUINT_TO_POINTER (0x043D), "/encyrillic" },
  { GUINT_TO_POINTER (0x2013), "/endash" },
  { GUINT_TO_POINTER (0xFE32), "/endashvertical" },
  { GUINT_TO_POINTER (0x04A3), "/endescendercyrillic" },
  { GUINT_TO_POINTER (0x014B), "/eng" },
  { GUINT_TO_POINTER (0x3125), "/engbopomofo" },
  { GUINT_TO_POINTER (0x04A5), "/enghecyrillic" },
  { GUINT_TO_POINTER (0x04C8), "/enhookcyrillic" },
  { GUINT_TO_POINTER (0x2002), "/enspace" },
  { GUINT_TO_POINTER (0x0119), "/eogonek" },
  { GUINT_TO_POINTER (0x3153), "/eokorean" },
  { GUINT_TO_POINTER (0x025B), "/eopen" },
  { GUINT_TO_POINTER (0x029A), "/eopenclosed" },
  { GUINT_TO_POINTER (0x025C), "/eopenreversed" },
  { GUINT_TO_POINTER (0x025E), "/eopenreversedclosed" },
  { GUINT_TO_POINTER (0x025D), "/eopenreversedhook" },
  { GUINT_TO_POINTER (0x24A0), "/eparen" },
  { GUINT_TO_POINTER (0x03B5), "/epsilon" },
  { GUINT_TO_POINTER (0x03AD), "/epsilontonos" },
  { GUINT_TO_POINTER (0x003D), "/equal" },
  { GUINT_TO_POINTER (0xFF1D), "/equalmonospace" },
  { GUINT_TO_POINTER (0xFE66), "/equalsmall" },
  { GUINT_TO_POINTER (0x207C), "/equalsuperior" },
  { GUINT_TO_POINTER (0x2261), "/equivalence" },
  { GUINT_TO_POINTER (0x3126), "/erbopomofo" },
  { GUINT_TO_POINTER (0x0440), "/ercyrillic" },
  { GUINT_TO_POINTER (0x0258), "/ereversed" },
  { GUINT_TO_POINTER (0x044D), "/ereversedcyrillic" },
  { GUINT_TO_POINTER (0x0441), "/escyrillic" },
  { GUINT_TO_POINTER (0x04AB), "/esdescendercyrillic" },
  { GUINT_TO_POINTER (0x0283), "/esh" },
  { GUINT_TO_POINTER (0x0286), "/eshcurl" },
  { GUINT_TO_POINTER (0x090E), "/eshortdeva" },
  { GUINT_TO_POINTER (0x0946), "/eshortvowelsigndeva" },
  { GUINT_TO_POINTER (0x01AA), "/eshreversedloop" },
  { GUINT_TO_POINTER (0x0285), "/eshsquatreversed" },
  { GUINT_TO_POINTER (0x3047), "/esmallhiragana" },
  { GUINT_TO_POINTER (0x30A7), "/esmallkatakana" },
  { GUINT_TO_POINTER (0xFF6A), "/esmallkatakanahalfwidth" },
  { GUINT_TO_POINTER (0x212E), "/estimated" },
  { GUINT_TO_POINTER (0xF6EC), "/esuperior" },
  { GUINT_TO_POINTER (0x03B7), "/eta" },
  { GUINT_TO_POINTER (0x0568), "/etarmenian" },
  { GUINT_TO_POINTER (0x03AE), "/etatonos" },
  { GUINT_TO_POINTER (0x00F0), "/eth" },
  { GUINT_TO_POINTER (0x1EBD), "/etilde" },
  { GUINT_TO_POINTER (0x1E1B), "/etildebelow" },
  { GUINT_TO_POINTER (0x0591), "/etnahtafoukhhebrew" },
  { GUINT_TO_POINTER (0x0591), "/etnahtafoukhlefthebrew" },
  { GUINT_TO_POINTER (0x0591), "/etnahtahebrew" },
  { GUINT_TO_POINTER (0x0591), "/etnahtalefthebrew" },
  { GUINT_TO_POINTER (0x01DD), "/eturned" },
  { GUINT_TO_POINTER (0x3161), "/eukorean" },
  { GUINT_TO_POINTER (0x20AC), "/euro" },
  { GUINT_TO_POINTER (0x09C7), "/evowelsignbengali" },
  { GUINT_TO_POINTER (0x0947), "/evowelsigndeva" },
  { GUINT_TO_POINTER (0x0AC7), "/evowelsigngujarati" },
  { GUINT_TO_POINTER (0x0021), "/exclam" },
  { GUINT_TO_POINTER (0x055C), "/exclamarmenian" },
  { GUINT_TO_POINTER (0x203C), "/exclamdbl" },
  { GUINT_TO_POINTER (0x00A1), "/exclamdown" },
  { GUINT_TO_POINTER (0xF7A1), "/exclamdownsmall" },
  { GUINT_TO_POINTER (0xFF01), "/exclammonospace" },
  { GUINT_TO_POINTER (0xF721), "/exclamsmall" },
  { GUINT_TO_POINTER (0x2203), "/existential" },
  { GUINT_TO_POINTER (0x0292), "/ezh" },
  { GUINT_TO_POINTER (0x01EF), "/ezhcaron" },
  { GUINT_TO_POINTER (0x0293), "/ezhcurl" },
  { GUINT_TO_POINTER (0x01B9), "/ezhreversed" },
  { GUINT_TO_POINTER (0x01BA), "/ezhtail" },
  { GUINT_TO_POINTER (0x0066), "/f" },
  { GUINT_TO_POINTER (0x095E), "/fadeva" },
  { GUINT_TO_POINTER (0x0A5E), "/fagurmukhi" },
  { GUINT_TO_POINTER (0x2109), "/fahrenheit" },
  { GUINT_TO_POINTER (0x064E), "/fathaarabic" },
  { GUINT_TO_POINTER (0x064E), "/fathalowarabic" },
  { GUINT_TO_POINTER (0x064B), "/fathatanarabic" },
  { GUINT_TO_POINTER (0x3108), "/fbopomofo" },
  { GUINT_TO_POINTER (0x24D5), "/fcircle" },
  { GUINT_TO_POINTER (0x1E1F), "/fdotaccent" },
  { GUINT_TO_POINTER (0x0641), "/feharabic" },
  { GUINT_TO_POINTER (0x0586), "/feharmenian" },
  { GUINT_TO_POINTER (0xFED2), "/fehfinalarabic" },
  { GUINT_TO_POINTER (0xFED3), "/fehinitialarabic" },
  { GUINT_TO_POINTER (0xFED4), "/fehmedialarabic" },
  { GUINT_TO_POINTER (0x03E5), "/feicoptic" },
  { GUINT_TO_POINTER (0x2640), "/female" },
  { GUINT_TO_POINTER (0xFB00), "/ff" },
  { GUINT_TO_POINTER (0xFB03), "/ffi" },
  { GUINT_TO_POINTER (0xFB04), "/ffl" },
  { GUINT_TO_POINTER (0xFB01), "/fi" },
  { GUINT_TO_POINTER (0x246E), "/fifteencircle" },
  { GUINT_TO_POINTER (0x2482), "/fifteenparen" },
  { GUINT_TO_POINTER (0x2496), "/fifteenperiod" },
  { GUINT_TO_POINTER (0x2012), "/figuredash" },
  { GUINT_TO_POINTER (0x25A0), "/filledbox" },
  { GUINT_TO_POINTER (0x25AC), "/filledrect" },
  { GUINT_TO_POINTER (0x05DA), "/finalkaf" },
  { GUINT_TO_POINTER (0xFB3A), "/finalkafdagesh" },
  { GUINT_TO_POINTER (0xFB3A), "/finalkafdageshhebrew" },
  { GUINT_TO_POINTER (0x05DA), "/finalkafhebrew" },
  { GUINT_TO_POINTER (0x05DA), "/finalkafqamats" },
  { GUINT_TO_POINTER (0x05B8), "/finalkafqamats" },
  { GUINT_TO_POINTER (0x05DA), "/finalkafqamatshebrew" },
  { GUINT_TO_POINTER (0x05B8), "/finalkafqamatshebrew" },
  { GUINT_TO_POINTER (0x05DA), "/finalkafsheva" },
  { GUINT_TO_POINTER (0x05B0), "/finalkafsheva" },
  { GUINT_TO_POINTER (0x05DA), "/finalkafshevahebrew" },
  { GUINT_TO_POINTER (0x05B0), "/finalkafshevahebrew" },
  { GUINT_TO_POINTER (0x05DD), "/finalmem" },
  { GUINT_TO_POINTER (0x05DD), "/finalmemhebrew" },
  { GUINT_TO_POINTER (0x05DF), "/finalnun" },
  { GUINT_TO_POINTER (0x05DF), "/finalnunhebrew" },
  { GUINT_TO_POINTER (0x05E3), "/finalpe" },
  { GUINT_TO_POINTER (0x05E3), "/finalpehebrew" },
  { GUINT_TO_POINTER (0x05E5), "/finaltsadi" },
  { GUINT_TO_POINTER (0x05E5), "/finaltsadihebrew" },
  { GUINT_TO_POINTER (0x02C9), "/firsttonechinese" },
  { GUINT_TO_POINTER (0x25C9), "/fisheye" },
  { GUINT_TO_POINTER (0x0473), "/fitacyrillic" },
  { GUINT_TO_POINTER (0x0035), "/five" },
  { GUINT_TO_POINTER (0x0665), "/fivearabic" },
  { GUINT_TO_POINTER (0x09EB), "/fivebengali" },
  { GUINT_TO_POINTER (0x2464), "/fivecircle" },
  { GUINT_TO_POINTER (0x278E), "/fivecircleinversesansserif" },
  { GUINT_TO_POINTER (0x096B), "/fivedeva" },
  { GUINT_TO_POINTER (0x215D), "/fiveeighths" },
  { GUINT_TO_POINTER (0x0AEB), "/fivegujarati" },
  { GUINT_TO_POINTER (0x0A6B), "/fivegurmukhi" },
  { GUINT_TO_POINTER (0x0665), "/fivehackarabic" },
  { GUINT_TO_POINTER (0x3025), "/fivehangzhou" },
  { GUINT_TO_POINTER (0x3224), "/fiveideographicparen" },
  { GUINT_TO_POINTER (0x2085), "/fiveinferior" },
  { GUINT_TO_POINTER (0xFF15), "/fivemonospace" },
  { GUINT_TO_POINTER (0xF735), "/fiveoldstyle" },
  { GUINT_TO_POINTER (0x2478), "/fiveparen" },
  { GUINT_TO_POINTER (0x248C), "/fiveperiod" },
  { GUINT_TO_POINTER (0x06F5), "/fivepersian" },
  { GUINT_TO_POINTER (0x2174), "/fiveroman" },
  { GUINT_TO_POINTER (0x2075), "/fivesuperior" },
  { GUINT_TO_POINTER (0x0E55), "/fivethai" },
  { GUINT_TO_POINTER (0xFB02), "/fl" },
  { GUINT_TO_POINTER (0x0192), "/florin" },
  { GUINT_TO_POINTER (0xFF46), "/fmonospace" },
  { GUINT_TO_POINTER (0x3399), "/fmsquare" },
  { GUINT_TO_POINTER (0x0E1F), "/fofanthai" },
  { GUINT_TO_POINTER (0x0E1D), "/fofathai" },
  { GUINT_TO_POINTER (0x0E4F), "/fongmanthai" },
  { GUINT_TO_POINTER (0x2200), "/forall" },
  { GUINT_TO_POINTER (0x0034), "/four" },
  { GUINT_TO_POINTER (0x0664), "/fourarabic" },
  { GUINT_TO_POINTER (0x09EA), "/fourbengali" },
  { GUINT_TO_POINTER (0x2463), "/fourcircle" },
  { GUINT_TO_POINTER (0x278D), "/fourcircleinversesansserif" },
  { GUINT_TO_POINTER (0x096A), "/fourdeva" },
  { GUINT_TO_POINTER (0x0AEA), "/fourgujarati" },
  { GUINT_TO_POINTER (0x0A6A), "/fourgurmukhi" },
  { GUINT_TO_POINTER (0x0664), "/fourhackarabic" },
  { GUINT_TO_POINTER (0x3024), "/fourhangzhou" },
  { GUINT_TO_POINTER (0x3223), "/fourideographicparen" },
  { GUINT_TO_POINTER (0x2084), "/fourinferior" },
  { GUINT_TO_POINTER (0xFF14), "/fourmonospace" },
  { GUINT_TO_POINTER (0x09F7), "/fournumeratorbengali" },
  { GUINT_TO_POINTER (0xF734), "/fouroldstyle" },
  { GUINT_TO_POINTER (0x2477), "/fourparen" },
  { GUINT_TO_POINTER (0x248B), "/fourperiod" },
  { GUINT_TO_POINTER (0x06F4), "/fourpersian" },
  { GUINT_TO_POINTER (0x2173), "/fourroman" },
  { GUINT_TO_POINTER (0x2074), "/foursuperior" },
  { GUINT_TO_POINTER (0x246D), "/fourteencircle" },
  { GUINT_TO_POINTER (0x2481), "/fourteenparen" },
  { GUINT_TO_POINTER (0x2495), "/fourteenperiod" },
  { GUINT_TO_POINTER (0x0E54), "/fourthai" },
  { GUINT_TO_POINTER (0x02CB), "/fourthtonechinese" },
  { GUINT_TO_POINTER (0x24A1), "/fparen" },
  { GUINT_TO_POINTER (0x2044), "/fraction" },
  { GUINT_TO_POINTER (0x20A3), "/franc" },
  { GUINT_TO_POINTER (0x0067), "/g" },
  { GUINT_TO_POINTER (0x0997), "/gabengali" },
  { GUINT_TO_POINTER (0x01F5), "/gacute" },
  { GUINT_TO_POINTER (0x0917), "/gadeva" },
  { GUINT_TO_POINTER (0x06AF), "/gafarabic" },
  { GUINT_TO_POINTER (0xFB93), "/gaffinalarabic" },
  { GUINT_TO_POINTER (0xFB94), "/gafinitialarabic" },
  { GUINT_TO_POINTER (0xFB95), "/gafmedialarabic" },
  { GUINT_TO_POINTER (0x0A97), "/gagujarati" },
  { GUINT_TO_POINTER (0x0A17), "/gagurmukhi" },
  { GUINT_TO_POINTER (0x304C), "/gahiragana" },
  { GUINT_TO_POINTER (0x30AC), "/gakatakana" },
  { GUINT_TO_POINTER (0x03B3), "/gamma" },
  { GUINT_TO_POINTER (0x0263), "/gammalatinsmall" },
  { GUINT_TO_POINTER (0x02E0), "/gammasuperior" },
  { GUINT_TO_POINTER (0x03EB), "/gangiacoptic" },
  { GUINT_TO_POINTER (0x310D), "/gbopomofo" },
  { GUINT_TO_POINTER (0x011F), "/gbreve" },
  { GUINT_TO_POINTER (0x01E7), "/gcaron" },
  { GUINT_TO_POINTER (0x0123), "/gcedilla" },
  { GUINT_TO_POINTER (0x24D6), "/gcircle" },
  { GUINT_TO_POINTER (0x011D), "/gcircumflex" },
  { GUINT_TO_POINTER (0x0123), "/gcommaaccent" },
  { GUINT_TO_POINTER (0x0121), "/gdot" },
  { GUINT_TO_POINTER (0x0121), "/gdotaccent" },
  { GUINT_TO_POINTER (0x0433), "/gecyrillic" },
  { GUINT_TO_POINTER (0x3052), "/gehiragana" },
  { GUINT_TO_POINTER (0x30B2), "/gekatakana" },
  { GUINT_TO_POINTER (0x2251), "/geometricallyequal" },
  { GUINT_TO_POINTER (0x059C), "/gereshaccenthebrew" },
  { GUINT_TO_POINTER (0x05F3), "/gereshhebrew" },
  { GUINT_TO_POINTER (0x059D), "/gereshmuqdamhebrew" },
  { GUINT_TO_POINTER (0x00DF), "/germandbls" },
  { GUINT_TO_POINTER (0x059E), "/gershayimaccenthebrew" },
  { GUINT_TO_POINTER (0x05F4), "/gershayimhebrew" },
  { GUINT_TO_POINTER (0x3013), "/getamark" },
  { GUINT_TO_POINTER (0x0998), "/ghabengali" },
  { GUINT_TO_POINTER (0x0572), "/ghadarmenian" },
  { GUINT_TO_POINTER (0x0918), "/ghadeva" },
  { GUINT_TO_POINTER (0x0A98), "/ghagujarati" },
  { GUINT_TO_POINTER (0x0A18), "/ghagurmukhi" },
  { GUINT_TO_POINTER (0x063A), "/ghainarabic" },
  { GUINT_TO_POINTER (0xFECE), "/ghainfinalarabic" },
  { GUINT_TO_POINTER (0xFECF), "/ghaininitialarabic" },
  { GUINT_TO_POINTER (0xFED0), "/ghainmedialarabic" },
  { GUINT_TO_POINTER (0x0495), "/ghemiddlehookcyrillic" },
  { GUINT_TO_POINTER (0x0493), "/ghestrokecyrillic" },
  { GUINT_TO_POINTER (0x0491), "/gheupturncyrillic" },
  { GUINT_TO_POINTER (0x095A), "/ghhadeva" },
  { GUINT_TO_POINTER (0x0A5A), "/ghhagurmukhi" },
  { GUINT_TO_POINTER (0x0260), "/ghook" },
  { GUINT_TO_POINTER (0x3393), "/ghzsquare" },
  { GUINT_TO_POINTER (0x304E), "/gihiragana" },
  { GUINT_TO_POINTER (0x30AE), "/gikatakana" },
  { GUINT_TO_POINTER (0x0563), "/gimarmenian" },
  { GUINT_TO_POINTER (0x05D2), "/gimel" },
  { GUINT_TO_POINTER (0xFB32), "/gimeldagesh" },
  { GUINT_TO_POINTER (0xFB32), "/gimeldageshhebrew" },
  { GUINT_TO_POINTER (0x05D2), "/gimelhebrew" },
  { GUINT_TO_POINTER (0x0453), "/gjecyrillic" },
  { GUINT_TO_POINTER (0x01BE), "/glottalinvertedstroke" },
  { GUINT_TO_POINTER (0x0294), "/glottalstop" },
  { GUINT_TO_POINTER (0x0296), "/glottalstopinverted" },
  { GUINT_TO_POINTER (0x02C0), "/glottalstopmod" },
  { GUINT_TO_POINTER (0x0295), "/glottalstopreversed" },
  { GUINT_TO_POINTER (0x02C1), "/glottalstopreversedmod" },
  { GUINT_TO_POINTER (0x02E4), "/glottalstopreversedsuperior" },
  { GUINT_TO_POINTER (0x02A1), "/glottalstopstroke" },
  { GUINT_TO_POINTER (0x02A2), "/glottalstopstrokereversed" },
  { GUINT_TO_POINTER (0x1E21), "/gmacron" },
  { GUINT_TO_POINTER (0xFF47), "/gmonospace" },
  { GUINT_TO_POINTER (0x3054), "/gohiragana" },
  { GUINT_TO_POINTER (0x30B4), "/gokatakana" },
  { GUINT_TO_POINTER (0x24A2), "/gparen" },
  { GUINT_TO_POINTER (0x33AC), "/gpasquare" },
  { GUINT_TO_POINTER (0x2207), "/gradient" },
  { GUINT_TO_POINTER (0x0060), "/grave" },
  { GUINT_TO_POINTER (0x0316), "/gravebelowcmb" },
  { GUINT_TO_POINTER (0x0300), "/gravecmb" },
  { GUINT_TO_POINTER (0x0300), "/gravecomb" },
  { GUINT_TO_POINTER (0x0953), "/gravedeva" },
  { GUINT_TO_POINTER (0x02CE), "/gravelowmod" },
  { GUINT_TO_POINTER (0xFF40), "/gravemonospace" },
  { GUINT_TO_POINTER (0x0340), "/gravetonecmb" },
  { GUINT_TO_POINTER (0x003E), "/greater" },
  { GUINT_TO_POINTER (0x2265), "/greaterequal" },
  { GUINT_TO_POINTER (0x22DB), "/greaterequalorless" },
  { GUINT_TO_POINTER (0xFF1E), "/greatermonospace" },
  { GUINT_TO_POINTER (0x2273), "/greaterorequivalent" },
  { GUINT_TO_POINTER (0x2277), "/greaterorless" },
  { GUINT_TO_POINTER (0x2267), "/greateroverequal" },
  { GUINT_TO_POINTER (0xFE65), "/greatersmall" },
  { GUINT_TO_POINTER (0x0261), "/gscript" },
  { GUINT_TO_POINTER (0x01E5), "/gstroke" },
  { GUINT_TO_POINTER (0x3050), "/guhiragana" },
  { GUINT_TO_POINTER (0x00AB), "/guillemotleft" },
  { GUINT_TO_POINTER (0x00BB), "/guillemotright" },
  { GUINT_TO_POINTER (0x2039), "/guilsinglleft" },
  { GUINT_TO_POINTER (0x203A), "/guilsinglright" },
  { GUINT_TO_POINTER (0x30B0), "/gukatakana" },
  { GUINT_TO_POINTER (0x3318), "/guramusquare" },
  { GUINT_TO_POINTER (0x33C9), "/gysquare" },
  { GUINT_TO_POINTER (0x0068), "/h" },
  { GUINT_TO_POINTER (0x04A9), "/haabkhasiancyrillic" },
  { GUINT_TO_POINTER (0x06C1), "/haaltonearabic" },
  { GUINT_TO_POINTER (0x09B9), "/habengali" },
  { GUINT_TO_POINTER (0x04B3), "/hadescendercyrillic" },
  { GUINT_TO_POINTER (0x0939), "/hadeva" },
  { GUINT_TO_POINTER (0x0AB9), "/hagujarati" },
  { GUINT_TO_POINTER (0x0A39), "/hagurmukhi" },
  { GUINT_TO_POINTER (0x062D), "/haharabic" },
  { GUINT_TO_POINTER (0xFEA2), "/hahfinalarabic" },
  { GUINT_TO_POINTER (0xFEA3), "/hahinitialarabic" },
  { GUINT_TO_POINTER (0x306F), "/hahiragana" },
  { GUINT_TO_POINTER (0xFEA4), "/hahmedialarabic" },
  { GUINT_TO_POINTER (0x332A), "/haitusquare" },
  { GUINT_TO_POINTER (0x30CF), "/hakatakana" },
  { GUINT_TO_POINTER (0xFF8A), "/hakatakanahalfwidth" },
  { GUINT_TO_POINTER (0x0A4D), "/halantgurmukhi" },
  { GUINT_TO_POINTER (0x0621), "/hamzaarabic" },
  { GUINT_TO_POINTER (0x0621), "/hamzadammaarabic" },
  { GUINT_TO_POINTER (0x064F), "/hamzadammaarabic" },
  { GUINT_TO_POINTER (0x0621), "/hamzadammatanarabic" },
  { GUINT_TO_POINTER (0x064C), "/hamzadammatanarabic" },
  { GUINT_TO_POINTER (0x0621), "/hamzafathaarabic" },
  { GUINT_TO_POINTER (0x064E), "/hamzafathaarabic" },
  { GUINT_TO_POINTER (0x0621), "/hamzafathatanarabic" },
  { GUINT_TO_POINTER (0x064B), "/hamzafathatanarabic" },
  { GUINT_TO_POINTER (0x0621), "/hamzalowarabic" },
  { GUINT_TO_POINTER (0x0621), "/hamzalowkasraarabic" },
  { GUINT_TO_POINTER (0x0650), "/hamzalowkasraarabic" },
  { GUINT_TO_POINTER (0x0621), "/hamzalowkasratanarabic" },
  { GUINT_TO_POINTER (0x064D), "/hamzalowkasratanarabic" },
  { GUINT_TO_POINTER (0x0621), "/hamzasukunarabic" },
  { GUINT_TO_POINTER (0x0652), "/hamzasukunarabic" },
  { GUINT_TO_POINTER (0x3164), "/hangulfiller" },
  { GUINT_TO_POINTER (0x044A), "/hardsigncyrillic" },
  { GUINT_TO_POINTER (0x21BC), "/harpoonleftbarbup" },
  { GUINT_TO_POINTER (0x21C0), "/harpoonrightbarbup" },
  { GUINT_TO_POINTER (0x33CA), "/hasquare" },
  { GUINT_TO_POINTER (0x05B2), "/hatafpatah" },
  { GUINT_TO_POINTER (0x05B2), "/hatafpatah16" },
  { GUINT_TO_POINTER (0x05B2), "/hatafpatah23" },
  { GUINT_TO_POINTER (0x05B2), "/hatafpatah2f" },
  { GUINT_TO_POINTER (0x05B2), "/hatafpatahhebrew" },
  { GUINT_TO_POINTER (0x05B2), "/hatafpatahnarrowhebrew" },
  { GUINT_TO_POINTER (0x05B2), "/hatafpatahquarterhebrew" },
  { GUINT_TO_POINTER (0x05B2), "/hatafpatahwidehebrew" },
  { GUINT_TO_POINTER (0x05B3), "/hatafqamats" },
  { GUINT_TO_POINTER (0x05B3), "/hatafqamats1b" },
  { GUINT_TO_POINTER (0x05B3), "/hatafqamats28" },
  { GUINT_TO_POINTER (0x05B3), "/hatafqamats34" },
  { GUINT_TO_POINTER (0x05B3), "/hatafqamatshebrew" },
  { GUINT_TO_POINTER (0x05B3), "/hatafqamatsnarrowhebrew" },
  { GUINT_TO_POINTER (0x05B3), "/hatafqamatsquarterhebrew" },
  { GUINT_TO_POINTER (0x05B3), "/hatafqamatswidehebrew" },
  { GUINT_TO_POINTER (0x05B1), "/hatafsegol" },
  { GUINT_TO_POINTER (0x05B1), "/hatafsegol17" },
  { GUINT_TO_POINTER (0x05B1), "/hatafsegol24" },
  { GUINT_TO_POINTER (0x05B1), "/hatafsegol30" },
  { GUINT_TO_POINTER (0x05B1), "/hatafsegolhebrew" },
  { GUINT_TO_POINTER (0x05B1), "/hatafsegolnarrowhebrew" },
  { GUINT_TO_POINTER (0x05B1), "/hatafsegolquarterhebrew" },
  { GUINT_TO_POINTER (0x05B1), "/hatafsegolwidehebrew" },
  { GUINT_TO_POINTER (0x0127), "/hbar" },
  { GUINT_TO_POINTER (0x310F), "/hbopomofo" },
  { GUINT_TO_POINTER (0x1E2B), "/hbrevebelow" },
  { GUINT_TO_POINTER (0x1E29), "/hcedilla" },
  { GUINT_TO_POINTER (0x24D7), "/hcircle" },
  { GUINT_TO_POINTER (0x0125), "/hcircumflex" },
  { GUINT_TO_POINTER (0x1E27), "/hdieresis" },
  { GUINT_TO_POINTER (0x1E23), "/hdotaccent" },
  { GUINT_TO_POINTER (0x1E25), "/hdotbelow" },
  { GUINT_TO_POINTER (0x05D4), "/he" },
  { GUINT_TO_POINTER (0x2665), "/heart" },
  { GUINT_TO_POINTER (0x2665), "/heartsuitblack" },
  { GUINT_TO_POINTER (0x2661), "/heartsuitwhite" },
  { GUINT_TO_POINTER (0xFB34), "/hedagesh" },
  { GUINT_TO_POINTER (0xFB34), "/hedageshhebrew" },
  { GUINT_TO_POINTER (0x06C1), "/hehaltonearabic" },
  { GUINT_TO_POINTER (0x0647), "/heharabic" },
  { GUINT_TO_POINTER (0x05D4), "/hehebrew" },
  { GUINT_TO_POINTER (0xFBA7), "/hehfinalaltonearabic" },
  { GUINT_TO_POINTER (0xFEEA), "/hehfinalalttwoarabic" },
  { GUINT_TO_POINTER (0xFEEA), "/hehfinalarabic" },
  { GUINT_TO_POINTER (0xFBA5), "/hehhamzaabovefinalarabic" },
  { GUINT_TO_POINTER (0xFBA4), "/hehhamzaaboveisolatedarabic" },
  { GUINT_TO_POINTER (0xFBA8), "/hehinitialaltonearabic" },
  { GUINT_TO_POINTER (0xFEEB), "/hehinitialarabic" },
  { GUINT_TO_POINTER (0x3078), "/hehiragana" },
  { GUINT_TO_POINTER (0xFBA9), "/hehmedialaltonearabic" },
  { GUINT_TO_POINTER (0xFEEC), "/hehmedialarabic" },
  { GUINT_TO_POINTER (0x337B), "/heiseierasquare" },
  { GUINT_TO_POINTER (0x30D8), "/hekatakana" },
  { GUINT_TO_POINTER (0xFF8D), "/hekatakanahalfwidth" },
  { GUINT_TO_POINTER (0x3336), "/hekutaarusquare" },
  { GUINT_TO_POINTER (0x0267), "/henghook" },
  { GUINT_TO_POINTER (0x3339), "/herutusquare" },
  { GUINT_TO_POINTER (0x05D7), "/het" },
  { GUINT_TO_POINTER (0x05D7), "/hethebrew" },
  { GUINT_TO_POINTER (0x0266), "/hhook" },
  { GUINT_TO_POINTER (0x02B1), "/hhooksuperior" },
  { GUINT_TO_POINTER (0x327B), "/hieuhacirclekorean" },
  { GUINT_TO_POINTER (0x321B), "/hieuhaparenkorean" },
  { GUINT_TO_POINTER (0x326D), "/hieuhcirclekorean" },
  { GUINT_TO_POINTER (0x314E), "/hieuhkorean" },
  { GUINT_TO_POINTER (0x320D), "/hieuhparenkorean" },
  { GUINT_TO_POINTER (0x3072), "/hihiragana" },
  { GUINT_TO_POINTER (0x30D2), "/hikatakana" },
  { GUINT_TO_POINTER (0xFF8B), "/hikatakanahalfwidth" },
  { GUINT_TO_POINTER (0x05B4), "/hiriq" },
  { GUINT_TO_POINTER (0x05B4), "/hiriq14" },
  { GUINT_TO_POINTER (0x05B4), "/hiriq21" },
  { GUINT_TO_POINTER (0x05B4), "/hiriq2d" },
  { GUINT_TO_POINTER (0x05B4), "/hiriqhebrew" },
  { GUINT_TO_POINTER (0x05B4), "/hiriqnarrowhebrew" },
  { GUINT_TO_POINTER (0x05B4), "/hiriqquarterhebrew" },
  { GUINT_TO_POINTER (0x05B4), "/hiriqwidehebrew" },
  { GUINT_TO_POINTER (0x1E96), "/hlinebelow" },
  { GUINT_TO_POINTER (0xFF48), "/hmonospace" },
  { GUINT_TO_POINTER (0x0570), "/hoarmenian" },
  { GUINT_TO_POINTER (0x0E2B), "/hohipthai" },
  { GUINT_TO_POINTER (0x307B), "/hohiragana" },
  { GUINT_TO_POINTER (0x30DB), "/hokatakana" },
  { GUINT_TO_POINTER (0xFF8E), "/hokatakanahalfwidth" },
  { GUINT_TO_POINTER (0x05B9), "/holam" },
  { GUINT_TO_POINTER (0x05B9), "/holam19" },
  { GUINT_TO_POINTER (0x05B9), "/holam26" },
  { GUINT_TO_POINTER (0x05B9), "/holam32" },
  { GUINT_TO_POINTER (0x05B9), "/holamhebrew" },
  { GUINT_TO_POINTER (0x05B9), "/holamnarrowhebrew" },
  { GUINT_TO_POINTER (0x05B9), "/holamquarterhebrew" },
  { GUINT_TO_POINTER (0x05B9), "/holamwidehebrew" },
  { GUINT_TO_POINTER (0x0E2E), "/honokhukthai" },
  { GUINT_TO_POINTER (0x0309), "/hookabovecomb" },
  { GUINT_TO_POINTER (0x0309), "/hookcmb" },
  { GUINT_TO_POINTER (0x0321), "/hookpalatalizedbelowcmb" },
  { GUINT_TO_POINTER (0x0322), "/hookretroflexbelowcmb" },
  { GUINT_TO_POINTER (0x3342), "/hoonsquare" },
  { GUINT_TO_POINTER (0x03E9), "/horicoptic" },
  { GUINT_TO_POINTER (0x2015), "/horizontalbar" },
  { GUINT_TO_POINTER (0x031B), "/horncmb" },
  { GUINT_TO_POINTER (0x2668), "/hotsprings" },
  { GUINT_TO_POINTER (0x2302), "/house" },
  { GUINT_TO_POINTER (0x24A3), "/hparen" },
  { GUINT_TO_POINTER (0x02B0), "/hsuperior" },
  { GUINT_TO_POINTER (0x0265), "/hturned" },
  { GUINT_TO_POINTER (0x3075), "/huhiragana" },
  { GUINT_TO_POINTER (0x3333), "/huiitosquare" },
  { GUINT_TO_POINTER (0x30D5), "/hukatakana" },
  { GUINT_TO_POINTER (0xFF8C), "/hukatakanahalfwidth" },
  { GUINT_TO_POINTER (0x02DD), "/hungarumlaut" },
  { GUINT_TO_POINTER (0x030B), "/hungarumlautcmb" },
  { GUINT_TO_POINTER (0x0195), "/hv" },
  { GUINT_TO_POINTER (0x002D), "/hyphen" },
  { GUINT_TO_POINTER (0xF6E5), "/hypheninferior" },
  { GUINT_TO_POINTER (0xFF0D), "/hyphenmonospace" },
  { GUINT_TO_POINTER (0xFE63), "/hyphensmall" },
  { GUINT_TO_POINTER (0xF6E6), "/hyphensuperior" },
  { GUINT_TO_POINTER (0x2010), "/hyphentwo" },
  { GUINT_TO_POINTER (0x0069), "/i" },
  { GUINT_TO_POINTER (0x00ED), "/iacute" },
  { GUINT_TO_POINTER (0x044F), "/iacyrillic" },
  { GUINT_TO_POINTER (0x0987), "/ibengali" },
  { GUINT_TO_POINTER (0x3127), "/ibopomofo" },
  { GUINT_TO_POINTER (0x012D), "/ibreve" },
  { GUINT_TO_POINTER (0x01D0), "/icaron" },
  { GUINT_TO_POINTER (0x24D8), "/icircle" },
  { GUINT_TO_POINTER (0x00EE), "/icircumflex" },
  { GUINT_TO_POINTER (0x0456), "/icyrillic" },
  { GUINT_TO_POINTER (0x0209), "/idblgrave" },
  { GUINT_TO_POINTER (0x328F), "/ideographearthcircle" },
  { GUINT_TO_POINTER (0x328B), "/ideographfirecircle" },
  { GUINT_TO_POINTER (0x323F), "/ideographicallianceparen" },
  { GUINT_TO_POINTER (0x323A), "/ideographiccallparen" },
  { GUINT_TO_POINTER (0x32A5), "/ideographiccentrecircle" },
  { GUINT_TO_POINTER (0x3006), "/ideographicclose" },
  { GUINT_TO_POINTER (0x3001), "/ideographiccomma" },
  { GUINT_TO_POINTER (0xFF64), "/ideographiccommaleft" },
  { GUINT_TO_POINTER (0x3237), "/ideographiccongratulationparen" },
  { GUINT_TO_POINTER (0x32A3), "/ideographiccorrectcircle" },
  { GUINT_TO_POINTER (0x322F), "/ideographicearthparen" },
  { GUINT_TO_POINTER (0x323D), "/ideographicenterpriseparen" },
  { GUINT_TO_POINTER (0x329D), "/ideographicexcellentcircle" },
  { GUINT_TO_POINTER (0x3240), "/ideographicfestivalparen" },
  { GUINT_TO_POINTER (0x3296), "/ideographicfinancialcircle" },
  { GUINT_TO_POINTER (0x3236), "/ideographicfinancialparen" },
  { GUINT_TO_POINTER (0x322B), "/ideographicfireparen" },
  { GUINT_TO_POINTER (0x3232), "/ideographichaveparen" },
  { GUINT_TO_POINTER (0x32A4), "/ideographichighcircle" },
  { GUINT_TO_POINTER (0x3005), "/ideographiciterationmark" },
  { GUINT_TO_POINTER (0x3298), "/ideographiclaborcircle" },
  { GUINT_TO_POINTER (0x3238), "/ideographiclaborparen" },
  { GUINT_TO_POINTER (0x32A7), "/ideographicleftcircle" },
  { GUINT_TO_POINTER (0x32A6), "/ideographiclowcircle" },
  { GUINT_TO_POINTER (0x32A9), "/ideographicmedicinecircle" },
  { GUINT_TO_POINTER (0x322E), "/ideographicmetalparen" },
  { GUINT_TO_POINTER (0x322A), "/ideographicmoonparen" },
  { GUINT_TO_POINTER (0x3234), "/ideographicnameparen" },
  { GUINT_TO_POINTER (0x3002), "/ideographicperiod" },
  { GUINT_TO_POINTER (0x329E), "/ideographicprintcircle" },
  { GUINT_TO_POINTER (0x3243), "/ideographicreachparen" },
  { GUINT_TO_POINTER (0x3239), "/ideographicrepresentparen" },
  { GUINT_TO_POINTER (0x323E), "/ideographicresourceparen" },
  { GUINT_TO_POINTER (0x32A8), "/ideographicrightcircle" },
  { GUINT_TO_POINTER (0x3299), "/ideographicsecretcircle" },
  { GUINT_TO_POINTER (0x3242), "/ideographicselfparen" },
  { GUINT_TO_POINTER (0x3233), "/ideographicsocietyparen" },
  { GUINT_TO_POINTER (0x3000), "/ideographicspace" },
  { GUINT_TO_POINTER (0x3235), "/ideographicspecialparen" },
  { GUINT_TO_POINTER (0x3231), "/ideographicstockparen" },
  { GUINT_TO_POINTER (0x323B), "/ideographicstudyparen" },
  { GUINT_TO_POINTER (0x3230), "/ideographicsunparen" },
  { GUINT_TO_POINTER (0x323C), "/ideographicsuperviseparen" },
  { GUINT_TO_POINTER (0x322C), "/ideographicwaterparen" },
  { GUINT_TO_POINTER (0x322D), "/ideographicwoodparen" },
  { GUINT_TO_POINTER (0x3007), "/ideographiczero" },
  { GUINT_TO_POINTER (0x328E), "/ideographmetalcircle" },
  { GUINT_TO_POINTER (0x328A), "/ideographmooncircle" },
  { GUINT_TO_POINTER (0x3294), "/ideographnamecircle" },
  { GUINT_TO_POINTER (0x3290), "/ideographsuncircle" },
  { GUINT_TO_POINTER (0x328C), "/ideographwatercircle" },
  { GUINT_TO_POINTER (0x328D), "/ideographwoodcircle" },
  { GUINT_TO_POINTER (0x0907), "/ideva" },
  { GUINT_TO_POINTER (0x00EF), "/idieresis" },
  { GUINT_TO_POINTER (0x1E2F), "/idieresisacute" },
  { GUINT_TO_POINTER (0x04E5), "/idieresiscyrillic" },
  { GUINT_TO_POINTER (0x1ECB), "/idotbelow" },
  { GUINT_TO_POINTER (0x04D7), "/iebrevecyrillic" },
  { GUINT_TO_POINTER (0x0435), "/iecyrillic" },
  { GUINT_TO_POINTER (0x3275), "/ieungacirclekorean" },
  { GUINT_TO_POINTER (0x3215), "/ieungaparenkorean" },
  { GUINT_TO_POINTER (0x3267), "/ieungcirclekorean" },
  { GUINT_TO_POINTER (0x3147), "/ieungkorean" },
  { GUINT_TO_POINTER (0x3207), "/ieungparenkorean" },
  { GUINT_TO_POINTER (0x00EC), "/igrave" },
  { GUINT_TO_POINTER (0x0A87), "/igujarati" },
  { GUINT_TO_POINTER (0x0A07), "/igurmukhi" },
  { GUINT_TO_POINTER (0x3044), "/ihiragana" },
  { GUINT_TO_POINTER (0x1EC9), "/ihookabove" },
  { GUINT_TO_POINTER (0x0988), "/iibengali" },
  { GUINT_TO_POINTER (0x0438), "/iicyrillic" },
  { GUINT_TO_POINTER (0x0908), "/iideva" },
  { GUINT_TO_POINTER (0x0A88), "/iigujarati" },
  { GUINT_TO_POINTER (0x0A08), "/iigurmukhi" },
  { GUINT_TO_POINTER (0x0A40), "/iimatragurmukhi" },
  { GUINT_TO_POINTER (0x020B), "/iinvertedbreve" },
  { GUINT_TO_POINTER (0x0439), "/iishortcyrillic" },
  { GUINT_TO_POINTER (0x09C0), "/iivowelsignbengali" },
  { GUINT_TO_POINTER (0x0940), "/iivowelsigndeva" },
  { GUINT_TO_POINTER (0x0AC0), "/iivowelsigngujarati" },
  { GUINT_TO_POINTER (0x0133), "/ij" },
  { GUINT_TO_POINTER (0x30A4), "/ikatakana" },
  { GUINT_TO_POINTER (0xFF72), "/ikatakanahalfwidth" },
  { GUINT_TO_POINTER (0x3163), "/ikorean" },
  { GUINT_TO_POINTER (0x02DC), "/ilde" },
  { GUINT_TO_POINTER (0x05AC), "/iluyhebrew" },
  { GUINT_TO_POINTER (0x012B), "/imacron" },
  { GUINT_TO_POINTER (0x04E3), "/imacroncyrillic" },
  { GUINT_TO_POINTER (0x2253), "/imageorapproximatelyequal" },
  { GUINT_TO_POINTER (0x0A3F), "/imatragurmukhi" },
  { GUINT_TO_POINTER (0xFF49), "/imonospace" },
  { GUINT_TO_POINTER (0x2206), "/increment" },
  { GUINT_TO_POINTER (0x221E), "/infinity" },
  { GUINT_TO_POINTER (0x056B), "/iniarmenian" },
  { GUINT_TO_POINTER (0x222B), "/integral" },
  { GUINT_TO_POINTER (0x2321), "/integralbottom" },
  { GUINT_TO_POINTER (0x2321), "/integralbt" },
  { GUINT_TO_POINTER (0xF8F5), "/integralex" },
  { GUINT_TO_POINTER (0x2320), "/integraltop" },
  { GUINT_TO_POINTER (0x2320), "/integraltp" },
  { GUINT_TO_POINTER (0x2229), "/intersection" },
  { GUINT_TO_POINTER (0x3305), "/intisquare" },
  { GUINT_TO_POINTER (0x25D8), "/invbullet" },
  { GUINT_TO_POINTER (0x25D9), "/invcircle" },
  { GUINT_TO_POINTER (0x263B), "/invsmileface" },
  { GUINT_TO_POINTER (0x0451), "/iocyrillic" },
  { GUINT_TO_POINTER (0x012F), "/iogonek" },
  { GUINT_TO_POINTER (0x03B9), "/iota" },
  { GUINT_TO_POINTER (0x03CA), "/iotadieresis" },
  { GUINT_TO_POINTER (0x0390), "/iotadieresistonos" },
  { GUINT_TO_POINTER (0x0269), "/iotalatin" },
  { GUINT_TO_POINTER (0x03AF), "/iotatonos" },
  { GUINT_TO_POINTER (0x24A4), "/iparen" },
  { GUINT_TO_POINTER (0x0A72), "/irigurmukhi" },
  { GUINT_TO_POINTER (0x3043), "/ismallhiragana" },
  { GUINT_TO_POINTER (0x30A3), "/ismallkatakana" },
  { GUINT_TO_POINTER (0xFF68), "/ismallkatakanahalfwidth" },
  { GUINT_TO_POINTER (0x09FA), "/issharbengali" },
  { GUINT_TO_POINTER (0x0268), "/istroke" },
  { GUINT_TO_POINTER (0xF6ED), "/isuperior" },
  { GUINT_TO_POINTER (0x309D), "/iterationhiragana" },
  { GUINT_TO_POINTER (0x30FD), "/iterationkatakana" },
  { GUINT_TO_POINTER (0x0129), "/itilde" },
  { GUINT_TO_POINTER (0x1E2D), "/itildebelow" },
  { GUINT_TO_POINTER (0x3129), "/iubopomofo" },
  { GUINT_TO_POINTER (0x044E), "/iucyrillic" },
  { GUINT_TO_POINTER (0x09BF), "/ivowelsignbengali" },
  { GUINT_TO_POINTER (0x093F), "/ivowelsigndeva" },
  { GUINT_TO_POINTER (0x0ABF), "/ivowelsigngujarati" },
  { GUINT_TO_POINTER (0x0475), "/izhitsacyrillic" },
  { GUINT_TO_POINTER (0x0477), "/izhitsadblgravecyrillic" },
  { GUINT_TO_POINTER (0x006A), "/j" },
  { GUINT_TO_POINTER (0x0571), "/jaarmenian" },
  { GUINT_TO_POINTER (0x099C), "/jabengali" },
  { GUINT_TO_POINTER (0x091C), "/jadeva" },
  { GUINT_TO_POINTER (0x0A9C), "/jagujarati" },
  { GUINT_TO_POINTER (0x0A1C), "/jagurmukhi" },
  { GUINT_TO_POINTER (0x3110), "/jbopomofo" },
  { GUINT_TO_POINTER (0x01F0), "/jcaron" },
  { GUINT_TO_POINTER (0x24D9), "/jcircle" },
  { GUINT_TO_POINTER (0x0135), "/jcircumflex" },
  { GUINT_TO_POINTER (0x029D), "/jcrossedtail" },
  { GUINT_TO_POINTER (0x025F), "/jdotlessstroke" },
  { GUINT_TO_POINTER (0x0458), "/jecyrillic" },
  { GUINT_TO_POINTER (0x062C), "/jeemarabic" },
  { GUINT_TO_POINTER (0xFE9E), "/jeemfinalarabic" },
  { GUINT_TO_POINTER (0xFE9F), "/jeeminitialarabic" },
  { GUINT_TO_POINTER (0xFEA0), "/jeemmedialarabic" },
  { GUINT_TO_POINTER (0x0698), "/jeharabic" },
  { GUINT_TO_POINTER (0xFB8B), "/jehfinalarabic" },
  { GUINT_TO_POINTER (0x099D), "/jhabengali" },
  { GUINT_TO_POINTER (0x091D), "/jhadeva" },
  { GUINT_TO_POINTER (0x0A9D), "/jhagujarati" },
  { GUINT_TO_POINTER (0x0A1D), "/jhagurmukhi" },
  { GUINT_TO_POINTER (0x057B), "/jheharmenian" },
  { GUINT_TO_POINTER (0x3004), "/jis" },
  { GUINT_TO_POINTER (0xFF4A), "/jmonospace" },
  { GUINT_TO_POINTER (0x24A5), "/jparen" },
  { GUINT_TO_POINTER (0x02B2), "/jsuperior" },
  { GUINT_TO_POINTER (0x006B), "/k" },
  { GUINT_TO_POINTER (0x04A1), "/kabashkircyrillic" },
  { GUINT_TO_POINTER (0x0995), "/kabengali" },
  { GUINT_TO_POINTER (0x1E31), "/kacute" },
  { GUINT_TO_POINTER (0x043A), "/kacyrillic" },
  { GUINT_TO_POINTER (0x049B), "/kadescendercyrillic" },
  { GUINT_TO_POINTER (0x0915), "/kadeva" },
  { GUINT_TO_POINTER (0x05DB), "/kaf" },
  { GUINT_TO_POINTER (0x0643), "/kafarabic" },
  { GUINT_TO_POINTER (0xFB3B), "/kafdagesh" },
  { GUINT_TO_POINTER (0xFB3B), "/kafdageshhebrew" },
  { GUINT_TO_POINTER (0xFEDA), "/kaffinalarabic" },
  { GUINT_TO_POINTER (0x05DB), "/kafhebrew" },
  { GUINT_TO_POINTER (0xFEDB), "/kafinitialarabic" },
  { GUINT_TO_POINTER (0xFEDC), "/kafmedialarabic" },
  { GUINT_TO_POINTER (0xFB4D), "/kafrafehebrew" },
  { GUINT_TO_POINTER (0x0A95), "/kagujarati" },
  { GUINT_TO_POINTER (0x0A15), "/kagurmukhi" },
  { GUINT_TO_POINTER (0x304B), "/kahiragana" },
  { GUINT_TO_POINTER (0x04C4), "/kahookcyrillic" },
  { GUINT_TO_POINTER (0x30AB), "/kakatakana" },
  { GUINT_TO_POINTER (0xFF76), "/kakatakanahalfwidth" },
  { GUINT_TO_POINTER (0x03BA), "/kappa" },
  { GUINT_TO_POINTER (0x03F0), "/kappasymbolgreek" },
  { GUINT_TO_POINTER (0x3171), "/kapyeounmieumkorean" },
  { GUINT_TO_POINTER (0x3184), "/kapyeounphieuphkorean" },
  { GUINT_TO_POINTER (0x3178), "/kapyeounpieupkorean" },
  { GUINT_TO_POINTER (0x3179), "/kapyeounssangpieupkorean" },
  { GUINT_TO_POINTER (0x330D), "/karoriisquare" },
  { GUINT_TO_POINTER (0x0640), "/kashidaautoarabic" },
  { GUINT_TO_POINTER (0x0640), "/kashidaautonosidebearingarabic" },
  { GUINT_TO_POINTER (0x30F5), "/kasmallkatakana" },
  { GUINT_TO_POINTER (0x3384), "/kasquare" },
  { GUINT_TO_POINTER (0x0650), "/kasraarabic" },
  { GUINT_TO_POINTER (0x064D), "/kasratanarabic" },
  { GUINT_TO_POINTER (0x049F), "/kastrokecyrillic" },
  { GUINT_TO_POINTER (0xFF70), "/katahiraprolongmarkhalfwidth" },
  { GUINT_TO_POINTER (0x049D), "/kaverticalstrokecyrillic" },
  { GUINT_TO_POINTER (0x310E), "/kbopomofo" },
  { GUINT_TO_POINTER (0x3389), "/kcalsquare" },
  { GUINT_TO_POINTER (0x01E9), "/kcaron" },
  { GUINT_TO_POINTER (0x0137), "/kcedilla" },
  { GUINT_TO_POINTER (0x24DA), "/kcircle" },
  { GUINT_TO_POINTER (0x0137), "/kcommaaccent" },
  { GUINT_TO_POINTER (0x1E33), "/kdotbelow" },
  { GUINT_TO_POINTER (0x0584), "/keharmenian" },
  { GUINT_TO_POINTER (0x3051), "/kehiragana" },
  { GUINT_TO_POINTER (0x30B1), "/kekatakana" },
  { GUINT_TO_POINTER (0xFF79), "/kekatakanahalfwidth" },
  { GUINT_TO_POINTER (0x056F), "/kenarmenian" },
  { GUINT_TO_POINTER (0x30F6), "/kesmallkatakana" },
  { GUINT_TO_POINTER (0x0138), "/kgreenlandic" },
  { GUINT_TO_POINTER (0x0996), "/khabengali" },
  { GUINT_TO_POINTER (0x0445), "/khacyrillic" },
  { GUINT_TO_POINTER (0x0916), "/khadeva" },
  { GUINT_TO_POINTER (0x0A96), "/khagujarati" },
  { GUINT_TO_POINTER (0x0A16), "/khagurmukhi" },
  { GUINT_TO_POINTER (0x062E), "/khaharabic" },
  { GUINT_TO_POINTER (0xFEA6), "/khahfinalarabic" },
  { GUINT_TO_POINTER (0xFEA7), "/khahinitialarabic" },
  { GUINT_TO_POINTER (0xFEA8), "/khahmedialarabic" },
  { GUINT_TO_POINTER (0x03E7), "/kheicoptic" },
  { GUINT_TO_POINTER (0x0959), "/khhadeva" },
  { GUINT_TO_POINTER (0x0A59), "/khhagurmukhi" },
  { GUINT_TO_POINTER (0x3278), "/khieukhacirclekorean" },
  { GUINT_TO_POINTER (0x3218), "/khieukhaparenkorean" },
  { GUINT_TO_POINTER (0x326A), "/khieukhcirclekorean" },
  { GUINT_TO_POINTER (0x314B), "/khieukhkorean" },
  { GUINT_TO_POINTER (0x320A), "/khieukhparenkorean" },
  { GUINT_TO_POINTER (0x0E02), "/khokhaithai" },
  { GUINT_TO_POINTER (0x0E05), "/khokhonthai" },
  { GUINT_TO_POINTER (0x0E03), "/khokhuatthai" },
  { GUINT_TO_POINTER (0x0E04), "/khokhwaithai" },
  { GUINT_TO_POINTER (0x0E5B), "/khomutthai" },
  { GUINT_TO_POINTER (0x0199), "/khook" },
  { GUINT_TO_POINTER (0x0E06), "/khorakhangthai" },
  { GUINT_TO_POINTER (0x3391), "/khzsquare" },
  { GUINT_TO_POINTER (0x304D), "/kihiragana" },
  { GUINT_TO_POINTER (0x30AD), "/kikatakana" },
  { GUINT_TO_POINTER (0xFF77), "/kikatakanahalfwidth" },
  { GUINT_TO_POINTER (0x3315), "/kiroguramusquare" },
  { GUINT_TO_POINTER (0x3316), "/kiromeetorusquare" },
  { GUINT_TO_POINTER (0x3314), "/kirosquare" },
  { GUINT_TO_POINTER (0x326E), "/kiyeokacirclekorean" },
  { GUINT_TO_POINTER (0x320E), "/kiyeokaparenkorean" },
  { GUINT_TO_POINTER (0x3260), "/kiyeokcirclekorean" },
  { GUINT_TO_POINTER (0x3131), "/kiyeokkorean" },
  { GUINT_TO_POINTER (0x3200), "/kiyeokparenkorean" },
  { GUINT_TO_POINTER (0x3133), "/kiyeoksioskorean" },
  { GUINT_TO_POINTER (0x045C), "/kjecyrillic" },
  { GUINT_TO_POINTER (0x1E35), "/klinebelow" },
  { GUINT_TO_POINTER (0x3398), "/klsquare" },
  { GUINT_TO_POINTER (0x33A6), "/kmcubedsquare" },
  { GUINT_TO_POINTER (0xFF4B), "/kmonospace" },
  { GUINT_TO_POINTER (0x33A2), "/kmsquaredsquare" },
  { GUINT_TO_POINTER (0x3053), "/kohiragana" },
  { GUINT_TO_POINTER (0x33C0), "/kohmsquare" },
  { GUINT_TO_POINTER (0x0E01), "/kokaithai" },
  { GUINT_TO_POINTER (0x30B3), "/kokatakana" },
  { GUINT_TO_POINTER (0xFF7A), "/kokatakanahalfwidth" },
  { GUINT_TO_POINTER (0x331E), "/kooposquare" },
  { GUINT_TO_POINTER (0x0481), "/koppacyrillic" },
  { GUINT_TO_POINTER (0x327F), "/koreanstandardsymbol" },
  { GUINT_TO_POINTER (0x0343), "/koroniscmb" },
  { GUINT_TO_POINTER (0x24A6), "/kparen" },
  { GUINT_TO_POINTER (0x33AA), "/kpasquare" },
  { GUINT_TO_POINTER (0x046F), "/ksicyrillic" },
  { GUINT_TO_POINTER (0x33CF), "/ktsquare" },
  { GUINT_TO_POINTER (0x029E), "/kturned" },
  { GUINT_TO_POINTER (0x304F), "/kuhiragana" },
  { GUINT_TO_POINTER (0x30AF), "/kukatakana" },
  { GUINT_TO_POINTER (0xFF78), "/kukatakanahalfwidth" },
  { GUINT_TO_POINTER (0x33B8), "/kvsquare" },
  { GUINT_TO_POINTER (0x33BE), "/kwsquare" },
  { GUINT_TO_POINTER (0x006C), "/l" },
  { GUINT_TO_POINTER (0x09B2), "/labengali" },
  { GUINT_TO_POINTER (0x013A), "/lacute" },
  { GUINT_TO_POINTER (0x0932), "/ladeva" },
  { GUINT_TO_POINTER (0x0AB2), "/lagujarati" },
  { GUINT_TO_POINTER (0x0A32), "/lagurmukhi" },
  { GUINT_TO_POINTER (0x0E45), "/lakkhangyaothai" },
  { GUINT_TO_POINTER (0xFEFC), "/lamaleffinalarabic" },
  { GUINT_TO_POINTER (0xFEF8), "/lamalefhamzaabovefinalarabic" },
  { GUINT_TO_POINTER (0xFEF7), "/lamalefhamzaaboveisolatedarabic" },
  { GUINT_TO_POINTER (0xFEFA), "/lamalefhamzabelowfinalarabic" },
  { GUINT_TO_POINTER (0xFEF9), "/lamalefhamzabelowisolatedarabic" },
  { GUINT_TO_POINTER (0xFEFB), "/lamalefisolatedarabic" },
  { GUINT_TO_POINTER (0xFEF6), "/lamalefmaddaabovefinalarabic" },
  { GUINT_TO_POINTER (0xFEF5), "/lamalefmaddaaboveisolatedarabic" },
  { GUINT_TO_POINTER (0x0644), "/lamarabic" },
  { GUINT_TO_POINTER (0x03BB), "/lambda" },
  { GUINT_TO_POINTER (0x019B), "/lambdastroke" },
  { GUINT_TO_POINTER (0x05DC), "/lamed" },
  { GUINT_TO_POINTER (0xFB3C), "/lameddagesh" },
  { GUINT_TO_POINTER (0xFB3C), "/lameddageshhebrew" },
  { GUINT_TO_POINTER (0x05DC), "/lamedhebrew" },
  { GUINT_TO_POINTER (0x05DC), "/lamedholam" },
  { GUINT_TO_POINTER (0x05B9), "/lamedholam" },
  { GUINT_TO_POINTER (0x05DC), "/lamedholamdagesh" },
  { GUINT_TO_POINTER (0x05B9), "/lamedholamdagesh" },
  { GUINT_TO_POINTER (0x05BC), "/lamedholamdagesh" },
  { GUINT_TO_POINTER (0x05DC), "/lamedholamdageshhebrew" },
  { GUINT_TO_POINTER (0x05B9), "/lamedholamdageshhebrew" },
  { GUINT_TO_POINTER (0x05BC), "/lamedholamdageshhebrew" },
  { GUINT_TO_POINTER (0x05DC), "/lamedholamhebrew" },
  { GUINT_TO_POINTER (0x05B9), "/lamedholamhebrew" },
  { GUINT_TO_POINTER (0xFEDE), "/lamfinalarabic" },
  { GUINT_TO_POINTER (0xFCCA), "/lamhahinitialarabic" },
  { GUINT_TO_POINTER (0xFEDF), "/laminitialarabic" },
  { GUINT_TO_POINTER (0xFCC9), "/lamjeeminitialarabic" },
  { GUINT_TO_POINTER (0xFCCB), "/lamkhahinitialarabic" },
  { GUINT_TO_POINTER (0xFDF2), "/lamlamhehisolatedarabic" },
  { GUINT_TO_POINTER (0xFEE0), "/lammedialarabic" },
  { GUINT_TO_POINTER (0xFD88), "/lammeemhahinitialarabic" },
  { GUINT_TO_POINTER (0xFCCC), "/lammeeminitialarabic" },
  { GUINT_TO_POINTER (0xFEDF), "/lammeemjeeminitialarabic" },
  { GUINT_TO_POINTER (0xFEE4), "/lammeemjeeminitialarabic" },
  { GUINT_TO_POINTER (0xFEA0), "/lammeemjeeminitialarabic" },
  { GUINT_TO_POINTER (0xFEDF), "/lammeemkhahinitialarabic" },
  { GUINT_TO_POINTER (0xFEE4), "/lammeemkhahinitialarabic" },
  { GUINT_TO_POINTER (0xFEA8), "/lammeemkhahinitialarabic" },
  { GUINT_TO_POINTER (0x25EF), "/largecircle" },
  { GUINT_TO_POINTER (0x019A), "/lbar" },
  { GUINT_TO_POINTER (0x026C), "/lbelt" },
  { GUINT_TO_POINTER (0x310C), "/lbopomofo" },
  { GUINT_TO_POINTER (0x013E), "/lcaron" },
  { GUINT_TO_POINTER (0x013C), "/lcedilla" },
  { GUINT_TO_POINTER (0x24DB), "/lcircle" },
  { GUINT_TO_POINTER (0x1E3D), "/lcircumflexbelow" },
  { GUINT_TO_POINTER (0x013C), "/lcommaaccent" },
  { GUINT_TO_POINTER (0x0140), "/ldot" },
  { GUINT_TO_POINTER (0x0140), "/ldotaccent" },
  { GUINT_TO_POINTER (0x1E37), "/ldotbelow" },
  { GUINT_TO_POINTER (0x1E39), "/ldotbelowmacron" },
  { GUINT_TO_POINTER (0x031A), "/leftangleabovecmb" },
  { GUINT_TO_POINTER (0x0318), "/lefttackbelowcmb" },
  { GUINT_TO_POINTER (0x003C), "/less" },
  { GUINT_TO_POINTER (0x2264), "/lessequal" },
  { GUINT_TO_POINTER (0x22DA), "/lessequalorgreater" },
  { GUINT_TO_POINTER (0xFF1C), "/lessmonospace" },
  { GUINT_TO_POINTER (0x2272), "/lessorequivalent" },
  { GUINT_TO_POINTER (0x2276), "/lessorgreater" },
  { GUINT_TO_POINTER (0x2266), "/lessoverequal" },
  { GUINT_TO_POINTER (0xFE64), "/lesssmall" },
  { GUINT_TO_POINTER (0x026E), "/lezh" },
  { GUINT_TO_POINTER (0x258C), "/lfblock" },
  { GUINT_TO_POINTER (0x026D), "/lhookretroflex" },
  { GUINT_TO_POINTER (0x20A4), "/lira" },
  { GUINT_TO_POINTER (0x056C), "/liwnarmenian" },
  { GUINT_TO_POINTER (0x01C9), "/lj" },
  { GUINT_TO_POINTER (0x0459), "/ljecyrillic" },
  { GUINT_TO_POINTER (0xF6C0), "/ll" },
  { GUINT_TO_POINTER (0x0933), "/lladeva" },
  { GUINT_TO_POINTER (0x0AB3), "/llagujarati" },
  { GUINT_TO_POINTER (0x1E3B), "/llinebelow" },
  { GUINT_TO_POINTER (0x0934), "/llladeva" },
  { GUINT_TO_POINTER (0x09E1), "/llvocalicbengali" },
  { GUINT_TO_POINTER (0x0961), "/llvocalicdeva" },
  { GUINT_TO_POINTER (0x09E3), "/llvocalicvowelsignbengali" },
  { GUINT_TO_POINTER (0x0963), "/llvocalicvowelsigndeva" },
  { GUINT_TO_POINTER (0x026B), "/lmiddletilde" },
  { GUINT_TO_POINTER (0xFF4C), "/lmonospace" },
  { GUINT_TO_POINTER (0x33D0), "/lmsquare" },
  { GUINT_TO_POINTER (0x0E2C), "/lochulathai" },
  { GUINT_TO_POINTER (0x2227), "/logicaland" },
  { GUINT_TO_POINTER (0x00AC), "/logicalnot" },
  { GUINT_TO_POINTER (0x2310), "/logicalnotreversed" },
  { GUINT_TO_POINTER (0x2228), "/logicalor" },
  { GUINT_TO_POINTER (0x0E25), "/lolingthai" },
  { GUINT_TO_POINTER (0x017F), "/longs" },
  { GUINT_TO_POINTER (0xFE4E), "/lowlinecenterline" },
  { GUINT_TO_POINTER (0x0332), "/lowlinecmb" },
  { GUINT_TO_POINTER (0xFE4D), "/lowlinedashed" },
  { GUINT_TO_POINTER (0x25CA), "/lozenge" },
  { GUINT_TO_POINTER (0x24A7), "/lparen" },
  { GUINT_TO_POINTER (0x0142), "/lslash" },
  { GUINT_TO_POINTER (0x2113), "/lsquare" },
  { GUINT_TO_POINTER (0xF6EE), "/lsuperior" },
  { GUINT_TO_POINTER (0x2591), "/ltshade" },
  { GUINT_TO_POINTER (0x0E26), "/luthai" },
  { GUINT_TO_POINTER (0x098C), "/lvocalicbengali" },
  { GUINT_TO_POINTER (0x090C), "/lvocalicdeva" },
  { GUINT_TO_POINTER (0x09E2), "/lvocalicvowelsignbengali" },
  { GUINT_TO_POINTER (0x0962), "/lvocalicvowelsigndeva" },
  { GUINT_TO_POINTER (0x33D3), "/lxsquare" },
  { GUINT_TO_POINTER (0x006D), "/m" },
  { GUINT_TO_POINTER (0x09AE), "/mabengali" },
  { GUINT_TO_POINTER (0x00AF), "/macron" },
  { GUINT_TO_POINTER (0x0331), "/macronbelowcmb" },
  { GUINT_TO_POINTER (0x0304), "/macroncmb" },
  { GUINT_TO_POINTER (0x02CD), "/macronlowmod" },
  { GUINT_TO_POINTER (0xFFE3), "/macronmonospace" },
  { GUINT_TO_POINTER (0x1E3F), "/macute" },
  { GUINT_TO_POINTER (0x092E), "/madeva" },
  { GUINT_TO_POINTER (0x0AAE), "/magujarati" },
  { GUINT_TO_POINTER (0x0A2E), "/magurmukhi" },
  { GUINT_TO_POINTER (0x05A4), "/mahapakhhebrew" },
  { GUINT_TO_POINTER (0x05A4), "/mahapakhlefthebrew" },
  { GUINT_TO_POINTER (0x307E), "/mahiragana" },
  { GUINT_TO_POINTER (0xF895), "/maichattawalowleftthai" },
  { GUINT_TO_POINTER (0xF894), "/maichattawalowrightthai" },
  { GUINT_TO_POINTER (0x0E4B), "/maichattawathai" },
  { GUINT_TO_POINTER (0xF893), "/maichattawaupperleftthai" },
  { GUINT_TO_POINTER (0xF88C), "/maieklowleftthai" },
  { GUINT_TO_POINTER (0xF88B), "/maieklowrightthai" },
  { GUINT_TO_POINTER (0x0E48), "/maiekthai" },
  { GUINT_TO_POINTER (0xF88A), "/maiekupperleftthai" },
  { GUINT_TO_POINTER (0xF884), "/maihanakatleftthai" },
  { GUINT_TO_POINTER (0x0E31), "/maihanakatthai" },
  { GUINT_TO_POINTER (0xF889), "/maitaikhuleftthai" },
  { GUINT_TO_POINTER (0x0E47), "/maitaikhuthai" },
  { GUINT_TO_POINTER (0xF88F), "/maitholowleftthai" },
  { GUINT_TO_POINTER (0xF88E), "/maitholowrightthai" },
  { GUINT_TO_POINTER (0x0E49), "/maithothai" },
  { GUINT_TO_POINTER (0xF88D), "/maithoupperleftthai" },
  { GUINT_TO_POINTER (0xF892), "/maitrilowleftthai" },
  { GUINT_TO_POINTER (0xF891), "/maitrilowrightthai" },
  { GUINT_TO_POINTER (0x0E4A), "/maitrithai" },
  { GUINT_TO_POINTER (0xF890), "/maitriupperleftthai" },
  { GUINT_TO_POINTER (0x0E46), "/maiyamokthai" },
  { GUINT_TO_POINTER (0x30DE), "/makatakana" },
  { GUINT_TO_POINTER (0xFF8F), "/makatakanahalfwidth" },
  { GUINT_TO_POINTER (0x2642), "/male" },
  { GUINT_TO_POINTER (0x3347), "/mansyonsquare" },
  { GUINT_TO_POINTER (0x05BE), "/maqafhebrew" },
  { GUINT_TO_POINTER (0x2642), "/mars" },
  { GUINT_TO_POINTER (0x05AF), "/masoracirclehebrew" },
  { GUINT_TO_POINTER (0x3383), "/masquare" },
  { GUINT_TO_POINTER (0x3107), "/mbopomofo" },
  { GUINT_TO_POINTER (0x33D4), "/mbsquare" },
  { GUINT_TO_POINTER (0x24DC), "/mcircle" },
  { GUINT_TO_POINTER (0x33A5), "/mcubedsquare" },
  { GUINT_TO_POINTER (0x1E41), "/mdotaccent" },
  { GUINT_TO_POINTER (0x1E43), "/mdotbelow" },
  { GUINT_TO_POINTER (0x0645), "/meemarabic" },
  { GUINT_TO_POINTER (0xFEE2), "/meemfinalarabic" },
  { GUINT_TO_POINTER (0xFEE3), "/meeminitialarabic" },
  { GUINT_TO_POINTER (0xFEE4), "/meemmedialarabic" },
  { GUINT_TO_POINTER (0xFCD1), "/meemmeeminitialarabic" },
  { GUINT_TO_POINTER (0xFC48), "/meemmeemisolatedarabic" },
  { GUINT_TO_POINTER (0x334D), "/meetorusquare" },
  { GUINT_TO_POINTER (0x3081), "/mehiragana" },
  { GUINT_TO_POINTER (0x337E), "/meizierasquare" },
  { GUINT_TO_POINTER (0x30E1), "/mekatakana" },
  { GUINT_TO_POINTER (0xFF92), "/mekatakanahalfwidth" },
  { GUINT_TO_POINTER (0x05DE), "/mem" },
  { GUINT_TO_POINTER (0xFB3E), "/memdagesh" },
  { GUINT_TO_POINTER (0xFB3E), "/memdageshhebrew" },
  { GUINT_TO_POINTER (0x05DE), "/memhebrew" },
  { GUINT_TO_POINTER (0x0574), "/menarmenian" },
  { GUINT_TO_POINTER (0x05A5), "/merkhahebrew" },
  { GUINT_TO_POINTER (0x05A6), "/merkhakefulahebrew" },
  { GUINT_TO_POINTER (0x05A6), "/merkhakefulalefthebrew" },
  { GUINT_TO_POINTER (0x05A5), "/merkhalefthebrew" },
  { GUINT_TO_POINTER (0x0271), "/mhook" },
  { GUINT_TO_POINTER (0x3392), "/mhzsquare" },
  { GUINT_TO_POINTER (0xFF65), "/middledotkatakanahalfwidth" },
  { GUINT_TO_POINTER (0x00B7), "/middot" },
  { GUINT_TO_POINTER (0x3272), "/mieumacirclekorean" },
  { GUINT_TO_POINTER (0x3212), "/mieumaparenkorean" },
  { GUINT_TO_POINTER (0x3264), "/mieumcirclekorean" },
  { GUINT_TO_POINTER (0x3141), "/mieumkorean" },
  { GUINT_TO_POINTER (0x3170), "/mieumpansioskorean" },
  { GUINT_TO_POINTER (0x3204), "/mieumparenkorean" },
  { GUINT_TO_POINTER (0x316E), "/mieumpieupkorean" },
  { GUINT_TO_POINTER (0x316F), "/mieumsioskorean" },
  { GUINT_TO_POINTER (0x307F), "/mihiragana" },
  { GUINT_TO_POINTER (0x30DF), "/mikatakana" },
  { GUINT_TO_POINTER (0xFF90), "/mikatakanahalfwidth" },
  { GUINT_TO_POINTER (0x2212), "/minus" },
  { GUINT_TO_POINTER (0x0320), "/minusbelowcmb" },
  { GUINT_TO_POINTER (0x2296), "/minuscircle" },
  { GUINT_TO_POINTER (0x02D7), "/minusmod" },
  { GUINT_TO_POINTER (0x2213), "/minusplus" },
  { GUINT_TO_POINTER (0x2032), "/minute" },
  { GUINT_TO_POINTER (0x334A), "/miribaarusquare" },
  { GUINT_TO_POINTER (0x3349), "/mirisquare" },
  { GUINT_TO_POINTER (0x0270), "/mlonglegturned" },
  { GUINT_TO_POINTER (0x3396), "/mlsquare" },
  { GUINT_TO_POINTER (0x33A3), "/mmcubedsquare" },
  { GUINT_TO_POINTER (0xFF4D), "/mmonospace" },
  { GUINT_TO_POINTER (0x339F), "/mmsquaredsquare" },
  { GUINT_TO_POINTER (0x3082), "/mohiragana" },
  { GUINT_TO_POINTER (0x33C1), "/mohmsquare" },
  { GUINT_TO_POINTER (0x30E2), "/mokatakana" },
  { GUINT_TO_POINTER (0xFF93), "/mokatakanahalfwidth" },
  { GUINT_TO_POINTER (0x33D6), "/molsquare" },
  { GUINT_TO_POINTER (0x0E21), "/momathai" },
  { GUINT_TO_POINTER (0x33A7), "/moverssquare" },
  { GUINT_TO_POINTER (0x33A8), "/moverssquaredsquare" },
  { GUINT_TO_POINTER (0x24A8), "/mparen" },
  { GUINT_TO_POINTER (0x33AB), "/mpasquare" },
  { GUINT_TO_POINTER (0x33B3), "/mssquare" },
  { GUINT_TO_POINTER (0xF6EF), "/msuperior" },
  { GUINT_TO_POINTER (0x026F), "/mturned" },
  { GUINT_TO_POINTER (0x00B5), "/mu" },
  { GUINT_TO_POINTER (0x00B5), "/mu1" },
  { GUINT_TO_POINTER (0x3382), "/muasquare" },
  { GUINT_TO_POINTER (0x226B), "/muchgreater" },
  { GUINT_TO_POINTER (0x226A), "/muchless" },
  { GUINT_TO_POINTER (0x338C), "/mufsquare" },
  { GUINT_TO_POINTER (0x03BC), "/mugreek" },
  { GUINT_TO_POINTER (0x338D), "/mugsquare" },
  { GUINT_TO_POINTER (0x3080), "/muhiragana" },
  { GUINT_TO_POINTER (0x30E0), "/mukatakana" },
  { GUINT_TO_POINTER (0xFF91), "/mukatakanahalfwidth" },
  { GUINT_TO_POINTER (0x3395), "/mulsquare" },
  { GUINT_TO_POINTER (0x00D7), "/multiply" },
  { GUINT_TO_POINTER (0x339B), "/mumsquare" },
  { GUINT_TO_POINTER (0x05A3), "/munahhebrew" },
  { GUINT_TO_POINTER (0x05A3), "/munahlefthebrew" },
  { GUINT_TO_POINTER (0x266A), "/musicalnote" },
  { GUINT_TO_POINTER (0x266B), "/musicalnotedbl" },
  { GUINT_TO_POINTER (0x266D), "/musicflatsign" },
  { GUINT_TO_POINTER (0x266F), "/musicsharpsign" },
  { GUINT_TO_POINTER (0x33B2), "/mussquare" },
  { GUINT_TO_POINTER (0x33B6), "/muvsquare" },
  { GUINT_TO_POINTER (0x33BC), "/muwsquare" },
  { GUINT_TO_POINTER (0x33B9), "/mvmegasquare" },
  { GUINT_TO_POINTER (0x33B7), "/mvsquare" },
  { GUINT_TO_POINTER (0x33BF), "/mwmegasquare" },
  { GUINT_TO_POINTER (0x33BD), "/mwsquare" },
  { GUINT_TO_POINTER (0x006E), "/n" },
  { GUINT_TO_POINTER (0x09A8), "/nabengali" },
  { GUINT_TO_POINTER (0x2207), "/nabla" },
  { GUINT_TO_POINTER (0x0144), "/nacute" },
  { GUINT_TO_POINTER (0x0928), "/nadeva" },
  { GUINT_TO_POINTER (0x0AA8), "/nagujarati" },
  { GUINT_TO_POINTER (0x0A28), "/nagurmukhi" },
  { GUINT_TO_POINTER (0x306A), "/nahiragana" },
  { GUINT_TO_POINTER (0x30CA), "/nakatakana" },
  { GUINT_TO_POINTER (0xFF85), "/nakatakanahalfwidth" },
  { GUINT_TO_POINTER (0x0149), "/napostrophe" },
  { GUINT_TO_POINTER (0x3381), "/nasquare" },
  { GUINT_TO_POINTER (0x310B), "/nbopomofo" },
  { GUINT_TO_POINTER (0x00A0), "/nbspace" },
  { GUINT_TO_POINTER (0x0148), "/ncaron" },
  { GUINT_TO_POINTER (0x0146), "/ncedilla" },
  { GUINT_TO_POINTER (0x24DD), "/ncircle" },
  { GUINT_TO_POINTER (0x1E4B), "/ncircumflexbelow" },
  { GUINT_TO_POINTER (0x0146), "/ncommaaccent" },
  { GUINT_TO_POINTER (0x1E45), "/ndotaccent" },
  { GUINT_TO_POINTER (0x1E47), "/ndotbelow" },
  { GUINT_TO_POINTER (0x306D), "/nehiragana" },
  { GUINT_TO_POINTER (0x30CD), "/nekatakana" },
  { GUINT_TO_POINTER (0xFF88), "/nekatakanahalfwidth" },
  { GUINT_TO_POINTER (0x20AA), "/newsheqelsign" },
  { GUINT_TO_POINTER (0x338B), "/nfsquare" },
  { GUINT_TO_POINTER (0x0999), "/ngabengali" },
  { GUINT_TO_POINTER (0x0919), "/ngadeva" },
  { GUINT_TO_POINTER (0x0A99), "/ngagujarati" },
  { GUINT_TO_POINTER (0x0A19), "/ngagurmukhi" },
  { GUINT_TO_POINTER (0x0E07), "/ngonguthai" },
  { GUINT_TO_POINTER (0x3093), "/nhiragana" },
  { GUINT_TO_POINTER (0x0272), "/nhookleft" },
  { GUINT_TO_POINTER (0x0273), "/nhookretroflex" },
  { GUINT_TO_POINTER (0x326F), "/nieunacirclekorean" },
  { GUINT_TO_POINTER (0x320F), "/nieunaparenkorean" },
  { GUINT_TO_POINTER (0x3135), "/nieuncieuckorean" },
  { GUINT_TO_POINTER (0x3261), "/nieuncirclekorean" },
  { GUINT_TO_POINTER (0x3136), "/nieunhieuhkorean" },
  { GUINT_TO_POINTER (0x3134), "/nieunkorean" },
  { GUINT_TO_POINTER (0x3168), "/nieunpansioskorean" },
  { GUINT_TO_POINTER (0x3201), "/nieunparenkorean" },
  { GUINT_TO_POINTER (0x3167), "/nieunsioskorean" },
  { GUINT_TO_POINTER (0x3166), "/nieuntikeutkorean" },
  { GUINT_TO_POINTER (0x306B), "/nihiragana" },
  { GUINT_TO_POINTER (0x30CB), "/nikatakana" },
  { GUINT_TO_POINTER (0xFF86), "/nikatakanahalfwidth" },
  { GUINT_TO_POINTER (0xF899), "/nikhahitleftthai" },
  { GUINT_TO_POINTER (0x0E4D), "/nikhahitthai" },
  { GUINT_TO_POINTER (0x0039), "/nine" },
  { GUINT_TO_POINTER (0x0669), "/ninearabic" },
  { GUINT_TO_POINTER (0x09EF), "/ninebengali" },
  { GUINT_TO_POINTER (0x2468), "/ninecircle" },
  { GUINT_TO_POINTER (0x2792), "/ninecircleinversesansserif" },
  { GUINT_TO_POINTER (0x096F), "/ninedeva" },
  { GUINT_TO_POINTER (0x0AEF), "/ninegujarati" },
  { GUINT_TO_POINTER (0x0A6F), "/ninegurmukhi" },
  { GUINT_TO_POINTER (0x0669), "/ninehackarabic" },
  { GUINT_TO_POINTER (0x3029), "/ninehangzhou" },
  { GUINT_TO_POINTER (0x3228), "/nineideographicparen" },
  { GUINT_TO_POINTER (0x2089), "/nineinferior" },
  { GUINT_TO_POINTER (0xFF19), "/ninemonospace" },
  { GUINT_TO_POINTER (0xF739), "/nineoldstyle" },
  { GUINT_TO_POINTER (0x247C), "/nineparen" },
  { GUINT_TO_POINTER (0x2490), "/nineperiod" },
  { GUINT_TO_POINTER (0x06F9), "/ninepersian" },
  { GUINT_TO_POINTER (0x2178), "/nineroman" },
  { GUINT_TO_POINTER (0x2079), "/ninesuperior" },
  { GUINT_TO_POINTER (0x2472), "/nineteencircle" },
  { GUINT_TO_POINTER (0x2486), "/nineteenparen" },
  { GUINT_TO_POINTER (0x249A), "/nineteenperiod" },
  { GUINT_TO_POINTER (0x0E59), "/ninethai" },
  { GUINT_TO_POINTER (0x01CC), "/nj" },
  { GUINT_TO_POINTER (0x045A), "/njecyrillic" },
  { GUINT_TO_POINTER (0x30F3), "/nkatakana" },
  { GUINT_TO_POINTER (0xFF9D), "/nkatakanahalfwidth" },
  { GUINT_TO_POINTER (0x019E), "/nlegrightlong" },
  { GUINT_TO_POINTER (0x1E49), "/nlinebelow" },
  { GUINT_TO_POINTER (0xFF4E), "/nmonospace" },
  { GUINT_TO_POINTER (0x339A), "/nmsquare" },
  { GUINT_TO_POINTER (0x09A3), "/nnabengali" },
  { GUINT_TO_POINTER (0x0923), "/nnadeva" },
  { GUINT_TO_POINTER (0x0AA3), "/nnagujarati" },
  { GUINT_TO_POINTER (0x0A23), "/nnagurmukhi" },
  { GUINT_TO_POINTER (0x0929), "/nnnadeva" },
  { GUINT_TO_POINTER (0x306E), "/nohiragana" },
  { GUINT_TO_POINTER (0x30CE), "/nokatakana" },
  { GUINT_TO_POINTER (0xFF89), "/nokatakanahalfwidth" },
  { GUINT_TO_POINTER (0x00A0), "/nonbreakingspace" },
  { GUINT_TO_POINTER (0x0E13), "/nonenthai" },
  { GUINT_TO_POINTER (0x0E19), "/nonuthai" },
  { GUINT_TO_POINTER (0x0646), "/noonarabic" },
  { GUINT_TO_POINTER (0xFEE6), "/noonfinalarabic" },
  { GUINT_TO_POINTER (0x06BA), "/noonghunnaarabic" },
  { GUINT_TO_POINTER (0xFB9F), "/noonghunnafinalarabic" },
  { GUINT_TO_POINTER (0xFEE7), "/noonhehinitialarabic" },
  { GUINT_TO_POINTER (0xFEEC), "/noonhehinitialarabic" },
  { GUINT_TO_POINTER (0xFEE7), "/nooninitialarabic" },
  { GUINT_TO_POINTER (0xFCD2), "/noonjeeminitialarabic" },
  { GUINT_TO_POINTER (0xFC4B), "/noonjeemisolatedarabic" },
  { GUINT_TO_POINTER (0xFEE8), "/noonmedialarabic" },
  { GUINT_TO_POINTER (0xFCD5), "/noonmeeminitialarabic" },
  { GUINT_TO_POINTER (0xFC4E), "/noonmeemisolatedarabic" },
  { GUINT_TO_POINTER (0xFC8D), "/noonnoonfinalarabic" },
  { GUINT_TO_POINTER (0x220C), "/notcontains" },
  { GUINT_TO_POINTER (0x2209), "/notelement" },
  { GUINT_TO_POINTER (0x2209), "/notelementof" },
  { GUINT_TO_POINTER (0x2260), "/notequal" },
  { GUINT_TO_POINTER (0x226F), "/notgreater" },
  { GUINT_TO_POINTER (0x2271), "/notgreaternorequal" },
  { GUINT_TO_POINTER (0x2279), "/notgreaternorless" },
  { GUINT_TO_POINTER (0x2262), "/notidentical" },
  { GUINT_TO_POINTER (0x226E), "/notless" },
  { GUINT_TO_POINTER (0x2270), "/notlessnorequal" },
  { GUINT_TO_POINTER (0x2226), "/notparallel" },
  { GUINT_TO_POINTER (0x2280), "/notprecedes" },
  { GUINT_TO_POINTER (0x2284), "/notsubset" },
  { GUINT_TO_POINTER (0x2281), "/notsucceeds" },
  { GUINT_TO_POINTER (0x2285), "/notsuperset" },
  { GUINT_TO_POINTER (0x0576), "/nowarmenian" },
  { GUINT_TO_POINTER (0x24A9), "/nparen" },
  { GUINT_TO_POINTER (0x33B1), "/nssquare" },
  { GUINT_TO_POINTER (0x207F), "/nsuperior" },
  { GUINT_TO_POINTER (0x00F1), "/ntilde" },
  { GUINT_TO_POINTER (0x03BD), "/nu" },
  { GUINT_TO_POINTER (0x306C), "/nuhiragana" },
  { GUINT_TO_POINTER (0x30CC), "/nukatakana" },
  { GUINT_TO_POINTER (0xFF87), "/nukatakanahalfwidth" },
  { GUINT_TO_POINTER (0x09BC), "/nuktabengali" },
  { GUINT_TO_POINTER (0x093C), "/nuktadeva" },
  { GUINT_TO_POINTER (0x0ABC), "/nuktagujarati" },
  { GUINT_TO_POINTER (0x0A3C), "/nuktagurmukhi" },
  { GUINT_TO_POINTER (0x0023), "/numbersign" },
  { GUINT_TO_POINTER (0xFF03), "/numbersignmonospace" },
  { GUINT_TO_POINTER (0xFE5F), "/numbersignsmall" },
  { GUINT_TO_POINTER (0x0374), "/numeralsigngreek" },
  { GUINT_TO_POINTER (0x0375), "/numeralsignlowergreek" },
  { GUINT_TO_POINTER (0x2116), "/numero" },
  { GUINT_TO_POINTER (0x05E0), "/nun" },
  { GUINT_TO_POINTER (0xFB40), "/nundagesh" },
  { GUINT_TO_POINTER (0xFB40), "/nundageshhebrew" },
  { GUINT_TO_POINTER (0x05E0), "/nunhebrew" },
  { GUINT_TO_POINTER (0x33B5), "/nvsquare" },
  { GUINT_TO_POINTER (0x33BB), "/nwsquare" },
  { GUINT_TO_POINTER (0x099E), "/nyabengali" },
  { GUINT_TO_POINTER (0x091E), "/nyadeva" },
  { GUINT_TO_POINTER (0x0A9E), "/nyagujarati" },
  { GUINT_TO_POINTER (0x0A1E), "/nyagurmukhi" },
  { GUINT_TO_POINTER (0x006F), "/o" },
  { GUINT_TO_POINTER (0x00F3), "/oacute" },
  { GUINT_TO_POINTER (0x0E2D), "/oangthai" },
  { GUINT_TO_POINTER (0x0275), "/obarred" },
  { GUINT_TO_POINTER (0x04E9), "/obarredcyrillic" },
  { GUINT_TO_POINTER (0x04EB), "/obarreddieresiscyrillic" },
  { GUINT_TO_POINTER (0x0993), "/obengali" },
  { GUINT_TO_POINTER (0x311B), "/obopomofo" },
  { GUINT_TO_POINTER (0x014F), "/obreve" },
  { GUINT_TO_POINTER (0x0911), "/ocandradeva" },
  { GUINT_TO_POINTER (0x0A91), "/ocandragujarati" },
  { GUINT_TO_POINTER (0x0949), "/ocandravowelsigndeva" },
  { GUINT_TO_POINTER (0x0AC9), "/ocandravowelsigngujarati" },
  { GUINT_TO_POINTER (0x01D2), "/ocaron" },
  { GUINT_TO_POINTER (0x24DE), "/ocircle" },
  { GUINT_TO_POINTER (0x00F4), "/ocircumflex" },
  { GUINT_TO_POINTER (0x1ED1), "/ocircumflexacute" },
  { GUINT_TO_POINTER (0x1ED9), "/ocircumflexdotbelow" },
  { GUINT_TO_POINTER (0x1ED3), "/ocircumflexgrave" },
  { GUINT_TO_POINTER (0x1ED5), "/ocircumflexhookabove" },
  { GUINT_TO_POINTER (0x1ED7), "/ocircumflextilde" },
  { GUINT_TO_POINTER (0x043E), "/ocyrillic" },
  { GUINT_TO_POINTER (0x0151), "/odblacute" },
  { GUINT_TO_POINTER (0x020D), "/odblgrave" },
  { GUINT_TO_POINTER (0x0913), "/odeva" },
  { GUINT_TO_POINTER (0x00F6), "/odieresis" },
  { GUINT_TO_POINTER (0x04E7), "/odieresiscyrillic" },
  { GUINT_TO_POINTER (0x1ECD), "/odotbelow" },
  { GUINT_TO_POINTER (0x0153), "/oe" },
  { GUINT_TO_POINTER (0x315A), "/oekorean" },
  { GUINT_TO_POINTER (0x02DB), "/ogonek" },
  { GUINT_TO_POINTER (0x0328), "/ogonekcmb" },
  { GUINT_TO_POINTER (0x00F2), "/ograve" },
  { GUINT_TO_POINTER (0x0A93), "/ogujarati" },
  { GUINT_TO_POINTER (0x0585), "/oharmenian" },
  { GUINT_TO_POINTER (0x304A), "/ohiragana" },
  { GUINT_TO_POINTER (0x1ECF), "/ohookabove" },
  { GUINT_TO_POINTER (0x01A1), "/ohorn" },
  { GUINT_TO_POINTER (0x1EDB), "/ohornacute" },
  { GUINT_TO_POINTER (0x1EE3), "/ohorndotbelow" },
  { GUINT_TO_POINTER (0x1EDD), "/ohorngrave" },
  { GUINT_TO_POINTER (0x1EDF), "/ohornhookabove" },
  { GUINT_TO_POINTER (0x1EE1), "/ohorntilde" },
  { GUINT_TO_POINTER (0x0151), "/ohungarumlaut" },
  { GUINT_TO_POINTER (0x01A3), "/oi" },
  { GUINT_TO_POINTER (0x020F), "/oinvertedbreve" },
  { GUINT_TO_POINTER (0x30AA), "/okatakana" },
  { GUINT_TO_POINTER (0xFF75), "/okatakanahalfwidth" },
  { GUINT_TO_POINTER (0x3157), "/okorean" },
  { GUINT_TO_POINTER (0x05AB), "/olehebrew" },
  { GUINT_TO_POINTER (0x014D), "/omacron" },
  { GUINT_TO_POINTER (0x1E53), "/omacronacute" },
  { GUINT_TO_POINTER (0x1E51), "/omacrongrave" },
  { GUINT_TO_POINTER (0x0950), "/omdeva" },
  { GUINT_TO_POINTER (0x03C9), "/omega" },
  { GUINT_TO_POINTER (0x03D6), "/omega1" },
  { GUINT_TO_POINTER (0x0461), "/omegacyrillic" },
  { GUINT_TO_POINTER (0x0277), "/omegalatinclosed" },
  { GUINT_TO_POINTER (0x047B), "/omegaroundcyrillic" },
  { GUINT_TO_POINTER (0x047D), "/omegatitlocyrillic" },
  { GUINT_TO_POINTER (0x03CE), "/omegatonos" },
  { GUINT_TO_POINTER (0x0AD0), "/omgujarati" },
  { GUINT_TO_POINTER (0x03BF), "/omicron" },
  { GUINT_TO_POINTER (0x03CC), "/omicrontonos" },
  { GUINT_TO_POINTER (0xFF4F), "/omonospace" },
  { GUINT_TO_POINTER (0x0031), "/one" },
  { GUINT_TO_POINTER (0x0661), "/onearabic" },
  { GUINT_TO_POINTER (0x09E7), "/onebengali" },
  { GUINT_TO_POINTER (0x2460), "/onecircle" },
  { GUINT_TO_POINTER (0x278A), "/onecircleinversesansserif" },
  { GUINT_TO_POINTER (0x0967), "/onedeva" },
  { GUINT_TO_POINTER (0x2024), "/onedotenleader" },
  { GUINT_TO_POINTER (0x215B), "/oneeighth" },
  { GUINT_TO_POINTER (0xF6DC), "/onefitted" },
  { GUINT_TO_POINTER (0x0AE7), "/onegujarati" },
  { GUINT_TO_POINTER (0x0A67), "/onegurmukhi" },
  { GUINT_TO_POINTER (0x0661), "/onehackarabic" },
  { GUINT_TO_POINTER (0x00BD), "/onehalf" },
  { GUINT_TO_POINTER (0x3021), "/onehangzhou" },
  { GUINT_TO_POINTER (0x3220), "/oneideographicparen" },
  { GUINT_TO_POINTER (0x2081), "/oneinferior" },
  { GUINT_TO_POINTER (0xFF11), "/onemonospace" },
  { GUINT_TO_POINTER (0x09F4), "/onenumeratorbengali" },
  { GUINT_TO_POINTER (0xF731), "/oneoldstyle" },
  { GUINT_TO_POINTER (0x2474), "/oneparen" },
  { GUINT_TO_POINTER (0x2488), "/oneperiod" },
  { GUINT_TO_POINTER (0x06F1), "/onepersian" },
  { GUINT_TO_POINTER (0x00BC), "/onequarter" },
  { GUINT_TO_POINTER (0x2170), "/oneroman" },
  { GUINT_TO_POINTER (0x00B9), "/onesuperior" },
  { GUINT_TO_POINTER (0x0E51), "/onethai" },
  { GUINT_TO_POINTER (0x2153), "/onethird" },
  { GUINT_TO_POINTER (0x01EB), "/oogonek" },
  { GUINT_TO_POINTER (0x01ED), "/oogonekmacron" },
  { GUINT_TO_POINTER (0x0A13), "/oogurmukhi" },
  { GUINT_TO_POINTER (0x0A4B), "/oomatragurmukhi" },
  { GUINT_TO_POINTER (0x0254), "/oopen" },
  { GUINT_TO_POINTER (0x24AA), "/oparen" },
  { GUINT_TO_POINTER (0x25E6), "/openbullet" },
  { GUINT_TO_POINTER (0x2325), "/option" },
  { GUINT_TO_POINTER (0x00AA), "/ordfeminine" },
  { GUINT_TO_POINTER (0x00BA), "/ordmasculine" },
  { GUINT_TO_POINTER (0x221F), "/orthogonal" },
  { GUINT_TO_POINTER (0x0912), "/oshortdeva" },
  { GUINT_TO_POINTER (0x094A), "/oshortvowelsigndeva" },
  { GUINT_TO_POINTER (0x00F8), "/oslash" },
  { GUINT_TO_POINTER (0x01FF), "/oslashacute" },
  { GUINT_TO_POINTER (0x3049), "/osmallhiragana" },
  { GUINT_TO_POINTER (0x30A9), "/osmallkatakana" },
  { GUINT_TO_POINTER (0xFF6B), "/osmallkatakanahalfwidth" },
  { GUINT_TO_POINTER (0x01FF), "/ostrokeacute" },
  { GUINT_TO_POINTER (0xF6F0), "/osuperior" },
  { GUINT_TO_POINTER (0x047F), "/otcyrillic" },
  { GUINT_TO_POINTER (0x00F5), "/otilde" },
  { GUINT_TO_POINTER (0x1E4D), "/otildeacute" },
  { GUINT_TO_POINTER (0x1E4F), "/otildedieresis" },
  { GUINT_TO_POINTER (0x3121), "/oubopomofo" },
  { GUINT_TO_POINTER (0x203E), "/overline" },
  { GUINT_TO_POINTER (0xFE4A), "/overlinecenterline" },
  { GUINT_TO_POINTER (0x0305), "/overlinecmb" },
  { GUINT_TO_POINTER (0xFE49), "/overlinedashed" },
  { GUINT_TO_POINTER (0xFE4C), "/overlinedblwavy" },
  { GUINT_TO_POINTER (0xFE4B), "/overlinewavy" },
  { GUINT_TO_POINTER (0x00AF), "/overscore" },
  { GUINT_TO_POINTER (0x09CB), "/ovowelsignbengali" },
  { GUINT_TO_POINTER (0x094B), "/ovowelsigndeva" },
  { GUINT_TO_POINTER (0x0ACB), "/ovowelsigngujarati" },
  { GUINT_TO_POINTER (0x0070), "/p" },
  { GUINT_TO_POINTER (0x3380), "/paampssquare" },
  { GUINT_TO_POINTER (0x332B), "/paasentosquare" },
  { GUINT_TO_POINTER (0x09AA), "/pabengali" },
  { GUINT_TO_POINTER (0x1E55), "/pacute" },
  { GUINT_TO_POINTER (0x092A), "/padeva" },
  { GUINT_TO_POINTER (0x21DF), "/pagedown" },
  { GUINT_TO_POINTER (0x21DE), "/pageup" },
  { GUINT_TO_POINTER (0x0AAA), "/pagujarati" },
  { GUINT_TO_POINTER (0x0A2A), "/pagurmukhi" },
  { GUINT_TO_POINTER (0x3071), "/pahiragana" },
  { GUINT_TO_POINTER (0x0E2F), "/paiyannoithai" },
  { GUINT_TO_POINTER (0x30D1), "/pakatakana" },
  { GUINT_TO_POINTER (0x0484), "/palatalizationcyrilliccmb" },
  { GUINT_TO_POINTER (0x04C0), "/palochkacyrillic" },
  { GUINT_TO_POINTER (0x317F), "/pansioskorean" },
  { GUINT_TO_POINTER (0x00B6), "/paragraph" },
  { GUINT_TO_POINTER (0x2225), "/parallel" },
  { GUINT_TO_POINTER (0x0028), "/parenleft" },
  { GUINT_TO_POINTER (0xFD3E), "/parenleftaltonearabic" },
  { GUINT_TO_POINTER (0xF8ED), "/parenleftbt" },
  { GUINT_TO_POINTER (0xF8EC), "/parenleftex" },
  { GUINT_TO_POINTER (0x208D), "/parenleftinferior" },
  { GUINT_TO_POINTER (0xFF08), "/parenleftmonospace" },
  { GUINT_TO_POINTER (0xFE59), "/parenleftsmall" },
  { GUINT_TO_POINTER (0x207D), "/parenleftsuperior" },
  { GUINT_TO_POINTER (0xF8EB), "/parenlefttp" },
  { GUINT_TO_POINTER (0xFE35), "/parenleftvertical" },
  { GUINT_TO_POINTER (0x0029), "/parenright" },
  { GUINT_TO_POINTER (0xFD3F), "/parenrightaltonearabic" },
  { GUINT_TO_POINTER (0xF8F8), "/parenrightbt" },
  { GUINT_TO_POINTER (0xF8F7), "/parenrightex" },
  { GUINT_TO_POINTER (0x208E), "/parenrightinferior" },
  { GUINT_TO_POINTER (0xFF09), "/parenrightmonospace" },
  { GUINT_TO_POINTER (0xFE5A), "/parenrightsmall" },
  { GUINT_TO_POINTER (0x207E), "/parenrightsuperior" },
  { GUINT_TO_POINTER (0xF8F6), "/parenrighttp" },
  { GUINT_TO_POINTER (0xFE36), "/parenrightvertical" },
  { GUINT_TO_POINTER (0x2202), "/partialdiff" },
  { GUINT_TO_POINTER (0x05C0), "/paseqhebrew" },
  { GUINT_TO_POINTER (0x0599), "/pashtahebrew" },
  { GUINT_TO_POINTER (0x33A9), "/pasquare" },
  { GUINT_TO_POINTER (0x05B7), "/patah" },
  { GUINT_TO_POINTER (0x05B7), "/patah11" },
  { GUINT_TO_POINTER (0x05B7), "/patah1d" },
  { GUINT_TO_POINTER (0x05B7), "/patah2a" },
  { GUINT_TO_POINTER (0x05B7), "/patahhebrew" },
  { GUINT_TO_POINTER (0x05B7), "/patahnarrowhebrew" },
  { GUINT_TO_POINTER (0x05B7), "/patahquarterhebrew" },
  { GUINT_TO_POINTER (0x05B7), "/patahwidehebrew" },
  { GUINT_TO_POINTER (0x05A1), "/pazerhebrew" },
  { GUINT_TO_POINTER (0x3106), "/pbopomofo" },
  { GUINT_TO_POINTER (0x24DF), "/pcircle" },
  { GUINT_TO_POINTER (0x1E57), "/pdotaccent" },
  { GUINT_TO_POINTER (0x05E4), "/pe" },
  { GUINT_TO_POINTER (0x043F), "/pecyrillic" },
  { GUINT_TO_POINTER (0xFB44), "/pedagesh" },
  { GUINT_TO_POINTER (0xFB44), "/pedageshhebrew" },
  { GUINT_TO_POINTER (0x333B), "/peezisquare" },
  { GUINT_TO_POINTER (0xFB43), "/pefinaldageshhebrew" },
  { GUINT_TO_POINTER (0x067E), "/peharabic" },
  { GUINT_TO_POINTER (0x057A), "/peharmenian" },
  { GUINT_TO_POINTER (0x05E4), "/pehebrew" },
  { GUINT_TO_POINTER (0xFB57), "/pehfinalarabic" },
  { GUINT_TO_POINTER (0xFB58), "/pehinitialarabic" },
  { GUINT_TO_POINTER (0x307A), "/pehiragana" },
  { GUINT_TO_POINTER (0xFB59), "/pehmedialarabic" },
  { GUINT_TO_POINTER (0x30DA), "/pekatakana" },
  { GUINT_TO_POINTER (0x04A7), "/pemiddlehookcyrillic" },
  { GUINT_TO_POINTER (0xFB4E), "/perafehebrew" },
  { GUINT_TO_POINTER (0x0025), "/percent" },
  { GUINT_TO_POINTER (0x066A), "/percentarabic" },
  { GUINT_TO_POINTER (0xFF05), "/percentmonospace" },
  { GUINT_TO_POINTER (0xFE6A), "/percentsmall" },
  { GUINT_TO_POINTER (0x002E), "/period" },
  { GUINT_TO_POINTER (0x0589), "/periodarmenian" },
  { GUINT_TO_POINTER (0x00B7), "/periodcentered" },
  { GUINT_TO_POINTER (0xFF61), "/periodhalfwidth" },
  { GUINT_TO_POINTER (0xF6E7), "/periodinferior" },
  { GUINT_TO_POINTER (0xFF0E), "/periodmonospace" },
  { GUINT_TO_POINTER (0xFE52), "/periodsmall" },
  { GUINT_TO_POINTER (0xF6E8), "/periodsuperior" },
  { GUINT_TO_POINTER (0x0342), "/perispomenigreekcmb" },
  { GUINT_TO_POINTER (0x22A5), "/perpendicular" },
  { GUINT_TO_POINTER (0x2030), "/perthousand" },
  { GUINT_TO_POINTER (0x20A7), "/peseta" },
  { GUINT_TO_POINTER (0x338A), "/pfsquare" },
  { GUINT_TO_POINTER (0x09AB), "/phabengali" },
  { GUINT_TO_POINTER (0x092B), "/phadeva" },
  { GUINT_TO_POINTER (0x0AAB), "/phagujarati" },
  { GUINT_TO_POINTER (0x0A2B), "/phagurmukhi" },
  { GUINT_TO_POINTER (0x03C6), "/phi" },
  { GUINT_TO_POINTER (0x03D5), "/phi1" },
  { GUINT_TO_POINTER (0x327A), "/phieuphacirclekorean" },
  { GUINT_TO_POINTER (0x321A), "/phieuphaparenkorean" },
  { GUINT_TO_POINTER (0x326C), "/phieuphcirclekorean" },
  { GUINT_TO_POINTER (0x314D), "/phieuphkorean" },
  { GUINT_TO_POINTER (0x320C), "/phieuphparenkorean" },
  { GUINT_TO_POINTER (0x0278), "/philatin" },
  { GUINT_TO_POINTER (0x0E3A), "/phinthuthai" },
  { GUINT_TO_POINTER (0x03D5), "/phisymbolgreek" },
  { GUINT_TO_POINTER (0x01A5), "/phook" },
  { GUINT_TO_POINTER (0x0E1E), "/phophanthai" },
  { GUINT_TO_POINTER (0x0E1C), "/phophungthai" },
  { GUINT_TO_POINTER (0x0E20), "/phosamphaothai" },
  { GUINT_TO_POINTER (0x03C0), "/pi" },
  { GUINT_TO_POINTER (0x3273), "/pieupacirclekorean" },
  { GUINT_TO_POINTER (0x3213), "/pieupaparenkorean" },
  { GUINT_TO_POINTER (0x3176), "/pieupcieuckorean" },
  { GUINT_TO_POINTER (0x3265), "/pieupcirclekorean" },
  { GUINT_TO_POINTER (0x3172), "/pieupkiyeokkorean" },
  { GUINT_TO_POINTER (0x3142), "/pieupkorean" },
  { GUINT_TO_POINTER (0x3205), "/pieupparenkorean" },
  { GUINT_TO_POINTER (0x3174), "/pieupsioskiyeokkorean" },
  { GUINT_TO_POINTER (0x3144), "/pieupsioskorean" },
  { GUINT_TO_POINTER (0x3175), "/pieupsiostikeutkorean" },
  { GUINT_TO_POINTER (0x3177), "/pieupthieuthkorean" },
  { GUINT_TO_POINTER (0x3173), "/pieuptikeutkorean" },
  { GUINT_TO_POINTER (0x3074), "/pihiragana" },
  { GUINT_TO_POINTER (0x30D4), "/pikatakana" },
  { GUINT_TO_POINTER (0x03D6), "/pisymbolgreek" },
  { GUINT_TO_POINTER (0x0583), "/piwrarmenian" },
  { GUINT_TO_POINTER (0x002B), "/plus" },
  { GUINT_TO_POINTER (0x031F), "/plusbelowcmb" },
  { GUINT_TO_POINTER (0x2295), "/pluscircle" },
  { GUINT_TO_POINTER (0x00B1), "/plusminus" },
  { GUINT_TO_POINTER (0x02D6), "/plusmod" },
  { GUINT_TO_POINTER (0xFF0B), "/plusmonospace" },
  { GUINT_TO_POINTER (0xFE62), "/plussmall" },
  { GUINT_TO_POINTER (0x207A), "/plussuperior" },
  { GUINT_TO_POINTER (0xFF50), "/pmonospace" },
  { GUINT_TO_POINTER (0x33D8), "/pmsquare" },
  { GUINT_TO_POINTER (0x307D), "/pohiragana" },
  { GUINT_TO_POINTER (0x261F), "/pointingindexdownwhite" },
  { GUINT_TO_POINTER (0x261C), "/pointingindexleftwhite" },
  { GUINT_TO_POINTER (0x261E), "/pointingindexrightwhite" },
  { GUINT_TO_POINTER (0x261D), "/pointingindexupwhite" },
  { GUINT_TO_POINTER (0x30DD), "/pokatakana" },
  { GUINT_TO_POINTER (0x0E1B), "/poplathai" },
  { GUINT_TO_POINTER (0x3012), "/postalmark" },
  { GUINT_TO_POINTER (0x3020), "/postalmarkface" },
  { GUINT_TO_POINTER (0x24AB), "/pparen" },
  { GUINT_TO_POINTER (0x227A), "/precedes" },
  { GUINT_TO_POINTER (0x211E), "/prescription" },
  { GUINT_TO_POINTER (0x02B9), "/primemod" },
  { GUINT_TO_POINTER (0x2035), "/primereversed" },
  { GUINT_TO_POINTER (0x220F), "/product" },
  { GUINT_TO_POINTER (0x2305), "/projective" },
  { GUINT_TO_POINTER (0x30FC), "/prolongedkana" },
  { GUINT_TO_POINTER (0x2318), "/propellor" },
  { GUINT_TO_POINTER (0x2282), "/propersubset" },
  { GUINT_TO_POINTER (0x2283), "/propersuperset" },
  { GUINT_TO_POINTER (0x2237), "/proportion" },
  { GUINT_TO_POINTER (0x221D), "/proportional" },
  { GUINT_TO_POINTER (0x03C8), "/psi" },
  { GUINT_TO_POINTER (0x0471), "/psicyrillic" },
  { GUINT_TO_POINTER (0x0486), "/psilipneumatacyrilliccmb" },
  { GUINT_TO_POINTER (0x33B0), "/pssquare" },
  { GUINT_TO_POINTER (0x3077), "/puhiragana" },
  { GUINT_TO_POINTER (0x30D7), "/pukatakana" },
  { GUINT_TO_POINTER (0x33B4), "/pvsquare" },
  { GUINT_TO_POINTER (0x33BA), "/pwsquare" },
  { GUINT_TO_POINTER (0x0071), "/q" },
  { GUINT_TO_POINTER (0x0958), "/qadeva" },
  { GUINT_TO_POINTER (0x05A8), "/qadmahebrew" },
  { GUINT_TO_POINTER (0x0642), "/qafarabic" },
  { GUINT_TO_POINTER (0xFED6), "/qaffinalarabic" },
  { GUINT_TO_POINTER (0xFED7), "/qafinitialarabic" },
  { GUINT_TO_POINTER (0xFED8), "/qafmedialarabic" },
  { GUINT_TO_POINTER (0x05B8), "/qamats" },
  { GUINT_TO_POINTER (0x05B8), "/qamats10" },
  { GUINT_TO_POINTER (0x05B8), "/qamats1a" },
  { GUINT_TO_POINTER (0x05B8), "/qamats1c" },
  { GUINT_TO_POINTER (0x05B8), "/qamats27" },
  { GUINT_TO_POINTER (0x05B8), "/qamats29" },
  { GUINT_TO_POINTER (0x05B8), "/qamats33" },
  { GUINT_TO_POINTER (0x05B8), "/qamatsde" },
  { GUINT_TO_POINTER (0x05B8), "/qamatshebrew" },
  { GUINT_TO_POINTER (0x05B8), "/qamatsnarrowhebrew" },
  { GUINT_TO_POINTER (0x05B8), "/qamatsqatanhebrew" },
  { GUINT_TO_POINTER (0x05B8), "/qamatsqatannarrowhebrew" },
  { GUINT_TO_POINTER (0x05B8), "/qamatsqatanquarterhebrew" },
  { GUINT_TO_POINTER (0x05B8), "/qamatsqatanwidehebrew" },
  { GUINT_TO_POINTER (0x05B8), "/qamatsquarterhebrew" },
  { GUINT_TO_POINTER (0x05B8), "/qamatswidehebrew" },
  { GUINT_TO_POINTER (0x059F), "/qarneyparahebrew" },
  { GUINT_TO_POINTER (0x3111), "/qbopomofo" },
  { GUINT_TO_POINTER (0x24E0), "/qcircle" },
  { GUINT_TO_POINTER (0x02A0), "/qhook" },
  { GUINT_TO_POINTER (0xFF51), "/qmonospace" },
  { GUINT_TO_POINTER (0x05E7), "/qof" },
  { GUINT_TO_POINTER (0xFB47), "/qofdagesh" },
  { GUINT_TO_POINTER (0xFB47), "/qofdageshhebrew" },
  { GUINT_TO_POINTER (0x05E7), "/qofhatafpatah" },
  { GUINT_TO_POINTER (0x05B2), "/qofhatafpatah" },
  { GUINT_TO_POINTER (0x05E7), "/qofhatafpatahhebrew" },
  { GUINT_TO_POINTER (0x05B2), "/qofhatafpatahhebrew" },
  { GUINT_TO_POINTER (0x05E7), "/qofhatafsegol" },
  { GUINT_TO_POINTER (0x05B1), "/qofhatafsegol" },
  { GUINT_TO_POINTER (0x05E7), "/qofhatafsegolhebrew" },
  { GUINT_TO_POINTER (0x05B1), "/qofhatafsegolhebrew" },
  { GUINT_TO_POINTER (0x05E7), "/qofhebrew" },
  { GUINT_TO_POINTER (0x05E7), "/qofhiriq" },
  { GUINT_TO_POINTER (0x05B4), "/qofhiriq" },
  { GUINT_TO_POINTER (0x05E7), "/qofhiriqhebrew" },
  { GUINT_TO_POINTER (0x05B4), "/qofhiriqhebrew" },
  { GUINT_TO_POINTER (0x05E7), "/qofholam" },
  { GUINT_TO_POINTER (0x05B9), "/qofholam" },
  { GUINT_TO_POINTER (0x05E7), "/qofholamhebrew" },
  { GUINT_TO_POINTER (0x05B9), "/qofholamhebrew" },
  { GUINT_TO_POINTER (0x05E7), "/qofpatah" },
  { GUINT_TO_POINTER (0x05B7), "/qofpatah" },
  { GUINT_TO_POINTER (0x05E7), "/qofpatahhebrew" },
  { GUINT_TO_POINTER (0x05B7), "/qofpatahhebrew" },
  { GUINT_TO_POINTER (0x05E7), "/qofqamats" },
  { GUINT_TO_POINTER (0x05B8), "/qofqamats" },
  { GUINT_TO_POINTER (0x05E7), "/qofqamatshebrew" },
  { GUINT_TO_POINTER (0x05B8), "/qofqamatshebrew" },
  { GUINT_TO_POINTER (0x05E7), "/qofqubuts" },
  { GUINT_TO_POINTER (0x05BB), "/qofqubuts" },
  { GUINT_TO_POINTER (0x05E7), "/qofqubutshebrew" },
  { GUINT_TO_POINTER (0x05BB), "/qofqubutshebrew" },
  { GUINT_TO_POINTER (0x05E7), "/qofsegol" },
  { GUINT_TO_POINTER (0x05B6), "/qofsegol" },
  { GUINT_TO_POINTER (0x05E7), "/qofsegolhebrew" },
  { GUINT_TO_POINTER (0x05B6), "/qofsegolhebrew" },
  { GUINT_TO_POINTER (0x05E7), "/qofsheva" },
  { GUINT_TO_POINTER (0x05B0), "/qofsheva" },
  { GUINT_TO_POINTER (0x05E7), "/qofshevahebrew" },
  { GUINT_TO_POINTER (0x05B0), "/qofshevahebrew" },
  { GUINT_TO_POINTER (0x05E7), "/qoftsere" },
  { GUINT_TO_POINTER (0x05B5), "/qoftsere" },
  { GUINT_TO_POINTER (0x05E7), "/qoftserehebrew" },
  { GUINT_TO_POINTER (0x05B5), "/qoftserehebrew" },
  { GUINT_TO_POINTER (0x24AC), "/qparen" },
  { GUINT_TO_POINTER (0x2669), "/quarternote" },
  { GUINT_TO_POINTER (0x05BB), "/qubuts" },
  { GUINT_TO_POINTER (0x05BB), "/qubuts18" },
  { GUINT_TO_POINTER (0x05BB), "/qubuts25" },
  { GUINT_TO_POINTER (0x05BB), "/qubuts31" },
  { GUINT_TO_POINTER (0x05BB), "/qubutshebrew" },
  { GUINT_TO_POINTER (0x05BB), "/qubutsnarrowhebrew" },
  { GUINT_TO_POINTER (0x05BB), "/qubutsquarterhebrew" },
  { GUINT_TO_POINTER (0x05BB), "/qubutswidehebrew" },
  { GUINT_TO_POINTER (0x003F), "/question" },
  { GUINT_TO_POINTER (0x061F), "/questionarabic" },
  { GUINT_TO_POINTER (0x055E), "/questionarmenian" },
  { GUINT_TO_POINTER (0x00BF), "/questiondown" },
  { GUINT_TO_POINTER (0xF7BF), "/questiondownsmall" },
  { GUINT_TO_POINTER (0x037E), "/questiongreek" },
  { GUINT_TO_POINTER (0xFF1F), "/questionmonospace" },
  { GUINT_TO_POINTER (0xF73F), "/questionsmall" },
  { GUINT_TO_POINTER (0x0022), "/quotedbl" },
  { GUINT_TO_POINTER (0x201E), "/quotedblbase" },
  { GUINT_TO_POINTER (0x201C), "/quotedblleft" },
  { GUINT_TO_POINTER (0xFF02), "/quotedblmonospace" },
  { GUINT_TO_POINTER (0x301E), "/quotedblprime" },
  { GUINT_TO_POINTER (0x301D), "/quotedblprimereversed" },
  { GUINT_TO_POINTER (0x201D), "/quotedblright" },
  { GUINT_TO_POINTER (0x2018), "/quoteleft" },
  { GUINT_TO_POINTER (0x201B), "/quoteleftreversed" },
  { GUINT_TO_POINTER (0x201B), "/quotereversed" },
  { GUINT_TO_POINTER (0x2019), "/quoteright" },
  { GUINT_TO_POINTER (0x0149), "/quoterightn" },
  { GUINT_TO_POINTER (0x201A), "/quotesinglbase" },
  { GUINT_TO_POINTER (0x0027), "/quotesingle" },
  { GUINT_TO_POINTER (0xFF07), "/quotesinglemonospace" },
  { GUINT_TO_POINTER (0x0072), "/r" },
  { GUINT_TO_POINTER (0x057C), "/raarmenian" },
  { GUINT_TO_POINTER (0x09B0), "/rabengali" },
  { GUINT_TO_POINTER (0x0155), "/racute" },
  { GUINT_TO_POINTER (0x0930), "/radeva" },
  { GUINT_TO_POINTER (0x221A), "/radical" },
  { GUINT_TO_POINTER (0xF8E5), "/radicalex" },
  { GUINT_TO_POINTER (0x33AE), "/radoverssquare" },
  { GUINT_TO_POINTER (0x33AF), "/radoverssquaredsquare" },
  { GUINT_TO_POINTER (0x33AD), "/radsquare" },
  { GUINT_TO_POINTER (0x05BF), "/rafe" },
  { GUINT_TO_POINTER (0x05BF), "/rafehebrew" },
  { GUINT_TO_POINTER (0x0AB0), "/ragujarati" },
  { GUINT_TO_POINTER (0x0A30), "/ragurmukhi" },
  { GUINT_TO_POINTER (0x3089), "/rahiragana" },
  { GUINT_TO_POINTER (0x30E9), "/rakatakana" },
  { GUINT_TO_POINTER (0xFF97), "/rakatakanahalfwidth" },
  { GUINT_TO_POINTER (0x09F1), "/ralowerdiagonalbengali" },
  { GUINT_TO_POINTER (0x09F0), "/ramiddlediagonalbengali" },
  { GUINT_TO_POINTER (0x0264), "/ramshorn" },
  { GUINT_TO_POINTER (0x2236), "/ratio" },
  { GUINT_TO_POINTER (0x3116), "/rbopomofo" },
  { GUINT_TO_POINTER (0x0159), "/rcaron" },
  { GUINT_TO_POINTER (0x0157), "/rcedilla" },
  { GUINT_TO_POINTER (0x24E1), "/rcircle" },
  { GUINT_TO_POINTER (0x0157), "/rcommaaccent" },
  { GUINT_TO_POINTER (0x0211), "/rdblgrave" },
  { GUINT_TO_POINTER (0x1E59), "/rdotaccent" },
  { GUINT_TO_POINTER (0x1E5B), "/rdotbelow" },
  { GUINT_TO_POINTER (0x1E5D), "/rdotbelowmacron" },
  { GUINT_TO_POINTER (0x203B), "/referencemark" },
  { GUINT_TO_POINTER (0x2286), "/reflexsubset" },
  { GUINT_TO_POINTER (0x2287), "/reflexsuperset" },
  { GUINT_TO_POINTER (0x00AE), "/registered" },
  { GUINT_TO_POINTER (0xF8E8), "/registersans" },
  { GUINT_TO_POINTER (0xF6DA), "/registerserif" },
  { GUINT_TO_POINTER (0x0631), "/reharabic" },
  { GUINT_TO_POINTER (0x0580), "/reharmenian" },
  { GUINT_TO_POINTER (0xFEAE), "/rehfinalarabic" },
  { GUINT_TO_POINTER (0x308C), "/rehiragana" },
  { GUINT_TO_POINTER (0x0631), "/rehyehaleflamarabic" },
  { GUINT_TO_POINTER (0xFEF3), "/rehyehaleflamarabic" },
  { GUINT_TO_POINTER (0xFE8E), "/rehyehaleflamarabic" },
  { GUINT_TO_POINTER (0x0644), "/rehyehaleflamarabic" },
  { GUINT_TO_POINTER (0x30EC), "/rekatakana" },
  { GUINT_TO_POINTER (0xFF9A), "/rekatakanahalfwidth" },
  { GUINT_TO_POINTER (0x05E8), "/resh" },
  { GUINT_TO_POINTER (0xFB48), "/reshdageshhebrew" },
  { GUINT_TO_POINTER (0x05E8), "/reshhatafpatah" },
  { GUINT_TO_POINTER (0x05B2), "/reshhatafpatah" },
  { GUINT_TO_POINTER (0x05E8), "/reshhatafpatahhebrew" },
  { GUINT_TO_POINTER (0x05B2), "/reshhatafpatahhebrew" },
  { GUINT_TO_POINTER (0x05E8), "/reshhatafsegol" },
  { GUINT_TO_POINTER (0x05B1), "/reshhatafsegol" },
  { GUINT_TO_POINTER (0x05E8), "/reshhatafsegolhebrew" },
  { GUINT_TO_POINTER (0x05B1), "/reshhatafsegolhebrew" },
  { GUINT_TO_POINTER (0x05E8), "/reshhebrew" },
  { GUINT_TO_POINTER (0x05E8), "/reshhiriq" },
  { GUINT_TO_POINTER (0x05B4), "/reshhiriq" },
  { GUINT_TO_POINTER (0x05E8), "/reshhiriqhebrew" },
  { GUINT_TO_POINTER (0x05B4), "/reshhiriqhebrew" },
  { GUINT_TO_POINTER (0x05E8), "/reshholam" },
  { GUINT_TO_POINTER (0x05B9), "/reshholam" },
  { GUINT_TO_POINTER (0x05E8), "/reshholamhebrew" },
  { GUINT_TO_POINTER (0x05B9), "/reshholamhebrew" },
  { GUINT_TO_POINTER (0x05E8), "/reshpatah" },
  { GUINT_TO_POINTER (0x05B7), "/reshpatah" },
  { GUINT_TO_POINTER (0x05E8), "/reshpatahhebrew" },
  { GUINT_TO_POINTER (0x05B7), "/reshpatahhebrew" },
  { GUINT_TO_POINTER (0x05E8), "/reshqamats" },
  { GUINT_TO_POINTER (0x05B8), "/reshqamats" },
  { GUINT_TO_POINTER (0x05E8), "/reshqamatshebrew" },
  { GUINT_TO_POINTER (0x05B8), "/reshqamatshebrew" },
  { GUINT_TO_POINTER (0x05E8), "/reshqubuts" },
  { GUINT_TO_POINTER (0x05BB), "/reshqubuts" },
  { GUINT_TO_POINTER (0x05E8), "/reshqubutshebrew" },
  { GUINT_TO_POINTER (0x05BB), "/reshqubutshebrew" },
  { GUINT_TO_POINTER (0x05E8), "/reshsegol" },
  { GUINT_TO_POINTER (0x05B6), "/reshsegol" },
  { GUINT_TO_POINTER (0x05E8), "/reshsegolhebrew" },
  { GUINT_TO_POINTER (0x05B6), "/reshsegolhebrew" },
  { GUINT_TO_POINTER (0x05E8), "/reshsheva" },
  { GUINT_TO_POINTER (0x05B0), "/reshsheva" },
  { GUINT_TO_POINTER (0x05E8), "/reshshevahebrew" },
  { GUINT_TO_POINTER (0x05B0), "/reshshevahebrew" },
  { GUINT_TO_POINTER (0x05E8), "/reshtsere" },
  { GUINT_TO_POINTER (0x05B5), "/reshtsere" },
  { GUINT_TO_POINTER (0x05E8), "/reshtserehebrew" },
  { GUINT_TO_POINTER (0x05B5), "/reshtserehebrew" },
  { GUINT_TO_POINTER (0x223D), "/reversedtilde" },
  { GUINT_TO_POINTER (0x0597), "/reviahebrew" },
  { GUINT_TO_POINTER (0x0597), "/reviamugrashhebrew" },
  { GUINT_TO_POINTER (0x2310), "/revlogicalnot" },
  { GUINT_TO_POINTER (0x027E), "/rfishhook" },
  { GUINT_TO_POINTER (0x027F), "/rfishhookreversed" },
  { GUINT_TO_POINTER (0x09DD), "/rhabengali" },
  { GUINT_TO_POINTER (0x095D), "/rhadeva" },
  { GUINT_TO_POINTER (0x03C1), "/rho" },
  { GUINT_TO_POINTER (0x027D), "/rhook" },
  { GUINT_TO_POINTER (0x027B), "/rhookturned" },
  { GUINT_TO_POINTER (0x02B5), "/rhookturnedsuperior" },
  { GUINT_TO_POINTER (0x03F1), "/rhosymbolgreek" },
  { GUINT_TO_POINTER (0x02DE), "/rhotichookmod" },
  { GUINT_TO_POINTER (0x3271), "/rieulacirclekorean" },
  { GUINT_TO_POINTER (0x3211), "/rieulaparenkorean" },
  { GUINT_TO_POINTER (0x3263), "/rieulcirclekorean" },
  { GUINT_TO_POINTER (0x3140), "/rieulhieuhkorean" },
  { GUINT_TO_POINTER (0x313A), "/rieulkiyeokkorean" },
  { GUINT_TO_POINTER (0x3169), "/rieulkiyeoksioskorean" },
  { GUINT_TO_POINTER (0x3139), "/rieulkorean" },
  { GUINT_TO_POINTER (0x313B), "/rieulmieumkorean" },
  { GUINT_TO_POINTER (0x316C), "/rieulpansioskorean" },
  { GUINT_TO_POINTER (0x3203), "/rieulparenkorean" },
  { GUINT_TO_POINTER (0x313F), "/rieulphieuphkorean" },
  { GUINT_TO_POINTER (0x313C), "/rieulpieupkorean" },
  { GUINT_TO_POINTER (0x316B), "/rieulpieupsioskorean" },
  { GUINT_TO_POINTER (0x313D), "/rieulsioskorean" },
  { GUINT_TO_POINTER (0x313E), "/rieulthieuthkorean" },
  { GUINT_TO_POINTER (0x316A), "/rieultikeutkorean" },
  { GUINT_TO_POINTER (0x316D), "/rieulyeorinhieuhkorean" },
  { GUINT_TO_POINTER (0x221F), "/rightangle" },
  { GUINT_TO_POINTER (0x0319), "/righttackbelowcmb" },
  { GUINT_TO_POINTER (0x22BF), "/righttriangle" },
  { GUINT_TO_POINTER (0x308A), "/rihiragana" },
  { GUINT_TO_POINTER (0x30EA), "/rikatakana" },
  { GUINT_TO_POINTER (0xFF98), "/rikatakanahalfwidth" },
  { GUINT_TO_POINTER (0x02DA), "/ring" },
  { GUINT_TO_POINTER (0x0325), "/ringbelowcmb" },
  { GUINT_TO_POINTER (0x030A), "/ringcmb" },
  { GUINT_TO_POINTER (0x02BF), "/ringhalfleft" },
  { GUINT_TO_POINTER (0x0559), "/ringhalfleftarmenian" },
  { GUINT_TO_POINTER (0x031C), "/ringhalfleftbelowcmb" },
  { GUINT_TO_POINTER (0x02D3), "/ringhalfleftcentered" },
  { GUINT_TO_POINTER (0x02BE), "/ringhalfright" },
  { GUINT_TO_POINTER (0x0339), "/ringhalfrightbelowcmb" },
  { GUINT_TO_POINTER (0x02D2), "/ringhalfrightcentered" },
  { GUINT_TO_POINTER (0x0213), "/rinvertedbreve" },
  { GUINT_TO_POINTER (0x3351), "/rittorusquare" },
  { GUINT_TO_POINTER (0x1E5F), "/rlinebelow" },
  { GUINT_TO_POINTER (0x027C), "/rlongleg" },
  { GUINT_TO_POINTER (0x027A), "/rlonglegturned" },
  { GUINT_TO_POINTER (0xFF52), "/rmonospace" },
  { GUINT_TO_POINTER (0x308D), "/rohiragana" },
  { GUINT_TO_POINTER (0x30ED), "/rokatakana" },
  { GUINT_TO_POINTER (0xFF9B), "/rokatakanahalfwidth" },
  { GUINT_TO_POINTER (0x0E23), "/roruathai" },
  { GUINT_TO_POINTER (0x24AD), "/rparen" },
  { GUINT_TO_POINTER (0x09DC), "/rrabengali" },
  { GUINT_TO_POINTER (0x0931), "/rradeva" },
  { GUINT_TO_POINTER (0x0A5C), "/rragurmukhi" },
  { GUINT_TO_POINTER (0x0691), "/rreharabic" },
  { GUINT_TO_POINTER (0xFB8D), "/rrehfinalarabic" },
  { GUINT_TO_POINTER (0x09E0), "/rrvocalicbengali" },
  { GUINT_TO_POINTER (0x0960), "/rrvocalicdeva" },
  { GUINT_TO_POINTER (0x0AE0), "/rrvocalicgujarati" },
  { GUINT_TO_POINTER (0x09C4), "/rrvocalicvowelsignbengali" },
  { GUINT_TO_POINTER (0x0944), "/rrvocalicvowelsigndeva" },
  { GUINT_TO_POINTER (0x0AC4), "/rrvocalicvowelsigngujarati" },
  { GUINT_TO_POINTER (0xF6F1), "/rsuperior" },
  { GUINT_TO_POINTER (0x2590), "/rtblock" },
  { GUINT_TO_POINTER (0x0279), "/rturned" },
  { GUINT_TO_POINTER (0x02B4), "/rturnedsuperior" },
  { GUINT_TO_POINTER (0x308B), "/ruhiragana" },
  { GUINT_TO_POINTER (0x30EB), "/rukatakana" },
  { GUINT_TO_POINTER (0xFF99), "/rukatakanahalfwidth" },
  { GUINT_TO_POINTER (0x09F2), "/rupeemarkbengali" },
  { GUINT_TO_POINTER (0x09F3), "/rupeesignbengali" },
  { GUINT_TO_POINTER (0xF6DD), "/rupiah" },
  { GUINT_TO_POINTER (0x0E24), "/ruthai" },
  { GUINT_TO_POINTER (0x098B), "/rvocalicbengali" },
  { GUINT_TO_POINTER (0x090B), "/rvocalicdeva" },
  { GUINT_TO_POINTER (0x0A8B), "/rvocalicgujarati" },
  { GUINT_TO_POINTER (0x09C3), "/rvocalicvowelsignbengali" },
  { GUINT_TO_POINTER (0x0943), "/rvocalicvowelsigndeva" },
  { GUINT_TO_POINTER (0x0AC3), "/rvocalicvowelsigngujarati" },
  { GUINT_TO_POINTER (0x0073), "/s" },
  { GUINT_TO_POINTER (0x09B8), "/sabengali" },
  { GUINT_TO_POINTER (0x015B), "/sacute" },
  { GUINT_TO_POINTER (0x1E65), "/sacutedotaccent" },
  { GUINT_TO_POINTER (0x0635), "/sadarabic" },
  { GUINT_TO_POINTER (0x0938), "/sadeva" },
  { GUINT_TO_POINTER (0xFEBA), "/sadfinalarabic" },
  { GUINT_TO_POINTER (0xFEBB), "/sadinitialarabic" },
  { GUINT_TO_POINTER (0xFEBC), "/sadmedialarabic" },
  { GUINT_TO_POINTER (0x0AB8), "/sagujarati" },
  { GUINT_TO_POINTER (0x0A38), "/sagurmukhi" },
  { GUINT_TO_POINTER (0x3055), "/sahiragana" },
  { GUINT_TO_POINTER (0x30B5), "/sakatakana" },
  { GUINT_TO_POINTER (0xFF7B), "/sakatakanahalfwidth" },
  { GUINT_TO_POINTER (0xFDFA), "/sallallahoualayhewasallamarabic" },
  { GUINT_TO_POINTER (0x05E1), "/samekh" },
  { GUINT_TO_POINTER (0xFB41), "/samekhdagesh" },
  { GUINT_TO_POINTER (0xFB41), "/samekhdageshhebrew" },
  { GUINT_TO_POINTER (0x05E1), "/samekhhebrew" },
  { GUINT_TO_POINTER (0x0E32), "/saraaathai" },
  { GUINT_TO_POINTER (0x0E41), "/saraaethai" },
  { GUINT_TO_POINTER (0x0E44), "/saraaimaimalaithai" },
  { GUINT_TO_POINTER (0x0E43), "/saraaimaimuanthai" },
  { GUINT_TO_POINTER (0x0E33), "/saraamthai" },
  { GUINT_TO_POINTER (0x0E30), "/saraathai" },
  { GUINT_TO_POINTER (0x0E40), "/saraethai" },
  { GUINT_TO_POINTER (0xF886), "/saraiileftthai" },
  { GUINT_TO_POINTER (0x0E35), "/saraiithai" },
  { GUINT_TO_POINTER (0xF885), "/saraileftthai" },
  { GUINT_TO_POINTER (0x0E34), "/saraithai" },
  { GUINT_TO_POINTER (0x0E42), "/saraothai" },
  { GUINT_TO_POINTER (0xF888), "/saraueeleftthai" },
  { GUINT_TO_POINTER (0x0E37), "/saraueethai" },
  { GUINT_TO_POINTER (0xF887), "/saraueleftthai" },
  { GUINT_TO_POINTER (0x0E36), "/sarauethai" },
  { GUINT_TO_POINTER (0x0E38), "/sarauthai" },
  { GUINT_TO_POINTER (0x0E39), "/sarauuthai" },
  { GUINT_TO_POINTER (0x3119), "/sbopomofo" },
  { GUINT_TO_POINTER (0x0161), "/scaron" },
  { GUINT_TO_POINTER (0x1E67), "/scarondotaccent" },
  { GUINT_TO_POINTER (0x015F), "/scedilla" },
  { GUINT_TO_POINTER (0x0259), "/schwa" },
  { GUINT_TO_POINTER (0x04D9), "/schwacyrillic" },
  { GUINT_TO_POINTER (0x04DB), "/schwadieresiscyrillic" },
  { GUINT_TO_POINTER (0x025A), "/schwahook" },
  { GUINT_TO_POINTER (0x24E2), "/scircle" },
  { GUINT_TO_POINTER (0x015D), "/scircumflex" },
  { GUINT_TO_POINTER (0x0219), "/scommaaccent" },
  { GUINT_TO_POINTER (0x1E61), "/sdotaccent" },
  { GUINT_TO_POINTER (0x1E63), "/sdotbelow" },
  { GUINT_TO_POINTER (0x1E69), "/sdotbelowdotaccent" },
  { GUINT_TO_POINTER (0x033C), "/seagullbelowcmb" },
  { GUINT_TO_POINTER (0x2033), "/second" },
  { GUINT_TO_POINTER (0x02CA), "/secondtonechinese" },
  { GUINT_TO_POINTER (0x00A7), "/section" },
  { GUINT_TO_POINTER (0x0633), "/seenarabic" },
  { GUINT_TO_POINTER (0xFEB2), "/seenfinalarabic" },
  { GUINT_TO_POINTER (0xFEB3), "/seeninitialarabic" },
  { GUINT_TO_POINTER (0xFEB4), "/seenmedialarabic" },
  { GUINT_TO_POINTER (0x05B6), "/segol" },
  { GUINT_TO_POINTER (0x05B6), "/segol13" },
  { GUINT_TO_POINTER (0x05B6), "/segol1f" },
  { GUINT_TO_POINTER (0x05B6), "/segol2c" },
  { GUINT_TO_POINTER (0x05B6), "/segolhebrew" },
  { GUINT_TO_POINTER (0x05B6), "/segolnarrowhebrew" },
  { GUINT_TO_POINTER (0x05B6), "/segolquarterhebrew" },
  { GUINT_TO_POINTER (0x0592), "/segoltahebrew" },
  { GUINT_TO_POINTER (0x05B6), "/segolwidehebrew" },
  { GUINT_TO_POINTER (0x057D), "/seharmenian" },
  { GUINT_TO_POINTER (0x305B), "/sehiragana" },
  { GUINT_TO_POINTER (0x30BB), "/sekatakana" },
  { GUINT_TO_POINTER (0xFF7E), "/sekatakanahalfwidth" },
  { GUINT_TO_POINTER (0x003B), "/semicolon" },
  { GUINT_TO_POINTER (0x061B), "/semicolonarabic" },
  { GUINT_TO_POINTER (0xFF1B), "/semicolonmonospace" },
  { GUINT_TO_POINTER (0xFE54), "/semicolonsmall" },
  { GUINT_TO_POINTER (0x309C), "/semivoicedmarkkana" },
  { GUINT_TO_POINTER (0xFF9F), "/semivoicedmarkkanahalfwidth" },
  { GUINT_TO_POINTER (0x3322), "/sentisquare" },
  { GUINT_TO_POINTER (0x3323), "/sentosquare" },
  { GUINT_TO_POINTER (0x0037), "/seven" },
  { GUINT_TO_POINTER (0x0667), "/sevenarabic" },
  { GUINT_TO_POINTER (0x09ED), "/sevenbengali" },
  { GUINT_TO_POINTER (0x2466), "/sevencircle" },
  { GUINT_TO_POINTER (0x2790), "/sevencircleinversesansserif" },
  { GUINT_TO_POINTER (0x096D), "/sevendeva" },
  { GUINT_TO_POINTER (0x215E), "/seveneighths" },
  { GUINT_TO_POINTER (0x0AED), "/sevengujarati" },
  { GUINT_TO_POINTER (0x0A6D), "/sevengurmukhi" },
  { GUINT_TO_POINTER (0x0667), "/sevenhackarabic" },
  { GUINT_TO_POINTER (0x3027), "/sevenhangzhou" },
  { GUINT_TO_POINTER (0x3226), "/sevenideographicparen" },
  { GUINT_TO_POINTER (0x2087), "/seveninferior" },
  { GUINT_TO_POINTER (0xFF17), "/sevenmonospace" },
  { GUINT_TO_POINTER (0xF737), "/sevenoldstyle" },
  { GUINT_TO_POINTER (0x247A), "/sevenparen" },
  { GUINT_TO_POINTER (0x248E), "/sevenperiod" },
  { GUINT_TO_POINTER (0x06F7), "/sevenpersian" },
  { GUINT_TO_POINTER (0x2176), "/sevenroman" },
  { GUINT_TO_POINTER (0x2077), "/sevensuperior" },
  { GUINT_TO_POINTER (0x2470), "/seventeencircle" },
  { GUINT_TO_POINTER (0x2484), "/seventeenparen" },
  { GUINT_TO_POINTER (0x2498), "/seventeenperiod" },
  { GUINT_TO_POINTER (0x0E57), "/seventhai" },
  { GUINT_TO_POINTER (0x00AD), "/sfthyphen" },
  { GUINT_TO_POINTER (0x0577), "/shaarmenian" },
  { GUINT_TO_POINTER (0x09B6), "/shabengali" },
  { GUINT_TO_POINTER (0x0448), "/shacyrillic" },
  { GUINT_TO_POINTER (0x0651), "/shaddaarabic" },
  { GUINT_TO_POINTER (0xFC61), "/shaddadammaarabic" },
  { GUINT_TO_POINTER (0xFC5E), "/shaddadammatanarabic" },
  { GUINT_TO_POINTER (0xFC60), "/shaddafathaarabic" },
  { GUINT_TO_POINTER (0x0651), "/shaddafathatanarabic" },
  { GUINT_TO_POINTER (0x064B), "/shaddafathatanarabic" },
  { GUINT_TO_POINTER (0xFC62), "/shaddakasraarabic" },
  { GUINT_TO_POINTER (0xFC5F), "/shaddakasratanarabic" },
  { GUINT_TO_POINTER (0x2592), "/shade" },
  { GUINT_TO_POINTER (0x2593), "/shadedark" },
  { GUINT_TO_POINTER (0x2591), "/shadelight" },
  { GUINT_TO_POINTER (0x2592), "/shademedium" },
  { GUINT_TO_POINTER (0x0936), "/shadeva" },
  { GUINT_TO_POINTER (0x0AB6), "/shagujarati" },
  { GUINT_TO_POINTER (0x0A36), "/shagurmukhi" },
  { GUINT_TO_POINTER (0x0593), "/shalshelethebrew" },
  { GUINT_TO_POINTER (0x3115), "/shbopomofo" },
  { GUINT_TO_POINTER (0x0449), "/shchacyrillic" },
  { GUINT_TO_POINTER (0x0634), "/sheenarabic" },
  { GUINT_TO_POINTER (0xFEB6), "/sheenfinalarabic" },
  { GUINT_TO_POINTER (0xFEB7), "/sheeninitialarabic" },
  { GUINT_TO_POINTER (0xFEB8), "/sheenmedialarabic" },
  { GUINT_TO_POINTER (0x03E3), "/sheicoptic" },
  { GUINT_TO_POINTER (0x20AA), "/sheqel" },
  { GUINT_TO_POINTER (0x20AA), "/sheqelhebrew" },
  { GUINT_TO_POINTER (0x05B0), "/sheva" },
  { GUINT_TO_POINTER (0x05B0), "/sheva115" },
  { GUINT_TO_POINTER (0x05B0), "/sheva15" },
  { GUINT_TO_POINTER (0x05B0), "/sheva22" },
  { GUINT_TO_POINTER (0x05B0), "/sheva2e" },
  { GUINT_TO_POINTER (0x05B0), "/shevahebrew" },
  { GUINT_TO_POINTER (0x05B0), "/shevanarrowhebrew" },
  { GUINT_TO_POINTER (0x05B0), "/shevaquarterhebrew" },
  { GUINT_TO_POINTER (0x05B0), "/shevawidehebrew" },
  { GUINT_TO_POINTER (0x04BB), "/shhacyrillic" },
  { GUINT_TO_POINTER (0x03ED), "/shimacoptic" },
  { GUINT_TO_POINTER (0x05E9), "/shin" },
  { GUINT_TO_POINTER (0xFB49), "/shindagesh" },
  { GUINT_TO_POINTER (0xFB49), "/shindageshhebrew" },
  { GUINT_TO_POINTER (0xFB2C), "/shindageshshindot" },
  { GUINT_TO_POINTER (0xFB2C), "/shindageshshindothebrew" },
  { GUINT_TO_POINTER (0xFB2D), "/shindageshsindot" },
  { GUINT_TO_POINTER (0xFB2D), "/shindageshsindothebrew" },
  { GUINT_TO_POINTER (0x05C1), "/shindothebrew" },
  { GUINT_TO_POINTER (0x05E9), "/shinhebrew" },
  { GUINT_TO_POINTER (0xFB2A), "/shinshindot" },
  { GUINT_TO_POINTER (0xFB2A), "/shinshindothebrew" },
  { GUINT_TO_POINTER (0xFB2B), "/shinsindot" },
  { GUINT_TO_POINTER (0xFB2B), "/shinsindothebrew" },
  { GUINT_TO_POINTER (0x0282), "/shook" },
  { GUINT_TO_POINTER (0x03C3), "/sigma" },
  { GUINT_TO_POINTER (0x03C2), "/sigma1" },
  { GUINT_TO_POINTER (0x03C2), "/sigmafinal" },
  { GUINT_TO_POINTER (0x03F2), "/sigmalunatesymbolgreek" },
  { GUINT_TO_POINTER (0x3057), "/sihiragana" },
  { GUINT_TO_POINTER (0x30B7), "/sikatakana" },
  { GUINT_TO_POINTER (0xFF7C), "/sikatakanahalfwidth" },
  { GUINT_TO_POINTER (0x05BD), "/siluqhebrew" },
  { GUINT_TO_POINTER (0x05BD), "/siluqlefthebrew" },
  { GUINT_TO_POINTER (0x223C), "/similar" },
  { GUINT_TO_POINTER (0x05C2), "/sindothebrew" },
  { GUINT_TO_POINTER (0x3274), "/siosacirclekorean" },
  { GUINT_TO_POINTER (0x3214), "/siosaparenkorean" },
  { GUINT_TO_POINTER (0x317E), "/sioscieuckorean" },
  { GUINT_TO_POINTER (0x3266), "/sioscirclekorean" },
  { GUINT_TO_POINTER (0x317A), "/sioskiyeokkorean" },
  { GUINT_TO_POINTER (0x3145), "/sioskorean" },
  { GUINT_TO_POINTER (0x317B), "/siosnieunkorean" },
  { GUINT_TO_POINTER (0x3206), "/siosparenkorean" },
  { GUINT_TO_POINTER (0x317D), "/siospieupkorean" },
  { GUINT_TO_POINTER (0x317C), "/siostikeutkorean" },
  { GUINT_TO_POINTER (0x0036), "/six" },
  { GUINT_TO_POINTER (0x0666), "/sixarabic" },
  { GUINT_TO_POINTER (0x09EC), "/sixbengali" },
  { GUINT_TO_POINTER (0x2465), "/sixcircle" },
  { GUINT_TO_POINTER (0x278F), "/sixcircleinversesansserif" },
  { GUINT_TO_POINTER (0x096C), "/sixdeva" },
  { GUINT_TO_POINTER (0x0AEC), "/sixgujarati" },
  { GUINT_TO_POINTER (0x0A6C), "/sixgurmukhi" },
  { GUINT_TO_POINTER (0x0666), "/sixhackarabic" },
  { GUINT_TO_POINTER (0x3026), "/sixhangzhou" },
  { GUINT_TO_POINTER (0x3225), "/sixideographicparen" },
  { GUINT_TO_POINTER (0x2086), "/sixinferior" },
  { GUINT_TO_POINTER (0xFF16), "/sixmonospace" },
  { GUINT_TO_POINTER (0xF736), "/sixoldstyle" },
  { GUINT_TO_POINTER (0x2479), "/sixparen" },
  { GUINT_TO_POINTER (0x248D), "/sixperiod" },
  { GUINT_TO_POINTER (0x06F6), "/sixpersian" },
  { GUINT_TO_POINTER (0x2175), "/sixroman" },
  { GUINT_TO_POINTER (0x2076), "/sixsuperior" },
  { GUINT_TO_POINTER (0x246F), "/sixteencircle" },
  { GUINT_TO_POINTER (0x09F9), "/sixteencurrencydenominatorbengali" },
  { GUINT_TO_POINTER (0x2483), "/sixteenparen" },
  { GUINT_TO_POINTER (0x2497), "/sixteenperiod" },
  { GUINT_TO_POINTER (0x0E56), "/sixthai" },
  { GUINT_TO_POINTER (0x002F), "/slash" },
  { GUINT_TO_POINTER (0xFF0F), "/slashmonospace" },
  { GUINT_TO_POINTER (0x017F), "/slong" },
  { GUINT_TO_POINTER (0x1E9B), "/slongdotaccent" },
  { GUINT_TO_POINTER (0x263A), "/smileface" },
  { GUINT_TO_POINTER (0xFF53), "/smonospace" },
  { GUINT_TO_POINTER (0x05C3), "/sofpasuqhebrew" },
  { GUINT_TO_POINTER (0x00AD), "/softhyphen" },
  { GUINT_TO_POINTER (0x044C), "/softsigncyrillic" },
  { GUINT_TO_POINTER (0x305D), "/sohiragana" },
  { GUINT_TO_POINTER (0x30BD), "/sokatakana" },
  { GUINT_TO_POINTER (0xFF7F), "/sokatakanahalfwidth" },
  { GUINT_TO_POINTER (0x0338), "/soliduslongoverlaycmb" },
  { GUINT_TO_POINTER (0x0337), "/solidusshortoverlaycmb" },
  { GUINT_TO_POINTER (0x0E29), "/sorusithai" },
  { GUINT_TO_POINTER (0x0E28), "/sosalathai" },
  { GUINT_TO_POINTER (0x0E0B), "/sosothai" },
  { GUINT_TO_POINTER (0x0E2A), "/sosuathai" },
  { GUINT_TO_POINTER (0x0020), "/space" },
  { GUINT_TO_POINTER (0x2660), "/spade" },
  { GUINT_TO_POINTER (0x2660), "/spadesuitblack" },
  { GUINT_TO_POINTER (0x2664), "/spadesuitwhite" },
  { GUINT_TO_POINTER (0x24AE), "/sparen" },
  { GUINT_TO_POINTER (0x033B), "/squarebelowcmb" },
  { GUINT_TO_POINTER (0x33C4), "/squarecc" },
  { GUINT_TO_POINTER (0x339D), "/squarecm" },
  { GUINT_TO_POINTER (0x25A9), "/squarediagonalcrosshatchfill" },
  { GUINT_TO_POINTER (0x25A4), "/squarehorizontalfill" },
  { GUINT_TO_POINTER (0x338F), "/squarekg" },
  { GUINT_TO_POINTER (0x339E), "/squarekm" },
  { GUINT_TO_POINTER (0x33CE), "/squarekmcapital" },
  { GUINT_TO_POINTER (0x33D1), "/squareln" },
  { GUINT_TO_POINTER (0x33D2), "/squarelog" },
  { GUINT_TO_POINTER (0x338E), "/squaremg" },
  { GUINT_TO_POINTER (0x33D5), "/squaremil" },
  { GUINT_TO_POINTER (0x339C), "/squaremm" },
  { GUINT_TO_POINTER (0x33A1), "/squaremsquared" },
  { GUINT_TO_POINTER (0x25A6), "/squareorthogonalcrosshatchfill" },
  { GUINT_TO_POINTER (0x25A7), "/squareupperlefttolowerrightfill" },
  { GUINT_TO_POINTER (0x25A8), "/squareupperrighttolowerleftfill" },
  { GUINT_TO_POINTER (0x25A5), "/squareverticalfill" },
  { GUINT_TO_POINTER (0x25A3), "/squarewhitewithsmallblack" },
  { GUINT_TO_POINTER (0x33DB), "/srsquare" },
  { GUINT_TO_POINTER (0x09B7), "/ssabengali" },
  { GUINT_TO_POINTER (0x0937), "/ssadeva" },
  { GUINT_TO_POINTER (0x0AB7), "/ssagujarati" },
  { GUINT_TO_POINTER (0x3149), "/ssangcieuckorean" },
  { GUINT_TO_POINTER (0x3185), "/ssanghieuhkorean" },
  { GUINT_TO_POINTER (0x3180), "/ssangieungkorean" },
  { GUINT_TO_POINTER (0x3132), "/ssangkiyeokkorean" },
  { GUINT_TO_POINTER (0x3165), "/ssangnieunkorean" },
  { GUINT_TO_POINTER (0x3143), "/ssangpieupkorean" },
  { GUINT_TO_POINTER (0x3146), "/ssangsioskorean" },
  { GUINT_TO_POINTER (0x3138), "/ssangtikeutkorean" },
  { GUINT_TO_POINTER (0xF6F2), "/ssuperior" },
  { GUINT_TO_POINTER (0x00A3), "/sterling" },
  { GUINT_TO_POINTER (0xFFE1), "/sterlingmonospace" },
  { GUINT_TO_POINTER (0x0336), "/strokelongoverlaycmb" },
  { GUINT_TO_POINTER (0x0335), "/strokeshortoverlaycmb" },
  { GUINT_TO_POINTER (0x2282), "/subset" },
  { GUINT_TO_POINTER (0x228A), "/subsetnotequal" },
  { GUINT_TO_POINTER (0x2286), "/subsetorequal" },
  { GUINT_TO_POINTER (0x227B), "/succeeds" },
  { GUINT_TO_POINTER (0x220B), "/suchthat" },
  { GUINT_TO_POINTER (0x3059), "/suhiragana" },
  { GUINT_TO_POINTER (0x30B9), "/sukatakana" },
  { GUINT_TO_POINTER (0xFF7D), "/sukatakanahalfwidth" },
  { GUINT_TO_POINTER (0x0652), "/sukunarabic" },
  { GUINT_TO_POINTER (0x2211), "/summation" },
  { GUINT_TO_POINTER (0x263C), "/sun" },
  { GUINT_TO_POINTER (0x2283), "/superset" },
  { GUINT_TO_POINTER (0x228B), "/supersetnotequal" },
  { GUINT_TO_POINTER (0x2287), "/supersetorequal" },
  { GUINT_TO_POINTER (0x33DC), "/svsquare" },
  { GUINT_TO_POINTER (0x337C), "/syouwaerasquare" },
  { GUINT_TO_POINTER (0x0074), "/t" },
  { GUINT_TO_POINTER (0x09A4), "/tabengali" },
  { GUINT_TO_POINTER (0x22A4), "/tackdown" },
  { GUINT_TO_POINTER (0x22A3), "/tackleft" },
  { GUINT_TO_POINTER (0x0924), "/tadeva" },
  { GUINT_TO_POINTER (0x0AA4), "/tagujarati" },
  { GUINT_TO_POINTER (0x0A24), "/tagurmukhi" },
  { GUINT_TO_POINTER (0x0637), "/taharabic" },
  { GUINT_TO_POINTER (0xFEC2), "/tahfinalarabic" },
  { GUINT_TO_POINTER (0xFEC3), "/tahinitialarabic" },
  { GUINT_TO_POINTER (0x305F), "/tahiragana" },
  { GUINT_TO_POINTER (0xFEC4), "/tahmedialarabic" },
  { GUINT_TO_POINTER (0x337D), "/taisyouerasquare" },
  { GUINT_TO_POINTER (0x30BF), "/takatakana" },
  { GUINT_TO_POINTER (0xFF80), "/takatakanahalfwidth" },
  { GUINT_TO_POINTER (0x0640), "/tatweelarabic" },
  { GUINT_TO_POINTER (0x03C4), "/tau" },
  { GUINT_TO_POINTER (0x05EA), "/tav" },
  { GUINT_TO_POINTER (0xFB4A), "/tavdages" },
  { GUINT_TO_POINTER (0xFB4A), "/tavdagesh" },
  { GUINT_TO_POINTER (0xFB4A), "/tavdageshhebrew" },
  { GUINT_TO_POINTER (0x05EA), "/tavhebrew" },
  { GUINT_TO_POINTER (0x0167), "/tbar" },
  { GUINT_TO_POINTER (0x310A), "/tbopomofo" },
  { GUINT_TO_POINTER (0x0165), "/tcaron" },
  { GUINT_TO_POINTER (0x02A8), "/tccurl" },
  { GUINT_TO_POINTER (0x0163), "/tcedilla" },
  { GUINT_TO_POINTER (0x0686), "/tcheharabic" },
  { GUINT_TO_POINTER (0xFB7B), "/tchehfinalarabic" },
  { GUINT_TO_POINTER (0xFB7C), "/tchehinitialarabic" },
  { GUINT_TO_POINTER (0xFB7D), "/tchehmedialarabic" },
  { GUINT_TO_POINTER (0xFB7C), "/tchehmeeminitialarabic" },
  { GUINT_TO_POINTER (0xFEE4), "/tchehmeeminitialarabic" },
  { GUINT_TO_POINTER (0x24E3), "/tcircle" },
  { GUINT_TO_POINTER (0x1E71), "/tcircumflexbelow" },
  { GUINT_TO_POINTER (0x0163), "/tcommaaccent" },
  { GUINT_TO_POINTER (0x1E97), "/tdieresis" },
  { GUINT_TO_POINTER (0x1E6B), "/tdotaccent" },
  { GUINT_TO_POINTER (0x1E6D), "/tdotbelow" },
  { GUINT_TO_POINTER (0x0442), "/tecyrillic" },
  { GUINT_TO_POINTER (0x04AD), "/tedescendercyrillic" },
  { GUINT_TO_POINTER (0x062A), "/teharabic" },
  { GUINT_TO_POINTER (0xFE96), "/tehfinalarabic" },
  { GUINT_TO_POINTER (0xFCA2), "/tehhahinitialarabic" },
  { GUINT_TO_POINTER (0xFC0C), "/tehhahisolatedarabic" },
  { GUINT_TO_POINTER (0xFE97), "/tehinitialarabic" },
  { GUINT_TO_POINTER (0x3066), "/tehiragana" },
  { GUINT_TO_POINTER (0xFCA1), "/tehjeeminitialarabic" },
  { GUINT_TO_POINTER (0xFC0B), "/tehjeemisolatedarabic" },
  { GUINT_TO_POINTER (0x0629), "/tehmarbutaarabic" },
  { GUINT_TO_POINTER (0xFE94), "/tehmarbutafinalarabic" },
  { GUINT_TO_POINTER (0xFE98), "/tehmedialarabic" },
  { GUINT_TO_POINTER (0xFCA4), "/tehmeeminitialarabic" },
  { GUINT_TO_POINTER (0xFC0E), "/tehmeemisolatedarabic" },
  { GUINT_TO_POINTER (0xFC73), "/tehnoonfinalarabic" },
  { GUINT_TO_POINTER (0x30C6), "/tekatakana" },
  { GUINT_TO_POINTER (0xFF83), "/tekatakanahalfwidth" },
  { GUINT_TO_POINTER (0x2121), "/telephone" },
  { GUINT_TO_POINTER (0x260E), "/telephoneblack" },
  { GUINT_TO_POINTER (0x05A0), "/telishagedolahebrew" },
  { GUINT_TO_POINTER (0x05A9), "/telishaqetanahebrew" },
  { GUINT_TO_POINTER (0x2469), "/tencircle" },
  { GUINT_TO_POINTER (0x3229), "/tenideographicparen" },
  { GUINT_TO_POINTER (0x247D), "/tenparen" },
  { GUINT_TO_POINTER (0x2491), "/tenperiod" },
  { GUINT_TO_POINTER (0x2179), "/tenroman" },
  { GUINT_TO_POINTER (0x02A7), "/tesh" },
  { GUINT_TO_POINTER (0x05D8), "/tet" },
  { GUINT_TO_POINTER (0xFB38), "/tetdagesh" },
  { GUINT_TO_POINTER (0xFB38), "/tetdageshhebrew" },
  { GUINT_TO_POINTER (0x05D8), "/tethebrew" },
  { GUINT_TO_POINTER (0x04B5), "/tetsecyrillic" },
  { GUINT_TO_POINTER (0x059B), "/tevirhebrew" },
  { GUINT_TO_POINTER (0x059B), "/tevirlefthebrew" },
  { GUINT_TO_POINTER (0x09A5), "/thabengali" },
  { GUINT_TO_POINTER (0x0925), "/thadeva" },
  { GUINT_TO_POINTER (0x0AA5), "/thagujarati" },
  { GUINT_TO_POINTER (0x0A25), "/thagurmukhi" },
  { GUINT_TO_POINTER (0x0630), "/thalarabic" },
  { GUINT_TO_POINTER (0xFEAC), "/thalfinalarabic" },
  { GUINT_TO_POINTER (0xF898), "/thanthakhatlowleftthai" },
  { GUINT_TO_POINTER (0xF897), "/thanthakhatlowrightthai" },
  { GUINT_TO_POINTER (0x0E4C), "/thanthakhatthai" },
  { GUINT_TO_POINTER (0xF896), "/thanthakhatupperleftthai" },
  { GUINT_TO_POINTER (0x062B), "/theharabic" },
  { GUINT_TO_POINTER (0xFE9A), "/thehfinalarabic" },
  { GUINT_TO_POINTER (0xFE9B), "/thehinitialarabic" },
  { GUINT_TO_POINTER (0xFE9C), "/thehmedialarabic" },
  { GUINT_TO_POINTER (0x2203), "/thereexists" },
  { GUINT_TO_POINTER (0x2234), "/therefore" },
  { GUINT_TO_POINTER (0x03B8), "/theta" },
  { GUINT_TO_POINTER (0x03D1), "/theta1" },
  { GUINT_TO_POINTER (0x03D1), "/thetasymbolgreek" },
  { GUINT_TO_POINTER (0x3279), "/thieuthacirclekorean" },
  { GUINT_TO_POINTER (0x3219), "/thieuthaparenkorean" },
  { GUINT_TO_POINTER (0x326B), "/thieuthcirclekorean" },
  { GUINT_TO_POINTER (0x314C), "/thieuthkorean" },
  { GUINT_TO_POINTER (0x320B), "/thieuthparenkorean" },
  { GUINT_TO_POINTER (0x246C), "/thirteencircle" },
  { GUINT_TO_POINTER (0x2480), "/thirteenparen" },
  { GUINT_TO_POINTER (0x2494), "/thirteenperiod" },
  { GUINT_TO_POINTER (0x0E11), "/thonangmonthothai" },
  { GUINT_TO_POINTER (0x01AD), "/thook" },
  { GUINT_TO_POINTER (0x0E12), "/thophuthaothai" },
  { GUINT_TO_POINTER (0x00FE), "/thorn" },
  { GUINT_TO_POINTER (0x0E17), "/thothahanthai" },
  { GUINT_TO_POINTER (0x0E10), "/thothanthai" },
  { GUINT_TO_POINTER (0x0E18), "/thothongthai" },
  { GUINT_TO_POINTER (0x0E16), "/thothungthai" },
  { GUINT_TO_POINTER (0x0482), "/thousandcyrillic" },
  { GUINT_TO_POINTER (0x066C), "/thousandsseparatorarabic" },
  { GUINT_TO_POINTER (0x066C), "/thousandsseparatorpersian" },
  { GUINT_TO_POINTER (0x0033), "/three" },
  { GUINT_TO_POINTER (0x0663), "/threearabic" },
  { GUINT_TO_POINTER (0x09E9), "/threebengali" },
  { GUINT_TO_POINTER (0x2462), "/threecircle" },
  { GUINT_TO_POINTER (0x278C), "/threecircleinversesansserif" },
  { GUINT_TO_POINTER (0x0969), "/threedeva" },
  { GUINT_TO_POINTER (0x215C), "/threeeighths" },
  { GUINT_TO_POINTER (0x0AE9), "/threegujarati" },
  { GUINT_TO_POINTER (0x0A69), "/threegurmukhi" },
  { GUINT_TO_POINTER (0x0663), "/threehackarabic" },
  { GUINT_TO_POINTER (0x3023), "/threehangzhou" },
  { GUINT_TO_POINTER (0x3222), "/threeideographicparen" },
  { GUINT_TO_POINTER (0x2083), "/threeinferior" },
  { GUINT_TO_POINTER (0xFF13), "/threemonospace" },
  { GUINT_TO_POINTER (0x09F6), "/threenumeratorbengali" },
  { GUINT_TO_POINTER (0xF733), "/threeoldstyle" },
  { GUINT_TO_POINTER (0x2476), "/threeparen" },
  { GUINT_TO_POINTER (0x248A), "/threeperiod" },
  { GUINT_TO_POINTER (0x06F3), "/threepersian" },
  { GUINT_TO_POINTER (0x00BE), "/threequarters" },
  { GUINT_TO_POINTER (0xF6DE), "/threequartersemdash" },
  { GUINT_TO_POINTER (0x2172), "/threeroman" },
  { GUINT_TO_POINTER (0x00B3), "/threesuperior" },
  { GUINT_TO_POINTER (0x0E53), "/threethai" },
  { GUINT_TO_POINTER (0x3394), "/thzsquare" },
  { GUINT_TO_POINTER (0x3061), "/tihiragana" },
  { GUINT_TO_POINTER (0x30C1), "/tikatakana" },
  { GUINT_TO_POINTER (0xFF81), "/tikatakanahalfwidth" },
  { GUINT_TO_POINTER (0x3270), "/tikeutacirclekorean" },
  { GUINT_TO_POINTER (0x3210), "/tikeutaparenkorean" },
  { GUINT_TO_POINTER (0x3262), "/tikeutcirclekorean" },
  { GUINT_TO_POINTER (0x3137), "/tikeutkorean" },
  { GUINT_TO_POINTER (0x3202), "/tikeutparenkorean" },
  { GUINT_TO_POINTER (0x02DC), "/tilde" },
  { GUINT_TO_POINTER (0x0330), "/tildebelowcmb" },
  { GUINT_TO_POINTER (0x0303), "/tildecmb" },
  { GUINT_TO_POINTER (0x0303), "/tildecomb" },
  { GUINT_TO_POINTER (0x0360), "/tildedoublecmb" },
  { GUINT_TO_POINTER (0x223C), "/tildeoperator" },
  { GUINT_TO_POINTER (0x0334), "/tildeoverlaycmb" },
  { GUINT_TO_POINTER (0x033E), "/tildeverticalcmb" },
  { GUINT_TO_POINTER (0x2297), "/timescircle" },
  { GUINT_TO_POINTER (0x0596), "/tipehahebrew" },
  { GUINT_TO_POINTER (0x0596), "/tipehalefthebrew" },
  { GUINT_TO_POINTER (0x0A70), "/tippigurmukhi" },
  { GUINT_TO_POINTER (0x0483), "/titlocyrilliccmb" },
  { GUINT_TO_POINTER (0x057F), "/tiwnarmenian" },
  { GUINT_TO_POINTER (0x1E6F), "/tlinebelow" },
  { GUINT_TO_POINTER (0xFF54), "/tmonospace" },
  { GUINT_TO_POINTER (0x0569), "/toarmenian" },
  { GUINT_TO_POINTER (0x3068), "/tohiragana" },
  { GUINT_TO_POINTER (0x30C8), "/tokatakana" },
  { GUINT_TO_POINTER (0xFF84), "/tokatakanahalfwidth" },
  { GUINT_TO_POINTER (0x02E5), "/tonebarextrahighmod" },
  { GUINT_TO_POINTER (0x02E9), "/tonebarextralowmod" },
  { GUINT_TO_POINTER (0x02E6), "/tonebarhighmod" },
  { GUINT_TO_POINTER (0x02E8), "/tonebarlowmod" },
  { GUINT_TO_POINTER (0x02E7), "/tonebarmidmod" },
  { GUINT_TO_POINTER (0x01BD), "/tonefive" },
  { GUINT_TO_POINTER (0x0185), "/tonesix" },
  { GUINT_TO_POINTER (0x01A8), "/tonetwo" },
  { GUINT_TO_POINTER (0x0384), "/tonos" },
  { GUINT_TO_POINTER (0x3327), "/tonsquare" },
  { GUINT_TO_POINTER (0x0E0F), "/topatakthai" },
  { GUINT_TO_POINTER (0x3014), "/tortoiseshellbracketleft" },
  { GUINT_TO_POINTER (0xFE5D), "/tortoiseshellbracketleftsmall" },
  { GUINT_TO_POINTER (0xFE39), "/tortoiseshellbracketleftvertical" },
  { GUINT_TO_POINTER (0x3015), "/tortoiseshellbracketright" },
  { GUINT_TO_POINTER (0xFE5E), "/tortoiseshellbracketrightsmall" },
  { GUINT_TO_POINTER (0xFE3A), "/tortoiseshellbracketrightvertical" },
  { GUINT_TO_POINTER (0x0E15), "/totaothai" },
  { GUINT_TO_POINTER (0x01AB), "/tpalatalhook" },
  { GUINT_TO_POINTER (0x24AF), "/tparen" },
  { GUINT_TO_POINTER (0x2122), "/trademark" },
  { GUINT_TO_POINTER (0xF8EA), "/trademarksans" },
  { GUINT_TO_POINTER (0xF6DB), "/trademarkserif" },
  { GUINT_TO_POINTER (0x0288), "/tretroflexhook" },
  { GUINT_TO_POINTER (0x25BC), "/triagdn" },
  { GUINT_TO_POINTER (0x25C4), "/triaglf" },
  { GUINT_TO_POINTER (0x25BA), "/triagrt" },
  { GUINT_TO_POINTER (0x25B2), "/triagup" },
  { GUINT_TO_POINTER (0x02A6), "/ts" },
  { GUINT_TO_POINTER (0x05E6), "/tsadi" },
  { GUINT_TO_POINTER (0xFB46), "/tsadidagesh" },
  { GUINT_TO_POINTER (0xFB46), "/tsadidageshhebrew" },
  { GUINT_TO_POINTER (0x05E6), "/tsadihebrew" },
  { GUINT_TO_POINTER (0x0446), "/tsecyrillic" },
  { GUINT_TO_POINTER (0x05B5), "/tsere" },
  { GUINT_TO_POINTER (0x05B5), "/tsere12" },
  { GUINT_TO_POINTER (0x05B5), "/tsere1e" },
  { GUINT_TO_POINTER (0x05B5), "/tsere2b" },
  { GUINT_TO_POINTER (0x05B5), "/tserehebrew" },
  { GUINT_TO_POINTER (0x05B5), "/tserenarrowhebrew" },
  { GUINT_TO_POINTER (0x05B5), "/tserequarterhebrew" },
  { GUINT_TO_POINTER (0x05B5), "/tserewidehebrew" },
  { GUINT_TO_POINTER (0x045B), "/tshecyrillic" },
  { GUINT_TO_POINTER (0xF6F3), "/tsuperior" },
  { GUINT_TO_POINTER (0x099F), "/ttabengali" },
  { GUINT_TO_POINTER (0x091F), "/ttadeva" },
  { GUINT_TO_POINTER (0x0A9F), "/ttagujarati" },
  { GUINT_TO_POINTER (0x0A1F), "/ttagurmukhi" },
  { GUINT_TO_POINTER (0x0679), "/tteharabic" },
  { GUINT_TO_POINTER (0xFB67), "/ttehfinalarabic" },
  { GUINT_TO_POINTER (0xFB68), "/ttehinitialarabic" },
  { GUINT_TO_POINTER (0xFB69), "/ttehmedialarabic" },
  { GUINT_TO_POINTER (0x09A0), "/tthabengali" },
  { GUINT_TO_POINTER (0x0920), "/tthadeva" },
  { GUINT_TO_POINTER (0x0AA0), "/tthagujarati" },
  { GUINT_TO_POINTER (0x0A20), "/tthagurmukhi" },
  { GUINT_TO_POINTER (0x0287), "/tturned" },
  { GUINT_TO_POINTER (0x3064), "/tuhiragana" },
  { GUINT_TO_POINTER (0x30C4), "/tukatakana" },
  { GUINT_TO_POINTER (0xFF82), "/tukatakanahalfwidth" },
  { GUINT_TO_POINTER (0x3063), "/tusmallhiragana" },
  { GUINT_TO_POINTER (0x30C3), "/tusmallkatakana" },
  { GUINT_TO_POINTER (0xFF6F), "/tusmallkatakanahalfwidth" },
  { GUINT_TO_POINTER (0x246B), "/twelvecircle" },
  { GUINT_TO_POINTER (0x247F), "/twelveparen" },
  { GUINT_TO_POINTER (0x2493), "/twelveperiod" },
  { GUINT_TO_POINTER (0x217B), "/twelveroman" },
  { GUINT_TO_POINTER (0x2473), "/twentycircle" },
  { GUINT_TO_POINTER (0x5344), "/twentyhangzhou" },
  { GUINT_TO_POINTER (0x2487), "/twentyparen" },
  { GUINT_TO_POINTER (0x249B), "/twentyperiod" },
  { GUINT_TO_POINTER (0x0032), "/two" },
  { GUINT_TO_POINTER (0x0662), "/twoarabic" },
  { GUINT_TO_POINTER (0x09E8), "/twobengali" },
  { GUINT_TO_POINTER (0x2461), "/twocircle" },
  { GUINT_TO_POINTER (0x278B), "/twocircleinversesansserif" },
  { GUINT_TO_POINTER (0x0968), "/twodeva" },
  { GUINT_TO_POINTER (0x2025), "/twodotenleader" },
  { GUINT_TO_POINTER (0x2025), "/twodotleader" },
  { GUINT_TO_POINTER (0xFE30), "/twodotleadervertical" },
  { GUINT_TO_POINTER (0x0AE8), "/twogujarati" },
  { GUINT_TO_POINTER (0x0A68), "/twogurmukhi" },
  { GUINT_TO_POINTER (0x0662), "/twohackarabic" },
  { GUINT_TO_POINTER (0x3022), "/twohangzhou" },
  { GUINT_TO_POINTER (0x3221), "/twoideographicparen" },
  { GUINT_TO_POINTER (0x2082), "/twoinferior" },
  { GUINT_TO_POINTER (0xFF12), "/twomonospace" },
  { GUINT_TO_POINTER (0x09F5), "/twonumeratorbengali" },
  { GUINT_TO_POINTER (0xF732), "/twooldstyle" },
  { GUINT_TO_POINTER (0x2475), "/twoparen" },
  { GUINT_TO_POINTER (0x2489), "/twoperiod" },
  { GUINT_TO_POINTER (0x06F2), "/twopersian" },
  { GUINT_TO_POINTER (0x2171), "/tworoman" },
  { GUINT_TO_POINTER (0x01BB), "/twostroke" },
  { GUINT_TO_POINTER (0x00B2), "/twosuperior" },
  { GUINT_TO_POINTER (0x0E52), "/twothai" },
  { GUINT_TO_POINTER (0x2154), "/twothirds" },
  { GUINT_TO_POINTER (0x0075), "/u" },
  { GUINT_TO_POINTER (0x00FA), "/uacute" },
  { GUINT_TO_POINTER (0x0289), "/ubar" },
  { GUINT_TO_POINTER (0x0989), "/ubengali" },
  { GUINT_TO_POINTER (0x3128), "/ubopomofo" },
  { GUINT_TO_POINTER (0x016D), "/ubreve" },
  { GUINT_TO_POINTER (0x01D4), "/ucaron" },
  { GUINT_TO_POINTER (0x24E4), "/ucircle" },
  { GUINT_TO_POINTER (0x00FB), "/ucircumflex" },
  { GUINT_TO_POINTER (0x1E77), "/ucircumflexbelow" },
  { GUINT_TO_POINTER (0x0443), "/ucyrillic" },
  { GUINT_TO_POINTER (0x0951), "/udattadeva" },
  { GUINT_TO_POINTER (0x0171), "/udblacute" },
  { GUINT_TO_POINTER (0x0215), "/udblgrave" },
  { GUINT_TO_POINTER (0x0909), "/udeva" },
  { GUINT_TO_POINTER (0x00FC), "/udieresis" },
  { GUINT_TO_POINTER (0x01D8), "/udieresisacute" },
  { GUINT_TO_POINTER (0x1E73), "/udieresisbelow" },
  { GUINT_TO_POINTER (0x01DA), "/udieresiscaron" },
  { GUINT_TO_POINTER (0x04F1), "/udieresiscyrillic" },
  { GUINT_TO_POINTER (0x01DC), "/udieresisgrave" },
  { GUINT_TO_POINTER (0x01D6), "/udieresismacron" },
  { GUINT_TO_POINTER (0x1EE5), "/udotbelow" },
  { GUINT_TO_POINTER (0x00F9), "/ugrave" },
  { GUINT_TO_POINTER (0x0A89), "/ugujarati" },
  { GUINT_TO_POINTER (0x0A09), "/ugurmukhi" },
  { GUINT_TO_POINTER (0x3046), "/uhiragana" },
  { GUINT_TO_POINTER (0x1EE7), "/uhookabove" },
  { GUINT_TO_POINTER (0x01B0), "/uhorn" },
  { GUINT_TO_POINTER (0x1EE9), "/uhornacute" },
  { GUINT_TO_POINTER (0x1EF1), "/uhorndotbelow" },
  { GUINT_TO_POINTER (0x1EEB), "/uhorngrave" },
  { GUINT_TO_POINTER (0x1EED), "/uhornhookabove" },
  { GUINT_TO_POINTER (0x1EEF), "/uhorntilde" },
  { GUINT_TO_POINTER (0x0171), "/uhungarumlaut" },
  { GUINT_TO_POINTER (0x04F3), "/uhungarumlautcyrillic" },
  { GUINT_TO_POINTER (0x0217), "/uinvertedbreve" },
  { GUINT_TO_POINTER (0x30A6), "/ukatakana" },
  { GUINT_TO_POINTER (0xFF73), "/ukatakanahalfwidth" },
  { GUINT_TO_POINTER (0x0479), "/ukcyrillic" },
  { GUINT_TO_POINTER (0x315C), "/ukorean" },
  { GUINT_TO_POINTER (0x016B), "/umacron" },
  { GUINT_TO_POINTER (0x04EF), "/umacroncyrillic" },
  { GUINT_TO_POINTER (0x1E7B), "/umacrondieresis" },
  { GUINT_TO_POINTER (0x0A41), "/umatragurmukhi" },
  { GUINT_TO_POINTER (0xFF55), "/umonospace" },
  { GUINT_TO_POINTER (0x005F), "/underscore" },
  { GUINT_TO_POINTER (0x2017), "/underscoredbl" },
  { GUINT_TO_POINTER (0xFF3F), "/underscoremonospace" },
  { GUINT_TO_POINTER (0xFE33), "/underscorevertical" },
  { GUINT_TO_POINTER (0xFE4F), "/underscorewavy" },
  { GUINT_TO_POINTER (0x222A), "/union" },
  { GUINT_TO_POINTER (0x2200), "/universal" },
  { GUINT_TO_POINTER (0x0173), "/uogonek" },
  { GUINT_TO_POINTER (0x24B0), "/uparen" },
  { GUINT_TO_POINTER (0x2580), "/upblock" },
  { GUINT_TO_POINTER (0x05C4), "/upperdothebrew" },
  { GUINT_TO_POINTER (0x03C5), "/upsilon" },
  { GUINT_TO_POINTER (0x03CB), "/upsilondieresis" },
  { GUINT_TO_POINTER (0x03B0), "/upsilondieresistonos" },
  { GUINT_TO_POINTER (0x028A), "/upsilonlatin" },
  { GUINT_TO_POINTER (0x03CD), "/upsilontonos" },
  { GUINT_TO_POINTER (0x031D), "/uptackbelowcmb" },
  { GUINT_TO_POINTER (0x02D4), "/uptackmod" },
  { GUINT_TO_POINTER (0x0A73), "/uragurmukhi" },
  { GUINT_TO_POINTER (0x016F), "/uring" },
  { GUINT_TO_POINTER (0x045E), "/ushortcyrillic" },
  { GUINT_TO_POINTER (0x3045), "/usmallhiragana" },
  { GUINT_TO_POINTER (0x30A5), "/usmallkatakana" },
  { GUINT_TO_POINTER (0xFF69), "/usmallkatakanahalfwidth" },
  { GUINT_TO_POINTER (0x04AF), "/ustraightcyrillic" },
  { GUINT_TO_POINTER (0x04B1), "/ustraightstrokecyrillic" },
  { GUINT_TO_POINTER (0x0169), "/utilde" },
  { GUINT_TO_POINTER (0x1E79), "/utildeacute" },
  { GUINT_TO_POINTER (0x1E75), "/utildebelow" },
  { GUINT_TO_POINTER (0x098A), "/uubengali" },
  { GUINT_TO_POINTER (0x090A), "/uudeva" },
  { GUINT_TO_POINTER (0x0A8A), "/uugujarati" },
  { GUINT_TO_POINTER (0x0A0A), "/uugurmukhi" },
  { GUINT_TO_POINTER (0x0A42), "/uumatragurmukhi" },
  { GUINT_TO_POINTER (0x09C2), "/uuvowelsignbengali" },
  { GUINT_TO_POINTER (0x0942), "/uuvowelsigndeva" },
  { GUINT_TO_POINTER (0x0AC2), "/uuvowelsigngujarati" },
  { GUINT_TO_POINTER (0x09C1), "/uvowelsignbengali" },
  { GUINT_TO_POINTER (0x0941), "/uvowelsigndeva" },
  { GUINT_TO_POINTER (0x0AC1), "/uvowelsigngujarati" },
  { GUINT_TO_POINTER (0x0076), "/v" },
  { GUINT_TO_POINTER (0x0935), "/vadeva" },
  { GUINT_TO_POINTER (0x0AB5), "/vagujarati" },
  { GUINT_TO_POINTER (0x0A35), "/vagurmukhi" },
  { GUINT_TO_POINTER (0x30F7), "/vakatakana" },
  { GUINT_TO_POINTER (0x05D5), "/vav" },
  { GUINT_TO_POINTER (0xFB35), "/vavdagesh" },
  { GUINT_TO_POINTER (0xFB35), "/vavdagesh65" },
  { GUINT_TO_POINTER (0xFB35), "/vavdageshhebrew" },
  { GUINT_TO_POINTER (0x05D5), "/vavhebrew" },
  { GUINT_TO_POINTER (0xFB4B), "/vavholam" },
  { GUINT_TO_POINTER (0xFB4B), "/vavholamhebrew" },
  { GUINT_TO_POINTER (0x05F0), "/vavvavhebrew" },
  { GUINT_TO_POINTER (0x05F1), "/vavyodhebrew" },
  { GUINT_TO_POINTER (0x24E5), "/vcircle" },
  { GUINT_TO_POINTER (0x1E7F), "/vdotbelow" },
  { GUINT_TO_POINTER (0x0432), "/vecyrillic" },
  { GUINT_TO_POINTER (0x06A4), "/veharabic" },
  { GUINT_TO_POINTER (0xFB6B), "/vehfinalarabic" },
  { GUINT_TO_POINTER (0xFB6C), "/vehinitialarabic" },
  { GUINT_TO_POINTER (0xFB6D), "/vehmedialarabic" },
  { GUINT_TO_POINTER (0x30F9), "/vekatakana" },
  { GUINT_TO_POINTER (0x2640), "/venus" },
  { GUINT_TO_POINTER (0x007C), "/verticalbar" },
  { GUINT_TO_POINTER (0x030D), "/verticallineabovecmb" },
  { GUINT_TO_POINTER (0x0329), "/verticallinebelowcmb" },
  { GUINT_TO_POINTER (0x02CC), "/verticallinelowmod" },
  { GUINT_TO_POINTER (0x02C8), "/verticallinemod" },
  { GUINT_TO_POINTER (0x057E), "/vewarmenian" },
  { GUINT_TO_POINTER (0x028B), "/vhook" },
  { GUINT_TO_POINTER (0x30F8), "/vikatakana" },
  { GUINT_TO_POINTER (0x09CD), "/viramabengali" },
  { GUINT_TO_POINTER (0x094D), "/viramadeva" },
  { GUINT_TO_POINTER (0x0ACD), "/viramagujarati" },
  { GUINT_TO_POINTER (0x0983), "/visargabengali" },
  { GUINT_TO_POINTER (0x0903), "/visargadeva" },
  { GUINT_TO_POINTER (0x0A83), "/visargagujarati" },
  { GUINT_TO_POINTER (0xFF56), "/vmonospace" },
  { GUINT_TO_POINTER (0x0578), "/voarmenian" },
  { GUINT_TO_POINTER (0x309E), "/voicediterationhiragana" },
  { GUINT_TO_POINTER (0x30FE), "/voicediterationkatakana" },
  { GUINT_TO_POINTER (0x309B), "/voicedmarkkana" },
  { GUINT_TO_POINTER (0xFF9E), "/voicedmarkkanahalfwidth" },
  { GUINT_TO_POINTER (0x30FA), "/vokatakana" },
  { GUINT_TO_POINTER (0x24B1), "/vparen" },
  { GUINT_TO_POINTER (0x1E7D), "/vtilde" },
  { GUINT_TO_POINTER (0x028C), "/vturned" },
  { GUINT_TO_POINTER (0x3094), "/vuhiragana" },
  { GUINT_TO_POINTER (0x30F4), "/vukatakana" },
  { GUINT_TO_POINTER (0x0077), "/w" },
  { GUINT_TO_POINTER (0x1E83), "/wacute" },
  { GUINT_TO_POINTER (0x3159), "/waekorean" },
  { GUINT_TO_POINTER (0x308F), "/wahiragana" },
  { GUINT_TO_POINTER (0x30EF), "/wakatakana" },
  { GUINT_TO_POINTER (0xFF9C), "/wakatakanahalfwidth" },
  { GUINT_TO_POINTER (0x3158), "/wakorean" },
  { GUINT_TO_POINTER (0x308E), "/wasmallhiragana" },
  { GUINT_TO_POINTER (0x30EE), "/wasmallkatakana" },
  { GUINT_TO_POINTER (0x3357), "/wattosquare" },
  { GUINT_TO_POINTER (0x301C), "/wavedash" },
  { GUINT_TO_POINTER (0xFE34), "/wavyunderscorevertical" },
  { GUINT_TO_POINTER (0x0648), "/wawarabic" },
  { GUINT_TO_POINTER (0xFEEE), "/wawfinalarabic" },
  { GUINT_TO_POINTER (0x0624), "/wawhamzaabovearabic" },
  { GUINT_TO_POINTER (0xFE86), "/wawhamzaabovefinalarabic" },
  { GUINT_TO_POINTER (0x33DD), "/wbsquare" },
  { GUINT_TO_POINTER (0x24E6), "/wcircle" },
  { GUINT_TO_POINTER (0x0175), "/wcircumflex" },
  { GUINT_TO_POINTER (0x1E85), "/wdieresis" },
  { GUINT_TO_POINTER (0x1E87), "/wdotaccent" },
  { GUINT_TO_POINTER (0x1E89), "/wdotbelow" },
  { GUINT_TO_POINTER (0x3091), "/wehiragana" },
  { GUINT_TO_POINTER (0x2118), "/weierstrass" },
  { GUINT_TO_POINTER (0x30F1), "/wekatakana" },
  { GUINT_TO_POINTER (0x315E), "/wekorean" },
  { GUINT_TO_POINTER (0x315D), "/weokorean" },
  { GUINT_TO_POINTER (0x1E81), "/wgrave" },
  { GUINT_TO_POINTER (0x25E6), "/whitebullet" },
  { GUINT_TO_POINTER (0x25CB), "/whitecircle" },
  { GUINT_TO_POINTER (0x25D9), "/whitecircleinverse" },
  { GUINT_TO_POINTER (0x300E), "/whitecornerbracketleft" },
  { GUINT_TO_POINTER (0xFE43), "/whitecornerbracketleftvertical" },
  { GUINT_TO_POINTER (0x300F), "/whitecornerbracketright" },
  { GUINT_TO_POINTER (0xFE44), "/whitecornerbracketrightvertical" },
  { GUINT_TO_POINTER (0x25C7), "/whitediamond" },
  { GUINT_TO_POINTER (0x25C8), "/whitediamondcontainingblacksmalldiamond" },
  { GUINT_TO_POINTER (0x25BF), "/whitedownpointingsmalltriangle" },
  { GUINT_TO_POINTER (0x25BD), "/whitedownpointingtriangle" },
  { GUINT_TO_POINTER (0x25C3), "/whiteleftpointingsmalltriangle" },
  { GUINT_TO_POINTER (0x25C1), "/whiteleftpointingtriangle" },
  { GUINT_TO_POINTER (0x3016), "/whitelenticularbracketleft" },
  { GUINT_TO_POINTER (0x3017), "/whitelenticularbracketright" },
  { GUINT_TO_POINTER (0x25B9), "/whiterightpointingsmalltriangle" },
  { GUINT_TO_POINTER (0x25B7), "/whiterightpointingtriangle" },
  { GUINT_TO_POINTER (0x25AB), "/whitesmallsquare" },
  { GUINT_TO_POINTER (0x263A), "/whitesmilingface" },
  { GUINT_TO_POINTER (0x25A1), "/whitesquare" },
  { GUINT_TO_POINTER (0x2606), "/whitestar" },
  { GUINT_TO_POINTER (0x260F), "/whitetelephone" },
  { GUINT_TO_POINTER (0x3018), "/whitetortoiseshellbracketleft" },
  { GUINT_TO_POINTER (0x3019), "/whitetortoiseshellbracketright" },
  { GUINT_TO_POINTER (0x25B5), "/whiteuppointingsmalltriangle" },
  { GUINT_TO_POINTER (0x25B3), "/whiteuppointingtriangle" },
  { GUINT_TO_POINTER (0x3090), "/wihiragana" },
  { GUINT_TO_POINTER (0x30F0), "/wikatakana" },
  { GUINT_TO_POINTER (0x315F), "/wikorean" },
  { GUINT_TO_POINTER (0xFF57), "/wmonospace" },
  { GUINT_TO_POINTER (0x3092), "/wohiragana" },
  { GUINT_TO_POINTER (0x30F2), "/wokatakana" },
  { GUINT_TO_POINTER (0xFF66), "/wokatakanahalfwidth" },
  { GUINT_TO_POINTER (0x20A9), "/won" },
  { GUINT_TO_POINTER (0xFFE6), "/wonmonospace" },
  { GUINT_TO_POINTER (0x0E27), "/wowaenthai" },
  { GUINT_TO_POINTER (0x24B2), "/wparen" },
  { GUINT_TO_POINTER (0x1E98), "/wring" },
  { GUINT_TO_POINTER (0x02B7), "/wsuperior" },
  { GUINT_TO_POINTER (0x028D), "/wturned" },
  { GUINT_TO_POINTER (0x01BF), "/wynn" },
  { GUINT_TO_POINTER (0x0078), "/x" },
  { GUINT_TO_POINTER (0x033D), "/xabovecmb" },
  { GUINT_TO_POINTER (0x3112), "/xbopomofo" },
  { GUINT_TO_POINTER (0x24E7), "/xcircle" },
  { GUINT_TO_POINTER (0x1E8D), "/xdieresis" },
  { GUINT_TO_POINTER (0x1E8B), "/xdotaccent" },
  { GUINT_TO_POINTER (0x056D), "/xeharmenian" },
  { GUINT_TO_POINTER (0x03BE), "/xi" },
  { GUINT_TO_POINTER (0xFF58), "/xmonospace" },
  { GUINT_TO_POINTER (0x24B3), "/xparen" },
  { GUINT_TO_POINTER (0x02E3), "/xsuperior" },
  { GUINT_TO_POINTER (0x0079), "/y" },
  { GUINT_TO_POINTER (0x334E), "/yaadosquare" },
  { GUINT_TO_POINTER (0x09AF), "/yabengali" },
  { GUINT_TO_POINTER (0x00FD), "/yacute" },
  { GUINT_TO_POINTER (0x092F), "/yadeva" },
  { GUINT_TO_POINTER (0x3152), "/yaekorean" },
  { GUINT_TO_POINTER (0x0AAF), "/yagujarati" },
  { GUINT_TO_POINTER (0x0A2F), "/yagurmukhi" },
  { GUINT_TO_POINTER (0x3084), "/yahiragana" },
  { GUINT_TO_POINTER (0x30E4), "/yakatakana" },
  { GUINT_TO_POINTER (0xFF94), "/yakatakanahalfwidth" },
  { GUINT_TO_POINTER (0x3151), "/yakorean" },
  { GUINT_TO_POINTER (0x0E4E), "/yamakkanthai" },
  { GUINT_TO_POINTER (0x3083), "/yasmallhiragana" },
  { GUINT_TO_POINTER (0x30E3), "/yasmallkatakana" },
  { GUINT_TO_POINTER (0xFF6C), "/yasmallkatakanahalfwidth" },
  { GUINT_TO_POINTER (0x0463), "/yatcyrillic" },
  { GUINT_TO_POINTER (0x24E8), "/ycircle" },
  { GUINT_TO_POINTER (0x0177), "/ycircumflex" },
  { GUINT_TO_POINTER (0x00FF), "/ydieresis" },
  { GUINT_TO_POINTER (0x1E8F), "/ydotaccent" },
  { GUINT_TO_POINTER (0x1EF5), "/ydotbelow" },
  { GUINT_TO_POINTER (0x064A), "/yeharabic" },
  { GUINT_TO_POINTER (0x06D2), "/yehbarreearabic" },
  { GUINT_TO_POINTER (0xFBAF), "/yehbarreefinalarabic" },
  { GUINT_TO_POINTER (0xFEF2), "/yehfinalarabic" },
  { GUINT_TO_POINTER (0x0626), "/yehhamzaabovearabic" },
  { GUINT_TO_POINTER (0xFE8A), "/yehhamzaabovefinalarabic" },
  { GUINT_TO_POINTER (0xFE8B), "/yehhamzaaboveinitialarabic" },
  { GUINT_TO_POINTER (0xFE8C), "/yehhamzaabovemedialarabic" },
  { GUINT_TO_POINTER (0xFEF3), "/yehinitialarabic" },
  { GUINT_TO_POINTER (0xFEF4), "/yehmedialarabic" },
  { GUINT_TO_POINTER (0xFCDD), "/yehmeeminitialarabic" },
  { GUINT_TO_POINTER (0xFC58), "/yehmeemisolatedarabic" },
  { GUINT_TO_POINTER (0xFC94), "/yehnoonfinalarabic" },
  { GUINT_TO_POINTER (0x06D1), "/yehthreedotsbelowarabic" },
  { GUINT_TO_POINTER (0x3156), "/yekorean" },
  { GUINT_TO_POINTER (0x00A5), "/yen" },
  { GUINT_TO_POINTER (0xFFE5), "/yenmonospace" },
  { GUINT_TO_POINTER (0x3155), "/yeokorean" },
  { GUINT_TO_POINTER (0x3186), "/yeorinhieuhkorean" },
  { GUINT_TO_POINTER (0x05AA), "/yerahbenyomohebrew" },
  { GUINT_TO_POINTER (0x05AA), "/yerahbenyomolefthebrew" },
  { GUINT_TO_POINTER (0x044B), "/yericyrillic" },
  { GUINT_TO_POINTER (0x04F9), "/yerudieresiscyrillic" },
  { GUINT_TO_POINTER (0x3181), "/yesieungkorean" },
  { GUINT_TO_POINTER (0x3183), "/yesieungpansioskorean" },
  { GUINT_TO_POINTER (0x3182), "/yesieungsioskorean" },
  { GUINT_TO_POINTER (0x059A), "/yetivhebrew" },
  { GUINT_TO_POINTER (0x1EF3), "/ygrave" },
  { GUINT_TO_POINTER (0x01B4), "/yhook" },
  { GUINT_TO_POINTER (0x1EF7), "/yhookabove" },
  { GUINT_TO_POINTER (0x0575), "/yiarmenian" },
  { GUINT_TO_POINTER (0x0457), "/yicyrillic" },
  { GUINT_TO_POINTER (0x3162), "/yikorean" },
  { GUINT_TO_POINTER (0x262F), "/yinyang" },
  { GUINT_TO_POINTER (0x0582), "/yiwnarmenian" },
  { GUINT_TO_POINTER (0xFF59), "/ymonospace" },
  { GUINT_TO_POINTER (0x05D9), "/yod" },
  { GUINT_TO_POINTER (0xFB39), "/yoddagesh" },
  { GUINT_TO_POINTER (0xFB39), "/yoddageshhebrew" },
  { GUINT_TO_POINTER (0x05D9), "/yodhebrew" },
  { GUINT_TO_POINTER (0x05F2), "/yodyodhebrew" },
  { GUINT_TO_POINTER (0xFB1F), "/yodyodpatahhebrew" },
  { GUINT_TO_POINTER (0x3088), "/yohiragana" },
  { GUINT_TO_POINTER (0x3189), "/yoikorean" },
  { GUINT_TO_POINTER (0x30E8), "/yokatakana" },
  { GUINT_TO_POINTER (0xFF96), "/yokatakanahalfwidth" },
  { GUINT_TO_POINTER (0x315B), "/yokorean" },
  { GUINT_TO_POINTER (0x3087), "/yosmallhiragana" },
  { GUINT_TO_POINTER (0x30E7), "/yosmallkatakana" },
  { GUINT_TO_POINTER (0xFF6E), "/yosmallkatakanahalfwidth" },
  { GUINT_TO_POINTER (0x03F3), "/yotgreek" },
  { GUINT_TO_POINTER (0x3188), "/yoyaekorean" },
  { GUINT_TO_POINTER (0x3187), "/yoyakorean" },
  { GUINT_TO_POINTER (0x0E22), "/yoyakthai" },
  { GUINT_TO_POINTER (0x0E0D), "/yoyingthai" },
  { GUINT_TO_POINTER (0x24B4), "/yparen" },
  { GUINT_TO_POINTER (0x037A), "/ypogegrammeni" },
  { GUINT_TO_POINTER (0x0345), "/ypogegrammenigreekcmb" },
  { GUINT_TO_POINTER (0x01A6), "/yr" },
  { GUINT_TO_POINTER (0x1E99), "/yring" },
  { GUINT_TO_POINTER (0x02B8), "/ysuperior" },
  { GUINT_TO_POINTER (0x1EF9), "/ytilde" },
  { GUINT_TO_POINTER (0x028E), "/yturned" },
  { GUINT_TO_POINTER (0x3086), "/yuhiragana" },
  { GUINT_TO_POINTER (0x318C), "/yuikorean" },
  { GUINT_TO_POINTER (0x30E6), "/yukatakana" },
  { GUINT_TO_POINTER (0xFF95), "/yukatakanahalfwidth" },
  { GUINT_TO_POINTER (0x3160), "/yukorean" },
  { GUINT_TO_POINTER (0x046B), "/yusbigcyrillic" },
  { GUINT_TO_POINTER (0x046D), "/yusbigiotifiedcyrillic" },
  { GUINT_TO_POINTER (0x0467), "/yuslittlecyrillic" },
  { GUINT_TO_POINTER (0x0469), "/yuslittleiotifiedcyrillic" },
  { GUINT_TO_POINTER (0x3085), "/yusmallhiragana" },
  { GUINT_TO_POINTER (0x30E5), "/yusmallkatakana" },
  { GUINT_TO_POINTER (0xFF6D), "/yusmallkatakanahalfwidth" },
  { GUINT_TO_POINTER (0x318B), "/yuyekorean" },
  { GUINT_TO_POINTER (0x318A), "/yuyeokorean" },
  { GUINT_TO_POINTER (0x09DF), "/yyabengali" },
  { GUINT_TO_POINTER (0x095F), "/yyadeva" },
  { GUINT_TO_POINTER (0x007A), "/z" },
  { GUINT_TO_POINTER (0x0566), "/zaarmenian" },
  { GUINT_TO_POINTER (0x017A), "/zacute" },
  { GUINT_TO_POINTER (0x095B), "/zadeva" },
  { GUINT_TO_POINTER (0x0A5B), "/zagurmukhi" },
  { GUINT_TO_POINTER (0x0638), "/zaharabic" },
  { GUINT_TO_POINTER (0xFEC6), "/zahfinalarabic" },
  { GUINT_TO_POINTER (0xFEC7), "/zahinitialarabic" },
  { GUINT_TO_POINTER (0x3056), "/zahiragana" },
  { GUINT_TO_POINTER (0xFEC8), "/zahmedialarabic" },
  { GUINT_TO_POINTER (0x0632), "/zainarabic" },
  { GUINT_TO_POINTER (0xFEB0), "/zainfinalarabic" },
  { GUINT_TO_POINTER (0x30B6), "/zakatakana" },
  { GUINT_TO_POINTER (0x0595), "/zaqefgadolhebrew" },
  { GUINT_TO_POINTER (0x0594), "/zaqefqatanhebrew" },
  { GUINT_TO_POINTER (0x0598), "/zarqahebrew" },
  { GUINT_TO_POINTER (0x05D6), "/zayin" },
  { GUINT_TO_POINTER (0xFB36), "/zayindagesh" },
  { GUINT_TO_POINTER (0xFB36), "/zayindageshhebrew" },
  { GUINT_TO_POINTER (0x05D6), "/zayinhebrew" },
  { GUINT_TO_POINTER (0x3117), "/zbopomofo" },
  { GUINT_TO_POINTER (0x017E), "/zcaron" },
  { GUINT_TO_POINTER (0x24E9), "/zcircle" },
  { GUINT_TO_POINTER (0x1E91), "/zcircumflex" },
  { GUINT_TO_POINTER (0x0291), "/zcurl" },
  { GUINT_TO_POINTER (0x017C), "/zdot" },
  { GUINT_TO_POINTER (0x017C), "/zdotaccent" },
  { GUINT_TO_POINTER (0x1E93), "/zdotbelow" },
  { GUINT_TO_POINTER (0x0437), "/zecyrillic" },
  { GUINT_TO_POINTER (0x0499), "/zedescendercyrillic" },
  { GUINT_TO_POINTER (0x04DF), "/zedieresiscyrillic" },
  { GUINT_TO_POINTER (0x305C), "/zehiragana" },
  { GUINT_TO_POINTER (0x30BC), "/zekatakana" },
  { GUINT_TO_POINTER (0x0030), "/zero" },
  { GUINT_TO_POINTER (0x0660), "/zeroarabic" },
  { GUINT_TO_POINTER (0x09E6), "/zerobengali" },
  { GUINT_TO_POINTER (0x0966), "/zerodeva" },
  { GUINT_TO_POINTER (0x0AE6), "/zerogujarati" },
  { GUINT_TO_POINTER (0x0A66), "/zerogurmukhi" },
  { GUINT_TO_POINTER (0x0660), "/zerohackarabic" },
  { GUINT_TO_POINTER (0x2080), "/zeroinferior" },
  { GUINT_TO_POINTER (0xFF10), "/zeromonospace" },
  { GUINT_TO_POINTER (0xF730), "/zerooldstyle" },
  { GUINT_TO_POINTER (0x06F0), "/zeropersian" },
  { GUINT_TO_POINTER (0x2070), "/zerosuperior" },
  { GUINT_TO_POINTER (0x0E50), "/zerothai" },
  { GUINT_TO_POINTER (0xFEFF), "/zerowidthjoiner" },
  { GUINT_TO_POINTER (0x200C), "/zerowidthnonjoiner" },
  { GUINT_TO_POINTER (0x200B), "/zerowidthspace" },
  { GUINT_TO_POINTER (0x03B6), "/zeta" },
  { GUINT_TO_POINTER (0x3113), "/zhbopomofo" },
  { GUINT_TO_POINTER (0x056A), "/zhearmenian" },
  { GUINT_TO_POINTER (0x04C2), "/zhebrevecyrillic" },
  { GUINT_TO_POINTER (0x0436), "/zhecyrillic" },
  { GUINT_TO_POINTER (0x0497), "/zhedescendercyrillic" },
  { GUINT_TO_POINTER (0x04DD), "/zhedieresiscyrillic" },
  { GUINT_TO_POINTER (0x3058), "/zihiragana" },
  { GUINT_TO_POINTER (0x30B8), "/zikatakana" },
  { GUINT_TO_POINTER (0x05AE), "/zinorhebrew" },
  { GUINT_TO_POINTER (0x1E95), "/zlinebelow" },
  { GUINT_TO_POINTER (0xFF5A), "/zmonospace" },
  { GUINT_TO_POINTER (0x305E), "/zohiragana" },
  { GUINT_TO_POINTER (0x30BE), "/zokatakana" },
  { GUINT_TO_POINTER (0x24B5), "/zparen" },
  { GUINT_TO_POINTER (0x0290), "/zretroflexhook" },
  { GUINT_TO_POINTER (0x01B6), "/zstroke" },
  { GUINT_TO_POINTER (0x305A), "/zuhiragana" },
  { GUINT_TO_POINTER (0x30BA), "/zukatakana" },
  { NULL, NULL }
};

/*! \brief Initializes the glyph translation table.
 *  \par Function Description
 *  Initializes the glyph translation table
 *
 *  \return 0
 */
int f_print_initialize_glyph_table(void)
{
  struct glyph_list *g;

  /* Is the hash already intialized? */
  if(unicode_char_to_glyph != NULL) return 0;

  /* No, allocate hash table */
  unicode_char_to_glyph = g_hash_table_new_full (g_direct_hash,
						 g_direct_equal,
						 NULL,
						 NULL);

  /* insert all the entries, from glyph name mapping table */
  for(g = glyphs; g->name != NULL; g++) {
    g_hash_table_insert(unicode_char_to_glyph, g->key, g->name);
  }

  return 0;
}
