/**
 * @file libgalago/galago-person.c Galago Person API
 *
 * @Copyright (C) 2004-2005 Christian Hammond
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA  02111-1307, USA.
 */
#include <libgalago/galago-private.h>
#include <libgalago/galago-person.h>
#include <libgalago/galago-assert.h>
#include <libgalago/galago-context.h>
#include <libgalago/galago-context-priv.h>
#include <libgalago/galago-core.h>
#include <libgalago/galago-hashtable.h>
#include <libgalago/galago-marshal.h>
#include <libgalago/galago-object-utils.h>
#include <libgalago/galago-utils.h>
#include <libgalago/galago-utils-priv.h>
#include <stdio.h>
#include <string.h>

struct _GalagoPersonPrivate
{
	galago_bool native;
	galago_bool me;

	char *id;

	GalagoPhoto *photo;

	GalagoHashTable *accounts_table;
	GalagoHashTable *properties_table;

	GalagoList *accounts;
};

typedef struct
{
	char *username;
	const GalagoService *service;

} AccountCacheKey;

static GalagoPerson *_galago_person_new_common(const char *id,
											   galago_bool native,
											   galago_bool me,
											   const char *obj_path);
static void _galago_dbus_person_set_photo(GalagoPerson *person,
										  GalagoPhoto *photo);
static void _galago_dbus_person_set_property(GalagoPerson *person,
											 const char *name,
                                             GalagoValue *value);
static GalagoValue *_galago_dbus_person_get_property(
	const GalagoPerson *person, const char *name);


/**************************************************************************
 * Object/Class support
 **************************************************************************/
GALAGO_REGISTER_CLASS(galago_person, GalagoPerson, NULL,
					  GALAGO_DBUS_PERSON_INTERFACE);
DEPRECATE_OLD_GET_CLASS(galago_people, galago_person);

static void
cache_key_free(AccountCacheKey *key)
{
	if (key->username != NULL)
		free(key->username);

	free(key);
}

static unsigned int
cache_key_hash(const void *p)
{
	const AccountCacheKey *key = (AccountCacheKey *)p;

	return galago_str_hash(key->username) +
	       galago_str_hash(galago_service_get_id(key->service));
}

static galago_bool
cache_key_equal(const void *a, const void *b)
{
	const AccountCacheKey *ka = (AccountCacheKey *)a;
	const AccountCacheKey *kb = (AccountCacheKey *)b;

	if (ka == kb)
		return TRUE;

	if (strcmp(ka->username, kb->username))
		return FALSE;

	if (strcmp(galago_service_get_id(ka->service),
			   galago_service_get_id(kb->service)))
	{
		return FALSE;
	}

	return TRUE;
}

static void
galago_person_object_init(GalagoPerson *person)
{
	person->priv = galago_new0(GalagoPersonPrivate, 1);

	person->priv->accounts_table =
		galago_hash_table_new_full(cache_key_hash, cache_key_equal,
								   (GalagoFreeFunc)cache_key_free, NULL);
}

static void
galago_person_object_finalize(GalagoObject *object)
{
	GalagoPerson *person = (GalagoPerson *)object;

	if (person->priv->accounts != NULL)
	{
		galago_list_foreach(person->priv->accounts,
							(GalagoForEachFunc)galago_object_unref, NULL);
		galago_list_destroy(person->priv->accounts);
	}

	galago_hash_table_destroy(person->priv->accounts_table);

	if (person->priv->properties_table != NULL)
		galago_hash_table_destroy(person->priv->properties_table);

	galago_context_push(galago_object_get_context(person));
	galago_context_remove_person(person);
	galago_context_pop();

	if (person->priv->id != NULL)
		free(person->priv->id);

	free(person->priv);
}

static void
galago_person_dbus_message_append(DBusMessageIter *iter,
								  const GalagoObject *object)
{
	GalagoPerson *person = (GalagoPerson *)object;
	const char *obj_path, *id;
	galago_bool is_me;

	obj_path = galago_object_get_dbus_path(person);
	id       = galago_person_get_id(person);
	is_me    = galago_person_is_me(person);

	galago_dbus_message_iter_append_string(iter,  obj_path);
	galago_dbus_message_iter_append_string(iter,  id);
	galago_dbus_message_iter_append_boolean(iter, is_me);
}

static void *
galago_person_dbus_message_get(DBusMessageIter *iter)
{
	GalagoPerson *person;
	char *obj_path, *id;
	galago_bool me;

	galago_dbus_message_iter_get_string(iter, obj_path);
	dbus_message_iter_next(iter);

	galago_dbus_message_iter_get_string(iter, id);
	dbus_message_iter_next(iter);

	galago_dbus_message_iter_get_boolean(iter, me);

	person = _galago_person_new_common(id, FALSE, me, obj_path);

#if !GALAGO_CHECK_DBUS_VERSION(0, 30)
	dbus_free(obj_path);
	dbus_free(id);
#endif

	return person;
}

static void
_push_key(void *key, void *val, void *user_data)
{
	const char *name     = (const char *)key;
	GalagoValue *value   = (GalagoValue *)val;
	GalagoPerson *person = (GalagoPerson *)user_data;

	_galago_dbus_person_set_property(person, name, value);
}

static void
galago_person_dbus_push_full(GalagoObject *object)
{
	GalagoPerson *person = (GalagoPerson *)object;

	_galago_dbus_person_set_photo(person,
		galago_person_get_photo(person, FALSE));

	if (person->priv->properties_table != NULL)
	{
		galago_hash_table_foreach(person->priv->properties_table,
								  _push_key, person);
	}

	/* NOTE: Accounts are already handled by this point. */
}

static void
galago_person_class_init(GalagoObjectClass *klass)
{
	klass->finalize            = galago_person_object_finalize;
	klass->dbus_message_append = galago_person_dbus_message_append;
	klass->dbus_message_get    = galago_person_dbus_message_get;
	klass->dbus_push_full      = galago_person_dbus_push_full;

	galago_signal_register(klass->signal_context, "photo-set",
						   galago_marshal_VOID__POINTER, 1,
						   GALAGO_TYPE_OBJECT);
	galago_signal_register(klass->signal_context, "updated",
						   galago_marshal_VOID, 0);
}


/**************************************************************************
 * GalagoPerson API
 **************************************************************************/
static GalagoPerson *
_galago_person_new_common(const char *id, galago_bool native,
						  galago_bool me, const char *obj_path)
{
	GalagoPerson *person;

	galago_return_val_if_fail(galago_is_initted(),       NULL);
	galago_return_val_if_fail(id != NULL && *id != '\0', NULL);

	person = galago_context_get_person(id, native);

	if (person == NULL)
	{
		const char *obj_prefix;

		person = galago_object_new(GALAGO_CLASS_PERSON);

		person->priv->id     = strdup(id);
		person->priv->native = native;
		person->priv->me     = me;

		if (obj_path != NULL)
		{
			galago_object_set_dbus_path(person, obj_path);
		}
		else if ((obj_prefix = galago_context_get_obj_path_prefix()) != NULL)
		{
			char *new_obj_path;
			size_t len;
			const char *escaped_id;

			if (!strcmp(id, GALAGO_ME_ID))
				escaped_id = GALAGO_ME_ID;
			else
				escaped_id = galago_dbus_normalize_name(id);

			len = strlen(obj_prefix) + strlen("/people/") +
			      strlen(escaped_id) + 1;

			new_obj_path = galago_new(char, len);
			snprintf(new_obj_path, len, "%s/people/%s", obj_prefix,
					 escaped_id);

			galago_object_set_dbus_path(person, new_obj_path);

			free(new_obj_path);
		}

		galago_context_add_person(person);
	}

	return person;
}

GalagoPerson *
galago_person_new(const char *id, galago_bool native)
{
	galago_return_val_if_fail(galago_is_initted(),       NULL);
	galago_return_val_if_fail(id != NULL && *id != '\0', NULL);

	return _galago_person_new_common(id, native, FALSE, NULL);
}

GalagoPerson *
galago_person_me_new(galago_bool native)
{
	return _galago_person_new_common(GALAGO_ME_ID, native, TRUE, NULL);
}

void
galago_person_set_me(GalagoPerson *person)
{
	galago_return_if_fail(person != NULL);
	galago_return_if_fail(GALAGO_IS_PERSON(person));

	if (galago_person_is_me(person))
		return;

	person->priv->me = TRUE;

	free(person->priv->id);

	person->priv->id = strdup(GALAGO_ME_ID);
}

galago_bool
galago_person_is_me(const GalagoPerson *person)
{
	galago_return_val_if_fail(person != NULL,           FALSE);
	galago_return_val_if_fail(GALAGO_IS_PERSON(person), FALSE);

	return person->priv->me;
}

const char *
galago_person_get_id(const GalagoPerson *person)
{
	galago_return_val_if_fail(person != NULL,           NULL);
	galago_return_val_if_fail(GALAGO_IS_PERSON(person), NULL);

	return person->priv->id;
}

galago_bool
galago_person_is_native(const GalagoPerson *person)
{
	galago_return_val_if_fail(person != NULL,           FALSE);
	galago_return_val_if_fail(GALAGO_IS_PERSON(person), FALSE);

	return person->priv->native;
}

const char *
galago_person_get_display_name(const GalagoPerson *person)
{
	static char buffer[BUFSIZ];
	const char *first_name, *last_name;

	galago_return_val_if_fail(person != NULL,           NULL);
	galago_return_val_if_fail(GALAGO_IS_PERSON(person), NULL);

	/* XXX This is all a hack, but for now, it should work. Improve this! */

	first_name = galago_person_get_property_string(person,
		GALAGO_PERSON_PROP_FIRST_NAME);
	last_name = galago_person_get_property_string(person,
		GALAGO_PERSON_PROP_LAST_NAME);

	if (first_name != NULL || last_name != NULL)
	{
		snprintf(buffer, sizeof(buffer), "%s%s%s",
				 (first_name == NULL ? "" : first_name),
				 (first_name != NULL && last_name != NULL ? " " : ""),
				 (last_name == NULL ? "" : last_name));

		return buffer;
	}

	return NULL;
}

void
galago_person_set_photo(GalagoPerson *person, GalagoPhoto *photo)
{
	galago_return_if_fail(person != NULL);
	galago_return_if_fail(GALAGO_IS_PERSON(person));
	galago_return_if_fail(photo == NULL || GALAGO_IS_PHOTO(photo));

	if (person->priv->photo == photo)
		return;

	if (person->priv->photo != NULL)
	{
		GalagoPhoto *old_photo = person->priv->photo;

		person->priv->photo = NULL;

		galago_object_unref(old_photo);
	}

	person->priv->photo = photo;

	if (galago_person_is_native(person))
		_galago_dbus_person_set_photo(person, photo);

	galago_signal_emit(person, "photo-set", photo);
	galago_signal_emit(person, "updated");
}

GalagoPhoto *
galago_person_get_photo(const GalagoPerson *person, galago_bool query)
{
	galago_return_val_if_fail(person != NULL,           NULL);
	galago_return_val_if_fail(GALAGO_IS_PERSON(person), NULL);

	if (person->priv->photo == NULL && query &&
		!galago_person_is_native(person) && !galago_is_daemon() &&
		galago_is_connected())
	{
		GalagoSignalContext *signal_context = GALAGO_SIGNAL_CONTEXT(person);

		galago_signal_context_freeze(signal_context);
		person->priv->photo =
			galago_dbus_send_message_with_reply(person, "GetPhoto",
				galago_value_new(GALAGO_TYPE_OBJECT, NULL, GALAGO_CLASS_PHOTO),
				NULL);
		galago_signal_context_thaw(signal_context);
	}

	return person->priv->photo;
}

galago_bool
galago_person_has_accounts(const GalagoPerson *person, galago_bool query)
{
	galago_return_val_if_fail(person != NULL,           FALSE);
	galago_return_val_if_fail(GALAGO_IS_PERSON(person), FALSE);

	if (query && !galago_person_is_native(person) && !galago_is_daemon() &&
		galago_is_connected())
	{
		size_t num_accounts = (size_t)galago_dbus_send_message_with_reply(
			person, "GetAccountsCount",
			galago_value_new(GALAGO_TYPE_UINT, NULL, NULL),
			NULL);

		return (num_accounts > 0);
	}
	else
	{
		return (galago_person_get_accounts(person, query) != NULL);
	}
}

const GalagoList *
galago_person_get_accounts(const GalagoPerson *person, galago_bool query)
{
	galago_return_val_if_fail(person != NULL,           NULL);
	galago_return_val_if_fail(GALAGO_IS_PERSON(person), NULL);

	if (query && !galago_person_is_native(person) && !galago_is_daemon() &&
		galago_is_connected())
	{
		GalagoList *temp;

		galago_context_push(galago_object_get_context(person));
		/* galago_signal_context_freeze(signal_context); */

		temp = galago_dbus_send_message_with_reply(person, "GetAccounts",
			galago_value_new_list(GALAGO_TYPE_OBJECT, NULL,
								  GALAGO_CLASS_ACCOUNT),
			NULL);
		galago_list_destroy(temp);

		/* galago_signal_context_thaw(signal_context); */
		galago_context_pop();
	}

	return person->priv->accounts;
}

GalagoAccount *
galago_person_get_priority_account(const GalagoPerson *person)
{
	const GalagoList *l, *accounts;
	GalagoPresence *priority_presence = NULL;
	GalagoAccount *account = NULL;

	galago_return_val_if_fail(person != NULL,           NULL);
	galago_return_val_if_fail(GALAGO_IS_PERSON(person), NULL);

	for (l = accounts = galago_person_get_accounts(person, TRUE);
		 l != NULL;
		 l = l->next)
	{
		GalagoAccount *account = (GalagoAccount *)l->data;
		GalagoPresence *presence = galago_account_get_presence(account, TRUE);

		if (galago_presence_compare(priority_presence, presence) > 0)
			priority_presence = presence;
	}

	if (priority_presence != NULL)
	{
		account = galago_presence_get_account(priority_presence);
	}
	else if (accounts != NULL)
	{
		account = (GalagoAccount *)accounts->data;
	}

	return account;
}

void
galago_person_add_account(GalagoPerson *person, GalagoAccount *account)
{
	const char *username;
	GalagoService *service;
	AccountCacheKey *key;

	galago_return_if_fail(person  != NULL);
	galago_return_if_fail(account != NULL);
	galago_return_if_fail(GALAGO_IS_PERSON(person));
	galago_return_if_fail(GALAGO_IS_ACCOUNT(account));

	username = galago_account_get_username(account);
	service  = galago_account_get_service(account);

	if (galago_person_get_account(person, service, username, FALSE) != NULL)
	{
#if 0
		galago_log_warning("An account with username %s on service %s has "
						   "already been added to person %s\n",
						   username, galago_service_get_id(service),
						   galago_person_get_id(person));
#endif

		return;
	}

	key = galago_new0(AccountCacheKey, 1);
	key->username = strdup(galago_service_normalize(service, username));
	key->service  = service;

	galago_hash_table_insert(person->priv->accounts_table, key, account);

	person->priv->accounts = galago_list_append(person->priv->accounts,
												account);

#if 0
	galago_signal_emit(service, "account-added", account);
	galago_signal_emit(service, "updated");
#endif
}

void
galago_person_remove_account(GalagoPerson *person, GalagoAccount *account)
{
	AccountCacheKey key;
	GalagoService *service;
	const char *username;

	galago_return_if_fail(person  != NULL);
	galago_return_if_fail(account != NULL);
	galago_return_if_fail(GALAGO_IS_PERSON(person));
	galago_return_if_fail(GALAGO_IS_ACCOUNT(account));

	service  = galago_account_get_service(account);
	username = galago_account_get_username(account);

	key.username = strdup(galago_service_normalize(service, username));
	key.service  = galago_account_get_service(account);

	galago_hash_table_remove(person->priv->accounts_table, &key);

	free(key.username);

	person->priv->accounts = galago_list_remove(person->priv->accounts,
												account);

#if 0
	galago_signal_emit(person, "account-removed", account);
	galago_signal_emit(person, "updated");
#endif
}

GalagoAccount *
galago_person_get_account(const GalagoPerson *person,
						  const GalagoService *service, const char *username,
						  galago_bool query)
{
	GalagoAccount *account;
	AccountCacheKey key;

	galago_return_val_if_fail(person   != NULL,           NULL);
	galago_return_val_if_fail(service  != NULL,           NULL);
	galago_return_val_if_fail(username != NULL,           NULL);
	galago_return_val_if_fail(GALAGO_IS_PERSON(person),   NULL);
	galago_return_val_if_fail(GALAGO_IS_SERVICE(service), NULL);
	galago_return_val_if_fail(galago_person_is_native(person) ==
							  galago_service_is_native(service), NULL);

	key.username = strdup(galago_service_normalize(service, username));
	key.service  = service;

	account = galago_hash_table_lookup(person->priv->accounts_table, &key);

	free(key.username);

	if (account == NULL && query &&
		!galago_person_is_native(person) && !galago_is_daemon() &&
		galago_is_connected())
	{
		galago_context_push(galago_object_get_context(person));
		/* galago_signal_context_freeze(signal_context); */

		account = galago_dbus_send_message_with_reply(person, "GetAccount",
			galago_value_new(GALAGO_TYPE_OBJECT, NULL, GALAGO_CLASS_ACCOUNT),
			galago_value_new(GALAGO_TYPE_OBJECT, &service,
							 GALAGO_CLASS_SERVICE),
			galago_value_new(GALAGO_TYPE_STRING, &username, NULL),
			NULL);

		/* galago_signal_context_thaw(signal_context); */
		galago_context_pop();
	}

	return account;
}

void
galago_person_set_property_string(GalagoPerson *person, const char *name,
								  const char *str_value)
{
	galago_return_if_fail(person    != NULL);
	galago_return_if_fail(name      != NULL && *name      != '\0');
	galago_return_if_fail(str_value != NULL && *str_value != '\0');
	galago_return_if_fail(GALAGO_IS_PERSON(person));

	galago_person_set_property(person, name,
		galago_value_new(GALAGO_TYPE_STRING, &str_value, NULL));
}

void
galago_person_set_property_bool(GalagoPerson *person, const char *name,
								galago_bool bool_value)
{
	galago_return_if_fail(person != NULL);
	galago_return_if_fail(name   != NULL && *name  != '\0');
	galago_return_if_fail(GALAGO_IS_PERSON(person));

	galago_person_set_property(person, name,
		galago_value_new(GALAGO_TYPE_BOOLEAN, &bool_value, NULL));
}

void
galago_person_set_property_uint32(GalagoPerson *person, const char *name,
								  dbus_uint32_t uint_value)
{
	galago_return_if_fail(person != NULL);
	galago_return_if_fail(name   != NULL && *name  != '\0');
	galago_return_if_fail(GALAGO_IS_PERSON(person));

	galago_person_set_property(person, name,
		galago_value_new(GALAGO_TYPE_UINT, &uint_value, NULL));
}

#define IS_TYPE_INTEGER(type) \
	((type) == GALAGO_TYPE_UINT    || \
	 (type) == GALAGO_TYPE_INT     || \
	 (type) == GALAGO_TYPE_SHORT   || \
	 (type) == GALAGO_TYPE_USHORT  || \
	 (type) == GALAGO_TYPE_LONG    || \
	 (type) == GALAGO_TYPE_ULONG)

void
galago_person_set_property(GalagoPerson *person, const char *name,
						   GalagoValue *value)
{
	GalagoType value_type;

	galago_return_if_fail(person != NULL);
	galago_return_if_fail(name   != NULL && *name  != '\0');
	galago_return_if_fail(value  != NULL);
	galago_return_if_fail(GALAGO_IS_PERSON(person));

	value_type = galago_value_get_type(value);

	galago_return_if_fail(value_type == GALAGO_TYPE_STRING  ||
						  value_type == GALAGO_TYPE_BOOLEAN ||
						  IS_TYPE_INTEGER(value_type));

	if (person->priv->properties_table == NULL)
	{
		person->priv->properties_table =
			galago_hash_table_new_full(galago_str_hash, galago_str_equal,
									   free,
									   (GalagoFreeFunc)galago_value_destroy);
	}

	galago_hash_table_replace(person->priv->properties_table,
							  galago_str_lower(name), value);

	if (galago_person_is_native(person))
		_galago_dbus_person_set_property(person, name, value);
}

galago_bool
galago_person_remove_property(GalagoPerson *person, const char *name)
{
	char *temp;

	galago_return_val_if_fail(person != NULL,                  FALSE);
	galago_return_val_if_fail(name   != NULL && *name != '\0', FALSE);
	galago_return_val_if_fail(GALAGO_IS_PERSON(person),        FALSE);

	if (person->priv->properties_table == NULL)
		return TRUE;

	temp = galago_str_lower(name);
	galago_hash_table_remove(person->priv->properties_table, temp);
	free(temp);

	if (galago_person_is_native(person) && galago_is_connected() &&
		galago_core_is_feed())
	{
		galago_dbus_send_message(person, "RemoveProperty",
			galago_value_new(GALAGO_TYPE_STRING, &name, NULL),
			NULL);
	}

	return TRUE;
}

const char *
galago_person_get_property_string(const GalagoPerson *person,
								  const char *name)
{
	const GalagoValue *value;

	galago_return_val_if_fail(person != NULL,                  NULL);
	galago_return_val_if_fail(name   != NULL && *name != '\0', NULL);
	galago_return_val_if_fail(GALAGO_IS_PERSON(person),        NULL);

	value = galago_person_get_property(person, name);

	if (value == NULL)
		return NULL;

	galago_return_val_if_fail(
		galago_value_get_type(value) == GALAGO_TYPE_STRING, NULL);

	return galago_value_get_string(value);
}

galago_bool
galago_person_get_property_bool(const GalagoPerson *person, const char *name)
{
	const GalagoValue *value;

	galago_return_val_if_fail(person != NULL,                  FALSE);
	galago_return_val_if_fail(name   != NULL && *name != '\0', FALSE);
	galago_return_val_if_fail(GALAGO_IS_PERSON(person),        FALSE);

	value = galago_person_get_property(person, name);

	if (value == NULL)
		return FALSE;

	galago_return_val_if_fail(
		galago_value_get_type(value) == GALAGO_TYPE_BOOLEAN, FALSE);

	return galago_value_get_boolean(value);
}

dbus_uint32_t
galago_person_get_property_uint32(const GalagoPerson *person,
								  const char *name)
{
	const GalagoValue *value;

	galago_return_val_if_fail(person != NULL,                  0);
	galago_return_val_if_fail(name   != NULL && *name != '\0', 0);
	galago_return_val_if_fail(GALAGO_IS_PERSON(person),        0);

	value = galago_person_get_property(person, name);

	if (value == NULL)
		return 0;

	galago_return_val_if_fail(
		galago_value_get_type(value) == GALAGO_TYPE_UINT, 0);

	return galago_value_get_uint(value);
}

const GalagoValue *
galago_person_get_property(const GalagoPerson *person, const char *name)
{
	GalagoValue *value = NULL;
	char *temp;

	galago_return_val_if_fail(person != NULL,                  0);
	galago_return_val_if_fail(name   != NULL && *name != '\0', 0);
	galago_return_val_if_fail(GALAGO_IS_PERSON(person),        0);

	temp = galago_str_lower(name);

	if (person->priv->properties_table != NULL)
		value = galago_hash_table_lookup(person->priv->properties_table, temp);

	if (value == NULL && !galago_person_is_native(person))
		value = _galago_dbus_person_get_property(person, temp);

	free(temp);

	return value;
}


/**************************************************************************
 * D-BUS Functions
 **************************************************************************/
static void
_galago_dbus_person_set_photo(GalagoPerson *person, GalagoPhoto *photo)
{
	if (photo == NULL)
	{
		galago_dbus_send_message(person, "UnsetPhoto", NULL);
	}
	else
	{
		galago_dbus_send_message(person, "SetPhoto",
			galago_value_new(GALAGO_TYPE_OBJECT, &photo, GALAGO_CLASS_PHOTO),
			NULL);
	}
}

static void
_galago_dbus_person_set_property(GalagoPerson *person, const char *name,
								 GalagoValue *value)
{
	GalagoValue *new_value;
	GalagoType type;

	if (!galago_is_connected() || !galago_core_is_feed())
		return;

	type = galago_value_get_type(value);

	switch (type)
	{
		case GALAGO_TYPE_STRING:
		{
			const char *str = galago_value_get_string(value);
			new_value = galago_value_new(GALAGO_TYPE_STRING, &str, NULL);
			break;
		}

		case GALAGO_TYPE_BOOLEAN:
		{
			galago_bool b = galago_value_get_boolean(value);
			new_value = galago_value_new(GALAGO_TYPE_BOOLEAN, &b, NULL);
			break;
		}

		case GALAGO_TYPE_UINT:
		{
			unsigned int i = galago_value_get_uint(value);
			new_value = galago_value_new(GALAGO_TYPE_UINT, &i, NULL);
			break;
		}

		default:
			galago_log_error("Unknown property type %d for property %s\n",
							 type, name);
			return;
	}

	galago_dbus_send_message(person, "SetProperty",
							 galago_value_new(GALAGO_TYPE_STRING, &name, NULL),
							 new_value, NULL);
}

static GalagoValue *
_galago_dbus_person_get_property(const GalagoPerson *person, const char *name)
{
	DBusConnection *dbus_conn;
	DBusMessage *message, *reply;
	DBusMessageIter iter;
	DBusError error;
	GalagoValue *value = NULL;

	if (!galago_is_connected())
		return NULL;

	dbus_conn = galago_core_get_dbus_conn();

	message = galago_dbus_message_new_method_call(person, "GetProperty",
												  TRUE, &iter);

	galago_return_val_if_fail(message != NULL, NULL);

	galago_dbus_message_iter_append_string(&iter, name);

	dbus_error_init(&error);

	reply = dbus_connection_send_with_reply_and_block(dbus_conn, message,
													  -1, &error);

	dbus_message_unref(message);

	if (dbus_error_is_set(&error))
	{
		if (!dbus_error_has_name(&error, GALAGO_DBUS_ERROR_INVALID_PROPERTY))
		{
			galago_log_error("Error sending Person.GetProperty(%s, %s): %s\n",
							 galago_person_get_id(person), name,
							 error.message);
		}

		goto exit;
	}

	dbus_message_iter_init(reply, &iter);

	int type = dbus_message_iter_get_arg_type(&iter);

	if (type == DBUS_TYPE_STRING)
	{
		const char *data;
		galago_dbus_message_iter_get_string(&iter, data);
		value = galago_value_new(GALAGO_TYPE_STRING, &data, NULL);
	}
	else if (type == DBUS_TYPE_BOOLEAN)
	{
		galago_bool data;
		galago_dbus_message_iter_get_boolean(&iter, data);
		value = galago_value_new(GALAGO_TYPE_BOOLEAN, &data, NULL);
	}
	else if (type == DBUS_TYPE_UINT32)
	{
		int data;
		galago_dbus_message_iter_get_uint32(&iter, data);
		value = galago_value_new(GALAGO_TYPE_UINT, &data, NULL);
	}
	else
	{
		galago_log_fatal("Unknown property type %d for property %s\n",
						 type, name);
	}

	dbus_message_unref(reply);

exit:
	dbus_error_free(&error);

	return value;
}
