/**
 * @file libgalago/galago-object.h Galago Object API
 *
 * @Copyright (C) 2004-2005 Christian Hammond
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA  02111-1307, USA.
 */
#ifndef _GALAGO_OBJECT_H_
#define _GALAGO_OBJECT_H_

#include <libgalago/galago-types.h>

typedef struct _GalagoObjectClass         GalagoObjectClass;
typedef struct _GalagoObjectClassInfo     GalagoObjectClassInfo;
typedef struct _GalagoObjectClassPrivate  GalagoObjectClassPrivate;
typedef struct _GalagoObject              GalagoObject;
typedef struct _GalagoObjectPrivate       GalagoObjectPrivate;

typedef void (*GalagoObjectClassInitFunc)(GalagoObjectClass *klass);
typedef void (*GalagoObjectClassFinalizeFunc)(GalagoObjectClass *klass);
typedef void (*GalagoObjectInitFunc)(GalagoObject *object);

typedef enum
{
	GALAGO_FLAG_FINALIZING = 1 << 1 /**< Object is being finalized. */

} GalagoObjectFlags;

typedef enum
{
	GALAGO_FLAG_ABSTRACT = 1 << 1 /**< Abstract object. */

} GalagoObjectClassFlags;

#include <libgalago/galago-context-base.h>
#include <libgalago/galago-signals.h>


/**
 * A class.
 */
struct _GalagoObjectClass
{
	unsigned long magic;

	GalagoObjectClassPrivate *priv;

	GalagoSignalContext *signal_context;

	void (*finalize)(GalagoObject *object);

	void (*dbus_message_append)(DBusMessageIter *iter,
								const GalagoObject *object);
	void *(*dbus_message_get)(DBusMessageIter *iter);
	void (*dbus_push_full)(GalagoObject *object);

	void (*reserved_1)(void);
	void (*reserved_2)(void);
	void (*reserved_3)(void);
	void (*reserved_4)(void);
};

/**
 * Class information.
 */
struct _GalagoObjectClassInfo
{
	unsigned long flags;
	size_t class_size;
	GalagoObjectClassInitFunc class_init;
	GalagoObjectClassFinalizeFunc class_finalize;
	size_t object_size;
	GalagoObjectInitFunc object_init;
};

/**
 * An object.
 */
struct _GalagoObject
{
	unsigned long magic;

	GalagoObjectPrivate *priv;

	void (*reserved_1)(void);
	void (*reserved_2)(void);
	void (*reserved_3)(void);
	void (*reserved_4)(void);
};

#define GALAGO_OBJECT_MAGIC 0xB00F
#define GALAGO_CLASS_MAGIC  0xB0B0

#define GALAGO_IS_OBJECT(ptr) \
	((ptr) != NULL && ((GalagoObject *)(ptr))->magic == GALAGO_OBJECT_MAGIC)

#define GALAGO_IS_CLASS(ptr) \
	((ptr) != NULL && ((GalagoObjectClass *)(ptr))->magic == GALAGO_CLASS_MAGIC)

#define GALAGO_OBJECT(ptr) \
	((ptr) == NULL ? NULL : \
	 (GALAGO_IS_OBJECT(ptr) ? (GalagoObject *)(ptr) : NULL))

#define GALAGO_OBJECT_CLASS(obj) (galago_object_get_class(obj))

#define GALAGO_OBJECT_CAST(obj, klass, type) \
	((obj) == NULL ? NULL : (type *)galago_object_check_cast((obj), (klass)))

#define GALAGO_OBJECT_GET_FLAGS(obj) (galago_object_get_flags(obj))
#define GALAGO_OBJECT_SET_FLAGS(obj, flags) \
	galago_object_set_flags((obj), (flags))

#define GALAGO_OBJECT_SET_FLAG(obj, flag) \
	GALAGO_OBJECT_SET_FLAGS((obj), GALAGO_OBJECT_GET_FLAGS(obj) | (flag))
#define GALAGO_OBJECT_UNSET_FLAG(obj, flag) \
	GALAGO_OBJECT_SET_FLAGS((obj), GALAGO_OBJECT_GET_FLAGS(obj) & ~(flag))

#define GALAGO_OBJECT_HAS_FLAG(obj, flag) \
	((GALAGO_OBJECT_GET_FLAGS(obj) & (flag)) == (flag))

#ifdef __cplusplus
extern "C" {
#endif

/**************************************************************************/
/** @name Class API                                                       */
/**************************************************************************/
/*@{*/

/**
 * Registers a new type of class.
 *
 * @param parent_class The optional parent class.
 * @param name         The name of the class.
 * @param dbus_iface   The optional D-BUS interface for this class.
 * @param info         The static class information structure.
 *
 * @return The new class, or NULL on error.
 */
GalagoObjectClass *galago_class_register(GalagoObjectClass *parent_class,
										 const char *name,
										 const char *dbus_iface,
										 const GalagoObjectClassInfo *info);

/**
 * Returns a class's parent class.
 *
 * @param klass The class.
 *
 * @return The parent class.
 */
GalagoObjectClass *galago_class_get_parent(const GalagoObjectClass *klass);

/**
 * Returns a class's name.
 *
 * @param klass The class.
 *
 * @return The class's name.
 */
const char *galago_class_get_name(const GalagoObjectClass *klass);

/**
 * Returns the D-BUS interface of a class.
 *
 * @param klass The class.
 *
 * @return The D-BUS interface.
 */
const char *galago_class_get_dbus_iface(const GalagoObjectClass *klass);

/**
 * Returns the signal context for a class.
 *
 * @param klass The class.
 *
 * @return The class's signal context, if set.
 */
GalagoSignalContext *galago_class_get_signal_context(
	const GalagoObjectClass *klass);

/*@}*/

/**************************************************************************/
/** @name Object API                                                      */
/**************************************************************************/
/*@{*/

/**
 * Creates an object.
 *
 * @param klass  The object's class.
 */
void *galago_object_new(GalagoObjectClass *klass);

/**
 * References an object.
 *
 * @param object The object to reference.
 *
 * @return The object.
 */
void *galago_object_ref(void *object);

/**
 * Unreferences an object.
 *
 * @param object The object to unreference.
 *
 * @return The object, or NULL if its reference count reached 0.
 */
void *galago_object_unref(void *object);

/**
 * Returns the object's class.
 *
 * @param object The object.
 *
 * @return The object's class.
 */
GalagoObjectClass *galago_object_get_class(const void *object);

/**
 * Sets the flags for an object.
 *
 * @param object The object.
 * @param flags  The flags.
 */
void galago_object_set_flags(void *object, unsigned long flags);

/**
 * Returns the flags for an object.
 *
 * @param object The object.
 *
 * @return The flags for the object.
 */
unsigned long galago_object_get_flags(const void *object);

/**
 * Sets the D-BUS object path of an object.
 *
 * @param object   The object.
 * @param obj_path The object path.
 */
void galago_object_set_dbus_path(void *object, const char *obj_path);

/**
 * Returns the D-BUS object path of an object.
 *
 * @param object The object.
 *
 * @return The object path.
 */
const char *galago_object_get_dbus_path(const void *object);

/**
 * Sets whether or not this object is watched for events.
 *
 * @param object The object.
 * @param watch  TRUE if this object should be watched, or FALSE.
 */
void galago_object_set_watch(void *object, galago_bool watch);

/**
 * Returns whether or not an object is watched for events.
 *
 * @param object The object.
 *
 * @return TRUE if this object is being watched, or FALSE.
 */
galago_bool galago_object_is_watched(const void *object);

/**
 * Sets custom data on an object.
 *
 * @param object The object.
 * @param name   The property name.
 * @param value  The data.
 */
void galago_object_set_data(void *object, const char *name, void *value);

/**
 * Returns custom data from an object.
 *
 * @param object The object.
 * @param name   The propert name.
 *
 * @return The data.
 */
void *galago_object_get_data(const void *object, const char *name);

/**
 * Returns the object's context.
 *
 * @param object The object.
 *
 * @return The object's context.
 */
GalagoContext *galago_object_get_context(const void *object);

/**
 * Casts the specified pointer to a GalagoObject, if the pointer is an object
 * and matches the type specified.
 *
 * @param ptr   The pointer to the object to cast.
 * @param klass The class type.
 *
 * @return @a ptr casted into a GalagoObject, if @a ptr is an object and if
 *         its class type matches @a klass.
 */
GalagoObject *galago_object_check_cast(const void *ptr,
									   const GalagoObjectClass *klass);

/*@}*/

#ifdef __cplusplus
}
#endif

#endif /* _GALAGO_OBJECT_H_ */
