/**
 * @file libgalago/galago-avatar.h Galago Avatar API
 *
 * @Copyright (C) 2004-2005 Christian Hammond
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA  02111-1307, USA.
 */
#ifndef _GALAGO_AVATAR_H_
#define _GALAGO_AVATAR_H_

#include <stdlib.h>

typedef struct _GalagoAvatar        GalagoAvatar;
typedef struct _GalagoAvatarClass   GalagoAvatarClass;
typedef struct _GalagoAvatarPrivate GalagoAvatarPrivate;

#include <libgalago/galago-account.h>
#include <libgalago/galago-image.h>

/**
 * An avatar for a user.
 *
 * The avatar is a visual representation of the user, usually in
 * PNG, GIF, or JPEG format.
 */
struct _GalagoAvatar
{
	GalagoImage parent_object;

	GalagoAvatarPrivate *priv;

	void (*reserved_1)(void);
	void (*reserved_2)(void);
	void (*reserved_3)(void);
	void (*reserved_4)(void);
};

/**
 * GalagoAvatar's class.
 */
struct _GalagoAvatarClass
{
	GalagoImageClass parent_class;

	void (*reserved_1)(void);
	void (*reserved_2)(void);
	void (*reserved_3)(void);
	void (*reserved_4)(void);
};

#define GALAGO_CLASS_AVATAR (galago_avatar_get_class())
#define GALAGO_DBUS_AVATAR_INTERFACE "org.freedesktop.Galago.Avatar"
#define GALAGO_IS_AVATAR(obj) (GALAGO_IS_OBJECT(obj) && galago_object_check_cast((obj), GALAGO_CLASS_AVATAR))

#ifdef __cplusplus
extern "C" {
#endif

/**************************************************************************/
/** @name Avatar API                                                      */
/**************************************************************************/
/*@{*/

/**
 * Returns the class for a GalagoAvatar.
 *
 * @return The GalagoAvatar class.
 */
GalagoObjectClass *galago_avatar_get_class(void);

/**
 * Creates a new avatar.
 *
 * @param account The account the avatar is for.
 * @param data    The image data.
 * @param len     The image data length.
 *
 * @return The avatar.
 */
GalagoAvatar *galago_avatar_new(GalagoAccount *account,
								const unsigned char *data, size_t len);

/**
 * Sets the avatar's image data.
 *
 * @param avatar The avatar.
 * @param data   The image data.
 * @param len    The image data length.
 */
void galago_avatar_set_image_data(GalagoAvatar *avatar,
								  const unsigned char *data, size_t len);

/**
 * Returns an avatar's account.
 *
 * @param avatar The avatar.
 *
 * @return The account.
 */
GalagoAccount *galago_avatar_get_account(const GalagoAvatar *avatar);

/**
 * Returns an avatar's image data.
 *
 * @param avatar   The avatar.
 * @param ret_data The returned image data.
 * @param ret_len  The returned image data length.
 */
void galago_avatar_get_image_data(const GalagoAvatar *avatar,
								  unsigned char **ret_data,
								  size_t *ret_len);

/*@}*/

#ifdef __cplusplus
}
#endif

#endif /* _GALAGO_AVATAR_H_ */
