/* Copyright (C) 1999 Hans Petter K. Jansson
 *
 * This library is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307, USA.
 *
 * You can contact the library's author by sending e-mail to <hpj@styx.net>.
 */

#include "config.h"
#include "flux.h"
#include <stdio.h>
#include <stdlib.h>
#include <string.h>

static void mt_print_xml_escaped_data(char *data, FILE *out);
static void mt_print_xml_attrs(MT *mt, FILE *out);
static void mt_print_xml_span(MT *mt, FILE *out);
static void mt_print_xml_empty(MT *mt, FILE *out);
static void mt_print_xml_data(MT *mt, FILE *out);
static void mt_print_xml_children(MT *mt, FILE *out);


static void mt_print_xml_escaped_data(char *data, FILE *out)
{
  char *p0;
  
  for (p0 = data; *p0; p0++)
  {
    switch(*p0)
    {
      case '<': fprintf(out, "&lt;"); break;
      case '>': fprintf(out, "&gt;"); break;
      case '&': fprintf(out, "&amp;"); break;
      case '"': fprintf(out, "&quot;"); break;
      case   9: fprintf(out, "&#9;"); break;
      case  10: fprintf(out, "&#10;"); break;
      case  13: fprintf(out, "&#13;"); break;
      default: fputc(*p0, out); break;
    }
  }
}


static void mt_print_xml_attrs(MT *mt, FILE *out)
{
  TT *attrs, *attr;
  char *arg, *val;
  
  attrs = mt_get_attrs(mt);
  if (!attrs) return;

  TT_FOR_EACH(attrs, attr)
  {
    arg = tt_data_get_str(attr);
    if (!arg) continue;

    if (!tt_is_leaf(attr)) val = tt_data_get_str(tt_get_first_child(attr));
    else val = 0;

    if (val)
    {
      fprintf(out, " %s=\"", arg);
      mt_print_xml_escaped_data(val, out);
      fputc('"', out);
    }
    else fprintf(out, " %s=\"\"", arg);
    
    if (val) free(val);
    free(arg);
  }
}


static void mt_print_xml_span(MT *mt, FILE *out)
{
  char *name;

  name = mt_get_name_str(mt);
  fprintf(out, "<%s", name);
  mt_print_xml_attrs(mt, out);
  fprintf(out, ">");

  mt_print_xml_children(mt, out);

  fprintf(out, "</%s>", name);
}


static void mt_print_xml_empty(MT *mt, FILE *out)
{
  char *name;
  
  name = mt_get_name_str(mt);
  fprintf(out, "<%s/", name);
  mt_print_xml_attrs(mt, out);
  fprintf(out, ">");
}


static void mt_print_xml_data(MT *mt, FILE *out)
{
  void *data;
  
  data = tt_data_get(mt_data_get(mt));
  if (!data) return;
  
  fwrite(data, mt_size(mt), 1, out);
}


static void mt_print_xml_children(MT *mt, FILE *out)
{
  MT *mt_child;
  
  MT_FOR_EACH(mt, mt_child)
  {
    switch (mt_get_type(mt_child))
    {
      case MT_SPAN: mt_print_xml_span(mt_child, out); break;
      case MT_EMPTY: mt_print_xml_empty(mt_child, out); break;
      case MT_DATA: mt_print_xml_data(mt_child, out); break;
      default: break;
    }
  }
}
  

int mt_print_to_xml_file(MT *mt, FILE *out)
{
  if (mt_get_type(mt) != MT_SPAN) return(-1);
  
  mt_print_xml_span(mt, out);
  return(0);
}
