/*--------------------------------------------------------------------------+
$Id: StateflowModelTest.java 26285 2010-02-18 11:22:54Z juergens $
|                                                                          |
| Copyright 2005-2010 Technische Universitaet Muenchen                     |
|                                                                          |
| Licensed under the Apache License, Version 2.0 (the "License");          |
| you may not use this file except in compliance with the License.         |
| You may obtain a copy of the License at                                  |
|                                                                          |
|    http://www.apache.org/licenses/LICENSE-2.0                            |
|                                                                          |
| Unless required by applicable law or agreed to in writing, software      |
| distributed under the License is distributed on an "AS IS" BASIS,        |
| WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. |
| See the License for the specific language governing permissions and      |
| limitations under the License.                                           |
+--------------------------------------------------------------------------*/
package edu.tum.cs.simulink.model.stateflow;

import java.io.FileNotFoundException;
import java.util.Map;

import edu.tum.cs.commons.collections.IdentityHashSet;
import edu.tum.cs.commons.logging.SimpleLogger;
import edu.tum.cs.commons.test.CCSMTestCaseBase;
import edu.tum.cs.commons.test.DeepCloneTestUtils;
import edu.tum.cs.simulink.builder.SimulinkModelBuilder;
import edu.tum.cs.simulink.builder.SimulinkModelBuildingException;
import edu.tum.cs.simulink.model.SimulinkBlock;
import edu.tum.cs.simulink.model.SimulinkModel;
import edu.tum.cs.simulink.util.SimulinkUtils;

/**
 * Test for the Stateflow part of the model.
 * 
 * @author deissenb
 * @author $Author: juergens $
 * @version $Rev: 26285 $
 * @levd.rating GREEN Hash: 014E8453400116C981ADBAF942700934
 */
public class StateflowModelTest extends CCSMTestCaseBase {

	/** Chart under test. */
	private StateflowChart chart;

	/** Load model and set block. */
	@Override
	public void setUp() throws FileNotFoundException,
			SimulinkModelBuildingException {
		SimulinkModelBuilder builder = new SimulinkModelBuilder(
				useTestFile("internal_remove.mdl"), new SimpleLogger());
		SimulinkModel model = builder.buildModel();
		StateflowBlock block = (StateflowBlock) model
				.getBlock("internal_remove/Stateflow");
		chart = block.getChart();
	}

	/** Test if chart removal works. */
	public void testRemoveChart() {

		StateflowMachine machine = chart.getMachine();

		IdentityHashSet<StateflowChart> charts = DeepCloneTestUtils
				.getAllReferencedObjects(machine, StateflowChart.class,
						getClass().getPackage().getName());

		for (StateflowChart chart : charts) {
			chart.getStateflowBlock().remove();
			assertFalse(machine.getCharts().contains(chart));
			assertNull(chart.getParent());
		}
	}

	/** Test if data removal works. */
	public void testRemoveData() {
		IdentityHashSet<StateflowData> data = DeepCloneTestUtils
				.getAllReferencedObjects(chart, StateflowData.class, getClass()
						.getPackage().getName());

		for (StateflowData date : data) {
			StateflowDeclContainerBase<?> parent = date.getParent();
			date.remove();
			assertFalse(parent.getData().contains(date));
			assertNull(date.getParent());
		}
	}

	/** Test if event removal works. */
	public void testRemoveEvent() {
		IdentityHashSet<StateflowEvent> events = DeepCloneTestUtils
				.getAllReferencedObjects(chart, StateflowEvent.class,
						getClass().getPackage().getName());

		for (StateflowEvent event : events) {
			StateflowDeclContainerBase<?> parent = event.getParent();
			event.remove();
			assertFalse(parent.getEvents().contains(event));
			assertNull(event.getParent());
		}
	}

	/** Test if machine removal works. */
	public void testRemoveMachine() {
		StateflowMachine machine = chart.getMachine();
		SimulinkModel model = machine.getModel();
		testRemoveChart();
		machine.remove();
		assertNull(machine.getModel());
		assertNull(model.getStateflowMachine());
		assertTrue(machine.getCharts().isEmpty());
		assertTrue(machine.getData().isEmpty());
		assertTrue(machine.getEvents().isEmpty());
		assertTrue(machine.getTargets().isEmpty());
	}

	/** Test if node removal works. */
	public void testRemoveNode() {
		IdentityHashSet<StateflowNodeBase> nodes = DeepCloneTestUtils
				.getAllReferencedObjects(chart, StateflowNodeBase.class,
						getClass().getPackage().getName());

		for (StateflowNodeBase node : nodes) {
			// cast must be valid
			IStateflowNodeContainer<?> parent = node.getParent();

			node.remove();

			assertFalse(parent.getNodes().contains(node));
			assertNull(node.getParent());

			assertTrue(node.getOutTransitions().isEmpty());
			assertTrue(node.getInTransitions().isEmpty());
		}

		// after we removed all nodes, there should be no transitions left
		IdentityHashSet<StateflowTransition> transitions = DeepCloneTestUtils
				.getAllReferencedObjects(chart, StateflowTransition.class,
						getClass().getPackage().getName());
		assertTrue(transitions.isEmpty());
	}

	/** Test if event removal works. */
	public void testRemoveTarget() {
		IdentityHashSet<StateflowTarget> targets = DeepCloneTestUtils
				.getAllReferencedObjects(chart, StateflowTarget.class,
						getClass().getPackage().getName());

		for (StateflowTarget target : targets) {
			StateflowMachine parent = target.getParent();
			target.remove();
			assertFalse(parent.getEvents().contains(target));
			assertNull(target.getParent());
		}
	}

	/** Test if transition removal works. */
	public void testRemoveTransition() {
		IdentityHashSet<StateflowTransition> transitions = DeepCloneTestUtils
				.getAllReferencedObjects(chart, StateflowTransition.class,
						getClass().getPackage().getName());

		for (StateflowTransition transition : transitions) {
			StateflowNodeBase src = transition.getSrc();
			StateflowNodeBase dst = transition.getDst();

			int dstCount = dst.getInTransitions().size();

			int srcCount = 1;
			if (src != null) {
				srcCount = src.getOutTransitions().size();
			}

			transition.remove();
			assertEquals(dstCount - 1, dst.getInTransitions().size());
			assertFalse(dst.getInTransitions().contains(transition));

			// default transition
			if (src == null) {
				continue;
			}

			assertEquals(srcCount - 1, src.getOutTransitions().size());
			assertFalse(src.getOutTransitions().contains(transition));

			assertNull(transition.getSrc());
			assertNull(transition.getDst());
		}
	}

	/**
	 * This tests if the blocks beneath the Stateflow block are read. See CR
	 * #1502.
	 */
	public void testHiddenBlocks() throws FileNotFoundException,
			SimulinkModelBuildingException {
		SimulinkModelBuilder builder = new SimulinkModelBuilder(
				useTestFile("chart_input.mdl"), new SimpleLogger());
		SimulinkModel model = builder.buildModel();
		StateflowBlock block = (StateflowBlock) model
				.getBlock("chart_input/Chart");

		Map<String, SimulinkBlock> map = SimulinkUtils.createIdToNodeMap(block);
		assertFalse(map.isEmpty());

		assertTrue(map.containsKey("chart_input/Chart/input1"));

		// The blocks really contain spaces
		assertTrue(map.containsKey("chart_input/Chart/ Demux "));
		assertTrue(map.containsKey("chart_input/Chart/ SFunction "));
		assertTrue(map.containsKey("chart_input/Chart/ Terminator "));

	}

}