/*--------------------------------------------------------------------------+
$Id: Assessment.java 27816 2010-05-20 16:13:15Z hummelb $
|                                                                          |
| Copyright 2005-2010 Technische Universitaet Muenchen                     |
|                                                                          |
| Licensed under the Apache License, Version 2.0 (the "License");          |
| you may not use this file except in compliance with the License.         |
| You may obtain a copy of the License at                                  |
|                                                                          |
|    http://www.apache.org/licenses/LICENSE-2.0                            |
|                                                                          |
| Unless required by applicable law or agreed to in writing, software      |
| distributed under the License is distributed on an "AS IS" BASIS,        |
| WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. |
| See the License for the specific language governing permissions and      |
| limitations under the License.                                           |
+--------------------------------------------------------------------------*/
package edu.tum.cs.commons.assessment;

import java.io.Serializable;
import java.util.Collection;

import edu.tum.cs.commons.clone.IDeepCloneable;

/**
 * This class stores an assessment. An assessment is a multiset of traffic light
 * colors (i.e. a mapping from traffic light colors to non-negative integers).
 * 
 * @author Benjamin Hummel
 * @author $Author: hummelb $
 * @version $Rev: 27816 $
 * @levd.rating GREEN Hash: E1A8D09889F5CFB5BAD863BEB224644F
 */
public class Assessment implements Cloneable, IDeepCloneable, Serializable {

	/** The "multimap". */
	private int[] mapping = new int[ETrafficLightColor.values().length];

	/** The assessment message. */
	private String message;

	/**
	 * Creates an empty assessment (i.e. one with all entries set to 0).
	 */
	public Assessment() {
		// Do nothing, but keep it to have a default constructor.
	}

	/**
	 * Create an assessment with a single color entry.
	 * 
	 * @param color
	 *            the color included in this assessment.
	 */
	public Assessment(ETrafficLightColor color) {
		add(color);
	}

	/**
	 * Create an assessement with a assessment message.
	 */
	public Assessment(ETrafficLightColor color, String message) {
		add(color);
		this.message = message;
	}

	/**
	 * Add a single entry of this color to this assessment.
	 * 
	 * @param color
	 *            the color added to this assessment.
	 */
	public void add(ETrafficLightColor color) {
		mapping[color.ordinal()]++;
	}

	/**
	 * Add a single entry of this color to this assessment.
	 * 
	 * @param color
	 *            the color added to this assessment.
	 * @param count
	 *            how often to add this color to the assessment.
	 */
	public void add(ETrafficLightColor color, int count) {
		if (count < 0) {
			throw new IllegalArgumentException("Count must be non-negative!");
		}
		mapping[color.ordinal()] += count;
	}

	/**
	 * Merge the provided assessment into this, i.e. increase all trafic light
	 * color counts by the values in the provided asseessment.
	 * 
	 * @param a
	 *            the assessment to merge in.
	 */
	public void add(Assessment a) {
		for (int i = 0; i < mapping.length; ++i) {
			mapping[i] += a.mapping[i];
		}
	}

	/**
	 * @param color
	 *            the color whose frequency to read.
	 * @return the number of occurrences of the provided color in this
	 *         assessment.
	 */
	public int getColorFrequency(ETrafficLightColor color) {
		return mapping[color.ordinal()];
	}

	/**
	 * @return the most dominant color. If there is a RED entry, RED is
	 *         returned. Otherwise, if there is a YELLOW entry, YELLOW is
	 *         returned. If there is neither a RED OR YELLOW entry, but a GREEN
	 *         one, GREEN is returned. If no color is there, UNKNOWN is
	 *         returned.
	 * 
	 *         This method relies on the fact that the entries in
	 *         {@link ETrafficLightColor} are ordered according to their
	 *         dominance.
	 */
	public ETrafficLightColor getDominantColor() {
		for (int i = 0; i < ETrafficLightColor.values().length; ++i) {
			if (mapping[i] > 0) {
				return ETrafficLightColor.values()[i];
			}
		}
		return ETrafficLightColor.UNKNOWN;
	}

	/**
	 * @return the color that is most frequent in this assessment. If all
	 *         frequencies are 0, UNKNOWN is returned. If there are ties, the
	 *         more dominant (see {@link #getDominantColor()}) one is returned.
	 */
	public ETrafficLightColor getMostFrequentColor() {
		ETrafficLightColor result = null;
		int resultCount = 0;

		for (int i = 0; i < mapping.length; ++i) {
			if (mapping[i] > resultCount) {
				resultCount = mapping[i];
				result = ETrafficLightColor.values()[i];
			}
		}

		if (result != null) {
			return result;
		}
		return ETrafficLightColor.UNKNOWN;
	}

	/** {@inheritDoc} */
	@Override
	public String toString() {
		int sum = 0;
		for (int i = 0; i < mapping.length; ++i) {
			sum += mapping[i];
		}

		if (sum == 0) {
			return "";
		}

		if (sum == 1) {
			for (int i = 0; i < mapping.length; ++i) {
				if (mapping[i] > 0) {
					return ETrafficLightColor.values()[i].toString();
				}
			}
		}

		StringBuilder builder = new StringBuilder("[");
		appendColor(builder, ETrafficLightColor.GREEN);
		builder.append(", ");
		appendColor(builder, ETrafficLightColor.YELLOW);
		builder.append(", ");
		appendColor(builder, ETrafficLightColor.RED);
		builder.append("]");
		return builder.toString();
	}

	/**
	 * append a string containing the color and its frequency to the given
	 * builder
	 * 
	 * @param builder
	 * @param color
	 */
	private void appendColor(StringBuilder builder, ETrafficLightColor color) {
		builder.append(color.toString().substring(0, 1));
		builder.append(": ");
		builder.append(getColorFrequency(color));
	}

	/** {@inheritDoc} */
	@Override
	protected Object clone() throws CloneNotSupportedException {
		Assessment a = (Assessment) super.clone();
		a.mapping = a.mapping.clone();
		return a;
	}

	/** {@inheritDoc} */
	public Assessment deepClone() {
		Assessment a = new Assessment();
		a.add(this);
		return a;
	}

	/** {@inheritDoc} */
	@Override
	public boolean equals(Object obj) {
		if (!(obj instanceof Assessment)) {
			return false;
		}

		Assessment a = (Assessment) obj;
		for (int i = 0; i < mapping.length; ++i) {
			if (mapping[i] != a.mapping[i]) {
				return false;
			}
		}
		return true;
	}

	/** {@inheritDoc} */
	@Override
	public int hashCode() {
		int hash = 0;
		for (int i = 0; i < mapping.length; ++i) {
			/*
			 * primes taken from
			 * http://planetmath.org/encyclopedia/GoodHashTablePrimes.html
			 */
			hash *= 97;
			hash += mapping[i];
			hash %= 50331653;
		}
		return hash;
	}

	/**
	 * Get assessment message.
	 * 
	 * @return the message or <code>null</code> if this assessment has no
	 *         message.
	 */
	public String getMessage() {
		return message;
	}

	/**
	 * Set assessment message.
	 * 
	 * @param message
	 *            the message or <code>null</code> if this assessment has no
	 *            message.
	 */
	public void setMessage(String message) {
		this.message = message;
	}

	/** Aggregate assessments. */
	public static Assessment aggregate(Collection<Assessment> values) {
		Assessment result = new Assessment();
		for (Assessment a : values) {
			result.add(a);
		}
		return result;
	}

}