#ifndef WS_H
#define WS_H

#include <glib.h>
#include <glib-object.h>

#include <X11/X.h>
#include <X11/Xlib.h>
#include <GL/gl.h>

typedef struct WsDisplay WsDisplay;
typedef struct WsDisplayClass WsDisplayClass;
typedef struct WsScreen WsScreen;
typedef struct WsScreenClass WsScreenClass;
typedef struct WsResource WsResource;
typedef struct WsResourceClass WsResourceClass;
typedef struct WsDrawable WsDrawable;
typedef struct WsDrawableClass WsDrawableClass;
typedef struct WsPixmap WsPixmap;
typedef struct WsPixmapClass WsPixmapClass;
typedef struct WsWindow WsWindow;
typedef struct WsWindowClass WsWindowClass;
typedef struct WsRegion WsRegion;
typedef struct WsRegionClass WsRegionClass;
typedef struct WsBits WsBits;

/* WsRectangle */

typedef struct
{
    int x, y;
    guint width, height;
} WsRectangle;

gboolean     ws_rectangle_intersect (WsRectangle *src1,
				     WsRectangle *src2,
				     WsRectangle *dest);

/* WsFormat */

/* This is my current understanding. In case anybody
 * is interested in this stuff.
 *
 * Suppose guchar *pixel points to a pixel in WS_FORMAT_ABCD. Then,
 * on little endian:
 *
 *	pixel[0] is the D component
 *	pixel[1] is the C component
 *	pixel[2] is the B component
 *	pixel[3] is the A component
 *
 * on big endian:
 *
 *	pixel[0] is the A component
 *	pixel[1] is the B component
 *      etc.
 *
 * Note that OpenGL uses a different scheme, where ABCD always
 * means that A is higher in memory than D, independent of
 * endianness. (This is the part I'm unsure about).
 *
 */
typedef enum
{
    WS_FORMAT_UNKNOWN,
    WS_FORMAT_INPUT_ONLY,
    WS_FORMAT_RGB_16,
    WS_FORMAT_RGB_24,
    WS_FORMAT_XRGB_32,
    WS_FORMAT_ARGB_32,
    WS_FORMAT_LAST
} WsFormat;

guint    ws_format_get_depth (WsFormat format);
gboolean ws_format_is_viewable (WsFormat format);
void     ws_format_get_masks (WsFormat format,
			      gulong *  red_mask,
			      gulong *  green_mask,
			      gulong *  blue_mask);

/* WsColor */

typedef struct
{
    gint alpha;
    gint red;
    gint green;
    gint blue;
} WsColor;

/* WsDisplay */
#define WS_TYPE_DISPLAY            (ws_display_get_type ())
#define WS_DISPLAY(obj)            (G_TYPE_CHECK_INSTANCE_CAST ((obj), WS_TYPE_DISPLAY, WsDisplay))
#define WS_DISPLAY_CLASS(klass)    (G_TYPE_CHECK_CLASS_CAST ((klass),  WS_TYPE_DISPLAY, WsDisplayClass))
#define WS_IS_DISPLAY(obj)         (G_TYPE_CHECK_INSTANCE_TYPE ((obj), WS_TYPE_DISPLAY))
#define WS_IS_DISPLAY_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass),  WS_TYPE_DISPLAY))
#define WS_DISPLAY_GET_CLASS(obj)  (G_TYPE_INSTANCE_GET_CLASS ((obj),  WS_TYPE_DISPLAY, WsDisplayClass))

GType      ws_display_get_type               (void);
WsDisplay *ws_display_new                    (const char  *dpy);
void       ws_display_process_xevent         (WsDisplay   *ws,
					      XEvent      *xev);
void       ws_display_process_error          (WsDisplay   *ws,
					      XErrorEvent *error);
void       ws_display_grab                   (WsDisplay   *ws);
void       ws_display_ungrab                 (WsDisplay   *ws);
gboolean   ws_display_init_composite         (WsDisplay   *ws);
gboolean   ws_display_init_damage            (WsDisplay   *ws);
gboolean   ws_display_init_fixes             (WsDisplay   *ws);
gboolean   ws_display_init_test              (WsDisplay   *ws);
GList *    ws_display_list_screens           (WsDisplay   *ws);
void       ws_display_sync                   (WsDisplay   *ws);
void       ws_display_sync_with_xdisplay     (WsDisplay   *ws,
					      gpointer     dpy);
void       ws_display_flush                  (WsDisplay   *ws);
void       ws_display_set_synchronize        (WsDisplay   *ws,
					      gboolean     sync);
void       ws_display_set_ignore_grabs       (WsDisplay   *ws,
					      gboolean     ignore);
void       ws_display_begin_error_trap       (WsDisplay   *ws);
void       ws_display_end_error_trap         (WsDisplay   *ws);
gboolean   ws_display_end_error_trap_with_return (WsDisplay *display);
WsScreen * ws_display_get_default_screen     (WsDisplay   *ws);
WsScreen * ws_display_get_screen_from_number (WsDisplay   *ws,
					      int          number);


/* WsScreen */
#define WS_TYPE_SCREEN            (ws_screen_get_type ())
#define WS_SCREEN(obj)            (G_TYPE_CHECK_INSTANCE_CAST ((obj), WS_TYPE_SCREEN, WsScreen))
#define WS_SCREEN_CLASS(klass)    (G_TYPE_CHECK_CLASS_CAST ((klass),  WS_TYPE_SCREEN, WsScreenClass))
#define WS_IS_SCREEN(obj)         (G_TYPE_CHECK_INSTANCE_TYPE ((obj), WS_TYPE_SCREEN))
#define WS_IS_SCREEN_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass),  WS_TYPE_SCREEN))
#define WS_SCREEN_GET_CLASS(obj)  (G_TYPE_INSTANCE_GET_CLASS ((obj),  WS_TYPE_SCREEN, WsScreenClass))

GType	  ws_screen_get_type	    (void);
WsWindow *ws_screen_get_root_window (WsScreen *screen);
int       ws_screen_get_number      (WsScreen *screen);
void      ws_screen_graphics_sync   (WsScreen *screen);
int	  ws_screen_get_height      (WsScreen *screen);
int       ws_screen_get_width       (WsScreen *screen);
WsWindow *ws_screen_get_gl_window   (WsScreen *screen);

/* WsResource */
#define WS_TYPE_RESOURCE            (ws_resource_get_type ())
#define WS_RESOURCE(obj)            (G_TYPE_CHECK_INSTANCE_CAST ((obj), WS_TYPE_RESOURCE, WsResource))
#define WS_RESOURCE_CLASS(klass)    (G_TYPE_CHECK_CLASS_CAST ((klass),  WS_TYPE_RESOURCE, WsResourceClass))
#define WS_IS_RESOURCE(obj)         (G_TYPE_CHECK_INSTANCE_TYPE ((obj), WS_TYPE_RESOURCE))
#define WS_IS_RESOURCE_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass),  WS_TYPE_RESOURCE))
#define WS_RESOURCE_GET_CLASS(obj)  (G_TYPE_INSTANCE_GET_CLASS ((obj),  WS_TYPE_RESOURCE, WsResourceClass))

GType ws_resource_get_type (void);


/* WsRegion */
#define WS_TYPE_REGION            (ws_region_get_type ())
#define WS_REGION(obj)            (G_TYPE_CHECK_INSTANCE_CAST ((obj), WS_TYPE_REGION, WsRegion))
#define WS_REGION_CLASS(klass)    (G_TYPE_CHECK_CLASS_CAST ((klass),  WS_TYPE_REGION, WsRegionClass))
#define WS_IS_REGION(obj)         (G_TYPE_CHECK_INSTANCE_TYPE ((obj), WS_TYPE_REGION))
#define WS_IS_REGION_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass),  WS_TYPE_REGION))
#define WS_REGION_GET_CLASS(obj)  (G_TYPE_INSTANCE_GET_CLASS ((obj),  WS_TYPE_REGION, WsRegionClass))

GType ws_region_get_type (void);

WsRegion *   ws_region_new              (WsDisplay       *ws);
WsRectangle *ws_region_query_rectangles (WsRegion *region,
					 guint    *n_rectangles);
void         ws_region_union            (WsRegion *region,
					 WsRegion *other);

/* WsDrawable */
#define WS_TYPE_DRAWABLE            (ws_drawable_get_type ())
#define WS_DRAWABLE(obj)            (G_TYPE_CHECK_INSTANCE_CAST ((obj), WS_TYPE_DRAWABLE, WsDrawable))
#define WS_DRAWABLE_CLASS(klass)    (G_TYPE_CHECK_CLASS_CAST ((klass),  WS_TYPE_DRAWABLE, WsDrawableClass))
#define WS_IS_DRAWABLE(obj)         (G_TYPE_CHECK_INSTANCE_TYPE ((obj), WS_TYPE_DRAWABLE))
#define WS_IS_DRAWABLE_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass),  WS_TYPE_DRAWABLE))
#define WS_DRAWABLE_GET_CLASS(obj)  (G_TYPE_INSTANCE_GET_CLASS ((obj),  WS_TYPE_DRAWABLE, WsDrawableClass))

GType ws_drawable_get_type (void);

typedef void (* WsWindowDamageCallback) (WsDrawable *window,
					 gpointer    data);
WsDisplay *   ws_drawable_get_display         (WsDrawable             *window);
WsScreen *    ws_drawable_query_screen        (WsDrawable             *window);
void          ws_drawable_copy_area           (WsDrawable             *src,
					       int                     src_x,
					       int                     src_y,
					       int                     width,
					       int                     height,
					       WsDrawable             *dst,
					       int                     dst_x,
					       int                     dst_y,
					       WsRegion               *clip);
gboolean      ws_drawable_query_geometry      (WsDrawable             *drawable,
					       WsRectangle            *rect);
int           ws_drawable_query_depth         (WsDrawable             *drawable);
WsBits   *    ws_drawable_get_bits            (WsDrawable             *drawable,
					       WsRectangle	      *rect);
WsFormat      ws_drawable_get_format	      (WsDrawable	      *drawable);
const guchar *ws_bits_get_info                (WsBits                 *bits,
					       guint                  *width,
					       guint                  *height,
					       guint                  *bytes_per_pixel,
					       guint                  *bytes_per_row);
void          ws_bits_free                    (WsBits                 *bits);
void	      ws_drawable_free		      (WsDrawable             *drawable);


/* WsPixmap */
#define WS_TYPE_PIXMAP            (ws_pixmap_get_type ())
#define WS_PIXMAP(obj)            (G_TYPE_CHECK_INSTANCE_CAST ((obj), WS_TYPE_PIXMAP, WsPixmap))
#define WS_PIXMAP_CLASS(klass)    (G_TYPE_CHECK_CLASS_CAST ((klass),  WS_TYPE_PIXMAP, WsPixmapClass))
#define WS_IS_PIXMAP(obj)         (G_TYPE_CHECK_INSTANCE_TYPE ((obj), WS_TYPE_PIXMAP))
#define WS_IS_PIXMAP_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass),  WS_TYPE_PIXMAP))
#define WS_PIXMAP_GET_CLASS(obj)  (G_TYPE_INSTANCE_GET_CLASS ((obj),  WS_TYPE_PIXMAP, WsPixmapClass))

typedef void (* WsDamageCallback) (WsPixmap *pixmap,
				   gpointer data);

GType ws_pixmap_get_type (void);

WsPixmap *ws_pixmap_new (WsDrawable *drawable,
			 int	     width,
			 int	     height);
GLuint	  ws_pixmap_get_texture (WsPixmap *pixmap);
void	  ws_pixmap_set_damage_callback (WsPixmap *pixmap,
					 WsDamageCallback cb,
					 gpointer  data);
void      ws_pixmap_set_updates (WsPixmap *pixmap,
				 gboolean  do_updates);
gboolean ws_pixmap_get_updates (WsPixmap *pixmap);

/* WsTexture */




/* WsWindow */
#define WS_TYPE_WINDOW            (ws_window_get_type ())
#define WS_WINDOW(obj)            (G_TYPE_CHECK_INSTANCE_CAST ((obj), WS_TYPE_WINDOW, WsWindow))
#define WS_WINDOW_CLASS(klass)    (G_TYPE_CHECK_CLASS_CAST ((klass),  WS_TYPE_WINDOW, WsWindowClass))
#define WS_IS_WINDOW(obj)         (G_TYPE_CHECK_INSTANCE_TYPE ((obj), WS_TYPE_WINDOW))
#define WS_IS_WINDOW_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass),  WS_TYPE_WINDOW))
#define WS_WINDOW_GET_CLASS(obj)  (G_TYPE_INSTANCE_GET_CLASS ((obj),  WS_TYPE_WINDOW, WsWindowClass))

typedef void (* WsConfigureCallback) (WsWindow *window,
				      int	x,
				      int	y,
				      int	width,
				      int	height,
				      int	border_width,
				      WsWindow *above,
				      gboolean  override_redirect,
				      gpointer  data);

GType ws_window_get_type (void);

gboolean  ws_window_query_mapped	   (WsWindow            *window);
void      ws_window_raise                  (WsWindow            *window);
void      ws_window_map                    (WsWindow            *window);
void	  ws_window_unmap                  (WsWindow            *window);
WsWindow *ws_window_new                    (WsWindow            *parent);
WsWindow *ws_window_new_gl                 (WsWindow            *parent);
GList *   ws_window_query_subwindows       (WsWindow            *window);
void      ws_window_unredirect             (WsWindow            *window);
void      ws_window_redirect               (WsWindow            *window);
void      ws_window_set_override_redirect  (WsWindow            *window,
					    gboolean             override_redirect);
WsPixmap *ws_window_name_pixmap            (WsWindow            *window);
void      ws_window_redirect_subwindows    (WsWindow            *window);
void	  ws_window_unredirect_subwindows  (WsWindow            *window);
gboolean  ws_window_query_input_only       (WsWindow            *window);
gboolean  ws_window_query_viewable         (WsWindow            *window);
char *    ws_window_query_title            (WsWindow            *window);
void      ws_window_set_input_shape        (WsWindow            *window,
					    WsRegion            *shape);
WsWindow *ws_window_lookup                 (WsDisplay                  *ws,
					    Window               xid);
void      ws_window_set_configure_callback (WsWindow            *window,
					    WsConfigureCallback  cb,
					    gpointer             data);

/* WsTexture */

#define WS_TYPE_TEXTURE            (ws_texture_get_type ())
#define WS_TEXTURE(obj)            (G_TYPE_CHECK_INSTANCE_CAST ((obj), WS_TYPE_TEXTURE, WsTexture))
#define WS_TEXTURE_CLASS(klass)    (G_TYPE_CHECK_CLASS_CAST ((klass),  WS_TYPE_TEXTURE, WsTextureClass))
#define WS_IS_TEXTURE(obj)         (G_TYPE_CHECK_INSTANCE_TYPE ((obj), WS_TYPE_TEXTURE))
#define WS_IS_TEXTURE_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass),  WS_TYPE_TEXTURE))
#define WS_TEXTURE_GET_CLASS(obj)  (G_TYPE_INSTANCE_GET_CLASS ((obj),  WS_TYPE_TEXTURE, WsTextureClass))

typedef struct _WsTexture WsTexture;
typedef struct _WsTextureClass WsTextureClass;

struct _WsTexture
{
    GObject parent_instance;
};

struct _WsTextureClass
{
    GObjectClass parent_class;
};

GType ws_texture_get_type (void);


/* Possible GL API */
void      ws_window_setup_gl              (WsWindow    *window);
void      ws_window_gl_begin              (WsWindow    *window);
void      ws_window_gl_end                (WsWindow    *window);
void      ws_window_gl_swap_buffers       (WsWindow    *window);

/* Maybe also "get texture from drawable/pixmap" */

#endif
