// 
// Copyright (c) 2006-2008 Ben Motmans
// 
// Permission is hereby granted, free of charge, to any person obtaining a copy
// of this software and associated documentation files (the "Software"), to deal
// in the Software without restriction, including without limitation the rights
// to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
// copies of the Software, and to permit persons to whom the Software is
// furnished to do so, subject to the following conditions:
//
// The above copyright notice and this permission notice shall be included in
// all copies or substantial portions of the Software.
//
// THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
// IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
// FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
// AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
// LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
// OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
// THE SOFTWARE.
//
// Author(s):
//    Ben Motmans <ben.motmans@gmail.com>
//

using System;
using System.Collections.Generic;

namespace Anculus.Core
{

	public class SetSearchAlgorithmWrapper : AbstractSetSearchAlgorithm
	{
		private ISearchAlgorithm algorithm;
		
		public SetSearchAlgorithmWrapper (ISearchAlgorithm algorithm)
		{
			if (algorithm == null)
				throw new ArgumentNullException ("algorithm");
			
			this.algorithm = algorithm;
		}
		
		public override SearchResult[] SearchAll (string text, int start, int count, params string[] keywords)
		{
			CheckArguments (text, start, count);
			CheckKeywords (keywords);

			List<SearchResult> results = new List<SearchResult> (count != int.MaxValue ? count : 4);
			
			foreach (string keyword in keywords) {
				int[] indices = algorithm.SearchAll (text, keyword);
				foreach (int index in indices)
					results.Add (new SearchResult (index, keyword));
			}

			new QuickSorter ().Sort<SearchResult> (results, new SearchResultComparer ());

			return results.ToArray ();
		}

		public override SearchResult SearchFirst (string text, int start, params string[] keywords)
		{
			CheckArguments (text, start, int.MaxValue);
			CheckKeywords (keywords);
			
			SearchResult first = SearchResult.Empty;
			foreach (string keyword in keywords) {
				int index = algorithm.SearchFirst (text, keyword);
				if (index < 0)
					continue;
				
				if (first == SearchResult.Empty)
					first = new SearchResult (index, keyword);
				else
					if (index < first.Index)
						first = new SearchResult (index, keyword);
			}
			
			return first;
		}
		
		public override string ToString ()
		{
			return algorithm.ToString ();
		}
	}
}
