/**
 * \file QRef.C
 * This file is part of LyX, the document processor.
 * Licence details can be found in the file COPYING.
 *
 * \author John Levon
 *
 * Full author contact details are available in file CREDITS
 */

#include <config.h>

#include "ControlRef.h"
#include "QRef.h"
#include "QRefDialog.h"
#include "Qt2BC.h"

#include <qlineedit.h>
#include <qcheckbox.h>
#include <qlistbox.h>
#include <qcombobox.h>
#include <qpushbutton.h>
#include <qtooltip.h>

#include "helper_funcs.h" // getStringFromVector
#include "support/lstrings.h" // frontStrip, strip
#include "qt_helpers.h"
#include "insets/insetref.h"

using std::find;
using std::max;
using std::sort;
using std::vector;
using std::endl;


typedef Qt2CB<ControlRef, Qt2DB<QRefDialog> > base_class;


QRef::QRef()
	: base_class(qt_("Cross Reference")),
	sort_(false), at_ref_(false)
{
}


void QRef::build_dialog()
{
	dialog_.reset(new QRefDialog(this));

	bc().setOK(dialog_->okPB);
	bc().setCancel(dialog_->closePB);
	bc().addReadOnly(dialog_->refsLB);
	bc().addReadOnly(dialog_->sortCB);
	bc().addReadOnly(dialog_->nameED);
	bc().addReadOnly(dialog_->referenceED);
	bc().addReadOnly(dialog_->typeCO);
	bc().addReadOnly(dialog_->bufferCO);
}


void QRef::update_contents()
{
	InsetCommandParams const & params = controller().params();

	dialog_->referenceED->setText(toqstr(params.getContents()));

	dialog_->nameED->setText(toqstr(params.getOptions()));
	dialog_->nameED->setReadOnly(!nameAllowed() && !readOnly());

	dialog_->typeCO->setCurrentItem(InsetRef::getType(params.getCmdName()));
	dialog_->typeCO->setEnabled(typeAllowed() && !readOnly());
	if (!typeAllowed())
		dialog_->typeCO->setCurrentItem(0);

	dialog_->sortCB->setChecked(sort_);

	// insert buffer list
	dialog_->bufferCO->clear();
	vector<string> const buffers = controller().getBufferList();
	for (vector<string>::const_iterator it = buffers.begin();
		it != buffers.end(); ++it) {
		dialog_->bufferCO->insertItem(toqstr(*it));
	}
	dialog_->bufferCO->setCurrentItem(controller().getBufferNum());

	updateRefs();
}


void QRef::apply()
{
	InsetCommandParams & params = controller().params();

	params.setCmdName(InsetRef::getName(dialog_->typeCO->currentItem()));
	params.setContents(fromqstr(dialog_->referenceED->text()));
	params.setOptions(fromqstr(dialog_->nameED->text()));
}


bool QRef::nameAllowed()
{
	return controller().docType() != ControlRef::LATEX &&
		controller().docType() != ControlRef::LITERATE;
}


bool QRef::typeAllowed()
{
	return controller().docType() != ControlRef::LINUXDOC &&
	    controller().docType() != ControlRef::DOCBOOK;
}


void QRef::setGoBack()
{
	dialog_->gotoPB->setText(qt_("&Go back"));
	QToolTip::remove(dialog_->gotoPB);
	QToolTip::add(dialog_->gotoPB, qt_("Go back"));
}


void QRef::setGotoRef()
{
	dialog_->gotoPB->setText(qt_("&Goto"));
	QToolTip::remove(dialog_->gotoPB);
	QToolTip::add(dialog_->gotoPB, qt_("Go to label"));
}


void QRef::gotoRef()
{
	string ref(fromqstr(dialog_->referenceED->text()));

	if (at_ref_) {
		// go back
		setGotoRef();
		controller().gotoBookmark();
	} else {
		// go to the ref
		setGoBack();
		controller().gotoRef(ref);
	}
	at_ref_ = !at_ref_;
}


void QRef::redoRefs()
{
	// Prevent these widgets from emitting any signals whilst
	// we modify their state.
	dialog_->refsLB->blockSignals(true);
	dialog_->referenceED->blockSignals(true);

	dialog_->refsLB->setAutoUpdate(false);
	dialog_->refsLB->clear();

	// need this because Qt will send a highlight() here for
	// the first item inserted
	QString const tmp(dialog_->referenceED->text());

	for (std::vector<string>::const_iterator iter = refs_.begin();
		iter != refs_.end(); ++iter) {
		if (sort_)
			dialog_->refsLB->inSort(toqstr(*iter));
		else
			dialog_->refsLB->insertItem(toqstr(*iter));
	}

	dialog_->referenceED->setText(tmp);

	for (unsigned int i = 0; i < dialog_->refsLB->count(); ++i) {
		if (tmp == dialog_->refsLB->text(i))
			dialog_->refsLB->setCurrentItem(i);
	}

	dialog_->refsLB->setAutoUpdate(true);
	dialog_->refsLB->update();

	// Re-activate the emission of signals by these widgets.
	dialog_->refsLB->blockSignals(false);
	dialog_->referenceED->blockSignals(false);
}


void QRef::updateRefs()
{
	refs_.clear();
	if (at_ref_)
		gotoRef();
	dialog_->refsLB->clear();
	string const name = controller().getBufferName(dialog_->bufferCO->currentItem());
	refs_ = controller().getLabelList(name);
	dialog_->sortCB->setEnabled(!refs_.empty());
	dialog_->refsLB->setEnabled(!refs_.empty());
	redoRefs();
}
