#include <stdlib.h>
#include <stdio.h>
#include <errno.h>
#include <string.h>
#include <SDL.h>
#include "lconf2bin.h"
#include "sbmp.h"

static int mainblock2bin(LSB_Main *block,SDL_RWops *rw,int *size);
static int overrideblock2bin(LSB_Override *block,SDL_RWops *rw, int *size);
static int objectblock2bin(LSB_Objects *object,SDL_RWops *rw, int *size);
static int paletteblock2bin(LSB_Palette *block,SDL_RWops *rw, int *size);
static int iconblock2bin(SDL_Surface *block,SDL_RWops *rw, int *size);

/* Convert the level settings structure into binary data stream */
SDL_RWops *LevelSetting2bin(LevelSettings *settings,int *size) {
  SDL_RWops *rw;
  FILE *fp;
  int len=0;
  /* Open the file */
  fp=tmpfile();
  if(!fp) {
    perror("temp file");
    exit(1);
  }
  rw=SDL_RWFromFP(fp,0);
  if(!rw) {
    perror(SDL_GetError());
    exit(1);
  }
  mainblock2bin(settings->mainblock,rw,&len);
  if(settings->override) overrideblock2bin(settings->override,rw,&len);
  if(settings->objects) objectblock2bin(settings->objects,rw,&len);
  if(settings->palette) paletteblock2bin(settings->palette,rw,&len);
  if(settings->icon) iconblock2bin(settings->icon,rw,&len);
  /* Rewind the file */
  SDL_RWseek(rw,0,SEEK_SET);
  *size=len;
  return rw;
}

/* Convert the main block into binary datastream */
static int mainblock2bin(LSB_Main *block,SDL_RWops *rw, int *size) {
  Uint16 len=0,i16;
  Uint8 type=0x01,i8;
  char tmps[4];
  int r;
  /* Block header */
  SDL_RWwrite(rw,&len,2,1);	/* We'll fix this later */
  SDL_RWwrite(rw,&type,1,1);
  /* Last modified date */
  SDL_RWwrite(rw,&type,1,1); len++;
  strncpy(tmps,block->modified,4);
  i16=atoi(tmps);
  SDL_RWwrite(rw,&i16,2,1); len+=2;
  tmps[2]='\0';
  for(r=4;r<12;r+=2) {
    strncpy(tmps,block->modified+r,2);
    i8=atoi(tmps);
    SDL_RWwrite(rw,&i8,1,1); len++;
  }
  /* Level name (including terminating \0) */
  if(block->name) {
    type=0x02;
    SDL_RWwrite(rw,&type,1,1); len++;
    SDL_RWwrite(rw,block->name,1,strlen(block->name)+1);
    len+=strlen(block->name)+1;
  }
  /* Level type */
  type=0x03;
  SDL_RWwrite(rw,&type,1,1); len++;
  switch(block->type) {
    case LuolaLevel: i8=0x01; break;
    case Luola8bitLevel: i8=0x02; break;
  }
  SDL_RWwrite(rw,&i8,1,1); len++;
  /* Horizontal scaling */
  if(block->scalex!=0) {
    type=0x04;
    SDL_RWwrite(rw,&type,1,1); len++;
    SDL_RWwrite(rw,&block->scalex,1,1); len++;
  }
  /* Vertical scaling */
  if(block->scaley!=0) {
    type=0x05;
    SDL_RWwrite(rw,&type,1,1); len++;
    SDL_RWwrite(rw,&block->scaley,1,1); len++;
  }
  /* Smooth scaling */
  if(block->smooth_scale==0) {
    type=0x06;
    SDL_RWwrite(rw,&type,1,1); len++;
    SDL_RWwrite(rw,&block->smooth_scale,1,1); len++;
  }
  /* Set proper length */
  SDL_RWseek(rw,-(len+3),SEEK_CUR);
  SDL_RWwrite(rw,&len,2,1);
  SDL_RWseek(rw,len+1,SEEK_CUR);
  printf("Main block length: %d (+3 bytes)\n",len);
  len+=3;
  /* Done */
  *size+=len;
}

/* Convert the override block into binary datastream */
static int overrideblock2bin(LSB_Override *block,SDL_RWops *rw, int *size) {
  Uint16 len=0,i16;
  Uint8 type=0x02,i8;
  int r;
  /* Block header */
  SDL_RWwrite(rw,&len,2,1);	/* We'll fix this later */
  SDL_RWwrite(rw,&type,1,1);
  /* Critters enabled */
  if(block->critters_enabled!=0xff) {
    type=0x01;
    SDL_RWwrite(rw,&type,1,1); len++;
    SDL_RWwrite(rw,&block->critters_enabled,1,1); len++;
  }
  /* Indestructable bases */
  if(block->indstr_base!=0xff) {
    type=0x02;
    SDL_RWwrite(rw,&type,1,1); len++;
    SDL_RWwrite(rw,&block->indstr_base,1,1); len++;
  }
  /* Stars */
  if(block->stars!=0xff) {
    type=0x03;
    SDL_RWwrite(rw,&type,1,1); len++;
    SDL_RWwrite(rw,&block->stars,1,1); len++;
  }
  /* Snowfall */
  if(block->snowfall!=0xff) {
    type=0x04;
    SDL_RWwrite(rw,&type,1,1); len++;
    SDL_RWwrite(rw,&block->snowfall,1,1); len++;
  }
  /* Turrets */
  if(block->turrets!=0xff) {
    type=0x05;
    SDL_RWwrite(rw,&type,1,1); len++;
    SDL_RWwrite(rw,&block->turrets,1,1); len++;
  }
  /* Jumpgates */
  if(block->jumpgates!=0xff) {
    type=0x06;
    SDL_RWwrite(rw,&type,1,1); len++;
    SDL_RWwrite(rw,&block->jumpgates,1,1); len++;
  }
  /* Cows */
  if(block->cows!=0xff) {
    type=0x07;
    SDL_RWwrite(rw,&type,1,1); len++;
    SDL_RWwrite(rw,&block->cows,1,1); len++;
  }
  /* Fish */
  if(block->fish!=0xff) {
    type=0x08;
    SDL_RWwrite(rw,&type,1,1); len++;
    SDL_RWwrite(rw,&block->fish,1,1); len++;
  }
  /* Birds */
  if(block->birds!=0xff) {
    type=0x09;
    SDL_RWwrite(rw,&type,1,1); len++;
    SDL_RWwrite(rw,&block->birds,1,1); len++;
  }
  /* Bats */
  if(block->bats!=0xff) {
    type=0x0a;
    SDL_RWwrite(rw,&type,1,1); len++;
    SDL_RWwrite(rw,&block->bats,1,1); len++;
  }
  /* Set proper length */
  SDL_RWseek(rw,-(len+3),SEEK_CUR);
  SDL_RWwrite(rw,&len,2,1);
  SDL_RWseek(rw,len+1,SEEK_CUR);
  printf("Override block length: %d (+3 bytes)\n",len);
  len+=3;
  /* Done */
  *size+=len;
}

/* Convert the objects block into binary datastream */
static int objectblock2bin(LSB_Objects *object,SDL_RWops *rw, int *size) {
  Uint16 len=0,i16;
  Uint8 type=0x03,i8;
  int r;
  /* Block header */
  SDL_RWwrite(rw,&len,2,1);	/* We'll fix this later */
  SDL_RWwrite(rw,&type,1,1);
  /* Object */
  r=19; /* Length of the block data */
  while(object) {
    SDL_RWwrite(rw,&r,2,1); len+=2;
    SDL_RWwrite(rw,&object->type,1,1); len++;
    SDL_RWwrite(rw,&object->x,4,1); len+=4;
    SDL_RWwrite(rw,&object->y,4,1); len+=4;
    SDL_RWwrite(rw,&object->ceiling_attach,1,1); len++;
    SDL_RWwrite(rw,&object->value,1,1); len++;
    SDL_RWwrite(rw,&object->id,4,1); len+=4;
    SDL_RWwrite(rw,&object->link,4,1); len+=4;
    object=object->next;
  }
  /* Set proper length */
  SDL_RWseek(rw,-(len+3),SEEK_CUR);
  SDL_RWwrite(rw,&len,2,1);
  SDL_RWseek(rw,len+1,SEEK_CUR);
  printf("Object block length: %d (+3 bytes)\n",len);
  len+=3;
  /* Done */
  *size+=len;

}

/* Spit out the palette block */
static int paletteblock2bin(LSB_Palette *block,SDL_RWops *rw, int *size) {
  Uint16 len=256;
  Uint8 type=0x04;
  /* Block header */
  SDL_RWwrite(rw,&len,2,1);
  SDL_RWwrite(rw,&type,1,1);
  /* Data */
  SDL_RWwrite(rw,&block->entries,1,256);
  *size+=len+3;
}

/* Pack up the icon into an SBMP */
static int iconblock2bin(SDL_Surface *block,SDL_RWops *rw, int *size) {
  Uint8 hascolorkey=0;
  Uint32 colorkey=0;
  SDL_Color color;
  Uint32 len;
  Uint8 *data;
  Uint8 type=0x05;
  Uint8 notunique=1;
  int r,breakfree=0;
  /* See if the icon has a colorkey (most likely) */
  if((block->flags&SDL_SRCCOLORKEY)) {
    colorkey=block->format->colorkey;
    printf("Icon has colorkey, pixevalue: 0x%x\n",colorkey);
    hascolorkey=1;
  }
  /* A hack to preserve the colorkey */
  /* SBMP rewrites the palette, so we need give the colorkey an unique colour */
  if(block->format->BytesPerPixel==1 && hascolorkey) {
    color=block->format->palette->colors[colorkey];
    while(notunique||breakfree>10000) {
      notunique=0;
      for(r=0;r<block->format->palette->ncolors;r++) {
        if(r==colorkey) continue;
	if(bcmp(&color,&block->format->palette[r],3)==0) {notunique=1;break;}
      }
      if(notunique) {
        color.r=rand()%255;
	color.g=rand()%255;
	color.b=rand()%255;
      }
      breakfree++;
    }
    if(breakfree>10000) {
      printf("Couldn't find a unique colorkey !\n");
    } else {
      printf("Unique colorkey colour is RGB(%d,%d,%d)\n",color.r,color.g,color.b);
      block->format->palette->colors[colorkey]=color;
    }
  }
  /* Convert the SDL_Surface into SBMP */
  data=surface_to_sbmp(&len,block);
  if(!data) {
    printf("Cannot pack icon !\n");
    return 0;
  }
  /* Another hack. Find the colorkey palette entry (it might have moved somewhere else) */
  if(block->format->BytesPerPixel==1 && hascolorkey) {
    Uint8 ccount;
    Uint8 *ptr;
    ptr=data+2;
    ccount=*ptr; ptr++;
    printf("SBMP has %d colours\n",ccount);
    for(r=0;r<ccount;r++) {
      if(ptr[0]==color.r && ptr[1]==color.g && ptr[2]==color.b) {
        printf("Colorkey found at SBMP palette entry %d !\n",r);
	colorkey=r;
      }
      ptr+=3;
    }
  }
  /* Write data to file */
  len+=5;
  SDL_RWwrite(rw,&len,2,1);
  SDL_RWwrite(rw,&type,1,1);
  SDL_RWwrite(rw,&hascolorkey,1,1);
  SDL_RWwrite(rw,&colorkey,4,1);
  SDL_RWwrite(rw,data,1,len);
  printf("Icon block length: %d (+3 bytes)\n",len);
  *size+=len+3;
}
