/*
 * Luola - 2D multiplayer cavern-flying game
 * Copyright (C) 2003 Calle Laakkonen
 *
 * File        : weather.c
 * Description : Weather effects
 * Author(s)   : Calle Laakkonen
 *
 * Luola is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * Luola is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include <stdlib.h>
#include "defines.h"
#include "console.h"
#include "game.h"
#include "level.h"
#include "player.h"
#include "weather.h"

/* List of weather effects */
struct WeatherFXList {
  WeatherFX type;
  int x, y;
  int timer1;
  int timer2;
  struct WeatherFXList *next;
  struct WeatherFXList *prev;
};

/* Internally used globals */
static struct WeatherFXList *weather_list;
static int weather_snowsource_i2;
static int weather_wind_targ_vector;
static int weather_windy;

/* Exported globals */
double weather_wind_vector; /* Ok, so this really isn't a vector, but we only need the X component */

/* Internally used function prototypes */
static struct WeatherFXList *weather_remove_fx(struct WeatherFXList *list);
static inline void draw_weather_fx(struct WeatherFXList *list);

/* Initialize */
void init_weather(void) {
  weather_list=NULL;
  weather_wind_vector=0;
  weather_windy=1;
}

/* Deinitialize. */
void deinit_weather(void) {
  struct WeatherFXList *next;
  while(weather_list) {
    next=weather_list->next;
    free(weather_list);
    weather_list=next;
  }
}

/* Prepare weather for the next level */
void prepare_weather(void) {
  struct WeatherFXList *newentry,*list=NULL;
  int r;
  deinit_weather();
  if(game_settings.ls.snowfall==0) return;
  weather_snowsource_i2=lev_level.width/15;
  for(r=1;r<weather_snowsource_i2;r++) {
    newentry=(struct WeatherFXList*)malloc(sizeof(struct WeatherFXList));
    newentry->x=r*lev_level.width/15+rand()%15;
    if(newentry->x>=lev_level.width) {free(newentry); continue;}
    newentry->y=find_rainy(newentry->x);
    if(newentry->y<0) {free(newentry); continue;}
    newentry->type=Snowsource;
    newentry->timer1=SNOWFLAKE_INTERVAL;
    newentry->timer2=0;
    newentry->next=NULL;
    newentry->prev=list;
    if(list) { list->next=newentry; list=list->next; } else list=newentry;
  }
  if(list) while(list->prev) list=list->prev; // Rewind
  weather_list=list;
}

/* Remove one element from the weather list, returns pointer to next */
struct WeatherFXList *weather_remove_fx(struct WeatherFXList *list) {
  struct WeatherFXList *next;
  next=list->next;
  if(list->prev) list->prev->next=list->next; else weather_list=list->next;
  if(next) next->prev=list->prev;
  free(list);
  return next;
}

/* Add a snowflake */
void add_snowflake(int x,int y) {
  struct WeatherFXList *newentry=NULL,*list=NULL;
  if(weather_list!=NULL) {
    list=weather_list;
    while(list->next) list=list->next;
  }
  newentry=(struct WeatherFXList*)malloc(sizeof(struct WeatherFXList));
  newentry->next=NULL;
  newentry->prev=list;
  newentry->x=x; newentry->y=y;
  newentry->type=Snowflake;
  if(weather_list==NULL)
    weather_list=newentry;
  else
    list->next=newentry;
}

/* Animate */
void animate_weather(void) {
  struct WeatherFXList *list=weather_list;
  signed char solid;
  if(weather_windy<=0) {
    int tmpi;
    weather_windy=rand()%MAX_WIND_TIME;
    tmpi=rand()%5;
    if(rand()%3==0) { if(weather_wind_targ_vector<0) weather_wind_targ_vector=tmpi; else weather_wind_targ_vector=-tmpi; }
    else { if(weather_wind_targ_vector<0) weather_wind_targ_vector=-tmpi; else weather_wind_targ_vector=tmpi; }
  } else weather_windy--;
  if(weather_wind_targ_vector<weather_wind_vector) weather_wind_vector-=0.05;
  else if(weather_wind_targ_vector>weather_wind_vector) weather_wind_vector+=0.05;
  while(list) {
    switch(list->type) {
      case Snowsource:
	list->timer1--;
	list->timer2++;
	if(list->timer2<weather_snowsource_i2)
	  list->x++;
	else {
	  list->x--;
	  if(list->timer2==weather_snowsource_i2*2) list->timer2=0;
	}
	if(list->timer1==0) {
	  add_snowflake(list->x,list->y);
	  list->timer1=SNOWFLAKE_INTERVAL;
	}
	break;
      case Snowflake:
        list->y++;
	list->x+=(rand()%4)-2+round(weather_wind_vector);
        if(list->y>=lev_level.height) {list=weather_remove_fx(list); continue;}
        if(list->x<=0 || list->x>=lev_level.width) {list=weather_remove_fx(list); continue;}
        solid=hit_solid(list->x,list->y);
        if(solid) {
          alter_level(list->x,list->y-1,1,Ice); /* change the recurse value (1) into 2 to get a nice thick coating of snow... ;) */
          list=weather_remove_fx(list);
          continue;
        }
	break;
    }
    if(list->type==Snowflake) draw_weather_fx(list);
    list=list->next;
  }
}

/* Draw on screen */
static inline void draw_weather_fx(struct WeatherFXList *list) {
  int x,y,p;
#ifndef HAVE_LIBSDL_GFX
  if( SDL_MUSTLOCK(screen) ) SDL_LockSurface(screen);
#endif
  for(p=0;p<4;p++) {
#ifdef DONT_UPDATE_DEAD
    if(players[p].active && players[p].pilot->dead==0) {
#else
    if(players[p].active) {
#endif
      x=list->x-cam_rects[p].x+lev_rects[p].x;
      y=list->y-cam_rects[p].y+lev_rects[p].y;
      if((x>lev_rects[p].x && x<lev_rects[p].x+lev_rects[p].w)
      && (y>lev_rects[p].y && y<lev_rects[p].y+lev_rects[p].h)) {
        putpixel(screen,x,y,col_snow);
      }
    }
  }
#ifndef HAVE_LIBSDL_GFX
  if (SDL_MUSTLOCK(screen)) SDL_UnlockSurface(screen);
#endif

}


