/*
 * Luola - 2D multiplayer cavern-flying game
 * Copyright (C) 2003 Calle Laakkonen
 *
 * File        : weapon.c
 * Description : Weapon code
 * Author(s)   : Calle Laakkonen
 *
 * Luola is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * Luola is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include <stdlib.h>
#include <string.h>
#include <math.h>
#include "defines.h"

#include "fs.h"
#include "console.h"
#include "level.h"
#include "weapon.h"
#include "sweapon.h"
#include "player.h"
#include "particle.h"
#include "game.h"
#include "special.h"
#include "critter.h"
#include "weather.h"

#if HAVE_LIBSDL_MIXER
#include "audio.h"
#endif

#include "hole.h"

/* Explosion */
typedef struct {
  int x;
  int y;
  int frame;
  ProjectileType cluster;
} Explosion;

/* Linked list containing all explosions */
struct Explosion_list {
  Explosion *explosion;
  struct Explosion_list *next;
  struct Explosion_list *prev;
};

/* Internally used globals */
static struct Projectile_list *projectiles;
#ifdef PROJECTILE_CACHE
static struct Projectile_list *projectile_cache;
static int projectile_cache_count;
#endif
static struct Explosion_list *explosions;
static struct Projectile_list *last_proj;

static Vector wea_gravity;
static SDL_Surface *wea_explosion[EXPL_FRAMES];
static int gravity_weapon_count; /* How many gravity weapons there are */

static Uint8 wea_gnaw[16][16];

static const double G=667.2590;

/* Internally used functions */
static inline void draw_projectile(Projectile *projectile);
static inline void draw_explosions(void);

/* Remove a projectile from the list */
static struct Projectile_list *remove_projectile(struct Projectile_list *list) {
  struct Projectile_list *next;
#ifdef PROJECTILE_CACHE
  if(projectile_cache_count<PROJECTILE_CACHE) {
    if(list->prev)
      list->prev->next=list->next;
    else
      projectiles=list->next;
    next=list->next;
    if(next)
      list->next->prev=list->prev;
    else last_proj=list->prev;
    list->next=projectile_cache;
    list->prev=NULL;
    projectile_cache=list;
    projectile_cache_count++;
    return next;
  }
#endif
  free(list->projectile);
  if(list->prev)
    list->prev->next=list->next;
  else
    projectiles=list->next;
  next=list->next;
  if(next)
    list->next->prev=list->prev;
  else last_proj=list->prev;
  free(list);
  return next;
}

/* Can this type of a weapon collide with a ship ? */
static char can_collide_with_ship(ProjectileType type) {
  if(type==GravityWell) return 0;
  if(type==FireStarter) return 0;
  if(type==Decor) return 0;
  if(type==Napalm) return 0;	/* Napalm has its own collision handler */
  if(type==Acid) return 0;	/* Acid too has its own collision handler */
  return 1;			/* Rest hit the ship */
}
/* How near this projectile has to be to the ship to explode ? */
static unsigned char hit_ship_proximity(ProjectileType type) {
  if(type==Energy) return 10;
  if(type==Mine||type==MagMine) return 16;
  if(type==Grenade) return 16;
  if(type==Missile) return 16;
  if(type==Tag) return 16;
  return 8;
}

/* Initialize */
void init_weapons(void) {
  LDAT *datafile;
  int r;
  projectiles=NULL;
  last_proj=NULL;
  explosions=NULL;
  wea_gravity.y=-WEAP_GRAVITY;
  wea_gravity.x=0;
  datafile=ldat_open_file((char*)getfullpath(GFX_DIRECTORY,"explosion.ldat"));
  for(r=0;r<EXPL_FRAMES;r++) {
    wea_explosion[r]=load_image_ldat(datafile,0,1,"EXPL",r);
  }
  ldat_free(datafile);
  set_hole_size(game_settings.holesize);
#ifdef PROJECTILE_CACHE
  projectile_cache=NULL;
  projectile_cache_count=0;
#endif
  things_loaded[TL_WEAPONS]=1;
}

void deinit_weapons(char just_clear) {
  struct Explosion_list *enext;
  struct Projectile_list *pnext;
  int r;
  while(explosions) {
    enext=explosions->next;
    free(explosions->explosion);
    free(explosions);
    explosions=enext;
  }
  while(projectiles) {
    pnext=projectiles->next;
    free(projectiles->projectile);
    free(projectiles);
    projectiles=pnext;
  }
#ifdef PROJECTILE_CACHE
  while(projectile_cache) {
    pnext=projectile_cache->next;
    free(projectile_cache->projectile);
    free(projectile_cache);
    projectile_cache=pnext;
  }
  projectile_cache_count=0;
#endif
  last_proj=0;
  gravity_weapon_count=0;
  if(!just_clear) {
    for(r=0;r<EXPL_FRAMES;r++)
      SDL_FreeSurface(wea_explosion[r]);
  }
}
/* Set the proper hole size */
void set_hole_size(Uint8 size) {
  int x,y;
  for(x=0;x<HOLE_W;x++)
    for(y=0;y<HOLE_H;y++)
      switch(size) {
        case 0: /* Tiny */
	  wea_gnaw[x][y]=wea_gnaw_tiny[x][y];
        case 1: /* Small */
	  wea_gnaw[x][y]=wea_gnaw_small[x][y];
	  break;
        case 2: /* Normal */
          wea_gnaw[x][y]=wea_gnaw_normal[x][y];
          break;
        case 3: /* Big */
	  wea_gnaw[x][y]=wea_gnaw_big[x][y];
      }
}

Projectile *make_projectile(int x,int y,Vector v) {
  Projectile *newproj;
#ifdef PROJECTILE_CACHE
  if(projectile_cache_count) {
    struct Projectile_list *next=projectile_cache->next;
    newproj=projectile_cache->projectile;
    free(projectile_cache);
    projectile_cache=next;
    projectile_cache_count--;
  } else
#endif  
  newproj=(Projectile*)malloc(sizeof(Projectile));

  newproj->x=x;
  newproj->y=y;
  newproj->gravity=NULL;
  newproj->vector.x=v.x*7.0;
  newproj->vector.y=v.y*7.0;
  newproj->color=col_default;
  newproj->type=Cannon;
  newproj->owner=-1;
  newproj->primed=0;
  newproj->life=-1;
  newproj->var1=0;
  newproj->maxspeed=PROJ_MAXSPEED;
  if(game_settings.gravity_bullets)
    newproj->gravity=&wea_gravity;
  else
    newproj->gravity=NULL;
  newproj->wind_affects=game_settings.wind_bullets;
  return newproj;
}
/* Add a new projectile to list */
void add_projectile(Projectile *proj) {
  struct Projectile_list *newentry;
  newentry=(struct Projectile_list*)malloc(sizeof(struct Projectile_list));
  newentry->next=NULL;
  newentry->prev=last_proj;
  newentry->projectile=proj;
  if(last_proj==NULL)
    projectiles=newentry;
  else
    last_proj->next=newentry;
  last_proj=newentry;
  if(proj->type==GravityWell) gravity_weapon_count++;
}

void add_explosion(int x,int y,ProjectileType cluster) {
  struct Explosion_list *newentry;
  char solid;
  int fx,fy,rx,ry;
#if HAVE_LIBSDL_MIXER
  playwave_3d(WAV_EXPLOSION,x,y);
#endif
  if(game_settings.explosions||(cluster!=Noproj&&cluster!=Zap)) {
    newentry=(struct Explosion_list*)malloc(sizeof(struct Explosion_list));
    newentry->next=explosions;
    newentry->prev=NULL;
    newentry->explosion=(Explosion*)malloc(sizeof(Explosion));
    newentry->explosion->x=x;
    newentry->explosion->y=y;
    newentry->explosion->frame=0;
    if(cluster==Zap) newentry->explosion->cluster=Noproj; else newentry->explosion->cluster=cluster;
    if(explosions==NULL)
      explosions=newentry;
    else {
      explosions->prev=newentry;
      explosions=newentry;
    }
  }
  /* Gnaw a hole in terrain */
  if(cluster!=Zap) {
  if(lev_level.solid[x][y] != TER_INDESTRUCT && !(level_settings.indstr_base && (lev_level.solid[x][y] == TER_BASE||lev_level.solid[x][y]==TER_BASEMAT)))
  for(fx=0;fx<HOLE_W;fx++)
    for(fy=0;fy<HOLE_H;fy++) {
      if(wea_gnaw[fx][fy]) continue;
      rx=fx-HOLE_W2+x;
      ry=fy-HOLE_H2+y;
      if(rx<0 || ry<0 || rx>=lev_level.width || ry>=lev_level.height) continue;
      solid=lev_level.solid[rx][ry];
      if(solid==TER_FREE) continue;
      if(solid==TER_INDESTRUCT || is_water(rx,ry)) continue;
      if(level_settings.indstr_base && (solid==TER_BASE||solid==TER_BASEMAT) ) continue;
      if(solid==TER_BASE && lev_level.base_area>0) lev_level.base_area--;
      if(solid==TER_UNDERWATER||solid==TER_ICE) {
        lev_level.solid[rx][ry]=TER_WATER;
	putpixel(lev_level.terrain,rx,ry,lev_watercol);
      } else {
        lev_level.solid[rx][ry]=TER_FREE;
        putpixel(lev_level.terrain,rx,ry,col_black);
      }
    }
  } else { /* Burn terrain */
    for(fx=0;fx<HOLE_W;fx++)
      for(fy=0;fy<HOLE_H;fy++) {
        if(wea_gnaw[fx][fy]) continue;
	rx=fx-HOLE_W2+x;
        ry=fy-HOLE_H2+y;
	if(rx<0 || ry<0 || rx>=lev_level.width || ry>=lev_level.height) continue;
        solid=lev_level.solid[rx][ry];
	if(solid==TER_GROUND||solid==TER_UNDERWATER||solid==TER_COMBUSTABLE||solid==TER_COMBUSTABL2||solid==TER_TUNNEL||solid==TER_WALKWAY)
	  putpixel(lev_level.terrain,rx,ry,col_gray);
      }
    rx=x+HOLE_W2;
    ry=y+HOLE_H2;
    solid=lev_level.solid[rx][ry];
    if(solid==TER_COMBUSTABLE||solid==TER_COMBUSTABL2) start_burning(rx,ry);
  }
}

/**** WEAPON ANIMATION CODE ****/
void animate_weapons(void) {
  struct Projectile_list *plist=projectiles,*plist2,*plist3;
  struct Explosion_list *elist=explosions,*enext;
  ProjectileType expl_cluster;
  signed int explode,solid;
  double f1;
  Projectile *proj;
  Explosion *expl;
  if(plist) {
    while(plist) {
      proj=plist->projectile;
      if(proj->type!=Zap) expl_cluster=Noproj; else expl_cluster=Zap;
      /*** Collision detection ***/
      if(proj->primed==0) {
      explode=0;
      /* Collide with a ship ? */
      if(can_collide_with_ship(proj->type)) {
        explode=(int)hit_ship(proj->x,proj->y,(proj->type!=Boomerang)?proj->owner:-1,hit_ship_proximity(proj->type));
	if(explode) explode=ship_damage((Ship*)explode,proj);
	if(explode) {
	  if(explode==-2) {/* Did the ship just swallow the projectile ? */
	    plist=remove_projectile(plist);
	    continue;
	  }
	}
	/* Collide with a level special ? (Same types of projectiles that collide with ships) */
	if(explode==0) explode=hit_special(proj);
      }
      /* Collide with a critter or a pilot ? Also check if the projectile is underwater */
      if(explode==0 && proj->type!=GravityWell && proj->type!=Energy && proj->type!=MagWave && proj->type!=FireStarter) {
        if(proj->type!=Decor) explode=hit_critter(proj->x,proj->y,proj->type);
        if(explode && (proj->type==Spear||proj->type==FireStarter||proj->type==Acid)) explode=0;
	else if(explode && (proj->type==Tag||proj->type==Napalm)) { explode=0; if(proj->vector.x>proj->vector.y) proj->vector.y*=-1; else proj->vector.x*=-1; }
        if(explode==0&&proj->type!=Decor) explode=hit_pilot(proj->x,proj->y,proj->owner);
      }
      /* Check if the projectile goes outside its boundaries */
      if(proj->x<=1) explode=-1; else if(proj->x>=lev_level.width-1) explode=-1;
      if(proj->y<=1) explode=-1; else if(proj->y>=lev_level.height-1) explode=-1;
      /* Mines collide with other projectiles */
      if(explode==0 && (proj->type==DividingMine || proj->type==Mine || proj->type==MagMine)) { /* Mines detonate when they hit other projectiles */
        plist3=projectiles;
	while(plist3) {
	  if(proj==plist3->projectile) {plist3=plist3->next; continue;}
	  if(plist3->projectile->type==GravityWell || plist3->projectile->type==Energy||plist3->projectile->type==MagWave) {plist3=plist3->next; continue;}
	  if(abs(proj->x-plist3->projectile->x)<2)
	    if(abs(proj->y-plist3->projectile->y)<2) {
	      explode++;
	      /* The other projectile is destroyed as well (except gravitywells and energybolts) */
	      if(plist3->prev) plist3->prev->next=plist3->next; else projectiles=plist3->next;
	      if(plist3->next) plist3->next->prev=plist3->prev; else last_proj=plist3->prev;
	      free(plist3->projectile);
	      free(plist3);
	      break;
	    }
	  plist3=plist3->next;
	}
      }
      /* A hack to make Claybomb react with things other than ground */
      if(explode && proj->type==Claybomb) {
        lev_level.solid[proj->x][proj->y]=(lev_level.solid[proj->x][proj->y]==TER_WATER)?TER_UNDERWATER:TER_GROUND;
	explode=0;
      }
      /* Terrain collision detection */
      if(explode==0 && proj->type!=Energy && proj->type!=GravityWell && proj->type!=MagWave) {
#ifdef HIRES_PROJ
	solid=hit_solid_line(proj->x,proj->y,proj->x-round(proj->vector.x),proj->y-round(proj->vector.y),&proj->x,&proj->y);
	if(solid<0) {
#else
	solid=lev_level.solid[proj->x][proj->y];
	if(solid==TER_WATER||(solid>=TER_WATERFU&&solid<=TER_WATERFR)) {
#endif
	  switch(proj->type) {
	    case DividingMine:
	    case Tag:
	    case Ember:
	    case Napalm:
	      explode=1; break;
	    case Acid:
	    case Decor:
	    case Landmine:
	      explode=-1; break;
	    case Snowball:	/* Snowball is special because it reacts with water */
	      explode=-1;
	      alter_level(proj->x,proj->y,solid==TER_GROUND?30:15,Ice);
#if HAVE_LIBSDL_MIXER
	      playwave_3d(WAV_FREEZE,proj->x,proj->y);
#endif
	    default: break;
	  }
	} else if(solid) {
	  if(solid==TER_EXPLOSIVE) expl_cluster=Cannon;
	  else if(solid==TER_EXPLOSIVE2) expl_cluster=Grenade;
	  switch(proj->type) {	/* Different types of reactions when projectiles hit something solid */
	    case FireStarter:
	      if((solid==TER_COMBUSTABLE||solid==TER_COMBUSTABL2)||(solid==TER_EXPLOSIVE||solid==TER_EXPLOSIVE2)) {
	        start_burning(proj->x,proj->y);
		explode=-1;
	      }
	      break;
	    case Landmine:
	    case Tag:
	    case Napalm:
	      if(proj->type==Landmine) proj->life=-1;
	      else if(solid==TER_COMBUSTABLE || solid==TER_COMBUSTABL2 || solid==TER_EXPLOSIVE || solid==TER_EXPLOSIVE2) start_burning(proj->x,proj->y);
	      proj->vector.x=0;
	      proj->vector.y=0;
	      proj->wind_affects=0;
	      break;
	    case Plastique:
	      explode=-1;
	      alter_level(proj->x,proj->y,5+3*(game_settings.holesize),Explosive);
#if HAVE_LIBSDL_MIXER
	      playwave_3d(WAV_SWOOSH,proj->x,proj->y);
#endif
	      break;
	    case Claybomb:
	      explode=-1;
	      alter_level(proj->x,proj->y,15,Earth);
#if HAVE_LIBSDL_MIXER
	      playwave_3d(WAV_SWOOSH,proj->x,proj->y);
#endif
	      break;
	    case Snowball:
	      explode=-1;
	      alter_level(proj->x,proj->y,solid==TER_GROUND?30:15,Ice);
#if HAVE_LIBSDL_MIXER
	      playwave_3d(WAV_FREEZE,proj->x,proj->y);
#endif
	      break;
	    case Acid:
	      explode=-1;
	      start_melting(proj->x,proj->y,10);
#if HAVE_LIBSDL_MIXER
              playwave_3d(WAV_STEAM,proj->x,proj->y);
#endif
	      break;
	    case Waterjet:
	    case Decor: explode=-1; break;
	    case Spear: {
	      int dx,dy;
	      dx=sin(proj->angle)*6;
	      dy=cos(proj->angle)*6;
	      draw_line(lev_level.terrain,proj->x-dx,proj->y-dy,proj->x+dx,proj->y+dy,proj->color);
	      explode=-1;
	      } break;
	    default:
	      explode=1;
	    break;
	  }
        }
      }
      /* End terrain code */
      /* Larger explosions */
      if(explode) {
        if(explode>0) add_explosion(proj->x,proj->y,expl_cluster);
        switch(proj->type) {
          case Grenade:
			spawn_clusters(proj->x,proj->y,16,-1,Cannon); spawn_clusters(proj->x,proj->y,16,-1,FireStarter);
			break;
          case MegaBomb:
			spawn_clusters(proj->x,proj->y,8,-1,Grenade); spawn_clusters(proj->x,proj->y,16,-1,FireStarter);
			break;
          case Missile:
			spawn_clusters(proj->x,proj->y,10,-1,Cannon); spawn_clusters(proj->x,proj->y,16,-1,FireStarter);
			break;
          case Boomerang:
		    spawn_clusters(proj->x,proj->y,4,-1,Cannon); spawn_clusters(proj->x,proj->y,16,-1,FireStarter);
			break;
          case Rocket:
            spawn_clusters(proj->x,proj->y,3,-1,Grenade);
            spawn_clusters(proj->x,proj->y,6,-1,Cannon);
            spawn_clusters(proj->x,proj->y,16,-1,FireStarter);
            break;
          case DividingMine:
            spawn_clusters(proj->x,proj->y,4,-1,Cannon);
            spawn_clusters(proj->x,proj->y,16,-1,FireStarter);
            break;
          case MagMine:
          case Mine:
            spawn_clusters(proj->x,proj->y,8,-1,Cannon);
            spawn_clusters(proj->x,proj->y,16,-1,FireStarter);
            break;
          case GravityWell: gravity_weapon_count--; break;
          case Mush:
            spawn_clusters(proj->x,proj->y,5,-1,Cannon);
            spawn_clusters(proj->x,proj->y,16,-1,FireStarter);
            break;
          case Ember:
            spawn_clusters(proj->x,proj->y,3*(proj->var1+1),-1,Napalm);
            spawn_clusters(proj->x,proj->y,16,-1,FireStarter);
            break;
	  case Mirv:
	    if(proj->var1==3)
	      spawn_clusters(proj->x,proj->y,3,-1,Grenade);
	    else if(proj->var1==2)
	      spawn_clusters(proj->x,proj->y,32,-1,Cannon);
	    else if(proj->var1==1)
	      spawn_clusters(proj->x,proj->y,12,-1,Cannon);
	    else 
	      spawn_clusters(proj->x,proj->y,6,-1,Cannon);
	    spawn_clusters(proj->x,proj->y,8,-1,FireStarter);
	    break;
	  case Zap:
	    proj->primed=proj->life;
	    break;
          default: break;
        }
        if((proj->type!=Zap || (proj->type==Zap && proj->life<=1)))
          plist=remove_projectile(plist);
	  continue;
      }
      }
      /* Weapon specific movement/fx code */
      switch(proj->type) {
        case Missile:
	  missile_movement(proj); break;
        case MagMine:
	  if(proj->primed<=0) magmine_movement(proj);
	  break;
	  case DividingMine:
	    dividingmine_movement(proj,projectiles); break;
  	  case Mine:
	    if(proj->primed<=0) mine_movement(proj);
	    break;
	  case Rocket:
	    rocket_movement(proj); break;
	  case Spear:
	    if(game_settings.gravity_bullets||gravity_weapon_count) spear_movement(proj);
	    break;
	  case Energy:
	    energy_movement(proj); break;
	  case Tag:
	    tag_movement(proj); break;
	  case Acid:
	    acid_movement(proj); break;
	  case Waterjet:
	    water_movement(proj); break;
	  case Ember:
	    ember_movement(proj); break;
	  case Napalm:
	    napalm_movement(proj); break;
	  case Boomerang:
	    boomerang_movement(proj); break;
	  case Mush:
	    mush_movement(proj); break;
	  case Zap:
	    expl_cluster=Zap;	/* This is to tell add_explosion to char the ground */
	  break;
	  case Mirv:
	    mirv_movement(proj); break;
	  case MagWave:
	    magwave_movement(proj); break;
	default: break;
      }
      /* End weapon specific movement/fx code */
      /* Check if someone is operating a force field nearby */
      if(proj->type!=GravityWell) {
        ShipList *ship=getshiplist();
        while(ship) {
          if(ship->ship->shieldup) {
  	    f1=hypot(abs(ship->ship->x-proj->x),abs(ship->ship->y-proj->y));
	    if(f1<100) {
	      proj->vector.x+=((ship->ship->x-proj->x)/(abs(ship->ship->x-proj->x)+0.1))/1.3;
	      proj->vector.y+=((ship->ship->y-proj->y)/(abs(ship->ship->y-proj->y)+0.1))/1.3;
	      if(proj->type==Rocket||proj->type==Mirv) proj->angle=atan2(proj->vector.x,proj->vector.y);
	    }
	  } else if(ship->ship->antigrav) {
	    f1=hypot(abs(ship->ship->x-proj->x),abs(ship->ship->y-proj->y));
	    if(f1<250 && proj->gravity) {
	      proj->vector.y-=proj->gravity->y*2.0;
	    }
	  }
	  ship=ship->next;
        }
      }
      /* Is this is a gravity weapon ? */
      if(proj->type==GravityWell && proj->primed==0) {
        ShipList *ship=getshiplist();
        plist2=projectiles;	/* Attract other projectiles */
        while(plist2) {
          if(plist2->projectile->type!=GravityWell && plist2->projectile->type!=Energy) {
            if(abs(plist2->projectile->x-proj->x)<150)
              if(abs(plist2->projectile->y-proj->y)<150) {
                 plist2->projectile->vector.x-=((proj->x-plist2->projectile->x)/(abs(proj->x-plist2->projectile->x)+0.1));
                 plist2->projectile->vector.y-=((proj->y-plist2->projectile->y)/(abs(proj->y-plist2->projectile->y)+0.1));
                 if(plist2->projectile->type==Rocket||plist2->projectile->type==Mirv) plist2->projectile->angle=atan2(plist2->projectile->vector.x,plist2->projectile->vector.y);
              }
          }
          plist2=plist2->next;
        }
	while(ship) {	/* Attract ships */
	  if(abs(ship->ship->x-proj->x)<100)
	    if(abs(ship->ship->y-proj->y)<100) {
	      ship->ship->vector.x+=(ship->ship->x-proj->x)/(abs(ship->ship->x-proj->x)+0.01)/1.3;
	      ship->ship->vector.y+=(ship->ship->y-proj->y)/(abs(ship->ship->y-proj->y)+0.01)/1.3;
	    }
	  ship=ship->next;
	}
	cow_storm(proj->x,proj->y); /* Attract critters */
      }
      /* End gravity weapon code */
      /* Common movement code */
      proj->x-=round(proj->vector.x);
      proj->y-=round(proj->vector.y);
      if(proj->x<0) proj->x=0; else if(proj->x>=lev_level.width) proj->x=lev_level.width-1;
      if(proj->y<0) proj->y=0; else if(proj->y>=lev_level.height) proj->y=lev_level.height-1;
      if(proj->wind_affects && lev_level.solid[proj->x][proj->y]==TER_FREE)
	proj->vector.x-=weather_wind_vector/50.0;
      if(proj->gravity) {
        proj->vector=addVectors(&proj->vector,proj->gravity);
	proj->vector.x/=1.005;
	proj->vector.y/=1.005;
	switch(lev_level.solid[proj->x][proj->y]) {
	  case TER_WATERFD:
	    proj->vector.y-=3;
	    proj->vector.x*=0.93;
	    break;
	  case TER_WATERFR:
	    proj->vector.x-=3;
	    proj->vector.y*=0.93;
	    break;
	  case TER_WATERFL:
	    proj->vector.x+=3;
	    proj->vector.y*=0.93;
	    break;
	  case TER_WATERFU:
	    proj->vector.y+=3;
	    proj->vector.x*=0.93;
	  case TER_WATER:
	    if(proj->vector.x>PROJ_UW_MAXSPEED) proj->vector.x=PROJ_UW_MAXSPEED;
            else if(proj->vector.x<-PROJ_UW_MAXSPEED) proj->vector.x=-PROJ_UW_MAXSPEED;
            if(proj->vector.y>PROJ_UW_MAXSPEED) proj->vector.y=PROJ_UW_MAXSPEED;
            else if(proj->vector.y<-PROJ_UW_MAXSPEED) proj->vector.y=-PROJ_UW_MAXSPEED;
	    break;
	  default:
	    if(proj->vector.x>proj->maxspeed) proj->vector.x=proj->maxspeed;
            else if(proj->vector.x<-proj->maxspeed) proj->vector.x=-proj->maxspeed;
            if(proj->vector.x>proj->maxspeed) proj->vector.y=proj->maxspeed;
            else if(proj->vector.y<-proj->maxspeed) proj->vector.y=-proj->maxspeed;
	  break;
	}
      }
      if(proj->life>0) proj->life--;
      if(proj->primed>0) proj->primed--;
      /* Draw the projectile */
      draw_projectile(proj);
      /* Remove old projectiles */
      if(plist) {
        if(proj->life==0)
	  plist=remove_projectile(plist);
	else plist=plist->next;
      }
    }
  }
  if(elist) {	/* Explosions */
    while(elist) {
      expl=elist->explosion;
      enext=elist->next;
      expl->frame++;
      if(expl->frame==EXPLOSION_CLUSTER_SPEED && expl->cluster!=Noproj) spawn_clusters(expl->x,expl->y,expl->cluster==Grenade?3:6,-1,expl->cluster);
      if(expl->frame==EXPL_FRAMES) { /* Animation is over */
        free(expl);
        if(elist->prev)
          elist->prev->next = elist->next;
        else
          explosions=elist->next;
        if(elist->next)
          elist->next->prev = elist->prev;
        free(elist);
      }
      elist=enext;
    }
  }
  draw_explosions();
}

static void draw_special_weapons(int x,int y,Projectile *proj) {
switch(proj->type) {
  case MegaBomb:
    putpixel(screen,x,y-1,proj->color);
    putpixel(screen,x,y+1,proj->color);
    putpixel(screen,x-1,y-1,proj->color);
    putpixel(screen,x+1,y-1,proj->color);
    break;
  case DividingMine:
  case Mine:
  case MagMine:
  case Acid:
    if(proj->color!=0) {
    putpixel(screen,x,y-1,proj->color);
    putpixel(screen,x,y+1,proj->color);
    putpixel(screen,x-1,y,proj->color);
    putpixel(screen,x+1,y,proj->color);
    }
    break;
  case Ember:
    if(proj->var1>1) {
      putpixel(screen,x,y-2,proj->color);
      putpixel(screen,x,y+2,proj->color);
      putpixel(screen,x-2,y,proj->color);
      putpixel(screen,x+2,y,proj->color);
    }
    if(proj->var1>0) {
      putpixel(screen,x,y-1,proj->color);
      putpixel(screen,x,y+1,proj->color);
      putpixel(screen,x-1,y,proj->color);
      putpixel(screen,x+1,y,proj->color);
    }
    break;
  case GravityWell: {
    double d;
    int r,r2;
    if(proj->var1<=0) proj->var1=8; else proj->var1--;
    for(r=0;r<12;r++) {
      d=-M_PI_4*r+proj->var1/4.0;
      for(r2=4;r2<=16;r2+=4)
        putpixel(screen,x+sin(d-r2/6.0)*r2,y+cos(d-r2/6.0)*r2,proj->color);
    }
    }
    break;
  case Spear:
  case Missile:
  case Rocket:
  case Mirv:
    {
    int dx,dy;
    dx=sin(proj->angle)*3;
    dy=cos(proj->angle)*3;
    draw_line(screen,x-dx,y-dy,x+dx,y+dy,proj->color);
    }
    break;
  case Zap:
    if(players[proj->owner].ship==NULL) break;
    {
    int targx,targy,dx,dy,seg;
    targx=x+(players[proj->owner].ship->x-proj->x);
    targy=y+(players[proj->owner].ship->y-proj->y);
    for(seg=0;seg<4;seg++) {
      dx=(targx-x+((rand()%20)-10))/3;
      dy=(targy-y+((rand()%20)-10))/3;
      if(x>0 && x<SCREEN_W && y>0 && y<SCREEN_H)
        if(x+dx>0 && x+dx<SCREEN_W && y+dy>0 && y+dy<SCREEN_H)
          draw_line(screen,x,y,x+dx,y+dy,col_yellow);
      x+=dx; y+=dy;
    }
    }
    break;
  case Boomerang:
    putpixel(screen,x-round(cos(proj->angle-0.5)),y-round(sin(proj->angle-0.5)),proj->color);
    putpixel(screen,x-round(cos(proj->angle+0.5)),y-round(sin(proj->angle+0.5)),proj->color);
    break;
  default: break;
}
}

static inline void draw_projectile(Projectile *projectile) {
  int x,y,p;
  /* Some projectiles are not drawn */
  if(projectile->type==FireStarter) return;
#ifndef HAVE_LIBSDL_GFX
  if( SDL_MUSTLOCK(screen) ) SDL_LockSurface(screen);
#endif
  /* Draw the projectile */
  for(p=0;p<4;p++) {
#ifdef DONT_UPDATE_DEAD
    if(players[p].active && players[p].pilot->dead==0) {
#else
    if(players[p].active) {
#endif
      x=projectile->x-cam_rects[p].x;
      y=projectile->y-cam_rects[p].y;
      if((x>0 && x<320) && (y>0 && y<240)) {
        putpixel(screen,x+lev_rects[p].x,y+lev_rects[p].y,projectile->color);
        draw_special_weapons(x+lev_rects[p].x,y+lev_rects[p].y,projectile);
      }
    }
  }
#ifndef HAVE_LIBSDL_GFX
  if (SDL_MUSTLOCK(screen)) SDL_UnlockSurface(screen);
#endif
}

static inline void draw_explosions(void) {
  struct Explosion_list *elist=explosions;
  SDL_Surface *tmpsurf;
  SDL_Rect rect;
  int p;
  while(elist) {
    for(p=0;p<4;p++) {
      tmpsurf=wea_explosion[elist->explosion->frame];
#ifdef DONT_UPDATE_DEAD
      if(players[p].active && players[p].pilot->dead==0) {
#else
      if(players[p].active) {
#endif
        rect.x=lev_rects[p].x+elist->explosion->x-cam_rects[p].x-8;
        rect.y=lev_rects[p].y+elist->explosion->y-cam_rects[p].y-8;
        if((rect.x>lev_rects[p].x && rect.x<lev_rects[p].x+lev_rects[p].w-tmpsurf->w)
	&& (rect.y>lev_rects[p].y && rect.y<lev_rects[p].y+lev_rects[p].h-tmpsurf->h))
          SDL_BlitSurface(tmpsurf,NULL,screen,&rect);
      }
    }
    elist=elist->next;
  }
}

void spawn_clusters(int x,int y,int count,int owner,ProjectileType type) {
  Projectile *cluster;
  double angle,incr,r;
  angle=0;
  incr=(2.0*M_PI)/(double)count;
  while(angle<2.0*M_PI) {
    r=(rand()%10)/10.0;
    cluster=make_projectile(x,y,makeVector(sin(angle+r)*0.8,cos(angle+r)*0.8));
    cluster->type=type;
    cluster->primed=1;
    if(type==Grenade) cluster->color=col_grenade;
    else if(type==Napalm) { cluster->life=30; cluster->color=col_white; }
    else if(type==FireStarter) {
      cluster->vector.y/=2.0;
      cluster->life=30;
    }
    cluster->gravity=&wea_gravity;
    cluster->owner=owner;
    add_projectile(cluster);
    angle+=incr;
  }
}

int find_nearest_player(int myX,int myY,int not_this,double *dist) {
  int p,plr=-1;
  double distance=9999999,d;
  for(p=0;p<4;p++) {
    if(not_this>=0)
      if(player_teams[p]==player_teams[not_this]) continue;
    if(players[p].ship==NULL) continue;
    if(!players[p].active || players[p].ship->dead) continue;
    d=hypot(abs(players[p].ship->x-myX),abs(players[p].ship->y-myY));
    if(d<distance) {plr=p; distance=d;}
  }
  if(dist) *dist=distance;
  return plr;
}

char detonate_remote(char plr) {
  struct Projectile_list *list=projectiles;
  float f;
  Projectile *proj;
  while(list) {
    if(list->projectile->type==Landmine && list->projectile->owner==plr) {
      spawn_clusters(list->projectile->x,list->projectile->y,5,-1,Cannon);
      for(f=list->projectile->angle-0.25;f<list->projectile->angle+0.25;f+=0.05) {
	      proj=make_projectile(list->projectile->x,list->projectile->y, makeVector( sin(f)*1.5,cos(f)*1.5 ) );
	      add_projectile(proj);
      }
      free(list->projectile);
      if(list->prev)
        list->prev->next = list->next;
      else
	projectiles=list->next;
      if(list->next)
        list->next->prev = list->prev;
      free(list);
      return 1;
    }
    list=list->next;
  }
  return 0;
}
