/*
 * Luola - 2D multiplayer cavern-flying game
 * Copyright (C) 2003 Calle Laakkonen
 *
 * File        : particle.c
 * Description : Particle engine
 * Author(s)   : Calle Laakkonen
 *
 * Luola is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * Luola is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include <stdlib.h>
#include <math.h>
#include <string.h>
#include "SDL.h"

#include "defines.h"
#include "console.h"
#include "level.h"
#include "player.h"
#include "particle.h"

/* Linked list containg all particles */
struct Particle_list {
  Particle *particle;
  struct Particle_list *next;
  struct Particle_list *prev;
};

/* Internally used globals */
static struct Particle_list *particles;
static struct Particle_list *last_particle;
#ifdef PARTICLE_CACHE
static struct Particle_list *particle_cache;
static int particle_cache_count;
#endif

/* Internally used function prototypes */
static inline void draw_particle(Particle *part);

/* Initialize */
void init_particles(void) {
  particles=NULL;
  last_particle=NULL;
#ifdef PARTICLE_CACHE
  particle_cache=NULL;
  particle_cache_count=0;
#endif
  things_loaded[TL_PARTICLES]=1;
}

void deinit_particles(void) {
  struct Particle_list *next;
  while(particles) {
    next=particles->next;
    free(particles->particle);
    free(particles);
    particles=next;
  }
#ifdef PARTICLE_CACHE
  while(particle_cache) {
    next=particle_cache->next;
    free(particle_cache->particle);
    free(particle_cache);
    particle_cache=next;
  }
  particle_cache_count=0;
#endif
  last_particle=NULL;
}

/* Utility function to create a new particle */
Particle *make_particle(int x,int y,int age) {
  Particle *newpart;
#ifdef PARTICLE_CACHE
  if(particle_cache_count) {
     struct Particle_list *next=particle_cache->next;
     newpart=particle_cache->particle;
     free(particle_cache);
     particle_cache=next;
     particle_cache_count--;
  } else
#endif
  newpart=(Particle*)malloc(sizeof(Particle));
  newpart->x=x;
  newpart->y=y;
  newpart->age=age;
  newpart->color[0]=200;
  newpart->color[1]=200;
  newpart->color[2]=255;
  newpart->color[3]=255;
  newpart->targ_color[0]=0;
  newpart->targ_color[1]=0;
  newpart->targ_color[2]=0;
  newpart->targ_color[3]=0;
  // We don't initialize the vector. This can be considered a 'feature' since now we get
  // random jiggling for free ;-) (looks good on missile trails)
  calc_color_deltas(newpart);
  return newpart;
}

/* Add a new particle to list*/
void addParticle(Particle *newpart) {
  struct Particle_list *newentry;
  newentry=(struct Particle_list*)malloc(sizeof(struct Particle_list));
  newentry->next=NULL;
  newentry->prev=last_particle;
  newentry->particle=newpart;
  if(last_particle==NULL)
    particles=newentry;
  else
    last_particle->next=newentry;
  last_particle=newentry;
}

void animate_particles(void) {
  struct Particle_list *list=particles,*next;
  Particle *part;
  if(list==NULL) return;
  while(list) {
    part=list->particle;
    next=list->next;
    part->x-=round(part->vector.x);
    part->y-=round(part->vector.y);
    part->color[0]+=part->rd;
    part->color[1]+=part->gd;
    part->color[2]+=part->bd;
    part->color[3]+=part->ad;
    part->age--;
    if(part->age<=0) { /* Particle has expired, delete it */
#ifdef PARTICLE_CACHE
      if(particle_cache_count<PARTICLE_CACHE) {
        if(list->prev)
          list->prev->next=list->next;
        else
          particles=list->next;
        next=list->next;
       if(next)
         list->next->prev=list->prev;
       else last_particle=list->prev;
       list->next=particle_cache;
       list->prev=NULL;
       particle_cache=list;
       particle_cache_count++;
      } else {
#endif
      free(part);
      if(list->prev)
        list->prev->next = list->next;
      else
        particles=list->next;
      if(list->next)
        list->next->prev = list->prev;
      else
        last_particle=list->prev;
      free(list);
#ifdef PARTICLE_CACHE
      }
#endif
    } else draw_particle(part);
    list=next;
  }
}

static inline void draw_particle(Particle *part) {
  Uint32 color;
  int x,y,p;
#ifndef HAVE_LIBSDL_GFX
  if( SDL_MUSTLOCK(screen) ) SDL_LockSurface(screen);
#endif

#if HAVE_LIBSDL_GFX
  color=(part->color[0]<<24)+(part->color[1]<<16)+(part->color[2]<<8)+part->color[3];
#else
  color=SDL_MapRGB(screen->format,part->color[0],part->color[1],part->color[2]);
#endif
  for(p=0;p<4;p++) {
#ifdef DONT_UPDATE_DEAD
    if(players[p].active && players[p].pilot->dead==0) {
#else
    if(players[p].active) {
#endif
      x=part->x-cam_rects[p].x;
      y=part->y-cam_rects[p].y;
      if((x>0 && x<lev_rects[p].w) && (y>0 && y<lev_rects[p].h))
        putpixel(screen,lev_rects[p].x+x,lev_rects[p].y+y,color);
    }
  }
#ifndef HAVE_LIBSDL_GFX
  if (SDL_MUSTLOCK(screen)) SDL_UnlockSurface(screen);
#endif
}

/* Calculate color delta values */
void calc_color_deltas(Particle *part) {
  part->rd=(part->targ_color[0]-part->color[0])/part->age;
  part->gd=(part->targ_color[1]-part->color[1])/part->age;
  part->bd=(part->targ_color[2]-part->color[2])/part->age;
  part->ad=(part->targ_color[3]-part->color[3])/part->age;
}

