#!/usr/bin/env python
# -*- coding: utf-8 -*-
#
#
# Copyright Nicolas Bertrand (nico@inattendu.org), 2009
#
# This file is part of Luciole.
#
#    Luciole is free software: you can redistribute it and/or modify
#    it under the terms of the GNU General Public License as published by
#    the Free Software Foundation, either version 3 of the License, or
#    (at your option) any later version.
#
#    Luciole is distributed in the hope that it will be useful,
#    but WITHOUT ANY WARRANTY; without even the implied warranty of
#    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#    GNU General Public License for more details.
#
#    You should have received a copy of the GNU General Public License
#    along with Luciole.  If not, see <http://www.gnu.org/licenses/>.
#
#
import pygst
pygst.require('0.10')
import gst
import gst.interfaces


class luciole_gstreamer_play(object) :
  """ implementation of image per image video playing with gstreamer .
       Description of pipeline : 
           _______________________________________________________________________________________________________
          |                                                                                                       | 
    -->-- | multifilesrc --> capsfilter(framerate, image ratio) --> decodebin --> ffmpegcolorspace -->xvimagesink |
          |_______________________________________________________________________________________________________|
  """

  _filter_caps_string = "image/jpeg, framerate=(fraction)%s/1 ,width=360,height=288 "
  def _get_nbImages(self): return self._nbImages
  def _set_nbImages(self, value):
    if isinstance(value,int) and self.MyImageSrc :
      self._nbImages = value   
      self.MyImageSrc.set_property('num-buffers',self._nbImages)
  nbImages = property(_get_nbImages, _set_nbImages, "Number of image to play")
 
  def _get_framerate(self): return self._framerate
  def _set_framerate(self, value):
    if isinstance(value,str) and self.ImageFilter :
      caps_string = self._filter_caps_string%value
      caps = gst.Caps(caps_string)
      self.ImageFilter.set_property("caps", caps)
      self._framerate = value   
  framerate = property(_get_framerate, _set_framerate, "Framerate for displaying image ")

  
  def __init__(self,videowidget,location,framerate="5",nbImages=-1,start_buf=0,on_eos_cb =None) :
    """ 
      init of gstreamer player 
      inputs :
        - videowidget : Drawing area widgert, in gstreamer style for playing video
        - location : where the image are stored 
        - framerate : the video framerate
        - nbImages  : Number of images to play
        - start_buf : index of the first image to play  
		- on_eos_cb : callback for EOS signal
    """
    
    self._videowidget = videowidget
    self._location =  location
    self._nbImages = nbImages
    self._framerate = framerate
    self._start_buf = start_buf

    self.playing = False    # playing is not started at init
    self.on_eos = on_eos_cb
    #self.on_eos = False
    # 
    # Creation of gstreamer pipeline
    #
    ElementList=[]
    self.pipe = gst.Pipeline()

    # image load
    self.MyImageSrc = gst.element_factory_make('multifilesrc')
    self.MyImageSrc.set_property('location',self._location)
    self.MyImageSrc.set_property('num-buffers',self._nbImages)
    self.MyImageSrc.set_property('index',self._start_buf)
    ElementList.append(self.MyImageSrc)

    # filter
    self.ImageFilter = gst.element_factory_make("capsfilter")
    caps_string = self._filter_caps_string%self._framerate
    caps = gst.Caps(caps_string)
    self.ImageFilter.set_property("caps", caps)
    ElementList.append(self.ImageFilter)

    # decodebin
    self.dec = gst.element_factory_make('decodebin')
    # Connect handler for 'new-decoded-pad' signal
    self.dec.connect('new-decoded-pad', self.on_new_decoded_pad)
    ElementList.append(self.dec)

    # ffmpegclorspace
    self.Myffmpeg = gst.element_factory_make('ffmpegcolorspace')
    # Reference used in self.on_new_decoded_pad()
    self.vpad = self.Myffmpeg.get_pad('sink')
    ElementList.append(self.Myffmpeg)

    # Image sink
    MyImageSink = gst.element_factory_make('xvimagesink')
    ElementList.append(MyImageSink)
    #
    # Add elements to pipeline
    #
    for elem in ElementList : self.pipe.add(elem)

    # link all elements
    gst.element_link_many(self.MyImageSrc,self.ImageFilter)
    self.ImageFilter.link(self.dec)
    gst.element_link_many(self.Myffmpeg,MyImageSink)
   
    # connect bus
    bus = self.pipe.get_bus()
    bus.enable_sync_message_emission()
    bus.add_signal_watch()
    bus.connect('sync-message::element', self.on_sync_message)
    bus.connect('message', self.on_message)

     
  def on_new_decoded_pad(self, element, pad, last):
    """ call back to connect decodebin to videopad """
    caps = pad.get_caps()
    name = caps[0].get_name()
    if  name == 'video/x-raw-yuv':
      if not self.vpad.is_linked(): # Only link once
        pad.link(self.vpad)
    
  def on_sync_message(self, bus, message):
    """ callback on video display """
    if message.structure is None:
      return
    if message.structure.get_name() == 'prepare-xwindow-id':
      self._videowidget.set_sink(message.src)
      message.src.set_property('force-aspect-ratio', True)
            
  def on_message(self, bus, message):
    """ call back on message sent by gstreamer """ 
    t = message.type
    if t == gst.MESSAGE_ERROR:
      err, debug = message.parse_error()
      print "Error: %s" % err, debug
      if self.on_eos:
        self.on_eos()
        self.playing = False
    elif t == gst.MESSAGE_EOS:
      if self.on_eos:
        self.on_eos()
      self.playing = False
  
  def play(self):
    """ Play the video """
    gst.info("playing player")
    self.pipe.set_state(gst.STATE_PLAYING)
    self.playing = True
        
  def stop(self):
    """ Stop the video """
    self.pipe.set_state(gst.STATE_NULL)
    gst.info("stopped player")
    # When video is stopped , set the video to start at the first image
    self.MyImageSrc.set_property('index',0)
    self.playing = False
  
  def get_state(self, timeout=1):
    """ Gstreamer state"""
    return self.pipe.get_state(timeout=timeout)
    
  def pause(self):
    """ Pause the video """
    gst.info("pausing player")
    self.pipe.set_state(gst.STATE_PAUSED)
    self.playing = False

  def is_playing(self):
    """ return playing state """
    return self.playing
 

