'''
Defines a L{Task} to execute when the selector changes.

@author: Pete Brunet
@organization: IBM Corporation
@copyright: Copyright (c) 2005, 2006 IBM Corporation
@license: Common Public License 1.0

All rights reserved. This program and the accompanying materials are made 
available under the terms of the Common Public License v1.0 which accompanies
this distribution, and is available at
U{http://www.opensource.org/licenses/cpl1.0.php}
'''

import Base, Constants
import AEEvent

class SelectorTask(Base.Task):
  '''
  Executed when a selector change occurs. 

  This class registers its name and whether it should be monitored by default in
  an L{AEMonitor.EventMonitor} using the L{Constants.registerTaskType} function
  when this module is first imported. The L{AEMonitor.EventMonitor} can use this
  information to build its menus.
  
  All class variables are replicated from L{AEEvent.ScreenChange} for 
  convience to advanced L{Perk}s.
  '''
  ACTIVE_ITEM_SELECT = AEEvent.SelectorChange.ACTIVE_ITEM_SELECT
  ADD_ITEM_SELECT = AEEvent.SelectorChange.ADD_ITEM_SELECT
  REMOVE_ITEM_SELECT = AEEvent.SelectorChange.ADD_ITEM_SELECT
  CHANGE_TEXT_SELECT = AEEvent.SelectorChange.CHANGE_TEXT_SELECT
  
  Constants.registerTaskType('SelectorTask', True)

  def getType(self):
    '''
    @return: Type of L{AEEvent} this L{Task} wants to handle
    @rtype: class
    '''
    return AEEvent.SelectorChange
  
  def update(self, por, text, kind, layer, **kwargs):
    '''
    Update this L{Task} in response to a consumed selector change event. Called 
    by L{Tier.Tier._executeTask}.
    
    @param text: The accessible text or name of the item at the POR
    @type text: string
    @param por: Point of regard for the related accessible
    @type por: L{POR}
    @param kind: Indicates the kind of selection event, one of 
      L{ACTIVE_ITEM_SELECT}, L{ADD_ITEM_SELECT}, L{REMOVE_ITEM_SELECT}, or 
      L{CHANGE_TEXT_SELECT}
    @type kind: integer
    @param layer: Layer on which the event occurred, one of L{Task.FOCUS_LAYER},
      L{Task.TIER_LAYER}, or L{Task.BACKGROUND_LAYER}
    @type layer: integer
    '''
    pass
  
  def execute(self, por, text, layer, kind, **kwargs):
    '''
    Executes this L{Task} in response to a selector change event. Called by 
    L{Tier.Tier._executeTask}.
    
    @param text: The accessible text or name of the item at the POR
    @type text: string
    @param por: Point of regard for the related accessible
    @type por: L{POR}
    @param layer: Layer on which the event occurred, one of L{Task.FOCUS_LAYER},
      L{Task.TIER_LAYER}, or L{Task.BACKGROUND_LAYER}
    @type layer: integer
    @param kind: Indicates the kind of selection event, one of 
      L{ACTIVE_ITEM_SELECT}, L{ADD_ITEM_SELECT}, L{REMOVE_ITEM_SELECT}, or 
      L{CHANGE_TEXT_SELECT}
    @type kind: integer
    @return: Should processing continue? Always returns True by default.
    @rtype: boolean
    '''
    if kind == self.ACTIVE_ITEM_SELECT:
      return self.executeActive(por=por, text=text, layer=layer, **kwargs)
    elif kind == self.ADD_ITEM_SELECT:
      return self.executeAdded(por=por, text=text, layer=layer, **kwargs)
    elif kind == self.REMOVE_ITEM_SELECT:
      return self.executeRemoved(por=por, text=text, layer=layer, **kwargs)
    elif kind == self.CHANGE_TEXT_SELECT:
      return self.executeText(por=por, text=text, layer=layer, **kwargs)
  
  def executeActive(self, por, text, layer, **kwargs):
    '''
    Executes this L{Task} in response to an active selection change event.
    Called by L{Tier.Tier._executeTask}.
    
    @param text: The accessible text or name of the item at the POR
    @type text: string
    @param por: Point of regard for the related accessible
    @type por: L{POR}
    @param layer: Layer on which the event occurred, one of L{Task.FOCUS_LAYER},
      L{Task.TIER_LAYER}, or L{Task.BACKGROUND_LAYER}
    @type layer: integer
    @return: Should processing continue? Always returns True by default.
    @rtype: boolean
    '''
    pass
  
  def executeText(self, por, text, layer, **kwargs):
    '''
    Executes this L{Task} in response to an addition to the current selection.
    Called by L{Tier.Tier._executeTask}.
    
    @param text: The accessible text or name of the item at the POR
    @type text: string
    @param por: Point of regard for the related accessible
    @type por: L{POR}
    @param layer: Layer on which the event occurred, one of L{Task.FOCUS_LAYER},
      L{Task.TIER_LAYER}, or L{Task.BACKGROUND_LAYER}
    @type layer: integer
    @return: Should processing continue? Always returns True by default.
    @rtype: boolean
    '''
    pass
  
  def executeAdded(self, por, text, layer, **kwargs):
    '''
    Executes this L{Task} in response to a removal from the current selection.
    Called by L{Tier.Tier._executeTask}.
    
    @param text: The accessible text or name of the item at the POR
    @type text: string
    @param por: Point of regard for the related accessible
    @type por: L{POR}
    @param layer: Layer on which the event occurred, one of L{Task.FOCUS_LAYER},
      L{Task.TIER_LAYER}, or L{Task.BACKGROUND_LAYER}
    @type layer: integer
    @return: Should processing continue? Always returns True by default.
    @rtype: boolean
    '''
    pass
  
  def executeRemoved(self, por, text, layer, **kwargs):
    '''
    Executes this L{Task} in response to a removal from the current selection.
    Called by L{Tier.Tier._executeTask}.
    
    @param text: The accessible text or name of the item at the POR
    @type text: string
    @param por: Point of regard for the related accessible
    @type por: L{POR}
    @param layer: Layer on which the event occurred, one of L{Task.FOCUS_LAYER},
      L{Task.TIER_LAYER}, or L{Task.BACKGROUND_LAYER}
    @type layer: integer
    @return: Should processing continue? Always returns True by default.
    @rtype: boolean
    '''
    pass
 