'''
Defines a device that uses the IBM TTS synthesizer via the pyibmtts package.
pyibmtts can be obtained from U{http://ibmtts-sdk.sf.net}.

@var IBMSpeechStyle: Singleton default speech style for all IBM-TTS flyweights
@type IBMSpeechStyle: L{AEOutput.AudioStyleDefault}

@author: Larry Weiss
@author: Peter Parente
@author: Brett Clippingdale
@organization: IBM Corporation
@copyright: Copyright (c) 2006 IBM Corporation
@license: Common Public License 1.0

All rights reserved. This program and the accompanying materials are made
available under the terms of the Common Public License v1.0 which accompanies
this distribution, and is available at
U{http://www.opensource.org/licenses/cpl1.0.php}
'''
import AEOutput
import pyibmtts
from i18n import _

class IBMSpeechStyle(AEOutput.AudioStyleDefault):
  '''
  Overrides the base L{AEOutput.Style.AudioStyleDefault} class, filling in 
  fields for supported style properties with their appropriate values. After
  this class is defined, it is immediately turned into a singleton so it may
  be used ubiquitously as the default style for this device.
  '''
  MaxVoice = 8
  MinVoice = 1
  MaxVolume = 65535
  MinVolume = 0
  Volume = 26987
  MinPitch = 40
  MaxPitch = 442
  MaxRate = 1297
  MinRate = 176
  Rate = 250
IBMSpeechStyle = IBMSpeechStyle()  #instantiate a singleton overriding the class

class IBMSpeech(AEOutput.Audio):
  '''
  Defines a class to send output from LSR to IBM TTS using the pyibmtts wrapper
  provided with ibmtts-sdk U{http://ibmtts-sdk.sf.net}.

  To be compliant with LSR requirements, this class implements the interface
  defined in the L{AEOutput.Audio} class.
  '''    
  USE_THREAD = False
  
  def _localize(self, text):
    '''
    Converts the given unicode text to latin-1, the encoding expected by the
    IBM TTS runtime.

    @todo: PP: Temporary kludge to convert all text to Latin-1 as IBM TTS is
      expecting that encoding instead of UTF-8. Should look at the style 
      language attribute in the future to do the correct encoding.
    @param text: Text to localize
    @type text: unicode
    @return: Localized string
    @rtype: string
    '''
    try:
      # kludge!
      return text.encode('latin-1', 'replace')
    except (TypeError, UnicodeDecodeError), e:
      return text

  def init(self):
    '''
    Initializes the IBM TTSspeech driver through gnome-speech.
    
    @raise AEOutput.InitError: When the device can not be initialized
    '''
    try:
      self.tts = pyibmtts.Engine()
    except pyibmtts.ECIException:
      raise AEOutput.InitError
    # use real units and start at some speedy rate
    self.tts.realWorldUnits = True
      
  def createDistinctStyles(self, num_styles):
    '''
    Creates distinct output styles differentiated by voice and  pitch up to 
    the maximum voices given by the default style for this device. Styles are
    repeated beyond that threshold.
      
    @param num_styles: Number of styles that the requestor would like the
      device to initialize
    @type num_styles: integer
    @return: Styles
    @rtype: list of L{AEOutput.Style}
    ''' 
    styles = []
    for i in xrange(1, num_styles+1):
      s = AEOutput.Style(IBMSpeechStyle)
      v_num = (i) % s.MaxVoice
      v = self.tts.getVoice(v_num)
      s.Voice = v_num
      s.Pitch = v.pitchBaseline
      styles.append(s)
    return styles

  def close(self):
    '''
    Closes the speech device.
    '''
    del self.tts

  def sendStop(self, style=None):
    '''
    Stops speech immediately.
    
    @param style: Ignored
    @type style: L{AEOutput.Style}
    '''
    try:
      self.tts.stop()
    except pyibmtts.ECIException:
      pass

  def sendTalk(self, style=None):
    '''
    Begins synthesis of the buffered text.
    
    @param style: Ignored
    @type style: L{AEOutput.Style}
    '''
    try:
      self.tts.synthesize()
    except pyibmtts.ECIException:
      pass
    
  def sendString(self, text, style):
    '''
    Adds the given string to the text to be synthesized.

    @param text: String to buffer on the device
    @type text: string
    @param style: Style with which this string should be output; None means no
      style change should be applied
    @type style: integer
    '''
    if style is not None:
      self._applyStyle(style)
    try:
      self.tts.addText(self._localize(text))
    except pyibmtts.ECIException:
      pass

  def isActive(self):
    '''
    Indicates whether the device is active meaning it is busy doing output or
    has text waiting to be output.

    @return: True when the speech device is synthesizing, False otherwise
    @rtype: boolean
    '''
    try:
      return self.tts.speaking()
    except pyibmtts.Exception:
      return False

  def getName(self):
    '''
    Gives the user displayable (localized) name for this output device.
    Relevant version and device status should be included.

    @return: Localized name for the device
    @rtype: string
    '''
    return 'IBM-TTS %s' % pyibmtts.getVersion()
  
  def sendIndex(self, marker, style):
    '''
    Sends an index marker to the device driver. The driver should notify the
    device when the marker has been reached during output.
    
    @todo: PP: implement when index working

    @param style: Style indicating channel in which the marker will be appended
    @type style: L{AEOutput.Style}
    @param marker: The value that should be returned when output has been 
      processed up to this point
    @type marker: integer
    '''
    raise NotImplementedError

  def _applyStyle(self, style):
    '''
    Applies a given style to this output device. All output following this 
    call will be presented in the given style until this method is called again
    with a different style.
    
    @param style: Style to apply to future output
    @type style: L{AEOutput.Style}
    '''
    try:
      v = self.tts.getVoice(style.Voice)
      v = self.tts.setActiveVoice(v)
    except pyibmtts.ECIException:
      # just quit if we can't even get the voice to which style params should be
      # applied
      return
    try:
      # restore voice independent parameters
      v.speed = style.Rate
    except pyibmtts.ECIException:
      pass
    try:
      v.volume = style.Volume
    except pyibmtts.ECIException:
      pass
    try:
      v.pitchBaseline = style.Pitch
    except pyibmtts.ECIException:
      pass    
    
  def getDefaultStyle(self):
    '''
    Gets the singleton default style for this device.
    
    @return: Default style of the device
    @rtype: L{AEOutput.AudioStyleDefault}
    '''
    return IBMSpeechStyle
