#!/usr/bin/perl -w

#$Header: /home2/cvsroot/LogTrend/Visu/Zone/AverageGraph.pm,v 1.18.2.3 2002/02/13 14:06:26 lsimonneau Exp $
##******************************************************************************
## Class ReportGraph  isa Zone
##  Description  : class from which inherit Report(Bar|Line)Graph
##  Project      : LogTrend 1.0.0.0 - Atrid Systemes
##  Author       : Laurent Simonneau (l.simonneau@atrid.fr)
##******************************************************************************
#$Log: AverageGraph.pm,v $
#Revision 1.18.2.3  2002/02/13 14:06:26  lsimonneau
#Bug fixed
#
#Revision 1.18.2.2  2002/02/13 13:52:51  lsimonneau
#Bug fixes
#
#Revision 1.18.2.1  2002/02/13 13:21:38  lsimonneau
#Major bug fixes
#
#Revision 1.18  2001/11/19 12:55:50  slhullier
#Accent bug fixed at all ...
#
#Revision 1.17  2001/11/19 08:23:18  lsimonneau
#Minor bugfixes.
#
#Revision 1.16  2001/11/16 16:45:01  slhullier
#Text color format
#
#Revision 1.15  2001/11/15 16:08:57  slhullier
#UserNode managing first version (still bugs)
#
#Revision 1.14  2001/11/14 15:59:45  slhullier
#No more global value for SystemsDeclaration
#
#Revision 1.13  2001/11/12 15:32:29  slhullier
#Relative times bug fixed
#
#Revision 1.12  2001/11/12 10:03:00  slhullier
#no more tab
#
#Revision 1.11  2001/11/09 13:42:42  slhullier
#Duration/Interval/StartStop modification
#
#Revision 1.10  2001/11/06 17:47:05  lsimonneau
#Move Duration and Start/Stop to a Time tag.
#No backward compatibility.
#
#Revision 1.9  2001/10/31 17:07:11  lsimonneau
#Remove backward compatibility for Type attribute.
#
#Revision 1.8  2001/10/30 15:02:15  lsimonneau
#Change period default value.
#
#Revision 1.7  2001/10/29 15:45:19  lsimonneau
#Add absolute date support for graph andReport/AlarmLog
#
#Revision 1.6  2001/10/26 13:16:12  lsimonneau
#Add "last month" and "current month" to Duration possible values.
#Type is now deprecated for AlarmLog and AverageGraph.
#
#Revision 1.5  2001/10/26 08:39:45  lsimonneau
#*** empty log message ***
#
#Revision 1.4  2001/10/26 08:16:39  lsimonneau
#Major bugfixes.
#
#Revision 1.3  2001/10/25 15:07:21  lsimonneau
#ReportAverageGrah improvement.
#
#Revision 1.2  2001/10/25 13:40:59  lsimonneau
#Average Graph improvement.
#
#Revision 1.1  2001/10/23 12:18:37  lsimonneau
#Now, LogTrend::Visu::Zone contains generic classes for Report/Web Zone.
#

package LogTrend::Visu::Zone::AverageGraph;

use strict;

use POSIX qw(strftime tmpnam);

use Time::Local;
use LogTrend::Common::Duration;
use LogTrend::Visu::Utils;
use LogTrend::Visu::TimeUtils;
use LogTrend::Visu::Zone;
use LogTrend::Visu::Request;
use LogTrend::Visu::Response;
use LogTrend::Visu::Constants;


use vars qw( @ISA );
@ISA = ("LogTrend::Visu::Zone::Graph");

##******************************************************************************
## Constructor  public
##  Description  : creat a new Graph object
##  Parameters   : the Type XML node, the Zone XML node
##******************************************************************************
sub new
{
   my ($classname,$rootnode,$realrootnode,$systemsDeclaration) = @_;
   my $self = $classname->SUPER::new($rootnode,$realrootnode,$systemsDeclaration);
   bless($self, $classname);

   return $self;
}



##******************************************************************************
## parseXML  protected (>Graph)
##  Description  : parse bar graph specific xml entry
##  Parameters   :
##******************************************************************************
sub parseXML {
   my ($self, $rootnode, $realrootnode) = @_;

   my ($list, $node, $attrnode, $attributes);

   ##===========================================================================
   ## AverageGraph tag
   ##===========================================================================
   $self->{PIXELHEIGHT} = 210;
   $attributes = $rootnode->getAttributes() || die("Error in \"AverageGraph\".");
   $attrnode = $attributes->getNamedItem("PixelHeight");
   if( defined( $attrnode ) )
   {  $self->{PIXELHEIGHT} = AccentsXML2Text( $attrnode->getValue() ); }

   $self->{LABELSPERLINE} = 3;

   ##===========================================================================
   ## Attributes Duration or Start/Stop
   ##===========================================================================

   $list = $rootnode->getElementsByTagName("Time") || die("Error in \"AverageGraph\".");
   die("Error in \"AverageGraph\".") if (@$list == 0);
   $node = $list->[0];
   $attributes = $node->getAttributes() || die("Error in \"AverageGraph\".");

   ##---------------------------------------------------------------------------
   if( defined( $attrnode = $attributes->getNamedItem("Duration") ) )
   {
      $self->{ORIGINAL_START} = Duration( $attrnode->getValue() );
      $self->{ORIGINAL_STOP}  = 0;
      $self->{USE_ABSOLUTE_DATE} = 0;
   }
   ##---------------------------------------------------------------------------
   elsif( defined( $attrnode = $attributes->getNamedItem("Interval") ) )
   {
      ($self->{ORIGINAL_START},$self->{ORIGINAL_STOP}) =
         TimeInterval( $attrnode->getValue() );
      $self->{USE_ABSOLUTE_DATE} = 1;
   }
   ##---------------------------------------------------------------------------
   else
   {
      $attrnode = $attributes->getNamedItem("Start") or
         die("No 'Start' field in \"X\" tag.");
      my $start_val = AccentsXML2Text( $attrnode->getValue() );

      $attrnode = $attributes->getNamedItem("Stop") or
         die("No 'Stop' field in \"X\" tag.");
      my $stop_val = AccentsXML2Text( $attrnode->getValue() );

      ($self->{ORIGINAL_START},$self->{ORIGINAL_STOP},$self->{USE_ABSOLUTE_DATE}) =
         StartStop( $start_val, $stop_val );
   }

   ##===========================================================================
   $attrnode = $attributes->getNamedItem("Period");
   if( defined( $attrnode ) ) {
       $self->{AVERAGE_PERIOD} = Duration($attrnode->getValue);
   }
   else {
       $self->{AVERAGE_PERIOD} = 600;
   }

   if($self->{USE_ABSOLUTE_DATE}) {
       $self->{FORCE_PRINTED_TIME_MIN} = $self->{ORIGINAL_START};
       $self->{FORCE_PRINTED_TIME_MAX} = $self->{ORIGINAL_STOP};
   }
   else {
       my $now = time;
       $self->{FORCE_PRINTED_TIME_MIN} = $now - $self->{ORIGINAL_START};
       $self->{FORCE_PRINTED_TIME_MAX} = $now - $self->{ORIGINAL_STOP};
   }

   $attrnode = $attributes->getNamedItem("IncludeWeekEnd");
   if( defined( $attrnode ) )
   {
       $self->{INCLUDEWEEKEND} = AccentsXML2Text( $attrnode->getValue() );
   }
   else {
       $self->{INCLUDEWEEKEND} = 0;
   }

   ##===========================================================================
   ## Colors tag
   ##===========================================================================
   $list = $rootnode->getElementsByTagName("Colors");
   $node = $list->item(0);
   my @pictureBGColor = RGBtoInt( $self->{SYSTEMSDECLARATION}->getColor('BackGround') );
   my @graphBGColor   = (208,208,208);
   my @gridColor      = (128,128,128);
   my @textColor      = RGBtoInt( $self->{SYSTEMSDECLARATION}->getColor('Text') );

   if( defined($node) )
   {
      $attributes = $node->getAttributes() || die("Error in \"Colors\".");

      ##------------------------------------------------------------------------
      $attrnode = $attributes->getNamedItem("PictureBG");
      if( defined( $attrnode ) )
      { @pictureBGColor = RGBtoInt( TextToRGB( $attrnode->getValue() ) ); }

      ##------------------------------------------------------------------------
      $attrnode = $attributes->getNamedItem("GraphBG");
      if( defined( $attrnode ) )
      { @graphBGColor = RGBtoInt( TextToRGB( $attrnode->getValue() ) ); }

      ##------------------------------------------------------------------------
      $attrnode = $attributes->getNamedItem("Grid");
      if( defined( $attrnode ) )
      { @gridColor = RGBtoInt( TextToRGB( $attrnode->getValue() ) ); }

      ##------------------------------------------------------------------------
      $attrnode = $attributes->getNamedItem("Text");
      if( defined( $attrnode ) )
      { @textColor = RGBtoInt( TextToRGB( $attrnode->getValue() ) ); }

   }



   $self->{PICTUREBGCOLOR} = \@pictureBGColor;
   $self->{GRAPHBGCOLOR}   = \@graphBGColor;
   $self->{GRIDCOLOR}      = \@gridColor;
   $self->{TEXTCOLOR}      = \@textColor;

   ##===========================================================================
   ## Colors3 tag
   ##===========================================================================
   $list = $rootnode->getElementsByTagName("Colors3");
   $node = $list->item(0);
   my $okColor  = "#00C000";
   my $criticalColor = "#FF0000";
   if( defined($node) )
   {
      $attributes = $node->getAttributes() || die("Error in \"Colors3\".");

      ##------------------------------------------------------------------------
      $attrnode = $attributes->getNamedItem("Ok");
      if( defined( $attrnode ) )
      { $okColor = AccentsXML2Text( $attrnode->getValue() ); }

      ##------------------------------------------------------------------------
      $attrnode = $attributes->getNamedItem("Critical");
      if( defined( $attrnode ) )
      { $criticalColor = AccentsXML2Text( $attrnode->getValue() ); }

   }
   $self-> {OKCOLOR} =  $okColor;
   $self->{CRITICALCOLOR} = $criticalColor;
   $self->{X_TOOBIGGAP} = Duration("1h");
}


##******************************************************************************
## Method evalExpression protected
##  Description  : eval expression and return data
##  Parameters   : a Request
##  Return value : a Response
##******************************************************************************
sub evalExpression
{
    my ($self) = @_;

    my $expressions = $self->{EXPRESSIONS};

    if(@$expressions != 1) {
        die "Error : AverageGraph could not display more than one expression.";
    }

    my $expr = $expressions->[0];
    my $varName = $expr->label();

    ## Get data
    my $data;
    if( $self->{USE_ABSOLUTE_DATE} == 0 )
    {
       $data = $expr->evalInTimeInterval( 100000,
                                          $self->{ORIGINAL_START}, $self->{ORIGINAL_STOP} );
    }
    else
    {
       $data = $expr->evalInAbsoluteTimeInterval( 100000,
                                                  $self->{ORIGINAL_START}, $self->{ORIGINAL_STOP} );
    }


    if($data == 0) {
        my %res = ( "Max values", 0, "Average", 0, "Min values", 0);
        return \%res;
    }

    my $nbData = @$data;

    my $low_bound;
    if($self->{USE_ABSOLUTE_DATE}) {
        $low_bound  = timelocal(0, 0, 0, (localtime($self->{ORIGINAL_START}))[3, 4, 5]);
    }
    else {
        $low_bound  = timelocal(0, 0, 0, (localtime($data->[0]->[1]))[3, 4, 5]);
    }

    my $up_bound = $low_bound + $self->{AVERAGE_PERIOD} -1;
    my $wday;

    my %nbval_list;
    my %data_list;
    my %max_list;
    my %min_list;

    my $key;
    my $first_time = 1;
    for(my $i = 0; $i < $nbData; $i++) {
        my ($cur_val, $cur_date) = @{$data->[$i]};

        while(($cur_date > $up_bound) or $first_time) {
            if($first_time) {
                $first_time = 0;
            }
            else {
                $low_bound = $up_bound+1;
                $up_bound = $low_bound+$self->{AVERAGE_PERIOD} -1;
            }

            my ($up_sec, $up_min, $up_hour) = (localtime($up_bound))[0, 1, 2];
            my ($low_sec, $low_min, $low_hour, $low_mday,$low_mon,$low_year) = (localtime($low_bound))[0, 1, 2, 3, 4, 5];

            if($up_hour < $low_hour) {
                $up_bound = timelocal(59, 59, 23, $low_mday,$low_mon,$low_year);
                ($up_sec, $up_min, $up_hour) = (59, 59, 23);
            }

            $key = sprintf("%.2d:%.2d:%.2d , %.2d:%.2d:%.2d", $low_sec, $low_min, $low_hour,  $up_sec, $up_min, $up_hour );
        }

        if(! exists $nbval_list{$key}) {
            $max_list{$key} = $min_list{$key} = $cur_val;
            $data_list{$key} = $nbval_list{$key} = 0;
        }

        $wday = (localtime($cur_date))[6];
        next if (($wday == 0 or $wday == 6) and !$self->{INCLUDEWEEKEND});

        $data_list{$key} += $cur_val;
        $nbval_list{$key}++;

        $max_list{$key} = $cur_val if $cur_val > $max_list{$key};
        $min_list{$key} = $cur_val if $cur_val < $min_list{$key};
    }


    my %res;

    my @key_list = sort {timelocal(($a =~ /^(\d+):(\d+):(\d+)/), 1, 1, 1970) <=> timelocal(($b =~ /^(\d+):(\d+):(\d+)/), 1, 1, 1970)} (keys %data_list);

    foreach $key (@key_list) {
        my ($low_sec, $low_min, $low_hour, $up_sec, $up_min, $up_hour) = ($key =~ /(\d+):(\d+):(\d+) \, (\d+):(\d+):(\d+)/);

        my $date = ($low_sec + 60 * $low_min + 3600 * $low_hour + $up_sec + 60 * $up_min + 3600*$up_hour)/2;

        $date += timelocal(0, 0, 0, 1, 0, 70);

        if($nbval_list{$key}) {
            push @{$res{"Min values"}}, [$min_list{$key}, $date];
            push @{$res{"Average"}}, [$data_list{$key}/$nbval_list{$key}, $date];
            push @{$res{"Max values"}}, [$max_list{$key}, $date];
        }
    }

    return \%res;
}



##******************************************************************************
## Method expressionColors protected
##  Description  : Return a hash of expression colors
##  Parameters   : a Request
##  Return value : a Response
##******************************************************************************
sub expressionColors
{
    my ($self) = @_;

    my $expressions = $self->{EXPRESSIONS};
    my $varName = $expressions->[0]->label();

    my %colors = ("Min values", $self->{OKCOLOR}, "Average", $expressions->[0]->color(), "Max values", $self->{CRITICALCOLOR});


    return \%colors;
}

##******************************************************************************
1;
