#!/usr/bin/perl -w

#$Header: /home2/cvsroot/LogTrend/MailBridge/MailBridge.pl,v 1.32.2.2 2002/02/20 14:25:34 lsimonneau Exp $
##******************************************************************************
## MailBridge.pl
##  Description  : for reading mail from MailDataSender and re-sending
##                 data via TcpDataSender
##  Project      : LogTrend 1.0.0.0 - Atrid Systemes
##  Author       : Laurent Simonneau  l.simonneau@atrid.fr
##  Author       : Sylvain Lhullier s.lhullier@atrid.fr
##******************************************************************************
#$Log: MailBridge.pl,v $
#Revision 1.32.2.2  2002/02/20 14:25:34  lsimonneau
#*** empty log message ***
#
#Revision 1.34  2002/02/20 13:51:27  lsimonneau
#Change Reply to instead of Return-Path in agents and MailBridge.pl.
#Major bugfixes in StorageServer GPG Key rings management.
#
#Revision 1.33  2002/02/14 15:45:00  lsimonneau
#Minor features enhancement
#Add a Sender attribut in the Mail tag of agents config file.
#Use SMTP Server when available or /usr/lib/sendmail otherwise.
#Add the -s <smtp_server> option to MailBridge.pl.
#
#Revision 1.32  2002/02/06 13:06:39  lsimonneau
#Minor bugfixes.
#
#Revision 1.31  2002/01/16 15:13:03  slhullier
#sendmail -f$self->{ADMINMAIL}
#
#Revision 1.30  2002/01/08 12:42:56  lsimonneau
#Major bugfixes : Correct signature crash in daemon mode and use /usr/lib/sendmail -t -i
#to send mail insetead of MIME::Entity->smtpsend
#
#Revision 1.29  2001/12/10 17:37:28  lsimonneau
#Minor bugfixes.
#
#Revision 1.28  2001/11/20 14:48:47  lsimonneau
#Bugfixes.
#Send mail to the real dest. and not to me :)
#
#Revision 1.27  2001/11/16 10:29:20  lsimonneau
#Major feature enhancement : Implementation of authentification with RSA signatu
#re.
#
#IMPORTANT : This mailbridge is incompatible with older version.
#
#Revision 1.26  2001/10/22 15:32:25  lsimonneau
#Minor bugfixes.
#
#Revision 1.25  2001/10/22 12:52:12  fdubuy
#second arg missing in mkdirs
#
#Revision 1.24  2001/10/15 13:05:07  lsimonneau
#Minor bugfixes.
#
#Revision 1.23  2001/10/12 16:30:30  lsimonneau
#Major feature improvement in MailBridge.
#Now, queued message when execution fail, automaticaly reexecute these mail
#And send alarm mail to Agent or Server admin.
#
#Revision 1.22  2001/10/05 11:55:08  slhullier
#in the other case -> otherwise
#
#Revision 1.21  2001/08/07 13:33:04  slhullier
#
#Regular expressions update for ' *'
#
#Revision 1.20  2001/07/27 14:41:23  lsimonneau
#Include UpdateAgentDescription in AgentDescriptionToDB.
#
#Revision 1.19  2001/07/27 11:59:37  lsimonneau
#Modification pour assurer la continuit des donnes entre deux versions d' un agent.
#
#Revision 1.18  2001/07/10 12:55:17  slhullier
#*** empty log message ***
#
#Revision 1.17  2001/06/08 15:54:40  slhullier
#*** empty log message ***
#
#Revision 1.16  2001/06/08 15:51:03  slhullier
#
#Pour debugage
#
#Revision 1.15  2001/06/07 14:08:54  slhullier
#
#Passage de  unshift @INC,'..';  aux packages Logtrend::....
#
#Revision 1.14  2001/04/20 07:13:31  slhullier
#
#Premier essais (concluant) de stockage des messages sur disque
#
#Revision 1.13  2001/04/11 16:15:09  slhullier
#
#MailBridge stock les donnees en cas de non reponse du serveur
#Teste, operationnel sur serveur.
#
#Revision 1.12  2001/04/10 15:43:01  slhullier
#
#Repository pour MailBridge en cours
#
#Revision 1.11  2001/04/09 16:28:07  slhullier
#
#Debut de stockage en cas de pb avec mailbridge
#
#Revision 1.10  2001/03/26 09:00:10  slhullier
#
#MailBridge : XML et gzip opperationnel, mis en place spa/serveur
#
#Revision 1.9  2001/03/23 17:14:35  slhullier
#
#MailBridge: todo en XML, donnees gzip
#Pas fini : gunzip description
#
#Revision 1.8  2001/03/22 14:51:21  slhullier
#
#Commentaires
#
#Revision 1.7  2001/03/12 16:10:59  slhullier
#
#Mise en place
#
#Revision 1.6  2001/02/20 16:58:37  slhullier
#
#Faute
#
#Revision 1.5  2001/02/09 13:58:45  slhullier
#
#
#MailBridge Description Ok.
#Tests en situation reelle.
#
#Revision 1.4  2001/02/08 16:40:22  slhullier
#
#
#Efface les fichiers de /tmp/
#
#Revision 1.3  2001/02/08 16:22:26  slhullier
#
#
#MailBridge avec attachement : partie MailBridge fonctionne
#
#Revision 1.2  2001/02/08 15:23:56  slhullier
#
#
#Changement de repertoire pour MailBridge.pl (rep MailBridge)
#
#Revision 1.1  2001/02/07 14:36:33  slhullier
#
#
#Le systeme MailBridge fonctionne.
#Mise en service sur spa.
#

package LogTrend::MailBridge::MailBridge;

use strict;
use POSIX qw(tmpnam);
use MIME::Parser;
use Net::SMTP;
use XML::DOM;

use LogTrend::Common::LogDie;
use LogTrend::Agent::TcpDataSender;
use LogTrend::Agent::AgentDescriptionToDB::DirectDescriptionSender;

my $mb_directory = "/var/cache/LogTrend/logtrend-mailbridge";
my $mb_index     = "$mb_directory/index.xml";

my @FunctionList = ("SendDataAndAlarms", "AgentDescriptionToDB", "UpdateAgentDescription");

my $SMTPServer;

##******************************************************************************
## Method MakeMailBridgeDir
##******************************************************************************
sub MakeMailBridgeDir
{
    if( ! -e $mb_directory )
    {
        mkdir $mb_directory, 0700 or 
            Die("mkdir $mb_directory: $!");
    }
    elsif( ! ((-d $mb_directory)&&(-w $mb_directory)) )
    {
        Die("$mb_directory: not a writable directory");
    }

    if(! -e "$mb_directory/mail_queue") {
        open (F_QUEUE, ">$mb_directory/mail_queue")
            or Die("open $mb_directory/mail_queue: $!");
        
        print F_QUEUE "<?xml version=\"1.0\" standalone=\"no\"?>\n".
                      "<MailQueue>\n".
                      "</MailQueue>\n";
        close F_QUEUE;
    }
}


##******************************************************************************
## Method SendDataAndAlarms
##  Description  : generic function for sends data and alarms via TcpDataSender
##  Parameters   : host, port, login, password, filename for data and alarms
##  Return value : Like DataSender->SendDataAndAlarms
##******************************************************************************
sub SendDataAndAlarms
{
    my ($host, $port, $xmldatafile, $signaturefile) = @_;

    my $xmldataandalarms;
    open( XML, "gunzip -c < $xmldatafile |" ) or
        return "ERROR IO gunzip $xmldatafile : $!";
    while( <XML> ) { $xmldataandalarms .= $_; }
    close( XML );

    my $signature;
    open( SIGNATURE, "$signaturefile" ) or
        return "ERROR IO gunzip $signaturefile : $!";
    while( <SIGNATURE> ) { $signature .= $_; }
    close( SIGNATURE );

    my $tcpSender = LogTrend::Agent::TcpDataSender->new( $host, $port );
    my $res       =    $tcpSender->SendData( $xmldataandalarms, $signature );
    return $res;
}

##******************************************************************************
## Function SendDataAndAlarms_Mail
##  Description  : sends data and alarms via TcpDataSender
##                 called once by process, just for the mail
##  Parameters   :
##  Return value : Like DataSender->SendDataAndAlarms
##******************************************************************************
sub SendDataAndAlarms_Mail
{
    my $entity = shift;
    my ($host, $port);
    my ($name, $todo);

    ##===========================================================================
    ## First file : infos
    ##===========================================================================
    $name = $entity->parts( 0 )->bodyhandle->path;
    open(TODO, "<$name") or
        return "ERROR IO open $name : $!";
    $todo = <TODO>;
    close( TODO );

    ($host, $port) =
        ( $todo =~ /^<DataAndAlarms H="(.*)" P="(.*)" *\/>/ );
    if( !defined($port) )
    {
        return "ERROR XML Invalid todo.xml file.\n";              
    }

    ##===========================================================================
    ## Sending via TCP
    ##===========================================================================
    my $xmldatafile = $entity->parts( 1 )->bodyhandle->path;
    my $signaturefile = $entity->parts( 2 )->bodyhandle->path;
    return SendDataAndAlarms( $host, $port, $xmldatafile, $signaturefile );
}

##******************************************************************************
## Method AgentDescriptionToDB
##  Description  : generic function for declaring agent to DB
##  Parameters   : 
##  Return value : Like DataSender->SendDataAndAlarms
##******************************************************************************
sub AgentDescriptionToDB
{
    my ($filename, $signaturefile, $databasename, $host, $port, $username, $password, $ancestor_source, $ancestor_number) = @_;
    my $descriptionSender = 
      LogTrend::Agent::AgentDescriptionToDB::DirectDescriptionSender->new($databasename, $host, $port,
                                                                          $username, $password, 
                                                                          $ancestor_source, $ancestor_number );

    my $xmldata;
    open( XML, "gunzip -c < $filename |" ) or
        return "ERROR IO gunzip $filename : $!";
    while( <XML> ) { $xmldata .= $_; }
    close( XML );

    my $signature;
    open( SIGNATURE, "$signaturefile" ) 
        or return "ERROR IO gunzip $signaturefile : $!";
    while( <SIGNATURE> ) { $signature .= $_; }
    close( SIGNATURE );


    return $descriptionSender->SendDescription($xmldata, $signature);
}


##******************************************************************************
## Function AgentDescriptionToDB_Mail
##  Description  : declare agent to DB
##                 called once by process, just for the mail
##  Parameters   :
##  Return value : Like DataSender->SendDataAndAlarms
##******************************************************************************
sub AgentDescriptionToDB_Mail
{
    my $entity = shift;
    my ($databasename, $host, $port, $username, $password, $ancestor_source, $ancestor_number);
    my ($name, $l);

    ##===========================================================================
    ## First file : infos
    ##===========================================================================
    $name = $entity->parts( 0 )->bodyhandle->path;

    open(TODO, "<$name") or
        return "ERROR IO $name : $!";
    $l = <TODO>;
    close( TODO );

    ($databasename, $host, $port, $username, $password, $ancestor_source, $ancestor_number) =
        ( $l =~ /^<Description D="(.*)" H="(.*)" P="(.*)" U="(.*)" Pw="(.*)" S="(.*)" N="(.*)" *\/>/ ) ;

    if( !defined($password) )
    {
        ($databasename, $host, $port, $username, $password) = ( $l =~ /^<Description D="(.*)" H="(.*)" P="(.*)" U="(.*)" Pw="(.*)" *\/>/ );
    }
    
    if( !defined($password) ) {
        return "ERROR XML Invalid todo.xml file.\n";
    }

    ##===========================================================================
    ## Second file : description
    ##===========================================================================
    ##---------------------------------------------------------------------------
    ## Directory creation
    ##---------------------------------------------------------------------------
    my $filename = $entity->parts( 1 )->bodyhandle->path;
    my $signaturefile = $entity->parts( 2 )->bodyhandle->path;


    ##===========================================================================
    ## Sending via TCP
    ##===========================================================================
    my $res = AgentDescriptionToDB( $filename, $signaturefile, $databasename, $host, $port, $username, $password,
                                    $ancestor_source, $ancestor_number);
    
    return $res;
}



##******************************************************************************
## Function EmptyQueue
##  Description  : Try to empty the mail queue
##                 
##  Parameters   : none
##  Return value : 1 if successfull and 0 otherwise
##******************************************************************************
sub EmptyQueue {
    ### Create a Lock for mail queue
    open(F_QUEUE_LOCK, "+>$mb_directory/mail_queue_lock") 
        or Die("can't open the lock file $mb_directory/mail_queue_lock : $!");
    
    flock(F_QUEUE_LOCK, 2)
        or Die("can't flock $mb_directory/mail_queue_lock : $!");

    ### open the mail Queue index
    open(F_QUEUE, "$mb_directory/mail_queue") 
        or Die("can't open $mb_directory/mail_queue : $!");
    
    ## parse the queue
    my $parser = new XML::DOM::Parser;
    my $doc = $parser->parse(*F_QUEUE);
    
    my $nodes = $doc->getElementsByTagName ("Mail");
    
    my $return_value = 1;
    my $node_number = $nodes->getLength;
    my $i;
    for($i = 0; $i < $node_number; $i++) {
        my $mail = $nodes->item($i);
        ### Retrieve mail file name
        my $file_arg = $mail->getAttributeNode("File")
            or Die("LogTrend MailBridge mail queue is corrupted !!! ($mb_directory/mail_queue)");
        my $filename = $file_arg->getValue;

        ### Open the mailfile
        if(!open(F_MAILFILE, "$mb_directory/$filename")) {
            warn "Can't open mail file $mb_directory/$filename";
            next;
        }

        ### Setup MIME parser
        my $parser = new MIME::Parser;
        
        if(! -d "$mb_directory/logtrend-mailbridge-$$") {
            mkdir("$mb_directory/logtrend-mailbridge-$$",0700)
                or Die("Can't create directory $mb_directory/logtrend-mailbridge-$$ : $!");
        }
        
        $parser->output_dir("$mb_directory/logtrend-mailbridge-$$");
        
        ### Parse the mail
        my $entity = $parser->parse(\*F_MAILFILE);    
        
        if(!TreatMail($entity, 0)) {
            $parser->filer->purge;
            $return_value = 0;
            close F_MAILFILE;
            last;
        }
        
        $parser->filer->purge;
        
        ### The mail is successfully treated, remove it
        close F_MAILFILE;
        unlink "$mb_directory/$filename";
    }

    ### rebuild index with no treated entry
    my $new_index="";
    for(; $i < $node_number; $i++) {
        my $mail = $nodes->item($i);
        $new_index .= "\t".$mail->toString."\n";
    }

    ### close mail queue index and recreate it
    close F_QUEUE;
    
    open(F_QUEUE, "+>$mb_directory/mail_queue") 
        or return 1;

    print F_QUEUE "<?xml version=\"1.0\" standalone=\"no\"?>\n".
                  "<MailQueue>\n".
                   $new_index.
                  "</MailQueue>\n";

    close F_QUEUE;
  
    ### Unlock and remove queue lock
    unlink("$mb_directory/mail_queue_lock");
    flock(F_QUEUE_LOCK, 8);
    close F_QUEUE_LOCK;

    return $return_value;
}

##******************************************************************************
## Function SendMail
##  Description  : Send a mail
##                 
##  Parameters   : 
##  Return value : 1 if successfull and 0 otherwise
##******************************************************************************
sub SendMail {
    my ($sender, $dest, $message, $entity) = @_;

    my $mail;
    if(defined $entity) {
        $mail = new MIME::Entity->build( Type        => "multipart/mixed",
                                         From        => $sender,
                                         To          => $dest,
                                         Subject     => "LogTrend MailBridge error report");    
   
        $mail->attach("Type"      => "text/plain",
                      "Data"      => $message);

        $mail->attach("Type"      => "message/rfc822",
                      "Data"      => $entity->stringify);
    }
    else {
        $mail = new MIME::Entity->build( Type        => "text/plain",
                                         From        => $sender,
                                         To          => $dest,                                   
                                         Subject     => "LogTrend MailBridge error report",
                                         Data        => [$message]);
    }

    

    if($SMTPServer eq 'localhost') {
        open(MAILER, "| /usr/lib/sendmail -t  -i -f$sender");
        $mail->print(\*MAILER);
        close(MAILER);
    }
    else {
        my $smtp = Net::SMTP->new($SMTPServer);
        $smtp->mail($sender);
        $smtp->to($dest);
        $smtp->data([ $mail->stringify ]);
        $smtp->quit;
    }

}

##******************************************************************************
## Function TreatMail
##  Description  : Try to treat a mail
##                 
##  Parameters   : a filehandle
##  Return value : 1 if successfull and 0 otherwise
##******************************************************************************
sub TreatMail{
    my $entity = shift;
    my $send_mail_to_admin = shift;

    $send_mail_to_admin = 1 if !defined $send_mail_to_admin;

    my $dest = $entity->head->get("Reply-To");
    $dest = $entity->head->get("From") unless defined $dest;
    chomp $dest;
    
    my $sender = $entity->head->get("To");
    chomp $sender;

    if( !defined($entity) )
    {
        Die("Can't parse mail (MIME::Parser->parse) : $!");
    }

    ### Extract subject. Subject is something like "LogTrend : FunctionToExecute"
    my $subject = $entity->head->get('subject');
    chomp $subject;

    if( $subject !~ /^LogTrend: (.*)$/) {
        ### Not a LogTrend function, skip this mail.
        return 1;
    }
    
    $subject = $1;

    ### Verify function validity 
    my $found = 0;
    foreach my $func_name (@FunctionList) {
        if($func_name eq $subject) {
            $found = 1;
            last;
        }
    }

    ### If the function is not valid, warn, return the message to the sender and return true (skip the message).
    if(! $found) {
        warn "Invalid function in mail subject : $subject";

        SendMail($sender, $dest, "Invalid LogTrend function in subject : $subject\n", $entity);

        return 1;
    }

    ### Run the function.
    my $response = eval "${subject}_Mail(\$entity)";
    
    ### If the returned value is 2, it's a login probleme, if it's 3, it's a XML data probleme, 
    ## else it's a server (storage server or database) probleme
    if( $response eq "OK" )
    {			
        return 1;
    }

    if($response !~ /^ERROR/) {
        $response = "ERROR PROTOCOL Unknown response \"$response\"";
    }
      
    my ($error_id, $error_message) = $response =~ /^ERROR\s+(\w+)(\s+.+)?$/sm;
    $error_message =~ s/^\s*// if  defined $error_message;
    
    ###
    ### Storage Server conection error (not critical)
    ### Add the mail to the queue
    ###
    if( $error_id eq "CONNECT" or 
        $error_id eq "IO" or 
        $error_id eq "DB_CONNECT" or
        $error_id eq "INTERNAL")
    {
        return 0 if(! $send_mail_to_admin);
        
        SysLog "$error_id, $error_message\n";
        
        SendMail($sender, $dest, "The storage server does not reply.\n".
                        "The reason is :\n\t$error_id : $error_message\n\n".
                        "Note : Data are not lost.\n".
                        "       Collected data are currently stored by the mailbridge and will be automatically sent to the server when the connection will be restored.\n");

        return 0;
    }
    ###
    ### Authentication problem. The source is not recognized by the Storage Server.
    ### Critical error : return the mail to sender.
    ###
    elsif( $error_id eq "AUTHENTICATION" )
    {
        return 1 if(! $send_mail_to_admin);
        
        SysLog "$error_id\n";

        SendMail($sender, $dest, "Authentication error on storage server.\n".
                        "If you have change your GnuPG key pair, you must modify the public key in the database.\n\n".
                        "The file todo.xml attached to this mail contains StorageServer references.\n".
                        "The others attached file contains the data and the signature.", $entity);

        return 1;        
    }
    ###
    ### DB_REQUEST error,  probably due to
    ### UNIQUE index or FOREIGN KEY violation.
    ### 
    ### Data are ignored, no error message sent.
    ### Add an error message in system log
    ###     
    elsif($error_id eq "DB_REQUEST") {
        SysLog("LogTrend MailBridge DB_REQUEST error : $error_message");
        return 1;
    }
    ###
    ### Protocol, XML and NOT_A_SIGNATURE error, Critical Error probably due to
    ### an incompatible version of the StorageServer an internal bug.
    ### 
    elsif( $error_id eq "PROTOCOL" or
           $error_id eq "NOT_A_SIGNATURE" or 
           $error_id eq "XML"){

        return 1 if(! $send_mail_to_admin);
        
        my $message = "$error_id";
        $message .= " $error_message" if defined $error_message;
        $message .= "\n";
        SysLog $message;

        ### Invalid XML data.
        $message = "An error occured while sending data to Storage Server. This one has replied : $error_id";
        $message .= " : $error_message" if defined $error_message;
        $message .= ".\nPlease, check if your Storage Server is compatible with this mailbridge.\n".
                    "If yes, it's probably due to a bug in the mailbridge. Please, forward this mail to the mailbridge's developer.\n\n".
                    "The file todo.xml attached to this mail contains StorageServer references.\n".
                    "The others attached file contains the data and the signature.";
        
        SendMail($sender, $dest, $message, $entity);
        
        return 1;
    }
    
    ###
    ### Agent description present in database is incoherent with sent data.
    ### Critical Error : perhaps the agent admin forgot to update the agent 
    ### description in database.
    ###
    elsif( $error_id eq "DB_AGENTDESCRIPTION") {
        return 1 if(! $send_mail_to_admin);
        
        SysLog "$error_id, $error_message\n";
                
        if($subject ne "AgentDescriptionToDB" and $subject ne "UpdateAgentDescription") {
            SendMail($sender, $dest, "The agent description present in database is incoherent with sent data.\n".
                            "Perhaps the agent admin has changed the agent description and forgot to update the agent description with AgentDescriptionToDB\n\n".
                            "The file todo.xml attached to this mail contains StorageServer references.\n".
                            "The others attached file contains the data and the signature.", $entity);
        }
        else {
            SendMail($sender, $dest, "The agent description can't be added to database.\n".
                            "Perhaps the agent has been already added to the database\n".
                            "The error message returned by the MailBridge is : $error_message\n\n".
                            "The file todo.xml attached to this mail contains database references.\n".
                            "The others attached file contains the data description and the signature.", $entity);            
        }
            
        return 1;
    }
    else {
        return 1 if(! $send_mail_to_admin);

        SysLog "PROTOCOL, Unknown response from the server\n";

        SendMail($sender, $dest, "The mailbridge has received an unknown response from the storage server ($response).\n".
                         "Please, check if your Storage Server is compatible with this mailbridge.\n".
                         "If yes, it's probably due to a bug in the mailbridge. Please, forward this mail to the mailbridge's developer.\n\n".
                         "The file todo.xml attached to this mail contains StorageServer references.\n".
                         "The others attached file contains the data and the signature.", $entity);
        return 1;
    }
}




##******************************************************************************
## Function queue_mail
##  Description  : Add the mail to the queue
##                 
##  Parameters   : an MIME::Entity
##******************************************************************************
sub queue_mail {
    my $entity = shift;

    ### Create a Lock for mail queue
    open(F_QUEUE_LOCK, "+>$mb_directory/mail_queue_lock") 
        or Die("can't open the lock file $mb_directory/mail_queue_lock : $!");
    
    flock(F_QUEUE_LOCK, 2)
        or Die("can't flock $mb_directory/mail_queue_lock : $!");
    
    ### print the current mail in a file
    my $tmpFileName = tmpnam();
    $tmpFileName =~ s/^\/tmp\/file//;
    my $filename = "logtrend-mail-$$-".$tmpFileName;
    
    open(F_MAIL, ">$mb_directory/$filename")
        or Die("Can't open file $mb_directory/$filename : $!");
    $entity->print(\*F_MAIL);
    close F_MAIL;
    
    ### open the mail Queue index
    open(F_QUEUE, "+<$mb_directory/mail_queue") 
        or Die("can't open $mb_directory/mail_queue : $!");
    
    my $new_index = "";
    my $line;
    while(<F_QUEUE>) {
        last if $_ eq "</MailQueue>\n";
        $new_index .= $_;
    } 
    
    seek(F_QUEUE, 0, 0);
    print F_QUEUE $new_index.
        "\t<Mail File=\"$filename\" />\n".
            "</MailQueue>\n";
    
    close F_QUEUE;
    
    ### Unlock and remove queue lock
    unlink("$mb_directory/mail_queue_lock");
    flock(F_QUEUE_LOCK, 8);
    close F_QUEUE_LOCK;    
}


##******************************************************************************
## Function main
##******************************************************************************
sub main
{
    $SIG{__WARN__} = sub { SysLog(@_) };

    ## parse command line option
    if($#ARGV != -1 and $ARGV[0] eq "-s") {
        if($#ARGV ne 1) {
            warn "Invalid options on command line, ignored".
                 "Usage : MailBridge.pl [-s smtpserver]\n";
        }
        else {
            $SMTPServer = $ARGV[1];
        }
    }
    
    $SMTPServer = 'localhost' unless defined $SMTPServer;


    MakeMailBridgeDir();

    ### Empty the mail queue
    EmptyQueue();

    ### And treat the current mail
    ### Setup MIME parser
    my $parser = new MIME::Parser;
    
    if(! -d "$mb_directory/logtrend-mailbridge-$$") {
        mkdir("$mb_directory/logtrend-mailbridge-$$",0700)
            or Die("Can't create directory $mb_directory/logtrend-mailbridge-$$ : $!");
    }
    
    $parser->output_dir("$mb_directory/logtrend-mailbridge-$$");
    
    ### Parse the mail
    my $entity = $parser->parse(\*STDIN);

    ### Returned value : 1 = StorageServer/DataBase connection error, 0 = OK.
    if(! TreatMail($entity)) {
        ### StorageServer/DataBase connection error, send a mail to the server admin, 
        ### add this mail to the queue
        queue_mail($entity);
    }
    
    $parser->filer->purge;

    rmdir("$mb_directory/logtrend-mailbridge-$$");

    return 0;
}

##******************************************************************************
main();

