#!/usr/bin/perl -w

#$Header: /var/lib/cvs/LogTrend/Consolidation/DataBaseAnalyzer.pm,v 1.4 2002/05/02 14:47:08 lsimonneau Exp $
##*****************************************************************************
##  ConsolidationSet
##  Description  : Parse XML File and create Consolidation object. 
##
##  Project      : LogTrend 1.0.0.0 - Atrid Systemes
##  Author       : Laurent Simonneau l.simonneau@atrid.fr
##*****************************************************************************
#$Log: DataBaseAnalyzer.pm,v $
#Revision 1.4  2002/05/02 14:47:08  lsimonneau
#A lot of bugfixes.
#Critical bugfixes in consolidation.
#
#Revision 1.3  2001/10/24 08:11:35  lsimonneau
#Enhanced databasae error support.
#
#Revision 1.2  2001/10/24 07:46:21  lsimonneau
#Don't crash when several data have the same collection date.
#
#Revision 1.1  2001/09/24 15:29:10  lsimonneau
#Reimplementation of consolidation.
#

package LogTrend::Consolidation::DataBaseAnalyzer;

use LogTrend::DataBaseAccess::PostgreSQLDataBase;


##*****************************************************************************
## Constructor  public
##  Description  : creat a new ConsolidationSet
##  Parameters   : the XML Root node of the consolidation and defaults rules.
##*****************************************************************************

sub new {
    my ($classname, $db_handle) = @_;
    
    my $self = {};
    
    bless($self, $classname);
    
    $self->{DB_HANDLE} = $db_handle;

    return $self;
}



##*****************************************************************************
##  GetZoneList public
##  Description  : Return the list of continous data zone. 
##                 A zone is a suite of data with approximately the same frequency
##  Parameters   : none
##  Return : a list of [mean time between two values, start date, end date]
##*****************************************************************************
sub GetZoneList {
    my ($self, $source, $agent, $var_name) = @_;

    my $list = $self->{DB_HANDLE}->GetCollectionDateInTimeInterval($source, $agent,
								   $var_name, 0);    
    my $nb_val = @$list;

    return [] if $nb_val < 2;

    my $delta;
    my $old_delta=0;
    my $derive;
    my $mean=0;
    my $i_start_mean=0;
    my @zone_list;

    my $i;
    my $database_error=0;
    for($i=1; $i < $nb_val; $i++) {
	$delta = $list->[$i] - $list->[$i-1];
	if($delta == 0) {
	    $database_error++;
	    next;
	}
	
        $derive = ($delta - $old_delta)/$delta;
	$derive = sprintf "%.1f", $derive;

	$old_delta = $delta;
	
	if($derive!=0) {
	    if (($i - $i_start_mean - $database_error) >= 5) {
		unshift @zone_list, [$mean/($i-$i_start_mean-$database_error), $list->[$i_start_mean], $list->[$i-1]];
	    }
	    
	    $mean=$delta;
	    $i_start_mean = $i;
	    $database_error=0;
	}
	else {
	    $mean += $delta;
	}
    }

    if (($list->[$i-1] - $list->[$i_start_mean]) != 0) {
	unshift @zone_list, [$mean/($i-$i_start_mean-$database_error), $list->[$i_start_mean], $list->[$i-1]];
    }

    return \@zone_list;
}


##*****************************************************************************
##  GetConsolidationList public
##  Description  : return the consolidation detected in database
##  Parameters   : none
##*****************************************************************************
sub GetConsolidationList {
    my ($self, $zone_list) = @_;

    return [] if $#$zone_list == -1;

    my @conso_list;
    my $zone;
    my $prev_zone;
    for (my $i = 1; defined $zone_list->[$i]; $i++) {
	$zone = $zone_list->[$i];
	$prev_zone = $zone_list->[$i-1];

	# if the there 35% of difference between the two zone, it's a consolidation threshold
	my $percentage = $prev_zone->[0]/$zone->[0];

	if($percentage <= 0.60) {
	    push @conso_list, [$prev_zone->[1], $zone->[2]];
	}
    }

    return \@conso_list;
}


1;

__END__
