#!/usr/bin/perl -w

#$Header: /home2/cvsroot/LogTrend/ComplexAlarm/Equation/EquationParser.pm,v 1.6 2001/12/13 16:01:54 lsimonneau Exp $
##*****************************************************************************
## Class EquationParser  virtual
##  Description  : Generic class for equation parsers
##                 'Real' parsers must inherit from this class and implements
##                 Function, AndOP, OrOP, NotOP and ParseError methods
##                 The public method Run can be implemented by children 
##                 but should call SUPER::Run.
##
##                 Protected methods :
##                 - Function is called when a function is parsed.
##                 - AndOP, OrOP and NotOP are called when an operator
##                   is parsed.
##                 - ParseError is called when a parse error occured
## 
##                 Public methods :
##
##  Project      : LogTrend 1.0.0.0 - Atrid Systemes
##  Author       : Laurent Simonneau l.simonneau@atrid.fr
##*****************************************************************************
#$Log: EquationParser.pm,v $
#Revision 1.6  2001/12/13 16:01:54  lsimonneau
#Minor bugfixes.
#
#Revision 1.5  2001/09/26 15:44:53  lsimonneau
#Reimplementation of the Equation parser.
#
#Revision 1.4  2001/07/25 12:00:47  lsimonneau
#Ajout du support des equations arithmtiques sur les variables en argument des fonctions de detection.
#
#Revision 1.3  2001/06/21 16:07:14  lsimonneau
#Dplacement de la clause 'package' avant les 'use'
#
#Revision 1.2  2001/06/07 14:41:04  lsimonneau
#Passage du unshift @INC, '..' au LogTrend::
#
#Revision 1.1  2001/05/30 09:36:57  lsimonneau
#Premire version du module d'alarmes complexes dans le CVS.
#Toutes les fonctionnalits ont t testes et correctement.
#

package LogTrend::ComplexAlarm::Equation::EquationParser;

use strict;
use LogTrend::ComplexAlarm::Parser;

@LogTrend::ComplexAlarm::Equation::EquationParser::ISA = ("LogTrend::ComplexAlarm::Parser");

my @tokenlist = ('\(',
		 '\)', 
		 ',', 
		 'AND',
		 'OR', 
		 'NOT',
		 '\+', 
		 '\-', 
		 '\*',
		 '\/',
		 '(\d+(\.\d+)?)',
		 '(([a-zA-Z0-9]+)|(("[^\"]*")|(\'[^\']*\')))\[\d+\s*,\s*\d+\]',
		 '([a-zA-Z0-9_]+)',
		 '("[^\"]*")|(\'[^\']*\')');

use constant OPEN_PAR => 0;
use constant CLOSE_PAR => 1;
use constant COMMA => 2;
use constant AND_OP => 3;
use constant OR_OP => 4;
use constant NOT_OP => 5;
use constant PLUS_OP => 6;
use constant MINUS_OP => 7;
use constant MULT_OP => 8;
use constant DIV_OP => 9;
use constant NUMBER => 10;
use constant EXT_VARNAME => 11;
use constant VARNAME => 12;
use constant FUNCNAME => 12;
use constant STRING => 13;
use constant END_OF_STRING => 14;


my %op_priority_hash = ("NOT", 3, "AND", 2, "OR", 1, "(", 0);

my %arithm_priority_hash = ("*", 2, "/", 2, "-", 1, "+", 1, "(", 0);

##*****************************************************************************
## Constructor  public
##  Description  : creat a new EquationParser
##  Parameters   : the equation string
##*****************************************************************************
sub new
{
    my ($classname, $equation) = @_;
    my $self = $classname->SUPER::new($equation, @tokenlist);

    bless($self, $classname);

    $self->{EQUATION} = $equation;
    $self->{SYMBOLS_STACK} = ();
    $self->{FUNCTIONS_STACK} = ();
    $self->{ARITHM_EQU_STACK} = ();
    $self->{ARITHM_SYMBOLS_STACK} = ();
    
    return $self;
}


##*****************************************************************************
## Method Run public
##  Description  : parse the equation and call he following protected methods.
##  Parameters   : none
##*****************************************************************************
sub Run
{
    my $self = shift;

    $self->NextToken;
    my $result = $self->Equation;    

    my ($token_id, $token_string) = $self->CurToken;
    $self->ParseError("Abnormal equation termination") if $token_id != END_OF_STRING;

    return $result;
}

##
## Equation -> Eq1 OR Equation |
##             Eq1
##
sub Equation {
    my $self = shift;
    my ($left_op, $right_op);

    $left_op = $self->Eq1;

    my ($token_id, $token_string) = $self->CurToken;

    if($token_id == OR_OP) {
	$self->NextToken;    
	$right_op = $self->Equation;

	return $self->APPLY_OR($left_op, $right_op);
    }

    return $left_op;
}


##
## Eq1 -> Eq2 OR Eq1 |
##        Eq2
##
sub Eq1 {
    my $self = shift;
    my ($left_op, $right_op);
    
    $left_op = $self->Eq2;

    my ($token_id, $token_string) = $self->CurToken;
    
    if($token_id == AND_OP) {
	$self->NextToken;
	$right_op = $self->Eq1;
	return $self->APPLY_AND($left_op, $right_op);
    }

    return $left_op;
}


##
## Eq2 -> ( Equation ) |
##        NOT ( Equation ) |
##        NOT DetFunc |
##        DetFunc
##
sub Eq2 {
    my $self = shift;
    
    my ($token_id, $token_string) = $self->CurToken;

    my $result;
    
    if($token_id == OPEN_PAR) {
	##
	## ( Equation )
	##
	$self->NextToken;

	$result = $self->Equation;
	
	($token_id, $token_string) = $self->CurToken;
	$self->ParseError("Expecting ), having $token_string") if $token_id != CLOSE_PAR;

	$self->NextToken;

	return $result;
    }
    elsif($token_id == NOT_OP) {
	##
	## NOT ( Equation ) |
	## NOT DetFun
	##
	($token_id, $token_string) = $self->NextToken;
	if($token_id == OPEN_PAR) {
	    ## NOT ( Equation )
	    $self->NextToken;
	    
	    $result = $self->Equation;
	    
	    ($token_id, $token_string) = $self->CurToken;
    
	    $self->ParseError("Expecting ), having $token_string") if $token_id != CLOSE_PAR;
	    
	    $self->NextToken;
	    return $self->APPLY_NOT($result);
	}
	else {
	    ## NOT DetFunc
	    $result = $self->DetFunc;
	    return $self->APPLY_NOT($result);
	}
    }
    else {
	##
	## Function
	##
	return $self->DetFunc;
    }
}

##
## DetFunc -> funcname( Arg ) |
##            funcname()
##
sub DetFunc {
    my $self = shift;
    my @arg_list;
    my $funcname;

    my ($token_id, $token_string) = $self->CurToken;
    $self->ParseError("Expecting FuncName, having $token_string") if $token_id != FUNCNAME;

    $funcname = $token_string;
    
    ($token_id, $token_string) = $self->NextToken;
    $self->ParseError("Expecting (, having $token_string") if $token_id != OPEN_PAR;
    
    ($token_id, $token_string) = $self->NextToken;
    if($token_id != CLOSE_PAR) {
	@arg_list = $self->Arg;

	($token_id, $token_string) = $self->CurToken;
	$self->ParseError("Expecting ), having $token_string") if $token_id != CLOSE_PAR;
    }

    $self->NextToken;

    return $self->FUNCTION($funcname, @arg_list);
}

##
## Arg -> ArEq, Arg |
##        ArEq
##
sub Arg {
    my $self = shift;
    my @arg_list;

    push @arg_list, $self->ArEq;
    
    my ($token_id, $token_string) = $self->CurToken;
    
    if($token_id == COMMA) {
	$self->NextToken;
	push @arg_list, $self->Arg;
    }

    return @arg_list;
}

##
## ArEq -> ArEq1 + ArEq |
##         ArEq1 - ArEq |
##         ArEq1
##
sub ArEq {
    my $self = shift;
    my ($left_op, $right_op);

    my ($token_id, $token_string) = $self->CurToken;

    $left_op = $self->ArEq1;

    ($token_id, $token_string) = $self->CurToken;

    if($token_id == PLUS_OP) {
	$self->NextToken;
	$right_op = $self->ArEq;

	return $self->APPLY_PLUS($left_op, $right_op);
    }
    elsif($token_id == MINUS_OP) {
	$self->NextToken;
	$right_op = $self->ArEq;

	return $self->APPLY_MINUS($left_op, $right_op);
    }

    return $left_op;
}


##
## ArEq1 -> ArEq2 * ArEq1 |
##          ArEq2 / ArEq1 |
##          ArEq2
##
sub ArEq1 {
    my $self = shift;
    my ($left_op, $right_op);

    $left_op = $self->ArEq2;

    my ($token_id, $token_string) = $self->CurToken;

    if($token_id == MULT_OP) {
	$self->NextToken;
	$right_op = $self->ArEq1;

  	return $self->APPLY_MULT($left_op, $right_op);
    }
    elsif($token_id == DIV_OP) {
	$self->NextToken;
	$right_op = $self->ArEq1;

  	return $self->APPLY_DIV($left_op, $right_op);
    }

    return $left_op;
}


##
## ArEq2 -> ( ArEq ) |
##          - ArEq |
##          variable | number | string 
##
sub ArEq2 {
    my $self = shift;    
    my ($token_id, $token_string) = $self->CurToken;
    my $result;
    

    if($token_id == OPEN_PAR) {
	$self->NextToken;	
	$result = $self->ArEq;

	($token_id, $token_string) = $self->CurToken;
	$self->ParseError("Expecting ), having $token_string") if $token_id != CLOSE_PAR;

	$self->NextToken;

	return $result;
    }
    elsif($token_id == MINUS_OP) {
	$self->NextToken;

	return $self->APPLY_MINUS(0, $self->ArEq);
    }
    elsif($token_id == VARNAME or $token_id == EXT_VARNAME or $token_id == STRING) {        
        $token_string =~ s/^[\"|\'](.*)[\"|\']$/$1/;
	$self->NextToken;
	return $token_string;
    }
    elsif($token_id == NUMBER) {
	$self->NextToken;
	return $token_string;
    }
    else {
	$self->ParseError("Unexpected $token_string");
    }
}



##*****************************************************************************
## Method FUNCTION virtual protected
##  Description  : Called by the parser when a function is parsed.
##  Parameters   : The function name
##                 The list of arguments name.
##*****************************************************************************
sub FUNCTION
{
}

##*****************************************************************************
## Method APPLY_AND virtual protected
##  Description  : Called by the parser when an AND operator is parsed.
##  Parameters   : none
##*****************************************************************************
sub APPLY_AND
{
}

##*****************************************************************************
## Method APPLY_OR virtual protected
##  Description  : Called by the parser when an OR operator is parsed.
##  Parameters   : none
##*****************************************************************************
sub APPLY_OR
{
}

##*****************************************************************************
## Method APPLY_NOT virtual protected
##  Description  : Called by the parser when an NOT operator is parsed.
##  Parameters   : none
##*****************************************************************************
sub APPLY_NOT
{
}

##*****************************************************************************
## Method APPLY_PLUS virtual protected
##  Description  : Called by the parser when an + arithmetic operator is parsed.
##  Parameters   : none
##*****************************************************************************
sub APPLY_PLUS
{
}

##*****************************************************************************
## Method APPLY_MINUS virtual protected
##  Description  : Called by the parser when an - arithmetic operator is parsed.
##  Parameters   : none
##*****************************************************************************
sub APPLY_MINUS
{
}


##*****************************************************************************
## Method MULT virtual protected
##  Description  : Called by the parser when an * arithmetic operator is parsed.
##  Parameters   : none
##*****************************************************************************
sub APPLY_MULT
{
}


##*****************************************************************************
## Method APPLY_DIV virtual protected
##  Description  : Called by the parser when an / arithmetic operator is parsed.
##  Parameters   : none
##*****************************************************************************
sub APPLY_DIV
{
}

##*****************************************************************************
## Method ParseError virtual protected
##  Description  : Called by the perser when an error occure
##  Parameters   : A string on the error
##*****************************************************************************
sub ParseError {
    my ($self, $string) = @_;
    die "Parse error : $string\n";
}

1;
