#!/usr/bin/perl -w

#$Header: /home2/cvsroot/LogTrend/ComplexAlarm/DataBase/ComplexAlarmPostgreSQLDataBase.pm,v 1.7 2002/02/05 15:52:50 lsimonneau Exp $
##*****************************************************************************
##  ComplexAlarmPostgreSQLDataBase
##  Description  : ComplexAlarmPostgreSQLDataBase is an PostgreSQLDataBase  
##                 with additionnal methods for Complex Alarm tables
##                 
##  Project      : LogTrend 1.0.0.0 - Atrid Systemes
##  Author       : Laurent Simonneau l.simonneau@atrid.fr
##*****************************************************************************
#$Log: ComplexAlarmPostgreSQLDataBase.pm,v $
#Revision 1.7  2002/02/05 15:52:50  lsimonneau
#Several critical bugs fixed
#
#Revision 1.6  2002/02/01 17:35:16  lsimonneau
#Modify database structure. *_values tables inherite of data.
#
#Revision 1.5  2001/10/10 16:33:53  lsimonneau
#Remove AcquitComplexAlarm. This feature is now in DatabaseAccess.
#
#Revision 1.4  2001/06/21 16:07:14  lsimonneau
#Dplacement de la clause 'package' avant les 'use'
#
#Revision 1.3  2001/06/11 15:08:23  lsimonneau
#last_connection_date est positionner a 'now' au lieu de 'epoch' lors de l'ajout d'une description.
#
#Revision 1.2  2001/06/07 14:41:04  lsimonneau
#Passage du unshift @INC, '..' au LogTrend::
#
#Revision 1.1  2001/05/30 09:36:57  lsimonneau
#Premire version du module d'alarmes complexes dans le CVS.
#Toutes les fonctionnalits ont t testes et correctement.
#

package LogTrend::ComplexAlarm::DataBase::ComplexAlarmPostgreSQLDataBase;

use strict;
use DBI;
use LogTrend::DataBaseAccess::PostgreSQLDataBase;
use LogTrend::ComplexAlarm::DataBase::ComplexAlarmDataBase;

@LogTrend::ComplexAlarm::DataBase::ComplexAlarmPostgreSQLDataBase::ISA = ("LogTrend::DataBaseAccess::PostgreSQLDataBase", "LogTrend::ComplexAlarm::DataBase::ComplexAlarmDataBase");

##*****************************************************************************
## Constructor  public
##  Description  : creat a new ComplexAlarmPostgreSQLDataBase
##
##  Parameters   : The database name,
##                 The database server host name,
##                 The database server port,
##                 An username and a password of a database user
##*****************************************************************************
sub new
{
    my ($classname, $databasename, $host, $port, $username, $password) = @_;

    ## Only PostgreSQLDataBase::new is called, not ComplexAlarmDataBase::new
    my $self = $classname->SUPER::new($databasename, $host, $port,
				      $username, $password);
    
    bless($self, $classname);

    return $self;
}

##*****************************************************************************
## StoreComplexAlarm public
##  Description  : Store a complex alarm in the database
##  Parameters   : The source name,
##                 The agent name,
##                 The complex alarm number.
##*****************************************************************************
sub StoreComplexAlarm
{
    my ($self, $source, $agent, $number) = @_;
    my $st_handle;

    my $result;
    
    my $id_alarm_description = $self->getIdAlarmDescription($source, $agent, $number);

    # Check if a non acquitted alarm exists
    $st_handle = $self->{DBHANDLE}->prepare("
SELECT COUNT(*) 

FROM   alarms

WHERE  (id_alarm_description = ?) AND 
       (acquittal_date IS NULL)")
	or die "Can't verify if a complex alarm is acquitted : $id_alarm_description (prepare)";
    
    $st_handle->execute($id_alarm_description)
	or die "Can't verify if a complex alarm is acquitted : $id_alarm_description (execute)";
    
    my @row = $st_handle->fetchrow_array();

    $st_handle->finish();
    
    if($#row == -1){
	$result = 0;
    }
    else{
	$result = $row[0];
    }
    

    if ($result == 0) {
	# if there is no non aquitted alarms, insert a new alarm
	$st_handle = $self->{DBHANDLE}->prepare("
INSERT INTO alarms(id_alarm_description, storage_date, release_date) 

VALUES (?, 'now()', 'now()')")
	    or die "Can't add a complex alarm : $id_alarm_description (prepare)";
	
	$st_handle->execute($id_alarm_description)
	    or die "Can't add a complex alarm : $id_alarm_description (execute)";
	
	$st_handle->finish();
    }
    else {
	# if there is a non acquitted alarm, update the storage date of this one
	$st_handle = $self->{DBHANDLE}->prepare("
UPDATE alarms 

SET    storage_date = 'now()' 

WHERE  (id_alarm_description = ?) AND 
       (acquittal_date IS NULL)")
	    or die "Can't update a complex alarm storage date : $id_alarm_description (prepare)";	    
	    
	$st_handle->execute($id_alarm_description)
	    or die "Can't update a complex alarm storage date : $id_alarm_description (execute)";
	
	$st_handle->finish();	
    }
}

##*****************************************************************************
## StoreComplexAlarmDescription  public
##  Description  : Store a complex alarm description in the database
##  Parameters   : the agent_id ; 
##                 the complex alarm description number ;
##                 the complex alarm name ;
##                 the complex alarm level.
##*****************************************************************************
sub StoreComplexAlarmDescription
{
    my ($self, $source, $agent, $number, $name, $level_name) = @_;
    my %level_hash = ("Info", 1, "Warning", 2, "Error", 3);
    my $id_level = $level_hash{$level_name};
    my $st_handle;

    # check if the agent is in the database
    my $id_agent = $self->getIdAgent($source, $agent)
	or die "Agent $agent on source $source, does not exist";
    
    # add the description
    $st_handle = $self->{DBHANDLE}->prepare("
INSERT INTO complex_alarms_descriptions(id_agent, alarm_number, message, id_alarm_level, last_checked_data_date) 

VALUES (?, ?, ?, ?, 'now()')")
	or die "Can't add complex alarm description number $number named $name for agent $agent on source $source (prepare)";
    
    $st_handle->execute($id_agent, $number, $name, $id_level)
	or die "Can't add complex alarm description number $number named $name for agent $agent on source $source (execute)";

    $st_handle->finish();
}

##*****************************************************************************
## getLastCheckedDataDate  private
##  Description  : Retrieve the date of the last checked data by a 
##                 complex alarm 
##  Parameters   : 
##*****************************************************************************
sub getLastCheckedDataDate
{
    my ($self, $id_alarm_description) = @_;
    my $st_handle;
    my $result = 0;
    
    # Prepare Database request
    $st_handle = $self->{DBHANDLE}->prepare("
SELECT date_part('epoch', last_checked_data_date)

FROM   complex_alarms_descriptions 

WHERE  id_alarm_description = ?")
	or die "Can't retrieve the last checked data date of $id_alarm_description (prepare)";
    
    # Run the request
    $st_handle->execute($id_alarm_description)
	or die "Can't retrieve the last checked data date of $id_alarm_description (execute)";

    my @row = $st_handle->fetchrow_array();
    
    $st_handle->finish();
    
    if($#row == -1) {
	return 0;
    }
    else {
	return $row[0];
    }

}

    
##*****************************************************************************
## updateLastCheckedDataDate  private
##  Description  : Update the date of the last checked data by a 
##                 complex alarm.
##  Parameters   : The complex alarm description id 
##                 and the date in seconde since epoch.
##*****************************************************************************
sub updateLastCheckedDataDate
{
    my ($self, $id_alarm_description, $new_date) = @_;
    my $st_handle;
    my $result = 0;
    my @row;
    
    $st_handle = $self->{DBHANDLE}->prepare("
UPDATE complex_alarms_descriptions 

SET    last_checked_data_date = datetime(?) 

WHERE  id_alarm_description = ?")
	or die "Can't update the last checked data date of complex alarm description $id_alarm_description to $new_date (prepare)";
    
    $st_handle->execute($new_date, $id_alarm_description)
	or die "Can't update the last checked data date of complex alarm description $id_alarm_description to $new_date (execute)";

    $st_handle->finish();   
}

##*****************************************************************************
## getIdAlarmDescription  private
##  Description  : Retrieve the id_complex_alarm_descpription 
##  Parameters   : The source name,
##                 The agent name,
##                 The complex alarm number
##*****************************************************************************
sub getIdAlarmDescription
{
    my ($self, $source, $agent, $alarm_number) = @_;
    my $st_handle;
    my @row;
    
    my $id_agent = $self->getIdAgent($source, $agent);

    $st_handle = $self->{DBHANDLE}->prepare("
SELECT id_alarm_description

FROM   agents_alarms_descriptions 

WHERE id_agent = ? AND 
      alarm_number = ?")
	or die "Can't retrieve the complex alarm description for alarm number $alarm_number on agent $agent on source $source (prepare)";
    
    $st_handle->execute($id_agent, $alarm_number)
	or die "Can't retrieve the complex alarm description for alarm number $alarm_number on agent $agent on source $source (execute)";

    @row = $st_handle->fetchrow_array();

    $st_handle->finish();

    if($#row == -1) {
	return 0;
    }
    else {
        return $row[0];
    }
}

1;


