//
//  Login.app
// 
//  Copyright (c) 1997, 1998 Per Liden
// 
//  This program is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 2 of the License, or
//  (at your option) any later version.
//
//  This program is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with this program; if not, write to the Free Software
//  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
//

#include "Panel.h"

 
Panel::Panel(Display* dpy, int scr, Window root)
{
    // Set display
    Dpy = dpy;
    Scr = scr;
    Root = root;

    // Init GC
    XGCValues gcv;
    unsigned long gcm;           
    gcm = GCForeground|GCBackground|GCGraphicsExposures|GCFont;
    gcv.foreground = GetColor("black");
    gcv.background = GetColor("white");
    gcv.graphics_exposures = False;
    gcv.font = XLoadFont(Dpy, INPUT_FONT);
    TextGC = XCreateGC(Dpy, Root, gcm, &gcv);

    // Load background xpm   
    PanelPixmapAttributes.valuemask = 0;
    int tmp = XpmCreatePixmapFromData(Dpy, Root, PanelXpm, 
				      &PanelPixmap, NULL, &PanelPixmapAttributes);
    // Check for error
    switch(tmp)
    {
    case XpmColorError:
	cerr << APPNAME << ": xpm image loaded but did not get all colors needed" << endl;
	break;

    case XpmColorFailed:
	cerr << APPNAME << ": could not load xpm image (not enough colors available)" << endl;
	exit(ERR_EXIT);
	break;

    case XpmNoMemory:
	cerr << APPNAME << ": could not load xpm image (not enough memory available)" << endl;
	exit(ERR_EXIT);
	break;

    case XpmOpenFailed:
    case XpmFileInvalid:
	cerr << APPNAME << ": could not load xpm image (image broken or corrupt)" << endl;
	exit(ERR_EXIT);
	break;

    case XpmSuccess:
    default:
	// Image loaded ok
	break;
    }

    
    ShrinkPixmap = XCreatePixmap(Dpy, Root, 
				 PanelPixmapAttributes.width, 
				 PanelPixmapAttributes.height,
				 DefaultDepth(Dpy, Scr));

    // Calculate position
    X = (XWidthOfScreen(ScreenOfDisplay(Dpy, Scr)) - PanelPixmapAttributes.width) / 2;
    Y = (XHeightOfScreen(ScreenOfDisplay(Dpy, Scr)) - PanelPixmapAttributes.height - 100) / 2;
  
    // Init In
    In = new Input;
}


Panel::~Panel()
{    
    XFreePixmap(Dpy, ShrinkPixmap);
    XFreeGC(Dpy, TextGC);
    delete In;
}


void Panel::OpenPanel()
{
    // Create window
    Win = XCreateSimpleWindow(Dpy, Root, X, Y, 
			      PanelPixmapAttributes.width,
			      PanelPixmapAttributes.height, 
			      1, GetColor("black"), GetColor("white"));
    
    // Events
    XSelectInput(Dpy, Win, ExposureMask | KeyPressMask);

    // Set background
    XSetWindowBackgroundPixmap(Dpy, Win, PanelPixmap);

    // Show window
    XMapWindow(Dpy, Win);
 
    // Grab keyboard
    XGrabKeyboard(Dpy, Win, False, GrabModeAsync, GrabModeAsync, CurrentTime);
	
    XFlush(Dpy);
}


void Panel::ClosePanel()
{
    XUngrabKeyboard(Dpy, CurrentTime);
    XUnmapWindow(Dpy, Win);
    XDestroyWindow(Dpy, Win);
    XFlush(Dpy);
}


void Panel::ClearPanel()
{
    In->Reset();
    XClearWindow(Dpy, Win);
    Cursor(SHOW);
    XFlush(Dpy);
}


void Panel::ShakePanel()
{
    // Erase name and passwd  
 /*
    char* name = In->GetName();
    char* passwd = In->GetHiddenPasswd();
    Cursor(HIDE);
    XSetForeground(Dpy, TextGC, GetColor(INPUT_BGCOLOR));
    XDrawString(Dpy, Win, TextGC, INPUT_NAME_X, INPUT_NAME_Y, name, strlen(name));
    XDrawString(Dpy, Win, TextGC, INPUT_PASSWD_X, INPUT_PASSWD_Y, passwd, strlen(passwd));
    In->Reset();
    Cursor(SHOW);
*/
    ClearPanel();

    // Shake panel
    for(int i=0;i<3;i++)
    {
	int x;
	for(x=2;x<=20;x+=2)
	        XMoveWindow(Dpy, Win, X+x, Y);
	for(x=20;x>=-20;x-=2)
	        XMoveWindow(Dpy, Win, X+x, Y);       
	for(x=-18;x<=0;x+=2)
	        XMoveWindow(Dpy, Win, X+x, Y);
	XFlush(Dpy);
    }
}


void Panel::ShrinkPanel()
{
    int x = X;
    int y = Y;
    int width = PanelPixmapAttributes.width;
    int height = PanelPixmapAttributes.height; 

    XCopyArea(Dpy, PanelPixmap, ShrinkPixmap, TextGC,
 	      0, 0, width, height, 0, 0);

    for(;width>0; width-=10, x+=5)
    {
	XMoveResizeWindow(Dpy, Win, x, y, width, height);
	XCopyArea(Dpy, ShrinkPixmap, ShrinkPixmap, TextGC, 
		  5, 0, width, height, 0, 0);
	XSetWindowBackgroundPixmap(Dpy, Win, ShrinkPixmap);
	XFlush(Dpy);
    }
    
    XSetWindowBackgroundPixmap(Dpy, Win, PanelPixmap);
}


void Panel::Message(char* text)
{
    XSetForeground(Dpy, TextGC, GetColor(MSG_COLOR));
    XDrawString(Dpy, Root, TextGC, 10, 25, text, strlen(text));
    XFlush(Dpy);
}


Input* Panel::GetInput()
{
    return In;
}


unsigned long Panel::GetColor(char* colorname)
{
    XColor color;
    XWindowAttributes attributes;
    
    XGetWindowAttributes(Dpy, Root, &attributes);
    color.pixel = 0;
    
    if(!XParseColor(Dpy, attributes.colormap, colorname, &color)) 
	cerr << APPNAME << ": can't parse color " << colorname << endl;
    else if(!XAllocColor(Dpy, attributes.colormap, &color)) 
	cerr << APPNAME << ": can't allocate color " << colorname << endl;   
    
    return color.pixel;
}


void Panel::Cursor(int visible)
{       
    char* text;
    int xx, yy;

    switch(In->GetField())
    {
    case GET_PASSWD:
	text = In->GetHiddenPasswd();
	xx = INPUT_PASSWD_X;
	yy = INPUT_PASSWD_Y;
	break;

    case GET_NAME:
	text = In->GetName();
	xx = INPUT_NAME_X;
	yy = INPUT_NAME_Y;
	break;
    }

    if(visible == SHOW)
	XSetForeground(Dpy, TextGC, GetColor(INPUT_FGCOLOR));
    else
	XSetForeground(Dpy, TextGC, GetColor(INPUT_BGCOLOR));
    
    xx += XTextWidth(XLoadQueryFont(Dpy, INPUT_FONT), text, strlen(text));    
    XDrawLine(Dpy, Win, TextGC, 
	      xx, yy-(INPUT_CURSOR_HEIGHT-3), 
	      xx, yy+3);
    XDrawLine(Dpy, Win, TextGC, 
	      xx+1, yy-(INPUT_CURSOR_HEIGHT-3), 
	      xx+1, yy+3); 
}


int Panel::EventHandler(XEvent* event)
{
    Action = WAIT;

    switch(event->type) 
    {	
    case Expose:
	OnExpose(event);
	break;
	
    case KeyPress:
	OnKeyPress(event);       
	break;
    }

    return Action;
}


void Panel::OnExpose(XEvent* event)
{
    char* name = In->GetName();
    char* passwd = In->GetHiddenPasswd();
    XSetForeground(Dpy, TextGC, GetColor(INPUT_FGCOLOR));
    XDrawString(Dpy, Win, TextGC, INPUT_NAME_X, INPUT_NAME_Y, name, strlen(name));
    XDrawString(Dpy, Win, TextGC, INPUT_PASSWD_X, INPUT_PASSWD_Y, passwd, strlen(passwd));
    Cursor(SHOW);
}
 

void Panel::OnKeyPress(XEvent* event)
{

    char del;
    char buffer;    
    KeySym keysym;
    XComposeStatus compstatus;
    int xx;
    int yy;
    char* text;

    Cursor(HIDE);
    XLookupString(&event->xkey, &buffer, 1, &keysym, &compstatus);
    del = In->Key(buffer, keysym);
    Action = In->GetAction();

    switch(In->GetField())
    {
    case GET_PASSWD:
	text = In->GetHiddenPasswd();
	xx = INPUT_PASSWD_X;
	yy = INPUT_PASSWD_Y;
	break;

    case GET_NAME:
	text = In->GetName();
	xx = INPUT_NAME_X;
	yy = INPUT_NAME_Y;
	break;	
    }

    
    if(del == 0)
    {
	// No character deleted
	XSetForeground(Dpy, TextGC, GetColor(INPUT_FGCOLOR));
	XDrawString(Dpy, Win, TextGC, xx, yy, text, strlen(text));
    }
    else
    {
	// Character deleted
	int i = 0;
	char tmp[2];
	tmp[0] = del;
	tmp[1] = '\0';

	XSetForeground(Dpy, TextGC, GetColor(INPUT_BGCOLOR));
	i = XTextWidth(XLoadQueryFont(Dpy, INPUT_FONT), text, strlen(text));
	XDrawString(Dpy, Win, TextGC, xx + i, yy, tmp, 1);
    }

    Cursor(SHOW);
}

