/*
 * midi_device.cpp - base-class for midi-device-drivers like ALSA-driver for
 *                   midi-port, pc-keyboard-wrapper etc.
 *
 * Linux MultiMedia Studio
 * Copyright (_c) 2004-2005 Tobias Doerffel <tobydox@users.sourceforge.net>
 * This file partly contains code from Fluidsynth, Peter Hanappe
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public
 * License along with this program (see COPYING); if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 *
 */


#include "qt3support.h"
#ifndef QT4
#include <qmap.h>
#endif

#include "midi_device.h"
/*#include "midi_mapper.h"*/
#include "note_play_handle.h"
#include "knob.h"
#include "song_editor.h"
#include "channel_track.h"
#include "templates.h"


#include "midi_alsa_raw.h"
#include "midi_oss.h"
#include "midi_dummy.h"



midiDevice::midiDevice( channelTrack * _ct ) :
	m_channelTrack( _ct ),
	m_pitchBendKnob( NULL )
{
	//midiMapper * m = new midiMapper("midi-maps/YamahaPSR400.map");
	for( int i = 0; i < NOTES_PER_OCTAVE * OCTAVES; ++i )
	{
		m_keys[i].m_noteForKey = NULL;
	}
}




midiDevice::~midiDevice()
{
	noteOffAll();
}




void midiDevice::processInEvent( const midiEvent & _me )
{
	switch( _me.m_type )
	{
		case NOTE_ON: 
			if( _me.m_data.m_param[1] > 0 )
			{
				noteOn( _me.m_data.m_param[0],
						_me.m_data.m_param[1] * 100 /
									128 );
			}
			else
			{
				noteOff( _me.m_data.m_param[0] );
			}
			break;

		case NOTE_OFF:
			noteOff( _me.m_data.m_param[0] );
			break;

		case PITCH_BEND:
			if( m_pitchBendKnob != NULL )
			{
				float range = tAbs(
						m_pitchBendKnob->maxValue() -
						m_pitchBendKnob->minValue() );
				m_pitchBendKnob->setValue(
						m_pitchBendKnob->minValue() +
						_me.m_data.m_param[0] *
								range / 16384 );
			}
			break;

		default:
			break;
	}
}




void midiDevice::processOutEvent( const midiEvent & _me )
{
	switch( _me.m_type )
	{
		case NOTE_ON:
		case NOTE_OFF:
			sendByte( _me.m_type | _me.m_channel );
//			sendByte( _me.m_channel );
			sendByte( _me.m_data.m_param[0] + NOTES_PER_OCTAVE );
			sendByte( tLimit( (int) _me.m_data.m_param[1], 0,
									127 ) );
			break;

		default:
			break;
	}
}




const midiEvent * midiDevice::parseData( Uint8 _c )
{
	/*********************************************************************/
	/* 'Process' system real-time messages                               */
	/*********************************************************************/
	/* There are not too many real-time messages that are of interest here.
	 * They can occur anywhere, even in the middle of a noteon message! 
	 * Real-time range: 0xF8 .. 0xFF
	 * Note: Real-time does not affect (running) status.
	 */  
	if( _c >= 0xF8 )
	{
		if( _c == MIDI_SYSTEM_RESET )
		{
			m_midiParseData.m_midiEvent.m_type = MIDI_SYSTEM_RESET;
			m_midiParseData.m_status = 0;
			return( &m_midiParseData.m_midiEvent );
		}
		return( NULL );
	}

	/*********************************************************************/
	/* 'Process' system common messages (again, just skip them)          */
	/*********************************************************************/
	/* There are no system common messages that are of interest here.
	 * System common range: 0xF0 .. 0xF7 
	 */
	if( _c > 0xF0 )
	{
	/* MIDI spec say: To ignore a non-real-time message, just discard all
	 * data up to the next status byte.  And our parser will ignore data
	 * that is received without a valid status.  
	 * Note: system common cancels running status. */
		m_midiParseData.m_status = 0;
		return( NULL );
	}

	/*********************************************************************/
	/* Process voice category messages:                                  */
	/*********************************************************************/
	/* Now that we have handled realtime and system common messages, only
	 * voice messages are left.
	 * Only a status byte has bit # 7 set.
	 * So no matter the status of the parser (in case we have lost sync),
	 * as soon as a byte >= 0x80 comes in, we are dealing with a status byte
	 * and start a new event.
	 */
	if( _c & 0x80 )
	{
		m_midiParseData.m_channel = _c & 0x0F;
		m_midiParseData.m_status = _c & 0xF0;
		/* The event consumes x bytes of data...
					(subtract 1 for the status byte) */
		m_midiParseData.m_bytesTotal = eventLength(
						m_midiParseData.m_status ) - 1;
		/* of which we have read 0 at this time. */
		m_midiParseData.m_bytes = 0;
		return( NULL );
	}

	/*********************************************************************/
	/* Process data                                                      */
	/*********************************************************************/
	/* If we made it this far, then the received char belongs to the data
	 * of the last event. */
	if( m_midiParseData.m_status == 0 )
	{
		/* We are not interested in the event currently received.
							Discard the data. */
		return( NULL );
	}

	/* Store the first couple of bytes */
	if( m_midiParseData.m_bytes < MIDI_PARSE_BUF_SIZE )
	{
		m_midiParseData.m_buffer[m_midiParseData.m_bytes] = _c;
	}
	++m_midiParseData.m_bytes;

	/* Do we still need more data to get this event complete? */
	if( m_midiParseData.m_bytes < m_midiParseData.m_bytesTotal )
	{
		return( NULL );
	}

	/*********************************************************************/
	/* Send the event                                                    */
	/*********************************************************************/
	/* The event is ready-to-go.  About 'running status': 
	 * 
	 * The MIDI protocol has a built-in compression mechanism. If several
	 * similar events are sent in-a-row, for example note-ons, then the
	 * event type is only sent once. For this case, the last event type
	 * (status) is remembered.
	 * We simply keep the status as it is, just reset the parameter counter.
	 * If another status byte comes in, it will overwrite the status. 
	 */
	m_midiParseData.m_midiEvent.m_type = static_cast<midiEventTypes>(
						m_midiParseData.m_status );
	m_midiParseData.m_midiEvent.m_channel = m_midiParseData.m_channel;
	m_midiParseData.m_bytes = 0; /* Related to running status! */
	switch( m_midiParseData.m_midiEvent.m_type )
	{
		case NOTE_OFF:
		case NOTE_ON:
		case KEY_PRESSURE:
		case CONTROL_CHANGE:
		case PROGRAM_CHANGE:
		case CHANNEL_PRESSURE:
			m_midiParseData.m_midiEvent.m_data.m_param[0] =
				m_midiParseData.m_buffer[0] - NOTES_PER_OCTAVE;
			m_midiParseData.m_midiEvent.m_data.m_param[1] =
						m_midiParseData.m_buffer[1];
			break;

		case PITCH_BEND:
			// Pitch-bend is transmitted with 14-bit precision.
			// Note: '|' does here the same as '+' (no common bits),
			// but might be faster
			m_midiParseData.m_midiEvent.m_data.m_param[0] =
				( ( m_midiParseData.m_buffer[1] * 128 ) |
						m_midiParseData.m_buffer[0] );
			break;

		default: 
			// Unlikely
			return( NULL );
	}

	return( &m_midiParseData.m_midiEvent );
}




void midiDevice::sendByte( Uint8 )
{
}




void midiDevice::noteOn( int _key, volume _vol )
{
	if( m_keys[_key].m_noteForKey == NULL && m_channelTrack != NULL )
	{
		// create temporary note
		note n( 0, 0, static_cast<tones>( _key % NOTES_PER_OCTAVE ),
				static_cast<octaves>( _key / NOTES_PER_OCTAVE ),
									_vol );
		// create note-play-handle for it
		notePlayHandle * nph = new notePlayHandle( m_channelTrack, 0,
								~0, &n );
		mixer::inst()->addPlayHandle( nph );
		m_keys[_key].m_noteForKey = nph;
	}
}




void midiDevice::noteOff( int _key )
{
	if( m_keys[_key].m_noteForKey != NULL )
	{
		notePlayHandle * n = m_keys[_key].m_noteForKey;
		note done_note( midiTime( static_cast<Uint32>(
						n->totalFramesPlayed() * 64 /
					songEditor::inst()->framesPerTact() ) ),
				0, n->tone(), n->octave(),
				n->getVolume(), n->getPanning() );
		n->noteOff();
		m_keys[_key].m_noteForKey = NULL;
		m_channelTrack->noteDone( done_note );
	}
}




void midiDevice::noteOffAll( void )
{
	for( int i = 0; i < NOTES_PER_OCTAVE * OCTAVES; ++i )
	{
		if( m_keys[i].m_noteForKey != NULL )
		{
			noteOff( i );
		}
	}
}




// Taken from Nagano Daisuke's USB-MIDI driver
Uint16 REMAINS_F0F6[] =
{
	0,	/* 0xF0 */
	2,	/* 0XF1 */
	3,	/* 0XF2 */
	2,	/* 0XF3 */
	2,	/* 0XF4 (Undefined by MIDI Spec, and subject to change) */
	2,	/* 0XF5 (Undefined by MIDI Spec, and subject to change) */
	1	/* 0XF6 */
} ;

Uint16 REMAINS_80E0[] =
{
	3,	/* 0x8X Note Off */
	3,	/* 0x9X Note On */
	3,	/* 0xAX Poly-key pressure */
	3,	/* 0xBX Control Change */
	2,	/* 0xCX Program Change */
	2,	/* 0xDX Channel pressure */
	3 	/* 0xEX PitchBend Change */
} ;



// Returns the length of the MIDI message starting with _event.
// Taken from Nagano Daisuke's USB-MIDI driver
Uint16 midiDevice::eventLength( Uint8 _event )
{
	if ( _event < 0xF0 )
	{
		return( REMAINS_80E0[( ( _event-0x80 ) >>4 ) & 0x0F] );
	}
	else if ( _event < 0xF7 )
	{
		return( REMAINS_F0F6[_event-0xF0] );
	}
	return( 1 );
}


