/*b
 * Copyright (C) 2001,2002  Rick Richardson
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 * Author: Rick Richardson <rickr@mn.rr.com>
b*/

/*
 * swissquote.com
 *
 * line oriented ASCII using stdio buffering.  Uses linuxtrade.sq helper
 * script to change symbols (which requires a new connection).
 */
#include <stdio.h>
#include <stdlib.h>
#include <ncurses.h>
#include <panel.h>
#include <errno.h>
#include <string.h>
#include <time.h>
#include <sys/time.h>
#include <unistd.h>
#include <sys/socket.h>
#include <netdb.h>
#include <netinet/in.h>
#include "error.h"
#include "debug.h"
#include "rc.h"
#include "streamer.h"
#include "linuxtrade.h"
#include "util.h"
#include "p2open.h"

#include "optchain.h"
#include "info.h"

#define         SYMCNT          200
#define         SYMBUFLEN       (SYMCNT * (SYMLEN+1) + 1)

typedef struct streamerpriv
{
	FILE	*fp[2];

	time_t	last_flush;

	char	symbuf[SYMBUFLEN];
	int	symcnt;

	int	delayed;
} STREAMERDATA;

typedef struct
{
	char	*canon, *sym;
} SYMMAP;

static SYMMAP SymMap[] =
{
	{	"$DJI",		".DJIA,U",	},
	{	"$DJT",		".DJTA,U",	},
	{	"$DJU",		".DJUA,U",	},
	{	"$NYA",		".NYA,U",	},
	{	"$COMP",	"COMP,U",	},
	{	"$NDX",		"NDX,U",	},
	{	"$SPX",		".SPX",		},
	{	"$OEX",		".OEX",		},
	{	"$MID",		".MID",		},
	{	"$SML",		".SML",		},
	{	"$XAX",		".XAX,U",	},
	{	"$IIX",		".IIX",		},
	{	"$RUI",		".RUI",		},
	{	"$RUT",		".RUT",		},
	{	"$RUA",		".RUA",		},
	{	"$SOX",		".SOX",		},
	{	"$XAU",		".XAU",		},
	{	"$GOX",		".GOX",		},
	{	NULL,		NULL		}
};

//
// Convert canonical index names to/from streamer index names
//
static void
streamer_canon2sym(char *out, char *in)
{
	char	*ip, *op;
	char	*p;
	char	*cp;
	int	len;
	SYMMAP	*map;

	ip = in;
	op = out;
	for (;;)
	{
		p = strchr(ip, '|');
		if (!p) p = strchr(ip, ' ');
		if (!p) p = strchr(ip, 0);

		len = p - ip;
		memcpy(op, ip, len); op[len] = 0;

		// Convert market designator
		cp = strchr(op, ':');
		if (cp) *cp = ',';

		for (map = SymMap; map->canon; ++map)
			if (strcmp(op, map->canon) == 0)
			{
				strcpy(op, map->sym);
				break;
			}

		if (*p == 0)
			break;

		ip += len + 1;
		op = strchr(op, 0);
		*op++ = *p;
		*op = 0;
	}
}

static void
streamer_sym2canon(char *out, char *in)
{
	SYMMAP	*map;
	char	*cp;

	for (map = SymMap; map->sym; ++map)
		if (strcmp(in, map->sym) == 0)
		{
			strcpy(out, map->canon);
			return;
		}

	if (in != out)
		strcpy(out, in);
	cp = strchr(out, ',');
	if (cp) *cp = ':';
}

static void
streamer_init(STREAMER sr)
{
	sr->refresh = 0;
	sr->fd[0] = -1;
	sr->nfd = 1;
	strcpy(sr->id, "swissquote");

	time(&sr->priv->last_flush);

	sr->priv->fp[0] = NULL;
	sr->priv->fp[1] = NULL;

	sr->priv->symcnt = 0;
	sr->priv->delayed = 1;

	putenv("TZ=Europe/Zurich");
}

static int 
streamer_open( STREAMER sr, RCFILE *rcp, FILE *readfile)
{
	int	rc;

	streamer_init(sr);
	++sr->cnt_opens;
	++sr->cnt_realopens;
	time(&sr->time_open);
	time(&sr->time_realopen);

	rc = p2open("/bin/bash", PROGNAMESTR ".sq 2>/dev/null",
			sr->priv->fp);
	if (rc < 0)
		return -1;

	fprintf(sr->priv->fp[1], "%s\n", get_rc_value(rcp, "username"));
	fprintf(sr->priv->fp[1], "%s\n", get_rc_value(rcp, "password"));
	fprintf(sr->priv->fp[1], "%s\n", get_rc_value(rcp, "hostname"));
	fprintf(sr->priv->fp[1], "%s\n", get_rc_value(rcp, "port"));
	fprintf(sr->priv->fp[1], "GE,\n");
	fflush(sr->priv->fp[1]);

	sr->fd[0] = fileno(sr->priv->fp[0]);
	debug(5, "fd=%d\n", sr->fd[0]);

	if (strncmp(get_rc_value(rcp, "hostname"), "private", 7) == 0)
		sr->priv->delayed = 0;
	return 0;
}
static int
streamer_select(
		STREAMER sr,
		int n, fd_set *readfds, fd_set *writefds,
		fd_set *exceptfds, struct timeval *timeout
		)
{
	if (readfds && FD_ISSET(sr->fd[0], readfds) && FRcnt(sr->priv->fp[0]))
	{
		FD_ZERO(readfds);
		FD_SET(sr->fd[0], readfds);
		if (writefds)
			FD_ZERO(writefds);
		if (exceptfds)
			FD_ZERO(exceptfds);
		return 1;
	}
	return select(n, readfds, writefds, exceptfds, timeout);
}

static void
streamer_close(STREAMER sr)
{
	p2close(sr->priv->fp);
	sr->fd[0] = -1;
}

static void
streamer_record(STREAMER sr, FILE *fp)
{
	sr->writefile = fp;
}

static void
streamer_timetick(STREAMER sr, time_t now)
{
	if (now > sr->priv->last_flush + 5)
	{
		if (sr->writefile)
			fflush(sr->writefile);
	}
}

static void
send_stocklist(STREAMER sr)
{
	char	buf[SYMBUFLEN];
	char	*p;
	int	i;

	// Walk list of all stocks and write symbols to streamer process
	p = buf;
	for (i = 0; i < NumStock; ++i)
	{
		char sym[SYMLEN+1];

		streamer_canon2sym(sym, Stock[i].sym);
		p += sprintf(p, "%s|", sym);
	}

	fprintf(sr->priv->fp[1], "%s\n", buf);
	fflush(sr->priv->fp[1]);

	if (Debug >= 5)
	{
		timestamp(stderr);
		fprintf(stderr, "> %s\n", buf);
	}
}

static void
streamer_send_quickquote(STREAMER sr, char *sym)
{
	char	csym[SYMLEN+1];

	if (sr->fd[0] < 0 || sr->readfile)
		return;

	streamer_canon2sym(csym, sym);
	fprintf(sr->priv->fp[1], "%s|\n", csym);
	fflush(sr->priv->fp[1]);

	if (Debug >= 5)
	{
		timestamp(stderr);
		fprintf(stderr, "> %s|\n", csym);
	}

	send_stocklist(sr);
}

static void
streamer_send_livequote(STREAMER sr, char *sym)
{
	if (sr->fd[0] < 0 || sr->readfile)
		return;

	if (sr->priv->symcnt >= SYMCNT)
		return;

	if (sr->priv->symcnt == 0)
		strcpy(sr->priv->symbuf, "");

	strcat(sr->priv->symbuf, sym);
	strcat(sr->priv->symbuf, "|");
	++sr->priv->symcnt;
}

static void
streamer_send_livequote_end(STREAMER sr)
{
	if (!sr->priv->symcnt)
		return;

	sr->priv->symcnt = 0;

	fprintf(sr->priv->fp[1], "%s\n", sr->priv->symbuf);
	fflush(sr->priv->fp[1]);

	if (Debug >= 5)
	{
		timestamp(stderr);
		fprintf(stderr, "> %s\n", sr->priv->symbuf);
	}

	send_stocklist(sr);
}

static void
streamer_send_symbols(STREAMER sr, char *symbols, int add) {}

static void
streamer_send_symbols_end(STREAMER sr, int add, int all)
{
	if (sr->fd[0] < 0 || sr->readfile)
		return;

	// Its very expensive to change symbols, so ignore deletes
	if (!add)
		return;

	// Send all the symbols in the current stocklist
	send_stocklist(sr);
}

static int
adv(char *p, char sep)
{
	char	*b = p;

	while (*p && *p++ != sep)
		{}
	return (p - b);
}

static void
do_fullquote(STREAMER sr, unsigned char *buf)
{
	char		*p;
	int		rc;
	int		len;
	QUOTE		q;
	QUICKQUOTE	qq;
	LIVEQUOTE	lq;
	int		hh, mm, ss;

	// SUNW|SUN MICROSYS INC|12.110|-.420|-3.46|12.150|12.290|11.950|
	// 12.530|0|12.230|500|12.260|200|34|60513100|35.125|7.520|16:0|
	// NSD|realtime|0|0|N/A
	
	// AT-N|ATEL N|860|830|850|4|860|215|860|850|25|134046
	// ATLAN|ATLAN SAR C 12/02|.08|.08|.07|50000|.09|50000|0|0|0|121616
	// SYM | NAME | LAST | PCLOSE | BID | BSIZE | ASK | ASIZE
	// | HI | LO | VOL | HHMMSS

	memset(&q, 0, sizeof(q));
	memset(&qq, 0, sizeof(qq));

	p = buf;
	len = adv(p, '|');
	if (len > SYMLEN+1)
		return;
	strncpy(q.sym, p, len-1); q.sym[len-1] = 0;

	streamer_sym2canon(q.sym, q.sym);

	strcpy(qq.sym, q.sym);
	strcpy(lq.sym, q.sym);
	p += len;

	sscanf(p, "%[^|]", qq.fullname);
	len = adv(p, '|'); /* fullname */; p += len;

	len = adv(p, '|'); lq.last = qq.last = q.last = atof(p); p += len;
	len = adv(p, '|'); lq.close=qq.prev_close = q.close = atof(p); p += len;
	len = adv(p, '|'); qq.bid = q.bid = atof(p); p += len;
	len = adv(p, '|'); q.bid_size = atoi(p) / 100; p += len;
	len = adv(p, '|'); qq.ask = q.ask = atof(p); p += len;
	len = adv(p, '|'); q.ask_size = atoi(p) / 100; p += len;
	len = adv(p, '|'); qq.high = q.high = atof(p); p += len;
	len = adv(p, '|'); qq.low = q.low = atof(p); p += len;
	len = adv(p, '|'); qq.volume = q.volume = atoi(p); p += len;
	rc = sscanf(p, "%2d%2d%2d", &hh, &mm, &ss);
		if (rc != 3) return;
		qq.timetrade = q.time = hh * 3600 + mm * 60 + ss;

	q.delayed = sr->priv->delayed;

	qq.bid_id = qq.ask_id = '?';
	qq.last_eps = 12345678;
	qq.cur_eps = 12345678;
	qq.sharesout = 12345678;

	display_quote(&q, 0);
	optchain_quote(&q);

	info_quickquote(&qq);

	display_livequote(&lq);
}

static int
streamer_process(STREAMER sr, int fdindex)
{
	char	buf[2048];
	char	*bp = buf;

	if (!fgets(buf, sizeof(buf), sr->priv->fp[0]))
		return -1;
	sr->cnt_rx += strlen(buf);

	if (sr->writefile)
		fputs(buf, sr->writefile);

	if (Debug >= 5)
	{
		bp = strchr(buf, '\n'); if (bp) *bp = 0;
		timestamp(stderr);
		fprintf(stderr, "< %s\n", buf);
	}

	do_fullquote(sr, buf);

	return 0;
}

STREAMER
swissquote_new(void)
{
	STREAMER	sr;

	sr = (STREAMER) malloc(sizeof(*sr));
	if (!sr)
		return NULL;
	memset(sr, 0, sizeof(*sr));

	sr->open = streamer_open;
	sr->select = streamer_select;
	sr->close = streamer_close;
	sr->record = streamer_record;
	sr->timetick = streamer_timetick;

	sr->send_quickquote = streamer_send_quickquote;
	sr->send_livequote = streamer_send_livequote;
	sr->send_livequote_end = streamer_send_livequote_end;
	sr->send_symbols = streamer_send_symbols;
	sr->send_symbols_end = streamer_send_symbols_end;

	// sr->send_disconnect = streamer_send_disconnect;
	// sr->send_top10 = streamer_send_top10;
	// sr->send_movers = streamer_send_movers;
	// sr->send_info = streamer_send_info;
	// sr->send_optchain = streamer_send_optchain;
	// sr->send_chart = streamer_send_chart;

	sr->process = streamer_process;

	sr->priv = (STREAMERDATA *) malloc(sizeof(*sr->priv));
	if (!sr->priv)
	{
		free(sr);
		return NULL;
	}

	time(&sr->time_start);

	streamer_init(sr);

	return (sr);
}
