/***************************************************************************
                          linphone  - sipomatic.c
This is a test program for linphone. It acts as a sip server and answers to linphone's
call.
                             -------------------
    begin                : ven mar  30
    copyright            : (C) 2001 by Simon MORLAT
    email                : simon.morlat@linphone.org
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

#include <signal.h>
#include "sipomatic.h"


int run_cond=1;

Sipomatic sipomatic;

static char *num2str(int a)
{
	char *p=smalloc(10);
	snprintf(p,10,"%i",a);
	return p;
}


void stop_handler(int signum)
{
	run_cond=0;
}

int invite_accepted_cb(OsipDialog *call,transaction_t *trn, sip_t *msg,void *p2)
{
	/* we should never be here since sipomatic is unable to to make invites*/
	return(0);
}

int bye_cb(OsipDialog *dialog,transaction_t *trn, sip_t *msg,void *p2)
{
	/* get the call */
	Call *call=(Call*)dialog->data;
	sipomatic_lock(&sipomatic);
	if ( sipomatic_check_call(&sipomatic,call))
		call->state=CALL_STATE_FINISHED;
	sipomatic_unlock(&sipomatic);
	return(0);
}

int invite_cb(OsipDialog *dialog,transaction_t *trn, sip_t *sipmsg,void *p2)
{
	/* we have just received an invite, so wait two seconds, then accept it*/
	char *p;
	Call *call;
	
	from_2char(sipmsg->from,&p);
	g_message("Contacted by %s.\n",p);
	sfree(p);
	call=call_new(dialog);
	call->current_trn=trn;
	call->root=&sipomatic;
	sipomatic_lock(&sipomatic);
	sipomatic.calls=g_list_append(sipomatic.calls,call);
	sipomatic_unlock(&sipomatic);
	return(0);
}

gint endoffile_cb(MSFilter *f,gint ev,gpointer arg,gpointer data)
{
	Call*call=(Call*)data;
	call->eof=1;
}

void call_accept(Call *call)
{
	SdpContext *ctx;
	int status;
	OsipDialog *dialog=call->dialog;
	CallParams *callparams;
	PayloadType *payload;
	gchar *hellofile;
	
	ctx=SDP_CONTEXT(osip_dialog_get_body_context(dialog,"application/sdp",0));
	callparams=(CallParams*)BODY_CONTEXT(ctx)->data;
	status=sdp_context_get_negociation_status(ctx);
	if (status!=200){
		g_message("Error during sdp negociation, cannot accept call.\n");
		osip_dialog_respond(dialog,call->current_trn,SDP_CONTEXT(ctx)->negoc_status);
		call->state=CALL_STATE_FINISHED;
		return;
	}
	payload=rtp_profile_get_payload(callparams->profile,callparams->pt);
	if (strcmp(payload->mime_type,"telephone-event")==0){
		/* telephone-event is not enough to accept a call */
		g_message("Cannot accept call with only telephone-event.\n");
		osip_dialog_respond(dialog,call->current_trn,415);
		call->state=CALL_STATE_FINISHED;
		return;
	}
	if (payload->clock_rate==16000){
		hellofile=call->root->file_path16000hz;
	}else hellofile=call->root->file_path8000hz;
	osip_dialog_accept_invite(dialog,call->current_trn);
 	call->audio_stream=audio_stream_start_with_files(callparams->profile,callparams->localport,
				callparams->remaddr,callparams->remoteport,callparams->pt,20,hellofile,"/dev/null",NULL);
	g_timer_reset(call->timer);
	call->state=CALL_STATE_RUNNING;
	call->params=callparams;
	ms_filter_set_notify_func(call->audio_stream->soundread,endoffile_cb,(gpointer)call);
}


int faillure_cb(OsipDialog *call,transaction_t *trn, sip_t *msg,void *p2)
{
	return(0);
}


int payload_is_supported(SdpPayload *payload,RtpProfile *local_profile,RtpProfile *dialog_profile)
{
	int localpt;
	if (payload->a_rtpmap!=NULL){
		localpt=rtp_profile_get_payload_number_from_rtpmap(local_profile,payload->a_rtpmap);
	}else{
		localpt=payload->pt;
		g_warning("payload has no rtpmap.");
	}
	
	if (localpt>=0){

		/* this payload is supported in our local rtp profile, so add it to the dialog rtp
		profile */
		PayloadType *rtppayload;
		rtppayload=rtp_profile_get_payload(local_profile,localpt);
		if (rtppayload==NULL) return 0;
		rtppayload=payload_type_clone(rtppayload);
		rtp_profile_set_payload(dialog_profile,payload->pt,rtppayload);
		/* add to the rtp payload type some other parameters (bandwidth) */
		if (payload->b_as_bandwidth!=0) rtppayload->normal_bitrate=payload->b_as_bandwidth*1000;
		return 1;
	}
	return 0;
}
int accept_audio_offer(SdpHandler *sdph,SdpContext *ctx,SdpPayload *payload)
{
	static int audioport=8000;
	OsipDialog *dialog=BODY_CONTEXT(ctx)->dialog;
	CallParams *callparams;
	int supported;
	if (BODY_CONTEXT(ctx)->data!=NULL){
		callparams=(CallParams *) BODY_CONTEXT(ctx)->data;
	}else{
		callparams=call_params_new();
		BODY_CONTEXT(ctx)->data=(void*)callparams;
	}
	
	/* see if this codec is supported in our local rtp profile*/
	supported=payload_is_supported(payload,&av_profile,callparams->profile);
	if (!supported) {
		g_message("Refusing codec %i (%s)",payload->pt,payload->a_rtpmap);
		return -1;
	}
	if (callparams->ncodecs==0){
		/* this is the first codec we may accept*/
		callparams->localport=payload->localport=audioport;
		callparams->remoteport=payload->remoteport;
		callparams->line=payload->line;
		callparams->pt=payload->pt; /* remember the first payload accepted */
		callparams->remaddr=payload->c_addr;
		callparams->ncodecs++;
		audioport+=4;
	}else{
		/* refuse all other audio lines*/
		if(callparams->line!=payload->line) return -1;
	}
	return 0;
}

void sipomatic_init(Sipomatic *obj, gchar *url)
{
	MSCodecInfo *info;
	GList *elem;
	BodyHandler *sdph;
	if (url==NULL){
		url=getenv("SIPOMATIC_URL");
		if (url==NULL) url="sip:robot@0.0.0.0:5064";
	}
	g_message("Starting using url %s",url);
	obj->audio_codecs=ms_codec_get_all_audio();
	obj->lock=g_mutex_new();
	obj->calls=NULL;
	obj->acceptance_time=5;
	obj->max_call_time=300;
	obj->file_path8000hz=g_strdup_printf("%s/%s",PACKAGE_SOUND_DIR,ANNOUCE_FILE8000HZ);
	obj->file_path16000hz=g_strdup_printf("%s/%s",PACKAGE_SOUND_DIR,ANNOUCE_FILE16000HZ);
	
	/* create a user agent */
	obj->ua=osip_ua_new();
	osip_ua_set_contact(obj->ua,url);
	osip_ua_signal_connect(obj->ua,"INVITE_ACCEPTED",invite_accepted_cb);
  	osip_ua_signal_connect(obj->ua,"BYE",bye_cb);
  	osip_ua_signal_connect(obj->ua,"FAILLURE",faillure_cb);
  	osip_ua_signal_connect(obj->ua,"INVITE",invite_cb);
	/* add sdp capabilities to the user agent */
	sdph=sdp_handler_new();
	sdp_handler_set_write_offer_fcn(SDP_HANDLER(sdph),NULL,NULL);
	sdp_handler_set_accept_offer_fcn(SDP_HANDLER(sdph),accept_audio_offer,NULL);
	sdp_handler_set_read_answer_fcn(SDP_HANDLER(sdph),NULL,NULL);
	
	osip_ua_add_body_handler(obj->ua,sdph);
	
}

void sipomatic_uninit(Sipomatic *obj)
{
	g_mutex_free(obj->lock);
	osip_ua_destroy(obj->ua);
}

void sipomatic_iterate(Sipomatic *obj)
{
	GList *elem;
	Call *call;
	gdouble time;
	sipomatic_lock(obj);
	elem=obj->calls;
	while(elem!=NULL){
		call=(Call*)elem->data;
		time=g_timer_elapsed(call->timer,NULL);
		switch(call->state){
			case CALL_STATE_INIT:
				if (time>obj->acceptance_time){
					call_accept(call);
				}
			break;
			case CALL_STATE_RUNNING:
				if (time>obj->max_call_time || call->eof){
					call_release(call);
					elem=obj->calls=g_list_remove(obj->calls,call);
					call_destroy(call);
				}
			break;
			case CALL_STATE_FINISHED:
				elem=obj->calls=g_list_remove(obj->calls,call);
				call_destroy(call);
			break;
		}
		elem=g_list_next(elem);
	}
	sipomatic_unlock(obj);
}

CallParams * call_params_new(){
	CallParams *obj;
	obj=g_new0(CallParams,1);
	obj->profile=rtp_profile_new("remote");
	return obj;
}
void call_params_destroy(CallParams *obj)
{
	rtp_profile_destroy(obj->profile);
	g_free(obj);
}

gboolean sipomatic_check_call(Sipomatic *obj,Call *call)
{
	GList *it;
	for (it=obj->calls;it!=NULL;it=g_list_next(it)){
		if ( ((Call*)it->data)==call) return 1;
	}
	return 0;
}

Call * call_new(OsipDialog *dialog)
{
	Call *obj=g_new0(Call,1);
	obj->timer=g_timer_new();
	obj->dialog=dialog;
	dialog->data=(void*)obj;
	g_timer_start(obj->timer);
	obj->audio_stream=NULL;
	obj->state=CALL_STATE_INIT;
	obj->eof=0;
	return obj;
}

void call_release(Call *call)
{
	osip_dialog_bye(call->dialog);
	call->state=CALL_STATE_FINISHED;
}

void call_destroy(Call *obj)
{
	if (obj->audio_stream!=NULL) audio_stream_stop(obj->audio_stream);
	g_timer_destroy(obj->timer);
	if (obj->params!=NULL) call_params_destroy(obj->params);
	g_free(obj);
}

void sipomatic_set_annouce_file(Sipomatic *obj, char *file)
{
	if (obj->file_path8000hz!=NULL){
		g_free(obj->file_path8000hz);
	}
	obj->file_path8000hz=g_strdup(file);
}

extern OsipManager *def_manager;

void display_help()
{
	printf("sipomatic [-u sip-url] [-f annouce-file ] [-s port]\n"
			"sipomatic -h or --help: display this help.\n"
			"sipomatic -v or --version: display version information.\n"
			"	-u sip-url : specify the sip url sipomatic listens and answers.\n"
			"	-f annouce-file : set the annouce file (16 bit raw format,8000Hz)\n"
			"	-s port	: set the port sipomatic uses to send its SIP answers.\n");
	exit(0);
}

char *getarg(int argc, char*argv[], int i)
{
	if (i<argc){
		return argv[i];
	}
	else display_help();
}

int main(int argc, char *argv[])
{
	int sendport=5070;
	char *file=NULL;
	gchar *url=NULL;
	int i;
	
	for(i=1;i<argc;i++){
		if ( (strcmp(argv[i],"-h")==0) || (strcmp(argv[i],"--help")==0) ){
			display_help();
			continue;
		}
		if ( (strcmp(argv[i],"-v")==0) || (strcmp(argv[i],"--version")==0) ){
			printf("version: " LINPHONE_VERSION "\n");
			exit(0);
		}
		if (strcmp(argv[i],"-u")==0){
			i++;
			url=getarg(argc,argv,i);
			continue;
		}
		if (strcmp(argv[i],"-s")==0){
			char *port;
			i++;
			port=getarg(argc,argv,i);
			sendport=atoi(port);
			continue;
		}
		if (strcmp(argv[i],"-f")==0){
			i++;
			file=getarg(argc,argv,i);
			continue;
		}
	}
	
	signal(SIGINT,stop_handler);
	ms_init();
	ms_speex_codec_init();
	ortp_init();
	ortp_set_debug_file("oRTP",NULL);
	rtp_profile_set_payload(&av_profile,115,&lpc1015);
	rtp_profile_set_payload(&av_profile,110,&speex_nb);
	rtp_profile_set_payload(&av_profile,111,&speex_wb);
	rtp_profile_set_payload(&av_profile,101,&telephone_event);
 	rtp_profile_set_payload(&av_profile,116,&truespeech);
	TRACE_INITIALIZE(TRACE_LEVEL6,stdout);
	osipua_init();
	osip_manager_set_send_port(def_manager,sendport);
	sipomatic_init(&sipomatic,url);
	if (file!=NULL) sipomatic_set_annouce_file(&sipomatic,file);
	
	while (run_cond){
		sipomatic_iterate(&sipomatic);
		usleep(20000);
	}
	
	return(0);
}
