/*  This file is part of LingoTeach, the Language Teaching program 
 *  Copyright (C) 2001-2003 The LingoTeach Team
 *
 *  This program is free software; you can redistribute it and/or modify 
 *  it under the terms of the GNU General Public License as published by 
 *  the Free Software Foundation; either version 2 of the License, or 
 *  (at your option) any later version. 
 *
 *  This program is distributed in the hope that it will be useful, 
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the 
 *  GNU General Public License for more details. 
 * 
 *  You should have received a copy of the GNU General Public License 
 *  along with this program; if not, write to the Free Software 
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA. 
 */

#include <gmodule.h>
#include <glib.h>
#include <string.h>

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif /* HAVE_CONFIG_H */

#include "lingoteach-i18n.h"
#include "module.h"
#include "errors.h"

/* external global variables */
extern GtkWidget *box_obo_module;
extern GtkWidget *box_mul_module;
extern GtkWidget *box_sab_module;

/* globally needed stuff */
enum
  {
    ONEBYONE,
    MULTICHOICE,
    SEARCH
  };

/* module struct */
typedef struct 
{
  gint       (*type) ();        /* pointer to the module function mod_type */
  gchar*     (*name) ();        /* pointer to the module function mod_name */
  gchar*     (*description) (); /* pointer to the module function mod_desc */
  GtkWidget* (*box) ();         /* pointer to the module function mod_box  */
  GtkWidget* (*info) ();        /* pointer to the module function mod_info  */
  gpointer*  (*free) ();        /* pointer to the module function mod_free */ 
  GModule*   module;            /* the module itself */
  gchar*     path;              /* the path of the module */
} lingMod;

GSList* plugins = NULL;  /* module list */


/* pototypes */
static gint find_module (lingMod *module, gchar *name);
static void reset_box (lingMod *module, gboolean fill);


/*********************
 * private functions *
 *********************/

/*
 * searches a module in the list
 */
static gint
find_module (lingMod *module, gchar *name)
{
  if (g_strncasecmp (module->name (), name, strlen (name)) == 0)
    return 0;
  return -1;
}

static void 
reset_box (lingMod *module, gboolean fill)
{
  GList     *module_data;
  GtkWidget *box = NULL;

  switch (module->type ())
    {
    case SEARCH:
      box = box_sab_module;
      break;
    case ONEBYONE:
      box = box_obo_module;
      break;
    case MULTICHOICE:
      box = box_mul_module;
      break;
    default:
      break;
    }
  
  module_data = gtk_container_get_children (GTK_CONTAINER (box));
  if (module_data != NULL)
    {
      gtk_widget_destroy (module_data->data);
      g_list_free (module_data);
    }
  if (fill == TRUE && box != NULL)
    gtk_box_pack_start (GTK_BOX (box), module->box (), 
			FALSE, FALSE, 0);
  return;
}



/********************
 * public functions *
 ********************/

/*
 * load a dynamic module
 */
void
module_load (GtkListStore *list, char *filename)
{
  lingMod     *module;
  gint        (*mod_type) ();
  gchar*      (*mod_name) ();
  gchar*      (*mod_desc) ();
  GtkWidget*  (*mod_box)  ();
  GtkWidget*  (*mod_info) ();
  gpointer*   (*mod_free) ();
  gchar       *name;
  GtkTreeIter module_iter;  

  /* allocate memory for module */
  module = g_new (lingMod, 1);
  if (module == NULL)
    {
      err_error_cb (_("Could not allocate enough memory for module!"));
      return;
    }
  
  /* open module */
  if ((module->module = g_module_open (filename, 0)) == NULL)
    {
      err_error_cb (_("Could not open module!"));
      g_free (module);
      return;
    }

  /* get the necessary module functions */
  if (!g_module_symbol (module->module, "module_name", (gpointer *) &mod_name)
      || !g_module_symbol (module->module, "module_desc", (gpointer *) &mod_desc)
      || !g_module_symbol (module->module, "module_box", (gpointer *) &mod_box)
      || !g_module_symbol (module->module, "module_free", (gpointer *) &mod_free)
      || !g_module_symbol (module->module, "module_type", (gpointer *) &mod_type)
      || !g_module_symbol (module->module, "module_info", (gpointer *) &mod_info))
    {
      err_error_cb (_("Could not find all methods in module!"));
      g_free (module);
      return;
    }
 
  /* set the name and description */
  module->type        = mod_type;
  module->name        = mod_name;
  module->description = mod_desc;
  module->box         = mod_box;
  module->info        = mod_info;
  module->free        = mod_free;

  /* check, if it is already loaded */
  name = module->name ();
  if (g_slist_find_custom (plugins, name, (GCompareFunc) find_module) != NULL)
    {
      err_error_cb (_("This module is already loaded."));
      g_module_close (module->module);
      g_free (module);
      return;
    }

  /* add module to the internal list */
  plugins = g_slist_append (plugins, module);

  /* add the module to the tree list */
  gtk_list_store_insert_before (list, &module_iter, NULL);
  gtk_list_store_set (list, 
		      &module_iter,
		      0, module->name (),
		      -1);

  /* save file path */
  module->path = malloc (strlen (filename));
  strcpy (module->path, filename);

  return;
}

/*
 * unloads (removes) a dynamic module
 */
lingbool 
module_unload (gchar *name)
{
  lingMod *module;
  GSList  *unload = g_slist_find_custom (plugins, name, 
					 (GCompareFunc) find_module);
  if (unload == NULL)
    return FALSE;

  module = unload->data;

  reset_box (module, FALSE);
  module->free ();                 /* call the freeing method in the module */
  free (module->path);
  g_module_close (module->module);
  plugins = g_slist_remove_link (plugins, unload);
  g_slist_free_1 (unload);
  return TRUE;
}

/*
 * gets the gtk stuff from the module and applies it to the box
 */
GtkWidget*
module_get_info (gchar *name)
{
  GSList *active = g_slist_find_custom (plugins, name, 
					(GCompareFunc) find_module);
  lingMod *module = active->data;

  reset_box (module, TRUE);
  return module->info ();
}

void
module_get_list (GtkListStore *list)
{
  GSList      *p = plugins;
  lingMod     *module;
  GtkTreeIter module_iter;

  while (p != NULL)
    {
      module = p->data;
      
      /* add the module to the tree list */
      gtk_list_store_insert_before (list, &module_iter, NULL);
      gtk_list_store_set (list, 
			  &module_iter,
			  0, module->name (),
			  -1);
      p = g_slist_next (p);
    }
  return;
}

/* 
 * return the path of a module with the given name 
 */
gchar*
module_get_path (gchar *name)
{
  GSList *active = g_slist_find_custom (plugins, name, 
					(GCompareFunc) find_module);
  lingMod *module = active->data;
  return module->path;
}
