/* $Id: gtklearn.c,v 1.21 2005/04/20 07:58:03 marcusva Exp $
 *
 *  This file is part of LingoTeach, the Language Teaching program
 *  Copyright (C) 2004-2005 Marcus von Appen. All rights reserved.
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include "../li18n.h"
#include "../errors.h"
#include "hig-widgets.h"
#include "gtkdefs.h"

#include "gtkconfig.h"
#include "gtksession.h"
#include "gtkprop.h"
#include "menu.h"
#include "util.h"
#include "gtklearn-wiz.h"
#include "gtklearn.h"

#define MEANING_CAPTION "<span weight=\"bold\">%s - %s</span>"
#define BOX_DETAILS "box_details"
#define IMAGE_NO "image-no"

enum
{
     LEARN_LANG,
     LEARN_TRANS,
     LEARN_PHONETIC,
     LEARN_COLUMNS
};

extern lingGtkMainWindow *main_app;
extern lingGtkPrefs *settings;

struct lingGtkLearnPage
{
     lingchar **img_paths;

     GtkWidget *frm_details;
     GtkWidget *lbl_description;

     GtkWidget *lbl_or_language;
     GtkWidget *lbl_original;
     GtkWidget *btn_next;
     GtkWidget *btn_prev;
     GtkWidget *tree_trans;

     lingchar  *main_language;
};

static struct lingGtkLearnPage page;

static GtkWidget* create_details (void);
static void create_next_entry (gpointer step);
static void show_image (void);
static void abort_session (void);
static void play_sound (GtkTreeView *view, GtkTreePath *path,
                        GtkTreeViewColumn *col, gpointer data);
static void load_prev_image (GtkWidget *button, GtkWidget *image);
static void load_next_image (GtkWidget *button, GtkWidget *image);

static GtkWidget*
create_details (void)
{
     GtkWidget *frm_details;
     GtkWidget *align_details;
     GtkWidget *box_details;
     GtkWidget *align_image;
     GtkWidget *btn_image;

     /* optional details */
     frm_details = hig_frame_new (_("Details"));
     gtk_container_set_border_width (GTK_CONTAINER (frm_details), 0);
     align_details = gtk_bin_get_child (GTK_BIN (frm_details));

     /* the box to hold the description and image */
     box_details = hig_hbox_new ();
     gtk_container_set_border_width (GTK_CONTAINER (box_details), 0);
     gtk_container_add (GTK_CONTAINER (align_details), box_details);
     
     page.lbl_description = gtk_label_new (NULL);
     gtk_label_set_line_wrap (GTK_LABEL (page.lbl_description), TRUE);
     gtk_misc_set_alignment (GTK_MISC (page.lbl_description), 0.0, 0.0);
     gtk_box_pack_start (GTK_BOX (box_details), page.lbl_description,
                         TRUE, TRUE, 0);
     
     align_image = gtk_alignment_new (0.5, 0, 1, 0);
     gtk_box_pack_start (GTK_BOX (box_details), align_image, FALSE, FALSE, 0);
     btn_image = gtk_button_new_with_mnemonic (_("_Image"));
     gtk_container_add (GTK_CONTAINER (align_image), btn_image);
     g_signal_connect (G_OBJECT (btn_image), "clicked",
                       G_CALLBACK (show_image), NULL);

     return frm_details;
}

static void
create_next_entry (gpointer step)
{
     lingMeaning *meaning_orig = NULL;
     lingMeaning *meaning = NULL;
     learnEntry *entry = NULL;
     gchar *text = NULL;
     learnLanguage *tmp = NULL;
     lingchar *language = NULL;
     lingchar *description = NULL;
     lingchar *phonetic = NULL;
     GtkListStore *list;
     GtkTreeIter iter;

     if (GPOINTER_TO_INT (step) == 0)
          entry = settings->learn_session->cur_entry;
     else if (GPOINTER_TO_INT (step) > 0)
          entry = settings->learn_session->cur_entry->next;
     else
          entry = settings->learn_session->cur_entry->prev;

     if (!entry->next)
          gtk_widget_set_sensitive (page.btn_next, FALSE);
     else if (!GTK_WIDGET_IS_SENSITIVE (page.btn_next))
          gtk_widget_set_sensitive (page.btn_next, TRUE);
     
     if (!entry->prev)
          gtk_widget_set_sensitive (page.btn_prev, FALSE);
     else if (!GTK_WIDGET_IS_SENSITIVE (page.btn_prev))
          gtk_widget_set_sensitive (page.btn_prev, TRUE);

     meaning_orig = ling_meaning_get_by_id (entry->lesson->lesson, entry->id,
                                            page.main_language);
     text = g_strdup_printf (MARKUP_ITALIC, meaning_orig->translation);
     gtk_label_set_markup (GTK_LABEL (page.lbl_original), text);
     g_free (text);

     /* create translations */
     list = GTK_LIST_STORE (gtk_tree_view_get_model 
                            (GTK_TREE_VIEW (page.tree_trans)));
     gtk_list_store_clear (list);
     
     tmp = settings->learn_session->language;
     while (tmp)
     {
          if (tmp->used)
          {
               gtk_list_store_append (list, &iter);
               language = 
                    ling_lang_get_lang_translation (settings->prefs->config,
                                                    tmp->language,
                                                    page.main_language);

               meaning = ling_meaning_get_by_id (entry->lesson->lesson,
                                                 entry->id, tmp->language);
               phonetic = ling_meaning_get_phonetic (meaning);

               gtk_list_store_set (list, &iter,
                                   LEARN_LANG,
                                   (language) ? language : tmp->language,
                                   LEARN_TRANS, meaning->translation,
                                   LEARN_PHONETIC, phonetic,
                                   -1);
               if (language)
                    ling_free (language);
               if (phonetic)
                    ling_free (phonetic);
               ling_meaning_free (meaning); /* info will be freed here, too */
          }
          tmp = tmp->next;
     }

     /* create hint stuff */
     if (settings->learn_session->hints)
     {
          description = ling_meaning_get_description (meaning_orig);
          gtk_label_set_text (GTK_LABEL (page.lbl_description), description);
          ling_free (description);

          if (page.img_paths) /* copy new image path */
               ling_strings_free (page.img_paths);
          
          /* add image path here! */
          page.img_paths = ling_meaning_get_images (meaning_orig);
     }
     ling_meaning_free (meaning_orig);
     settings->learn_session->cur_entry = entry;
     return;
}

static void
show_image (void)
{
     GtkWidget *dlg_image;
     GtkWidget *scr_win;
     GtkWidget *box_main;
     GtkWidget *box_btns;
     GtkWidget *box_nextprev;
     GtkWidget *btn_next;
     GtkWidget *btn_prev;
     GtkWidget *btn_close;
     GtkWidget *image;
    
     /* dialog window */
     dlg_image = hig_window_new (_("Image"));
     gtk_window_set_transient_for (GTK_WINDOW (dlg_image),
                                   GTK_WINDOW (main_app->window));
     gtk_window_set_position (GTK_WINDOW (dlg_image),
                              GTK_WIN_POS_CENTER_ON_PARENT);
     gtk_window_set_modal (GTK_WINDOW (dlg_image), TRUE);

     box_main = hig_vbox_new ();
     gtk_container_set_border_width (GTK_CONTAINER (box_main), 0);
     gtk_container_add (GTK_CONTAINER (dlg_image), box_main);

     /* image window */
     scr_win = hig_scrolled_window_new ();

     if (page.img_paths)
     {
          image = gtk_image_new_from_file (page.img_paths[0]);
          g_object_set_data (G_OBJECT (image), IMAGE_NO, GINT_TO_POINTER (0));
     }
     else
          image = gtk_image_new_from_stock (GTK_STOCK_MISSING_IMAGE,
                                            GTK_ICON_SIZE_DIALOG);
     gtk_scrolled_window_add_with_viewport (GTK_SCROLLED_WINDOW (scr_win),
                                            image);
     gtk_box_pack_start (GTK_BOX (box_main), scr_win, TRUE, TRUE, 0);

     /* buttons */
     box_btns = hig_hbox_new ();
     gtk_box_set_spacing (GTK_BOX (box_btns), 2 * BOX_SPACING);
     gtk_box_pack_start (GTK_BOX (box_main), box_btns, FALSE, TRUE, 0);

     /* button box */
     box_nextprev = gtk_hbutton_box_new ();
     gtk_box_pack_start (GTK_BOX (box_btns), box_nextprev, TRUE, TRUE, 0);
     gtk_button_box_set_layout (GTK_BUTTON_BOX (box_nextprev),
                                GTK_BUTTONBOX_START);
     gtk_box_set_spacing (GTK_BOX (box_nextprev), 2 * BOX_SPACING);

     /* next and prev */
     btn_prev = gtk_button_new_from_stock (GTK_STOCK_GO_BACK);
     g_signal_connect (G_OBJECT (btn_prev), "clicked",
                       G_CALLBACK (load_prev_image), image);
     gtk_container_add (GTK_CONTAINER (box_nextprev), btn_prev);
     gtk_tooltips_set_tip (main_app->tips, btn_prev, _("Previous image"),
                           NULL);

     btn_next = gtk_button_new_from_stock (GTK_STOCK_GO_FORWARD);
     g_signal_connect (G_OBJECT (btn_next), "clicked",
                       G_CALLBACK (load_next_image), image);
     gtk_container_add (GTK_CONTAINER (box_nextprev), btn_next);
     gtk_tooltips_set_tip (main_app->tips, btn_next,
                           _("Next image"), NULL);
     if (!page.img_paths)
     {
          gtk_widget_set_sensitive (btn_prev, FALSE);
          gtk_widget_set_sensitive (btn_next, FALSE);
     }

     /* button close */
     btn_close = gtk_button_new_from_stock (GTK_STOCK_CLOSE);
     gtk_box_pack_start (GTK_BOX (box_btns), btn_close, FALSE, FALSE, 0);
     g_signal_connect_swapped (G_OBJECT (btn_close), "clicked",
                               G_CALLBACK (gtk_widget_destroy), dlg_image);

     gtk_widget_show_all (dlg_image);
     return;
}

static void
abort_session (void)
{
     lingbool ok = FALSE;
     
     ok = util_ask_for_confirmation (_("Confirm abort"),
                                     _("Do you really want to abort the "
                                       "current learning session? "
                                       "Your session will be saved "
                                       "automatically, if you confirm."));
     if (ok)
     {
          gtksession_save_session ();
          learn_session_free (settings->learn_session);
          settings->learn_session = NULL;
          gtk_notebook_set_page (GTK_NOTEBOOK (main_app->notebook),
                                 PAGE_INIT);
          menu_set_active (PAGE_INIT);
     }
     return;
}

static void
play_sound (GtkTreeView *view, GtkTreePath *path, GtkTreeViewColumn *col,
            gpointer data)
{
     GtkTreeIter iter;
     GtkTreeModel *model = gtk_tree_view_get_model (view);
     lingchar *language = NULL;
     lingchar *orig = NULL;
     lingMeaning *meaning = NULL;
     lingchar *sound = NULL;
     lingchar *file = NULL;
     
     if (gtk_tree_model_get_iter (model, &iter, path))
     {
          gtk_tree_model_get (model, &iter, LEARN_LANG, &language, -1);
          orig = ling_lang_get_trans_language (settings->prefs->config,
                                               language, page.main_language);
          if (!orig || g_utf8_strlen (orig, -1) == 0)
          {
               ling_free (orig);
               orig = NULL;
          }
          meaning = ling_meaning_get_by_id
               (settings->learn_session->cur_entry->lesson->lesson,
                settings->learn_session->cur_entry->id,
                (orig != NULL) ? orig : language);
          
          sound = ling_meaning_get_sound (meaning);
          
          file = g_build_filename (settings->prefs->soundpath, sound, NULL);
          util_play_sound (file);
          g_free (language);

          if (orig)
               ling_free (orig);
          ling_free (sound);
          ling_meaning_free (meaning);
     }
     return;
}

static void
load_prev_image (GtkWidget *button, GtkWidget *image)
{
     int no = GPOINTER_TO_INT (g_object_get_data (G_OBJECT (image), IMAGE_NO));
     
     if (no <= 0)
          return;
     no--;

     gtk_image_set_from_file (GTK_IMAGE (image),
                              (const gchar *) page.img_paths[no]);
     g_object_set_data (G_OBJECT (image), IMAGE_NO, GINT_TO_POINTER (no));
     return;
}

static void
load_next_image (GtkWidget *button, GtkWidget *image)
{
     int no = GPOINTER_TO_INT (g_object_get_data (G_OBJECT (image), IMAGE_NO));
    
     no++;
     if (page.img_paths[no] == NULL)
          return;

     debug ("aha2: %i\n", no);

     gtk_image_set_from_file (GTK_IMAGE (image),
                              (const gchar *) page.img_paths[no]);
     g_object_set_data (G_OBJECT (image), IMAGE_NO, GINT_TO_POINTER (no));
     return;
}

GtkWidget*
gtklearn_create_learn_page (void)
{
     GtkWidget *box_main;
     GtkWidget *lbl_learn;
     GtkWidget *frm_meaning;
     GtkWidget *align_meaning;
     GtkWidget *box_meaning;

     GtkWidget *frm_translation;
     GtkWidget *align_translation;
     GtkWidget *scr_win_translation;
     GtkListStore *list;
     GtkCellRenderer *renderer;
     gint pos = 0;

     GtkWidget *box_buttons;
     GtkWidget *btn_cancel;
     gchar *text = NULL;

     box_main = hig_vbox_new ();
     gtk_box_set_spacing (GTK_BOX (box_main), 2 * BOX_SPACING);
     gtk_container_set_border_width (GTK_CONTAINER (box_main),
                                     2 * WIDGET_BORDER);
     page.img_paths = NULL;
     page.main_language = NULL;

     /* label */
     text = g_strdup_printf (MARKUP_LARGER_BOLD, _("Learn"));
     lbl_learn = gtk_label_new (NULL);
     gtk_label_set_markup (GTK_LABEL (lbl_learn), text);
     gtk_misc_set_alignment (GTK_MISC (lbl_learn), 1.0, 0.5);
     gtk_box_pack_start (GTK_BOX (box_main), lbl_learn, FALSE, FALSE, 0);
     g_free (text);
     
     /* the meaning / description frame */
     frm_meaning = hig_frame_new ("");
     gtk_container_set_border_width (GTK_CONTAINER (frm_meaning), 0);
     align_meaning = gtk_bin_get_child (GTK_BIN (frm_meaning));
     page.lbl_or_language =
          gtk_frame_get_label_widget (GTK_FRAME (frm_meaning));
     gtk_box_pack_start (GTK_BOX (box_main), frm_meaning, FALSE, TRUE, 0);

     /* the box to hold label and stuff */
     box_meaning = hig_hbox_new ();
     gtk_container_set_border_width (GTK_CONTAINER (box_meaning), 0);
     gtk_container_add (GTK_CONTAINER (align_meaning), box_meaning);

     /* translation label */
     page.lbl_original = gtk_label_new (NULL);
     gtk_box_pack_start (GTK_BOX (box_meaning), page.lbl_original,
                         FALSE, FALSE, 0);

     /* the details */
     page.frm_details = create_details ();
     gtk_box_pack_start (GTK_BOX (box_main), page.frm_details, FALSE, TRUE, 0);

     /* now the translation frame */
     frm_translation = hig_frame_new (_("Translations"));
     gtk_container_set_border_width (GTK_CONTAINER (frm_translation), 0);
     align_translation = gtk_bin_get_child (GTK_BIN (frm_translation));
     gtk_box_pack_start (GTK_BOX (box_main), frm_translation, TRUE, TRUE, 0);

     scr_win_translation = hig_scrolled_window_new ();
     gtk_container_set_border_width (GTK_CONTAINER (scr_win_translation), 0);
     gtk_container_add (GTK_CONTAINER (align_translation),
                        scr_win_translation);
     
     /* the list store */
     list = gtk_list_store_new (LEARN_COLUMNS, G_TYPE_STRING, G_TYPE_STRING,
                                G_TYPE_STRING);

     /* create the treeview */
     page.tree_trans = gtk_tree_view_new_with_model (GTK_TREE_MODEL (list));
     g_object_unref (G_OBJECT (list));
     g_signal_connect (G_OBJECT (page.tree_trans), "row-activated",
                       G_CALLBACK (play_sound), NULL);
     gtk_container_add (GTK_CONTAINER (scr_win_translation), page.tree_trans);

     renderer = gtk_cell_renderer_text_new ();
     gtk_tree_view_insert_column_with_attributes
          (GTK_TREE_VIEW (page.tree_trans), LEARN_LANG, _("Language"),
           renderer, "text", LEARN_LANG, NULL);

     renderer = gtk_cell_renderer_text_new ();
     pos = gtk_tree_view_insert_column_with_attributes
               (GTK_TREE_VIEW (page.tree_trans), LEARN_TRANS, _("Translation"),
                renderer, "text", LEARN_TRANS, NULL);
     /* set the column to occupy all extra space */
     gtk_tree_view_column_set_expand (gtk_tree_view_get_column 
                                      (GTK_TREE_VIEW (page.tree_trans),
                                       pos - 1), TRUE);

     renderer = gtk_cell_renderer_text_new ();
     gtk_tree_view_insert_column_with_attributes
          (GTK_TREE_VIEW (page.tree_trans), LEARN_PHONETIC,
           _("Phonetic transliteration"), renderer, "text",
           LEARN_PHONETIC, NULL);

     /* the button box */
     box_buttons = gtk_hbutton_box_new ();
     gtk_button_box_set_layout (GTK_BUTTON_BOX (box_buttons),
                                GTK_BUTTONBOX_END);
     gtk_box_set_spacing (GTK_BOX (box_buttons), 2 * BOX_SPACING);
     gtk_box_pack_start (GTK_BOX (box_main), box_buttons, FALSE, TRUE, 0);
     
     btn_cancel = gtk_button_new_from_stock (GTK_STOCK_CANCEL);
     gtk_container_add (GTK_CONTAINER (box_buttons), btn_cancel);
     g_signal_connect (G_OBJECT (btn_cancel), "clicked",
                       G_CALLBACK (abort_session), NULL);

     page.btn_prev = UTIL_CREATE_BUTTON_PREV ();
     gtk_container_add (GTK_CONTAINER (box_buttons), page.btn_prev);
     g_signal_connect_swapped (G_OBJECT (page.btn_prev), "clicked",
                               G_CALLBACK (create_next_entry),
                               GINT_TO_POINTER (-1));

     page.btn_next = UTIL_CREATE_BUTTON_NEXT ();
     gtk_container_add (GTK_CONTAINER (box_buttons), page.btn_next);
     g_signal_connect_swapped (G_OBJECT (page.btn_next), "clicked",
                               G_CALLBACK (create_next_entry),
                               GINT_TO_POINTER (1));

     return box_main;
}

void
gtklearn_init_session (void)
{
     GtkWidget *widget;
     static guint id = 0;
     learnLanguage *tmp = NULL;
     lingchar *language = NULL;
     
     gchar *text = NULL;
     page.main_language = NULL; /* reset the main language */

     /* properties entry in menu */
     widget = menu_get_entry (main_app->menu, "LingoTeach/Properties");
     if (id != 0)
          g_signal_handler_disconnect (G_OBJECT (widget), id);
     id = g_signal_connect_swapped (G_OBJECT (widget), "activate",
                                    G_CALLBACK (gtkprop_activate),
                                    settings->learn_session);

     if (!settings->learn_session->hints)
          gtk_widget_hide (page.frm_details);
     else
          gtk_widget_show_all (page.frm_details);

     tmp = settings->learn_session->language;
     while (tmp)
     {
          if (tmp->main)
          {
               page.main_language = tmp->language;
               break;
          }
          tmp = tmp->next;
     }
     
     language = ling_lang_get_lang_translation (settings->prefs->config,
                                                page.main_language,
                                                page.main_language);
     if (!language)
          text = g_strdup_printf (MEANING_CAPTION, _("Meaning"),
                                                     page.main_language);
     else
     {
          text = g_strdup_printf (MEANING_CAPTION, _("Meaning"), language);
          ling_free (language);
     }

     gtk_label_set_markup (GTK_LABEL (page.lbl_or_language), text);
     g_free (text);

     create_next_entry (GINT_TO_POINTER (0));
     return;
}
