/* $Id: gtkedit.c,v 1.27 2005/03/31 15:54:51 marcusva Exp $
 *
 *  This file is part of LingoTeach, the Language Teaching program
 *  Copyright (C) 2004-2005 Marcus von Appen. All rights reserved.
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include <string.h>

#include "../li18n.h"
#include "../errors.h"
#include "../lesson.h"
#include "hig-widgets.h"
#include "lingmeaningtree.h"
#include "gtkdefs.h"

#include "gtkconfig.h"
#include "gtklesson.h"
#include "util.h"
#include "menu.h"
#include "gtkedit-meaning.h"
#include "gtkedit.h"

extern lingGtkMainWindow *main_app;
extern lingGtkPrefs *settings;

static const gchar *labels[LING_TREE_N_COLUMNS - 2] = {
     N_("Id"), N_("Language"), N_("Translation")
};

/* editing stuff */
static void edit_lesson (GtkTreeView *treeview, GtkTreePath *path,
                         GtkTreeViewColumn *col, lingGtkEditor *editor);
static void lesson_name_edit (GtkCellRendererText *cell, gchar *path,
                              gchar *new_text, GtkTreeView *view);
static void lesson_sound_edit (GtkCellRendererText *cell, gchar *path,
                               gchar *new_text, GtkTreeView *view);
static void edit_wizard_new (GtkAction *act, lingGtkEditor *edit);

/* general editor stuff */
static void set_menu_sensitive (lingGtkEditor *editor, gboolean sensitive);
static void set_search_id (GtkComboBox *combo, GtkEntry *entry);

static void save_lesson (GtkAction *act, lingGtkEditor *editor);
static void save_lesson_to_file (GtkAction *act, lingGtkEditor *editor);

static void create_view_lesson (lingGtkEditor *editor, GtkListStore *list);
static GtkListStore* create_lesson_list (void);
static void refresh_lesson_list (GtkTreeView *tree);
static gint calc_translation_percent (lingLesson *lesson);
static void lesson_selection_changed (lingGtkEditor *editor);
static gboolean check_modified_lesson (lingGtkEditor *editor); 

static GtkUIManager* create_menu (lingGtkEditor *editor);
static void menu_quit_editor (GtkAction *act, lingGtkEditor *editor);

static GtkWidget* create_toolbar (lingGtkEditor *editor);
static void reload_lesson (GtkAction *act, lingGtkEditor *editor);

static void
edit_lesson (GtkTreeView *treeview, GtkTreePath *path, GtkTreeViewColumn *col,
             lingGtkEditor *editor)
{
     GtkTreeModel *tree;
     GtkTreeIter iter;
     lingLesson *lesson = NULL;
     GtkTreeModel *model = gtk_tree_view_get_model (treeview);

     if (!gtk_tree_model_get_iter (model, &iter, path))
          return;

     gtk_tree_model_get (model, &iter, GTKEDIT_LESSON_POINTER, &lesson, -1);

     /* create meaning list */
     tree = gtkedit_meaning_create_entry_list (lesson);
     gtk_tree_view_set_model (GTK_TREE_VIEW (editor->view_meanings), tree);
     g_object_unref (tree);

     set_menu_sensitive (editor, TRUE);
     gtkedit_meaning_update_menus (editor);
     util_new_status (GTK_STATUSBAR (editor->status),
                      _("Meaning tree created."));
     return;
}

static void
lesson_name_edit (GtkCellRendererText *cell, gchar *path, gchar *new_text,
                  GtkTreeView *view)
{
     lingLesson *lesson = NULL;
     GtkTreeIter iter;
     GtkTreeModel *model = gtk_tree_view_get_model (view);

     debug ("Editing lesson name...\n");
     if (gtk_tree_model_get_iter_from_string (model, &iter, path))
     {
          /* get the lesson pointer - does this lead to refs
           * or inconsistency? */
          gtk_tree_model_get (model, &iter,
                              GTKEDIT_LESSON_POINTER, &lesson, -1);
          
          if (g_utf8_collate (lesson->type, new_text) == 0)
               return;

          /* TODO: maybe change this code later on */
          ling_free (lesson->type);
          lesson->type = NULL;
          lesson->type = ling_malloc (strlen (new_text) + 1);
          strncpy (lesson->type, new_text, strlen (new_text) + 1);
          
          /* replace the old one */
          gtk_list_store_set (GTK_LIST_STORE (model), &iter,
                              GTKEDIT_LESSON_NAME, lesson->type,
                              GTKEDIT_LESSON_EDITED, TRUE,
                              -1);
     }
     return;
}

static void
lesson_sound_edit (GtkCellRendererText *cell, gchar *path, gchar *new_text,
                   GtkTreeView *view)
{
     lingLesson *lesson = NULL;
     GtkTreeIter iter;
     GtkTreeModel *model = gtk_tree_view_get_model (view);

     debug ("Editing lesson sound...\n");
     if (gtk_tree_model_get_iter_from_string (model, &iter, path))
     {
          /* get the lesson pointer - does this lead to refs or
           * inconsistency? */
          gtk_tree_model_get (model, &iter,
                              GTKEDIT_LESSON_POINTER, &lesson, -1);
          
          if (g_utf8_collate (lesson->sound, new_text) == 0)
               return;

          /* TODO: maybe change this code later on */
          ling_free (lesson->sound);
          lesson->sound = NULL;
          lesson->sound = ling_malloc (strlen (new_text) + 1);
          strncpy (lesson->sound, new_text, strlen (new_text) + 1);
          
          /* replace the old one */
          gtk_list_store_set (GTK_LIST_STORE (model), &iter,
                              GTKEDIT_LESSON_SOUND, lesson->sound,
                              GTKEDIT_LESSON_EDITED, TRUE,
                              -1);
     }
     return;
}

static void
edit_wizard_new (GtkAction *act, lingGtkEditor *editor)
{
     GtkTreeIter iter;
     lingLesson *lesson = NULL;
     lingMeaning *meaning = NULL;
     GtkTreeModel *model = NULL;
     
     /* create the new lesson */
     meaning = ling_meaning_new ();
     meaning->id = 1;
     lesson = ling_meaning_create_lesson (meaning, settings->prefs->config,
                                          "Fill me", "Fill me");

     /* set the data within the model */
     model = gtk_tree_view_get_model (GTK_TREE_VIEW (editor->view_lesson));
     gtk_list_store_append (GTK_LIST_STORE (model), &iter);
     gtk_list_store_set (GTK_LIST_STORE (model), &iter, 
                         GTKEDIT_LESSON_NAME, "Fill me",
                         GTKEDIT_LESSON_MEANINGS, 0,
                         GTKEDIT_LESSON_PERCENT, 0,
                         GTKEDIT_LESSON_SOUND, "Fill me",
                         GTKEDIT_LESSON_POINTER, lesson,
                         GTKEDIT_LESSON_EDITED, TRUE,
                         -1);

     settings->prefs->lessons = ling_lesson_add (settings->prefs->lessons,
                                                 lesson);

     debug ("Creating new lesson...\n");
     /* TODO: add a wizard here later! */
     
     return;
}

static void
set_menu_sensitive (lingGtkEditor *editor, gboolean sensitive)
{
     GtkWidget *entry;

     /* set save entries */
     entry = gtk_ui_manager_get_widget (editor->menu, "/EditorMenu/File/Save");
     gtk_widget_set_sensitive (entry, sensitive);
     entry = gtk_ui_manager_get_widget (editor->menu,
                                        "/EditorMenu/File/SaveAs");
     gtk_widget_set_sensitive (entry, sensitive);
     gtkedit_set_toolbar_item_sensitive (editor, GTKEDIT_TOOL_SAVE, sensitive);

     entry = gtk_ui_manager_get_widget (editor->menu,
                                        "/EditorMenu/Edit/Refresh");
     gtk_widget_set_sensitive (entry, sensitive);
     gtkedit_set_toolbar_item_sensitive (editor, GTKEDIT_TOOL_REFRESH,
                                         sensitive);
     
     return;
}

static void
set_search_id (GtkComboBox *combo, GtkEntry *entry)
{
     g_object_set_data (G_OBJECT (entry), GTKEDIT_SEARCH_ID, 
                        GINT_TO_POINTER (gtk_combo_box_get_active (combo)));
     return;
}

static void
save_lesson (GtkAction *act, lingGtkEditor *editor)
{
     GtkTreeModel *model;
     GtkTreeIter iter;
     lingMeaning *meaning;
     lingLesson *lesson = NULL;
     char *path = NULL;
     const gchar *fail = _("Lesson was not saved.");

     debug ("Saving lesson...\n");

     model = gtk_tree_view_get_model (GTK_TREE_VIEW (editor->view_meanings));
     /* it makes no sense to save a complete empty lesson */
     if (!gtk_tree_model_get_iter_first (model, &iter))
     {
          error_warning (_("Lesson could not be saved!"),
                         _("The lesson seems to completely empty and thus "
                           "will not be saved."), ERR_NOT_AVAILABLE);
          return; 
     }
     gtk_tree_model_get (model, &iter, LING_TREE_POINTER, &meaning, -1);
    
     /* check the path of the lesson and select a new one, if the lesson does
      * not have a path yet */
     path = ling_lesson_get_path (meaning->lesson);
     if (!path)
     {
          path = util_run_fileselection (_("Save lesson as..."),
                                         GTK_FILE_CHOOSER_ACTION_SAVE);
          if (!path)
          {
               util_new_status (GTK_STATUSBAR (editor->status), fail);
               return;
          }
     }

     /* save lesson */
     lesson = gtkedit_meaning_create_lesson_from_list
                (GTK_TREE_VIEW (editor->view_meanings));
     if (!lesson)
     {
          error_warning (_("Lesson could not be created!"),
                         _("The lesson could not be created."),
                         ERR_NOT_AVAILABLE);
          util_new_status (GTK_STATUSBAR (editor->status), fail);
          return;
     }

     ling_lesson_set_path (lesson, path);
     if (!ling_lesson_save_lesson (lesson, NULL))
     {
          error_warning (_("Lesson could not be saved!"),
                         _("The lesson could not be saved."),
                         _("Make sure, you have proper access rights "
                           "on the directory and/or file you want to save "
                           "the lesson to."));
          util_new_status (GTK_STATUSBAR (editor->status), fail);
          return;
     }
     /* preserve path and clean up */
     path = g_strdup (ling_lesson_get_path (lesson));
     ling_lesson_free (lesson);
     
     /* reload lessons with the new one */
     /* TODO: check for open sessions and fix up that ugly
        load_lesson_to_prefs, it resorts the list and stuff. */
     settings->prefs->lessons = ling_lesson_remove (settings->prefs->lessons,
                                                    meaning->lesson);
     lesson_load_lesson_to_prefs (settings->prefs, path);

     /* emit state change */
     event_emit_signal (main_app->objects, "lessons-changed");

     /* and update the meaning tree to reflect the changes */
     lesson = settings->prefs->lessons;
     while (lesson->next)
          lesson = lesson->next;

     model = gtkedit_meaning_create_entry_list (lesson);
     gtk_tree_view_set_model (GTK_TREE_VIEW (editor->view_meanings), model);
     g_object_unref (model);

     util_new_status (GTK_STATUSBAR (editor->status),
                      _("Lesson successfully saved."));
     return;
}

static void
save_lesson_to_file (GtkAction *act, lingGtkEditor *editor)
{
     gchar *file = NULL;
     lingLesson *lesson = NULL;
     const gchar *fail = _("Lesson was not saved.");
     
     file = util_run_fileselection (_("Save lesson as..."),
                                    GTK_FILE_CHOOSER_ACTION_SAVE);
     if (!file)
     {
          util_new_status (GTK_STATUSBAR (editor->status), fail);
          return;
     }

     /* save lesson */
     lesson = gtkedit_meaning_create_lesson_from_list
          (GTK_TREE_VIEW (editor->view_meanings));
     if (!lesson || !ling_lesson_save_lesson (lesson, file))
     {
          error_warning (_("Lesson could not be saved!"),
                         _("The lesson could not be saved"),
                         _("Make sure, you have proper access rights "
                           "on the directory and/or file you want to save "
                           "the lesson to."));
          util_new_status (GTK_STATUSBAR (editor->status), fail);
          return;
     }

     util_new_status (GTK_STATUSBAR (editor->status),
                      _("Lesson successfully saved."));
     return;
}

static void
create_view_lesson (lingGtkEditor *editor, GtkListStore *list)
{
     GtkCellRenderer *renderer;
     GtkTreeSelection *select;

     editor->view_lesson =
          gtk_tree_view_new_with_model (GTK_TREE_MODEL (list));
     g_object_unref (list);

     /* register object */
     main_app->objects =
          event_append_listener (main_app->objects,
                                 editor->view_lesson,
                                 "lessons-changed",
                                 EV_CALLBACK (refresh_lesson_list),
                                 NULL);
     g_signal_connect (G_OBJECT (editor->view_lesson), "destroy",
                       G_CALLBACK (util_event_unregister), NULL);

     renderer = gtk_cell_renderer_text_new ();
     g_object_set (G_OBJECT (renderer), "editable", TRUE, "foreground", "red",
                   NULL);
     gtk_tree_view_insert_column_with_attributes
          (GTK_TREE_VIEW (editor->view_lesson),
           GTKEDIT_LESSON_NAME, _("Lesson Name"), renderer, "text",
           GTKEDIT_LESSON_NAME, "foreground-set", GTKEDIT_LESSON_EDITED, NULL);
     g_signal_connect (G_OBJECT (renderer), "edited",
                       G_CALLBACK (lesson_name_edit),
                       GTK_TREE_VIEW (editor->view_lesson));

     renderer = gtk_cell_renderer_text_new ();
     g_object_set (G_OBJECT (renderer), "foreground", "red", NULL);
     gtk_tree_view_insert_column_with_attributes
          (GTK_TREE_VIEW (editor->view_lesson),
           GTKEDIT_LESSON_MEANINGS, _("Meaning Amount"), renderer, "text",
           GTKEDIT_LESSON_MEANINGS, "foreground-set", GTKEDIT_LESSON_EDITED,
           NULL);

     renderer = gtk_cell_renderer_text_new ();
     g_object_set (G_OBJECT (renderer), "foreground", "red", NULL);
     gtk_tree_view_insert_column_with_attributes
          (GTK_TREE_VIEW (editor->view_lesson),
           GTKEDIT_LESSON_PERCENT, _("Translated (in %)"), renderer, "text",
           GTKEDIT_LESSON_PERCENT, "foreground-set", GTKEDIT_LESSON_EDITED,
           NULL);

     renderer = gtk_cell_renderer_text_new ();
     g_object_set (G_OBJECT (renderer), "editable", TRUE, "foreground", "red",
                   NULL);
     gtk_tree_view_insert_column_with_attributes
          (GTK_TREE_VIEW (editor->view_lesson),
           GTKEDIT_LESSON_SOUND, _("Sound subdir."), renderer, "text",
           GTKEDIT_LESSON_SOUND, "foreground-set", GTKEDIT_LESSON_EDITED,
           NULL);
     g_signal_connect(G_OBJECT (renderer), "edited",
                      G_CALLBACK (lesson_sound_edit),
                      GTK_TREE_VIEW (editor->view_lesson));

     /* set selection mode */
     select =
          gtk_tree_view_get_selection (GTK_TREE_VIEW (editor->view_lesson));
     gtk_tree_selection_set_mode (select, GTK_SELECTION_SINGLE);
     g_signal_connect_swapped (G_OBJECT (select), "changed",
                               G_CALLBACK (lesson_selection_changed), editor);

     return;
}

static void
refresh_lesson_list (GtkTreeView *tree)
{
     GtkListStore *list = create_lesson_list ();
     gtk_tree_view_set_model (tree, GTK_TREE_MODEL (list));
     g_object_unref (list);
     return;
}

static GtkListStore*
create_lesson_list ()
{
     GtkListStore *list;
     GtkTreeIter iter;
     lingLesson *tmp = settings->prefs->lessons;

     /* create the list and fill it with data */
     list = gtk_list_store_new (GTKEDIT_LESSON_COLUMNS, G_TYPE_STRING,
                                G_TYPE_INT, G_TYPE_INT, G_TYPE_STRING,
                                G_TYPE_POINTER, G_TYPE_BOOLEAN);

     debug ("Creating lesson list\n");

     while (tmp)
     {
          gtk_list_store_append (list, &iter);
          gtk_list_store_set (list, &iter,
                              GTKEDIT_LESSON_NAME, tmp->type,
                              GTKEDIT_LESSON_MEANINGS,
                              ling_lesson_get_meaning_amount (tmp),
                              GTKEDIT_LESSON_PERCENT,
                              calc_translation_percent (tmp),
                              GTKEDIT_LESSON_SOUND, tmp->sound,
                              GTKEDIT_LESSON_POINTER, tmp,
                              GTKEDIT_LESSON_EDITED, FALSE,
                              -1);
          tmp = tmp->next;
     }
     return list;
}

static gint
calc_translation_percent (lingLesson *lesson)
{
     gint amount = ling_lesson_get_meaning_amount (lesson);
     gint langs = ling_lang_get_language_amount (settings->prefs->config);
     gint meanings = ling_lesson_get_translation_amount (lesson);

     if (amount < 0 || langs < 0 || meanings < 0)
          return -1;
     
     return (meanings * 100) / (amount * langs);
}

static void
lesson_selection_changed (lingGtkEditor *editor)
{
     GtkTreeSelection *select;
     GtkWidget *entry;
     gboolean val = FALSE;

     select =
          gtk_tree_view_get_selection (GTK_TREE_VIEW (editor->view_lesson));

     val = gtk_tree_selection_get_selected (select, NULL, NULL);
     
     entry = gtk_ui_manager_get_widget (editor->menu,
                                        "/EditorMenu/Edit/Refresh");
     gtk_widget_set_sensitive (entry, val);
     gtkedit_set_toolbar_item_sensitive (editor, GTKEDIT_TOOL_REFRESH, val);
     return;
}

static gboolean
check_modified_lesson (lingGtkEditor *editor)
{
     GtkTreeModel *model;
     GtkTreeIter iter;
     gboolean edited = FALSE;

     model = gtk_tree_view_get_model (GTK_TREE_VIEW (editor->view_lesson));

     if (gtk_tree_model_get_iter_first (model, &iter))
     {
          do
          {
               gtk_tree_model_get (model, &iter,
                                   GTKEDIT_LESSON_EDITED, &edited, -1);
               if (edited)
                    return edited;
          } while (gtk_tree_model_iter_next (model, &iter));
     }
     return FALSE;
}

static GtkUIManager*
create_menu (lingGtkEditor *editor)
{
     const gchar *main_menu =
          "<ui>"
          "  <menubar name='EditorMenu'>"
          "    <menu action='File'>"
          "      <menuitem action='New'/>"
          "      <menuitem action='Open'/>"
          "      <menuitem action='Save'/>"
          "      <menuitem action='SaveAs'/>"
          "      <separator/>"
          "      <menuitem action='Quit'/>"
          "    </menu>"
          "    <menu action='Edit'>"
          "      <menuitem action='Refresh'/>"
          "    </menu>"
          "  </menubar>"
          "</ui>";
     
     GtkActionEntry entries[] = {
          { "File", NULL, "_File", NULL, NULL, NULL },
          { "New", GTK_STOCK_NEW, NULL, NULL, N_("Create a new lesson"),
            G_CALLBACK (edit_wizard_new) },
          { "Open", GTK_STOCK_OPEN, NULL, NULL, N_("Open a lesson"),
            G_CALLBACK (gtklesson_add_lesson) },
          { "Save", GTK_STOCK_SAVE, NULL, NULL, N_("Save lesson"),
            G_CALLBACK (save_lesson) },
          { "SaveAs", GTK_STOCK_SAVE_AS, NULL, NULL, N_("Save lesson"),
            G_CALLBACK (save_lesson_to_file) },
          { "Quit", GTK_STOCK_QUIT, NULL, NULL, N_("Quit"), 
            G_CALLBACK (menu_quit_editor) },
          { "Edit", NULL, "_Edit", NULL, NULL, NULL },
          { "Refresh", GTK_STOCK_REFRESH, NULL, NULL, N_("Reload the lesson"),
            G_CALLBACK (reload_lesson) },
     };
     
     GtkUIManager *menu;
     GError *error = NULL;

     menu = gtk_ui_manager_new ();
     editor->actions = gtk_action_group_new ("Actions");
     gtk_action_group_add_actions (editor->actions, entries,
                                   G_N_ELEMENTS (entries), editor);
     gtk_ui_manager_insert_action_group (menu, editor->actions, 0);
     g_object_unref (editor->actions);

     if (!gtk_ui_manager_add_ui_from_string (menu, main_menu, -1, &error))
     {
          error_critical (_("Editor menu could not be created!"),
                          error->message, ERR_NOT_AVAILABLE);
          g_error_free (error);
     }
     return menu;
}

static void
menu_quit_editor (GtkAction *act, lingGtkEditor *editor)
{
     gtkedit_quit_editor (editor);
     return;
}

static GtkWidget*
create_toolbar (lingGtkEditor *editor)
{
     GtkWidget *box_handle;
     GtkWidget *toolbar;

     GtkToolItem *button;
     GtkToolItem *sep;

     box_handle = gtk_handle_box_new ();

     toolbar = gtk_toolbar_new ();
     gtk_toolbar_set_style (GTK_TOOLBAR (toolbar), GTK_TOOLBAR_BOTH);
     gtk_toolbar_set_tooltips (GTK_TOOLBAR (toolbar), TRUE);
     gtk_toolbar_set_icon_size (GTK_TOOLBAR (toolbar),
                                GTK_ICON_SIZE_SMALL_TOOLBAR);
     gtk_toolbar_set_style (GTK_TOOLBAR (toolbar), GTK_TOOLBAR_ICONS);
     gtk_container_add (GTK_CONTAINER (box_handle), toolbar);

     /* new */
     button = gtk_tool_button_new_from_stock (GTK_STOCK_NEW);
     g_signal_connect (G_OBJECT (button), "clicked",
                       G_CALLBACK (edit_wizard_new), editor);
     gtk_toolbar_insert (GTK_TOOLBAR (toolbar), button, -1);
     gtk_tool_item_set_tooltip (button, main_app->tips,
                                _("Create a new lesson from scratch"), NULL);
     
     /* open */
     button = gtk_tool_button_new_from_stock (GTK_STOCK_OPEN);
     g_signal_connect (G_OBJECT (button), "clicked",
                       G_CALLBACK (gtklesson_add_lesson), NULL);
     gtk_toolbar_insert (GTK_TOOLBAR (toolbar), button, -1);
     gtk_tool_item_set_tooltip (button, main_app->tips,
                                _("Open an existing lesson"), NULL);

     /* save */
     button = gtk_tool_button_new_from_stock (GTK_STOCK_SAVE);
     g_signal_connect (G_OBJECT (button), "clicked",
                       G_CALLBACK (save_lesson), editor);
     gtk_toolbar_insert (GTK_TOOLBAR (toolbar), button, -1);
     gtk_tool_item_set_tooltip (button, main_app->tips,
                                _("Save the currently selected lesson"), NULL);

     /* separator */
     sep = gtk_separator_tool_item_new ();
     gtk_toolbar_insert (GTK_TOOLBAR (toolbar), sep, -1);

     /* refresh */
     button = gtk_tool_button_new_from_stock (GTK_STOCK_REFRESH);
     gtk_toolbar_insert (GTK_TOOLBAR (toolbar), button, -1);
     g_signal_connect (G_OBJECT (button), "clicked",
                       G_CALLBACK (reload_lesson), editor);
     gtk_tool_item_set_tooltip (button, main_app->tips,
                                _("Reload the currently selected lesson"),
                                NULL);

     /* separator */
     sep = gtk_separator_tool_item_new ();
     gtk_toolbar_insert (GTK_TOOLBAR (toolbar), sep, -1);

     return box_handle;
}

static void
reload_lesson (GtkAction *act, lingGtkEditor *editor)
{
     GtkTreeSelection *select;
     GtkTreeModel *model;
     GtkTreeIter iter;
     lingLesson *lesson = NULL;

     select =
          gtk_tree_view_get_selection (GTK_TREE_VIEW (editor->view_lesson));

     if (gtk_tree_selection_get_selected (select, &model, &iter))
     {
          gtk_tree_model_get (model, &iter,
                              GTKEDIT_LESSON_POINTER, &lesson, -1);
          /* reset editing status */
          gtk_list_store_set (GTK_LIST_STORE (model), &iter,
                              GTKEDIT_LESSON_EDITED, FALSE, -1);

          model = gtkedit_meaning_create_entry_list (lesson);
          gtk_tree_view_set_model (GTK_TREE_VIEW (editor->view_meanings),
                                   model);
          g_object_unref (model);
          set_menu_sensitive (editor, TRUE);
          util_new_status (GTK_STATUSBAR (editor->status),
                           _("Lesson reloaded."));
     }
     return;
}

lingGtkEditor*
gtkedit_init_editor (void)
{
     GtkWidget *box_main;
     GtkWidget *box_handle;

     GtkWidget *pane;
     
     GtkWidget *frm_lesson;
     GtkWidget *align_lesson;
     GtkWidget *scr_win_lesson;
     GtkListStore *list_lesson;

     GtkWidget *frm_trans;
     GtkWidget *align_trans;
     GtkWidget *box_trans;
     GtkWidget *scr_win_trans;

     GtkWidget *box_search;
     GtkWidget *lbl_search;
     GtkWidget *txt_search;
     GtkEntryCompletion *cmpl_search;
     GtkListStore *cmpl_list;
     GtkWidget *cmb_search;
     guint i = 0;

     lingGtkEditor *editor;

     editor = g_new0 (lingGtkEditor, 1);
     
     /* main window */
     editor->window = hig_window_new (_("LingoTeach Editor"));
     gtk_container_set_border_width (GTK_CONTAINER (editor->window), 0);
     gtk_window_set_position (GTK_WINDOW (editor->window),
                              GTK_WIN_POS_CENTER_ON_PARENT);
     gtk_window_set_modal (GTK_WINDOW (editor->window), TRUE);
     gtk_window_set_default_size (GTK_WINDOW (editor->window), 600, 400);

     box_main = hig_vbox_new ();
     gtk_box_set_spacing (GTK_BOX (box_main), 0);
     gtk_container_set_border_width (GTK_CONTAINER (box_main), 0);
     gtk_container_add (GTK_CONTAINER (editor->window), box_main);

     /* menu creation */
     editor->menu = create_menu (editor);
     gtk_box_pack_start (GTK_BOX (box_main),
                         gtk_ui_manager_get_widget (editor->menu,
                                                    "/EditorMenu"),
                         FALSE, FALSE, 0);

     /* handle box and toolbar */
     box_handle = create_toolbar (editor);
     editor->toolbar = gtk_bin_get_child (GTK_BIN (box_handle));
     gtk_box_pack_start (GTK_BOX (box_main), box_handle, FALSE, FALSE, 0);

     set_menu_sensitive (editor, FALSE);

     /* statusbar */
     editor->status = gtk_statusbar_new ();
     gtk_box_pack_end (GTK_BOX (box_main), editor->status, FALSE, FALSE, 0);

     /* pane */
     pane = gtk_vpaned_new ();
     gtk_container_set_border_width (GTK_CONTAINER (pane), 12);
     gtk_box_pack_start (GTK_BOX (box_main), pane, TRUE, TRUE, 0);
     
     /* lesson content */
     frm_lesson = hig_frame_new (_("Lesson"));
     gtk_container_set_border_width (GTK_CONTAINER (frm_lesson), 0);
     gtk_paned_pack1 (GTK_PANED (pane), frm_lesson, FALSE, FALSE);
     align_lesson = gtk_bin_get_child (GTK_BIN (frm_lesson));

     scr_win_lesson = hig_scrolled_window_new ();
     gtk_container_set_border_width (GTK_CONTAINER (scr_win_lesson), 0);
     gtk_container_add (GTK_CONTAINER (align_lesson), scr_win_lesson);

     list_lesson = create_lesson_list ();
     create_view_lesson (editor, list_lesson);
     gtk_container_add (GTK_CONTAINER (scr_win_lesson), editor->view_lesson);

     /* translation frame */

     frm_trans = hig_frame_new (_("Meanings"));
     gtk_container_set_border_width (GTK_CONTAINER (frm_trans), 0);
     gtk_paned_pack2 (GTK_PANED (pane), frm_trans, TRUE, FALSE);
     align_trans = gtk_bin_get_child (GTK_BIN (frm_trans));

     box_trans = hig_vbox_new ();
     gtk_container_set_border_width (GTK_CONTAINER (box_trans), 0);
     gtk_container_add (GTK_CONTAINER (align_trans), box_trans);

     /* search stuff */
     box_search = hig_hbox_new ();
     gtk_container_set_border_width (GTK_CONTAINER (box_search), 0);
     gtk_box_pack_start (GTK_BOX (box_trans), box_search, FALSE, TRUE, 0);

     lbl_search = gtk_label_new_with_mnemonic (_("_Search:"));
     gtk_box_pack_start (GTK_BOX (box_search), lbl_search, FALSE, FALSE, 0);

     txt_search = gtk_entry_new ();
     gtk_box_pack_start (GTK_BOX (box_search), txt_search, TRUE, TRUE, 0);
     gtk_label_set_mnemonic_widget (GTK_LABEL (lbl_search), txt_search);
     g_object_set_data (G_OBJECT (txt_search), GTKEDIT_SEARCH_ID,
                        GINT_TO_POINTER (LING_TREE_ID));

     cmpl_search = gtk_entry_completion_new ();
     gtk_entry_set_completion (GTK_ENTRY (txt_search), cmpl_search);
     g_object_unref (cmpl_search);

     cmpl_list = gtk_list_store_new (1, G_TYPE_STRING);
     gtk_entry_completion_set_model (cmpl_search, GTK_TREE_MODEL (cmpl_list));
     g_object_unref (cmpl_list);
     gtk_entry_completion_set_text_column (cmpl_search, 0);

     cmb_search = gtk_combo_box_new_text ();
     while (i < G_N_ELEMENTS (labels))
     {
          gtk_combo_box_append_text (GTK_COMBO_BOX (cmb_search), labels[i]);
          i++;
     }
     g_signal_connect (G_OBJECT (cmb_search), "changed",
                       G_CALLBACK (set_search_id), txt_search);

     gtk_combo_box_set_active (GTK_COMBO_BOX (cmb_search), LING_TREE_ID);
     gtk_box_pack_start (GTK_BOX (box_search), cmb_search, FALSE, FALSE, 0);

     /* meaning window */
     scr_win_trans = hig_scrolled_window_new ();
     gtk_container_set_border_width (GTK_CONTAINER (scr_win_trans), 0);
     gtk_box_pack_start (GTK_BOX (box_trans), scr_win_trans, TRUE, TRUE, 0);

     editor->view_meanings = gtkedit_meaning_create_view (editor, labels);
     gtk_container_add (GTK_CONTAINER (scr_win_trans), editor->view_meanings);

     /* row double click in lessons for loading their content */
     g_signal_connect (G_OBJECT (editor->view_lesson), "row-activated",
                       G_CALLBACK (edit_lesson), editor);

     /* filtering */
     g_signal_connect (G_OBJECT (txt_search), "activate",
                       G_CALLBACK (gtkedit_meaning_apply_search),
                       editor->view_meanings);
     return editor;
}

void
gtkedit_quit_editor (lingGtkEditor *editor)
{
     debug ("Quitting editor...\n");

     /* check for edited lessons */
     if (check_modified_lesson (editor)
         && !util_ask_for_confirmation (_("Save lesson changes?"),
                                        _("There are unsaved lessons. If "
                                          "you do not save them now, all "
                                          "changes will be lost. Do you "
                                          "really want to quit the "
                                          "editor?")))
          return;
     gtk_widget_destroy (editor->window);
     g_free (editor);
     return;
}

void
gtkedit_set_toolbar_item_sensitive (lingGtkEditor *editor, int no,
                                    gboolean sensitive)
{
     GtkToolItem *item;
     
     item = gtk_toolbar_get_nth_item (GTK_TOOLBAR (editor->toolbar), no);
     gtk_widget_set_sensitive (GTK_WIDGET (item), sensitive);
     return;
}
