/**
 * @file ui_tabs.c browser tabs
 *
 * Copyright (C) 2004 Lars Lindner <lars.lindner@gmx.net>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA 
 */

#ifdef HAVE_CONFIG_H
#  include <config.h>
#endif

#include <string.h>
#include "support.h"
#include "ui_mainwindow.h"
#include "ui_htmlview.h"
#include "ui_itemlist.h"
#include "ui_tabs.h"

extern GtkWidget	*mainwindow;

static gboolean on_tab_url_entry_activate(GtkWidget *widget, gpointer user_data) {

	ui_htmlview_launch_URL(GTK_WIDGET(user_data), (gchar *)gtk_entry_get_text(GTK_ENTRY(widget)), UI_HTMLVIEW_LAUNCH_INTERNAL);
	return TRUE;
}

static gboolean on_tab_close_clicked(GtkWidget *button, GdkEvent *event, gpointer user_data) {
	GtkNotebook	*notebook;
	int 		i;
	
	notebook = GTK_NOTEBOOK(lookup_widget(mainwindow, "browsertabs"));
	i = gtk_notebook_get_current_page(notebook);
	gtk_notebook_remove_page(notebook, i);
	
	/* check if all tabs are closed */
	if(1 == gtk_notebook_get_n_pages(notebook))
		gtk_notebook_set_show_tabs(notebook, FALSE);
		
	return TRUE;
}

static void on_tab_switched(GtkNotebook *notebook, GtkNotebookPage *page, guint page_num, gpointer user_data) {
	GtkTreeViewColumn 	*column;
	GtkTreePath		*path;
		
	/* needed because switching does sometimes returns to the tree 
	   view with a very disturbing horizontal scrolling state */
	if(0 == page_num && ui_itemlist_get_two_pane_mode() == FALSE) {
		gtk_tree_view_get_cursor(GTK_TREE_VIEW(lookup_widget(mainwindow, "Itemlist")), &path, &column);
		if(path) {
			column = gtk_tree_view_get_column(GTK_TREE_VIEW(lookup_widget(mainwindow, "Itemlist")), 1);
			gtk_tree_view_set_cursor(GTK_TREE_VIEW(lookup_widget(mainwindow, "Itemlist")), path, column, FALSE);
			gtk_tree_path_free(path);
		}
	}
}

void ui_tabs_init(void) {

	gtk_notebook_set_show_tabs(GTK_NOTEBOOK(lookup_widget(mainwindow, "browsertabs")), FALSE);
	g_signal_connect((gpointer)lookup_widget(mainwindow, "browsertabs"), "switch-page", G_CALLBACK(on_tab_switched), NULL);
}

static gchar* ui_tabs_condense_text(const gchar* title) {
	int i;
	gchar *tmp, *tmp2;

	/* make nice short title */
	if (title != NULL) {
		if(!strncmp(title, "http://",7))
			title += strlen("http://");
		tmp2 = tmp = g_strstrip(g_strdup(title));
		if (*tmp == '\0') {
			g_free(tmp);
			tmp = tmp2 = g_strdup(_("New tab"));
		}
	} else {
		tmp2 = tmp = g_strdup(_("New tab"));
	}
	
	if(g_utf8_strlen(tmp, -1) > 20) {
		for(i = 0; i < 20; i++)
			tmp2 = g_utf8_find_next_char(tmp2, NULL);
        *tmp2 = 0;
		tmp2 = g_strdup_printf("%s...", tmp);
		g_free(tmp);
		tmp = tmp2;
	}
	return tmp;
}

GtkWidget* ui_tabs_new(const gchar *url, const gchar *title, gboolean activate) {
	GtkWidget	*widget, *label, *vbox, *toolbar, *htmlframe, *htmlview, *image;
	gchar		*tmp;
	int		i;

	tmp = ui_tabs_condense_text(title != NULL ? title : _("New tab"));
	label = gtk_label_new(tmp);
	gtk_widget_show(label);
	g_free(tmp);
	
	/* create widgets */
	vbox = gtk_vbox_new(FALSE, 0);
	htmlview = ui_htmlview_new(TRUE);
	toolbar = gtk_hbox_new(FALSE, 6);
	
	g_object_set_data(G_OBJECT(vbox), "htmlview", htmlview);

	widget = gtk_label_new(_("URL"));
	gtk_box_pack_start(GTK_BOX(toolbar), widget, FALSE, FALSE, 6);

	widget = gtk_entry_new();
	gtk_entry_set_text(GTK_ENTRY(widget), url != NULL ? url : "");
	g_object_set_data(G_OBJECT(vbox), "url_entry", widget);
	
	gtk_box_pack_start(GTK_BOX(toolbar), widget, TRUE, TRUE, 0);
	g_signal_connect((gpointer)widget, "activate", G_CALLBACK(on_tab_url_entry_activate), (gpointer)htmlview);

	widget = gtk_button_new();
	gtk_button_set_relief(GTK_BUTTON(widget), GTK_RELIEF_NONE);
	image = gtk_image_new_from_stock("gtk-close", GTK_ICON_SIZE_BUTTON);
	gtk_widget_show(image);
	gtk_container_add(GTK_CONTAINER(widget), image);
	gtk_box_pack_end(GTK_BOX(toolbar), widget, FALSE, FALSE, 0);
	g_signal_connect((gpointer)widget, "clicked", G_CALLBACK(on_tab_close_clicked), (gpointer)htmlview);
	
	htmlframe = gtk_frame_new(NULL);
	gtk_frame_set_shadow_type(GTK_FRAME(htmlframe), GTK_SHADOW_IN);
	gtk_container_add(GTK_CONTAINER(htmlframe), htmlview);	
	
	gtk_box_pack_start(GTK_BOX(vbox), toolbar, FALSE, FALSE, 0);
	gtk_box_pack_end(GTK_BOX(vbox), htmlframe, TRUE, TRUE, 0);
	gtk_widget_show_all(vbox);
	
	i = gtk_notebook_append_page(GTK_NOTEBOOK(lookup_widget(mainwindow, "browsertabs")), vbox, label);
	gtk_notebook_set_show_tabs(GTK_NOTEBOOK(lookup_widget(mainwindow, "browsertabs")), TRUE);
	
	if((TRUE == activate) && (i != -1))
		gtk_notebook_set_current_page(GTK_NOTEBOOK(lookup_widget(mainwindow, "browsertabs")), i);

	if(NULL != url)	
		ui_htmlview_launch_URL(htmlview, (gchar *)url, UI_HTMLVIEW_LAUNCH_INTERNAL);
	return htmlview;
}

void ui_tabs_show_headlines(void) {
	
	gtk_notebook_set_current_page(GTK_NOTEBOOK(lookup_widget(mainwindow, "browsertabs")), 0);
}

static GtkWidget* ui_tabs_find_notebook_child(GtkWidget *parent) {
	GtkWidget *child;
	if (parent == NULL)
		return NULL;
	do {
		child = parent;
		parent = gtk_widget_get_parent(child);
		if (parent == NULL)
			return NULL;
	} while (!GTK_IS_NOTEBOOK(parent));
	return child;
}

void ui_tabs_close_tab(GtkWidget *child) {
	GtkWidget *vbox;
	int n;
	vbox = ui_tabs_find_notebook_child(child);
	g_assert(vbox != NULL);
	n = gtk_notebook_page_num(GTK_NOTEBOOK(lookup_widget(mainwindow, "browsertabs")),vbox);
	g_assert(n >= 0);
	if (n == 0) /* Can't really do anything.... can't close the first tab */
		return;
	gtk_notebook_remove_page(GTK_NOTEBOOK(lookup_widget(mainwindow, "browsertabs")), n);

	/* check if all tabs are closed */
	if(1 == gtk_notebook_get_n_pages(GTK_NOTEBOOK(lookup_widget(mainwindow, "browsertabs"))))
		gtk_notebook_set_show_tabs(GTK_NOTEBOOK(lookup_widget(mainwindow, "browsertabs")), FALSE);
}

void ui_tabs_set_location(GtkWidget *child, const gchar *uri) {
	GtkWidget *vbox;
	GtkEntry *entry;
	int newPosition = 0;
	
	vbox = ui_tabs_find_notebook_child(child);
	g_assert(vbox != NULL);
	if (gtk_notebook_page_num(GTK_NOTEBOOK(lookup_widget(mainwindow, "browsertabs")),vbox) < 1)
		return;

	entry = g_object_get_data(G_OBJECT(vbox), "url_entry");
	g_assert(entry != NULL);

	gtk_editable_delete_text(GTK_EDITABLE(entry), 0, -1);
	gtk_editable_insert_text(GTK_EDITABLE(entry), uri, strlen(uri), &newPosition);
}

void ui_tabs_set_title(GtkWidget *child, const gchar *title) {
    gchar *text;
	GtkWidget *label;
	
	if (gtk_notebook_page_num(GTK_NOTEBOOK(lookup_widget(mainwindow, "browsertabs")),ui_tabs_find_notebook_child(child)) < 1)
		return;
	
	text = ui_tabs_condense_text(title != NULL ? title : _("New tab"));
	label = gtk_label_new(text);
	gtk_widget_show(label);
	gtk_notebook_set_tab_label(GTK_NOTEBOOK(lookup_widget(mainwindow, "browsertabs")),
							   ui_tabs_find_notebook_child(child),
							   label);
	g_free(text);
}

GtkWidget * ui_tabs_get_active_htmlview(void) {
	int current;

	current = gtk_notebook_get_current_page(GTK_NOTEBOOK(lookup_widget(mainwindow, "browsertabs")));
	if(0 == current)
		return ui_mainwindow_get_active_htmlview();
		
	return g_object_get_data(G_OBJECT(gtk_notebook_get_nth_page(GTK_NOTEBOOK(lookup_widget(mainwindow, "browsertabs")), current)), "htmlview");
}

void on_popup_open_link_in_tab_selected(gpointer url, guint callback_action, GtkWidget *widget) {

	ui_tabs_new((gchar *)url, (gchar *)url, FALSE);
}
