#!/usr/bin/python
# -*- coding: utf-8 -*-
### BEGIN LICENSE
# Copyright (C) 2009 Jono Bacon <jono@ubuntu.com>
# Copyright (C) 2010 Michael Budde <mbudde@gmail.com>
#
#This program is free software: you can redistribute it and/or modify it
#under the terms of the GNU General Public License version 3, as published
#by the Free Software Foundation.
#
#This program is distributed in the hope that it will be useful, but
#WITHOUT ANY WARRANTY; without even the implied warranties of
#MERCHANTABILITY, SATISFACTORY QUALITY, or FITNESS FOR A PARTICULAR
#PURPOSE.  See the GNU General Public License for more details.
#
#You should have received a copy of the GNU General Public License along
#with this program.  If not, see <http://www.gnu.org/licenses/>.
### END LICENSE


from desktopcouch.records.server import CouchDatabase
from desktopcouch.records.record import Record
from ConfigParser import RawConfigParser
from lernid.lernidconfig import save_cache_path
from lernid.LernidOptions import Options
import logging


class Preferences(object):

    _instance = None

    def __init__(self):
        self._db_name = 'lernid'
        self._parser = RawConfigParser()
        self._parser.add_section(self._db_name)
        try:
            self._database = CouchDatabase(self._db_name, create=True)
        except:
            self._database = False
            logging.debug('desktopcouch not available')
        if Options.get('avoid_desktopcouch'):
            logging.info('avoiding desktopcouch')
            self._database = False
        self._record_type = "http://wiki.ubuntu.com/Quickly/RecordTypes/Lernid/Preferences"
        self._key = None
        self._load_preferences()

    def _load_preferences(self):
        self._preferences = {
            'record_type': self._record_type,
            'vertical': False,
            'show_appindicator': True,
            'main_window_maximize': False,
            'statusbar': True,
            'nick': '',
            'show_irc_time': False,
            'links_in_default_browser': False

        }
        if self._database:
            results = self._database.get_records(record_type=self._record_type, create_view=True)
            if len(results.rows) == 0:
                #no preferences have ever been saved
                #save them before returning
                self._key = self._database.put_record(Record(self._preferences))
            else:
                prefs = results.rows[0].value
                self._preferences.update(prefs)
                self._key = prefs["_id"]
                self._database.update_fields(self._key, self._preferences)
        else:
            file_name = save_cache_path('lernid')+'/config'
            try:
                config_file = open(file_name, 'r')
            except: 
                return
            self._parser.readfp(config_file)
            config_file.close()
            for item in self._parser.items(self._db_name):
                if item[1].startswith('{'):
                    continue
                elif item[1] == "True":
                    self._preferences[item[0]] = True
                elif item[1] == "False":
                    self._preferences[item[0]] = False
                else:
                    self._preferences[item[0]] = item[1]

    def _update(self, prefs):
        if '_id' in prefs: del prefs['_id']
        if '_rev' in prefs: del prefs['_rev']
        self._preferences.update(prefs)
        if self._database:
            self._database.update_fields(self._key, prefs)

    def _flush(self):
       for item in self._preferences:
           if item == '_id':
               continue
           if item == '_rev':
               continue
           if item == 'record_type':
               continue
           self._parser.set(self._db_name, item, self._preferences[item])
       
       config_file = open(save_cache_path('lernid')+'/config', 'w')
       self._parser.write(config_file)
       config_file.close()

    @classmethod
    def get_instance(cls):
        if not cls._instance:
            cls._instance = cls()
        return cls._instance

    @classmethod
    def get(cls, name=None, default=None):
        """Return dict with preferences if @name is not given, else return
        the value of preference @name. If @name is not set, return @default
        which defaults to None.
        """
        klass = cls.get_instance()
        if name:
            return klass._preferences.get(name, default)
        return klass._preferences

    @classmethod
    def set(cls, *args, **kwargs):
        """set(dict) -> updates preferences with values from dict
        set(name, value) -> updates a single preference
        set(key=val, key=val, ...) -> updates preferences with key-value pairs
        """
        klass = cls.get_instance()
        if len(args) == 1:
            d = args[0]
        elif len(args) == 2:
            name, value = args
            d = {name: value}
        elif len(args) == 0 and kwargs:
            d = kwargs
        else:
            raise ValueError('set should be called as either set(key, val), '
                             'set(dict) or set(key=val, key=val, ...)')
        klass._update(d)

    @classmethod
    def flush(cls):
        klass = cls.get_instance()
        klass._flush()

if __name__ == '__main__':
    print Preferences.get('width')
    Preferences.set('width', 100)
    print Preferences.get('width')
    Preferences.set(width=200, height=300)
    print Preferences.get()
    Preferences.set({'width': 500, 'height': 200})
    print Preferences.get()
