<?php
/*
$Id: groupOfNames.inc,v 1.13 2008-09-28 12:25:34 roland Exp $

  This code is part of LDAP Account Manager (http://www.sourceforge.net/projects/lam)
  Copyright (C) 2006 - 2007  Roland Gruber

*/

/**
* Manages group members for the object class groupOfNames.
*
* @package modules
* @author Roland Gruber
*/

/**
* Manages group members for the object class groupOfNames.
*
* @package modules
*/
class groupOfNames extends baseModule {
	
	/** name of managed object class */
	protected $OBJECT_CLASS_NAME = "groupOfNames";
	
	/** name of member attribute */
	protected $ATTR_MEMBER_NAME = "member";

	/**
	* Returns meta data that is interpreted by parent class
	*
	* @return array array with meta data
	* 
	* @see baseModule::get_metaData()
	*/
	function get_metaData() {
		$return = array();
		// icon
		$return['icon'] = 'groupBig.png';
		// manages host accounts
		$return["account_types"] = array("gon");
		// alias name
		$return["alias"] = _("Group of names");
		// this is a base module
		$return["is_base"] = true;
		// LDAP filter
		$return["ldap_filter"] = array('or' => "(objectClass=" . $this->OBJECT_CLASS_NAME . ")");
		// RDN attribute
		$return["RDN"] = array("cn" => "normal");
		// module dependencies
		$return['dependencies'] = array('depends' => array(), 'conflicts' => array());
		// managed object classes
		$return['objectClasses'] = array($this->OBJECT_CLASS_NAME);
		// managed attributes
		$return['attributes'] = array($this->ATTR_MEMBER_NAME, 'cn', 'owner', 'description');
		// help Entries
		$return['help'] = array(
			'cn' => array(
				"Headline" => _("Group name"),
				"Text" => _("This is the name of this group.")
			),
			'description' => array(
				"Headline" => _("Description"),
				"Text" => _("Here you can enter a description for this group.")
			),
			'owner' => array(
				"Headline" => _("Owner"),
				"Text" => _("The owner of this group.")
			),
			'members' => array(
				"Headline" => _("Members"),
				"Text" => _("This is a list of members of this group.")
			),
			'members' => array(
				"Headline" => _("Members"),
				"Text" => _("This is a list of members of this group. Multiple members are separated by semicolons.")
			)
		);
		// upload fields
		$return['upload_columns'] = array(
			array(
				'name' => 'groupOfNames_name',
				'description' => _('Group name'),
				'help' => 'cn',
				'example' => _('adminstrators'),
				'required' => true
			),
			array(
				'name' => 'groupOfNames_description',
				'description' => _('Description'),
				'help' => 'description',
				'example' => _('Administrators group'),
			),
			array(
				'name' => 'groupOfNames_owner',
				'description' => _('Owner'),
				'help' => 'owner',
				'example' => 'uid=admin,ou=people,o=test'
			),
			array(
				'name' => 'groupOfNames_members',
				'description' => _('Members'),
				'help' => 'membersUpload',
				'example' => 'uid=user1,o=test;uid=user2,o=test',
				'required' => true
			)
		);
		// available PDF fields
		$return['PDF_fields'] = array(
			'name', 'description', 'owner', 'members'
		);
		return $return;
	}

	/**
	* This function fills the $messages variable with output messages from this module.
	*/
	function load_Messages() {
		$this->messages['cn'][0] = array('ERROR', _('Please enter a group name!'));
		$this->messages['member'][0] = array('ERROR', _("You need to add at least one member to this group."));
		$this->messages['member'][1] = array('INFO', _("You need to add at least one member to this group."));
	}
	
	/**
	 * Returns the HTML meta data for the main account page.
	 * 
	 * @return array HTML meta data
	 */
	function display_html_attributes() {
		// check members
		if (!isset($this->attributes[$this->ATTR_MEMBER_NAME]) || (sizeof($this->attributes[$this->ATTR_MEMBER_NAME]) < 1)) {
			call_user_func_array('StatusMessage', $this->messages['member'][1]);
		}
		$return = array();
		// name
		$return[] = array(
			array('kind' => 'text', 'text' => _('Name') . '*'),
			array('kind' => 'input', 'name' => 'cn', 'type' => 'text', 'size' => '20', 'maxlength' => '200', 'value' => $this->attributes['cn'][0]),
			array('kind' => 'help', 'value' => 'cn'));
		// description
		$return[] = array(
			array('kind' => 'text', 'text' => _('Description')),
			array('kind' => 'input', 'name' => 'description', 'type' => 'text', 'size' => '20', 'maxlength' => '200', 'value' => $this->attributes['description'][0]),
			array('kind' => 'help', 'value' => 'description'));
		// owner
		if (isset($this->attributes['owner'][0])) {
			$return[] = array(
				array('kind' => 'text', 'text' => _('Owner'), 'td' => array('valign' => 'top')),
				array('kind' => 'table', 'value' => array(
					array(
						array('kind' => 'text', 'text' => $this->attributes['owner'][0])
					),
					array(
						array('kind' => 'table', 'value' => array(
							array(
								array('kind' => 'input', 'type' => 'submit', 'name' => 'form_subpage_' . get_class($this) . '_owner_add', 'value' => _('Change')),
								array('kind' => 'input', 'type' => 'submit', 'name' => 'form_subpage_' . get_class($this) . '_attributes_delOwner', 'value' => _('Remove'))
							)
						))
					)
				)),
				array('kind' => 'help', 'value' => 'owner', 'td' => array('valign' => 'top')));
		}
		else {
			$return[] = array(
				array('kind' => 'text', 'text' => _('Owner')),
				array('kind' => 'input', 'type' => 'submit', 'name' => 'form_subpage_' . get_class($this) . '_owner_add', 'value' => _("Add")),
				array('kind' => 'help', 'value' => 'owner'));
		}
		// members
		$return[] = array(
			array('kind' => 'text', 'text' => _('Members') . '*'),
			array('kind' => 'input', 'type' => 'submit', 'value' => _('Change'), 'name' => 'form_subpage_' . get_class($this) . '_members_change'),
			array('kind' => 'help', 'value' => 'members')
		);
		$members = array();
		for ($i = 0; $i < sizeof($this->attributes[$this->ATTR_MEMBER_NAME]); $i++) {
			$members[] = array(array('kind' => 'text', 'text' => $this->attributes[$this->ATTR_MEMBER_NAME][$i]));
		}
		$return[] = array(
			array('kind' => 'text', 'text' => ''),
			array('kind' => 'table', 'td' => array('colspan' => 2), 'value' => $members)			
		);
		return $return;
	}

	/**
	* Processes user input of the primary module page.
	* It checks if all input values are correct and updates the associated LDAP attributes.
	*
	* @return array list of info/error messages
	*/
	function process_attributes() {
		$return = array();
		// set cn
		$this->attributes['cn'][0] = $_POST['cn'];
		// set description
		$this->attributes['description'][0] = $_POST['description'];
		// remove owner
		if (isset($_POST['form_subpage_' . get_class($this) . '_attributes_delOwner'])) {
			if (isset($this->attributes['owner'][0])) unset($this->attributes['owner'][0]);
		}
		// check input
		if (!isset($this->attributes['cn'][0]) || ($this->attributes['cn'][0] == '')) {
			$return[] = $this->messages['cn'][0];
		}
		return $return;
	}

	/**
	* This function will create the meta HTML code to show a page to change the owner attribute.
	*/
	function display_html_owner() {
		$return = array();
		// add new or change owner
		if (isset($_POST['form_subpage_' . get_class($this) . '_owner_add'])) {
			// account type
			$types = $_SESSION['config']->get_ActiveTypes();
			$options = array();
			for ($i = 0; $i < sizeof($types); $i++) {
				$options[] = array($types[$i], getTypeAlias($types[$i]));
			}
			$return[] = array(
				array('kind' => 'text', 'text' => _('Account type')),
				array('kind' => 'select', 'name' => 'type', 'options' => $options, 'descriptiveOptions' => true));
			$return[] = array(
				array('kind' => 'table', 'td' => array('colspan' => 2), 'value' => array(
					array(
						array('kind' => 'input', 'type' => 'submit', 'value' => _('Ok'), 'name' => 'form_subpage_' . get_class($this) . '_owner_select'),
						array('kind' => 'input', 'type' => 'submit', 'value' => _('Cancel'), 'name' => 'form_subpage_' . get_class($this) . '_attributes_cancel')
					)
				))
			);
		}
		// change existing owner
		// select new owner
		if (isset($_POST['form_subpage_' . get_class($this) . '_owner_select'])) {
			$options = array();
			$filter = get_ldap_filter($_POST['type']);
			$suffix = $_SESSION["config"]->get_Suffix($_POST['type']);
			$sr = @ldap_search($_SESSION["ldap"]->server(), escapeDN($suffix), $filter, array('dn'));
			if ($sr) {
				$info = ldap_get_entries($_SESSION["ldap"]->server(), $sr);
				ldap_free_result($sr);
				if (isset($info['count'])) unset($info['count']);
				for ($i = 0; $i < sizeof($info); $i++) {
					$options[] = $info[$i]['dn'];
				}
			}
			$return[] = array(
				array('kind' => 'text', 'text' => _('New owner')),
				array('kind' => 'select', 'name' => 'newOwner', 'options' => $options));
			$return[] = array(
				array('kind' => 'table', 'td' => array('colspan' => 2), 'value' => array(
					array(
						array('kind' => 'input', 'type' => 'submit', 'value' => _('Ok'), 'name' => 'form_subpage_' . get_class($this) . '_attributes_setOwner'),
						array('kind' => 'input', 'type' => 'submit', 'value' => _('Cancel'), 'name' => 'form_subpage_' . get_class($this) . '_attributes_cancel')
					)
				))
			);
		}
		return $return;
	}

	/**
	* Processes user input of the owner page.
	* It checks if all input values are correct and updates the associated LDAP attributes.
	*
	* @return array list of info/error messages
	*/
	function process_owner() {
		if (isset($_POST['form_subpage_' . get_class($this) . '_attributes_setOwner'])) {
			if (isset($_POST['newOwner']) && ($_POST['newOwner'] != '')) {
				$this->attributes['owner'][0] = $_POST['newOwner'];
			}
		}
		return array();
	}

	/**
	* This function will create the meta HTML code to show a page to change the member attribute.
	*/
	function display_html_members() {
		$return = array();
		// show list of possible new members
		if (isset($_POST['form_subpage_' . get_class($this) . '_members_select']) && isset($_POST['type'])) {
			$options = array();
			$filter = get_ldap_filter($_POST['type']);
			$suffix = $_SESSION["config"]->get_Suffix($_POST['type']);
			$sr = @ldap_search($_SESSION["ldap"]->server(), escapeDN($suffix), $filter, array('dn'));
			if ($sr) {
				$info = ldap_get_entries($_SESSION["ldap"]->server(), $sr);
				ldap_free_result($sr);
				if (isset($info['count'])) unset($info['count']);
				for ($i = 0; $i < sizeof($info); $i++) {
					if (!isset($this->attributes[$this->ATTR_MEMBER_NAME]) || !in_array($info[$i]['dn'], $this->attributes[$this->ATTR_MEMBER_NAME])) {
						$options[] = $info[$i]['dn'];
					}
				}
			}
			$size = 20;
			if (sizeof($options) < 20) $size = sizeof($options);
			$return[] = array(
				array('kind' => 'select', 'name' => 'members', 'options' => $options, 'multiple' => true, 'size' => $size)
			);
			$return[] = array(
				array('kind' => 'table', 'value' => array(
					array(
						array('kind' => 'input', 'type' => 'submit', 'value' => _('Add'), 'name' => 'form_subpage_' . get_class($this) . '_members_addMembers'),
						array('kind' => 'input', 'type' => 'submit', 'value' => _('Cancel'), 'name' => 'form_subpage_' . get_class($this) . '_members_cancel')
					)
				))
			);
			return $return;
		}
		// show existing members
		$members = $this->attributes[$this->ATTR_MEMBER_NAME];
		$size = 20;
		if (sizeof($this->attributes[$this->ATTR_MEMBER_NAME]) < 20) {
			$size = sizeof($this->attributes[$this->ATTR_MEMBER_NAME]);
		}
		if (sizeof($members) > 0) {
			$return[] = array(
				array('kind' => 'select', 'name' => 'members', 'options' => $members, 'multiple' => true, 'size' => $size)
			);
			$return[] = array(
				array('kind' => 'input', 'type' => 'submit', 'value' => _('Remove selected entries'), 'name' => 'form_subpage_' . get_class($this) . '_members_remove', 'td' => array('colspan' => 3))
			);
			$return[] = array(
				array('kind' => 'text', 'text' => '&nbsp;')
			);
		}
		$types = $_SESSION['config']->get_ActiveTypes();
		$options = array();
		for ($i = 0; $i < sizeof($types); $i++) {
			$options[] = array($types[$i], getTypeAlias($types[$i]));
		}
		$return[] = array(
			array('kind' => 'table', 'value' => array(
				array(
					array('kind' => 'text', 'text' => _('Add entries of this type:') . ' '),
					array('kind' => 'select', 'name' => 'type', 'options' => $options, 'descriptiveOptions' => true),
					array('kind' => 'input', 'type' => 'submit', 'value' => _('Ok'), 'name' => 'form_subpage_' . get_class($this) . '_members_select')
				)
			))
		);
		$return[] = array(
			array('kind' => 'text', 'text' => '&nbsp;')
		);
		$return[] = array(
			array('kind' => 'input', 'type' => 'submit', 'value' => _('Back'), 'name' => 'form_subpage_' . get_class($this) . '_attributes_membersBack')
		);
		return $return;
	}
	
	/**
	* Processes user input of the members page.
	* It checks if all input values are correct and updates the associated LDAP attributes.
	*
	* @return array list of info/error messages
	*/
	function process_members() {
		$return = array();
		if (isset($_POST['form_subpage_' . get_class($this) . '_members_remove']) && isset($_POST['members'])) {
			$members = array_flip($this->attributes[$this->ATTR_MEMBER_NAME]);
			for ($i = 0; $i < sizeof($_POST['members']); $i++) {
				if (isset($members[$_POST['members'][$i]])) {
					unset($members[$_POST['members'][$i]]);
				}
			}
			$this->attributes[$this->ATTR_MEMBER_NAME] = array_values(array_flip($members));
		}
		elseif (isset($_POST['form_subpage_' . get_class($this) . '_members_addMembers']) && isset($_POST['members'])) {
			for ($i = 0; $i < sizeof($_POST['members']); $i++) {
				$this->attributes[$this->ATTR_MEMBER_NAME][] = $_POST['members'][$i];
				$this->attributes[$this->ATTR_MEMBER_NAME] = array_unique($this->attributes[$this->ATTR_MEMBER_NAME]);
			}
		}
		// check input
		if (!isset($_POST['form_subpage_' . get_class($this) . '_members_select'])) {
			if (!isset($this->attributes[$this->ATTR_MEMBER_NAME]) || (sizeof($this->attributes[$this->ATTR_MEMBER_NAME]) < 1)) {
				$return[] = $this->messages['member'][0];
			}
		}
		return $return;
	}

	/**
	* In this function the LDAP account is built up.
	*
	* @param array $rawAccounts list of hash arrays (name => value) from user input
	* @param array $partialAccounts list of hash arrays (name => value) which are later added to LDAP
	* @param array $ids list of IDs for column position (e.g. "posixAccount_uid" => 5)
	* @return array list of error messages if any
	*/
	function build_uploadAccounts($rawAccounts, $ids, &$partialAccounts) {
		$messages = array();
		for ($i = 0; $i < sizeof($rawAccounts); $i++) {
			// add object class
			if (!in_array($this->OBJECT_CLASS_NAME, $partialAccounts[$i]['objectClass'])) $partialAccounts[$i]['objectClass'][] = $this->OBJECT_CLASS_NAME;
			// add cn
			$partialAccounts[$i]['cn'] = $rawAccounts[$i][$ids['groupOfNames_name']];
			// add description
			if ($rawAccounts[$i][$ids['groupOfNames_description']] != "") {
				$partialAccounts[$i]['description'] = $rawAccounts[$i][$ids['groupOfNames_description']];
			}
			// add owner
			if ($rawAccounts[$i][$ids['groupOfNames_owner']] != "") {
				$partialAccounts[$i]['owner'] = $rawAccounts[$i][$ids['groupOfNames_owner']];
			}
			// add members
			$partialAccounts[$i][$this->ATTR_MEMBER_NAME] = split(";", $rawAccounts[$i][$ids['groupOfNames_members']]);
		}
		return $messages;
	}
	
	/**
	* This functions is used to check if all settings for this module have been made.
	*
	* @return boolean true, if settings are complete
	*/
	function module_complete() {
		if (isset($this->attributes[$this->ATTR_MEMBER_NAME]) && (sizeof($this->attributes[$this->ATTR_MEMBER_NAME]) > 0)) {
			return true;
		}
		else {
			return false;
		}
	}

	/**
	* Returns a list of PDF entries
	*/
	function get_pdfEntries() {
		$return = array();
		// name
		if (isset($this->attributes['cn'][0])) {
			$return[$this->OBJECT_CLASS_NAME . '_name'] = array('<block><key>' . _('Group name') . '</key><value>' . $this->attributes['cn'][0] . '</value></block>');
		}
		// description
		if (isset($this->attributes['description'][0])) {
			$return[$this->OBJECT_CLASS_NAME . '_description'] = array('<block><key>' . _('Description') . '</key><value>' . $this->attributes['description'][0] . '</value></block>');
		}
		// owner
		if (isset($this->attributes['owner'][0])) {
			$return[$this->OBJECT_CLASS_NAME . '_owner'] = array('<block><key>' . _('Owner') . '</key><value>' . $this->attributes['owner'][0] . '</value></block>');
		}
		// members
		if (sizeof($this->attributes[$this->ATTR_MEMBER_NAME]) > 0) {
			$return[$this->OBJECT_CLASS_NAME . '_members'][0] = '<block><key>' . _('Members') . '</key><tr><td align=\"L\">' . $this->attributes[$this->ATTR_MEMBER_NAME][0] . '</td></tr></block>';
			for ($i = 1; $i < sizeof($this->attributes[$this->ATTR_MEMBER_NAME]); $i++) {
				$return[$this->OBJECT_CLASS_NAME . '_members'][] = '<block><tr><td align=\"L\">' . $this->attributes[$this->ATTR_MEMBER_NAME][$i] . '</td></tr></block>';
			}
		}
		return $return;
	}

}


?>
