/**
 * Copyright (c) Members of the EGEE Collaboration. 2004-2010.
 * See http://www.eu-egee.org/partners/ for details on the copyright
 * holders.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 *
 *  Authors:
 *  2009-
 *     Oscar Koeroo <okoeroo@nikhef.nl>
 *     Mischa Sall\'e <msalle@nikhef.nl>
 *     David Groep <davidg@nikhef.nl>
 *     NIKHEF Amsterdam, the Netherlands
 *     <grid-mw-security@nikhef.nl>
 *
 *  2007-2009
 *     Oscar Koeroo <okoeroo@nikhef.nl>
 *     David Groep <davidg@nikhef.nl>
 *     NIKHEF Amsterdam, the Netherlands
 *
 *  2003-2007
 *     Martijn Steenbakkers <martijn@nikhef.nl>
 *     Gerben Venekamp <venekamp@nikhef.nl>
 *     Oscar Koeroo <okoeroo@nikhef.nl>
 *     David Groep <davidg@nikhef.nl>
 *     NIKHEF Amsterdam, the Netherlands
 *
 */


/*!
    \page lcmaps_voms_poolaccount.mod voms poolaccount plugin

    \section vomspoolaccountsyn SYNOPSIS
    \b lcmaps_voms_poolaccount.mod
        [-gridmapfile|-GRIDMAPFILE|-gridmap|-GRIDMAP \<location gridmapfile\>]
        [-gridmapdir|-GRIDMAPDIR \<location gridmapdir\>]
        [-override_inconsistency|-OVERRIDE_INCONSISTENCY]
        [-max_mappings_per_credential \<max nr of mappings\>]
        [-do_not_use_secondary_gids]
        [-do_not_require_primary_gid]

    \section vomspoolaccountdesc DESCRIPTION

    This poolaccount acquisition plugin is a 'VOMS-aware' modification of the 'poolaccount' plugin.
    The plugin tries to find a poolaccount (more specifically a UID) based on the VOMS information
    that has been retrieved by the plugin \ref lcmaps_voms.mod "lcmaps_voms.mod"
    from the user's grid credential.
    It will try to match a VO-GROUP-ROLE combination from the user's grid credential with an entry
    in a gridmapfile (most likely the traditional gridmapfile, used by the localaccount and
    poolaccount plugins)
    In this file VO-GROUP-ROLE combinations are listed with a poolaccount entry, as shown in the
    following example.

    EXAMPLE:

    \c "/VO=wilma/GROUP=*" \c .wilma

    \c "/VO=fred/GROUP=*"  \c .fred

    If the first matching VO-GROUP-ROLE combination is \c "/VO=wilma/GROUP=*" the plugin will
    get a poolaccount from the '.test' pool.
    This could result in 'wilma001' as a poolaccount for this user.
    The linking between \c "/VO=wilma/GROUP=*", this user and a poolaccount must be made in
    the same directory as the for the \e 'poolaccount' plugin (the \e gridmapdir),
    otherwise it gives rise to inconsistancies when both are used on a site.
    The actual account assigned to the user is based on his VO information matched in the
    gridmapfile, the user's DN and the primary (and secondary) GIDs gathered so far.
    In the \e gridmapdir directory this is reflected in the leasename, which consists of the
    url-encoded DN + a concatenation of the gathered groupnames.
    So a lease name could look like this:

    EXAMPLE DN with pool/localgroups attached:
    \c %2fo%3ddutchgrid%2fo%3dusers%2fo%3dnikhef%2fcn%3dmartijn%20steenbakkers:pool001:bogus1:bogus2:bogus3:pool003:pool004:pool005

    If a user changes his VO-GROUP-ROLE combinations (but not his VO), in this case he will be mapped to
    a different account (UID) within the same pool.

    \section vomspoolaccountnote1 NOTE 1
        This plugin should only be used in combination with the \e 'voms_localgroup'
        and/or \e 'voms_poolgroup' plugins.

    \section vomspoolaccountnote2 NOTE 2
        The options '-do_not_require_primary_gid' and '-do_not_use_secondary_gids'
        can not be used together, because at least one GID is needed.

    \section vomspoolaccountoptions OPTIONS
    \subsection vomspoolaccountoptie1 -GRIDMAPFILE \<gridmapfile\>
        See \ref vomspoolaccountoptie4 "-gridmap"

    \subsection vomspoolaccountoptie2 -gridmapfile \<gridmapfile\>
        See \ref vomspoolaccountoptie4 "-gridmap"

    \subsection vomspoolaccountoptie3 -GRIDMAP \<gridmapfile\>
        See \ref vomspoolaccountoptie4 "-gridmap"

    \subsection vomspoolaccountoptie4 -gridmap \<gridmapfile\>
        When this option is set it will override the default path to the gridmapfile.
        It is advised to use an absolute path to the gridmapfile to avoid usage of the wrong file(path).

    \subsection vomspoolaccountoptie5 -GRIDMAPDIR \<gridmapdir\>
        See \ref vomspoolaccountoptie6 "-gridmapdir"

    \subsection vomspoolaccountoptie6 -gridmapdir \<gridmapdir\>
        If this option is set, it will override the default path to the gridmapdir.
        It is advised to use an absolute path to the gridmapdir to avoid usage of the wrong path.

    \subsection vomspoolaccountoptie7 -do_not_use_secondary_gids
        The determination of the poolaccount will not be based on the secondary GIDs found, but only
        on the user's DN, the VOMS info for the user and the primary GID that has been found.
        Cannot be used with \ref vomspoolaccountoptie8 "-do_not_require_primary_gid".

    \subsection vomspoolaccountoptie8 -do_not_require_primary_gid
        The determination of the poolaccount will not be based on the primary GID found, but only
        on the user's DN, the VOMS info for the user and the secondary GIDs found.
        Normally this option should not be used, but it can be useful for debugging.
        Cannot be used with \ref vomspoolaccountoptie7 "-do_not_use_secondary_gids".

    \subsection vomspoolaccountoptie9 -OVERRIDE_INCONSISTENCY
        See \ref vomspoolaccountoptie10 "-override_inconsistency"

    \subsection vomspoolaccountoptie10 -override_inconsistency
        Moving a user from one pool to another (because of a VO change)
        should only be done by changing the gridmapfile indicating the new pool for this user.
        If a user has already been mapped previously to a poolaccount, there is a link present
        between this poolaccount and his DN.
        In the good old days prior to LCMAPS, a 'pool change' would still result in a mapping to
        the old pool account, neglecting the administrative changes in the gridmapfile.
        LCMAPS corrects this behaviour:
        By default the voms_poolaccount plugin will \e fail if the pool designated by the gridmapfile
        doesn't match the previously mapped voms_poolaccount leasename.
        If the site doesn't want a failure on this inconsistency it can turn on this parameter.
        When the inconsistency is detected the plugin will automatically unlink the previous mapping
        and will proceed by making a \e new lease from the new pool.

    \subsection vomspoolaccountoptie11 -max_mappings_per_credential \<max nr of mappings\>
        This value indicates the maximum number of accounts a user, or more specifically
        a set of credentials (=DN + FQANS), can be mapped to. Normally this number is 1.
        But if each job should run under its own account the number should be increased.
        The leasename (or poolindex) in this case looks like:
                url_encoded(<DN>):gid1[:gid2[:gid3[...]]]:mapcount=<mapnumber>)

    \subsection vomspoolaccountoptie11 -strict_poolprefix_match [yes|no]. Default is 'yes'.
        If this is set to 'yes', a line in the gridmapfile like
        <FQAN> .pool
        will result in accounts matching the regexp 'pool[0-9]+'.
        Otherwise it will be allowed to match 'pool.*' (legacy behaviour).

\section vomspoolaccountReturnvalue RETURN VALUES
        \li LCMAPS_MOD_SUCCESS : Success
        \li LCMAPS_MOD_FAIL    : Failure


\section vomspoolaccountErrors ERRORS
        See bugzilla for known errors (http://marianne.in2p3.fr/datagrid/bugzilla/)

\section vomspoolaccountSeeAlso SEE ALSO
        \ref lcmaps_voms.mod "lcmaps_voms.mod",
        \ref lcmaps_voms_localgroup.mod "lcmaps_voms_localgroup.mod",
        \ref lcmaps_voms_poolgroup.mod "lcmaps_voms_poolgroup.mod",
        \ref lcmaps_localaccount.mod "lcmaps_localaccount.mod",
        \ref lcmaps_poolaccount.mod "lcmaps_poolaccount.mod",
        \ref lcmaps_posix_enf.mod "lcmaps_posix_enf.mod",
        \ref lcmaps_ldap_enf.mod "lcmaps_ldap_enf.mod",
*/

/*!
    \file   lcmaps_voms_poolaccount.c
    \brief  Interface to the LCMAPS plugins
    \author Martijn Steenbakkers for the EU DataGrid.

    This file contains the code of the voms_poolaccount plugin
    -# plugin_initialize()
    -# plugin_run()
    -# plugin_terminate()
    -# plugin_introspect()
*/

/*****************************************************************************
                            Include header files
******************************************************************************/
/* Try to provide RTLD_DEFAULT */
#define _GNU_SOURCE
#include <dlfcn.h>

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <pwd.h>
#include <ctype.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <unistd.h>

#include "lcmaps_voms_config.h"
#include <lcmaps/lcmaps_modules.h>
#include <lcmaps/lcmaps_arguments.h>
#include <lcmaps/lcmaps_cred_data.h>
#include "lcmaps_gridlist.h"

/******************************************************************************
                                Definitions
******************************************************************************/
#define LCMAPS_MAXGIDBUFFER 256
#define LCMAPS_ABSOLUTE_MAPPING_MAX 9999
#define LCMAPS_DEFAULT_MAPPING_MAX 1
#define LCMAPS_INVALID_MAPPING_MAX -1
#define LCMAPS_MAPPING_MIN 1
#define PLUGIN_RUN      0
#define PLUGIN_VERIFY   1

/******************************************************************************
                          Module specific prototypes
******************************************************************************/
static int plugin_run_or_verify(int, lcmaps_argument_t *, int);
static char * add_mapcount_to_leasename(int, int, const char *);

/******************************************************************************
                       Define module specific variables
******************************************************************************/

static char *gridmapfile            = NULL;
static char *gridmapdir             = NULL;
static int  require_primary_gid     = 0;
static int  use_secondary_gids      = 1;
static int  override_inconsistency  = 0;
static int  mapping_max             = LCMAPS_DEFAULT_MAPPING_MAX;
static int  strict_poolprefix_match = 1; /* By default strict matching */

static int   do_not_map_primary_gid                 = 0;
static int   add_primary_gid_from_mapped_account    = 0;
static int   add_primary_gid_as_secondary_gid_from_mapped_account = 0;
static int   add_secondary_gids_from_mapped_account = 0;


/******************************************************************************
Function:   plugin_initialize
Description:
    Initialize plugin
Parameters:
    argc, argv
    argv[0]: the name of the plugin
Returns:
    LCMAPS_MOD_SUCCESS : succes
    LCMAPS_MOD_FAIL    : failure
    LCMAPS_MOD_NOFILE  : db file not found (will halt LCMAPS initialization)
******************************************************************************/
int plugin_initialize(
        int argc,
        char ** argv
)
{
    char *  logstr = "lcmaps_plugin_voms_poolaccount-plugin_initialize()";
    int i = 0, j = 0;
    struct stat s;

    lcmaps_log_debug(5,"%s: passed arguments:\n", logstr);
    for (i=0; i < argc; i++)
    {
       lcmaps_log_debug(5,"%s: arg %d is %s\n", logstr, i, argv[i]);
    }

    /*
     * the first will be the thing to edit/select (gridmap(file))
     * the second will be the path && filename of the gridmapfile
     */

    /*
     * Parse arguments, argv[0] = name of plugin, so start with i = 1
     */
    for (i = 1; i < argc; i++)
    {
        if ( ((strcmp(argv[i], "-gridmap") == 0) ||
              (strcmp(argv[i], "-GRIDMAP") == 0) ||
              (strcmp(argv[i], "-gridmapfile") == 0) ||
              (strcmp(argv[i], "-GRIDMAPFILE") == 0))
             && (i + 1 < argc))
        {
            if ((argv[i + 1] != NULL) && (strlen(argv[i + 1]) > 0))
            {
                /* check if the setting exists */
                if (stat (argv[i + 1], &s) < 0)
                {
                    lcmaps_log(LOG_ERR, "%s: Error: grid-mapfile not accessible at \"%s\"\n", logstr, argv[i + 1]);
                    return LCMAPS_MOD_FAIL;
                }

                gridmapfile = strdup(argv[i + 1]);
            }
            i++;
        }
        else if ( ((strcmp(argv[i], "-gridmapdir") == 0) ||
              (strcmp(argv[i], "-GRIDMAPDIR") == 0))
             && (i + 1 < argc))
        {
            if ((argv[i + 1] != NULL) && (strlen(argv[i + 1]) > 0))
            {
                /* check if the setting exists */
                if (stat (argv[i + 1], &s) < 0)
                {
                    lcmaps_log(LOG_ERR, "%s: Error: gridmapdir not accessible at \"%s\"\n", logstr, argv[i + 1]);
                    return LCMAPS_MOD_FAIL;
                }

                gridmapdir = strdup(argv[i + 1]);
            }
            i++;
        }
        else if (strcmp(argv[i], "--do-not-add-primary-gid-from-mapped-account") == 0)
        {
            do_not_map_primary_gid = 1;
        }
        else if (strcmp(argv[i], "--add-primary-gid-from-mapped-account") == 0)
        {
            add_primary_gid_from_mapped_account = 1;
        }
        else if (strcmp(argv[i], "--add-primary-gid-as-secondary-gid-from-mapped-account") == 0)
        {
            add_primary_gid_as_secondary_gid_from_mapped_account = 1;
        }
        else if (strcmp(argv[i], "--add-secondary-gids-from-mapped-account") == 0)
        {
            add_secondary_gids_from_mapped_account = 1;
        }
        else if (strcmp(argv[i], "-do_not_use_secondary_gids") == 0)
        {
            use_secondary_gids = 0;
        }
        else if ((strcmp(argv[i], "--do-not-require-primary-gid") == 0) ||
                 (strcmp(argv[i], "-do_not-require-primary-gid") == 0) ||
                 (strcmp(argv[i], "-do_not_require_primary_gid") == 0))
        {
            require_primary_gid = 0;
        }
        else if (strcmp(argv[i], "--require-primary-gid") == 0)
        {
            require_primary_gid = 1;
        }
        else if ( (strcmp(argv[i], "-override_inconsistency") == 0) ||
                  (strcmp(argv[i], "-OVERRIDE_INCONSISTENCY") == 0))
        {
            override_inconsistency = 1;
        }
        else if ( (strcmp(argv[i], "-max_mappings_per_credential") == 0)
             && (i + 1 < argc) )
        {
            if ((argv[i + 1] != NULL) && (strlen(argv[i + 1]) > 0))
            {
                lcmaps_log_debug(4,"%s: Checking if argument behind \"-max_mappings_per_credential\" is a number\n", logstr);
                for (j = 0; j < strlen(argv[i + 1]); j++)
                {
                    if (!isdigit((argv[i + 1])[j]))
                    {
                        lcmaps_log(LOG_ERR,"%s: Maybe found some digits, but there is at least one char corrupting this parameter: %s\n",  logstr, argv[i + 1]);
                        return LCMAPS_MOD_FAIL;
                    }
                }
                mapping_max = atoi(argv[i + 1]);
                if ((mapping_max < LCMAPS_INVALID_MAPPING_MAX) || (mapping_max > LCMAPS_ABSOLUTE_MAPPING_MAX))
                {
                    lcmaps_log(LOG_ERR,"%s: Illegal value for \"-max_mappings_per_credential\" (%d): Should be in the range: %d-%d (failure)\n", logstr, mapping_max, LCMAPS_INVALID_MAPPING_MAX, LCMAPS_ABSOLUTE_MAPPING_MAX);
                    return LCMAPS_MOD_FAIL;
                }
            }
            else
            {
                lcmaps_log(LOG_ERR,"%s: no argument found for %s (failure)\n", logstr, argv[i]);
                return LCMAPS_MOD_FAIL;
            }
            i++;
        }
        else if  ( (strcmp(argv[i], "-strict_poolprefix_match") == 0)
             && (i + 1 < argc) )
        {
            if ((argv[i + 1] != NULL) && (strlen(argv[i + 1]) > 0))
            {
                 if (strcmp(argv[i+1],"yes") == 0)
                 {
                     strict_poolprefix_match = 1;
                 }
                 else if (strcmp(argv[i+1],"no") == 0)
                 {
                     strict_poolprefix_match = 0;
                 }
                 else
                 {
                     lcmaps_log(LOG_ERR,"%s: use \"yes\" or \"no\" for option %s\n", logstr, argv[i]);
                     return LCMAPS_MOD_FAIL;
                 }
            }
            else
            {
                lcmaps_log(LOG_ERR,"%s: no argument found for %s (failure)\n", logstr, argv[i]);
                return LCMAPS_MOD_FAIL;
            }
            i++;
        }
        else
        {
            lcmaps_log(LOG_ERR,"%s: Error in initialization parameter: %s (failure)\n", logstr, argv[i]);
            return LCMAPS_MOD_FAIL;
        }
    }

    /* Post mortum check */
    if (do_not_map_primary_gid && add_primary_gid_from_mapped_account)
    {
        lcmaps_log(LOG_ERR,"%s: Error: can't set both --do-not-add-primary-gid-from-mapped-account and --add-primary-gid-from-mapped-account\n", logstr);
        return LCMAPS_MOD_FAIL;
    }

    return LCMAPS_MOD_SUCCESS;
}

/******************************************************************************
Function:   plugin_introspect
Description:
    return list of required arguments
Parameters:

Returns:
    LCMAPS_MOD_SUCCESS : succes
    LCMAPS_MOD_FAIL    : failure
******************************************************************************/
int plugin_introspect(
        int * argc,
        lcmaps_argument_t ** argv
)
{
    char *                   logstr = "lcmaps_plugin_voms_poolaccount-plugin_introspect()";
    static lcmaps_argument_t argList[] = {
        {"user_dn"          ,   "char *"    ,   0   ,   NULL},
        {"fqan_list"        ,   "char **"   ,   0   ,   NULL},
        {"nfqan"            ,   "int"       ,   0   ,   NULL},
        {"mapcounter"       ,   "int"       ,   1   ,   NULL},
        {"requested_uid"    ,   "uid_t"     ,   0   ,   NULL},
        {NULL               ,   NULL        ,   -1  ,   NULL},
        {NULL               ,   NULL        ,   -1  ,   NULL}
    };

    /* Get the version of LCMAPS being used: we need at least 1.5.8 to be able
     * to demand "requested_username" in the argList */
    int major=0,minor=0,patch=0;
    /* Most UNIX now support RTLD_DEFAULT (POSIX reserved) */
#ifdef RTLD_DEFAULT
    char *errorstr;
    int (*lcmaps_major)(void),(*lcmaps_minor)(void),(*lcmaps_patch)(void);
    dlerror();
    lcmaps_major=dlsym(RTLD_DEFAULT,"lcmaps_get_major_version");
    lcmaps_minor=dlsym(RTLD_DEFAULT,"lcmaps_get_minor_version");
    lcmaps_patch=dlsym(RTLD_DEFAULT,"lcmaps_get_patch_version");
    if (dlerror()==NULL)    {
	major=lcmaps_major();
	minor=lcmaps_minor();
	patch=lcmaps_patch();
    }
#else
#warning old LCMAPS versions do not have lcmaps_get_major_version() functions
    /* No RTLD_DEFAULT, just hope the symbol exists in LCMAPS */
    major=lcmaps_get_major_version();
    minor=lcmaps_get_minor_version();
    patch=lcmaps_get_patch_version();
#endif

    /* Too old when older than 1.5.8 */
    if (major<1 || (major==1 && (minor<5 || (minor==5 && patch<8))))	{
	lcmaps_log_debug(4,
	    "%s: Old LCMAPS found (%d.%d.%d), not using requested_username\n",
	    logstr,major,minor,patch);
    } else {
	lcmaps_log_debug(4,
	    "%s LCMAPS (%d.%d.%d) supports using requested_username\n",
	    logstr,major,minor,patch);
	argList[5].argName="requested_username";
	argList[5].argType="char *";
	argList[5].argInOut=1;
	argList[5].value=NULL;
    }    

    lcmaps_log_debug(5,"%s: introspecting\n", logstr);

    *argv = argList;
    *argc = lcmaps_cntArgs(argList);
    lcmaps_log_debug(5,"%s: address first argument: 0x%x\n", logstr, argList);

    return LCMAPS_MOD_SUCCESS;
}

/******************************************************************************
Function:   plugin_run
Description:
    Gather credentials for LCMAPS
Parameters:
    argc: number of arguments
    argv: list of arguments
Returns:
    LCMAPS_MOD_SUCCESS: authorization succeeded
    LCMAPS_MOD_FAIL   : authorization failed
******************************************************************************/
int plugin_run(
        int argc,
        lcmaps_argument_t * argv
)
{
    return plugin_run_or_verify(argc, argv, PLUGIN_RUN);
}

/******************************************************************************
Function:   plugin_verify
Description:
    Verify if user is entitled to use local credentials based on his grid
    credentials. This means that the site should already have been set up
    by, e.g., LCMAPS in a previous run. This method will not try to setup
    account leases, modify (distributed) passwd/group files, etc. etc.
    The outcome should be identical to that of plugin_run().

    Policy: This method will not fail if the uid found is not the
            uid requested for the user, but it will issue a warning.
            The full check is deferred to the stage after which all plugins have run.

Parameters:
    argc: number of arguments
    argv: list of arguments
Returns:
    LCMAPS_MOD_SUCCESS: authorization succeeded
    LCMAPS_MOD_FAIL   : authorization failed
******************************************************************************/
int plugin_verify(
        int argc,
        lcmaps_argument_t * argv
)
{
    return plugin_run_or_verify(argc, argv, PLUGIN_VERIFY);
}

/******************************************************************************
Function: add_mapcount_to_leasename

Description:
    Add the mapcount to the leasename.

   Check the mapcounter against maximum number of allowed mappings
   The following scenarios may happen:
    1. mapping_max is not set (==LCMAPS_INVALID_MAPPING_MAX), mapcounter is set (>0):
       Enforce only one mapping with mapcounter is 1, reflected in the poolindex.
       If mapcounter>1: error.
    2. mapping_max is not set (==LCMAPS_INVALID_MAPPING_MAX), mapcounter is not set (==-1, 0)
       Legacy: one mapping is allowed, not reflected in the poolindex
    3. mapping_max is set (>LCMAPS_INVALID_MAPPING_MAX), mapcounter is not set
       If mapping_max==0: error
    4. mapping_max is set (>LCMAPS_INVALID_MAPPING_MAX), mapcounter is set (>0)
       Mapping is allowed, if mapcounter <= mapping_max
       If mapping_max==0: error

    NOTE: to separate the pure DN from attached strings we insert the character '\001'
          after the DN. The url encoding (in lcmaps_gridlist()) will be done up to this
          point.
          The '\001' character may already be there if a gidstring was inserted first,
          so this is checked for. In that case a ':' is inserted.

Parameters:
    mapping_max:  maximum number of account mappings for this user+FQANS combination
    mapcounter:   account mapping number
    leasename:    the leasename after which the mapcount string should be appended
Returns:
    the new leasename (should be freed)
    NULL in case of error
******************************************************************************/
static char * add_mapcount_to_leasename(
        int mapping_max,
        int mapcounter,
        const char * leasename
)
{
    char * newleasename = NULL;
    int    newleasenamelen = 0;
    char * logstr = "lcmaps_plugin_voms_poolaccount-add_mapcount_to_leasename()";

    if (mapping_max == LCMAPS_INVALID_MAPPING_MAX)
    {
        if (mapcounter < LCMAPS_MAPPING_MIN)
        {
            /* legacy mode, nothing is set */
            /* do nothing, leasename unchanged */
            lcmaps_log_debug(3,"%s: mapcounter not used, leasename unchanged\n", logstr);
            if (leasename) newleasename = strdup(leasename);
        }
        else if (mapcounter == LCMAPS_MAPPING_MIN) /* OK, create a leasename with a ':mapcount=1' at the end */
        {
            char mapcount_string[14];

            snprintf(mapcount_string, 14, "mapcount=%04d", mapcounter);

            if (leasename) newleasename = strdup(leasename);
            newleasenamelen = strlen(newleasename) + 15;
            newleasename = realloc(newleasename, newleasenamelen*sizeof(char));
            lcmaps_log_debug(3,"%s: leasename before adding mapcount: %s\n", logstr, newleasename);
            if (strstr(leasename, "\001"))
                strncat(newleasename, ":", 2);
            else
                strncat(newleasename, "\001", 2);
            strncat(newleasename, mapcount_string, 14);
            lcmaps_log_debug(3,"%s: leasename after adding mapcount: %s\n", logstr, newleasename);
        }
        else
        {
            lcmaps_log(LOG_ERR,"%s: mapping_max is not set, so only 1 mapping is allowed. The request (%d), exceeds this (failure)\n", logstr, mapcounter);
            return NULL;
        }
    }
    else if (mapping_max >= LCMAPS_MAPPING_MIN)
    {
        if (mapcounter < LCMAPS_MAPPING_MIN)
        {
            /* legacy mode, nothing is set */
            /* do nothing, leasename unchanged */
            lcmaps_log_debug(3,"%s: mapcounter not used, leasename unchanged\n", logstr);
            if (leasename) newleasename = strdup(leasename);
        }
        else if (mapcounter <= mapping_max) /* Create a leasename */
        {
            char mapcount_string[14];

            snprintf(mapcount_string, 14, "mapcount=%04d", mapcounter);

            if (leasename) newleasename = strdup(leasename);
            newleasenamelen = strlen(newleasename) + 15;
            newleasename = realloc(newleasename, newleasenamelen*sizeof(char));
            lcmaps_log_debug(3,"%s: leasename before adding mapcount: %s\n", logstr, newleasename);
            if (strstr(leasename, "\001"))
                strncat(newleasename, ":", 2);
            else
                strncat(newleasename, "\001", 2);
            strncat(newleasename, mapcount_string, 14);
            lcmaps_log_debug(3,"%s: leasename after adding mapcount: %s\n", logstr, newleasename);
        }
        else
        {
            lcmaps_log(LOG_ERR,"%s: the request for mapping nr. %d, exceeds the maximum of %d\n", logstr, mapcounter, mapping_max);
            return NULL;
        }
    }
    else if (mapping_max == 0) /* This means no mapping is allowed at all! */
    {
        lcmaps_log(LOG_ERR,"%s: The maximum number of account mappings per credential set is zero, so nothing to map (failure)! (You may want to adjust the \"-max_mappings_per_credential\" configuration parameter!)\n", logstr);
        return NULL;
    }
    else
    {
        lcmaps_log(LOG_ERR,"%s: Illegal value for \"-max_mappings_per_credential\" (%d). Should at least be %d (failure)\n", logstr, mapping_max, LCMAPS_MAPPING_MIN);
        return NULL;
    }
    return newleasename;
}


static int plugin_run_or_verify(
        int argc,
        lcmaps_argument_t * argv,
        int lcmaps_mode
)
{
    char *              logstr = "lcmaps_plugin_voms_poolaccount-plugin_run()";
    char *              dn                  = NULL;
    int                 dn_cnt              = 0;
    char *              groupname           = NULL;
    struct group        *group_info         = NULL;
    char *              username            = NULL;
    struct passwd       *user_info          = NULL;
    int                 i                   = 0;
    int                 cnt_pri_gid         = 0;
    gid_t *             pri_gid             = NULL;
    int                 cnt_sec_gid         = 0;
    gid_t *             sec_gid             = NULL;
    char *              gidstring           = NULL;
    char                gidbuffer[LCMAPS_MAXGIDBUFFER];
    int                 index               = 0;
    char **             vo_cred_string_list = NULL;
    int                 cnt_vo_cred_string  = 0;
    int                 found_mapping       = 0;
    char *              leasename           = NULL;
    int                 leasenamelen        = 0;
    char *              newleasename        = NULL;
    unsigned short      matching_type       = ((unsigned short)0x0000);
    int                 rc                  = 0;
    char **             fqan_list           = NULL;
    int                 nfqan               = -1;
    int                 mapcounter          = -1;
    int *               pmapcounter         = NULL;
    char *              requested_username  = NULL;
    uid_t               requested_uid       = -1;
    struct passwd       *requested_user_info = NULL;
    void *              value               = NULL;

    /*
     * The beginning
     */
    if (lcmaps_mode == PLUGIN_RUN)
        logstr = "lcmaps_plugin_voms_poolaccount-plugin_run()";
    else if (lcmaps_mode == PLUGIN_VERIFY)
        logstr = "lcmaps_plugin_voms_poolaccount-plugin_verify()";
    else
    {
        lcmaps_log(LOG_ERR, "lcmaps_plugin_voms_poolaccount-plugin_run_or_verify(): attempt to run plugin in invalid mode: %d\n", lcmaps_mode);
        goto fail_voms_poolaccount;
    }
    lcmaps_log_debug(5,"%s:\n", logstr);

    /*
     * Try to get the ordered values:
     */
    if ( ( value = lcmaps_getArgValue("user_dn", "char *", argc, argv) ) )
    {
	dn = *(char **) value;
        lcmaps_log_debug(5,"%s: found dn: %s\n", logstr, dn);

        /* Check if we don't have a DN already registered, if not, add it to the internal registry */
        getCredentialData (DN, &dn_cnt);
        if (dn_cnt == 0)
        {
            lcmaps_log_debug (5, "%s: Adding DN: %s\n", logstr, dn);
            addCredentialData(DN, &dn);
        }
    }
    else
        lcmaps_log_debug(1,"%s: could not get value of dn !\n", logstr);

    if ( ( value = lcmaps_getArgValue("mapcounter", "int", argc, argv) ) )
    {
	pmapcounter = (int *) value;
        lcmaps_log_debug(3,"%s: mapcounter: %d\n", logstr, *pmapcounter);
        mapcounter = *pmapcounter;
    }
    else
    {
        /* Don't quit because of backwards compatibility */
        lcmaps_log_debug(1,"%s: could not get value of mapcounter !\n", logstr);
    }

    /*
     * Try to fetch uid the invocator of LCMAPS wants to be
     * verified. (only in PLUGIN_VERIFY mode).
     */
    if (lcmaps_mode == PLUGIN_VERIFY)
    {
        if ( (value = lcmaps_getArgValue("requested_uid", "uid_t", argc, argv)) > 0 )
        {
            struct passwd * requested_user = NULL;
	    requested_uid = *(uid_t *) value;
            if ((requested_user = getpwuid(requested_uid)) != NULL)
            {
                requested_username = strdup(requested_user->pw_name);
		if (requested_username)
		    lcmaps_log_debug(3,"%s: the requested user is %s(%d)\n",
			    logstr, requested_username, (int) requested_uid);
		else    {
		    lcmaps_log(LOG_ERR,"%s: out of memory\n",logstr);
		    goto fail_voms_poolaccount;
		}
            }
            else
            {
                lcmaps_log(LOG_ERR,"%s: the requested uid is illegal\n", logstr, (int) requested_uid);
                goto fail_voms_poolaccount;
            }
        }
        else if (requested_uid == 0)
        {
            lcmaps_log(LOG_ERR,"%s: illegal request for uid == 0 (failure)\n", logstr);
            goto fail_voms_poolaccount;
        }
        else
        {
            lcmaps_log(LOG_ERR,"%s: illegal request for uid < 0 (failure)\n", logstr);
            goto fail_voms_poolaccount;
        }
    } else {
	/*
	 * See if we have a requested username: make sure not to get value
	 * immediately or we'll get a segfault
	 */
	requested_username=NULL;
	if ( (value = lcmaps_getArgValue("requested_username", "char *", argc, argv))!=NULL &&
	     (*(char **)value)!=NULL &&
	     (requested_username=strdup(*(char **)value))==NULL)
	{
	    lcmaps_log(LOG_ERR,"%s: out of memory\n",logstr);
	    goto fail_voms_poolaccount;
	}
	lcmaps_log_debug(1,"%s: requested username is %s\n", logstr,
		requested_username ? requested_username : "unset");
    }

    /*
     * Get the VO user information.
     * We can either order it by lcmaps_argument_t or use the getCredentialData() function.
     * The latter case requires the voms parsing plugin (lcmaps_voms.mod) to have run beforehand.
     * Unfortunately the formats of the VOMS strings (from getCredentialData()) and
     * FQANs (from lcmaps_argument_t) are not the same. We may have to introduce
     * two-way conversion functions.
     * The VOMS info has to matched against the info in the gridmapfile
     */
    lcmaps_log_debug(1,"%s: First try to get the FQAN list from input credential repository ...\n", logstr);
    if ( ( value = lcmaps_getArgValue("nfqan", "int", argc, argv) ) )
    {
	nfqan = *(int *) value;
        lcmaps_log_debug(4,"%s: the list of FQANs should contain %d elements\n", logstr, nfqan);
        if ( ( value = lcmaps_getArgValue("fqan_list", "char **", argc, argv) ) )   {
	    fqan_list = *(char ***) value;
            lcmaps_log_debug(4, "%s: found list of FQANs\n", logstr);
	}
        else
        {
            lcmaps_log_debug(1, "%s: could not retrieve list of FQANs (failure)!\n", logstr);
            goto fail_voms_poolaccount;
        }
        for (i = 0; i < nfqan; i++)
        {
            lcmaps_log_debug(3, "%s: FQAN %d: %s\n", logstr, i, fqan_list[i]);
        }
        vo_cred_string_list = fqan_list;
        cnt_vo_cred_string = nfqan;
    }
    else
    {
        lcmaps_log_debug(1,"%s: ... did not find input credentials in input credential repository. Trying the internal credential repository ...\n", logstr);

        vo_cred_string_list = getCredentialData(LCMAPS_VO_CRED_STRING, &cnt_vo_cred_string);
    }

    if (cnt_vo_cred_string == 0)
    {
        lcmaps_log(LOG_ERR,"%s: no VOMS group info --> no mapping (failure)\n", logstr);
        goto fail_voms_poolaccount;
    }
    else if (cnt_vo_cred_string < 0)
    {
        lcmaps_log(LOG_ERR,"%s: negative number of VOMS groups found ! (failure)\n", logstr);
        goto fail_voms_poolaccount;
    }

    /*
     * Check the gridmapfile
     */
    if ((gridmapfile != NULL) && (strlen(gridmapfile) > 0))
        lcmaps_log_debug(3,"%s: gridmapfile is: %s\n", logstr, gridmapfile);
    else
    {
        if (gridmapfile) free(gridmapfile);
        gridmapfile = NULL;
        lcmaps_log_debug(1,"%s: No gridmapfile assigned, so function must find out for it self\n", logstr);
    }

    /*
     * Check gridmapdir
     */
    if (gridmapdir == NULL) /* try if GRIDMAPDIR is already set */
    {
        char * tmpptr=NULL;
        if ((tmpptr = getenv("GRIDMAPDIR")) == NULL)
        {
            lcmaps_log(LOG_ERR,"%s: GRIDMAPDIR unknown! Specify as option or set GRIDMAPDIR\n", logstr);
            goto fail_voms_poolaccount;
        }
        else
        {
            gridmapdir = strdup(tmpptr);
        }
    }
    if (strlen(gridmapdir) == 0)
    {
        lcmaps_log(LOG_ERR,"%s: cannot set MAPDIR (strlen(gridmapdir) == 0)\n", logstr);
        goto fail_voms_poolaccount;
    }
    lcmaps_log_debug(3,"%s: setting MAPDIR to %s\n", logstr, gridmapdir);
    if (setenv("MAPDIR", gridmapdir, 1))
    {
        lcmaps_log(LOG_ERR,"%s: cannot set MAPDIR\n", logstr);
        goto fail_voms_poolaccount;
    }

    /*
     * Get the (VOMS) gids found so far and build a string out of it.
     * First primary Gid(s), behind it the secondary Gids.
     * For the moment, the same order is taken as found in the VOMS credential.
     * We might consider to sort the gids.
     * We cannot order them by lcmaps_argument_t, but have to use the getCredentialData() function
     * since it was stored there by a plugin (lcmaps_voms.mod)
     */
    pri_gid = (gid_t *)getCredentialData(PRI_GID, &cnt_pri_gid);
    if (cnt_pri_gid == 0)
    {
        if (require_primary_gid) {
            lcmaps_log(LOG_INFO,"%s: warning: no primary group found ! \n", logstr);
        }
    }
    else if (cnt_pri_gid < 0)
    {
        lcmaps_log(LOG_ERR,"%s: negative number of primary groups found ! (failure)\n", logstr);
        goto fail_voms_poolaccount;
    }
    else if (cnt_pri_gid > 1)
    {
        lcmaps_log(LOG_WARNING,"%s: warning more than 1 primary group found\n", logstr);
    }
    if ( (require_primary_gid == 1) && (cnt_pri_gid < 1) )
    {
        lcmaps_log(LOG_INFO,"%s: no primary group found (failure)\n", logstr);
        goto fail_voms_poolaccount;
    }

    sec_gid = (gid_t *)getCredentialData(SEC_GID, &cnt_sec_gid);
    if (cnt_sec_gid < 0)
    {
        lcmaps_log(LOG_ERR,"%s: negative number of secondary groups found ! (failure)\n", logstr);
        goto fail_voms_poolaccount;
    }
    /*
     * Cat the gids into a string
     */
    gidbuffer[0] = '\0';
    index = 0;
    /* First the primary gids */
    for (i = 0; i < cnt_pri_gid; i++)
    {
        if ( ( group_info = getgrgid(pri_gid[i]) ) )
        {
            groupname = group_info->gr_name;
            if ( (strlen(groupname) + 1) < (LCMAPS_MAXGIDBUFFER - index) )
            {
                strncat(gidbuffer, ":", 1);
                strncat(gidbuffer, groupname, (LCMAPS_MAXGIDBUFFER - index - 2));
                index += strlen(groupname) + 1;
                lcmaps_log_debug(3,"%s: primary groups, i=%d(%d), group_info->gr_name: %s\n", logstr,
                                 i,cnt_pri_gid,group_info->gr_name);
                lcmaps_log_debug(3,"%s: gidbuffer: %s\n", logstr,gidbuffer);
            }
            else
            {
                lcmaps_log(LOG_ERR,"%s: gidbuffer is full (%d bytes) (failure)\n", logstr, LCMAPS_MAXGIDBUFFER);
                goto fail_voms_poolaccount;
            }
        }
        else
        {
            lcmaps_log(LOG_ERR,"%s: no group id found for groupname = \"%s\"\n", logstr, groupname);
            goto fail_voms_poolaccount;
        }
    }
    /* Then the secondary gids, but only if use_secondary_gids == 1 */
    if (use_secondary_gids) /* Default Enabled */
    {
        for (i = 0; i < cnt_sec_gid; i++)
        {
            if ( ( group_info = getgrgid(sec_gid[i]) ) )
            {
                groupname = group_info->gr_name;
                if ( (strlen(groupname) + 1) < (LCMAPS_MAXGIDBUFFER - index) )
                {
                    strncat(gidbuffer, ":", 1);
                    strncat(gidbuffer, groupname, (LCMAPS_MAXGIDBUFFER - index - 2));
                    index += strlen(groupname) + 1;
                    lcmaps_log_debug(3,"%s: secondary groups, i=%d(%d), group_info->gr_name: %s\n", logstr,
                                     i,cnt_sec_gid,group_info->gr_name);
                    lcmaps_log_debug(3,"%s: gidbuffer: %s\n", logstr,gidbuffer);
                }
                else
                {
                    lcmaps_log(LOG_ERR,"%s: gidbuffer is full (%d bytes) (failure)\n", logstr, LCMAPS_MAXGIDBUFFER);
                    goto fail_voms_poolaccount;
                }
            }
            else
            {
                lcmaps_log_debug(1,"%s: no group id found for groupname = \"%s\"\n", logstr, groupname);
                goto fail_voms_poolaccount;
            }
        }
    }
    /* Create leasename based on dn+gids */
    if (gidbuffer[0] == '\0')
    {
        gidstring = NULL;
        leasenamelen = strlen(dn) + 1;
        leasename = strdup(dn);
    }
    else
    {
        gidstring = gidbuffer;
        if ( *gidstring == ':' ) gidstring++;
        leasenamelen = strlen(dn) + strlen(gidstring) + 2;
        leasename = malloc(leasenamelen*sizeof(char));
        snprintf(leasename, leasenamelen, "%s\001%s", dn, gidstring);
    }

    /*
     * Try to match the VO strings with the gridmapfile info
     * normally the first available VO string should match
     */
    found_mapping = 0;
    lcmaps_log_debug(3,"%s: using leasename: %s\n", logstr,leasename);

    matching_type = MATCH_INCLUDE|MATCH_WILD_CHARS;

    /* if override_consistency is set add this to the matchin_type so it will take effect */
    if (override_inconsistency)
        matching_type = matching_type|OVERRIDE_INCONSISTANCY;

    /* Do not create new leases in verification mode */
    if (lcmaps_mode == PLUGIN_VERIFY)
        matching_type = matching_type|ONLY_USE_EXISTING_LEASE;

    /* if strict_poolprefix_match is set add this to the matchin_type so it will take effect */
    if (strict_poolprefix_match)
        matching_type = matching_type|MATCH_STRICT_PREFIX_NUM;

    for (i = 0; i < cnt_vo_cred_string; i++)
    {
        if ( (i > 0) && (require_primary_gid == 1) )
        {
            lcmaps_log(LOG_ERR,"%s: no match (or no poolaccount available) for primary group (%s) in %s (failure)\n", logstr, vo_cred_string_list[0], gridmapfile);
            goto fail_voms_poolaccount;
        }

        /*
         * TODO: Need verify method for lcmaps_gridlist(): given the uid return the poolindex.
         * In that case we should check that the corresponding mapcount is still allowed.
         * Benefit of doing it this ways is that we don't have  to loop through all imaps
         * (maybe expensive calls).
         * But for now: loop through all possible mapcounts
         */
        if ((lcmaps_mode == PLUGIN_VERIFY) && (mapcounter < LCMAPS_MAPPING_MIN))
        {
            int imap = 0;

            for (imap = 1; imap <= mapping_max; imap++)
            {
                /* clean username and newleasename before each call to lcmaps_gridlist */
                if (username) free(username);
                username = NULL;
                if (newleasename) free(newleasename);
                newleasename = NULL;

                if ((newleasename = add_mapcount_to_leasename(mapping_max, imap, leasename)) == NULL)
                {
                    lcmaps_log(LOG_ERR, "%s: Error creating leasename including mapcounter (%d)\n", logstr, imap);
                    goto fail_voms_poolaccount;
                }

                if ( (rc = lcmaps_gridlist(vo_cred_string_list[i], &username, gridmapfile, matching_type, ".", newleasename)) == 0)
                {
		    /* in verify mode, requested_username!=NULL */
		    if ( strcmp(username, requested_username) == 0)
                    {
                        lcmaps_log_debug(3,"%s: found requested username: %s\n", logstr, username);
                        found_mapping = 1;
                        /*
                         * The following break statement places us outside the mapcount (imap) loop.
                         * Another break is needed to get out of the fqan loop.
                         */
                        break;
                    }
		    lcmaps_log_debug(3,"%s: found username %s but does not match requested username %s\n", logstr, username, requested_username);
                }
                else if (rc == LCMAPS_MOD_NOFILE)
                {
                    lcmaps_log(LOG_ERR, "%s: Could not find the gridmapfile %s\n", logstr, gridmapfile);
                    goto fail_voms_poolaccount;
                }
                else
                {
                    lcmaps_log_debug(1, "%s: no match (or no poolaccount available) for group (%s), mapcount = %d, in %s\n",
                                     logstr, vo_cred_string_list[i], imap, gridmapfile);
                }
            }
            if (found_mapping == 0) /* Let's try the legacy behaviour (use the (invalid) mapcounter) */
            {
                /* clean username and newleasename before each call to lcmaps_gridlist */
                if (username) free(username);
                username = NULL;
                if (newleasename) free(newleasename);
                newleasename = NULL;

                if ((newleasename = add_mapcount_to_leasename(mapping_max, mapcounter, leasename)) == NULL)
                {
                    lcmaps_log(LOG_ERR, "%s: Error creating leasename including mapcounter (%d)\n", logstr, mapcounter);
                    goto fail_voms_poolaccount;
                }
                if ( (rc = lcmaps_gridlist(vo_cred_string_list[i], &username, gridmapfile, matching_type, ".", newleasename)) == 0)
                {
		    /* in verify mode, requested_username!=NULL */
		    if ( strcmp(username, requested_username) == 0)
		    {
                        lcmaps_log_debug(3,"%s: found requested username: %s\n", logstr, username);
                        found_mapping = 1;
                    }
		    else
                        lcmaps_log_debug(3,"%s: found username %s but does not match requested username: %s\n", logstr, username, requested_username);

                }
                else if (rc == LCMAPS_MOD_NOFILE)
                {
                    lcmaps_log(LOG_ERR, "%s: Could not find the gridmapfile %s\n", logstr, gridmapfile);
                    goto fail_voms_poolaccount;
                }
                else
                {
                    lcmaps_log_debug(1, "%s: No entry found for %s in %s, mapcount=%d\n", logstr, dn, gridmapfile, mapcounter);
                }
            }
            /* Break again out of fqan loop, if username found */
            if (found_mapping) break;
        }
        else
        {
            /* clean username and newleasename before each call to lcmaps_gridlist */
            if (username) free(username);
            username = NULL;
            if (newleasename) free(newleasename);
            newleasename = NULL;

            if ((newleasename = add_mapcount_to_leasename(mapping_max, mapcounter, leasename)) == NULL)
            {
                lcmaps_log(LOG_ERR, "%s: Error creating leasename including mapcounter (%d)\n", logstr, mapcounter);
                goto fail_voms_poolaccount;
            }

            if ( (rc = lcmaps_gridlist(vo_cred_string_list[i], &username, gridmapfile, matching_type, ".", newleasename)) == 0)
            {
                if (lcmaps_mode == PLUGIN_VERIFY)
                {
                    lcmaps_log_debug(3,"%s: found a username: %s\n", logstr, username);
		    /* in verify mode, requested_username!=NULL */
                    if (strcmp(username, requested_username) == 0)
                    {
                        lcmaps_log_debug(3,"%s: found requested username: %s\n", logstr, username);
                        found_mapping = 1;
                        break;
                    }
                    else
                    {
                        lcmaps_log_debug(1,"%s: The requested username \"%s\" was not found for %s in %s, mapcount=%d\n",
                                         logstr, requested_username, vo_cred_string_list[i], gridmapfile, mapcounter);
                    }
                }
                else
                {
		    /* requested_username can be NULL */
		    if (requested_username==NULL)   {
			found_mapping = 1;
			lcmaps_log_debug(3,"%s: found username: %s\n", logstr, username);
		    }
		    else if (strcmp(username, requested_username) == 0)
		    {
			found_mapping = 1;
			lcmaps_log_debug(3,"%s: found requested username: %s\n", logstr, username);
		    }
		    else
		    {
			lcmaps_log(LOG_NOTICE,"%s: The requested username \"%s\" does not match the poolaccount \"%s\" for %s in %s, mapcount=%d\n",
				logstr, requested_username, username, dn, gridmapfile, mapcounter);
			goto fail_voms_poolaccount;
		    }
                    break;
                }
            }
            else if (rc == LCMAPS_MOD_NOFILE)
            {
                lcmaps_log(LOG_ERR, "%s: Could not find the gridmapfile %s\n", logstr, gridmapfile);
                goto fail_voms_poolaccount;
            }
            else
            {
                lcmaps_log_debug(1, "%s: no match (or no poolaccount available) for group (%s) in %s\n", logstr, vo_cred_string_list[i], gridmapfile);
            }
        }
    }
    /*
     * Because in the verify method we can end with a non-matching username, checking if the username exists
     * is not enough. So we check the found_mapping flag instead.
     */
    if (found_mapping == 0)
    {
        if (lcmaps_mode == PLUGIN_VERIFY)
        {
            lcmaps_log(LOG_ERR, "%s: account \"%s\" could not be matched for any of the fqans for user %s in %s\n",
                       logstr, requested_username, dn, gridmapfile);
        }
        else
        {
            lcmaps_log(LOG_ERR, "%s: no poolaccount available for any of the fqans for user %s in %s\n",
                       logstr, dn, gridmapfile);
        }
        goto fail_voms_poolaccount;
    }

    /*
     * Get userid to pwd_t structure
     */
    if (username && (strlen(username) > 0))
    {
        if ( ( user_info = getpwnam(username) ) )
        {
            char *  encoded_lease = NULL;

            lcmaps_log_debug(5,"%s: address user_info: %p\n", logstr, user_info);
            lcmaps_log_debug(3,"%s: username : %s, char ptr: %p, address char ptr: %p\n", logstr, user_info->pw_name, user_info->pw_name, &(user_info->pw_name));
            lcmaps_log_debug(3,"%s: password : %s\n", logstr, user_info->pw_passwd);
            lcmaps_log_debug(3,"%s: user_id  : %d, address uid: %p\n", logstr, user_info->pw_uid, &(user_info->pw_uid));
            lcmaps_log_debug(3,"%s: group_id : %d\n", logstr, user_info->pw_gid);
            lcmaps_log_debug(3,"%s: realname : %s\n", logstr, user_info->pw_gecos);
            lcmaps_log_debug(3,"%s: home dir : %s\n", logstr, user_info->pw_dir);
            lcmaps_log_debug(3,"%s: shellprg : %s\n", logstr, user_info->pw_shell);

            /*
             * Add this credential data to the credential data repository in the plugin manager
             */
            addCredentialData(UID, &(user_info->pw_uid));

            /* Map primary Unix GID from the account info */
            if ((!do_not_map_primary_gid) &&
                (add_primary_gid_from_mapped_account))
            {
                lcmaps_log_debug(4,"%s: adding primary GID (%d) from pool account to CredentialData\n", logstr, user_info->pw_gid);
                addCredentialData(PRI_GID, &(user_info->pw_gid));
            }

            /* Add the primary GID from the mapped account as an secondary GID to the result */
            if (add_primary_gid_as_secondary_gid_from_mapped_account)
            {
                lcmaps_log_debug(4,"%s: adding primary GID (%d) from pool account as a secondary GID to CredentialData\n", logstr, user_info->pw_gid);
                addCredentialData(SEC_GID, &(user_info->pw_gid));
            }

            /* Add secondary Unix group IDs from the mapped local account */
            if (add_secondary_gids_from_mapped_account)
            {
                /* Retrieve secondary group id's */
                if (lcmaps_get_gidlist(username, &cnt_sec_gid, &sec_gid)==0)
                {
                    lcmaps_log_debug(4,"%s: adding secondary GIDs (%d) from pool account to CredentialData\n", logstr, user_info->pw_gid);
                    for (i = 0; i < cnt_sec_gid; i++)
                    {
                        addCredentialData(SEC_GID, &(sec_gid[i]));
                    }
                    free(sec_gid);
                }
            }

            /* Added because of the POOL_INDEX request for the DAS */
            encoded_lease = gridmapdir_urlencode(newleasename);
            addCredentialData(POOL_INDEX, &encoded_lease);
            if (encoded_lease)
            {
                free(encoded_lease);
                encoded_lease = NULL;
            }
            /* Check if the uid found matches the uid requested and issue a warning if not*/
            if (lcmaps_mode == PLUGIN_VERIFY)
            {
                if (requested_uid != user_info->pw_uid)
                {
                    if ( ( requested_user_info = getpwuid(requested_uid) ) )
                    {
                        lcmaps_log(LOG_ERR, "%s: warning: mismatch between username found (%d/%s) and requested (%d/%s)\n",
                            logstr,
                            (int) user_info->pw_uid, user_info->pw_name,
                            (int) requested_user_info->pw_uid, requested_user_info->pw_name);
                    }
                    else
                    {
                        lcmaps_log(LOG_ERR, "%s: warning: mismatch between username found (%d/%s) and requested (%d/%s)\n",
                            logstr,
                            (int) user_info->pw_uid, user_info->pw_name,
                            (int) requested_uid, NULL);
                    }
                }
            }
        }
        else
        {
            lcmaps_log(LOG_ERR,"%s: no valid user account found with the name \"%s\"\n", logstr, username);
            goto fail_voms_poolaccount;
        }
    }
    else
    {
        /* error (msg is already given) */
        goto fail_voms_poolaccount;
    }

    /* succes */
 success_voms_poolaccount:
    if (requested_username) free(requested_username);
    if (username) free(username);
    if (leasename) free(leasename);
    if (newleasename) free(newleasename);
    lcmaps_log(LOG_INFO,"%s: voms_poolaccount plugin succeeded\n", logstr);
    return LCMAPS_MOD_SUCCESS;

 fail_voms_poolaccount:
    if (requested_username) free(requested_username);
    if (username) free(username);
    if (leasename) free(leasename);
    if (newleasename) free(newleasename);
    lcmaps_log(LOG_INFO,"%s: voms_poolaccount plugin failed\n", logstr);
    return LCMAPS_MOD_FAIL;
}

/******************************************************************************
Function:   plugin_terminate
Description:
    Terminate plugin
Parameters:

Returns:
    LCMAPS_MOD_SUCCESS : succes
    LCMAPS_MOD_FAIL    : failure
******************************************************************************/
int plugin_terminate()
{
    char * logstr = "lcmaps_plugin_voms_poolaccount-plugin_terminate()";

    lcmaps_log_debug(4,"%s: terminating\n", logstr);

    if (gridmapfile) free(gridmapfile);
    if (gridmapdir) free(gridmapdir);

    return LCMAPS_MOD_SUCCESS;
}

/******************************************************************************
CVS Information:
    $Source: /srv/home/dennisvd/svn/mw-security/lcmaps-plugins-voms/src/voms/lcmaps_voms_poolaccount.c,v $
    $Date: 2010-02-19 06:01:37 $
    $Revision: 1.21 $
    $Author: okoeroo $
******************************************************************************/
