/**
 * Copyright (c) Members of the EGEE Collaboration. 2004-2010. 
 * See http://www.eu-egee.org/partners/ for details on the copyright
 * holders.  
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"); 
 * you may not use this file except in compliance with the License. 
 * You may obtain a copy of the License at 
 * 
 *     http://www.apache.org/licenses/LICENSE-2.0 
 * 
 * Unless required by applicable law or agreed to in writing, software 
 * distributed under the License is distributed on an "AS IS" BASIS, 
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. 
 * See the License for the specific language governing permissions and 
 * limitations under the License.
 *
 *
 *  Authors:
 *  2009-
 *     Oscar Koeroo <okoeroo@nikhef.nl>
 *     Mischa Sall\'e <msalle@nikhef.nl>
 *     David Groep <davidg@nikhef.nl>
 *     NIKHEF Amsterdam, the Netherlands
 *     <grid-mw-security@nikhef.nl> 
 *
 *  2007-2009
 *     Oscar Koeroo <okoeroo@nikhef.nl>
 *     David Groep <davidg@nikhef.nl>
 *     NIKHEF Amsterdam, the Netherlands
 *
 *  2003-2007
 *     Martijn Steenbakkers <martijn@nikhef.nl>
 *     Gerben Venekamp <venekamp@nikhef.nl>
 *     Oscar Koeroo <okoeroo@nikhef.nl>
 *     David Groep <davidg@nikhef.nl>
 *     NIKHEF Amsterdam, the Netherlands
 *
 */


/*!
    \page lcmaps_voms_localgroup.mod voms localgroup plugin

    \section vomslocalgroupsyn SYNOPSIS
    \b lcmaps_voms_localgroup.mod
        -GROUPMAPFILE|-groupmapfile|-GROUPMAP|-groupmap \<groupmapfile\>
        [-mapall] [-mapmin \<group count\>]

    \section vomslocalgroupdesc DESCRIPTION

    The localgroup acquisition plugin is a 'VOMS-aware' plugin.
    It uses the VOMS information (acquired by the plugin \ref lcmaps_voms.mod "lcmaps_voms.mod")
    to gather primary and secondary GIDs.
    This is accomplished by matching VO-GROUP-ROLE(-CAPABILITY) combinations in the so-called
    \e groupmapfile (gridmapfile style) and by finding the corresponding local GID.
    Wildcards can be used in the groupmapfile to match VO-GROUP-ROLE combinations.

    EXAMPLE 'groupmapfile':

    \c "/VO=atlas/GROUP=mcprod"   \c atmcprod

    \c "/VO=atlas/GROUP=*"        \c atlasgrps


    A VO-GROUP combination /VO=atlas/GROUP=mcprod matches \c "/VO=atlas/GROUP=mcprod",
    resulting in a mapping to the GID of the 'atmcprod' group.
    All the other groups within the 'atlas' VO will be mapped to 'atlasgrps'.
    A user with /VO=cms/GROUP=user will not be mapped to any local system group, unless there will 
    be an extra row in the groupmapfile like '\c "/VO=*"  \c allothers'
    resulting in a mapping from any other VO-GROUP-ROLE combination to 'allothers'.
    The mapping is based on the first match found for a VO-GROUP-ROLE combination, implying
    that the most significant row must be on top.

    The poolgroup plugin will try to match each VO-GROUP-ROLE combination that was
    found by the plugin \ref lcmaps_voms.mod "lcmaps_voms.mod".
    The first VO-GROUP-ROLE combination will become the primary group, the others secondary groups.
    As the primary GID may be used for auditing and accounting purposes it is important that
    the user uses the correct ordering of VO-GROUP-ROLE combinations in his grid credential (X509
    certificate).

    \section options OPTIONS
    \subsection vomslocalgroupoptie1 -GROUPMAPFILE \<groupmapfile\>
        See \ref vomslocalgroupoptie4 "-groupmap"
 
    \subsection vomslocalgroupoptie2 -groupmapfile \<groupmapfile\>
        See \ref vomslocalgroupoptie4 "-groupmap"
 
    \subsection vomslocalgroupoptie3 -GROUPMAP \<groupmapfile\>
        See \ref vomslocalgroupoptie4 "-groupmap"
 
    \subsection vomslocalgroupoptie4 -groupmap \<groupmapfile\>
        If this option is set, it will override the default path to the groupmapfile.
        It is advised to use an absolute path to the groupmapfile to avoid usage of the wrong file(path).
    
    \subsection vomslocalgroupoptie5 -mapall
        If this parameter is set, the plugin only succeeds if it manages to map all voms data entries
        to (system) groups and find their GID.
        There is no communication between different plugins (like the voms_poolgroup plugin)
        about the failures.
        A log entry will state the VO-GROUP-ROLE combination that made the plugin fail.

    \subsection vomslocalgroupoptie6 -mapmin \<group count\>
        This option will set a minimum amount of groups that have to be resolved for later mapping.
        If the minimum is not set then the minimum amount is set to '0' by default.
        If the plugin is not able to the required number of local groups it will fail.
        Note: if the minimum is set to zero or the minimum is not set
        the plugin will return a success if no other errors occur, even if no local groups were found.

\section vomslocalgroupReturnvalue RETURN VALUES
        \li LCMAPS_MOD_SUCCESS : Success
        \li LCMAPS_MOD_FAIL    : Failure
 
 
\section vomslocalgroupErrors ERRORS
        See bugzilla for known errors (http://marianne.in2p3.fr/datagrid/bugzilla/)
 
\section vomslocalgroupSeeAlso SEE ALSO
        \ref lcmaps_voms.mod "lcmaps_voms.mod",
        \ref lcmaps_voms_poolaccount.mod "lcmaps_voms_poolaccount.mod",
        \ref lcmaps_voms_poolgroup.mod "lcmaps_voms_poolgroup.mod",
        \ref lcmaps_localaccount.mod "lcmaps_localaccount.mod",
        \ref lcmaps_poolaccount.mod "lcmaps_poolaccount.mod",
        \ref lcmaps_posix_enf.mod "lcmaps_posix_enf.mod",
        \ref lcmaps_ldap_enf.mod "lcmaps_ldap_enf.mod",
*/

/*!
    \file   lcmaps_voms_localgroup.c
    \brief  Interface to the LCMAPS plugins
    \author Martijn Steenbakkers for the EU DataGrid.

    This file contains the code of the voms_localgroup plugin
    -# plugin_initialize()
    -# plugin_run()
    -# plugin_terminate()
    -# plugin_introspect()
*/

/*****************************************************************************
                            Include header files
******************************************************************************/
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <pwd.h>
#include <ctype.h>

#include "lcmaps_voms_config.h"
#include <lcmaps/lcmaps_modules.h>
#include <lcmaps/lcmaps_arguments.h>
#include <lcmaps/lcmaps_cred_data.h>
#include "lcmaps_gridlist.h"

/******************************************************************************
                                Definitions
******************************************************************************/
#define PLUGIN_RUN      0
#define PLUGIN_VERIFY   1

/******************************************************************************
                          Module specific prototypes
******************************************************************************/
static int plugin_run_or_verify(int, lcmaps_argument_t *, int);

/******************************************************************************
                       Define module specific variables
******************************************************************************/

static char *groupmapfile            = NULL;
static int   mapall                  = 0;
static int   mapmin                  = 0;
static int   map_to_secondary_groups = 0;

/******************************************************************************
Function:   plugin_initialize
Description:
    Initialize plugin
Parameters:
    argc, argv
    argv[0]: the name of the plugin
Returns:
    LCMAPS_MOD_SUCCESS : succes
    LCMAPS_MOD_FAIL    : failure
    LCMAPS_MOD_NOFILE  : db file not found (will halt LCMAPS initialization)
******************************************************************************/
int plugin_initialize(
        int argc,
        char ** argv
)
{
    char * logstr = "lcmaps_plugin_voms_localgroup-plugin_initialize()";
    int i, j;

    lcmaps_log_debug(5,"%s: passed arguments:\n", logstr);
    for (i=0; i < argc; i++)
    {
       lcmaps_log_debug(5,"%s: arg %d is %s\n", logstr, i, argv[i]);
    }

    /*
     * the first will be the thing to edit/select (groupmap(file))
     */

    /*
     * Parse arguments, argv[0] = name of plugin, so start with i = 1
     */
    for (i = 1; i < argc; i++)
    {
        if ( ((strcmp(argv[i], "-groupmap") == 0) ||
              (strcmp(argv[i], "-GROUPMAP") == 0) ||
              (strcmp(argv[i], "-groupmapfile") == 0) ||
              (strcmp(argv[i], "-GROUPMAPFILE") == 0))
             && (i + 1 < argc))
        {
            if ((argv[i + 1] != NULL) && (strlen(argv[i + 1]) > 0))
            {
                 groupmapfile = strdup(argv[i + 1]);
            }
            i++;
        }
        else if (strcmp(argv[i], "--map-to-secondary-groups") == 0)
        {
             map_to_secondary_groups = 1;
        }
        else if (strcmp(argv[i], "-mapall") == 0)
        {
             mapall = 1;
        }
        else if ((strcmp(argv[i], "-mapmin") == 0)
                 && (i + 1 < argc))
        {
            if ((argv[i + 1] != NULL) && (strlen(argv[i + 1]) > 0))
            {
                 /* check parameter integrety */
                 for (j = 0; j < (strlen(argv[i + 1])); j++)
                 {
                     if (isdigit((argv[i + 1])[j]) == 0)
                     {
                         lcmaps_log(LOG_ERR,"%s: Error in initialization parameter: %s (%s is not a number)\n", logstr, argv[i], argv[i + 1]);
                         return LCMAPS_MOD_FAIL;
                     }
                 }

                 mapmin = atoi(argv[i + 1]);
            }
            i++;
        }
        else
        {
            lcmaps_log(LOG_ERR,"%s: Error in initialization parameter: %s (failure)\n", logstr,
                       argv[i]);
            return LCMAPS_MOD_FAIL;
        }
    }

    return LCMAPS_MOD_SUCCESS;
} 

/******************************************************************************
Function:   plugin_introspect
Description:
    return list of required arguments
Parameters:

Returns:
    LCMAPS_MOD_SUCCESS : succes
    LCMAPS_MOD_FAIL    : failure
******************************************************************************/
int plugin_introspect(
        int * argc,
        lcmaps_argument_t ** argv
)
{
    char *                   logstr = "lcmaps_plugin_voms_localgroup-plugin_introspect()";
    static lcmaps_argument_t argList[] = {
        {"user_dn"      ,       "char *"        , 0,   NULL},
        {"fqan_list"    ,       "char **"       , 0,   NULL},
        {"nfqan"        ,       "int"           , 0,   NULL},
        {NULL           ,       NULL            , -1,   NULL}
    };

    lcmaps_log_debug(4,"%s: introspecting\n", logstr);

    *argv = argList;
    *argc = lcmaps_cntArgs(argList);
    lcmaps_log_debug(5,"%s: address first argument: 0x%x\n", logstr,argList);

    return LCMAPS_MOD_SUCCESS;
}


/******************************************************************************
Function:   plugin_run
Description:
    Gather credentials for LCMAPS
Parameters:
    argc: number of arguments
    argv: list of arguments
Returns:
    LCMAPS_MOD_SUCCESS: authorization succeeded
    LCMAPS_MOD_FAIL   : authorization failed
******************************************************************************/
int plugin_run(
        int argc,
        lcmaps_argument_t * argv

)
{
    return plugin_run_or_verify(argc, argv, PLUGIN_RUN);
}

/******************************************************************************
Function:   plugin_verify
Description:
    Verify if user is entitled to use local credentials based on his grid
    credentials. This means that the site should already have been set up
    by, e.g., LCMAPS in a previous run. This method will not try to setup
    account leases, modify (distributed) passwd/group files, etc. etc.
    The outcome should be identical to that of plugin_run().
    In this particular case "plugin_verify()" is identical to "plugin_run()"

Parameters:
    argc: number of arguments
    argv: list of arguments
Returns:
    LCMAPS_MOD_SUCCESS: authorization succeeded
    LCMAPS_MOD_FAIL   : authorization failed
******************************************************************************/
int plugin_verify(
        int argc,
        lcmaps_argument_t * argv
)
{
    return plugin_run_or_verify(argc, argv, PLUGIN_VERIFY);
}

static int plugin_run_or_verify(
        int argc,
        lcmaps_argument_t * argv,
        int lcmaps_mode
)
{
    char *                logstr = "lcmaps_plugin_voms_localgroup-plugin_run()";
    char *                dn                  = NULL; 
    int                   dn_cnt              = 0;
    char *                groupname           = NULL;
    struct group *        group_info          = NULL;
    int                   i                   = 0;
    char **               vo_cred_string_list = NULL;
    int                   cnt_vo_cred_string  = 0;
    int                   group_counter       = 0;
    int                   rc                  = 0;
    lcmaps_vo_mapping_t * lcmaps_vo_mapping   = NULL;
    char **               fqan_list           = NULL;
    int                   nfqan               = -1;
    void *                value               = NULL;

    /*
     * The beginning
     */
    if (lcmaps_mode == PLUGIN_RUN)
        logstr = "lcmaps_plugin_voms_localgroup-plugin_run()";
    else if (lcmaps_mode == PLUGIN_VERIFY)
        logstr = "lcmaps_plugin_voms_localgroup-plugin_verify()";
    else
    {
        lcmaps_log(LOG_ERR, "lcmaps_plugin_voms_localgroup-plugin_run_or_verify(): attempt to run plugin in invalid mode: %d\n", lcmaps_mode);
        goto fail_voms_localgroup;
    }
    lcmaps_log_debug(4,"%s:\n", logstr);

    /*
     * Try to get the ordered values:
     */
    if ( (value = lcmaps_getArgValue("user_dn", "char *", argc, argv) ) )
    {
	dn = *(char **)value; 
        lcmaps_log_debug(5,"%s: found dn: %s\n", logstr, dn);

        /* Check if we don't have a DN already registered, if not, add it to the internal registry */
        getCredentialData (DN, &dn_cnt);
        if (dn_cnt == 0)
        {
            lcmaps_log_debug (5, "%s: Adding DN: %s\n", logstr, dn);
            addCredentialData(DN, &dn);
        }
    }
    else    {
	dn = NULL;
        lcmaps_log_debug(1,"%s: could not get value of dn !\n", logstr);
    }

    /*
     * Get the VO user information.
     * We can either order it by lcmaps_argument_t or use the getCredentialData() function.
     * The latter case requires the voms parsing plugin (lcmaps_voms.mod) to have run beforehand.
     * Unfortunately the formats of the VOMS strings (from getCredentialData()) and
     * FQANs (from lcmaps_argument_t) are not the same. We may have to introduce
     * two-way conversion functions.
     * The VOMS info has to matched against the info in the gridmapfile
     */
    lcmaps_log_debug(1,"%s: First try to get the FQAN list from input credential repository ...\n", logstr);
    if ( ( value = lcmaps_getArgValue("nfqan", "int", argc, argv) ) )
    {
	nfqan = *(int *) value;
        lcmaps_log_debug(1,"%s: the list of FQANs should contain %d elements\n", logstr, nfqan);
        if ( ( value = lcmaps_getArgValue("fqan_list", "char **", argc, argv) ) )   {
	    fqan_list = *(char ***) value;
            lcmaps_log_debug(1, "%s: found list of FQANs\n", logstr);
	}
	else
        {
            lcmaps_log_debug(1, "%s: could not retrieve list of FQANs (failure)!\n", logstr);
            goto fail_voms_localgroup;
        }
        for (i = 0; i < nfqan; i++)
        {
            lcmaps_log_debug(3, "%s: FQAN %d: %s\n", logstr, i, fqan_list[i]);
        }
        vo_cred_string_list = fqan_list;
        cnt_vo_cred_string = nfqan;
    }
    else
    {
        lcmaps_log_debug(1,"%s: ... did not find input credentials in input credential repository...\n", logstr);
        lcmaps_log_debug(1,"%s: ... trying the internal credential repository ...\n", logstr);

        vo_cred_string_list = getCredentialData(LCMAPS_VO_CRED_STRING, &cnt_vo_cred_string);
    }

    if (cnt_vo_cred_string == 0)
    {
        lcmaps_log(LOG_INFO,"%s: no VOMS group info --> no mapping\n", logstr);
        goto fail_voms_localgroup;
    }
    else if (cnt_vo_cred_string < 0)
    {
        lcmaps_log(LOG_ERR,"%s: negative number of VOMS groups found ! (failure)\n", logstr);
        goto fail_voms_localgroup;
    }

    /*
     * Check the groupmapfile
     */

    if ((groupmapfile != NULL) && (strlen(groupmapfile) > 0))
        lcmaps_log_debug(1,"%s: groupmapfile is: %s\n", logstr, groupmapfile);
    else
    {
        lcmaps_log(LOG_ERR,"%s: error finding the groupmapfile: %s (hint: use the option \"-groupmapfile <groupmapfile>\")\n", logstr, groupmapfile);
        goto fail_voms_localgroup;
    }

    /*
     * Try to find the unix groups from the VO info in the groupmapfile
     * The first group (if found) should become the primary group
     */
    for (i = 0; i < cnt_vo_cred_string; i++)
    {
        /* clean groupname before each call to lcmaps_gridlist */
        if (groupname) free(groupname);
        groupname = NULL;
        if ( (rc = lcmaps_gridlist(vo_cred_string_list[i], &groupname, groupmapfile, MATCH_EXCLUDE|MATCH_WILD_CHARS, ".", NULL) ) == 0)
        {
            lcmaps_log_debug(1,"%s: found groupname: %s\n", logstr, groupname);
            group_counter++;

            if (groupname && (strlen(groupname) > 0))
            {
                if ( ( group_info = getgrnam(groupname) ) )
                {
                    /* When map_to_secondary_groups is true, all results will be stored as secondary Unix group IDs */
                    if ((i == 0) && (!map_to_secondary_groups))
                    {
                        /* First VO group */
                        addCredentialData(PRI_GID, (void *) &(group_info->gr_gid));
                    }
                    else
                    {
                        /* Other VO groups */
                        addCredentialData(SEC_GID, (void *) &(group_info->gr_gid));
                    }
                    /*
                     * The coupling between VO information and the GID is maintained
                     * in the lcmaps_vo_mapping structure, which is added to the credential data
                     */
                    lcmaps_vo_mapping=lcmaps_createVoMapping(
                        vo_cred_string_list[i],
                        groupname,
                        group_info->gr_gid
                    );
                    if (! lcmaps_vo_mapping)
                    {
                        lcmaps_log(LOG_ERR,"%s: could not create VoMapping structure (failure)\n", logstr);
                        goto fail_voms_localgroup;
                    }
//                        lcmaps_printVoMapping(2, lcmaps_vo_mapping);
                    /* Add credential */
                    addCredentialData(LCMAPS_VO_CRED_MAPPING, (void *) lcmaps_vo_mapping);
                    if ( lcmaps_deleteVoMapping(&lcmaps_vo_mapping) )
                    {
                        lcmaps_log(LOG_ERR,"%s: error while deleting VoMapping structure (failure)\n", logstr);
                        goto fail_voms_localgroup;
                    }
                }
                else
                {
                    lcmaps_log(LOG_ERR,"%s: no Unix group id found for groupname = \"%s\"\n", logstr, groupname);
                    goto fail_voms_localgroup;
                }
            }
            else
            {
                lcmaps_log(LOG_ERR,"%s: error getting value of groupname (failure)!\n", logstr);
                goto fail_voms_localgroup;
            }
        }
        else if (rc == LCMAPS_MOD_NOFILE)
        {
            lcmaps_log(LOG_ERR, "%s: Could not find the groupmapfile %s\n", logstr, groupmapfile);
            goto fail_voms_localgroup;
        }
        else
        {
            lcmaps_log_debug(1,"%s: could not get value of groupname !\n", logstr);
            if (mapall)
            {
                lcmaps_log(LOG_INFO,"%s: no mapping for VO group %s\n", logstr, vo_cred_string_list[i]);
                goto fail_voms_localgroup;
            }
        }
    }

    if (group_counter < mapmin)
    {
        lcmaps_log(LOG_ERR,"%s: Not enough groups found. The minimum is set to %d. The plugin found %d\n", logstr, mapmin, group_counter);
        goto fail_voms_localgroup;
    }

    /* success */
 success_voms_localgroup:
    if (groupname) free(groupname);
    lcmaps_log(LOG_INFO,"%s: voms_localgroup plugin succeeded\n", logstr);
    return LCMAPS_MOD_SUCCESS;

 fail_voms_localgroup:
    if (groupname) free(groupname);
    lcmaps_log(LOG_INFO,"%s: voms_localgroup plugin failed\n", logstr);
    return LCMAPS_MOD_FAIL;
}

/******************************************************************************
Function:   plugin_terminate
Description:
    Terminate plugin
Parameters:

Returns:
    LCMAPS_MOD_SUCCESS : succes
    LCMAPS_MOD_FAIL    : failure
******************************************************************************/
int plugin_terminate()
{
    char *           logstr = "lcmaps_plugin_voms_localgroup-plugin_terminate()";

    lcmaps_log_debug(4,"%s: terminating\n", logstr);

    if (groupmapfile) free(groupmapfile);

    return LCMAPS_MOD_SUCCESS;
}

/******************************************************************************
CVS Information:
    $Source: /srv/home/dennisvd/svn/mw-security/lcmaps-plugins-voms/src/voms/lcmaps_voms_localgroup.c,v $
    $Date: 2010-02-19 06:01:37 $
    $Revision: 1.8 $
    $Author: okoeroo $
******************************************************************************/
