/* vim: set noet ts=4:
 *
 * Copyright (c) 2002-2003 Martin A. Godisch <martin@godisch.de>.
 *
 * This program is free software; you can redistribute it and/or modify it under
 * the terms of the GNU General Public License as published by the Free Software
 * Foundation; either version 2 of the License, or (at your option) any later
 * version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License along with
 * this program; if not, write to the Free Software Foundation, Inc., 59 Temple
 * Place, Suite 330, Boston, MA 02111-1307 USA.
 */
/* Based on mdb-export.c from http://sourceforge.net/projects/mdbtools.
 */
#include <lsvt2.h>
#ifdef HAVE_LIBMDB

#include <charset.h>
#include <latrine.h>
#include <mdbtools.h>
#include <memory.h>

#define NUM_COLS 6

static MdbHandle   *dict  = NULL;
static MdbTableDef *table = NULL;
static char *value[NUM_COLS];

/* open the dictionary
 *
 * returns  0: success
 * returns -1: failure (errmsg called)
 */
int open_lsvt2(const char *file)
{
	MdbCatalogEntry *entry;
	int i;

	assert(dict == NULL);
	mdb_init();
	if ((dict = mdb_open((char*)file)) != NULL) {
		mdb_read_catalog(dict, MDB_TABLE);
		for (i = 0; i < dict->num_catalog; i++) {
			entry = g_ptr_array_index(dict->catalog, i);
			if (entry->object_type == MDB_TABLE && strcmp(entry->object_name, "Voc") == 0) {
				table = mdb_read_table(entry);
				break;
			}
		}
	}
	if (table == NULL) {
		errmsg(_("cannot open dictionary: %s"), errno == 0 ? _("invalid or unknown format") : strerror(errno));
		return -1;
	}
	mdb_read_columns(table);
	mdb_rewind_table(table);
	for (i = 0; i < NUM_COLS; i++) {
		value[i]    = (char*)MALLOC(MDB_BIND_SIZE);
		value[i][0] = '\0';
		mdb_bind_column(table, i + 1, value[i]);
	}
	/* can be used to initialize --label
	if (header_row)
		for (i = 0; i < NUM_COLS; i++) {
			col = g_ptr_array_index(table->columns, i);
			errmsg("column: %s", col->name);
		}
	*/
	return 0;
}

/* read a struct word from dictionary and wordlist
 *
 * returns  1: success
 * returns  0: no more words available
 * returns -1: failure (errmsg called)
 */
int read_lsvt2(gzFile hits, struct word *w)
{
	char buffer[BUFSIZE];
	char *s, *t;
	MdbColumn *col;
	static size_t n    = 0;
	static size_t line = 1;
	int i;

	assert(table != NULL);
	memset(w, 0, sizeof(struct word));
	while (mdb_fetch_row(table) != 0) {
		for (i = 0; i <= 1; i++) {
			col = g_ptr_array_index(table->columns, i);
			strncpy(buffer, value[i], BUFSIZE);
			if ((s = strstr(buffer, "<lin>")) == NULL || (t = strstr(buffer, "</lin>")) == NULL || t < s) {
				FREE(&w->lang[0]);
				return read_lsvt2(hits, w);
			}
			*t = 0;
			memcpy(buffer, s + 5, strlen(s) - 4);
			iso2utf(buffer, BUFSIZE);
			if (strncmp(buffer, "<sty 0-204>", 11) == 0) {
				wchar_t buf2[BUFSIZE];
				memcpy(buffer, buffer + 11, BUFSIZE - 11);
				mbstowcs(buf2, buffer, BUFSIZE);
				wcsaddn(buf2, 0x350);
				wcstombs(buffer, buf2, BUFSIZE);
			}
			w->lang[i] = STRDUP(buffer);
		}
		if (w->lang[0] != NULL && w->lang[1] != NULL) {
			w->pos = n++;
			for (; hits != NULL && gzgets(hits, buffer, BUFSIZE) != Z_NULL; line++) {
				if ((s = index(buffer, '\n')) != NULL)
					*s = 0;
				if (*buffer == 0 || *buffer == '#')
					continue;
				if (sscanf(buffer, "%d:%d", &w->correct, &w->mistaken) != 2) {
					errmsg(_("ignoring invalid format in wordlist line %d"), line);
					w->correct  = 0;
					w->mistaken = 0;
				}
				break;
			}
			return 1;
		}
		FREE(&w->lang[0]);
		FREE(&w->lang[1]);
	}
	return 0;
}

/* close the dictionary
 *
 * returns 0: success or failure (no errmsg)
 */
int close_lsvt2(void)
{
	int i;

	assert(dict != NULL);
	for (i = 0; i < NUM_COLS; i++)
		FREE(&value[i]);
	mdb_free_handle(dict);
	dict = NULL;
	mdb_exit();
	return 0;
}

#endif
