/***************************************************************************
 *   Copyright (C) 2005 - 2007 by                                          *
 *      Christian Muehlhaeuser, Last.fm Ltd <chris@last.fm>                *
 *      Erik Jalevik, Last.fm Ltd <erik@last.fm>                           *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   51 Franklin Steet, Fifth Floor, Boston, MA  02111-1307, USA.          *
 ***************************************************************************/

#ifndef METADATA_EXTENSION_H
#define METADATA_EXTENSION_H

#include <QMovie>
#include <QScrollArea>
#include <QStackedWidget>
#include "interfaces/ExtensionInterface.h"
#include "metadata.h"
#include "URLLabel.h"
#include "watermarkwidget.h"
#include "WebService/fwd.h"
#include "ui_notplayingwidget.h"
#include "ui_metadata3.h"


class MetaDataWidget;
class MetaDataScrollArea;
class Http;
class Container;


enum MetaDataFetchedState
{
    NothingFetched = 0,
    AlbumPicFetched,
    ArtistPicFetched
};


class MetaDataExtension : public ExtensionInterface
{
    Q_OBJECT
    Q_ENUMS( MetaDataFetchedState )
    Q_INTERFACES( ExtensionInterface )

    friend class MetaDataWidget;
    friend class MetaDataScrollArea;

    public:
        MetaDataExtension();

        QString name() const;
        QString version() const;
        QString tabCaption() const { return tr( "Now Playing" ); }

        virtual bool hasGui() { return true; }

        QWidget* gui();
        bool guiEnabled() { return ( m_widget != 0 ); }

        QWidget* owner() { return m_parent; }
        void setOwner( QWidget* parent ) { m_parent = parent; setParent( parent ); }
        void setContainer( Container* cont ) { m_container = cont; }
		void setAlwaysFetchMetaData ( bool value ) { m_alwaysFetchMetaData = value; }

        //HACK for Container::onRestStateTimerTimeout()
        bool isTuningIn() const { return m_stack.currentIndex() == 1; }

    signals:
        void trackModerated( const MetaData& );
        void tagButtonClicked();
        void urlHovered( const QString& url );
		void metadataFetched( MetaData song );

    public slots:
        // Used by player listener and radio when a new track starts
        void setMetaData( MetaData metaData );

        // Call this when a radio station has just been requested
        void displayTuningIn();
        
        void clear();

    private:
        void initGui();
        void adjustTopLabels();
        void adjustTrackLabels();
        void adjustArtistLabels();
        int adjustLabelWidth( QWidget* label, int y = -1, bool adaptToViewport = true );
        int adjustLabelSize( QWidget* label, int y = -1 );
        void moveLabel( QWidget* label, int y );
        void adjustWidgetSize();
        int viewWidth();
        
        void applyMarginAndSpacing();
        void translateItem( QWidget* item, int xOffset, int yOffset );

        virtual void resizeEvent( QResizeEvent* event );

        bool isVisible();

        void clearArtist();
        void clearAlbum();
        void clearTrack();
        void resetRequests();

        void setDefaultAlbumCover();
        void setDefaultArtistPic();
        void downloadPic( Http* loader, const QUrl& url );
        void renderWithDropShadow( const QByteArray& from, ImageButton* to );
        void render( const QByteArray& from, ImageButton* to );

        void requestMetaData();
        void populateWiki();
        void updateDragData();

        void buildHomeBrewedUrls();
        void buildTooltips();

        QString urlEncodeItem( QString item );

        void fixDragLabelSizes();
        void resizeBottomSpacers();
        
        Ui::MetaDataExtension ui;
        Ui::NotPlayingWidget ui_notPlaying;

        QStackedWidget m_stack;
        MetaDataScrollArea* m_scrollArea;
        QWidget* m_widget;
        QWidget* m_parent;
        Container* m_container;
        
        QMovie m_spinnerMovie;

        QMutex m_metadataMutex;

        MetaData m_metaData;
        Http* m_coverArtistLoader;
        Http* m_coverAlbumLoader;
        
        // we guard them behind QPointers so we don't manipulate dead objects
        QPointer<ArtistMetaDataRequest> m_artistMetaDataRequest;
        QPointer<TrackMetaDataRequest> m_trackMetaDataRequest;

        QPixmap m_buyAlbum;
        QPixmap m_buyTrack;

        // Calculated positions of where the wiki link should go depending on
        // whether the Write icon is visible or not.
        QPoint m_wikiLinkReadPos;
        QPoint m_wikiLinkWritePos;

        QString m_urlBase; // jp or www

        bool m_haveTrack;
        bool m_requestsReset;
        bool m_offline;

        class QTimer *m_tuning_in_timer;
		
		MetaDataFetchedState m_metadataFetched;
        void checkMetaDataFetched( MetaDataFetchedState state );
		bool m_alwaysFetchMetaData;

    private slots:
        void artistMetaDataReady();
        void trackMetaDataReady();

        void artistPicClicked();
        void albumPicClicked();

        void coverArtistLoaded( const QByteArray& to );
        void coverAlbumLoaded( const QByteArray& to );
        
        void urlLeft();
		
		void buyTrackIconClicked();
		void buyAlbumIconClicked();
        
        void onDelayedTuningIn();
};


class MetaDataWidget : public WatermarkWidget
{
    Q_OBJECT

    public:
        MetaDataWidget( MetaDataExtension* owner, QWidget* parent = 0 ) :
            WatermarkWidget(parent), m_owner(owner) {}

    protected:
        virtual void paintEvent( QPaintEvent * event );

    private:
        MetaDataExtension* m_owner;
};


class MetaDataScrollArea : public QScrollArea
{
    Q_OBJECT

    public:
        MetaDataScrollArea( MetaDataExtension* owner, QWidget* parent = 0 ) :
            QScrollArea(parent), m_owner(owner) {}

    protected:
        virtual void resizeEvent( QResizeEvent* event )
        {
            m_owner->resizeEvent( event );
            QScrollArea::resizeEvent( event );
        }

        virtual void paintEvent( QPaintEvent * event );

    private:
        MetaDataExtension* m_owner;
};

#endif
