
#ifndef LAST_FM_H
#define LAST_FM_H

#include "lastfmtools_common.h"
#include "StationUrl.h"

/** 
 * Predeclare some common class names to save effort and development time
 */

class Request;

enum ItemType { ItemArtist = 1, ItemTrack, ItemAlbum, ItemTag, ItemUser, ItemStation, ItemUnknown };



/** 
 * Add functions common to any part of the Application here
 * keep includes to a minimum by checkign for the Qt header guards
 */

#ifdef QT_GUI_LIB
#include <QIcon>
namespace LastFm
{
    MCEXP QIcon icon( const char *name );
}
#endif


// Some nice macros for writing getters and setters automatically
#define PROP_GET( Type, lower ) \
    private: Type m_##lower; \
    public: Type lower() const { return m_##lower; } \
    private:
#define PROP_GET_SET( Type, lower, Upper ) \
    PROP_GET( Type, lower ) \
    public: void set##Upper( Type lower ) { m_##lower = lower; } \
    private:
    
    

#include <QString>
#include <QVariant>

class MCEXP Track 
{
    //TODO we have issues with naming and many different classes for this kind of data
    // main naming issue is class name and the track member data string

    PROP_GET_SET( QString, artist, Artist )
    PROP_GET_SET( QString, title, Title )
    
public:
    bool operator==( const Track& other )
    {   
        return other.m_artist == m_artist && m_title == other.m_title;
    }

    QString displayText() const
    {
        //FIXME duplicated in TrackInfo.cpp
        if (m_artist.isEmpty())
            return m_title; //NOTE could be empty too
        else if (m_title.isEmpty())
            return m_artist;
        else 
            return m_artist + " - " + m_title;
    }
    
    QString toString() const { return displayText(); }
    operator QString() const { return displayText(); }
    operator QVariant() const { return QVariant(displayText()); }
    
    bool isEmpty() const 
    {
        return m_title.isEmpty() && m_artist.isEmpty();
    }
    
    static Track fromMimeData( const class QMimeData* );
};


class MCEXP Station
{
    QString m_name;
    StationUrl m_url;

public:
    Station() {}
    Station( const StationUrl& u, const QString& n ) : m_name( n ), m_url( u ) 
    {}

    QString name() const { return m_name.isEmpty() ? m_url : m_name; }
    StationUrl url() const { return m_url; }
    
    void setUrl( const QString& s ) { m_url = StationUrl( s ); }
    void setName( const QString& s ) { m_name = s; }
};


#include <QStringList>

class MCEXP WeightedString : public QString
{
    union {
        int weighting;
        int count;
    } u;
    
public:
    WeightedString() { u.weighting = -1; }
    
    explicit WeightedString( QString name, int w = -1 ) : QString( name ) { u.weighting = w; }
    
    static WeightedString weighted( QString name, int w )
    {
        WeightedString t( name );
        t.u.weighting = w;
        return t;
    }
    
    static WeightedString counted( QString name, int c )
    {
        WeightedString t( name );
        t.u.count = c;
        return t;
    }

    int count() const { return u.count; }
    int weighting() const { return u.weighting; }
};


class MCEXP WeightedStringList : public QList<WeightedString>
{
public:
    WeightedStringList() {}
    WeightedStringList( QList<WeightedString> list ) : QList<WeightedString>( list ) {}

    operator QStringList() 
    {
        QStringList strings;
        foreach (WeightedString t, *this)
            strings << QString(t);
        return strings;
    }

    void sortWeightingAscending() 
    {
        qSort( begin(), end(), weightLessThan ); 
    }
    
    void sortWeightingDescending() 
    {
        qSort( begin(), end(), weightMoreThan );
    }
    
    void sortAscending()
    {
        qSort( begin(), end(), caseInsensitiveLessThan );
    }
    
    void sortDescending()    
    {
        qSort( begin(), end(), qGreater<WeightedString>() );        
    }
    
    static bool caseInsensitiveLessThan(const QString &s1, const QString &s2)
    {
        return s1.toLower() < s2.toLower();
    } 
    
    static bool weightLessThan(const WeightedString &s1, const WeightedString &s2)
    {
        return s1.weighting() < s2.weighting();
    }
    
    static bool weightMoreThan(const WeightedString &s1, const WeightedString &s2)
    {
        return s1.weighting() > s2.weighting();
    }
};



/** 
 * @author <max@last.fm>
 */

#include <QMimeData>

namespace LastFm
{
    class MCEXP MimeData : public QMimeData
    {
    public:
        Track track() const;
        QString username() const;
        QString tag() const;
        Station station() const;
        
        bool hasTrack() const { return hasFormat( "item/track" ); }
        bool hasUser() const { return hasFormat( "item/user" ); }
        bool hasTag() const { return hasFormat( "item/tag" ); }
        bool hasStation() const { return hasFormat( "item/station" ); }
        
        ItemType itemType() const;
        QString toString() const;
    };
}



namespace LastFm
{
    /** 
     * @author <max@last.fm>
     * QStringList::sort() sorts with uppercase first
     */
    QStringList sortedCaseInsensitively( QStringList input );
}



/** make life easier for us */
#include "logger.h"
#include "RadioEnums.h"


#ifdef QT_NO_STL
	#define toStdString() toAscii().data()
	#define fromStdString( x ) fromAscii( x.c_str() )
#endif


namespace Teh = The;

#endif
