/***************************************************************************
 *   Copyright (C) 2005 - 2007 by                                          *
 *      Christian Muehlhaeuser, Last.fm Ltd <chris@last.fm>                *
 *      Erik Jaelevik, Last.fm Ltd <erik@last.fm>                          *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   51 Franklin Steet, Fifth Floor, Boston, MA  02111-1307, USA.          *
 ***************************************************************************/

#include "TrackInfo.h"
#include "Settings.h"
#include "stopwatch.h"
#include "lastfm_common.h"
#include <QDir>
#include <QMimeData>

QStringList TrackInfo::mInvalidArtists = initInvalid(); //static


QStringList
TrackInfo::initInvalid()
{
    return QStringList() 
            << "unknown artist"
            << "unknown"
            << "[unknown]"
            << "[unknown artist]";
}


TrackInfo::TrackInfo() :
        m_playCount( 0 ),
        m_duration( 0 ),
        m_timeStamp( 0 ),
        m_source( Unknown ),
        m_nextPath( 0 ),
        m_stopWatch( 0 )
{}


TrackInfo::TrackInfo( const QDomElement& e )
{
    setArtist( e.namedItem( "artist" ).toElement().text() );
    setAlbum( e.namedItem( "album" ).toElement().text() );
    setTrack( e.namedItem( "track" ).toElement().text() );
    setDuration( e.namedItem( "duration" ).toElement().text() );
    setPlayCount( e.namedItem( "playcount" ).toElement().text().toInt() );
    setFileName( e.namedItem( "filename" ).toElement().text() );
    setUniqueID( e.namedItem( "uniqueID" ).toElement().text() );
    setSource( (Source)e.namedItem( "source" ).toElement().text().toInt() );
    setAuthCode( e.namedItem( "authorisationKey" ).toElement().text() );
    
    // support 1.1.3 stringed timestamps, and 1.3.0 Unix Timestamps
    QString const timestring = e.namedItem( "timestamp" ).toElement().text();
    QDateTime const timestamp = QDateTime::fromString( timestring, "yyyy-MM-dd hh:mm:ss" );
    if (timestamp.isValid())
        setTimeStamp( timestamp.toTime_t() );
    else
        setTimeStamp( timestring.toUInt() );
}


bool
TrackInfo::sameAs( const TrackInfo& that ) const
{
    if (this->artist() != that.artist())
        return false;

    if (this->track() != that.track())
        return false;

    if (this->album() != that.album())
        return false;

    return true;
}


void
TrackInfo::timeStampMe()
{
    setTimeStamp( QDateTime::currentDateTime().toTime_t() );
}


TrackInfo::ScrobblableStatus
TrackInfo::scrobblableStatus()
{
    // Check duration
    if ( duration() < Constants::kScrobbleMinLength )
    {
        LOG( 3, "Track length is " << duration() << " s which is too short, will not submit.\n" );
        return TooShort;
    }

    // Radio tracks above preview length always scrobblable
    if ( source() == Radio )
    {
        return OkToScrobble;
    }

    // Check timestamp
    if ( timeStamp() == 0 )
    {
        LOG( 3, "Track has no timestamp, will not submit.\n" );
        return NoTimeStamp;
    }

    // Check if any required fields are empty
    if ( artist().isEmpty() )
    {
        LOG( 3, "Artist was missing, will not submit.\n" );
        return ArtistNameMissing;
    }
    if ( track().isEmpty() )
    {
        LOG( 3, "Artist, track or duration was missing, will not submit.\n" );
        return TrackNameMissing;
    }

    // Check if dir excluded
    if ( dirExcluded( path() ) )
    {
        LOG( 3, "Track is in excluded directory '" << path() << "', " << "will not submit.\n" );
        return ExcludedDir;
    }

    // Check if artist name is an invalid one like "unknown"
    foreach( QString invalid, mInvalidArtists )
    {
        if ( artist().toLower() == invalid )
        {
            LOG( 3, "Artist '" << artist() << "' is an invalid artist name, will not submit.\n" );
            return ArtistInvalid;
        }
    }

    // All tests passed!
    return OkToScrobble;
}


bool
TrackInfo::dirExcluded( const QString& path )
{
    if (path.isEmpty())
        return false;
    
    QDir dirToTest( path );
    QString pathToTest = dirToTest.canonicalPath();
    #ifdef WIN32
        pathToTest = pathToTest.toLower();
    #endif

    foreach (QString bannedPath, The::currentUser().excludedDirs())
    {
        QDir bannedDir( bannedPath );
        bannedPath = bannedDir.canonicalPath();
        #ifdef WIN32
            bannedPath = bannedPath.toLower();
        #endif

        // Try and match start of given path with banned dir
        if ( pathToTest.startsWith( bannedPath ) )
        {
            // Found, this path is from a banned dir
            return true;
        }
    }

    // The path wasn't found in exclusions list
    return false;
}


int
TrackInfo::scrobbleTime()
{
    // If we don't have a length or it's less than the minimum, return the
    // threshold
    if (duration() <= 0 || duration() < Constants::kScrobbleMinLength)
        return Constants::kScrobbleTimeMax;
    
    float scrobPoint = qBound(
        Constants::kScrobblePointMin, 
        The::settings().currentUser().scrobblePoint(),
        Constants::kScrobblePointMax );

    scrobPoint /= 100.0f;

    return qMin( Constants::kScrobbleTimeMax, int(duration() * scrobPoint) );
}


QDomElement
TrackInfo::toDomElement( QDomDocument& document ) const
{
    QDomElement item = document.createElement( "item" );

    QDomElement artist = document.createElement( "artist" );
    QDomText artistText = document.createTextNode( m_artist );
    artist.appendChild( artistText );
    item.appendChild( artist );

    QDomElement album = document.createElement( "album" );
    QDomText albumText = document.createTextNode( m_album );
    album.appendChild( albumText );
    item.appendChild( album );

    QDomElement title = document.createElement( "track" );
    QDomText titleText = document.createTextNode( m_track );
    title.appendChild( titleText );
    item.appendChild( title );

    QDomElement length = document.createElement( "duration" );
    QDomText lengthText = document.createTextNode( QString::number( m_duration ) );
    length.appendChild( lengthText );
    item.appendChild( length );

    QDomElement playtime = document.createElement( "timestamp" );
    QDomText playtimeText = document.createTextNode( QString::number( m_timeStamp ) );
    playtime.appendChild( playtimeText );
    item.appendChild( playtime );

    QDomElement playcount = document.createElement( "playcount" );
    QDomText playcountText = document.createTextNode( QString::number( m_playCount ) );
    playcount.appendChild( playcountText );
    item.appendChild( playcount );

    QDomElement filename = document.createElement( "filename" );
    QDomText filenameText = document.createTextNode( m_fileName );
    filename.appendChild( filenameText );
    item.appendChild( filename );

    QDomElement uniqueID = document.createElement( "uniqueID" );
    QDomText uniqueIDText = document.createTextNode( m_uniqueID );
    uniqueID.appendChild( uniqueIDText );
    item.appendChild( uniqueID );

    QDomElement source = document.createElement( "source" );
    QDomText sourceText = document.createTextNode( QString::number( m_source ) );
    source.appendChild( sourceText );
    item.appendChild( source );

    QDomElement authKey = document.createElement( "authorisationKey" );
    QDomText authKeyText = document.createTextNode( m_authCode );
    authKey.appendChild( authKeyText );
    item.appendChild( authKey );

    return item;
}


const QString
TrackInfo::path() const
{
    return m_paths.isEmpty() ? "" : m_paths.at( 0 );
}


void
TrackInfo::setPath( QString path )
{
    m_paths.clear();
    m_paths.append( path );
}


const QString
TrackInfo::nextPath() const
{
    if ( m_nextPath < m_paths.size() )
    {
        return m_paths.at( m_nextPath++ );
    }
    else
    {
        return "";
    }
}


void
TrackInfo::setPaths( QStringList paths )
{
    m_paths = paths;
}


QString
TrackInfo::sourceString() const
{
    switch (m_source)
    {
        case Radio:
            return "L";
        
        case Player:
        case MediaDevice:
            return "P";
        
        default:
            return "U";
    }
}
