/***************************************************************************
 *   Copyright (C) 2005 - 2007 by                                          *
 *      Christian Muehlhaeuser, Last.fm Ltd <chris@last.fm>                *
 *      Erik Jaelevik, Last.fm Ltd <erik@last.fm>                          *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   51 Franklin Steet, Fifth Floor, Boston, MA  02111-1307, USA.          *
 ***************************************************************************/

#include "TrackProgressFrame.h"
#include "lastfm_common.h"
#include "Settings.h"

TrackProgressFrame::TrackProgressFrame( QWidget *parent ) :
        ProgressFrame( parent ),
        m_mode( TrackInfo::Unknown ),
        m_trackLength( -1 ),
        m_scrobblingEnabled( true ),
        m_scrobbled( false ),
        m_clockShowsTrackTime( true )
{
    setMouseTracking( true );
    
    m_clockShowsTrackTime = The::settings().currentUser().trackFrameClockMode();
}

void
TrackProgressFrame::setMode( TrackInfo::Source mode )
{
    m_mode = mode;
    if ( m_mode == TrackInfo::Player )
    {
        // Force this off if we're doing local playback as we have no track time
        m_clockShowsTrackTime = false;
    }
    else
    {
        // Restore it to what it was previously
        m_clockShowsTrackTime = The::settings().currentUser().trackFrameClockMode();
    }
}


void
TrackProgressFrame::setNewTrack( const MetaData& track )
{
    setText( track.toString() );
    setTrackLength( track.duration() );
    m_scrobbled = false;
}


void
TrackProgressFrame::setScrobblingEnabled( bool en )
{
    m_scrobblingEnabled = en;
    setProgressEnabled( en );
}


void
TrackProgressFrame::paintEvent( QPaintEvent * event )
{
    QRect rect = event->rect();
    rect.adjust( 8, 0, -8, 0 );

    QPainter painter( this );
    drawFrame( &painter );

    int progressWidth = 0;
    if ( isScrobblable() )
    {
        if ( value() >= m_watch->getTimeOut() )
        {
            progressWidth = paintProgressBar( &painter, m_scrobbledGradient );
        }
        else
        {
            progressWidth = paintProgressBar( &painter, m_foregroundGradient );
        }
    }
    
    paintBackground( &painter, progressWidth, m_backgroundGradient );
    
    paintIcon( &painter, rect, icon() );
    
    // This calls the overridden version
    QRect clockRect = paintClock( &painter, rect );

    paintText( &painter, rect, clockRect );
}

QRect
TrackProgressFrame::paintClock( QPainter* painter, QRect rect )
{
    // texts on top
    painter->setPen( Qt::black );
    painter->setBrush( Qt::black );

    // Draw time
    QString timeString;

    if ( !m_clockText.isEmpty() )
    {
        // Clock text always overrides time string
        timeString = m_clockText;
    }
    else if ( m_clockEnabled )
    {
        //                Radio     Local
        // Track time     yes       no
        // Scrobble time  yes       yes

        Q_ASSERT( m_mode != TrackInfo::Unknown );

        if ( m_clockShowsTrackTime )
        {
            // Display track timer
            
            if ( m_mode == TrackInfo::Player )
            {
                // Have no track time info for local scrobbles
                timeString = "";
            }
            else
            {
                // Radio has track length
                int displayTime = m_reverse ? m_trackLength - value() : value();
                timeString = secondsToTimeString( displayTime );
            }
        }
        else
        {
            // Display scrobble timer

            bool scrobblingOff = !isScrobblable(); 
            if ( scrobblingOff && !m_scrobbled )
            {
                timeString = tr( "won't scrobble", "Needs to be very short, displayed in track progress bar" );
            }
            else
            {
                bool scrobbled = ( m_watch->getTimeOut() - value() ) < 1;
                if ( scrobbled )
                {
                    timeString = tr( "scrobbled", "Needs to be very short, displayed in track progress bar" );
                    m_scrobbled = true;
                }
                else
                {
                    int displayTime = m_reverse ? m_watch->getTimeOut() - value() : value();
                    timeString = secondsToTimeString( displayTime );
                }
            }
        }

    }
    
    painter->drawText( rect, Qt::AlignRight | Qt::AlignVCenter, timeString );

    return painter->fontMetrics().boundingRect( timeString );
}


void
TrackProgressFrame::mousePressEvent( QMouseEvent* event )
{
    if ( m_mode != TrackInfo::Radio )
        return;

    if ( event->button() == Qt::LeftButton )
    {
        // HACK: it's late and I want to go home
        if ( event->x() > ( width() - 40 ) )
        {
            m_clockShowsTrackTime = !m_clockShowsTrackTime;
            update();

            The::settings().currentUser().setTrackFrameClockMode( m_clockShowsTrackTime );
        }
    }    
}

void
TrackProgressFrame::mouseMoveEvent( QMouseEvent* event )
{
    if ( m_mode != TrackInfo::Radio )
        return;

    if ( event->x() > ( width() - 40 ) )
        QToolTip::showText( mapToGlobal( QPoint( width() - 40, height() / 2 ) ), 
                            tr( "Click to toggle between track time and scrobble time" ), this );
    else
        QToolTip::hideText();
}

bool
TrackProgressFrame::isScrobblable()
{
    return m_scrobblingEnabled && m_trackLength > Constants::kScrobbleMinLength;
}