//LabPlot : FFTListDialog.cc

#ifdef HAVE_GSL
#include <gsl/gsl_fft_complex.h>
#endif
#ifdef HAVE_FFTW3
#include <fftw3.h>
#elif defined(HAVE_FFTW)
#include <fftw.h>
#endif
#include "FFTListDialog.h"
#include "defs.h"

using namespace std;

#define REAL(z,i) ((z)[2*(i)])
#define IMAG(z,i) ((z)[2*(i)+1])

FFTListDialog::FFTListDialog(MainWin *mw, const char *name)
	: ListDialog(mw, name)
{
	setCaption(i18n("FFT Dialog"));
	KConfig *config = mw->Config();
	config->setGroup( "FFT" );

	QTabWidget *tw = new QTabWidget(vbox);
	QVBox *tab1 = new QVBox(tw);

	QHBox *hb = new QHBox(tab1);
	new QLabel(i18n("Library : "),hb);
        library = new KComboBox(hb);
	QStringList llist;
#if defined(HAVE_FFTW) || defined(HAVE_FFTW3)
	llist << i18n("fftw");
#endif
#ifdef HAVE_GSL
	llist << i18n("gsl");
#endif
	library->insertStringList(llist);
	library->setCurrentItem(config->readNumEntry("Library",0));

	hb = new QHBox(tab1);
	new QLabel(i18n("Transform : "),hb);
        transform = new KComboBox(hb);
	QStringList tlist;
	tlist << i18n("forward") << i18n("backward");
	transform->insertStringList(tlist);
	transform->setCurrentItem(config->readNumEntry("Transformation",0));

	hb = new QHBox(tab1);
	new QLabel(i18n("x-values : "),hb);
        xvalues = new KComboBox(hb);
	QStringList vlist;
	vlist << i18n("index")<< i18n("frequency") << i18n("period");
	xvalues->insertStringList(vlist);
	xvalues->setCurrentItem(config->readNumEntry("XValues",0));

	hb = new QHBox(tab1);
	new QLabel(i18n("y-values : "),hb);
        yvalues = new KComboBox(hb);
	QStringList ylist;
	ylist << i18n("magnitude") << i18n("real") << i18n("imag") << i18n("phase");
	yvalues->insertStringList(ylist);
	yvalues->setCurrentItem(config->readNumEntry("YValues",0));

	hb = new QHBox(tab1);
	new QLabel(i18n("number of threads : "),hb);
	threadsni = new KIntNumInput(config->readNumEntry("Threads",1),hb);
	threadsni->setRange(1,INF,1,false);
#if ! defined(HAVE_FFTW3_THREADS)
	threadsni->hide();
#endif

	Style *style=0;
	Symbol *symbol=0;
	QVBox *styletab;
	if(p && p->getPlot(p->API())->Type() == PSURFACE)
		styletab = surfaceStyle(tw,true);
	else
		styletab = simpleStyle(tw, style, symbol);

	tw->addTab(tab1,i18n("Parameter"));
	tw->addTab(styletab,i18n("Style"));

	QObject::connect(ok,SIGNAL(clicked()),SLOT(ok_clicked()));
        QObject::connect(apply,SIGNAL(clicked()),SLOT(apply_clicked()));
	QObject::connect(save,SIGNAL(clicked()),SLOT(saveSettings()));

	setMinimumWidth(vbox->minimumSizeHint().width());
	setMinimumHeight(gbox->minimumSizeHint().height()+vbox->minimumSizeHint().height());
	resize(minimumSize());
}

void FFTListDialog::saveSettings() {
	KConfig *config = mw->Config();
	config->setGroup( "FFT" );

	config->writeEntry("Library", library->currentItem());
	config->writeEntry("Transformation", transform->currentItem());
	config->writeEntry("XValues", xvalues->currentItem());
	config->writeEntry("YValues", yvalues->currentItem());
	config->writeEntry("Threads", threadsni->value());
}

int FFTListDialog::apply_clicked() {
	kdDebug()<<"	FFTListDialog::apply()"<<endl;
	
	if(s) {
		//TODO
		KMessageBox::error(this,i18n("Sorry. This function is not yet implemented!"));
		return 0;
	}

	GraphList *gl = p->getPlot(p->API())->getGraphList();
	if(gl->Number()==0) {
		KMessageBox::error(this,i18n("No graph found!"));
		return -2;
	}
	int item = (int) (lv->itemPos(lv->currentItem())/lv->currentItem()->height());

	GRAPHType st = gl->getStruct(item);

	Style *style=0;
	Symbol *symbol=0;
	if(st != GRAPHM) {
		style = new Style(cb2->currentItem(),color->color(),filled->isChecked(),fcolor->color(),
			width->value(),pencb->currentItem(),brushcb->currentItem());
		style->setBoxWidth(boxwidth->value());
		style->setAutoBoxWidth(autobox->isChecked());
		style->setPointsSorting(sortpointscb->isChecked());
		symbol = new Symbol((SType)symbolcb->currentItem(),scolor->color(),ssize->value(),
			(FType)symbolfillcb->currentItem(),sfcolor->color(),sbrushcb->currentItem());
	}
	
	QString lib = library->currentText();
	int trans = transform->currentItem();
	int xval = xvalues->currentItem();
	int yval = yvalues->currentItem();
	
	// set sign for fftw
#if defined(HAVE_FFTW3)
	int sign=0;
#elif defined(HAVE_FFTW)
	fftw_direction sign=FFTW_FORWARD;
#endif
#if defined(HAVE_FFTW3) || defined(HAVE_FFTW)
	switch(trans) {
	case 0:	sign = FFTW_FORWARD; break;
	case 1:	sign = FFTW_BACKWARD; break;
	}
#endif

#if defined(HAVE_FFTW3_THREADS)
	fftw_init_threads();
	fftw_plan_with_nthreads(threadsni->value());
#endif
	
	int nx=0,ny=0;
	double *tmp=0;
	LRange xrange, yrange;
	QString label; 
	if (st == GRAPH2D || (st == GRAPH3D && (type == P2D || type == PPOLAR || type == PTERNARY)) 
		|| st == GRAPH4D) {
		kdDebug()<<"	1D FFT"<<endl;
 
		 if(st == GRAPH2D) {
			Graph2D *g = gl->getGraph2D(item);
			xrange = g->Range(0);
			label = g->getLabel()->simpleTitle();
			nx = g->Number();
			Point *a = g->Data();
			tmp = (double *) malloc(nx*sizeof(double));	

			for (int i = 0;i<nx;i++)
				tmp[i]=a[i].Y();
		}
		else if (st == GRAPH3D) {
			Graph3D *g = gl->getGraph3D(item);
			xrange = g->Range(0);
			label = g->getLabel()->simpleTitle();
			nx = g->Number();
			Point3D *a = g->Data();
			tmp = (double *) malloc(nx*sizeof(double));	

			for (int i = 0;i<nx;i++)
				tmp[i]=a[i].Y();
		}
		else if (st == GRAPH4D) {
			Graph4D *g = gl->getGraph4D(item);
			xrange = g->Range(0);
			label = g->getLabel()->simpleTitle();
			nx = g->Number();
			Point4D *a = g->Data();
			tmp = (double *) malloc(nx*sizeof(double));	
		
			for (int i = 0;i<nx;i++)
				tmp[i]=a[i].Y();
		}

		// copy the data
		int N=(int)(nx/2.0);
#ifdef HAVE_FFTW3
		fftw_complex *in = 0, *out = 0;
		if (lib == QString("fftw")) { // only if fftw is selected
			in = (fftw_complex *) fftw_malloc(sizeof(fftw_complex) * nx);
			out = (fftw_complex *) fftw_malloc(sizeof(fftw_complex) * nx);
		}
#elif defined(HAVE_FFTW)
		fftw_complex *in = 0, *out = 0;
		if (lib == QString("fftw")) { // only if fftw is selected
			in = new fftw_complex[nx];
			out = new fftw_complex[nx];
		}
#endif

		// initialize
#ifdef HAVE_GSL
		double *data=0;
		if (lib == QString("gsl"))	// only if gsl is selected
			data = new double[2*nx];
#endif

		if (lib == QString("fftw")) {	// using FFTW
			// fft the data set
#ifdef HAVE_FFTW3
			for (int i=0;i<nx;i++) {
				in[i][0] = tmp[i];
				in[i][1] = 0;
			}

			fftw_plan plan = fftw_plan_dft_1d(nx, in, out, sign, FFTW_ESTIMATE);
			fftw_execute(plan);
			fftw_destroy_plan(plan);

			out[0][0]=out[0][1]=0.0;
			for(int i=1;i<N;i++) {
				out[i][0] += out[nx-i][0];
				out[i][1] -= out[nx-i][1];
			}
#elif defined(HAVE_FFTW)
			for (int i=0;i<nx;i++) {
				in[i].re = tmp[i];
				in[i].im = 0;
			}

			fftw_plan plan = fftw_create_plan(nx, sign, FFTW_ESTIMATE);
          
			fftw_one(plan, in, out);	
			fftw_destroy_plan(plan);

			out[0].re=out[0].im=0.0;
			for(int i=1;i<N;i++) {
				out[i].re += out[nx-i].re;
				out[i].im -= out[nx-i].im;
			}
#endif
		}
		else {	// using GSL
#ifdef HAVE_GSL
			for (int i = 0; i < nx; i++) {
				REAL(data,i) = tmp[i];
				IMAG(data,i) = 0.0;
			}

			gsl_fft_complex_workspace *workspace = gsl_fft_complex_workspace_alloc(nx);
			gsl_fft_complex_wavetable *wavetable = gsl_fft_complex_wavetable_alloc(nx);

#ifdef HAVE_GSL17
			gsl_fft_direction sign = gsl_fft_forward;
			switch(trans) {
			case 0:	sign = gsl_fft_forward; break;
			case 1:	sign = gsl_fft_backward; break;
			}
#else
			gsl_fft_direction sign = forward;
			switch(trans) {
			case 0:	sign = forward; break;
			case 1:	sign = backward; break;
			}
#endif

		 	gsl_fft_complex_transform (data,1,nx,wavetable,workspace,sign);
			gsl_fft_complex_wavetable_free(wavetable);
			gsl_fft_complex_workspace_free(workspace);

			REAL(data,0)=0.0;
			IMAG(data,0)=0.0;
			for(int i=1;i<N;i++) {
 				REAL(data,i) += REAL(data,nx-i);
				IMAG(data,i) -= IMAG(data,nx-i);
			}
#endif
		}

		// create new data set
		Point *ptr = new Point[nx];
		double xmin=0, xmax=1, ymin=0, ymax=1;
		for (int i = 0;i<N;i++) {
			double dx = xrange.rMax()-xrange.rMin();
			
			// x-values
			double x = i;
			switch (xval) {
			case 0: x = i; break;
			case 1: x = (double)i/dx; break;
			case 2: x = 2*M_PI*(double)i/dx; break;
			}

			//y-values
			double y=0;
			if (lib == QString("fftw")) {
#ifdef HAVE_FFTW3
				switch (yval) {
				case 0:	y=hypot(out[i][0],out[i][1]); break;
				case 1:	y=out[i][0]; break;
				case 2:	y=out[i][1]; break;
				case 3:	y=atan2(out[i][0],out[i][1]); break;
				}
#elif defined(HAVE_FFTW)
				switch (yval) {
				case 0:	y=hypot(out[i].re,out[i].im); break;
				case 1:	y=out[i].re; break;
				case 2:	y=out[i].im; break;
				case 3:	y=atan2(out[i].re,out[i].im); break;
				}
#endif
			}
			else {
#ifdef HAVE_GSL
				switch (yval) {
				case 0:	y=hypot(REAL(data,i),IMAG(data,i)); break;
				case 1:	y=REAL(data,i); break;
				case 2:	y=IMAG(data,i); break;
				case 3:	y=atan2(REAL(data,i),IMAG(data,i)); break;
				}
#endif
			}

			ptr[i].setPoint(x,y);
		}
		
		mw->calculateRanges2D(ptr,N,&xmin,&xmax,&ymin,&ymax);

#ifdef HAVE_FFTW3
		if (lib == QString("fftw")) {
			fftw_free(in); 
			fftw_free(out);
		}
#elif defined(HAVE_FFTW)
		if (lib == QString("fftw")) {
			free(in);
			free(out);
		}
#endif
#ifdef HAVE_GSL
		if (lib == QString("gsl"))
			free(data);
#endif

		// create the new Graph only if fft is possible
#if defined(HAVE_FFTW3) || defined(HAVE_FFTW) || defined(HAVE_GSL)
		LRange range[2];
		range[0] = LRange(xmin,xmax);
		range[1] = LRange(ymin,ymax);

		QString fun = i18n("fft of")+QString(" ")+label;

		Graph2D *ng = new Graph2D(fun,fun,range,SSPREADSHEET,P2D,style,symbol,ptr,N);
		mw->addGraph2D(ng,sheetcb->currentItem());
#endif
	}
	else if (st == GRAPH3D || st == GRAPHM) {	// 2D FFT
		kdDebug()<<"	2D FFT"<<endl;
		
		// we need fftw here since gsl dont seems to support this
		if (lib == QString("gsl")) {
			KMessageBox::error(this,i18n("Sorry. GSL doesn't support 2D FFT!"));
			return -3;
		}

		if (st == GRAPH3D) {
			kdDebug()<<"	GRAPH3D"<<endl;
			Graph3D *g = gl->getGraph3D(item);
			xrange = g->Range(0);
			yrange = g->Range(1);
			label = g->getLabel()->simpleTitle();
			nx = g->NX();
			ny = g->NY();
			tmp = (double *) malloc(nx*ny*sizeof(double));
			Point3D *a = g->Data();

			for (int i = 0;i<nx*ny;i++)
					tmp[i]=a[i].Z();
		}
		else if (st == GRAPHM) {
			kdDebug()<<"	GRAPHM"<<endl;
			GraphM *g = gl->getGraphM(item);
			xrange = g->Range(0);
			yrange = g->Range(1);
			label = g->getLabel()->simpleTitle();
			nx = g->NX();
			ny = g->NY();
			tmp = (double *) malloc(nx*ny*sizeof(double));
			double *a = g->Data();

			for (int i = 0;i<nx*ny;i++)
					tmp[i]=a[i];
		}

		kdDebug()<<"	creating structures"<<endl;
		int N=(int)(nx*ny/4.0);
#ifdef HAVE_FFTW3
		fftw_complex *in = 0, *out = 0;
		in = (fftw_complex *) fftw_malloc(sizeof(fftw_complex) * nx*ny);
		out = (fftw_complex *) fftw_malloc(sizeof(fftw_complex) * nx*ny);
#elif defined(HAVE_FFTW)
		fftw_complex *in = 0, *out = 0;
		in = new fftw_complex[nx*ny];
		out = new fftw_complex[nx*ny];
#endif

#ifdef HAVE_FFTW3
		for (int i=0;i<nx*ny;i++) {
			in[i][0] = tmp[i];
			in[i][1] = 0;
		}

		fftw_plan plan = fftw_plan_dft_2d(nx, ny, in, out, sign, FFTW_ESTIMATE);
		kdDebug()<<"Calling fftw_execute()"<<endl;
		fftw_execute(plan);
		kdDebug()<<"DONE"<<endl;
		fftw_destroy_plan(plan);
		
		out[0][0]=out[0][1]=0.0;

		for(int i=1;i<N;i++) {
			out[i][0] += out[nx*ny-i][0];
			out[i][1] -= out[nx*ny-i][1];
		}

#elif defined(HAVE_FFTW)
		for (int i=0;i<nx*ny;i++) {
			in[i].re = tmp[i];
			in[i].im = 0;
		}

		fftwnd_plan plan = fftw2d_create_plan(nx, ny, sign, FFTW_ESTIMATE);
	
		fftwnd_one(plan, in, out);	
		fftwnd_destroy_plan(plan);
	
		out[0].re=out[0].im=0.0;
		for(int i=1;i<N;i++) {
			out[i].re += out[nx*ny-i].re;
			out[i].im -= out[nx*ny-i].im;
		}
#endif

		// create new data set
		Point3D *ptr = new Point3D[N];
		double xmin=0, xmax=1, ymin=0, ymax=1, zmin=0, zmax=1;
		for (int i = 0;i<N;i++) {
			double dx = xrange.rMax()-xrange.rMin();
			double dy = yrange.rMax()-yrange.rMin();
			
			double x = i,y = i;
			switch (xval) {
			case 0: x = (int)i/ny*2; y = i%ny/2; break;
			case 1: x = (double)i/(ny/2*dx); y = (double) (i%ny/2/dy); break;
			case 2: x = 2*M_PI*(double)i/(ny/2*dx); y = 2*M_PI*(double)(i%ny/2/dy); break;
			}

			double z=0;
#ifdef HAVE_FFTW3
			switch (yval) {
			case 0:	z=hypot(out[i][0],out[i][1]); break;
			case 1:	z=out[i][0]; break;
			case 2:	z=out[i][1]; break;
			case 3:	z=atan2(out[i][0],out[i][1]); break;
			}
#elif defined(HAVE_FFTW)
			switch (yval) {
			case 0:	z=hypot(out[i].re,out[i].im); break;
			case 1:	z=out[i].re; break;
			case 2:	z=out[i].im; break;
			case 3:	z=atan2(out[i].re,out[i].im); break;
			}
#endif
			ptr[i].setPoint(x,y,z);
		}
		
		mw->calculateRanges3D(ptr,N,&xmin,&xmax,&ymin,&ymax,&zmin,&zmax);

#ifdef HAVE_FFTW3
		fftw_free(in); 
		fftw_free(out);
#elif defined(HAVE_FFTW)
		free(in);
		free(out);
#endif

		// create the new Graph only if 2D fft is possible
#if defined(HAVE_FFTW3) || defined(HAVE_FFTW)
		LRange range[3];
		range[0] = LRange(xmin,xmax);
		range[1] = LRange(ymin,ymax);
		range[2] = LRange(zmin,zmax);

		QString fun = i18n("2D FFT of")+QString(" ")+label;

		Graph3D *ng = new Graph3D(fun,fun,range,SSPREADSHEET,type,style,symbol,ptr,nx/2,ny/2);
		ng->setNumber(N);
		ng->setNumberX(nx/2);
		ng->setNumberY(ny/2);
		mw->addGraph3D(ng,sheetcb->currentItem(),type);
#endif
	}

#if defined(HAVE_FFTW3_THREADS)
	fftw_cleanup_threads();
#endif
	
	updateList();
	
	return 0;
}
