//LabPlot : FitListDialog.cc

#include <kconfig.h>
#ifdef HAVE_GSL
#include <gsl/gsl_rng.h>
#include <gsl/gsl_blas.h>
#endif
#include "FitListDialog.h"
#include "fit.h"

using namespace std;

struct data {
	int n;
	double *x;
	double *y;
	double *sigma;
	int np;
	Model model;
	double base;
	QString fun;
	MainWin *mw;
};

FitListDialog::FitListDialog(MainWin *mw, const char *name)
	: ListDialog(mw, name)
{
	setCaption(i18n("Fit Dialog"));

	if(p==0) {
		KMessageBox::error(this,i18n("No worksheet found!"));
		return;
	}

	Plot *plot = p->getPlot(p->API());

	QTabWidget *tw = new QTabWidget(vbox);
	QVBox *tab1 = new QVBox(tw);

	QHBox *hb = new QHBox(tab1);
	QLabel *tmp = new QLabel(i18n("Model : "),hb);
	tmp->setMaximumWidth(50);
	modelcb = new KComboBox(hb);
	modelcb->insertStrList(modelitems);

	KConfig *config = mw->Config();
	config->setGroup("Fit Functions" );
	// read functions
	for(int i=0;i<10;i++) {
		QString fun = config->readEntry(QString("function%1").arg(i+1));
		if (fun.length()>0) {
			modelcb->insertItem(fun);
		}
		else
			break;
	}
	
	modelcb->setCurrentItem(0);
	QObject::connect(modelcb,SIGNAL(activated (int)),SLOT(updateModel(int)));
	// custom fit function
	hb = new QHBox(tab1);
	new QLabel(i18n("Fit Function :"),hb);
	new QLabel(i18n(" y = "),hb);
	GraphList *gl = plot->getGraphList();
	Graph *g = gl->getGraph(0);
	funle = new KLineEdit(g==0?QString("a*x+b"):g->FitFunction(),hb);
	funle->setReadOnly(true);

	hb = new QHBox(tab1);
	new QLabel(i18n("Nr. of Parameter : "),hb);
	parle = new KLineEdit(QString("2"),hb);
	parle->setReadOnly(true);
	parle->setValidator(new QIntValidator(0,9,parle));
	QObject::connect(parle,SIGNAL(textChanged (const QString&)),SLOT(updateParameter()));

	new QLabel(i18n("Initial Values : "),tab1);
	hb = new QHBox(tab1);
	new QLabel(i18n("a = "),hb);
	parNle[0] = new KLineEdit(QString("1.0"),hb);
	new QLabel(i18n("b = "),hb);
	parNle[1] = new KLineEdit(QString("1.0"),hb);
	new QLabel(i18n("c = "),hb);
	parNle[2] = new KLineEdit(QString("1.0"),hb);
	hb = new QHBox(tab1);
	new QLabel(i18n("d = "),hb);
	parNle[3] = new KLineEdit(QString("1.0"),hb);
	new QLabel(i18n("e = "),hb);
	parNle[4] = new KLineEdit(QString("1.0"),hb);
	new QLabel(i18n("f = "),hb);
	parNle[5] = new KLineEdit(QString("1.0"),hb);
	hb = new QHBox(tab1);
	new QLabel(i18n("g = "),hb);
	parNle[6] = new KLineEdit(QString("1.0"),hb);
	new QLabel(i18n("h = "),hb);
	parNle[7] = new KLineEdit(QString("1.0"),hb);
	new QLabel(i18n("i = "),hb);
	parNle[8] = new KLineEdit(QString("1.0"),hb);

	for (int i=0;i<NR_PARS;i++) {
		parNle[i]->setValidator(new QDoubleValidator(parNle[i]));
		if (i>1) parNle[i]->setEnabled(false);
	}
	
	hb = new QHBox(tab1);
	resultcb = new QCheckBox(i18n("show result in plot "),hb);
	resultcb->setChecked(true);
	
	QVBox *tab2 = new QVBox(tw);
	hb = new QHBox(tab2);
	new QLabel(i18n("Maximum Steps : "),hb);
	stepsle = new KLineEdit(QString("100"),hb);
	stepsle->setValidator(new QIntValidator(stepsle));
	new QLabel(i18n(" Tolerance : "),hb);
	tolle = new KLineEdit(QString("0.001"),hb);
	tolle->setValidator(new QDoubleValidator(0,1,20,tolle));

	hb = new QHBox(tab2);
	new QLabel(i18n("Weight : "),hb);
	weightcb = new KComboBox(hb);
	int i=0;
	while(weightitems[i] != 0) weightcb->insertItem(i18n(weightitems[i++]));
	weightcb->setCurrentItem(0);
	QObject::connect(weightcb,SIGNAL(activated(int)),SLOT(weightChanged()));
	hb = new QHBox(tab2);
	new QLabel(i18n("Weight Function : "),hb);
	weightle = new KLineEdit(i18n("equal"),hb);
	weightle->setReadOnly(true);

	hb = new QHBox(tab2);
	regioncb = new QCheckBox(i18n("use Region "),hb);
	if(plot->RegionMin() != plot->RegionMax() )
		regioncb->setChecked(true);
	else
		regioncb->setChecked(false);
        new QLabel(i18n("( From "),hb);
        regionminle = new KLineEdit(QString::number(plot->RegionMin()),hb);
	regionminle->setValidator(new QDoubleValidator(regionminle));
        new QLabel(i18n(" To "),hb);
        regionmaxle = new KLineEdit(QString::number(plot->RegionMax()),hb);
	regionmaxle->setValidator(new QDoubleValidator(regionmaxle));
        new QLabel(i18n(" )"),hb);

	hb = new QHBox(tab2);
        baselinecb = new QCheckBox(i18n("Use Baseline @ y = "),hb);
       	baselinecb->setChecked(false);
	baselinele = new KLineEdit(QString::number(plot->Baseline()),hb);
	baselinele->setValidator(new QDoubleValidator(baselinele));

	hb = new QHBox(tab2);
	new QLabel(i18n("Number of Points for fit function : "),hb);
	GRAPHType st = gl->getStruct(0);
	int number=100;
	if (st == GRAPH2D) {
		Graph2D *g = gl->getGraph2D(0);
		number = g->Number();
	}
	numberle = new KLineEdit(QString::number(number),hb);
	numberle->setValidator(new QIntValidator(numberle));

	hb = new QHBox(tab2);
	new QLabel(i18n("Range of fit function : "),hb);
	LRange *range = plot->Ranges();
	minle = new KLineEdit(QString::number(range[0].rMin()),hb);
	minle->setValidator(new QDoubleValidator(minle));
	new QLabel(i18n(" .. "),hb);
	maxle = new KLineEdit(QString::number(range[0].rMax()),hb);
	maxle->setValidator(new QDoubleValidator(maxle));

	hb = new QHBox(tab2);
	rescb = new QCheckBox(i18n("Show Residuals"),hb);
	rescb->setChecked(false);

	Style *style=0;
	Symbol *symbol=0;
	QVBox *styletab;
	if(p->getPlot(p->API())->Type() == PSURFACE)
		styletab = surfaceStyle(tw,true);
	else
		styletab = simpleStyle(tw, style, symbol);

	tw->addTab(tab1,i18n("Parameter"));
	tw->addTab(tab2,i18n("Advanced"));
	tw->addTab(styletab,i18n("Style"));

	infote = new QTextEdit(vbox);

#if QT_VERSION > 0x030005
	infote->setTextFormat( Qt::LogText );
#else
	infote->setTextFormat( Qt::PlainText );
#endif

	QObject::connect(ok,SIGNAL(clicked()),SLOT(ok_clicked()));
        QObject::connect(apply,SIGNAL(clicked()),SLOT(apply_clicked()));

	setMinimumWidth(vbox->minimumSizeHint().width());
	setMinimumHeight(gbox->minimumSizeHint().height()+vbox->minimumSizeHint().height());
	resize(minimumSize());
}

void FitListDialog::weightChanged() {
	kdDebug()<<"FitListDialog::weightChanged()"<<endl;
	int item = weightcb->currentItem();
	
	weightle->setText(i18n(weightitems[item]));
	if(item==WUSER)
		weightle->setReadOnly(false);
	else
		weightle->setReadOnly(true);
}

//! called when the number of parameter is changed in user fit model
void FitListDialog::updateParameter() {
	int np = parle->text().toInt();
	
	for (int i=0;i<NR_PARS;i++) {
		parNle[i]->setEnabled(false);
		if(np>i) parNle[i]->setEnabled(true);
	}	
}

//! called when a model is selected
void FitListDialog::updateModel(int model) {
	// TODO : set inital values using selected graph values
	parle->setReadOnly(true);
	if((Model)model<=MUSER)
		funle->setText(QString(modelitems[model]));	
	funle->setReadOnly(true);
	
	if ((Model)model == MEXP || (Model)model == MLORENTZ) {
		parle->setText(QString("3"));
		parNle[2]->setEnabled(true);
	}
	else if ((Model)model == MMULTIEXP2) {
		parle->setText(QString("4"));
		parNle[2]->setEnabled(true);
		parNle[3]->setEnabled(true);
	}
	else if ((Model)model == MMULTIEXP3) {
		parle->setText(QString("6"));
		parNle[2]->setEnabled(true);
		parNle[3]->setEnabled(true);
		parNle[4]->setEnabled(true);
		parNle[5]->setEnabled(true);
	}
	else if ((Model)model == MUSER) {
		parle->setText(QString("2"));
		parle->setReadOnly(false);
		// set from selected graph
		int item = (int) (lv->itemPos(lv->currentItem())/lv->currentItem()->height());
		GraphList *gl = p->getPlot(p->API())->getGraphList();
		Graph *g = gl->getGraph(item);
		funle->setText(g==0?QString("a*x+b"):g->FitFunction());
		funle->setReadOnly(false);
	}
	else if ((Model)model > MUSER) {
		QString string = modelcb->currentText();
		QStringList list = QStringList::split(QRegExp(";"),string);
		parle->setText(list[1]);
		parle->setReadOnly(false);
		funle->setText(list[0]);
		funle->setReadOnly(false);	
		for(int i=0;i<list[1].toInt();i++)
			parNle[i]->setEnabled(true);
	}
	else  {
		parle->setText(QString("2"));
		parNle[2]->setEnabled(false);
		parNle[3]->setEnabled(false);
		parNle[4]->setEnabled(false);
		parNle[5]->setEnabled(false);
	}
}

void FitListDialog::setFunction(QString fun) { 
	modelcb->setCurrentItem(MUSER); 
	funle->setText(fun); 
}

void FitListDialog::setNrParameter(int par) {
	if(par<0) par=2;
	if(par>9) {
		KMessageBox::error(this,i18n("Not more than 9 parameters allowed!"));
		par=9;
	}
	parle->setText(QString::number(par));
	for(int i=0;i<par;i++)
		parNle[i]->setEnabled(true);
}

void FitListDialog::setInitialValue(int par, double v) {
	if(par>9 || par<0) {
		KMessageBox::error(this,i18n("Not more than 9 parameters allowed!"));
		return;
	}
	
	parNle[par]->setText(QString::number(v));
}

double FitListDialog::initialValue(int par) {
	if(par>9 || par<0) {
		KMessageBox::error(this,i18n("Not more than 9 parameters allowed!"));
		return 0;
	}

	return parNle[par]->text().toDouble();	
}

void FitListDialog::setWeightFunction(QString w) { 
	weightcb->setCurrentItem(WUSER); 
	weightle->setText(w); 
}

void FitListDialog::setRegion(double a,double b) {
	regioncb->setChecked(true); 
	regionminle->setText(QString::number(a));
	regionmaxle->setText(QString::number(b));
}

void FitListDialog::setBaseline(double b) {
	baselinecb->setChecked(true);
	baselinele->setText(QString::number(b));
}

void FitListDialog::setRange(double a,double b) {
	minle->setText(QString::number(a));
	maxle->setText(QString::number(b));
}

#ifdef HAVE_GSL
int fun_f(const gsl_vector *v, void *params, gsl_vector *f) {
//	kdDebug()<<"fun_f() "<<endl;
	int n = ((struct data *)params)->n;
	int np = ((struct data *)params)->np;
	double *x = ((struct data *)params)->x;
	double *y = ((struct data *)params)->y;
	double *sigma = ((struct data *) params)->sigma;
	Model model = ((struct data *) params)->model;
	double base = ((struct data *) params)->base;
	QString fun = ((struct data *)params)->fun;
	MainWin *mw =  ((struct data *)params)->mw;

	// parameter
	double* p = new double[np];
	for (int i=0;i<np;i++) {
		p[i]=gsl_vector_get(v,i);
//		kdDebug()<<"fun_f Parameter "<<i<<" = "<<p[i]<<endl;
	}

	QString userfun(fun);
	if (model >= MUSER) {
		if (userfun.length()==1) 
			userfun += " ";		// "x" problem

		for (int j=0;j<np;j++)
			userfun = mw->parseExpression(userfun, p[j], j);	// "a",...,"f"
	}

	for (int i = 0; i < n; i++) {
		double t = x[i];
//		kdDebug()<<"x["<<i<<"]="<<t<<endl;
		double Yi=0;
		if (model == MLINEAR)
			Yi = p[0] * t + p[1];
		else if (model == MEXP)
			Yi = p[0] * exp (-p[1] * t) + p[2];
		else if (model == MPOT)
			Yi = p[0] * pow(t,p[1]);
		else if (model == MLN) {
			double tlog;
			if (t<=0)
				tlog = 0;
			else
				tlog=log(t);
			Yi = p[0] + p[1]*tlog;
		}
		else if (model == M1L)
			Yi = 1/(p[0]+p[1]*t);
		else if (model == MEXP2)
			Yi = p[0]*t*exp(-p[1]*t);
		else if (model == MGAUSSIAN)
			Yi = 1/(sqrt(2*M_PI)*p[0])*exp(-(t-p[1])*(t-p[1])/(2*p[0]*p[0]));
		else if (model == MMAXWELL)
			Yi = p[0]*t*t*exp(-p[1]*t*t);
		else if (model == MPLANCK) {
			if (t==0)
				Yi = 0;
			else
				Yi = p[0]*t*t*t/(exp(p[1]*t)-1);
		}
		else if (model == MLORENTZ)
			Yi = p[0]/((t-p[1])*(t-p[1])+p[2]*p[2]/4);
		else if (model == MMULTIEXP2)
			Yi = p[0] * exp (p[1] * t) + p[2] * exp (p[3] * t);
		else if (model == MMULTIEXP3)
			Yi = p[0] * exp (p[1] * t) + p[2] * exp (p[3] * t) + p[4] * exp(p[5] * t);
		else if (model >= MUSER) {
			QString tmp(userfun);
			tmp = mw->parseExpression(tmp, t, 23);	// "x"
			Yi = parse((char *) tmp.latin1());
			
			if(parse_errors()>0) {
				KMessageBox::error(mw, i18n("Parse Error!\n Please check the given function."));
				return GSL_EINVAL;
			}
		}

		Yi += base;
//		kdDebug()<<"Yi = "<<Yi<<endl;

		gsl_vector_set (f, i, (Yi - y[i])/sigma[i]);
	}

	return GSL_SUCCESS;
}

int fun_df(const gsl_vector *v, void *params, gsl_matrix *J) {
//	kdDebug()<<"fun_df() "<<endl;
	int n = ((struct data *)params)->n;
	int np = ((struct data *)params)->np;
	double *x = ((struct data *)params)->x;
	double *sigma = ((struct data *) params)->sigma;
	Model model = ((struct data *) params)->model;
	QString fun = ((struct data *)params)->fun;
	MainWin *mw =  ((struct data *)params)->mw;

	double* p = new double[np];
	for (int i=0;i<np;i++) {
		p[i]=gsl_vector_get(v,i);
//		kdDebug()<<"fun_df Parameter "<<i<<" = "<<p[i]<<endl;
	}

	for (int i = 0; i < n; i++) {
		/* Jacobian matrix J(i,j) = dfi / dxj, */
		/* where fi = (Yi - yi)/sigma[i],      */
		/*       Yi = model  */
		/* and the xj are the parameters */
		double t = x[i];
//		kdDebug()<<"x["<<i<<"]="<<t<<endl;
		double s = sigma[i];
//		kdDebug()<<"fun_df : sigma["<<i<<"]="<<s<<endl;

		if (model == MLINEAR) {
			gsl_matrix_set (J, i, 0, t/s);
			gsl_matrix_set (J, i, 1, 1.0/s);
		}
		else if (model == MEXP) {
			double e = exp(-p[1] * t);
			gsl_matrix_set (J, i, 0, e/s);
			gsl_matrix_set (J, i, 1, - t * p[0] * e/s);
			gsl_matrix_set (J, i, 2, 1.0/s);
		}
		else if (model == MPOT) {
			double tlog;
			if (t<=0)
				tlog = 0;
			else
				tlog=log(t);
			gsl_matrix_set (J, i, 0, pow(t,p[1])/s);
			gsl_matrix_set (J, i, 1, p[0] * pow(t,p[1]) * tlog/s);
		}
		else if (model == MLN) {
			double plog;
			if (p[1]==0)
				plog = 0;
			else if (p[1]<0)
				plog = log(-p[1]);
			else
				plog=log(p[1]);
			gsl_matrix_set (J, i, 0, 1/s);
			gsl_matrix_set (J, i, 1, plog/s);
		}
		else if (model == M1L) {
			double tmp = s*(p[0]+p[1]*t)*(p[0]+p[1]*t);
			gsl_matrix_set (J, i, 0, -1/tmp);
			gsl_matrix_set (J, i, 1, - t/tmp);
		}
		else if (model == MEXP2) {
			gsl_matrix_set (J, i, 0, t*exp(-p[1]*t)/s);
			gsl_matrix_set (J, i, 1, -p[0]*exp(-p[1]*t)*t*t/s);
		}
		else if (model == MGAUSSIAN) {
			double e = exp(-(t-p[1])*(t-p[1])/(2*p[0]*p[0]));
			double p2 = p[0]*p[0];
			gsl_matrix_set (J, i, 0, e*(t*t-2*t*p[1]+p[1]*p[1]-p2)/sqrt(2*M_PI*p2*p2)/s);
			gsl_matrix_set (J, i, 1, e*(t-p[1])/(sqrt(2*M_PI)*p[0]*p2)/s);
		}
		else if (model == MMAXWELL) {
			double e = exp(-p[1]*t*t);
			gsl_matrix_set (J, i, 0, t*t*e/s);
			gsl_matrix_set (J, i, 1, -p[0]*e*t*t*t*t/s);
		}
		else if (model == MPLANCK) {
			if (t==0) {
				gsl_matrix_set (J, i, 0, 0);
			 	gsl_matrix_set (J, i, 1, 0);
			}
			else {
				double e = exp(p[1]*t);
				gsl_matrix_set (J, i, 0, t*t*t/(e-1)/s);
			 	gsl_matrix_set (J, i, 1, -p[0]*e*t*t*t*t/((e-1)*(e-1))/s);
			}
		}
		else if (model == MLORENTZ) {
			double tmp = p[2]*p[2]/4+(t-p[1])*(t-p[1]);
			gsl_matrix_set (J, i, 0, 1/tmp/s);
			gsl_matrix_set (J, i, 1, 2*p[0]*(t-p[1])/(tmp*tmp)/s);
			gsl_matrix_set (J, i, 2, -p[0]*p[2]/(2*tmp*tmp)/s);
		}
		else if (model == MMULTIEXP2) {
			double e1 = exp(p[1] * t);
			double e2 = exp(p[3] * t);
			gsl_matrix_set (J, i, 0, e1/s);
			gsl_matrix_set (J, i, 1, t * p[0] * e1/s);
			gsl_matrix_set (J, i, 2, e2/s);
			gsl_matrix_set (J, i, 3, t * p[2] * e2/s);
		}
		else if (model == MMULTIEXP3) {
			double e1 = exp(p[1] * t);
			double e2 = exp(p[3] * t);
			double e3 = exp(p[5] * t);
			gsl_matrix_set (J, i, 0, e1/s);
			gsl_matrix_set (J, i, 1, t * p[0] * e1/s);
			gsl_matrix_set (J, i, 2, e2/s);
			gsl_matrix_set (J, i, 3, t * p[2] * e2/s);
			gsl_matrix_set (J, i, 4, e3/s);
			gsl_matrix_set (J, i, 5, t * p[4] * e3/s);
		}
		else if (model >= MUSER) {
			QString userfun(fun);
			userfun = mw->parseExpression(userfun, t, 23);	// "x"
			
//			kdDebug()<<"userfun = "<<userfun<<endl;
			for(int j=0;j<np;j++) {	//parameter
				double dp=1.0e-5;	// variation of parameter
				QString tmp(userfun);
				for(int k=0;k<np;k++) {	// set other parameter
					if(k!=j) tmp =  mw->parseExpression(tmp, p[k], k);	// "a".."f" (not )
				}
//				kdDebug()<<"df to parse = "<<tmp<<endl;
				QString tmpf_p = mw->parseExpression(tmp, p[j], j);
				QString tmpf_pdp = mw->parseExpression(tmp, p[j]+dp*p[j], j);
//				kdDebug()<<"df f_p to parse = "<<tmpf_p<<endl;
//				kdDebug()<<"df f_pdp to parse = "<<tmpf_pdp<<endl;
				double f_p = parse((char *) tmpf_p.latin1());
				double f_pdp = parse((char *) tmpf_pdp.latin1());
				gsl_matrix_set(J,i,j,1.0/s*(f_pdp-f_p)/(dp*p[j]));
			}
		}
	}
	return GSL_SUCCESS;
}

int fun_fdf(const gsl_vector *x, void *params, gsl_vector *f,gsl_matrix *J) {
        fun_f (x, params, f);
        fun_df (x, params, J);

        return GSL_SUCCESS;
}

void FitListDialog::print_state(int iter, gsl_multifit_fdfsolver * s) {
	int np = parle->text().toInt();

	QString text;
	text+= "iter : "+QString::number(iter)+"| x = ";
	for (int i=0;i<np;i++)
		text+=QString::number(gsl_vector_get (s->x, i))+" ";
	text+="|f(x)| = "+QString::number(gsl_blas_dnrm2 (s->f));

	infote->append(text);
}
#endif

int FitListDialog::apply_clicked() {
	// TODO : all selected graphs
#ifdef HAVE_GSL
	Model model = (Model)  modelcb->currentItem();
	
	// save fit function in config
	KConfig *config = mw->Config();
	config->setGroup("Fit Functions" );
	// rotate if necessary
	int nr_functions=10;
	if(config->readEntry(QString("function%1").arg(nr_functions)).length()>0) {
		for(int i=0;i<nr_functions-1;i++) {
			config->writeEntry(QString("function%1").arg(i+1),config->readEntry(QString("function%1").arg(i+2)));
		}
		config->writeEntry(QString("function%1").arg(nr_functions),"");
	}
	//save values
	if(model >= MUSER) {
		for(int i=0;i<nr_functions;i++) {
			QString fun = config->readEntry(QString("function%1").arg(i+1));
			if(fun.length()==0) {
				QString newfun = funle->text()+QString(";%1").arg(parle->text().toInt());
				bool isnew=true;
				for (int j=1;j<=i;j++) {
					if(newfun == config->readEntry(QString("function%1").arg(j)))
						isnew=false;
				}
				if(isnew)
					config->writeEntry(QString("function%1").arg(i+1),newfun);
				break;
			}
		}
	}

	Plot *plot = p->getPlot(p->API());
	GraphList *gl = plot->getGraphList();
	if(gl->Number()==0) {
		KMessageBox::error(this,i18n("No graph found!"));
		return -2;
	}
	int item = (int) (lv->itemPos(lv->currentItem())/lv->currentItem()->height());
	GRAPHType st = gl->getStruct(item);

	double base=0;
	if (baselinecb->isChecked()) {
		base = baselinele->text().toDouble();
		plot->setBaseline(base);
	}

	// 2d : x-y, 3d : x-y-dy, 4d : x-y-dx-dy+x-y-dy1-dy2
	if (st == GRAPH2D || st == GRAPH3D || st == GRAPH4D) {
		int nx=0;
		Point *a2d=0;
		Point3D *a3d=0;
		Point4D *a4d=0;
		QString funlabel;
		bool g4type=0;
		if(st == GRAPH2D) {
			Graph2D *g = gl->getGraph2D(item);
			nx = g->Number();
			a2d = g->Data();
			funlabel = g->Label();
		}
		else if(st == GRAPH3D) {
			Graph3D *g = gl->getGraph3D(item);
			nx = g->Number();
			a3d = g->Data();
			funlabel = g->Label();
		}
		else if(st == GRAPH4D) {
			Graph4D *g = gl->getGraph4D(item);
			nx = g->Number();
			a4d = g->Data();
			funlabel = g->Label();
			g4type = g->Type();
		}

		int np = parle->text().toInt();	 //  number of parameter
		double* xdata = new double[nx];
		double* ydata = new double[nx];
		double* weight = new double[nx];
		double* sigma = new double[nx];
		kdDebug()<<" MODEL = "<<model<<endl;
		kdDebug()<<" NP = "<<np<<endl;

		// start values
		double* x_init = new double[np];
		if ((Model)model == MEXP || (Model)model == MLORENTZ) {
			for (int i=0;i<3;i++)
				x_init[i]=parNle[i]->text().toDouble();
		}
		else if ((Model)model == MMULTIEXP2) {
			for (int i=0;i<4;i++)
				x_init[i]=parNle[i]->text().toDouble();
		}
		else if ((Model)model == MMULTIEXP3) {
			for (int i=0;i<6;i++)
				x_init[i]=parNle[i]->text().toDouble();
		}
		else if ((Model)model >= MUSER) {
			for (int i=0;i<NR_PARS;i++)
				if(np>i) x_init[i]=parNle[i]->text().toDouble();
		}
		else {
			for (int i=0;i<2;i++)
				x_init[i]=parNle[i]->text().toDouble();
		}

		gsl_vector_view v = gsl_vector_view_array (x_init, np);
		gsl_rng_env_setup();

		int N=0;
		//create weight
		for (int i = 0; i < nx; i++) {
			double xx=0,yy=0;
			if(st == GRAPH2D) {
				xx=a2d[i].X();
				yy=a2d[i].Y();
				if(a2d[i].Masked())
					continue;
			}
			else if(st == GRAPH3D) {
				xx=a3d[i].X();
				yy=a3d[i].Y();
				if(a3d[i].Masked())
					continue;
			}
			else if(st == GRAPH4D) {
				xx=a4d[i].X();
				yy=a4d[i].Y();
				if(a4d[i].Masked())
					continue;
			}
			if(!regioncb->isChecked() || xx > regionminle->text().toDouble() && xx < regionmaxle->text().toDouble()) {
				xdata[N] = xx;
				ydata[N] = yy;
				
				// create weight
				if (weightcb->currentItem() == WY)
					weight[N]=a2d[N].Y();
				else if(weightcb->currentItem() == WYY)
					weight[N]=a2d[i].Y()*a2d[i].Y();
				else if(weightcb->currentItem() == W1Y) {
					if (a2d[i].Y() != 0)
						weight[N]=1.0/a2d[i].Y();
				}
				else if(weightcb->currentItem() == W1YY) {
					if (a2d[i].Y() != 0)
						weight[N]=1.0/(a2d[i].Y()*a2d[i].Y());
				}
				else if(weightcb->currentItem() == WX)
					weight[N]=a2d[i].X();
				else if(weightcb->currentItem() == WXX)
					weight[N]=a2d[i].X()*a2d[i].X();
				else if(weightcb->currentItem() == W1X) {
					if (a2d[i].X() != 0)
						weight[N]=1.0/a2d[i].X();
				}
				else if(weightcb->currentItem() == W1XX) {
					if (a2d[i].X() != 0)
						weight[N]=1.0/(a2d[i].X()*a2d[i].X());
				}
				else if(weightcb->currentItem() == WERROR) {
					double e=1.0;
					if(st == GRAPH3D)
						e = a3d[i].Z();
					else if (st == GRAPH4D) {
						if(g4type)
							e = a4d[i].Z()+a4d[i].T();	// x-y-dy1-dy2
						else
							e = a4d[i].T();				// x-y-dx-dy
					}
					if(e!=0)
						weight[N]=1.0/(e*e);
				}
				else if(weightcb->currentItem() == WUSER) {
					QString tmp;
					if(st == GRAPH2D) {
						tmp = mw->parseExpression(weightle->text(), a2d[i].X(), 23);  // "x"
						tmp = mw->parseExpression(tmp, a2d[i].Y(), 24);  // "y"
					}
					else if(st == GRAPH3D) {
						tmp = mw->parseExpression(weightle->text(), a3d[i].X(), 23);  // "x"
						tmp = mw->parseExpression(tmp, a3d[i].Y(), 24);  // "y"
					}
					else if(st == GRAPH4D) {
						tmp = mw->parseExpression(weightle->text(), a4d[i].X(), 23);  // "x"
						tmp = mw->parseExpression(tmp, a4d[i].Y(), 24);  // "y"
					}

					double value = parse((char *) tmp.latin1());
					if(parse_errors()>0) {
						KMessageBox::error(mw, i18n("Parse Error!\n Please check the given weight function."));
						return -3;
					}

					if(!finite(value))
						value=0;
					weight[N]=value;
				}

				else
					weight[N]=1.0;

				// TODO : normalize? : No; scaling problem
				//sum += sigma[N];
				
				// catch sigma[i]==0 problems
				if(weight[N]<1.0e-15)
					weight[N]=1.0e-15;

				sigma[N]=1.0/sqrt(weight[N]);

				N++;
			}
        	};

	        struct data d = { N, xdata, ydata, sigma, np, model, base, funle->text(), p->getMainWin()};
		gsl_multifit_function_fdf f;
		f.f = &fun_f;
		f.df = &fun_df;
		f.fdf = &fun_fdf;
		f.n = N;
		f.p = np;
		f.params = &d;

		const gsl_multifit_fdfsolver_type *T = gsl_multifit_fdfsolver_lmsder;
        	gsl_multifit_fdfsolver *s = gsl_multifit_fdfsolver_alloc (T, N, np);
        	gsl_multifit_fdfsolver_set (s, &f, &v.vector);
		int status,iter = 0;
		int maxsteps = stepsle->text().toInt();

		print_state (iter, s);

        	do {
                	iter++;
                	status = gsl_multifit_fdfsolver_iterate (s);

                	print_state (iter, s);

                	if (status)
                        	break;

			double tolerance = tolle->text().toDouble();
                	status = gsl_multifit_test_delta (s->dx, s->x,tolerance, tolerance);
        	} while (status == GSL_CONTINUE && iter < maxsteps);

		gsl_matrix *covar = gsl_matrix_alloc (np, np);
        	gsl_multifit_covar (s->J, 0.0, covar);

#define FIT(i) gsl_vector_get(s->x, i)
#define ERR(i) sqrt(gsl_matrix_get(covar,i,i))

		// info
		QString text;
		double chi=chi=gsl_blas_dnrm2(s->f);

		if ((Model)model == MEXP || (Model)model == MLORENTZ) {
			text += "a = "+QString::number(FIT(0))+" +/- "+QString::number(ERR(0));
			text += "\nb = "+QString::number(FIT(1))+" +/- "+QString::number(ERR(1));
			text += "\nc = "+QString::number(FIT(2))+" +/- "+QString::number(ERR(2));
		}
		if ((Model)model == MMULTIEXP2 ) {
			text += "a = "+QString::number(FIT(0))+" +/- "+QString::number(ERR(0));
			text += "\nb = "+QString::number(FIT(1))+" +/- "+QString::number(ERR(1));
			text += "\nc = "+QString::number(FIT(2))+" +/- "+QString::number(ERR(2));
			text += "\nd = "+QString::number(FIT(3))+" +/- "+QString::number(ERR(3));
		}
		if ((Model)model == MMULTIEXP3) {
			text += "a = "+QString::number(FIT(0))+" +/- "+QString::number(ERR(0));
			text += "\nb = "+QString::number(FIT(1))+" +/- "+QString::number(ERR(1));
			text += "\nc = "+QString::number(FIT(2))+" +/- "+QString::number(ERR(2));
			text += "\nd = "+QString::number(FIT(3))+" +/- "+QString::number(ERR(3));
			text += "\ne = "+QString::number(FIT(4))+" +/- "+QString::number(ERR(4));
			text += "\nf = "+QString::number(FIT(5))+" +/- "+QString::number(ERR(5));
		}
		if ((Model)model >= MUSER) {
			if(np>0) text += "a = "+QString::number(FIT(0))+" +/- "+QString::number(ERR(0));
			if(np>1) text += "\nb = "+QString::number(FIT(1))+" +/- "+QString::number(ERR(1));
			if(np>2) text += "\nc = "+QString::number(FIT(2))+" +/- "+QString::number(ERR(2));
			if(np>3) text += "\nd = "+QString::number(FIT(3))+" +/- "+QString::number(ERR(3));
			if(np>4) text += "\ne = "+QString::number(FIT(4))+" +/- "+QString::number(ERR(4));
			if(np>5) text += "\nf = "+QString::number(FIT(5))+" +/- "+QString::number(ERR(5));
			if(np>6) text += "\ng = "+QString::number(FIT(6))+" +/- "+QString::number(ERR(6));
			if(np>7) text += "\nh= "+QString::number(FIT(7))+" +/- "+QString::number(ERR(7));
			if(np>8) text += "\ni = "+QString::number(FIT(8))+" +/- "+QString::number(ERR(8));
		}
		else {
			text += "a = "+QString::number(FIT(0))+" +/- "+QString::number(ERR(0));
			text += "\nb = "+QString::number(FIT(1))+" +/- "+QString::number(ERR(1));
		}
		text += "\nstatus = "+i18n(gsl_strerror(status));
		text += "\nchi^2 = "+QString::number(chi*chi);
		infote->append(text);
		infote->scrollToBottom();

		// update parameter start values
		if ((Model)model == MEXP || (Model)model == MLORENTZ) {
			for (int i=0;i<3;i++)
				parNle[i]->setText(QString::number(FIT(i)));
		}
		else if ((Model)model == MMULTIEXP2 ) {
			for (int i=0;i<4;i++)
				parNle[i]->setText(QString::number(FIT(i)));
		}
		else if ((Model)model == MMULTIEXP3 ) {
			for (int i=0;i<6;i++)
				parNle[i]->setText(QString::number(FIT(i)));
		}
		else if ((Model)model >= MUSER) {
			for (int i=0;i<NR_PARS;i++)
				if(np>i) parNle[i]->setText(QString::number(FIT(i)));
		}
		else {	// everything else
			for (int i=0;i<2;i++)
				parNle[i]->setText(QString::number(FIT(i)));
		}

		// create fit function
		int numberx = numberle->text().toInt();
		Point *ptr = new Point[numberx];
		double rangemin=minle->text().toDouble();
		double rangemax=maxle->text().toDouble();
		double xmin=0,xmax=0,ymin=0, ymax=1;
		// reset values for residuals
		if(rescb->isChecked()) {
			LRange *range = plot->Ranges();
			numberx=N;
			rangemin=range[0].rMin();
			rangemax=range[0].rMax();
		}
		for (int i = 0;i<numberx;i++) {
			// use scale of x-axis of plot
			Axis *axis = plot->getAxis(0);
			double y=0,x=0;
			switch(axis->Scale()) {
			case LINEAR:
			case SQRT:
			case SX2:
				x =rangemin+i*(rangemax-rangemin)/(double)(numberx-1);
				break;
			case LOG10:
				x = pow(10,log10(rangemin)+i*(log10(rangemax/rangemin))/(numberx-1));
				break;
			case LOG2:
				x = pow(2,log2(rangemin)+i*(log2(rangemax/rangemin))/(numberx-1));
				break;
			case LN:
				x = pow(M_E,log(rangemin)+i*(log(rangemax/rangemin))/(numberx-1));
				break;
			}
			
			if(model == MLINEAR)
				y = FIT(0)*x+FIT(1);
			else if(model == MEXP)
				y = FIT(0)*exp(-FIT(1)*x)+FIT(2);
			else if(model == MPOT)
				y = FIT(0)*pow(x,FIT(1));
			else if(model == MLN) {
				if(x<0)
					y = FIT(0);
				else if (x==0)
					y = 0;	// -inf
				else
					y = FIT(0) + FIT(1)*log(x);
			}
			else if(model == M1L)
				y = 1/(FIT(0) + FIT(1)*x);
			else if (model == MEXP2)
				y = FIT(0)*x*exp(-FIT(1)*x);
			else if (model == MGAUSSIAN)
				y = 1/(sqrt(2*M_PI)*FIT(0))*exp(-(x-FIT(1))*(x-FIT(1))/(2*FIT(0)*FIT(0)));
			else if (model == MMAXWELL)
				y = FIT(0)*x*x*exp(-FIT(1)*x*x);
			else if (model == MPLANCK) {
				if(x==0)
					y=0;
				else
					y = FIT(0)*x*x*x/(exp(FIT(1)*x)-1);
			}
			else if (model == MLORENTZ)
				y = FIT(0)/((x-FIT(1))*(x-FIT(1))+FIT(2)*FIT(2)/4);
			else if (model == MMULTIEXP2)
				y = FIT(0)*exp(FIT(1)*x)+FIT(2)*exp(FIT(3)*x);
			else if (model == MMULTIEXP3)
				y = FIT(0)*exp(FIT(1)*x)+FIT(2)*exp(FIT(3)*x)+FIT(4)*exp(FIT(5)*x);
			else if (model >= MUSER) {
				QString tmp(funle->text());
				tmp = p->getMainWin()->parseExpression(tmp, x, 23);	// "x"
				for(int j=0;j<np;j++) {
					tmp = p->getMainWin()->parseExpression(tmp, FIT(j), j);	// "a",...,"f"
				}
				y = parse((char *) tmp.latin1());
			
				if(parse_errors()>0) {
					KMessageBox::error(mw, i18n("Parse Error!\n Please check the given function."));
					 return -3;
				}
			}

			y += base;

			if(rescb->isChecked())
				y = ydata[i]-y;

			ptr[i].setPoint(x,y);
		}

		mw->calculateRanges2D(ptr,numberx,&xmin,&xmax,&ymin,&ymax);

		gsl_multifit_fdfsolver_free (s);

		LRange range[2];
		range[0] = LRange(xmin,xmax);
		range[1] = LRange(ymin,ymax);

		// add result label to active worksheet
		
		// label for fit result in plot
		if (resultcb->isChecked()) {
			Label *label = new Label(i18n("fit result : (f(x) = "+
				funle->text()+")<br>"+text.replace(QChar('\n'),QString("<br>"))),QFont("TImes",10));
			// TODO : dont overwrite other label
			label->setPosition(0.13,0.1);
			label->setBoxed(true);
			p->setLabel(0,label);
		}

		// TODO : use fit function? (might be long)
		QString fun = QString(i18n("fit of")+QString(" ")+ funlabel);
		if(rescb->isChecked())
			fun.prepend(i18n("residuals of "));
		
		Style *style = new Style(cb2->currentItem(),color->color(),filled->isChecked(),fcolor->color(),
			widthle->text().toInt(),pencb->currentItem(),brushcb->currentItem());
		Symbol *symbol = new Symbol((SType)symbolcb->currentItem(),scolor->color(),ssize->text().toInt(),
			(FType)symbolfillcb->currentItem(),sfcolor->color(),sbrushcb->currentItem());
		
		Graph2D *ng = new Graph2D(fun,fun,range,SSPREADSHEET,P2D,style,symbol,ptr,numberx);
		ng->setFitFunction(funle->text());

		mw->addGraph2D(ng,sheetcb->currentItem());
	}
	else if (st == GRAPHM) {
		KMessageBox::error(this,i18n("Sorry. This function is not yet implemented!"));
		return -1;
		// TODO
	}

	updateList();
#else
	KMessageBox::error(this, i18n("Sorry. Your installation doesn't support the GSL!"));
#endif

	return 0;
}
